/************************************************************************/
/*									*/
/*  Read a plain text file into a BufferDocument			*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"

/************************************************************************/
/*									*/
/*  Read a document as a whole.						*/
/*									*/
/************************************************************************/

typedef struct PlainReadingContext
    {
    TextAttribute	prcTextAttribute;
    BufferItem *	prcParaItem;
    } PlainReadingContext;

static void docPlainInitReadingContext( PlainReadingContext *	prc )
    {
    docInitTextAttribute( &(prc->prcTextAttribute) );

    prc->prcParaItem= (BufferItem *)0;
    }

/************************************************************************/

static int docPlainReadParaCharacter(	SimpleInputStream *	sis,
					int *			pC,
					TextAttribute *		pTa )
    {
    int		c= sioInGetCharacter( sis );
    int		cc;

    if  ( c == EOF )
	{ return 1;	}

    if  ( c == '\t' || c == '\n' )
	{
	*pC= c; pTa->taFontIsBold= 0; pTa->taIsUnderlined= 0;
	return 0;
	}

    cc= sioInGetCharacter( sis );
    if  ( cc != '\b' )
	{
	if  ( cc != EOF )
	    { sioInUngetLastRead( sis );	}

	*pC= c; pTa->taFontIsBold= 0; pTa->taIsUnderlined= 0;
	return 0;
	}
    else{
	cc= sioInGetCharacter( sis );

	if  ( cc == EOF || cc == '\n' || cc == '\t' )
	    {
	    if  ( cc != EOF )
		{ sioInUngetLastRead( sis );	}

	    *pC= c; pTa->taFontIsBold= 0; pTa->taIsUnderlined= 0;
	    return 0;
	    }

	if  ( c == '_' )
	    {
	    *pC= cc; pTa->taFontIsBold= 0; pTa->taIsUnderlined= 1;
	    return 0;
	    }

	if  ( c == cc )
	    {
	    *pC= cc; pTa->taFontIsBold= 1; pTa->taIsUnderlined= 0;
	    return 0;
	    }

	*pC= cc; pTa->taFontIsBold= 0; pTa->taIsUnderlined= 0;
	return 0;
	}
    }

static int docPlainReadParagraph(	const BufferDocument *	bd,
					BufferItem *		cellBi,
					SimpleInputStream *	sis,
					PlainReadingContext *	prc )
    {
    int			c;
    BufferItem *	bi;
    TextAttribute	ta= prc->prcTextAttribute;

    int			res;

    res= docPlainReadParaCharacter( sis, &c, &ta );

    if  ( res < 0 )
	{ LDEB(res); return res;	}
    if  ( res > 0 )
	{ return res;	}

    bi= docInsertItem( bd, cellBi, -1, DOClevPARA );
    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    while( c != '\n' )
	{
	int		off= bi->biParaStrlen;
	TextAttribute	prevTa= ta;

	if  ( c == '\f' )
	    {
	    if  ( bi->biParaStrlen == 0 )
		{ bi->biParaStartsOnNewPage= 1;	}

	    res= docPlainReadParaCharacter( sis, &c, &ta );

	    if  ( res < 0 )
		{ LDEB(res); return res;	}
	    if  ( res > 0 )
		{ break;	}

	    continue;
	    }

	if  ( c == '\t' )
	    {
	    if  ( docInflateTextString( bi, 1 ) )
		{ LDEB(bi->biParaStrlen); return -1;	}

	    if  ( ! docInsertTextParticule( bi, -1, bi->biParaStrlen,
				    1, DOCkindTAB, prc->prcTextAttribute ) )
		{ LDEB(bi->biParaParticuleCount); return -1;	}

	    bi->biParaString[bi->biParaStrlen++]= ' ';
	    bi->biParaString[bi->biParaStrlen  ]= '\0';

	    res= docPlainReadParaCharacter( sis, &c, &ta );

	    if  ( res < 0 )
		{ LDEB(res); return res;	}
	    if  ( res > 0 )
		{ break;	}

	    continue;
	    }

	while( c != ' ' )
	    {
	    if  ( c != '\r' )
		{
		if  ( docInflateTextString( bi, 1 ) )
		    { LDEB(bi->biParaStrlen); return -1;	}

		bi->biParaString[bi->biParaStrlen++]= c;
		bi->biParaString[bi->biParaStrlen  ]= '\0';
		}

	    res= docPlainReadParaCharacter( sis, &c, &ta );
	    if  ( res )
		{ break;	}

	    if  ( c == '\n'					||
		  c == '\t'					||
		  ! docEqualTextAttributes( &ta, &prevTa )	)
		{ break;	}
	    }

	if  ( res < 0 )
	    { LDEB(res); return res;	}

	if  ( ! res && docEqualTextAttributes( &ta, &prevTa ) )
	    {
	    while( c == ' ' )
		{
		if  ( docInflateTextString( bi, 1 ) )
		    { LDEB(bi->biParaStrlen); return -1;	}

		bi->biParaString[bi->biParaStrlen++]= c;
		bi->biParaString[bi->biParaStrlen  ]= '\0';

		res= docPlainReadParaCharacter( sis, &c, &ta );
		if  ( res )
		    { break;	}

		if  ( ! docEqualTextAttributes( &ta, &prevTa )	)
		    { break;	}
		}
	    }

	if  ( res < 0 )
	    { LDEB(res); return res;	}

	if  ( ! docInsertTextParticule( bi, -1,
			off, bi->biParaStrlen- off, DOCkindTEXT, prevTa ) )
	    { LDEB(bi->biParaParticuleCount); return -1;	}

	if  ( res > 0 )
	    { break;	}
	}

    if  ( bi->biParaParticuleCount == 0 )
	{
	if  ( ! docInsertTextParticule( bi, 0, 0, 0,
				    DOCkindTEXT, prc->prcTextAttribute ) )
	    { LDEB( bi->biParaParticuleCount); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Read a plain text file.						*/
/*									*/
/*  1)  Assume 12 cpi and courier. ( 12 cpi .. 6pt wide .. 10 pt high. )*/
/*	But make it one less: 9 pt to match the rfc1234.txt make up.	*/
/*  2)  Tab every 8 characters. 12 cpi .. 8/12 inch= 960 twips.		*/
/*									*/
/************************************************************************/

BufferDocument * docPlainReadFile(	SimpleInputStream *		sis,
					const DocumentGeometry *	dg )
    {
    BufferDocument *	bd;
    BufferItem *	bi;

    PlainReadingContext	prc;

    bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docPlainInitReadingContext( &prc );
    prc.prcTextAttribute.taFontNumber= 0;
    prc.prcTextAttribute.taFontSizeHalfPoints= 18;
    bd->bdProperties.dpTabIntervalTwips= 960;

    docInitDocument( bd );
    bd->bdProperties.dpGeometry= *dg;

    if  ( ! docInsertFont( &(bd->bdProperties.dpFontList), 0,
						    "fmodern", "Courier" ) )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0;	}

    bi= docInsertItem( bd, &(bd->bdItem), -1, DOClevSECT );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;	}
    bi= docInsertItem( bd, bi, -1, DOClevROW );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;	}
    bi= docInsertItem( bd, bi, -1, DOClevCELL );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;	}

    for (;;)
	{
	int	res;

	res= docPlainReadParagraph( bd, bi, sis, &prc );

	if  ( res > 0 )
	    { break;	}

	if  ( res < 0 )
	    { LDEB(res); docFreeDocument( bd ); return (BufferDocument *)0; }
	}

    if  ( bi->biChildCount == 0 )
	{
	bi= docInsertItem( bd, bi, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0; }

	if  ( ! docInsertTextParticule( bi, 0, 0, 0,
					DOCkindTEXT, prc.prcTextAttribute ) )
	    { LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }
	}

    return bd;
    }
