/************************************************************************/
/*									*/
/*  Save a BufferDocument into an RTF file.				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<bitmap.h>

#   include	<appSystem.h>
#   include	<appWinMeta.h>

#   include	"docBuf.h"
#   include	<sioGeneral.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>
#   include	<appImage.h>

#   include	<appDebugon.h>

#   include	<charnames.h>

#   define	TWIPS_TO_SIZE(x)	(((x)+9)/18)

/************************************************************************/
/*									*/
/*  Information used when writing HTML.					*/
/*									*/
/************************************************************************/

#   define	HTMLmaskFONT		0x01
#   define	HTMLmaskSUPER		0x02
#   define	HTMLmaskSUB		0x04
#   define	HTMLmaskITALIC		0x08
#   define	HTMLmaskBOLD		0x10
#   define	HTMLmaskUNDERLINE	0x20

typedef struct HtmlPushedAttribute
    {
    struct HtmlPushedAttribute *	hpaPrevious;
    unsigned int			hpaChangeMask;
    TextAttribute			hpaAttributeNew;
    } HtmlPushedAttribute;

typedef struct HtmlWritingContext
    {
    const char *	hwcFilename;
    int			hwcBaselength;
    int			hwcRelativeOffset;
    int			hwcUseTableForFirstIndent;
    int			hwcInHyperlink;
    int			hwcInBookmark;
    int			hwcInPageref;
    int			hwcBytesInLink;
    TextAttribute	hwcDefaultAttribute;
    ParagraphProperties	hwcParagraphProperties;
    int			hwcMadeTransparentGif;
    } HtmlWritingContext;

static void docInitHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    hwc->hwcFilename= (const char *)0;
    hwc->hwcBaselength= 0;
    hwc->hwcRelativeOffset= 0;
    hwc->hwcUseTableForFirstIndent= 0;
    hwc->hwcInHyperlink= 0;
    hwc->hwcInBookmark= 0;
    hwc->hwcInPageref= 0;
    hwc->hwcBytesInLink= 0;

    docInitTextAttribute( &(hwc->hwcDefaultAttribute) );
    hwc->hwcDefaultAttribute.taFontSizeHalfPoints= 24;
    hwc->hwcDefaultAttribute.taFontNumber= -1;

    docInitParagraphProperties( &(hwc->hwcParagraphProperties) );

    hwc->hwcMadeTransparentGif= 0;

    return;
    }

static void docCleanHtmlWritingContext(	HtmlWritingContext *	hwc )
    {
    docCleanParagraphProperties( &(hwc->hwcParagraphProperties) );

    return;
    }

/************************************************************************/
/*									*/
/*  Save a tag with an argument.					*/
/*									*/
/************************************************************************/

static void docHtmlWriteArgTag(	const char *		tag,
				int			arg,
				SimpleOutputStream *	sos )
    {
    char	scratch[20];

    sioOutPutString( " ", sos );
    sioOutPutString( tag, sos );

    sprintf( scratch, "=%d", arg );
    sioOutPutString( scratch, sos );

    return;
    }

static void docHtmlEscapeCharacters(	const unsigned char *	s,
					int			len,
					int *			pCol,
					SimpleOutputStream *	sos )
    {
    int		col= *pCol;

    while( len > 0 )
	{
	switch( *s )
	    {
	    case '"':
		sioOutPutString( "&quot;", sos ); col += 6;
		break;
	    case '&':
		sioOutPutString( "&amp;", sos ); col += 5;
		break;
	    case '>':
		sioOutPutString( "&gt;", sos ); col += 4;
		break;
	    case '<':
		sioOutPutString( "&lt;", sos ); col += 4;
		break;
	    default:
		sioOutPutCharacter( *s, sos ); col += 1;
		break;
	    }

	s++; len--;
	}

    *pCol= col; return;
    }

static void docHtmlEscapeString(	const unsigned char *	s,
					int *			pCol,
					SimpleOutputStream *	sos )
    {
    docHtmlEscapeCharacters( s, strlen( (char *)s ), pCol, sos );
    return;
    }

static int docHtmlFontSize(	int	halfPoints )
    {
    if  ( halfPoints/2 < 6 )	{ return 1;	}
    if  ( halfPoints/2 < 9 )	{ return 2;	}

    if  ( halfPoints/2 > 24 )	{ return 7;	}
    if  ( halfPoints/2 > 19 )	{ return 6;	}
    if  ( halfPoints/2 > 15 )	{ return 5;	}
    if  ( halfPoints/2 > 11 )	{ return 4;	}

    return 3;
    }

static void docHtmlFinishGroup(		SimpleOutputStream *	sos,
					HtmlWritingContext *	hwc )
    {
    if  ( hwc->hwcUseTableForFirstIndent )
	{
	sioOutPutString( "</TABLE>\n", sos );
	hwc->hwcUseTableForFirstIndent= 0;
	}
    }

static void docHtmlPopAttributes(	SimpleOutputStream *	sos,
					int *			pCol,
					HtmlPushedAttribute *	hpaOld,
					HtmlPushedAttribute **	pHpaNew )
    {
    HtmlPushedAttribute *	hpaNew;
    int				madeTags= 0;

    if  ( ! hpaOld )
	{ XDEB(hpaOld); *pHpaNew= hpaOld; return;		}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskUNDERLINE ) )
	{ sioOutPutString( "</U>", sos ); madeTags= 1; *pCol += 4;	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskBOLD ) )
	{ sioOutPutString( "</B>", sos ); madeTags= 1; *pCol += 4;	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskITALIC ) )
	{ sioOutPutString( "</I>", sos ); madeTags= 1; *pCol += 4;	}

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskSUB ) )
	{ sioOutPutString( "</SMALL></SUB>", sos ); madeTags= 1; *pCol += 14; }

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskSUPER ) )
	{ sioOutPutString( "</SMALL></SUP>", sos ); madeTags= 1; *pCol += 14; }

    if  ( ( hpaOld->hpaChangeMask & HTMLmaskFONT ) )
	{ sioOutPutString( "</FONT>", sos ); madeTags= 1; *pCol += 7;	}

    hpaNew= hpaOld->hpaPrevious;

    free( hpaOld );

    *pHpaNew= hpaNew; return;
    }

/************************************************************************/
/*									*/
/*  Change attributes.							*/
/*									*/
/*  1)  Pop all attributes until nothing is to be turned off compared	*/
/*	to the old attributes. (And the font remains the same)		*/
/*									*/
/************************************************************************/

static int docHtmlChangeAttributes(	SimpleOutputStream *		sos,
					int *				pCol,
					const BufferDocument *		bd,
					HtmlPushedAttribute *		hpaOld,
					HtmlPushedAttribute **		pHpaNew,
					TextAttribute			taDflt,
					TextAttribute			taNew )
    {
    HtmlPushedAttribute *	hpaNew;

    int				col= *pCol;

    TextAttribute		taOld;

    /*  1  */
    while( hpaOld )
	{
	taOld= hpaOld->hpaAttributeNew;

	if  ( ( ! taOld.taFontIsSlanted			||
		taNew.taFontIsSlanted			)	&&

	      ( ! taOld.taFontIsBold			||
		taNew.taFontIsBold			)	&&

	      ( ! taOld.taIsUnderlined			||
		taNew.taIsUnderlined			)	&&

	      ( taOld.taSuperSub == DOCfontREGULAR	||
		taNew.taSuperSub != DOCfontREGULAR )		&&

	      taOld.taFontNumber ==
				taNew.taFontNumber		&&

	      taOld.taFontSizeHalfPoints ==
				taNew.taFontSizeHalfPoints	)
	    { break;	}

	docHtmlPopAttributes( sos, &col, hpaOld, &hpaOld );
	}

    if  ( hpaOld						&&
	  docEqualFont( &(hpaOld->hpaAttributeNew), &taNew )	)
	{ *pHpaNew= hpaOld; *pCol= col; return 0; }

    hpaNew= (HtmlPushedAttribute *)malloc( sizeof(HtmlPushedAttribute) );
    if  ( ! hpaNew )
	{ XDEB(hpaNew); return -1;	}

    hpaNew->hpaPrevious= hpaOld;
    hpaNew->hpaChangeMask= 0;
    if  ( hpaOld )
	{ taOld= hpaOld->hpaAttributeNew;	}
    else{ taOld= taDflt;			}
    hpaNew->hpaAttributeNew= taNew;

    if  ( taNew.taFontNumber != taOld.taFontNumber			||
	  taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints	)
	{
	const DocumentProperties *	dp= &(bd->bdProperties);
	const DocumentFontList *	dfl= &dp->dpFontList;
	const DocumentFont *		df= dfl->dflFonts+ taNew.taFontNumber;
	char *				font= (char *)0;

	sioOutPutString( "<FONT", sos );
	col += 5;

	if  ( taNew.taFontNumber != taOld.taFontNumber )
	    {
	    if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
		{ font= "Helvetica,Arial";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
		{ font= "Times,Times New Roman";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
		{ font= "Courier";	}
	    if  ( ! strcmp( df->dfFamilyStyle, "ftech" ) )
		{ font= "Symbol";	}

	    if  ( font )
		{
		sioOutPutString( " FACE=\"", sos );
		col += 6;
		sioOutPutString( font, sos );
		sioOutPutString( "\"", sos );
		col += strlen( font )+ 2;
		}
	    }

	if  ( taNew.taFontSizeHalfPoints != taOld.taFontSizeHalfPoints )
	    {
	    int		oldSize= docHtmlFontSize( taOld.taFontSizeHalfPoints );
	    int		newSize= docHtmlFontSize( taNew.taFontSizeHalfPoints );

	    if  ( newSize != oldSize )
		{
		docHtmlWriteArgTag( "SIZE", newSize, sos );
		col += 8;
		}
	    }

	hpaNew->hpaChangeMask |= HTMLmaskFONT;
	sioOutPutString( ">", sos );
	col++;
	}

    if  ( taNew.taSuperSub == DOCfontSUPERSCRIPT	&&
	  taOld.taSuperSub == DOCfontREGULAR		)
	{
	hpaNew->hpaChangeMask |= HTMLmaskSUPER;
	sioOutPutString( "<SUP><SMALL>", sos ); col += 5;
	}

    if  ( taNew.taSuperSub == DOCfontSUBSCRIPT	&&
	  taOld.taSuperSub == DOCfontREGULAR	)
	{
	hpaNew->hpaChangeMask |= HTMLmaskSUB;
	sioOutPutString( "<SUB><SMALL>", sos ); col += 5;
	}

    if  ( taNew.taFontIsSlanted && ! taOld.taFontIsSlanted )
	{
	hpaNew->hpaChangeMask |= HTMLmaskITALIC;
	sioOutPutString( "<I>", sos ); col += 3;
	}

    if  ( taNew.taFontIsBold && ! taOld.taFontIsBold )
	{
	hpaNew->hpaChangeMask |= HTMLmaskBOLD;
	sioOutPutString( "<B>", sos ); col += 3;
	}

    if  ( taNew.taIsUnderlined && ! taOld.taIsUnderlined )
	{
	hpaNew->hpaChangeMask |= HTMLmaskUNDERLINE;
	sioOutPutString( "<U>", sos ); col += 3;
	}

    *pCol= col; *pHpaNew= hpaNew; return 0;
    }

/************************************************************************/
/*									*/
/*  Save a completely transparent gif to use for positive First Indents	*/
/*									*/
/************************************************************************/

static int docHtmlSaveTransparentGif(	HtmlWritingContext *	hwc )
    {
    static char *	scratch;
    char *		fresh;
    AppBitmapImage *	abi;

    fresh= (char *)realloc( scratch, hwc->hwcBaselength+ 50 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;  }
    scratch= fresh;

    strncpy( scratch, hwc->hwcFilename, hwc->hwcBaselength );
    strcpy( scratch+ hwc->hwcBaselength, ".img" );

    if  ( appTestDirectory( scratch )	&&
	  appMakeDirectory( scratch )	)
	{ SDEB(scratch); return -1;	}

    sprintf( scratch, "%.*s.img/transp.gif",
				    hwc->hwcBaselength, hwc->hwcFilename );


    abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );
    if  ( ! abi )
	{ XDEB(abi); return -1;	}
    appInitBitmapImage( abi );

    if  ( bmTransparentImage( &(abi->abiBitmap), &(abi->abiBuffer),
						    BMcoRGB8PALETTE, 1, 1 ) )
	{ free(abi); return -1;	}

    if  ( bmWriteGifFile( scratch, abi->abiBuffer, &(abi->abiBitmap),
								89, 10.0 ) )
	{ appCleanBitmapImage( abi ); free( abi ); return -1;	}

    appCleanBitmapImage( abi ); free( abi ); return 0;
    }

static int docHtmlSaveImgTag(	const InsertedObject *		io,
				const BitmapDescription *	bd,
				SimpleOutputStream *		sos,
				const char *			filename )
    {
    int		w;
    int		h;
    int		d;

    sioOutPutString( "<IMG SRC=\"", sos );
    sioOutPutString( filename, sos );
    sioOutPutString( "\"", sos );

    w= TWIPS_TO_SIZE( ( io->ioScaleX* io->ioTwipsWide )/100 );
    h= TWIPS_TO_SIZE( ( io->ioScaleY* io->ioTwipsHigh )/100 );

    d= ( 100* bd->bdPixelsWide- 100* w )/ bd->bdPixelsWide;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ w= bd->bdPixelsWide;	}
    docHtmlWriteArgTag( "WIDTH", w, sos );

    d= ( 100* bd->bdPixelsHigh- 100* h )/ bd->bdPixelsHigh;
    if  ( d < 0 )
	{ d= -d;	}
    if  ( d <= 15 )
	{ h= bd->bdPixelsHigh;	}
    docHtmlWriteArgTag( "HEIGHT", h, sos );

    sioOutPutString( " ALT=\"&lt;IMG&gt;\">\n", sos );

    return 0;
    }

static int docHtmlGetBitmap(	bmReadBitmap		readBitmap,
				InsertedObject *	io,
				ObjectData *		od )
    {
    AppBitmapImage *	abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );

    SimpleInputStream *	sisMem;
    SimpleInputStream *	sisBitmap;

    MemoryBuffer	mb;

    mb.mbCapacity= od->odSize;
    mb.mbBytes= od->odBytes;

    if  ( ! abi )
	{ XDEB(abi); return -1;	}
    appInitBitmapImage( abi );

    sisMem= sioInMemoryOpen( &mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisBitmap= sioInHexOpen( sisMem );
    if  ( ! sisBitmap )
	{ XDEB(sisMem); return -1;	}

    if  ( (*readBitmap)( &abi->abiBitmap, &abi->abiBuffer, sisBitmap ) )
	{ LDEB(1); return -1;	}

    sioInClose( sisBitmap );
    sioInClose( sisMem );

    io->ioPrivate= (void *)abi;

    return 0;
    }

static int docHtmlGetBitmapForObject(	InsertedObject *	io )
    {
    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    if  ( ! io->ioPrivate					&&
		  docHtmlGetBitmap( appMetaPlayWmfImg,
						io, &io->ioObjectData )	)
		{ LDEB(1); return 0;	}
	    break;

	case DOCokPICTPNGBLIP:
	    if  ( ! io->ioPrivate					&&
		  docHtmlGetBitmap( bmPngReadPng,
					    io, &io->ioObjectData )	)
		{ LDEB(1); return 0;	}
	    break;

	case DOCokPICTJPEGBLIP:
	    if  ( ! io->ioPrivate					&&
		  docHtmlGetBitmap( bmJpegReadJfif,
					    io, &io->ioObjectData )	)
		{ LDEB(1); return 0;	}

	    break;

	case DOCokOLEOBJECT:
	    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
		{
		if  ( ! io->ioPrivate					&&
		      docHtmlGetBitmap( bmJpegReadJfif,
					    io, &io->ioResultData )	)
		    { LDEB(1); return 0;	}

		return 0;
		}
	    return 0;

	case DOCokMACPICT:
	case DOCokPICTEMFBLIP:
	default:
	    LDEB(io->ioKind); return 0;
	}

    return 0;
    }

static int docHtmlSavePicture(	InsertedObject *	io,
				SimpleOutputStream *	sos,
				int *			pDone,
				HtmlWritingContext *	hwc )
    {
    static char *	scratch;
    char *		fresh;
    AppBitmapImage *	abi;

    if  ( ! hwc->hwcFilename )
	{ return 0;	}

    if  ( ! io->ioPrivate )
	{
	if  ( docHtmlGetBitmapForObject( io ) )
	    { LDEB(1); return 0;	}
	}

    abi= (AppBitmapImage *)io->ioPrivate;
    if  ( ! abi )
	{ return 0;	}

    if  ( bmCanWriteGifFile( &abi->abiBitmap, 89, 10.0 )	&&
	  bmCanWriteJpegFile( &abi->abiBitmap, 0, 10.0 )	)
	{ return 0;	}

    fresh= (char *)realloc( scratch, hwc->hwcBaselength+ 50 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;  }
    scratch= fresh;

    strncpy( scratch, hwc->hwcFilename, hwc->hwcBaselength );
    strcpy( scratch+ hwc->hwcBaselength, ".img" );

    if  ( appTestDirectory( scratch )	&&
	  appMakeDirectory( scratch )	)
	{ SDEB(scratch); return -1;	}

    if  ( io->ioBliptag == 0 )
	{ io->ioBliptag= appGetTimestamp();	}

    if  ( ! bmCanWriteGifFile( &abi->abiBitmap, 89, 10.0 ) )
	{
	sprintf( scratch, "%.*s.img/%08x.gif",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );

	if  ( bmWriteGifFile( scratch, abi->abiBuffer, &abi->abiBitmap,
								89, 10.0 ) )
	    { SDEB(scratch); return -1;	}

	if  ( docHtmlSaveImgTag( io, &abi->abiBitmap, sos,
					    scratch+ hwc->hwcRelativeOffset ) )
	    { SDEB(scratch); return -1;	}

	*pDone= 1; return 0;
	}

    if  ( ! bmCanWriteJpegFile( &abi->abiBitmap, 89, 10.0 ) )
	{
	sprintf( scratch, "%.*s.img/%08x.jpg",
			hwc->hwcBaselength, hwc->hwcFilename, io->ioBliptag );

	if  ( bmWriteJpegFile( scratch, abi->abiBuffer, &abi->abiBitmap,
								89, 10.0 ) )
	    { SDEB(scratch); return -1;	}

	if  ( docHtmlSaveImgTag( io, &abi->abiBitmap, sos,
					    scratch+ hwc->hwcRelativeOffset ) )
	    { SDEB(scratch); return -1;	}

	*pDone= 1; return 0;
	}

    return 0;
    }

static int docHtmlStartHyperlink( SimpleOutputStream *		sos,
				int *				pCol,
				const char *			fileName,
				int				fileSize,
				const char *			markName,
				int				markSize )
    {
    int			col= *pCol;

    sioOutPutString( "<A HREF=\"", sos ); col += 8;

    if  ( fileName )
	{
	while( fileSize > 0 )
	    {
	    sioOutPutCharacter( *fileName, sos );
	    fileName++; fileSize--; col++;
	    }
	}

    if  ( markName && markSize > 0 )
	{
	sioOutPutCharacter( '#', sos );

	while( markSize > 0 )
	    {
	    sioOutPutCharacter( *markName, sos );
	    markName++; markSize--; col++;
	    }
	}

    sioOutPutString( "\">", sos ); col++;

    *pCol= col; return 0;
    }

static int docHtmlStartBookmark( SimpleOutputStream *		sos,
				int *				pCol,
				const char *			markName,
				int				markSize )
    {
    int			col= *pCol;

    sioOutPutString( "<A NAME=\"", sos ); col += 8;

    if  ( markName )
	{
	while( markSize > 0 )
	    {
	    sioOutPutCharacter( *markName, sos );
	    markName++; markSize--; col++;
	    }
	}

    sioOutPutString( "\">", sos ); col++;

    *pCol= col; return 0;
    }

static int docHtmlStartField(	SimpleOutputStream *		sos,
				const DocumentField *		df,
				int *				pCol,
				HtmlWritingContext *		hwc,
				const BufferDocument *		bd,
				HtmlPushedAttribute *		hpaOld,
				HtmlPushedAttribute **		pHpaNew )
    {
    int			col= *pCol;

    const char *	fileName= (const char *)0;
    int			fileSize= 0;
    const char *	markName= (const char *)0;
    int			markSize= 0;

    switch( df->dfKind )
	{
	case DOCfkHYPERLINK:
	    hwc->hwcInHyperlink++;
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		if  ( ! docFieldGetHyperlink( df,
				&fileName, &fileSize, &markName, &markSize ) )
		    {
		    if  ( docHtmlChangeAttributes( sos, &col, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
			{ LDEB(1); return -1;	}

		    docHtmlStartHyperlink( sos, &col,
				    fileName, fileSize, markName, markSize );

		    hwc->hwcBytesInLink= 0;
		    }
		}
	    break;

	case DOCfkBOOKMARK:
	    hwc->hwcInBookmark++;
	    if  ( ! hwc->hwcInHyperlink && hwc->hwcInBookmark == 1 )
		{
		if  ( ! docFieldGetBookmark( df, &markName, &markSize ) )
		    {
		    if  ( docHtmlChangeAttributes( sos, &col, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
			{ LDEB(1); return -1;	}

		    docHtmlStartBookmark( sos, &col, markName, markSize );
		    }
		}
	    break;

	case DOCfkPAGEREF:
	    hwc->hwcInPageref++;
	    break;

	default:
	    break;
	}

    *pCol= col; return 0;
    }

static int docHtmlFinishField(	SimpleOutputStream *		sos,
				const DocumentField *		df,
				int *				pCol,
				HtmlWritingContext *		hwc,
				const BufferDocument *		bd,
				HtmlPushedAttribute *		hpaOld,
				HtmlPushedAttribute **		pHpaNew )
    {
    int			col= *pCol;

    switch( df->dfKind )
	{
	case DOCfkHYPERLINK:
	    if  ( ! hwc->hwcInBookmark && hwc->hwcInHyperlink == 1 )
		{
		if  ( docHtmlChangeAttributes( sos, &col, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
		    { LDEB(1); return -1;	}

		sioOutPutString( "</A>", sos ); col += 4;
		}
	    hwc->hwcInHyperlink--;
	    break;

	case DOCfkBOOKMARK:
	    if  ( ! hwc->hwcInHyperlink && hwc->hwcInBookmark == 1 )
		{
		if  ( docHtmlChangeAttributes( sos, &col, bd,
					    hpaOld, pHpaNew,
					    hwc->hwcDefaultAttribute,
					    hwc->hwcDefaultAttribute ) )
		    { LDEB(1); return -1;	}

		sioOutPutString( "</A>", sos ); col += 4;
		}
	    hwc->hwcInBookmark--;
	    break;

	case DOCfkPAGEREF:
	    hwc->hwcInPageref--;
	    break;

	default:
	    break;
	}
    
    *pCol= col; return 0;
    return 0;
    }

static int docHtmlSaveParticules( SimpleOutputStream *		sos,
				int *				pCol,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				int				part,
				int				partUpto,
				HtmlWritingContext *		hwc )
    {
    int					done= 0;

    int					col= *pCol;
    int					afterSpace= 0;

    TextParticule *			tp= bi->biParaParticules+ part;
    int					stroff= tp->tpStroff;
    unsigned char *			s= bi->biParaString+ stroff;

    int					pictureDone;
    InsertedObject *			io;

    HtmlPushedAttribute *		hpa= (HtmlPushedAttribute *)0;

    const DocumentField *		df;
    const FieldKindInformation *	fki;

    int					len;

    while( part < partUpto )
	{
	switch( tp->tpKind )
	    {
	    case DOCkindTAB:
		sioOutPutString( " ", sos );
		col += 1; afterSpace= 1;
		s++; stroff++;
		break;

	    case DOCkindTEXT:
		if  ( docHtmlChangeAttributes( sos, &col, bd, hpa, &hpa,
			    hwc->hwcDefaultAttribute, tp->tpTextAttribute ) )
		    { LDEB(1); return -1;	}

		if  ( afterSpace && col+ tp->tpStrlen > 72 )
		    { sioOutPutString( "\n", sos ); col= 0; }

		len= tp->tpStroff+ tp->tpStrlen- stroff;

		if  ( ! hwc->hwcInHyperlink	||
		      ! hwc->hwcInPageref	||
		      hwc->hwcBytesInLink == 0	)
		    {
		    docHtmlEscapeCharacters( s, len, &col, sos );

		    if  ( hwc->hwcInHyperlink )
			{ hwc->hwcBytesInLink += len;	}
		    }

		s += len;
		stroff += len;

		afterSpace= 0;
		if  ( tp->tpStrlen > 0 && s[-1] == ' ' )
		    { afterSpace= 1;	}
		break;

	    case DOCkindOBJECT:
		pictureDone= 0;
		io= bi->biParaObjects+ tp->tpObjectNumber;

		if  (   io->ioKind == DOCokPICTWMETAFILE		||
			io->ioKind == DOCokPICTPNGBLIP			||
			io->ioKind == DOCokPICTJPEGBLIP			||
			io->ioKind == DOCokMACPICT			||
		      ( io->ioKind == DOCokOLEOBJECT 		&&
		        io->ioResultKind == DOCokPICTWMETAFILE	)	)
		    {
		    if  ( docHtmlSavePicture( io, sos, &pictureDone, hwc ) )
			{ XDEB(io);	}
		    }

		if  ( ! pictureDone )
		    { sioOutPutString( " ", sos ); col += 1;	}
		else{ col= 0;					}
		afterSpace= 0; s++; stroff++;
		break;

	    case DOCkindFIELDSTART:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    { docHtmlStartField( sos, df, &col, hwc, bd, hpa, &hpa ); }

		if  ( df->dfKind == DOCfkBOOKMARK )
		    { docHtmlStartField( sos, df, &col, hwc, bd, hpa, &hpa ); }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;

	    case DOCkindFIELDEND:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( df->dfKind == DOCfkBOOKMARK )
		    {
		    docHtmlFinishField( sos, df, &col, hwc, bd, hpa, &hpa );
		    }

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    {
		    docHtmlFinishField( sos, df, &col, hwc, bd, hpa, &hpa );
		    }

		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		done= 1;
		break;

	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
		sioOutPutString( "<BR>\r\n", sos );
		col= 0; afterSpace= 0;
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 1 */
		break;

	    case DOCkindXE:
	    case DOCkindTC:
		s += tp->tpStrlen; stroff += tp->tpStrlen; /* += 0 */
		break;

	    default:
		LDEB(tp->tpKind);
		s += tp->tpStrlen; stroff += tp->tpStrlen;
		break;
	    }

	done++; part++; tp++;
	}

    while( hpa )
	{ docHtmlPopAttributes( sos, &col, hpa, &hpa ); }

    *pCol= col; return done;
    }

/************************************************************************/
/*									*/
/*  The body of a paragraph is written as a <DIV> or <TD> division.	*/
/*									*/
/************************************************************************/

static void docHtmlStartParagraphBody(	const BufferItem *	bi,
					const char *		tag,
					int *			pCol,
					SimpleOutputStream *	sos )
    {
    int		col= *pCol;

    sioOutPutCharacter( '<', sos ); col++;
    sioOutPutString( tag, sos ); col += strlen( tag );

    switch( bi->biParaAlignment )
	{
	case DOCiaLEFT:
	    break;
	case DOCiaRIGHT:
	    sioOutPutString( " ALIGN=\"RIGHT\"", sos );
	    col += 14;
	    break;
	case DOCiaCENTERED:
	    sioOutPutString( " ALIGN=\"CENTER\"", sos );
	    col += 15;
	    break;
	case DOCiaJUSTIFIED:
	    sioOutPutString( " ALIGN=\"JUSTIFY\"", sos );
	    col += 16;
	    break;
	default:
	    LDEB(bi->biParaAlignment);
	    break;
	}

    sioOutPutString( ">", sos ); col++;

    *pCol= col;
    }

/************************************************************************/
/*									*/
/*  Use a HTML table for the implementation of the 'First Indent' of an	*/
/*  RTF paragraph?							*/
/*									*/
/************************************************************************/

static int docHtmlUseTableForIndent(	int *			pTabParticule,
					const BufferDocument *	bd,
					const BufferItem *	bi )
    {
    int			stroff;
    int			part;
    TextParticule *	tp;

    if  ( bi->biParaAlignment != DOCiaLEFT	&&
	  bi->biParaAlignment != DOCiaJUSTIFIED	)
	{ return 0;	}

    if  ( bi->biParaLeftIndentTwips <= 0 )
	{ return 0;	}
    if  ( bi->biParaFirstIndentTwips >= 0 )
	{ return 0;	}
    if  ( bi->biParaFirstIndentTwips < -bi->biParaLeftIndentTwips )
	{ return 0;	}

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	switch( tp->tpKind )
	    {
	    const DocumentField *		df;
	    const FieldKindInformation *	fki;

	    case DOCkindTAB:
		*pTabParticule= part;
		return 1;

	    case DOCkindFIELDSTART:
	    case DOCkindFIELDEND:
		df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
		fki= DOC_FieldKinds+ df->dfKind;

		if  ( fki->fkiIsFieldInRtf		&&
		      fki->fkiLevel == DOClevTEXT	)
		    { return 0;	}

		if  ( df->dfKind == DOCfkBOOKMARK )
		    { return 0;	}

		break;

	    default:
		break;
	    }

	stroff= tp->tpStroff+ tp->tpStrlen;
	if  ( stroff > 10 )
	    { return 0;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save a 'Paragraph'							*/
/*									*/
/*  But not as a <P>...</P>.						*/
/*									*/
/************************************************************************/

static int docHtmlSaveParaItem(	SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				HtmlWritingContext *		hwc )
    {
    TextParticule *			tp;
    unsigned char *			s;

    int					useTABLE= 0;

    int					part= 0;
    int					stroff= 0;
    int					tabParticule;

    int					col= 0;

    int					fontHeight;

    if  ( bi->biParaParticuleCount == 0		||
	  bi->biParaStrlen == 0			)
	{
	if  ( hwc->hwcUseTableForFirstIndent )
	    {
	    sioOutPutString( "</TABLE>&nbsp;<BR>\n", sos );
	    hwc->hwcUseTableForFirstIndent= 0;
	    }
	else{ sioOutPutString( "<DIV>&nbsp;</DIV>\n", sos ); }


	return 0;
	}

    useTABLE= docHtmlUseTableForIndent( &tabParticule, bd, bi );

    part= 0;
    stroff= 0;
    tp= bi->biParaParticules+ part;
    s= bi->biParaString+ stroff;
    fontHeight= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

    if  ( ! useTABLE && hwc->hwcUseTableForFirstIndent )
	{
	hwc->hwcUseTableForFirstIndent= useTABLE;
	sioOutPutString( "</TABLE>\n", sos );
	}

    if  ( useTABLE )
	{
	int	left= TWIPS_TO_SIZE( bi->biParaLeftIndentTwips );

	if  ( ! hwc->hwcUseTableForFirstIndent )
	    {
	    sioOutPutString(
		"<TABLE CELLPADDING=0 CELLSPACING=0><TR VALIGN=\"TOP\">",
									sos );
	    col= 36;
	    }
	else{
	    sioOutPutString( "<TR VALIGN=\"TOP\">", sos );
	    col= 20;
	    }

	sioOutPutString( "<TD", sos );
	docHtmlWriteArgTag( "WIDTH", left, sos );

	if  ( ! hwc->hwcUseTableForFirstIndent )
	    { sioOutPutString( ">\n", sos ); col= 0;	}
	else{ sioOutPutString( ">", sos ); col += 12;	}

	if  ( bi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	    { sioOutPutString( "&nbsp;<BR>\n", sos );	}

	if  ( docHtmlSaveParticules( sos, &col, bd, bi,
					    part, tabParticule, hwc ) < 0 )
	    { LDEB(part); return -1;	}

	sioOutPutString( "</TD><TD>", sos );

	hwc->hwcUseTableForFirstIndent= useTABLE;

	docHtmlStartParagraphBody( bi, "TD", &col, sos );

	if  ( bi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	    { sioOutPutString( "&nbsp;<BR>\n", sos );	}

	part= tabParticule+ 1;
	}
    else{
	docHtmlStartParagraphBody( bi, "DIV", &col, sos );

	if  ( bi->biParaSpaceBeforeTwips > fontHeight/ 2 )
	    { sioOutPutString( "&nbsp;<BR>\n", sos );	}
	}

    if  ( bi->biParaTopBorder.bpIsSet )
	{ sioOutPutString( "<HR NOSHADE WIDTH=100%>\n", sos ); }

    if  ( ! useTABLE && bi->biParaFirstIndentTwips > 40 )
	{
	if  ( docHtmlSaveTransparentGif( hwc ) )
	    { LDEB(bi->biParaFirstIndentTwips); return -1;	}

	sioOutPutString( "<IMG SRC=\"", sos );
	sioOutWriteBytes( sos,
		(const unsigned char *)hwc->hwcFilename, hwc->hwcBaselength );
	sioOutPutString( ".img/transp.gif", sos );
	sioOutPutString( "\"", sos );
	docHtmlWriteArgTag( "WIDTH",
			TWIPS_TO_SIZE( bi->biParaFirstIndentTwips ), sos );
	docHtmlWriteArgTag( "HEIGHT", 1, sos );
	sioOutPutString( ">\n", sos );
	col= 0;
	}

    if  ( docHtmlSaveParticules( sos, &col, bd, bi,
				part, bi->biParaParticuleCount, hwc ) < 0 )
	{ LDEB(part); return -1;	}

    if  ( bi->biParaBottomBorder.bpIsSet )
	{ sioOutPutString( "<HR NOSHADE WIDTH=100%>\n", sos ); }

    if  ( hwc->hwcUseTableForFirstIndent )
	{ sioOutPutString( "</TD></TR>\n", sos ); }
    else{
	if  ( bi->biParaSpaceAfterTwips > fontHeight/ 2 )
	    { sioOutPutString( "<BR>&nbsp;</DIV>\n", sos );	}
	else{ sioOutPutString( "</DIV>\n", sos );		}
	}

    if  ( docCopyParagraphProperties( &(hwc->hwcParagraphProperties),
						&(bi->biParaProperties) ) )
	{ LDEB(1);	}

    return 0;
    }

static int docHtmlSaveRowItem(	SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const BufferItem *		row,
				HtmlWritingContext *		hwc )
    {
    int			i;

    if  ( ! hwc->hwcParagraphProperties.ppInTable )
	{
	int			cellPadding;
	int			useBorder= 0;
	CellProperties *	cp;

	cp= row->biRowCells;
	for ( i= 0; i < row->biChildCount; cp++, i++ )
	    {
	    if  ( cp->cpTopBorder.bpIsSet	||
		  cp->cpLeftBorder.bpIsSet	||
		  cp->cpRightBorder.bpIsSet	||
		  cp->cpBottomBorder.bpIsSet	)
		{ useBorder= 1; break;	}
	    }

	sioOutPutString( "<TABLE CELLSPACING=0", sos );

	if  ( useBorder )
	    { sioOutPutString( " BORDER=\"1\"", sos ); }

	cellPadding= TWIPS_TO_SIZE( row->biRowHalfGapWidthTwips )- 4;
	if  ( cellPadding < 1 )
	    { docHtmlWriteArgTag( "CELLPADDING", 0, sos ); }
	if  ( cellPadding > 1 )
	    { docHtmlWriteArgTag( "CELLPADDING", cellPadding, sos ); }

	sioOutPutString( "><TR VALIGN=\"TOP\">", sos );
	hwc->hwcParagraphProperties.ppInTable= 1;
	}
    else{
	sioOutPutString( "<TR VALIGN=\"TOP\">", sos );
	}

    for ( i= 0; i < row->biChildCount; i++ )
	{
	int		j;
	BufferItem *	cellBi= row->biChildren[i];

	FormattingFrame	ff;

	docParagraphFrameTwips( &ff, -1, -1, cellBi->biChildren[0] );

	sioOutPutString( "<TD", sos );
	docHtmlWriteArgTag( "WIDTH",
		    TWIPS_TO_SIZE( ff.ffX1Geometry- ff.ffX0Geometry ), sos );
	sioOutPutString( ">\n", sos );

	for ( j= 0; j < cellBi->biChildCount; j++ )
	    {
	    BufferItem *	para= cellBi->biChildren[j];

	    if  ( docHtmlSaveParaItem( sos, bd, para, hwc ) )
		{ LDEB(1); return -1;	}
	    }

	docHtmlFinishGroup( sos, hwc );

	if  ( i < row->biChildCount- 1 )
	    { sioOutPutString( "</TD>\n", sos );	}
	else{ sioOutPutString( "</TD>", sos );		}
	}

    sioOutPutString( "</TR>\n", sos );

    return 0;
    }

static int docHtmlSaveItem(	SimpleOutputStream *		sos,
				const BufferDocument *		bd,
				const BufferItem *		bi,
				HtmlWritingContext *		hwc )
    {
    int		i;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevDOC:
	case DOClevCELL:
	rowAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docHtmlSaveItem( sos, bd,
					    bi->biChildren[i], hwc ) )
		    { LDEB(i); return -1;	}
		}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{
		if  ( hwc->hwcParagraphProperties.ppInTable )
		    {
		    sioOutPutString( "</TABLE>\n", sos );
		    hwc->hwcParagraphProperties.ppInTable= 0;
		    }

		goto rowAsGroup;
		}

	    if  ( docHtmlSaveRowItem( sos, bd, bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    if  ( docHtmlSaveParaItem( sos, bd, bi, hwc ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }

int docHtmlSaveDocument(	SimpleOutputStream *	sos,
				const BufferDocument *	bd,
				const char *		filename )
    {
    HtmlWritingContext		hwc;

    const BufferItem *		bi= &bd->bdItem;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg= &(dp->dpGeometry);

#   if 0
    const DocumentFontList *	dfl= &bd->bdFontList;
    const DocumentFont *	df;
    int				i;
#   endif

    docInitHtmlWritingContext( &hwc );

    if  ( filename )
	{
	const char *	s;

	hwc.hwcFilename= filename;
	s= strrchr( filename, '.' );
	if  ( s )
	    { hwc.hwcBaselength= s- filename;		}
	else{ hwc.hwcBaselength= strlen( filename );	}

	s= strrchr( filename, '/' );
	if  ( s )
	    { hwc.hwcRelativeOffset= s- filename+ 1;	}
	else{ hwc.hwcRelativeOffset= 0;			}
	}

    sioOutPutString( "<HTML>\n", sos );

#   if 0
    sioOutPutString( "<STYLE>\n", sos );

    df= dfl->dflFonts;
    for ( i= 0; i < dfl->dflCount; df++, i++ )
	{
	char *			font= (char *)0;
	char			scratch[15];

	sprintf( scratch, "%d", i );

	if  ( ! strcmp( df->dfFamilyStyle, "fswiss" ) )
	    { font= "Helvetica,Arial,sans-serif";	}
	if  ( ! strcmp( df->dfFamilyStyle, "froman" ) )
	    { font= "Times,Times New Roman,serif";	}
	if  ( ! strcmp( df->dfFamilyStyle, "fmodern" ) )
	    { font= "Courier,monospace";	}
	if  ( ! strcmp( df->dfFamilyStyle, "ftech" ) )
	    { font= "Symbol";	}

	sioOutPutString( "FONT.f", sos );
	sioOutPutString( scratch, sos );
	sioOutPutString( " {", sos );
	if  ( font )
	    {
	    sioOutPutString( "font-family: ", sos );
	    sioOutPutString( font, sos );
	    }

	sioOutPutString( " }\n", sos );
	}

    sioOutPutString( "</STYLE>\n", sos );
#   endif

    if  ( docHasDocumentInfo( dp ) )
	{
	int		col= 0;

	sioOutPutString( "<HEAD>\n", sos );

	if  ( dp->dpTitle )
	    {
	    sioOutPutString( "<TITLE>", sos );
	    docHtmlEscapeString( dp->dpTitle, &col, sos );
	    sioOutPutString( "</TITLE>\n", sos );
	    }

	if  ( dp->dpSubject )
	    {
	    sioOutPutString( "<META NAME=\"description\" CONTENT=\"", sos );
	    docHtmlEscapeString( dp->dpSubject, &col, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( dp->dpKeywords )
	    {
	    sioOutPutString( "<META NAME=\"keywords\" CONTENT=\"", sos );
	    docHtmlEscapeString( dp->dpKeywords, &col, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( dp->dpComment )
	    {
	    sioOutPutString( "<META NAME=\"comment\" CONTENT=\"", sos );
	    docHtmlEscapeString( dp->dpComment, &col, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( dp->dpAuthor )
	    {
	    sioOutPutString( "<META NAME=\"author\" CONTENT=\"", sos );
	    docHtmlEscapeString( dp->dpAuthor, &col, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	if  ( dp->dpHlinkbase )
	    {
	    sioOutPutString( "<BASE HREF=\"", sos );
	    docHtmlEscapeString( dp->dpHlinkbase, &col, sos );
	    sioOutPutString( "\">\n", sos );
	    }

	sioOutPutString( "</HEAD>\n", sos );
	}

    sioOutPutString( "<BODY BGCOLOR=\"#ffffff\" TEXT=\"#000000\">\n", sos );

    if  ( dg->dgLeftMarginTwips > 300		||
	  dg->dgRightMarginTwips > 300		||
	  dg->dgTopMarginTwips > 300		||
	  dg->dgBottomMarginTwips > 300		)
	{
	sioOutPutString( "<TABLE>\n", sos );

	if  ( dg->dgTopMarginTwips > 300 )
	    {
	    sioOutPutString( "<TR><TD", sos );

	    docHtmlWriteArgTag( "HEIGHT",
		TWIPS_TO_SIZE( dg->dgTopMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD></TR>", sos );
	    }

	sioOutPutString( "<TR>", sos );

	if  ( dg->dgLeftMarginTwips > 300 )
	    {
	    sioOutPutString( "<TD", sos );

	    docHtmlWriteArgTag( "WIDTH",
		TWIPS_TO_SIZE( dg->dgLeftMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD>", sos );
	    }

	sioOutPutString( "<TD>\n", sos );
	}

    if  ( docHtmlSaveItem( sos, bd, bi, &hwc ) )
	{ LDEB(bi->biLevel); return -1; }

    docHtmlFinishGroup( sos, &hwc );

    if  ( dg->dgLeftMarginTwips > 300		||
	  dg->dgRightMarginTwips > 300		||
	  dg->dgTopMarginTwips > 300		||
	  dg->dgBottomMarginTwips > 300		)
	{
	sioOutPutString( "</TD>", sos );

	if  ( dg->dgRightMarginTwips > 300 )
	    {
	    sioOutPutString( "<TD", sos );

	    docHtmlWriteArgTag( "WIDTH",
		TWIPS_TO_SIZE( dg->dgRightMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD>", sos );
	    }

	sioOutPutString( "</TR>", sos );

	if  ( dg->dgBottomMarginTwips > 300 )
	    {
	    sioOutPutString( "<TR><TD", sos );

	    docHtmlWriteArgTag( "HEIGHT",
		TWIPS_TO_SIZE( dg->dgBottomMarginTwips ), sos );

	    sioOutPutString( ">&nbsp;</TD></TR>", sos );
	    }

	sioOutPutString( "</TABLE>\n", sos );
	}

    sioOutPutString( "</BODY></HTML>\n", sos );

    docCleanHtmlWritingContext( &hwc );

    return 0;
    }
