/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<limits.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>
#   include	"docBuf.h"

/************************************************************************/
/*									*/
/*  Manage and evaluate fields.						*/
/*									*/
/************************************************************************/

#   define	FIC_COUNT	15

/************************************************************************/
/*									*/
/*  Format a page number to be used in a certain paragraph.		*/
/*									*/
/************************************************************************/

static int docGetPageNumberOffset(	const BufferItem *	sectBi )
    {
    if  ( ! sectBi->biParent )
	{ XDEB(sectBi->biParent); return 0;	}

    while( sectBi->biNumberInParent > 0 )
	{
	if  ( sectBi->biSectRestartPageNumbers )
	    { break;	}

	sectBi= sectBi->biParent->biChildren[sectBi->biNumberInParent- 1];
	}

    if  ( sectBi->biSectRestartPageNumbers )
	{
	return sectBi->biTopPosition.lpPage- sectBi->biSectStartPageNumber;
	}
    else{ return sectBi->biTopPosition.lpPage; }
    }

static void docFormatPageNumber(	char *			target,
					int			targetSize,
					const BufferItem *	sectBi,
					int			pageNumber )
    {
    int			style= sectBi->biSectPageNumberStyle;

    if  ( targetSize < 20 )
	{ LDEB(targetSize); strcpy( target, "?" ); return;	}

    pageNumber -= docGetPageNumberOffset( sectBi );

    switch( style )
	{
	default:
	    LDEB(style);
	    /*FALLTHROUGH*/

	case DOCpgnDEC:
	    sprintf( target, "%d", pageNumber+ 1 );
	    break;

	case DOCpgnUCLTR:
	    if  ( pageNumber >= 0 && pageNumber <= 26 )
		{ target[0]= 'A'+ pageNumber; target[1]= '\0';	}
	    else{ sprintf( target, "UCLTR:%d", pageNumber+ 1 );	}
	    break;

	case DOCpgnLCLTR:
	    if  ( pageNumber >= 0 && pageNumber <= 26 )
		{ target[0]= 'a'+ pageNumber; target[1]= '\0';	}
	    else{ sprintf( target, "lcltr:%d", pageNumber+ 1 );	}
	    break;

	case DOCpgnUCRM:
	    if  ( appRomanString( target, targetSize, pageNumber+ 1, 1 ) )
		{ sprintf( target, "UCRM:%d", pageNumber+ 1 );	}
	    break;

	case DOCpgnLCRM:
	    if  ( appRomanString( target, targetSize, pageNumber+ 1, 0 ) )
		{ sprintf( target, "lcrm:%d", pageNumber+ 1 );	}
	    break;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Substitute the text fields in a buffer item.			*/
/*									*/
/************************************************************************/

int docRecalculateTextFields(	int *			pChanged,
				BufferDocument *	bd,
				BufferItem *		bi,
				unsigned int		whenMask )
    {
    char		scratch[500+ 1];
    int			i;

    int			stroffShift;
    int			part;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docRecalculateTextFields( pChanged, bd,
					    bi->biChildren[i], whenMask ) )
		    { LDEB(i); return -1;	}
		}

	    return 0;
	case DOClevPARA:
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}

    stroffShift= 0;
    for ( part= 0; part < bi->biParaParticuleCount; part++ )
	{
	TextParticule *			tp= bi->biParaParticules+ part;

	DocumentField *			df;
	const FieldKindInformation *	fki;
	int				d;

	int				calculated;
	int				newSize;
	int				oldSize;

	int				partCount;
	int				partsDone;
	int				past;
	int				stroff;

	TextAttribute			ta;

	tp->tpStroff += stroffShift;

	if  ( tp->tpKind != DOCkindFIELDSTART )
	    { continue;	}

	df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;
	if  ( df->dfKind < 0 || df->dfKind >= DOC_FieldKindCount )
	    { LLDEB(df->dfKind,DOC_FieldKindCount); continue;	}
	fki= DOC_FieldKinds+ df->dfKind;
	if  ( fki->fkiLevel != DOClevTEXT		||
	      ! ( fki->fkiCalculateWhen & whenMask )	)
	    { continue;	}

	partCount= 0;
	while( part+ 1+ partCount < bi->biParaParticuleCount	&&
	       tp[1+ partCount].tpKind == DOCkindTEXT		)
	    { partCount++;	}

	if  ( part+ 1+ partCount >= bi->biParaParticuleCount )
	    { LLDEB(part+ 1+ partCount,bi->biParaParticuleCount); continue; }
	if  ( tp[1+partCount].tpKind != DOCkindFIELDEND	)
	    { LDEB(1); continue;	}

	if  ( (*fki->fkiCalculateText)( &calculated, &newSize, scratch,
					    sizeof(scratch)- 1, bd, bi, df ) )
	    { SDEB(fki->fkiLabel); return -1;	}

	if  ( ! calculated )
	    { continue;	}
	if  ( tp->tpStrlen == newSize					&&
	      ! memcmp( bi->biParaString+ tp->tpStroff+ stroffShift,
						    scratch, newSize )	)
	    { continue;	}

	*pChanged= 1;

	ta= tp[1].tpTextAttribute;
	past= tp[1+partCount].tpStroff+ stroffShift;
	stroff= tp[1].tpStroff+ stroffShift;
	oldSize= past- stroff;
	d= newSize- oldSize;

	if  ( d > 0				&&
	      docInflateTextString( bi, d )	)
	    { LDEB(d); return -1;	}

	if  ( d != 0 && bi->biParaStrlen+ 1 > past )
	    {
	    memmove( bi->biParaString+ past+ d, bi->biParaString+ past,
						bi->biParaStrlen- past+ 1 );
	    }

	memcpy( bi->biParaString+ stroff, scratch, newSize );

	bi->biParaString[bi->biParaStrlen+ d]= '\0';

	partsDone= docRedivideStringInParticules( bi, stroff, newSize,
						    part+ 1, partCount, ta );
	if  ( partsDone < partCount )
	    {
	    docDeleteParticules( bi, part+ 1+ partsDone,
						    partCount- partsDone );
	    }

	part += 1+ partsDone- 1; /* -1: loop increment */

	stroffShift += d;
	bi->biParaStrlen += d;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the value of a ref field.					*/
/*									*/
/************************************************************************/

static int docCalculateRefField(	int *			pCalculated,
					int *			pNewSize,
					char *			target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	dfRef )
    {
    const char *		markName;
    int				markSize;

    SelectionScope		ssNew;
    BufferSelection		bsNew;

    int				len;

    if  ( dfRef->dfKind != DOCfkREF )
	{ *pCalculated= 0; return 0;	}

    if  ( docFieldGetRef( dfRef, &markName, &markSize ) )
	{ LDEB(1); *pCalculated= 0; return 0;	}

    docInitSelection( &bsNew );
    docInitSelectionScope( &ssNew );

    if  ( docFindBookmarkInDocument( &ssNew, &bsNew, bd, markName, markSize ) )
	{
	SLDEB(markName,markSize);

	strcpy( target, "<<??>>" );
	len= strlen( target );
	}
    else{
	int			beginMoved= 0;
	int			endMoved= 0;
	const unsigned char *	string;

	docConstrainSelectionToOneParagraph( &beginMoved, &endMoved, &bsNew );

	if  ( bsNew.bsEnd.bpStroff == bsNew.bsBegin.bpStroff )
	    { LLDEB(bsNew.bsEnd.bpStroff,bsNew.bsBegin.bpStroff);	}

	string= bsNew.bsBegin.bpBi->biParaString;
	len= bsNew.bsEnd.bpStroff- bsNew.bsBegin.bpStroff;

	if  ( len > targetSize )
	    { len=  targetSize;	}
	memcpy( target, string+ bsNew.bsBegin.bpStroff, len );
	target[len]= '\0';
	}

    *pNewSize= len;
    *pCalculated= 1;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the value of a pageref field.				*/
/*									*/
/************************************************************************/

static int docCalculatePagerefField(	int *			pCalculated,
					int *			pNewSize,
					char *			target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	dfRef )
    {
    const DocumentField *	dfMark;
    int				pageNumber;
    int				i;
    const BufferItem *		sectBi= (const BufferItem *)0;

    const char *		markName;
    int				markSize;

    if  ( dfRef->dfKind != DOCfkPAGEREF )
	{ *pCalculated= 0; return 0;	}

    if  ( docFieldGetPageref( dfRef, &markName, &markSize ) )
	{ LDEB(1); *pCalculated= 0; return 0;	}

    dfMark= docFindBookmarkField( &(bd->bdFieldList), markName, markSize );
    if  ( ! dfMark )
	{ SXDEB(markName,dfMark); *pCalculated= 0; return 0;	}

    pageNumber= dfMark->dfPage;

    for ( i= 0; i < bd->bdItem.biChildCount; i++ )
	{
	sectBi= bd->bdItem.biChildren[i];
	if  ( sectBi->biBelowPosition.lpPage >= pageNumber )
	    { break;	}
	}

    if  ( i >= bd->bdItem.biChildCount )
	{ LDEB(1); *pCalculated= 0; return 0;	}

    docFormatPageNumber( target, targetSize, sectBi, pageNumber );

    *pNewSize= strlen( target );
    *pCalculated= 1;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Return the value of a page/numpages field.				*/
/*									*/
/************************************************************************/

static int docCalculatePageField(	int *			pCalculated,
					int *			pNewSize,
					char *			target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df )
    {
    const BufferItem *	headerFooterBi;

    if  ( df->dfKind != DOCfkPAGE )
	{ *pCalculated= 0; return 0;	}

    headerFooterBi= paraBi->biParent;
    headerFooterBi= headerFooterBi->biParent;
    headerFooterBi= headerFooterBi->biParent;

    docFormatPageNumber( target, targetSize,
			    headerFooterBi->biSectHeaderFooterUseForSectBi,
			    headerFooterBi->biSectHeaderFooterUseForPage );
    *pNewSize= strlen( target );

    *pCalculated= 1; return 0;
    }

static int docCalculateNumpagesField(	int *			pCalculated,
					int *			pNewSize,
					char *			target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df )
    {
    if  ( targetSize < 20 )
	{ LDEB(targetSize); return -1;	}

    sprintf( target, "%d", bd->bdItem.biBelowPosition.lpPage );
    *pNewSize= strlen( target );

    *pCalculated= 1; return 0;
    }

/************************************************************************/
/*									*/
/*  Return the value of one of the document dates.			*/
/*									*/
/************************************************************************/

static int docCalculateDocDateField(	int *			pCalculated,
					int *			pNewSize,
					char *			target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df )
    {
    FieldInstructionsComponent	fic[FIC_COUNT];
    int				n;
    int				comp;
    unsigned char *		bytes= df->dfInstructions.odBytes;

    const DocumentProperties *	dp= &(bd->bdProperties);
    const struct tm *		tm;

    int				mergeFormat= 0;
    int				pictureComp= -1;

    n= docSplitFieldInstructions( &(df->dfInstructions), fic, FIC_COUNT );
    if  ( n < 2 )
	{ LDEB(n); return -1;	}

    for ( comp= 1; comp < n; comp++ )
	{
	if  ( comp < n-1					&&
	      fic[comp].ficSize == 2				&&
	      ! memcmp( bytes+ fic[comp].ficOffset, "\\*", 2 )	&&
	      fic[comp+ 1].ficSize == 11				)
	    {
	    const char *	mergeformat= "mergeformat";
	    int			i;

	    for ( i= 0; i < 11; i++ )
		{
		if  ( tolower( bytes[fic[comp+1].ficOffset+ i] ) != 
							mergeformat[i] )
		    { break;	}
		}

	    if  ( i == 11 )
		{ mergeFormat= 1;	}
	    }

	if  ( comp < n-1					&&
	      fic[comp].ficSize == 2				&&
	      ! memcmp( bytes+ fic[comp].ficOffset, "\\@", 2 )	)
	    { pictureComp= comp+ 1; }
	}

    if  ( targetSize < 40 )
	{ LDEB(targetSize); return -1;	}

    switch( df->dfKind )
	{
	case DOCfkCREATEDATE:	tm= &(dp->dpCreatim);	break;
	case DOCfkSAVEDATE:	tm= &(dp->dpRevtim);	break;
	case DOCfkPRINTDATE:	tm= &(dp->dpPrintim);	break;

	default:
	    LDEB(df->dfKind); *pCalculated= 0; return 0;
	}

    if  ( tm->tm_mday == 0 )
	{ /*LDEB(tm->tm_mday);*/ *pCalculated= 0; return 0;	}

    if  ( pictureComp >= 0 )
	{
	char		scratch[50+1];

	if  ( fic[pictureComp].ficSize > sizeof(scratch)- 1 )
	    {
	    LLDEB(fic[pictureComp].ficSize,sizeof(scratch));
	    *pCalculated= 0; return 0;
	    }
	memcpy( scratch, bytes+ fic[pictureComp].ficOffset,
						    fic[pictureComp].ficSize );
	scratch[fic[pictureComp].ficSize]= '\0';

	appWordFormatDate( target, targetSize, tm, scratch );
	*pNewSize= strlen( target );
	*pCalculated= 1; return 0;
	}

    if  ( mergeFormat )
	{
	/*  GCC: Sut Up! */
	const char *	frm= "%c";
	if  ( strftime( target, targetSize, frm, tm ) < 1 )
	    { *pCalculated= 0; return 0;	}

	*pNewSize= strlen( target );
	*pCalculated= 1; return 0;
	}


    *pCalculated= 0; return 0;
    }

/************************************************************************/
/*									*/
/*  Return the value of one of the document properties.			*/
/*									*/
/************************************************************************/

static int docCalculateDocStringField(	int *			pCalculated,
					int *			pNewSize,
					char *			target,
					int			targetSize,
					BufferDocument *	bd,
					const BufferItem *	paraBi,
					const DocumentField *	df )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    char *			val;
    int				len;
    
    switch( df->dfKind )
	{
	case DOCfkAUTHOR:	val= (char *)dp->dpAuthor;	break;
	case DOCfkCOMMENTS:	val= (char *)dp->dpComment;	break;
	case DOCfkKEYWORDS:	val= (char *)dp->dpKeywords;	break;
	case DOCfkSUBJECT:	val= (char *)dp->dpSubject;	break;
	case DOCfkTITLE:	val= (char *)dp->dpTitle;	break;

	default:
	    LDEB(df->dfKind); *pCalculated= 0; return 0;
	}

    if  ( ! val )
	{ val= "";	}
    len= strlen( val );

    if  ( len > targetSize )
	{ LLDEB(len,targetSize); *pCalculated= 0; return 0;	}

    strcpy( target, val );
    *pNewSize= len;

    *pCalculated= 1; return 0;
    }

/************************************************************************/
/*									*/
/*  The Kinds of Fields actively supported.				*/
/*									*/
/************************************************************************/

#   define	NOT_IN_RTF	0
#   define	FIELD_IN_RTF	1

#   define	NO_DEST		0
#   define	DEST_IN_RTF	1

#   define	RESULT_READONLY	0
#   define	RESULT_EDITABLE	1

const FieldKindInformation DOC_FieldKinds[]=
{
    {
	"-UNKNOWN", (char *)0,
		DOClevANY,
		NOT_IN_RTF,
		NO_DEST,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_READONLY,
    },
    {
	"HYPERLINK", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_EDITABLE,
    },
    {
	"-XE", "xe",
		DOClevTEXT,
		NOT_IN_RTF,
		DEST_IN_RTF,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_READONLY,
    },
    {
	"-TC", "tc",
		DOClevTEXT,
		NOT_IN_RTF,
		DEST_IN_RTF,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_READONLY,
    },
    {
	"-TCN", "tcn",
		DOClevTEXT,
		NOT_IN_RTF,
		DEST_IN_RTF,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_READONLY,
    },
    {
	"-BOOKMARK", (char *)0,
		DOClevTEXT,
		NOT_IN_RTF,
		NO_DEST,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_EDITABLE,
    },
    {
	"PAGE", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculatePageField,
		FIELDdoPAGE_NUMBER,
		RESULT_READONLY,
    },
    {
	"PAGEREF", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculatePagerefField,
		FIELDdoDOC_FORMATTED,
		RESULT_READONLY,
    },
    {
	"NUMPAGES", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateNumpagesField,
		FIELDdoDOC_FORMATTED,
		RESULT_READONLY,
    },
    {
	"REF", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateRefField,
		FIELDdoDOC_COMPLETE,
		RESULT_READONLY,
    },
    {
	"CREATEDATE", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocDateField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"SAVEDATE", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocDateField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"PRINTDATE", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocDateField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"AUTHOR", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocStringField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"COMMENTS", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocStringField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"KEYWORDS", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocStringField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"SUBJECT", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocStringField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"TITLE", (char *)0,
		DOClevTEXT,
		FIELD_IN_RTF,
		NO_DEST,
		docCalculateDocStringField,
		FIELDdoDOC_INFO,
		RESULT_READONLY,
    },
    {
	"-FOOTNOTE", (char *)0,
		DOClevTEXT,
		NOT_IN_RTF,
		NO_DEST,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_READONLY,
    },
    {
	"-CHFTN", (char *)0,
		DOClevTEXT,
		NOT_IN_RTF,
		NO_DEST,
		(CALCULATE_TEXT_FIELD)0,
		FIELDdoNEVER,
		RESULT_READONLY,
    },
};

const int DOC_FieldKindCount=
			sizeof(DOC_FieldKinds)/sizeof(FieldKindInformation);

