#   include	"config.h"

#   include	<stdlib.h>

#   include	"docDraw.h"
#   include	"docPs.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  For dividing table rows over pages.					*/
/*									*/
/************************************************************************/

typedef struct CellDrawingProgress
    {
    int			cdpAdvanced;
    int			cdpPara;
    int			cdpLinesDone;

    int			cdpX0Twips;
    int			cdpX1Twips;
    int			cdpX11Twips;	/*  Includes right border	*/

    int			cdpX0Pixels;
    int			cdpX1Pixels;
    int			cdpX11Pixels;	/*  Includes right border	*/

    FormattingFrame	cdpFormattingFrameTwips;
    FormattingFrame	cdpFormattingFramePixels;
    } CellDrawingProgress;

/************************************************************************/
/*									*/
/*  Initialise a DrawingContext						*/
/*									*/
/************************************************************************/

void docInitDrawingContext(	DrawingContext *	dc )
    {
    dc->dcCurrentPhysicalFont= -1;

    dc->dcDrawingData= (AppDrawingData *)0;
    dc->dcDocument= (BufferDocument *)0;
    dc->dcClipRect= (DocumentRectangle *)0;
    dc->dcSelection= (BufferSelection *)0;
    dc->dcFirstPage= -1;
    dc->dcLastPage= -1;
    dc->dcDrawHeadersFooters= 0;

    dc->dcDrawTableGrid= 0;

    dc->dcDrawTextLine= (DRAW_TEXT_LINE)0;
    dc->dcDrawParaTop= (DRAW_PARA_TOP)0;
    dc->dcDrawParaBottom= (DRAW_PARA_BOTTOM)0;
    dc->dcDrawCellTop= (DRAW_CELL_TOP)0;
    dc->dcDrawCellBottom= (DRAW_CELL_BOTTOM)0;
    dc->dcDrawCellLeft= (DRAW_CELL_LEFT)0;
    dc->dcDrawCellRight= (DRAW_CELL_RIGHT)0;
    dc->dcFinishPage= (FINISH_PAGE)0;
    dc->dcStartPage= (START_PAGE)0;
    dc->dcParaSidesPixels= (PARA_SIDES_PIXELS)0;

    dc->dcLayoutHeader= (LAYOUT_EXTERNAL)0;
    dc->dcLayoutFooter= (LAYOUT_EXTERNAL)0;

    return;
    }

/************************************************************************/
/*									*/
/*  Generic Document drawing machinery.					*/
/*									*/
/************************************************************************/

static int docDrawOneLine(	void *				through,
				const BufferItem *		bi,
				int				part,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				int				line,
				DrawingContext *		dc )
    {
    const TextLine *		tl= bi->biParaLines+ line;

    int				accepted;

    if  ( part == 0				&&
	  bi->biParaBorderAboveParagraph	&&
	  dc->dcDrawParaTop			)
	{
	if  ( (*dc->dcDrawParaTop)( bi, bi->biParaBorderAboveParagraph,
					ffTwips, ffPixels, through, dc ) )
	    { LDEB(1); return -1;	}
	}

    if  ( dc->dcDrawTextLine )
	{
	accepted= (*dc->dcDrawTextLine)( bi, line,
				    ffTwips, ffPixels, through, dc );
	}
    else{ accepted= tl->tlParticuleCount;	}

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( part+ accepted == bi->biParaParticuleCount )
	{
	if  ( bi->biParaBorderBelowParagraph	&&
	      dc->dcDrawParaBottom			)
	    {
	    if  ( (*dc->dcDrawParaBottom)( bi,
					bi->biParaBorderBelowParagraph,
					ffTwips, ffPixels, through, dc ) )
		{ LDEB(1); return -1;	}
	    }
	}

    return accepted;
    }

static int docDrawTextLines(	void *				through,
				const BufferItem *		bi,
				int				page,
				const FormattingFrame *		ffTwips,
				const FormattingFrame *		ffPixels,
				int *				pLinesDone,
				DrawingContext *		dc )
    {
    int				done= 0;
    int				pastSelectionEnd= 0;

    int				line= *pLinesDone;
    const TextLine *		tl= bi->biParaLines+ line;

    while( line < bi->biParaLineCount )
	{
	int	accepted;
	int	beforeSelectionBegin= 0;

	if  ( tl->tlTopPosition.lpPage > page )
	    { break;	}

	if  ( tl->tlTopPosition.lpPage < page )
	    { beforeSelectionBegin= 1;	}

	if  ( dc->dcClipRect )
	    {
	    AppDrawingData *		add= dc->dcDrawingData;
	    double			xfac= add->addMagnifiedPixelsPerTwip;

	    int	y1= tl->tlTopPosition.lpYPixels+
				TWIPStoPIXELS( xfac, tl->tlLineSpacingTwips );

	    if  ( tl->tlTopPosition.lpYPixels > dc->dcClipRect->drY1 )
		{ pastSelectionEnd= 1;	}
	    if  ( y1 < dc->dcClipRect->drY0 )
		{ beforeSelectionBegin= 1;	}
	    }

	if  ( dc->dcFirstPage >= 0		&&
	      page < dc->dcFirstPage		)
	    { beforeSelectionBegin= 1;	}

	if  ( dc->dcSelection					&&
	      ! beforeSelectionBegin				&&
	      ! pastSelectionEnd				&&
	      ( bi == dc->dcSelection->bsBegin.bpBi	||
	        bi == dc->dcSelection->bsEnd.bpBi	)	)
	    {
	    const BufferSelection *	bs= dc->dcSelection;

	    BufferPosition		bpLineBegin;
	    BufferPosition		bpLineEnd;
	    const int			mindLine= 1;

	    docLinePositions( &bpLineBegin, &bpLineEnd, bi, line );

	    if  ( bi == bs->bsEnd.bpBi					&&
		  docComparePositions( &bpLineBegin,
				    &(bs->bsEnd), mindLine ) > 0	)
		{ pastSelectionEnd= 1;	}

	    if  ( bi == bs->bsBegin.bpBi				&&
		  docComparePositions( &bpLineEnd,
				    &(bs->bsBegin), mindLine ) < 0	)
		{ beforeSelectionBegin= 1;	}
	    }

	if  ( beforeSelectionBegin || pastSelectionEnd )
	    { accepted= tl->tlParticuleCount;		}
	else{
	    accepted= docDrawOneLine( through, bi, tl->tlFirstParticule,
						ffTwips, ffPixels, line, dc );
	    }

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1; }

	done += accepted; line++; tl++;
	}

    *pLinesDone= line;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Print/Draw Headers/Footers.						*/
/*									*/
/************************************************************************/

static int docDrawHeaderItem(	HeaderFooter *			hf,
				int				pageNumber,
				int				inHeaderFooter,
				const BufferItem *		sectBi,
				DrawingContext *		dc,
				void *				through )
    {
    if  ( pageNumber != hf->hfPageUsedFor				&&
	  docLayoutHeaderFooter( hf, pageNumber, inHeaderFooter,
					    dc->dcDocument, sectBi,
					    dc->dcDrawingData,
					    dc->dcLayoutHeader,
					    (DocumentRectangle *)0 )	)
	{ LDEB(1); return -1;	}

    if  ( docDrawItem( hf->hfItem, through, dc ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static int docDrawFooterItem(	HeaderFooter *			hf,
				int				pageNumber,
				int				inHeaderFooter,
				const BufferItem *		sectBi,
				DrawingContext *		dc,
				void *				through )
    {
    if  ( pageNumber != hf->hfPageUsedFor				&&
	  docLayoutHeaderFooter( hf, pageNumber, inHeaderFooter,
					    dc->dcDocument, sectBi,
					    dc->dcDrawingData,
					    dc->dcLayoutFooter,
					    (DocumentRectangle *)0 )	)
	{ LDEB(1); return -1;	}

    if  ( docDrawItem( hf->hfItem, through, dc ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

int docDrawPageHeader(	BufferItem *			sectBi,
			void *				through,
			DrawingContext *		dc,
			int				page )
    {
    HeaderFooter *		hf= (HeaderFooter *)0;
    int				inHeaderFooter= DOCinBODY;
    BufferDocument *		bd= dc->dcDocument;
    DocumentProperties *	dp= &(bd->bdProperties);

    if  ( dc->dcClipRect )
	{
	DocumentRectangle		drHeader;
	AppDrawingData *		add= dc->dcDrawingData;
	double				xfac= add->addMagnifiedPixelsPerTwip;
	const SectionProperties *	sp= &(sectBi->biSectProperties);
	const DocumentGeometry *	dg= &(sp->spDocumentGeometry);
	int				pageY0;

	drHeader.drX0= add->addBackRect.drX0;
	drHeader.drX1= add->addBackRect.drX1;

	pageY0= page* add->addPageStepPixels;
	drHeader.drY0= pageY0;
	drHeader.drY1= pageY0+ TWIPStoPIXELS( xfac, dg->dgTopMarginTwips );

	if  ( ! docIntersectRectangle( &drHeader, &drHeader, dc->dcClipRect ) )
	    { return 0;	}
	}

    inHeaderFooter= docWhatPageHeader( &hf, sectBi, page, dp );

    if  ( hf								&&
	  hf->hfItem							&&
	  docDrawHeaderItem( hf, page, inHeaderFooter,
					    sectBi, dc, through )	)
	{ XLDEB(hf,page); return -1;	}

    return 0;
    }

int docDrawPageFooter(	BufferItem *			sectBi,
			void *				through,
			DrawingContext *		dc,
			int				page )
    {
    HeaderFooter *		hf= (HeaderFooter *)0;
    int				inHeaderFooter= DOCinBODY;
    BufferDocument *		bd= dc->dcDocument;
    DocumentProperties *	dp= &(bd->bdProperties);

    if  ( dc->dcClipRect )
	{
	DocumentRectangle		drFooter;
	AppDrawingData *		add= dc->dcDrawingData;
	double				xfac= add->addMagnifiedPixelsPerTwip;
	const SectionProperties *	sp= &(sectBi->biSectProperties);
	const DocumentGeometry *	dg= &(sp->spDocumentGeometry);
	int				pageY0;

	drFooter.drX0= add->addBackRect.drX0;
	drFooter.drX1= add->addBackRect.drX1;

	pageY0= page* add->addPageStepPixels;
	drFooter.drY0= pageY0+ TWIPStoPIXELS( xfac,
			    dg->dgPageHighTwips- dg->dgBottomMarginTwips );
	drFooter.drY1= pageY0+ TWIPStoPIXELS( xfac, dg->dgPageHighTwips );

	if  ( ! docIntersectRectangle( &drFooter, &drFooter, dc->dcClipRect ) )
	    { return 0;	}
	}

    inHeaderFooter= docWhatPageFooter( &hf, sectBi, page, dp );

    if  ( hf								&&
	  hf->hfItem							&&
	  docDrawFooterItem( hf, page, inHeaderFooter,
					    sectBi, dc, through )	)
	{ XLDEB(hf,page); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/*  1)  Just to make the compiler happy, if nothing is to be printed	*/
/*	anymore, the current position is not going to be used.		*/
/*									*/
/************************************************************************/

static int docDrawToNextPage(	BufferItem *			thisBi,
				BufferItem *			nextBi,
				int				thisPage,
				void *				through,
				DrawingContext *		dc )
    {
    const BufferDocument *	bd= dc->dcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const DocumentGeometry *	dg= &(dp->dpGeometry);
    int				nextPage= thisPage+ 1;

    const int			asLast= 0;

    if  ( thisBi->biLevel != nextBi->biLevel	||
	  thisBi->biLevel <  DOClevSECT		)
	{
	SSDEB(docLevelStr(thisBi->biLevel),docLevelStr(nextBi->biLevel));
	return -1;
	}

    while( thisBi && thisBi->biLevel > DOClevSECT )
	{ thisBi= thisBi->biParent; nextBi= nextBi->biParent;	}

    if  ( thisBi						&&
	  dc->dcDrawHeadersFooters				&&
	  docDrawPageFooter( thisBi, through, dc, thisPage )	)
	{ LDEB(thisPage); return -1;	}

    if  ( dc->dcFinishPage					&&
	  (*dc->dcFinishPage)( through, dc, thisPage, asLast )	)
	{ LDEB(1); return -1;	}

    if  ( dc->dcLastPage < 0				||
	  nextPage <= dc->dcLastPage			)
	{
	if  ( dc->dcStartPage					&&
	      (*dc->dcStartPage)( through, dg, dc, nextPage )	)
	    { LDEB(1); return -1;	}

	if  ( nextBi							&&
	      dc->dcDrawHeadersFooters					&&
	      docDrawPageHeader( nextBi, through, dc, nextPage )	)
	    { LDEB(nextPage); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Print a node in the BufferItem hierarchy.				*/
/*									*/
/*  1)  The last page is finished by the caller.			*/
/*									*/
/************************************************************************/

static int docDrawParaItem(	BufferItem *			paraBi,
				void *				through,
				DrawingContext *		dc )
    {
    FormattingFrame		ffTwips;
    FormattingFrame		ffPixels;
    int				page;
    int				line= 0;

    int				bottomTwips= -1;
    int				stripHigh= -1;

    docParagraphFrameTwips( &ffTwips, bottomTwips, stripHigh, paraBi );

    if  ( dc->dcParaSidesPixels )
	{ (*dc->dcParaSidesPixels) ( &ffPixels, dc->dcDrawingData, paraBi ); }

    page= paraBi->biTopPosition.lpPage;
    while( line < paraBi->biParaLineCount )
	{
	if  ( docDrawTextLines( through, paraBi, page,
					    &ffTwips, &ffPixels, &line, dc ) )
	    { LDEB(page); return -1;	}

	/*  1  */
	if  ( dc->dcLastPage >= 0	&&
	      page >= dc->dcLastPage	)
	    { break;	}

	if  ( line < paraBi->biParaLineCount )
	    {
	    if  ( docDrawToNextPage( paraBi, paraBi, page, through, dc ) )
		{ SDEB(docLevelStr(paraBi->biLevel));return -1;	}

	    page++;
	    }
	}

    return 0;
    }

static int psAdvanceCell(	void *				through,
				int *				pAdvanced,
				BufferItem *			cellBi,
				int				page,
				CellDrawingProgress *		cdp,
				DrawingContext *		dc,
				LayoutPosition *		lpBelow )
				
    {
    int				bottomTwips= -1;
    int				stripHigh= -1;

    int				advanced= 0;

    LayoutPosition		lp;

    while( cdp->cdpPara < cellBi->biChildCount )
	{
	BufferItem *		paraBi= cellBi->biChildren[cdp->cdpPara];
	int			linesDone= cdp->cdpLinesDone;

	if  ( cdp->cdpLinesDone == 0 )
	    {
	    docParagraphFrameTwips( &(cdp->cdpFormattingFrameTwips),
					    bottomTwips, stripHigh, paraBi );

	    if  ( dc->dcParaSidesPixels )
		{
		(*dc->dcParaSidesPixels) ( &(cdp->cdpFormattingFramePixels),
						dc->dcDrawingData, paraBi );
		}
	    }

	if  ( docDrawTextLines( through, paraBi, page,
					&(cdp->cdpFormattingFrameTwips),
					&(cdp->cdpFormattingFramePixels),
					&linesDone, dc ) )
	    { LDEB(page); return -1;	}

	if  ( linesDone > cdp->cdpLinesDone )
	    { cdp->cdpAdvanced= advanced= 1;	}

	cdp->cdpLinesDone= linesDone;

	if  ( cdp->cdpLinesDone < paraBi->biParaLineCount )
	    {
	    if  ( cdp->cdpLinesDone > 0 )
		{
		AppDrawingData *	add= dc->dcDrawingData;
		double			xfac= add->addMagnifiedPixelsPerTwip;
		TextLine *		tl;

		tl= paraBi->biParaLines+ cdp->cdpLinesDone- 1;

		lp.lpPage= tl->tlTopPosition.lpPage;
		lp.lpPageYTwips= tl->tlTopPosition.lpPageYTwips+
						    tl->tlLineSpacingTwips;
		lp.lpYPixels= tl->tlTopPosition.lpYPixels+
			    TWIPStoPIXELS( xfac, tl->tlLineSpacingTwips );
		}
	    else{ lp= paraBi->biTopPosition;	}
	    }
	else{ lp= paraBi->biBelowPosition; }

	if  ( lpBelow->lpPage < lp.lpPage )
	    { lpBelow->lpPage=  lp.lpPage; }
	if  ( lpBelow->lpPageYTwips < lp.lpPageYTwips )
	    { lpBelow->lpPageYTwips=  lp.lpPageYTwips;	}
	if  ( lpBelow->lpYPixels < lp.lpYPixels )
	    { lpBelow->lpYPixels=  lp.lpYPixels;	}

	if  ( cdp->cdpLinesDone < paraBi->biParaLineCount )
	    { break;	}

	cdp->cdpLinesDone= 0;
	cdp->cdpPara++;
	}

    if  ( advanced )
	{ *pAdvanced= 1;	}

    return 0;
    }

static int docDrawCellSides(	const BufferItem *		rowBi,
				const CellDrawingProgress *	cdp,
				int				drawTop,
				int				drawSides,
				int				drawBottom,
				void *				through,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    const CellProperties *	cp= rowBi->biRowCells;

    int				col;

    const BufferItem *		prevRow= (const BufferItem *)0;
    const BufferItem *		nextRow= (const BufferItem *)0;

    const CellProperties *	prevCp= (const CellProperties *)0;
    const CellProperties *	nextCp= (const CellProperties *)0;

    if  ( dc->dcDrawTableGrid )
	{
	const BufferItem *	sectBi= rowBi->biParent;

	if  ( drawTop && rowBi->biNumberInParent )
	    {
	    prevRow= sectBi->biChildren[rowBi->biNumberInParent- 1];

	    if  ( ! prevRow->biRowHasTableParagraphs			||
		  ! docAlignedColumns( &(prevRow->biRowProperties),
					    &(rowBi->biRowProperties) )	)
		{ prevRow= (const BufferItem *)0;	}
	    else{ prevCp= prevRow->biRowCells;		}
	    }

	if  ( rowBi->biNumberInParent < sectBi->biChildCount- 1 )
	    {
	    nextRow= sectBi->biChildren[rowBi->biNumberInParent+ 1];

	    if  ( ! nextRow->biRowHasTableParagraphs			||
		  ! docAlignedColumns( &(nextRow->biRowProperties),
					    &(rowBi->biRowProperties) )	)
		{ nextRow= (const BufferItem *)0;	}
	    else{ nextCp= nextRow->biRowCells;	}
	    }
	}

    for ( col= 0; col < rowBi->biChildCount; cdp++, cp++, col++ )
	{
	int		topAsGrid= 0;
	int		leftAsGrid= 0;
	int		rightAsGrid= 0;
	int		bottomAsGrid= 0;

	int		drawLeft= drawSides;
	int		drawRight= drawSides;

	if  ( cp->cpMergedWithLeft )
	    { drawLeft= 0;	}
	if  ( col < rowBi->biChildCount- 1	&&
	      cp[1].cpMergedWithLeft		)
	    { drawRight= 0;	}

	if  ( dc->dcDrawTableGrid )
	    {
	    if  ( ! cp->cpTopBorder.bpIsSet				&&
		  ( ! prevRow || ! prevCp->cpBottomBorder.bpIsSet )	)
		{ topAsGrid= 1;	}

	    if  ( col == 0 && ! cp->cpLeftBorder.bpIsSet )
		{ leftAsGrid= 1;	}

	    if  ( col == rowBi->biChildCount- 1 )
		{
		if  ( ! cp->cpRightBorder.bpIsSet )
		    { rightAsGrid= 1;	}
		}
	    else{
		if  ( ! cp->cpRightBorder.bpIsSet	&&
		      ! cp[1].cpLeftBorder.bpIsSet	)
		    { rightAsGrid= 1;	}
		}

	    if  ( ! cp->cpBottomBorder.bpIsSet				&&
		  ( ! nextRow || ! nextCp->cpTopBorder.bpIsSet )	)
		{ bottomAsGrid= 1;	}
	    }

	if  ( drawTop							&&
	      dc->dcDrawCellTop						&&
	      ( cp->cpTopBorder.bpIsSet || topAsGrid )			)
	    {
	    if  ( (*dc->dcDrawCellTop)(		&(cp->cpTopBorder),
						topAsGrid,
						cdp->cdpX0Twips,
						cdp->cdpX0Pixels,
						cdp->cdpX11Twips,
						cdp->cdpX11Pixels,
						through, dc,
						&(rowBi->biTopPosition) ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( drawLeft							&&
	      dc->dcDrawCellLeft					&&
	      ( cp->cpLeftBorder.bpIsSet || leftAsGrid )		)
	    {
	    if  ( (*dc->dcDrawCellLeft)(	&(cp->cpLeftBorder),
						leftAsGrid,
						through, dc,
						cdp->cdpX0Twips,
						cdp->cdpX0Pixels,
						lpTop, lpBelow ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( drawRight							&&
	      dc->dcDrawCellRight					&&
	      ( cp->cpRightBorder.bpIsSet || rightAsGrid )		)
	    {
	    if  ( (*dc->dcDrawCellRight)(	&(cp->cpRightBorder),
						rightAsGrid,
						through, dc,
						cdp->cdpX1Twips,
						cdp->cdpX1Pixels,
						lpTop, lpBelow ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( drawBottom						&&
	      dc->dcDrawCellBottom					&&
	      ( cp->cpBottomBorder.bpIsSet || bottomAsGrid )		)
	    {
	    if  ( (*dc->dcDrawCellBottom)(	&(cp->cpBottomBorder),
						bottomAsGrid,
						cdp->cdpX0Twips,
						cdp->cdpX0Pixels,
						cdp->cdpX11Twips,
						cdp->cdpX11Pixels,
						through, dc,
						&(rowBi->biBelowPosition) ) )
		{ LDEB(1); return -1;	}
	    }

	if  ( prevRow )
	    { prevCp++;	}
	if  ( nextRow )
	    { nextCp++;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Print as much of a table row as fits on the current page.		*/
/*									*/
/************************************************************************/

static int docDrawRowPageStrip(	const BufferItem *		rowBi,
				int				page,
				int *				pToNextPage,
				CellDrawingProgress *		cdp,
				void *				through,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop )
    {
    int				col;

    int				toNextPage= 0;

    int				advanced;
    int				atTop= 1;

    LayoutPosition		lpBottom;

    advanced= 0;
    for ( col= 0; col < rowBi->biChildCount; col++ )
	{
	BufferItem *	cellBi= rowBi->biChildren[col];

	lpBottom= *lpTop;

	if  ( dc->dcSelection )
	    {
	    const BufferSelection *	bs= dc->dcSelection;

	    if  ( docCompareItemPositions( cellBi, bs->bsBegin.bpBi ) < 0 )
		{ continue;	}

	    if  ( docCompareItemPositions( cellBi, bs->bsEnd.bpBi ) > 0	)
		{ continue;	}

	    if  ( ( bs->bsCol0 >= 0 && col < bs->bsCol0	)	||
	          ( bs->bsCol1 >= 0 && col > bs->bsCol1	)	)
		{ continue;	}
	    }

	if  ( cdp[col].cdpAdvanced )
	    { atTop= 0;	}

	if  ( psAdvanceCell( through, &advanced, cellBi, page,
						    cdp+ col, dc, &lpBottom ) )
	    { LDEB(1); return -1;	}

	if  ( cdp[col].cdpPara < cellBi->biChildCount	)
	    { toNextPage= 1;	}
	}

    if  ( advanced )
	{
	const int	drawTop= atTop != 0;
	const int	drawSides= toNextPage != 0;
	const int	drawBottom= 0;

	if  ( drawTop || drawSides || drawBottom )
	    {
	    if  ( docDrawCellSides( rowBi, cdp,
					drawTop, drawSides, drawBottom,
					through, dc, lpTop, &lpBottom ) )
		{ LDEB(1); return -1;	}
	    }
	}

    *pToNextPage= toNextPage; return 0;
    }

/************************************************************************/
/*									*/
/*  Draw a table row by successively drawing the portions that fit on	*/
/*  the current page and moving to the next page.			*/
/*									*/
/*  1)  The last page is finished by the caller.			*/
/*									*/
/************************************************************************/

static int docDrawRowItem(	BufferItem *			rowBi,
				void *				through,
				DrawingContext *		dc )
    {
    AppDrawingData *		add= dc->dcDrawingData;
    double			xfac= add->addMagnifiedPixelsPerTwip;

    int				col;

    const RowProperties *	rp= &(rowBi->biRowProperties);
    const SectionProperties *	sp= &(rowBi->biParent->biSectProperties);
    const DocumentGeometry *	dg= &(sp->spDocumentGeometry);
    CellProperties *		cp;

    int				toNextPage;
    int				page= rowBi->biTopPosition.lpPage;

    const int			drawTop= 0;
    const int			drawSides= 1;
    const int			drawBottom= 1;

    static CellDrawingProgress *	colP;
    CellDrawingProgress *		cdp;

    int				x00Pixels;

    int				x0Twips;
    int				x1Twips;
    int				x0Pixels;
    int				x1Pixels;

    LayoutPosition		lpTop;

    if  ( rp->rpCellCount < rowBi->biChildCount )
	{
	LLDEB(rp->rpCellCount,rowBi->biChildCount);
	docListItem( 0, rowBi );
	return -1;
	}

    cdp= (CellDrawingProgress *)realloc( colP,
				rp->rpCellCount* sizeof(CellDrawingProgress) );
    if  ( ! cdp )
	{ LXDEB(rp->rpCellCount,cdp); return -1;	}
    colP= cdp;

    toNextPage= 0;

    x00Pixels= TWIPStoPIXELS( xfac, dg->dgLeftMarginTwips );

    x1Twips= x0Twips= dg->dgLeftMarginTwips+ rowBi->biRowLeftIndentTwips;
    x1Pixels= x0Pixels= x00Pixels+ rowBi->biRowLeftIndentPixels;

    cp= rp->rpCells;
    for ( col= 0; col < rowBi->biChildCount; cp++, col++ )
	{
	x1Twips= dg->dgLeftMarginTwips+ cp->cpRightBoundaryTwips;
	x1Pixels= x00Pixels+ cp->cpRightBoundaryPixels;

	cdp[col].cdpAdvanced= 0;

	cdp[col].cdpPara= 0;
	cdp[col].cdpLinesDone= 0;

	cdp[col].cdpX0Twips= x0Twips;
	cdp[col].cdpX1Twips= x1Twips;
	cdp[col].cdpX11Twips= x1Twips;

	cdp[col].cdpX0Pixels= x0Pixels;
	cdp[col].cdpX1Pixels= x1Pixels;
	cdp[col].cdpX11Pixels= x1Pixels;

	if  ( cp->cpRightBorder.bpIsSet )
	    { cdp[col].cdpX11Twips +=  cp->cpRightBorder.bpWidthTwips; }

	x0Twips= x1Twips;
	x0Pixels= x1Pixels;
	}

    lpTop= rowBi->biTopPosition;

    if  ( docDrawRowPageStrip( rowBi, page, &toNextPage, cdp,
						    through, dc, &lpTop ) )
	{ LDEB(1); return -1;	}

    while( toNextPage )
	{
	/*  1  */
	if  ( dc->dcLastPage >= 0	&&
	      page >= dc->dcLastPage	)
	    { break;	}

	if  ( docDrawToNextPage( rowBi, rowBi, page, through, dc ) )
	    { SDEB(docLevelStr(rowBi->biLevel)); return -1;	}

	page++;
	docPsStartPageLayout( page, add, sp, &lpTop );

	if  ( docDrawRowPageStrip( rowBi, page, &toNextPage, cdp,
						    through, dc, &lpTop ) )
	    { LDEB(1); return -1;	}
	}

    if  ( docDrawCellSides( rowBi, cdp,
			    drawTop, drawSides, drawBottom,
			    through, dc, &lpTop, &(rowBi->biBelowPosition) ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the children of a group item.					*/
/*									*/
/************************************************************************/

static int docDrawGroupItem(	const BufferItem *		bi,
				void *				through,
				DrawingContext *		dc )
    {
    int			i;
    int			page;

    BufferItem *	prevBi= (BufferItem *)0;

    if  ( bi->biChildCount > 0 )
	{ prevBi= bi->biChildren[0];	}

    page= bi->biTopPosition.lpPage;
    for ( i= 0; i < bi->biChildCount; i++ )
	{
	BufferItem *	thisBi= bi->biChildren[i];

	if  ( dc->dcClipRect						&&
	      thisBi->biTopPosition.lpYPixels > dc->dcClipRect->drY1	)
	    { break;	}

	if  ( dc->dcSelection					&&
	      docCompareItemPositions( thisBi,
			    dc->dcSelection->bsEnd.bpBi ) > 0	)
	    { break;	}

	if  ( dc->dcLastPage >= 0				&&
	      thisBi->biTopPosition.lpPage > dc->dcLastPage	)
	    { break;	}

	if  ( dc->dcFirstPage < 0				||
	      thisBi->biBelowPosition.lpPage >= dc->dcFirstPage	)
	    {
	    if  ( thisBi->biTopPosition.lpPage > page			&&
		  thisBi->biTopPosition.lpPage > dc->dcFirstPage	&&
		  docDrawToNextPage( prevBi, thisBi, page,
						    through, dc )	)
		{ SDEB(docLevelStr(thisBi->biLevel)); return -1;	}

	    if  ( docDrawItem( thisBi, through, dc ) )
		{ LDEB(i); return -1;	}

	    page= thisBi->biBelowPosition.lpPage;
	    }

	prevBi= thisBi;
	}

    return 0;
    }

int docDrawItem(	BufferItem *			bi,
			void *				through,
			DrawingContext *		dc )
    {
    if  ( dc->dcClipRect )
	{
	if  ( bi->biTopPosition.lpYPixels > dc->dcClipRect->drY1 )
	    { return 0;	}

	if  ( bi->biBelowPosition.lpYPixels < dc->dcClipRect->drY0 )
	    { return 0; }
	}

    if  ( dc->dcSelection )
	{
	if  ( docCompareItemPositions( bi, dc->dcSelection->bsBegin.bpBi ) < 0 )
	    { return 0;	}

	if  ( docCompareItemPositions( bi, dc->dcSelection->bsEnd.bpBi ) > 0 )
	    { return 0; }
	}

    if  ( dc->dcFirstPage >= 0					&&
	  bi->biBelowPosition.lpPage < dc->dcFirstPage		)
	{ return 0;	}

    if  ( dc->dcLastPage >= 0					&&
	  bi->biTopPosition.lpPage > dc->dcLastPage		)
	{ return 0;	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevCELL:
	rowAsGroup:
	    if  ( docDrawGroupItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    if  ( docDrawRowItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevPARA:
	    if  ( docDrawParaItem( bi, through, dc ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    return 0;
    }
