/************************************************************************/
/*									*/
/*  Text Editor Buffer structure.					*/
/*									*/
/************************************************************************/

#   ifndef	DOC_BUF_H
#   define	DOC_BUF_H

#   include	<time.h>

#   include	<bitmap.h>
#   include	<docFont.h>
#   include	"docShape.h"
#   include	<appGeo.h>
#   include	<sioGeneral.h>

/************************************************************************/
/*									*/
/*  Forward declarations of structs.					*/
/*									*/
/************************************************************************/

struct BufferDocument;
struct BufferItem;
struct DocumentField;

/************************************************************************/
/*									*/
/*  An homogenous piece of text. Without hyphenation, it will not be	*/
/*  divided over lines.							*/
/*  It consists of some characters and then some spaces.		*/
/*									*/
/************************************************************************/

#   define	DOCkindUNKNOWN		0
#   define	DOCkindTEXT		1
#   define	DOCkindTAB		2
#   define	DOCkindOBJECT		3
#   define	DOCkindFIELDSTART	4
#   define	DOCkindFIELDEND		5
#   define	DOCkindTC		6
#   define	DOCkindXE		7
#   define	DOCkindLINEBREAK	8
#   define	DOCkindPAGEBREAK	9

typedef struct TextParticule
    {
    unsigned int	tpStroff:16;
    unsigned int	tpStrlen:12;
    unsigned int	tpX0:12;
    unsigned int	tpPixelsWide:12;
    unsigned int	tpKind:6;
    unsigned int	tpInField:1;
    short int		tpPhysicalFont;
    short int		tpObjectNumber;
    TextAttribute	tpTextAttribute;
    } TextParticule;

/************************************************************************/
/*									*/
/*  Used to layout lines of text.					*/
/*									*/
/************************************************************************/

typedef struct FormattingFrame
    {
    int		ffX0Geometry;
    int		ffX0TextLines;
    int		ffX0FirstLine;
    int		ffX1Geometry;
    int		ffX1TextLines;

    int		ffY1;

    int		ffPageHigh;
    int		ffStripHigh;
    } FormattingFrame;

typedef struct LayoutPosition
    {
    int			lpYPixels;
    short int		lpPageYTwips;
    short int		lpPage;
    } LayoutPosition;

/************************************************************************/
/*									*/
/*  Describes tabs.							*/
/*									*/
/*  !)	RS6000/AIX compler only supports unsigned bitfields.		*/
/*									*/
/************************************************************************/

typedef enum TabKind
    {
    DOCtkLEFT= 0,

    DOCtkRIGHT,
    DOCtkCENTRE,
    DOCtkDECIMAL
    } TabKind;

typedef enum TabLeader
    {
    DOCtlNONE= 0,

    DOCtlDOTS,
    DOCtlHYPH,
    DOCtlUNDERLINE,
    DOCtlTHICK,
    DOCtlEQUAL
    } TabLeader;

typedef struct TabStop
    {
    int			tsTwips;
    int			tsPixels;
    unsigned int	tsKind:2;	/*  (enum)	*/
    unsigned int	tsLeader:3;	/*  (enum)	*/
    } TabStop;

/************************************************************************/
/*									*/
/*  Used to lay out the text on the page.				*/
/*  A text line consists of a series of particules.			*/
/*									*/
/*  1)  What part of the string in the paragraph.			*/
/*  2)  What particules.						*/
/*  3)  Where is it geo/typographically?				*/
/*  4)  Postscript geometry information.				*/
/*	lineSpacing includes the leading that is irrelevant for the	*/
/*	last line of a page.						*/
/*  5)  Screen geometry.						*/
/*									*/
/************************************************************************/

typedef struct TextLine
    {
								/*  1,2	*/
    int			tlStroff;
    int			tlFirstParticule;
    short int		tlStrlen;
    short int		tlParticuleCount;
    short int		tlWordCount;
								/*  3  */
    LayoutPosition	tlTopPosition;
								/*  4  */
    short int		tlLineAscentTwips;
    short int		tlLineHeightTwips;
    short int		tlLineSpacingTwips;
								/*  5  */
    short int		tlLineAscentPixels;
    short int		tlHasPageBreak;
    } TextLine;

/************************************************************************/
/*									*/
/*  Information on Lists.						*/
/*									*/
/************************************************************************/

typedef struct DocumentList
    {
    int			dlListId;
    int			dlListTemplateId;
    int			dlListIsSimple;
    int			dlRestartForEverySection;
    char *		dlListName;
    } DocumentList;

/************************************************************************/
/*									*/
/*  Information on Paragraph Numbering.					*/
/*									*/
/************************************************************************/

typedef struct ParagraphNumber
    {
    short		pnLevel;
    short		pnStyle;
#			define	DOCpnCARD	0
#			define	DOCpnDEC	1
#			define	DOCpnUCLTR	2
#			define	DOCpnUCRM	3
#			define	DOCpnLCLTR	4
#			define	DOCpnLCRM	5
#			define	DOCpnORD	6
#			define	DOCpnORDT	7

    short		pnUseHangingIndent;

    int			pnStartNumber;
    int			pnIndentTwips;

    TextAttribute	pnTextAttribute;

    unsigned char	pnTextBefore[32+1];
    unsigned char	pnTextAfter[32+1];

    } ParagraphNumber;

/************************************************************************/
/*									*/
/*  A paragraph in a document.						*/
/*									*/
/*  !)	RS6000/AIX compler only supports unsigned bitfields.		*/
/*									*/
/************************************************************************/

typedef enum BorderStyle
    {
    DOCbsSH= 0,
    DOCbsDB,
    DOCbsDOT,
    DOCbsDASH,
    DOCbsHAIR,
    DOCbsDASHSM,
    DOCbsDASHD,
    DOCbsDASHDD,
    DOCbsTRIPLE,
    DOCbsTNTHSG,
    DOCbsTHTNSG,
    DOCbsTNTHTNSG,
    DOCbsTNTHMG,
    DOCbsTHTNMG,
    DOCbsTNTHTNMG,
    DOCbsTNTHLG,
    DOCbsTHTNLG,
    DOCbsTNTHTNLG,
    DOCbsWAVY,
    DOCbsWAVYDB,
    DOCbsDASHDOTSTR,
    DOCbsEMBOSS,
    DOCbsENGRAVE,

    DOCbs_COUNT
    } BorderStyle;

typedef struct BorderProperties
    {
    unsigned int	bpColor:8;
    unsigned int	bpSpacingTwips:16;
    unsigned int	bpWidthTwips:8;
    unsigned int	bpStyle:5;		/*  (enum)	*/
    unsigned int	bpThickness:2;		/*  0, 1 or 2	*/
    unsigned int	bpIsSet:1;
    } BorderProperties;

#   define	docCopyBorderProperties( bp1, bp2 ) *(bp1)= *(bp2)

typedef enum ItemAlignment
    {
    DOCiaLEFT= 0,
    DOCiaRIGHT,
    DOCiaCENTERED,
    DOCiaJUSTIFIED,

    DOCia_COUNT
    } ItemAlignment;

typedef enum ObjectKind
    {
    DOCokUNKNOWN= 0,
    DOCokPICTWMETAFILE,
    DOCokPICTPNGBLIP,
    DOCokPICTJPEGBLIP,
    DOCokPICTEMFBLIP,
    DOCokMACPICT,
    DOCokOLEOBJECT
    } ObjectKind;

typedef enum VerticalTextAlignment
    {
    DOCvtaTOP= 0,
    DOCvtaCENTERED,
    DOCvtaBOTTOM,
    DOCvtaVERT_LEFT,
    DOCvtaVERT_RIGHT
    } VerticalTextAlignment;

typedef enum ShadingPattern
    {
    DOCspSOLID= 0,

    DOCspBGHORIZ,
    DOCspBGVERT,
    DOCspBGFDIAG,
    DOCspBGBDIAG,
    DOCspBGCROSS,
    DOCspBGDCROSS,
    DOCspBGDKHORIZ,
    DOCspBGDKVERT,
    DOCspBGDKFDIAG,
    DOCspBGDKBDIAG,
    DOCspBGDKCROSS,
    DOCspBGDKDCROSS,

    DOCsp_COUNT
    } ShadingPattern;

typedef struct CellProperties
    {
    int			cpRightBoundaryTwips;
    int			cpRightBoundaryPixels;

    int			cpForegroundColor;
    int			cpBackgroundColor;

    BorderProperties	cpTopBorder;
    BorderProperties	cpBottomBorder;
    BorderProperties	cpLeftBorder;
    BorderProperties	cpRightBorder;

    unsigned int	cpShadingLevel:10;
    unsigned int	cpShadingPattern:4;		/*  (enum)	*/

    unsigned int	cpLeftInMergedRow:1;
    unsigned int	cpMergedWithLeft:1;
    unsigned int	cpTopInMergedColumn:1;
    unsigned int	cpMergedWithAbove:1;

    unsigned int	cpVerticalTextAlignment:3;	/*  (enum)	*/
    } CellProperties;

#   define	docCopyCellProperties( cp1, cp2 ) \
					    ( ( *(cp1)= *(cp2) ), 0 )

#   define	docCleanCellProperties( cp ) ( ( *(cp)= *(cp) ), 0 )

typedef struct RowProperties
    {
    int			rpCellCount;
    CellProperties *	rpCells;

    int			rpHalfGapWidthTwips;
    int			rpHalfGapWidthPixels;

    int			rpHeightTwips;

    int			rpLeftIndentTwips;
    int			rpLeftIndentPixels;

			/************************************************/
			/*  Row borders are not stored in the word	*/
			/*  binary format, and seem to be irrelevant:	*/
			/************************************************/
    BorderProperties	rpTopBorder;
    BorderProperties	rpBottomBorder;
    BorderProperties	rpLeftBorder;
    BorderProperties	rpRightBorder;
    BorderProperties	rpHorizontalBorder;
    BorderProperties	rpVerticalBorder;

    ItemAlignment	rpAlignment:3;
    unsigned int	rpHasTableParagraphs:1;

    unsigned int	rpHasHorizontalBorders:1;
    unsigned int	rpHasVerticalBorders:1;

    unsigned int	rpIsTableHeader:1;
    unsigned int	rpKeepOnPage:1;
    } RowProperties;

typedef enum SectionBreak
    {
    DOCsbkNONE= 0,
    DOCsbkCOL,
    DOCsbkPAGE,
    DOCsbkEVEN,
    DOCsbkODD,

    DOCsbk_COUNT
    } SectionBreak;

typedef enum PageNumberStyle
    {
    DOCpgnDEC= 0,
    DOCpgnUCRM,
    DOCpgnLCRM,
    DOCpgnUCLTR,
    DOCpgnLCLTR,

    DOCpgn_COUNT
    } PageNumberStyle;

typedef struct SectionProperties
    {
    DocumentGeometry	spDocumentGeometry;

    int			spColumnCount;
    int			spColumnSpacingTwips;
    int			spColumnWidthTwips;

    unsigned int	spHasTitlePage:1;
    unsigned int	spBreakKind:3;
    unsigned int	spPageNumberStyle:3;
    unsigned int	spRestartPageNumbers:1;

    int			spStartPageNumber;

    ParagraphNumber *	spParagraphNumbers;
    int			spParagraphNumberCount;
    } SectionProperties;

#   define	SPupdNONE		(0)

#   define	SPupdDOCUMENT_GEOMETRY	DGupdALL

#   define	SPupdTITLEPG		(1<<9)

#   define	SPupdNUMBER_STYLE	(1<<10)
#   define	SPupdBREAK_KIND		(1<<11)
#   define	SPupdPAGE_RESTART	(1<<12)
#   define	SPupdSTART_PAGE		(1<<13)

#   define	SPupdCOLUMN_COUNT	(1<<14)
#   define	SPupdCOLUMN_SPACING	(1<<15)
#   define	SPupdCOLUMN_WIDTH	(1<<16)
#   define	SPupdCOLUMN_PROPS	( SPupdCOLUMN_COUNT	|\
					  SPupdCOLUMN_SPACING	|\
					  SPupdCOLUMN_WIDTH	)

#   define	SPupdALL		( SPupdDOCUMENT_GEOMETRY	|\
					  SPupdTITLEPG			|\
					  SPupdNUMBER_STYLE		|\
					  SPupdBREAK_KIND		|\
					  SPupdPAGE_RESTART		|\
					  SPupdSTART_PAGE		|\
					  SPupdCOLUMN_PROPS		)

/************************************************************************/
/*									*/
/*  Document Properties.						*/
/*									*/
/************************************************************************/

#   define	DPfetFOOT_ONLY		0
#   define	DPfetEND_ONLY		1
#   define	DPfetFOOT_END_BOTH	2

#   define	DPftnPOS_SECT_END	0
#   define	DPftnPOS_DOC_END	1
#   define	DPftnPOS_BELOW_TEXT	2
#   define	DPftnPOS_PAGE_BOTTOM	3
#   define	DPftnPOS__COUNT		4

#   define	DPftnRST_CONTINUOUS	0
#   define	DPftnRST_PER_SECTION	1
#   define	DPftnRST_PER_PAGE	2
#   define	DPftnRST__COUNT		3

typedef enum NoteNumberStyle
    {
    DPftnNAR= 0,
    DPftnNALC,
    DPftnNAUC,
    DPftnNRLC,
    DPftnNRUC,
    DPftnNCHI,

    DPftn_NCOUNT
    } NoteNumberStyle;

#   define	DPupdNONE		(0)

#   define	DPupdDOCUMENT_GEOMETRY	DGupdALL

#   define	DPupdSTART_PAGE		(1<<9)
#   define	DPupdFACING_PAGES	(1<<10)

#   define	DPupdFOOTNOTE_POSITION	(1<<11)
#   define	DPupdFOOTNOTE_RESTART	(1<<12)
#   define	DPupdFOOTNOTE_STYLE	(1<<13)
#   define	DPupdFOOTNOTE_STARTNR	(1<<14)
#   define	DPupdFOOTNOTE_PROPS	( DPupdFOOTNOTE_POSITION |\
					  DPupdFOOTNOTE_RESTART |\
					  DPupdFOOTNOTE_STYLE |\
					  DPupdFOOTNOTE_STARTNR )

#   define	DPupdENDNOTE_POSITION	(1<<15)
#   define	DPupdENDNOTE_RESTART	(1<<16)
#   define	DPupdENDNOTE_STYLE	(1<<17)
#   define	DPupdENDNOTE_STARTNR	(1<<18)
#   define	DPupdENDNOTE_PROPS	( DPupdENDNOTE_POSITION |\
					  DPupdENDNOTE_RESTART |\
					  DPupdENDNOTE_STYLE |\
					  DPupdENDNOTE_STARTNR )

#   define	DPupdNOTE_PROPS		( DPupdFOOTNOTE_PROPS |\
					  DPupdENDNOTE_PROPS )

#   define	DPupdDEFTAB		(1<<19)
#   define	DPupdANSICPG		(1<<20)

#   define	DPupdALL		( DPupdDOCUMENT_GEOMETRY |\
					  DPupdSTART_PAGE |\
					  DPupdFACING_PAGES |\
					  DPupdNOTE_PROPS |\
					  DPupdDEFTAB |\
					  DPupdANSICPG )

typedef struct DocumentProperties
    {
    int			dpContainsTables;
    int			dpTabIntervalTwips;
    int			dpDefaultColor;
    int			dpDefaultFont;
    int			dpAnsiCodePage;

    int			dpStartPageNumber;
    int			dpStartFootnoteNumber;
    int			dpStartEndnoteNumber;

    unsigned int	dpHasFacingPages:1;

    unsigned int	dpFootEndNoteType:2;
    unsigned int	dpFootnotePosition:2;
    unsigned int	dpFootnotesRestart:2;
    unsigned int	dpFootnoteNumberStyle:3;
    unsigned int	dpEndnotePosition:2;
    unsigned int	dpEndnotesRestart:2;
    unsigned int	dpEndnoteNumberStyle:3;

    DocumentGeometry	dpGeometry;
    DocumentFontList	dpFontList;

    RGB8Color *		dpColors;
    int			dpColorCount;

    unsigned char **	dpAuthors;
    int			dpAuthorCount;

    struct tm		dpCreatim;
    struct tm		dpRevtim;
    struct tm		dpPrintim;

    unsigned char *	dpTitle;
    unsigned char *	dpSubject;
    unsigned char *	dpKeywords;
    unsigned char *	dpComment;
    unsigned char *	dpAuthor;
    unsigned char *	dpHlinkbase;
    } DocumentProperties;

#   define	docHasDocumentInfo( dp )		\
		( (dp)->dpTitle			||	\
		  (dp)->dpSubject		||	\
		  (dp)->dpKeywords		||	\
		  (dp)->dpComment		||	\
		  (dp)->dpHlinkbase		||	\
		  (dp)->dpAuthor		||	\
		  (dp)->dpCreatim.tm_mday != 0	||	\
		  (dp)->dpRevtim.tm_mday != 0	||	\
		  (dp)->dpPrintim.tm_mday != 0	)

/************************************************************************/
/*									*/
/*  Paragraph properties.						*/
/*									*/
/*  Unless explicitly stated, all properties except numbers in pixels	*/
/*  are properties of the paragraph.					*/
/*									*/
/*  1)  Tab stops.							*/
/*  2)  Indentations.							*/
/*  3)  White space above/after the paragraph.				*/
/*  4)  Style.								*/
/*  5)  Borders.							*/
/*									*/
/************************************************************************/

typedef struct ParagraphProperties
    {
    /*  1  */
    TabStop *		ppTabStops;
    short int		ppTabCount;

    /*  2  */
    short int		ppFirstIndentTwips;
    short int		ppLeftIndentTwips;
    short int		ppRightIndentTwips;

    /*  3  */
    short int		ppSpaceBeforeTwips;
    short int		ppSpaceAfterTwips;

    short int		ppLineSpacingTwips;

    /*  4  */
    short int		ppStyle;

    /*  5  */
    BorderProperties	ppTopBorder;
    BorderProperties	ppBottomBorder;
    BorderProperties	ppLeftBorder;
    BorderProperties	ppRightBorder;
    BorderProperties	ppBetweenBorder;
    BorderProperties	ppBoxBorder;
    BorderProperties	ppBar;

    unsigned int	ppOutlineLevel:4;

    unsigned int	ppShadingLevel:10;
    unsigned int	ppShadingPattern:4;		/*  (enum)	*/

    unsigned int	ppAlignment:3;			/*  (enum)	*/

    unsigned int	ppStartsOnNewPage:1;
    unsigned int	ppInTable:1;
    unsigned int	ppLineSpacingIsMultiple:1;

    unsigned int	ppKeepOnPage:1;
    unsigned int	ppKeepWithNext:1;
    } ParagraphProperties;

/************************************************************************/
/*									*/
/*  Masks for updating paragraph properies.				*/
/*									*/
/************************************************************************/

#   define	PPupdNONE		0x0000

#   define	PPupdLEFT_INDENT	0x0001
#   define	PPupdFIRST_INDENT	0x0002
#   define	PPupdRIGHT_INDENT	0x0004
#   define	PPupdTAB_STOPS		0x0008

#   define	PPupdNEWPAGE		0x0010
#   define	PPupdSPACE_BEFORE	0x0020
#   define	PPupdSPACE_AFTER	0x0040
#   define	PPupdTOP_BORDER		0x0080
#   define	PPupdBOTTOM_BORDER	0x0100
#   define	PPupdALIGNMENT		0x0200
#   define	PPupdLINE_SPACING	0x0400

#   define	PPupdALL		0x7fff

#   define	PPupdCOLUMNS		0x8000		/*  not real	*/

typedef enum ShapeHorizontalAttachment
    {
    SHPshaPAGE= 0,
    SHPshaMARGIN,
    SHPshaCOLUMN
    } ShapeHorizontalAttachment;

typedef enum ShapeVerticalAttachment
    {
    SHPsvaPAGE= 0,
    SHPsvaMARGIN,
    SHPsvaPARAGRAPH
    } ShapeVerticalAttachment;

typedef enum ShapeWrapStyle
    {
    SHPswsTOPBOTTOM= 1,
    SHPswsAROUND,
    SHPswsNONE,
    SHPswsTIGHTLY,
    SHPswsTHROUGH
    } ShapeWrapStyle;

typedef enum ShapeWrapSide
    {
    SHPswsBOTH= 0,
    SHPswsLEFT,
    SHPswsRIGHT,
    SHPswsLARGEST
    } ShapeWrapSide;

typedef struct ShapeProperties
    {
    int			spTwipsLeftOfAnchor;
    int			spTwipsAboveAnchor;
    int			spTwipsBelowAnchor;
    int			spTwipsRightOfAnchor;

    unsigned int	spHorizontalAttachment:2;	/*  (enum)	*/
    unsigned int	spVerticalAttachment:2;		/*  (enum)	*/
    unsigned int	spWrapStyle:3;			/*  (enum)	*/
    unsigned int	spWrapSide:2;			/*  (enum)	*/
    unsigned int	spLockAnchor:1;
    } ShapeProperties;

typedef struct ObjectData
    {
    unsigned char *	odBytes;
    int			odSize;
    } ObjectData;

typedef struct InsertedObject
    {
    int			ioKind;		/*  Kind of object.		*/
    int			ioResultKind;	/*  Kind of object.		*/
    int			ioTwipsWide;	/*  Width of object.		*/
    int			ioTwipsHigh;	/*  Height of object.		*/
    int			ioScaleX;	/*  In %.			*/
    int			ioScaleY;	/*  In %.			*/
    int			ioPixelsWide;	/*  Width of object on screen	*/
    int			ioPixelsHigh;	/*  Height of object on screen	*/
    int			ioXExtent;	/*  Of metafile picture.	*/
    int			ioYExtent;	/*  Of metafile picture.	*/
    int			ioUnitsPerInch;	/*  Only use if > 0		*/

    int			ioTopCropTwips;
    int			ioBottomCropTwips;
    int			ioLeftCropTwips;
    int			ioRightCropTwips;

    int			ioDragWide;	/*  PixelsWide during resize.	*/
    int			ioDragHigh;	/*  PixelsHigh during resize.	*/

    ObjectData		ioObjectData;
    ObjectData		ioResultData;

    unsigned char *	ioObjectName;
    unsigned char *	ioObjectClass;
    int			ioBliptag;

#   ifdef USE_MOTIF
    unsigned long	ioPixmap;
#   endif

#   ifdef USE_GTK
    void *		ioPixmap;
#   endif

    void *		ioPrivate;
    } InsertedObject;

/************************************************************************/
/*									*/
/*  Kind of field. Is an index in a descriptive array of field kinds.	*/
/*									*/
/*  Fields in the 'Ted' sense are not necessarily fileds in the 'Word'	*/
/*  sense. Everything inside the text, that can possibly result in	*/
/*  a piece of document, but is possibly under the control of the	*/
/*  application is a field in the sense of 'Ted'.			*/
/*									*/
/*  FieldKindInformation:						*/
/*  a)  The (case insensitive) string that identifies the field in the	*/
/*	field instructions. NOTE that the notion of a field is abused	*/
/*	by 'Ted' for a few things that are not fields in MS-Word.	*/
/*  b)  Some things that are a destination in RTF are stored as a field	*/
/*	in Ted. This is the control word of the destination.		*/
/*  c)  The level at which this kind of field occurs.			*/
/*  d)  Tells whether this kind of field is a field in the RTF file.	*/
/*  e)  Tells whether this kind of field is a destination in the RTF	*/
/*	file.								*/
/*  f)  Recalculate the field when it is a TEXT level field.		*/
/*  g)  When to recalculate fields in the text.				*/
/*  h)  Is the field result editable such as with a HYPERLINK or	*/
/*	readonly such as with a PAGEREF?				*/
/*									*/
/************************************************************************/

/*  f  */
typedef int (*CALCULATE_TEXT_FIELD)(
				int *				pCalculated,
				int *				pNewSize,
				char *				target,
				int				targetSize,
				struct BufferDocument *		bd,
				const struct BufferItem *	paraBi,
				const struct DocumentField *	df );

/*  g  */
#   define	FIELDdoNOTHING		0
#   define	FIELDdoNEVER		0
#   define	FIELDdoDOC_FORMATTED	(1<<0)
#   define	FIELDdoPAGE_NUMBER	(1<<1)
#   define	FIELDdoDOC_INFO		(1<<2)
#   define	FIELDdoDOC_COMPLETE	(1<<3)

typedef struct FieldKindInformation
    {
    const char *		fkiLabel;			/*  a	*/
    const char *		fkiDestinationLabel;		/*  b	*/
    int				fkiLevel;			/*  c	*/
    int				fkiIsFieldInRtf;		/*  d	*/
    int				fkiIsDestinationInRtf;		/*  e	*/
    CALCULATE_TEXT_FIELD	fkiCalculateText;		/*  f	*/
    unsigned int		fkiCalculateWhen;		/*  g	*/
    unsigned int		fkiResultEditable;		/*  h	*/
    } FieldKindInformation;

typedef enum FieldKind
    {
    DOCfkFREE= -1,
    DOCfkUNKNOWN= 0,
    DOCfkHYPERLINK,
    DOCfkXE,
    DOCfkTC,
    DOCfkTCN,
    DOCfkBOOKMARK,
    DOCfkPAGE,
    DOCfkPAGEREF,
    DOCfkNUMPAGES,
    DOCfkREF,

    DOCfkCREATEDATE,
    DOCfkSAVEDATE,
    DOCfkPRINTDATE,

    DOCfkAUTHOR,
    DOCfkCOMMENTS,
    DOCfkKEYWORDS,
    DOCfkSUBJECT,
    DOCfkTITLE,

    DOCfkFOOTNOTE,
    DOCfkCHFTN,

    DOCfk_COUNT
    } FieldKind;

typedef struct DocumentField
    {
    int			dfPage;
    FieldKind		dfKind;
    ObjectData		dfData;
    ObjectData		dfInstructions;
    struct BufferItem *	dfReferencedItem;
    } DocumentField;

typedef struct FieldInstructionsComponent
    {
    int		ficOffset;
    int		ficSize;
    } FieldInstructionsComponent;

#   define	DOCmaxBOOKMARK	39

typedef struct DocumentFieldList
    {
    DocumentField *	dflFields;
    int			dflFieldCount;
    } DocumentFieldList;

typedef struct BufferPara
    {
    unsigned char *		btString;
    unsigned int		btStrlen;

    int				btParticuleCount;
    TextParticule *		btParticules;

    int				btObjectCount;
    InsertedObject *		btObjects;

    int				btLineCount;
    TextLine *			btLines;

    int				btShapeCount;
    DrawingShape *		btShapes;

    ParagraphProperties		btProperties;

    int				btAscentTwips;
    int				btDescentTwips;
    int				btLeadingTwips;

    int				btSpaceAboveLinesTwips;
    const BorderProperties *	btBorderAboveParagraph;

    int				btSpaceBelowLinesTwips;
    const BorderProperties *	btBorderBelowParagraph;
    } BufferPara;

typedef struct HeaderFooter
    {
    struct BufferItem *	hfItem;
    int			hfPageUsedFor;
    } HeaderFooter;

typedef struct BufferSect
    {
    SectionProperties		bsProperties;

    HeaderFooter		bsHeader;
    HeaderFooter		bsFirstPageHeader;
    HeaderFooter		bsLeftPageHeader;
    HeaderFooter		bsRightPageHeader;

    HeaderFooter		bsFooter;
    HeaderFooter		bsFirstPageFooter;
    HeaderFooter		bsLeftPageFooter;
    HeaderFooter		bsRightPageFooter;

				/****************************************/
				/*  Only used for Headers and Footers:	*/
				/*  A)  Normally not set. Is set by the	*/
				/*	formatter to pass information	*/
				/*	to calculate page numbers.	*/
				/****************************************/
    int				bsUseForPage;
    const struct BufferItem *	bsUseForSectBi;
#   define			biSectHeaderFooterUseForPage \
						BIU.biuSect.bsUseForPage
				/*  A  */
#   define			biSectHeaderFooterUseForSectBi \
						BIU.biuSect.bsUseForSectBi
    } BufferSect;

typedef struct BufferRow
    {
    RowProperties		bguRowProperties;
    } BufferRow;

/************************************************************************/
/*									*/
/*  Levels of nesting.							*/
/*									*/
/************************************************************************/

typedef enum ItemLevel
    {
    DOClevANY,
    DOClevOUT,
			/****************************************/
			/*  Ignore, Garbage values.		*/
			/****************************************/
    DOClevDOC,
			/****************************************/
			/*					*/
			/*  Document.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
    DOClevSECT,
			/****************************************/
			/*					*/
			/*  Section.				*/
			/*					*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/*					*/
			/****************************************/

#	define	biSectHeader		BIU.biuSect.bsHeader
#	define	biSectFirstPageHeader	BIU.biuSect.bsFirstPageHeader
#	define	biSectLeftPageHeader	BIU.biuSect.bsLeftPageHeader
#	define	biSectRightPageHeader	BIU.biuSect.bsRightPageHeader

#	define	biSectFooter		BIU.biuSect.bsFooter
#	define	biSectFirstPageFooter	BIU.biuSect.bsFirstPageFooter
#	define	biSectLeftPageFooter	BIU.biuSect.bsLeftPageFooter
#	define	biSectRightPageFooter	BIU.biuSect.bsRightPageFooter

#	define	biSectProperties	BIU.biuSect.bsProperties

#	define	biSectDocumentGeometry	biSectProperties.spDocumentGeometry
#	define	biSectPageWideTwips	biSectProperties.spPageWideTwips
#	define	biSectPageHighTwips	biSectProperties.spPageHighTwips
#	define	biSectLeftMarginTwips	biSectProperties.spLeftMarginTwips
#	define	biSectTopMarginTwips	biSectProperties.spTopMarginTwips
#	define	biSectRightMarginTwips	biSectProperties.spRightMarginTwips
#	define	biSectBottomMarginTwips	biSectProperties.spBottomMarginTwips
#	define	biSectHeaderYTwips	biSectProperties.spHeaderYTwips
#	define	biSectFooterYTwips	biSectProperties.spFooterYTwips

#	define	biSectHasTitlePage	biSectProperties.spHasTitlePage
#	define	biSectBreakKind		biSectProperties.spBreakKind
#	define	biSectPageNumberStyle	biSectProperties.spPageNumberStyle
#	define	biSectRestartPageNumbers \
					biSectProperties.spRestartPageNumbers
#	define	biSectStartPageNumber	biSectProperties.spStartPageNumber

#	define	biSectParagraphNumbers	biSectProperties.spParagraphNumbers
#	define	biSectParagraphNumberCount \
					biSectProperties.spParagraphNumberCount

			/****************************************/

    DOClevROW,
			/****************************************/
			/*  Row.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
#	define	biRowProperties		BIU.biuRow.bguRowProperties
#	define	biRowHasTableParagraphs	biRowProperties.rpHasTableParagraphs
#	define	biRowHalfGapWidthTwips	biRowProperties.rpHalfGapWidthTwips
#	define	biRowHalfGapWidthPixels \
					biRowProperties.rpHalfGapWidthPixels
#	define	biRowLeftIndentTwips	biRowProperties.rpLeftIndentTwips
#	define	biRowLeftIndentPixels	biRowProperties.rpLeftIndentPixels
#	define	biRowHeightTwips	biRowProperties.rpHeightTwips
#	define	biRowIsTableHeader	biRowProperties.rpIsTableHeader
#	define	biRowKeepOnPage		biRowProperties.rpKeepOnPage
#	define	biRowHasHorizontalBorders \
					biRowProperties.rpHasHorizontalBorders
#	define	biRowHasVerticalBorders \
					biRowProperties.rpHasVerticalBorders
#	define	biRowCells		biRowProperties.rpCells
#	define	biRowCellCount		biRowProperties.rpCellCount
#	define	biRowAlignment		biRowProperties.rpAlignment

#	define	biRowTopBorder		biRowProperties.rpTopBorder
#	define	biRowBottomBorder	biRowProperties.rpBottomBorder
#	define	biRowLeftBorder		biRowProperties.rpLeftBorder
#	define	biRowRightBorder	biRowProperties.rpRightBorder
#	define	biRowVerticalBorder	biRowProperties.rpVerticalBorder
#	define	biRowHorizontalBorder	biRowProperties.rpHorizontalBorder

    DOClevCELL,
			/****************************************/
			/*  Cell.				*/
			/*  * biGroupChildren			*/
			/*  * biGroupChildCount			*/
			/****************************************/
	/* No.. Information is stored with the row.
	define	biCellProperties	BIU.biuGroup.BGU.bguCellProperties
	*/

    DOClevPARA,
			/********************************/
			/*  Paragraph.			*/
			/********************************/
#	define	biParaString		BIU.biuPara.btString
#	define	biParaStrlen		BIU.biuPara.btStrlen
#	define	biParaParticuleCount	BIU.biuPara.btParticuleCount
#	define	biParaParticules	BIU.biuPara.btParticules
#	define	biParaObjectCount	BIU.biuPara.btObjectCount
#	define	biParaObjects		BIU.biuPara.btObjects
#	define	biParaLineCount		BIU.biuPara.btLineCount
#	define	biParaLines		BIU.biuPara.btLines
#	define	biParaShapeCount	BIU.biuPara.btShapeCount
#	define	biParaShapes		BIU.biuPara.btShapes
#	define	biParaProperties	BIU.biuPara.btProperties
#	define	biParaInTable		biParaProperties.ppInTable
#	define	biParaLineSpacingIsMultiple \
					biParaProperties.ppLineSpacingIsMultiple
#	define	biParaStartsOnNewPage	biParaProperties.ppStartsOnNewPage
#	define	biParaFirstIndentTwips	biParaProperties.ppFirstIndentTwips
#	define	biParaLeftIndentTwips	biParaProperties.ppLeftIndentTwips
#	define	biParaRightIndentTwips	biParaProperties.ppRightIndentTwips
#	define	biParaSpaceBeforeTwips	biParaProperties.ppSpaceBeforeTwips
#	define	biParaSpaceAfterTwips	biParaProperties.ppSpaceAfterTwips
#	define	biParaLineSpacingTwips	biParaProperties.ppLineSpacingTwips
#	define	biParaAlignment		biParaProperties.ppAlignment
#	define	biParaTabStops		biParaProperties.ppTabStops
#	define	biParaTabCount		biParaProperties.ppTabCount
#	define	biParaKeepOnPage	biParaProperties.ppKeepOnPage
#	define	biParaKeepWithNext	biParaProperties.ppKeepWithNext
#	define	biParaShadingLevel	biParaProperties.ppShadingLevel
#	define	biParaShadingPattern	biParaProperties.ppShadingPattern
#	define	biParaOutlineLevel	biParaProperties.ppOutlineLevel
#	define	biParaTopBorder		biParaProperties.ppTopBorder
#	define	biParaBottomBorder	biParaProperties.ppBottomBorder
#	define	biParaLeftBorder	biParaProperties.ppLeftBorder
#	define	biParaRightBorder	biParaProperties.ppRightBorder
#	define	biParaBoxBorder		biParaProperties.ppBoxBorder
#	define	biParaBetweenBorder	biParaProperties.ppBetweenBorder
#	define	biParaBar		biParaProperties.ppBar
#	define	biParaStyle		biParaProperties.ppStyle

#	define	biParaAscentTwips	BIU.biuPara.btAscentTwips
#	define	biParaDescentTwips	BIU.biuPara.btDescentTwips
#	define	biParaLeadingTwips	BIU.biuPara.btLeadingTwips

#	define	biParaSpaceAboveLinesTwips \
					BIU.biuPara.btSpaceAboveLinesTwips
#	define	biParaBorderAboveParagraph \
					BIU.biuPara.btBorderAboveParagraph
#	define	biParaSpaceBelowLinesTwips \
					BIU.biuPara.btSpaceBelowLinesTwips
#	define	biParaBorderBelowParagraph \
					BIU.biuPara.btBorderBelowParagraph

    DOClevTEXT
			/********************************/
			/*  Handeled inside TEXT.	*/
			/********************************/
    } ItemLevel;

typedef struct BufferItem
    {
    ItemLevel			biLevel;
    struct BufferItem *		biParent;
    int				biNumberInParent;
    struct BufferItem **	biChildren;
    int				biChildCount;


    LayoutPosition		biTopPosition;
    LayoutPosition		biBelowPosition;

    union	BIU
	{
	BufferPara	biuPara;
	BufferSect	biuSect;
	BufferRow	biuRow;
	} BIU;

    } BufferItem;

/************************************************************************/
/*									*/
/*  A position in a document.						*/
/*									*/
/*  1)  Geometry is derived from the position, NOT the other way round.	*/
/*									*/
/************************************************************************/

typedef struct BufferPosition
    {
    BufferItem *	bpBi;
    int			bpLine;
    int			bpParticule;
    int			bpStroff;

    /*  1  */
    int			bpXPixels;
    LayoutPosition	bpTopPosition;
    int			bpY1Pixels;
    int			bpBaselinePixels;
    } BufferPosition;

typedef struct SelectionScope
    {
    int			ssInHeaderFooter;
#			define	DOCinBODY		0
#			define	DOCinSECT_HEADER	1
#			define	DOCinFIRST_HEADER	2
#			define	DOCinLEFT_HEADER	3
#			define	DOCinRIGHT_HEADER	4
#			define	DOCinSECT_FOOTER	(-1)
#			define	DOCinFIRST_FOOTER	(-2)
#			define	DOCinLEFT_FOOTER	(-3)
#			define	DOCinRIGHT_FOOTER	(-4)
    int			ssHeaderFooterSection;
    int			ssHeaderFooterPage;
    } SelectionScope;

typedef struct BufferSelection
    {
    BufferPosition	bsBegin;
    BufferPosition	bsEnd;
    BufferPosition	bsAnchor;
    int			bsCol0;
    int			bsCol1;
    int			bsDirection;
    } BufferSelection;

typedef struct TableRectangle
    {
    int		trSectionNumber;

    int		trCol0;
    int		trCol1;
    int		trCol11;

    int		trRow00;
    int		trRow0;
    int		trRow1;
    int		trRow11;
    } TableRectangle;

#   define	docPositionsInsideCell(b,e) \
		((b)->bpBi->biParent == (e)->bpBi->biParent)
#   define	docPositionsInsideRow(b,e) \
		((b)->bpBi->biParent->biParent == (e)->bpBi->biParent->biParent)

#   define	docSelectionInsideCell(bs) \
		docPositionsInsideCell(&(bs)->bsBegin,&(bs)->bsEnd)
#   define	docSelectionInsideRow(bs) \
		docPositionsInsideRow(&(bs)->bsBegin,&(bs)->bsEnd)

/************************************************************************/
/*									*/
/*  Styles in a document.						*/
/*									*/
/*  dsBusy is used to prevent recursion when styles are expanded.	*/
/*									*/
/************************************************************************/

typedef struct DocumentStyle
    {
    int			dsStyleNumber;
    int			dsBasedOn;
    int			dsIsCharacterStyle;
    int			dsBusy;
    ParagraphProperties	dsParagraphProperties;
    TextAttribute	dsTextAttribute;
    char *		dsName;
    } DocumentStyle;

/************************************************************************/
/*									*/
/*  The document as a whole.						*/
/*									*/
/************************************************************************/

typedef struct BufferDocument
    {
    BufferItem		bdItem;
    DocumentProperties	bdProperties;

    DocumentStyle *	bdStyles;
    int			bdStyleCount;

    DocumentFieldList	bdFieldList;
    } BufferDocument;

typedef void (*DOC_CLOSE_OBJECT)(	BufferDocument *	bd,
					BufferItem *		bi,
					TextParticule *		tp,
					void *			voiddis );

/************************************************************************/
/*									*/
/*  Statistics about a document. Used in the 'Document Properties'	*/
/*  dialog.								*/
/*									*/
/************************************************************************/

typedef struct DocumentStatistics
    {
    int		dsPageCount;
    int		dsParagraphCount;
    int		dsLineCount;
    int		dsWordCount;
    int		dsCharacterCount;
    } DocumentStatistics;

/************************************************************************/
/*									*/
/*  Field Kind Information.						*/
/*									*/
/************************************************************************/

extern const FieldKindInformation	DOC_FieldKinds[];
extern const int			DOC_FieldKindCount;

/************************************************************************/
/*									*/
/*  Routine declarations.						*/
/*									*/
/************************************************************************/

extern BufferDocument *	docPlainReadFile( SimpleInputStream *	sis,
				    const DocumentGeometry *	dg );

extern BufferDocument *	docRtfReadFile(		SimpleInputStream *	sis );
extern BufferDocument *	docLeafReadFile(	SimpleInputStream *	sis );
extern BufferDocument *	docWordReadFile(	SimpleInputStream *	sis );

extern BufferDocument * docNewFile(	const char *		fontName,
					int			fontSize,
					const DocumentGeometry * dg );

extern int docRtfSaveDocument(		SimpleOutputStream *	sos,
					BufferDocument *	bd,
					const SelectionScope *	ss,
					const BufferSelection *	bs,
					int			saveBookmarks );

extern int docRtfSaveSelectionAsLink(	SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const SelectionScope *	ss,
					const BufferSelection *	bs,
					int			asRef,
					int			asPageref,
					const unsigned char *	fileName,
					int			fileSize,
					const unsigned char *	markName,
					int			markSize );

extern int docHtmlSaveDocument(		SimpleOutputStream *	sos,
					const BufferDocument *	bd,
					const char *		filename );

extern int docPlainSaveDocument(	SimpleOutputStream *	sos,
					BufferDocument *	bd,
					const SelectionScope *	ss,
					const BufferSelection *	bs,
					int			fold,
					int			closed );

extern int docRtfSaveRuler(	SimpleOutputStream *		sos,
				const ParagraphProperties *	pp );

extern int docRtfReadRuler(	SimpleInputStream *	sis,
				ParagraphProperties *	pp );

extern int docInflateTextString(	BufferItem *    bi,
					int		by	);

extern void docInitTextLine(		TextLine *	tl );

extern TextLine * docInsertTextLine(	BufferItem *	bi,
					int		line );

extern void docCleanItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeItem(	BufferDocument *	bd,
				BufferItem *		bi );

extern void docFreeDocument( BufferDocument *	bd );

extern void docInitDocument(	BufferDocument *	bd	);

extern void docInitParagraphProperties(	ParagraphProperties *	pp );
extern void docInitDocumentProperties(	DocumentProperties *	dp );
extern void docInitSectionProperties(	SectionProperties *	sp );

extern void docInitTabStop(	TabStop *	ts );

extern void docCleanParagraphProperties(	ParagraphProperties *	pp );
extern void docCleanSectionProperties(		SectionProperties *	sp );
extern void docCleanDocumentProperties(		DocumentProperties *	dp );

extern int docCopyParagraphProperties(	ParagraphProperties *		to,
					const ParagraphProperties *	from );
extern int docCopySectionProperties(	SectionProperties *		to,
					const SectionProperties *	from );
extern int docCopyRowProperties(	RowProperties *			to,
					const RowProperties *		from );

extern int docCopyParagraphRuler(	ParagraphProperties *		to,
					const ParagraphProperties *	from );

extern BufferItem * docInsertItem(	const BufferDocument *	bd,
					BufferItem *		parent,
					int			n,
					ItemLevel		level );

extern BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biCellTo,
					int			n,
					BufferItem *		biParaFrom,
					const char *		filename );

extern BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biSectTo,
					int			n,
					BufferItem *		biRowFrom,
					const char *		filename );

extern void docDeleteItems(	BufferDocument *	bd,
				BufferItem *		bi,
				int			first,
				int			count	);

extern TextParticule *	docInsertTextParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind,
						TextAttribute	ta	);

extern TextParticule * docCopyParticule(	BufferItem *		bi,
						int			n,
						int			off,
						int			len,
						int			kind,
						const TextParticule *	from );

void docDeleteParticules	(	BufferItem *	bi,
					int		first,
					int		count	);

extern void docDeleteLines	(	BufferItem *	bi,
					int		first,
					int		count	);

extern int docParaAddTab(	ParagraphProperties *	pp,
				const TabStop *		ts );

extern void docInitPosition(		BufferPosition *	bp );
extern void docInitSelection(		BufferSelection *	bs );
extern void docInitTableRectangle(	TableRectangle *	tr );

extern int docParaReplaceText(	BufferDocument *	bd,
				BufferItem *		bi,
				int			part,
				unsigned int		stroffBegin,
				int *			pPartShift,
				int *			pStroffShift,
				unsigned int		stroffEnd,
				const unsigned char *	addedText,
				unsigned int		addedLength,
				TextAttribute		addedAttribute,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject );

extern int docSplitParaItem(	BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem *		oldBi,
				int			stroff );

extern int docSplitGroupItem(	const BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem *		oldBi,
				int			n );

extern BufferItem * docNextParagraph(	BufferItem *	bi	);
extern BufferItem * docPrevParagraph(	BufferItem *	bi	);

extern BufferItem * docNextSection(	BufferItem *	bi	);
extern BufferItem * docPrevSection(	BufferItem *	bi	);

extern int docNextPosition(	BufferPosition *	bp	);

extern int docPrevPosition(	BufferPosition *	bp,
				int			lastOne );

extern int docNextWord(		BufferPosition *	bp,
				int			lastOne );
extern int docPrevWord(		BufferPosition *	bp,
				int			lastOne );

extern int docParaBegin(	BufferPosition *	bp,
				int			lastOne );
extern int docParaEnd(		BufferPosition *	bp,
				int			lastOne );

extern int docNextLine(		TextParticule **	pTp,
				TextLine **		pTl,
				BufferPosition *	bp	);
extern int docPrevLine(		TextParticule **	pTp,
				TextLine **		pTl,
				BufferPosition *	bp	);

extern int docFirstPosition(	BufferItem *		bi,
				BufferPosition *	bp	);

extern int docLastPosition(	BufferItem *		bi,
				BufferPosition *	bp	);

extern void docSetParaSelection(	BufferSelection *	bs,
				BufferItem *		bi,
				int			direction,
				int			start,
				int			length	);

extern int docComparePositions(	const BufferPosition *	bp1,
				const BufferPosition *	bp2,
				int			mindLine );

extern int docCompareItemPositions(	const BufferItem *	bi1,
					const BufferItem *	bi2 );

extern TextParticule * docParaSpecialParticule(	BufferItem *	bi,
					int			kind,
					int			part,
					int			stroff,
					int *			pPartShift,
					int *			pStroffShift );

extern int docFindLineAndParticule(	BufferItem *		bi,
					int			stroff,
					BufferPosition *	bp ,
					int			lastOne );

extern int docFindParticule(		const BufferItem *	bi,
					int			stroff,
					int *			pPart,
					int			lastOne );

extern int docFindLineOfParticule(	const BufferItem *	bi,
					int			part,
					int *			pLine );

extern void docDeleteItem(		BufferDocument *	bd,
				    BufferItem *		bi );

extern void docListItem(	int			indent,
				const BufferItem *	bi );

extern void docListParticule(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextParticule *	tp );

extern void docListTextLine(	int			indent,
				const char *		label,
				int			n,
				const BufferItem *	bi,
				const TextLine *	tl );

extern int docReplaceSelection(	BufferDocument *	bd,
				const BufferSelection *	bs,
				int *			pPartShift,
				int *			pStroffShift,
				const unsigned char *	addedText,
				int			addedLength,
				TextAttribute		addedAttribute,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject );

extern void docInitItem(	BufferItem *		bi,
				BufferItem *		parent,
				const BufferDocument *	bd,
				int			numberInParent,
				int			level );

extern const char * docKindStr(		int			kind );
extern const char * docLevelStr(	int			level );
extern const char * docFieldKindStr(	int			kind );
extern const char * docAttributeStr(	TextAttribute		ta );

extern TextParticule *	docInsertObject(	BufferItem *		bi,
						InsertedObject **	pIo,
						int			n,
						int			off,
						TextAttribute		ta );

extern int docObjectSetData(	InsertedObject *	io,
				const unsigned char *	bytes,
				int			size );

extern int docCopyObjectData(	ObjectData *		odTo,
				const ObjectData *	odFrom );

extern int docSetResultData(	InsertedObject *	io,
				const unsigned char *	bytes,
				int			size );

extern int docSaveObjectTag(	InsertedObject *	io,
				const char *		tag,
				int			arg );

extern int docSaveResultTag(	InsertedObject *	io,
				const char *		tag,
				int			arg );

extern void docCloseItemObjects(	BufferDocument *	bd,
					BufferItem *		bi,
					void *			voidadd,
					DOC_CLOSE_OBJECT	closeObject );

extern void docCleanParticuleObject(	BufferItem *	bi,
					TextParticule *	tp );

extern InsertedObject *	docClaimObject(	int *			pNr,
					BufferItem *		bi );

extern InsertedObject * docClaimObjectCopy( BufferItem *	bi,
					int *			pNr,
					const InsertedObject *	ioFrom );

extern int docSetObjectName(	InsertedObject *	io,
				const unsigned char *	name,
				int			len );

extern int docSetObjectClass(	InsertedObject *	io,
				const unsigned char *	name,
				int			len );

extern DocumentField * docClaimField(	int *			pNr,
					DocumentFieldList *	dfl );

extern int docSetFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern int docAddToFieldInst(	DocumentField *		df,
				const unsigned char *	bytes,
				int			size );

extern void docInitRowProperties(	RowProperties *	rp );
extern void docCleanRowProperties(	RowProperties *	rp );
extern void docCleanInitRowProperties(	RowProperties *	rp );

extern void docInitCellProperties(	CellProperties *	cp );

extern int docInsertRowColumn(	RowProperties *			rp,
				int				n,
				const CellProperties *	cp );

extern void docInitBorderProperties(	BorderProperties *	bp );

extern int docAlignedColumns(	const RowProperties *	rp1,
				const RowProperties *	rp2 );

extern int docEqualRows(	const RowProperties *	rp1,
				const RowProperties *	rp2 );

extern int docDelimitTable(	BufferItem *		paraBi,
				BufferItem **		pSectBi,
				int *			pCol,
				int *			pRow0,
				int *			pRow,
				int *			pRow1 );

extern void docParagraphFrameTwips(	FormattingFrame *	ff,
					int			bottom,
					int			stripHigh,
					const BufferItem *	bi );

extern void docInitShapeProperties(	ShapeProperties *	sp );


extern int docGetBookmarkForPosition(	const BufferDocument *	bd,
					const BufferPosition *	bp,
					int *			pPart,
					int *			pEndPart,
					const char **		pMarkName,
					int *			pMarkSize );

extern DocumentField *	docFindBookmarkField( const DocumentFieldList * dfl,
					const char *		markName,
					int			markSize );

extern int docFieldGetHyperlink(		const DocumentField *	df,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldGetBookmark(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldGetPageref(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldGetRef(		const DocumentField *	df,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docGetHyperlinkForPosition(	const BufferDocument *	bd,
					const BufferPosition *	bp,
					int *			pStartPart,
					int *			pEndPart,
					const char **		pFileName,
					int *			pFileSize,
					const char **		pMarkName,
					int *			pMarkSize );

extern int docFieldSetHyperlink(	DocumentField *		df,
					const unsigned char *	file,
					int			fileSize,
					const unsigned char *	mark,
					int			markSize );

extern int docFieldSetBookmark(		DocumentField *		df,
					const unsigned char *	mark,
					int			markSize );

extern int docFieldSetRef(		DocumentField *		df,
					const unsigned char *	mark,
					int			markSize );

extern int docFieldSetPageref(		DocumentField *		df,
					const unsigned char *	mark,
					int			markSize );

extern void docInitField(		DocumentField *	df );

extern void docCleanField(		BufferDocument *	bd,
					DocumentField *		df );

extern void docCleanFieldList(		BufferDocument *	bd,
					DocumentFieldList *	dfl );
extern void docInitFieldList(		DocumentFieldList *	dfl );

DocumentField *	docClaimFieldCopy(	DocumentFieldList *	dfl,
					int *			pNr,
					const DocumentField *	dfFrom );

extern int docCopyParticuleData(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom );

extern int docCopyParticules(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		biTo,
				const BufferItem *	biFrom,
				int			partTo,
				int			partFrom,
				int			countFrom,
				int *			pParticulesInserted,
				int *			pCharactersCopied,
				const char *		refFileName );

extern int docEqualBorder(	const BorderProperties *	bp1,
				const BorderProperties *	bp2 );

extern void docInitShape(	DrawingShape *	ds );
extern void docCleanShape(	DrawingShape *	ds );

extern DrawingShape * docClaimShape(	int *			pNr,
					BufferItem *		bi );

extern void docInitStyle(	DocumentStyle *	ds );
extern void docCleanStyle(	DocumentStyle *	ds );

extern DocumentStyle * docInsertStyle(	BufferDocument *	bd,
					int			n,
					const char *		name );

extern int docParticuleInField(		BufferItem *	bi,
					int		part );
extern int docSaveParticules(	BufferItem *		bi,
				TextAttribute		ta,
				const unsigned char	inputMapping[256],
				const unsigned char *	text,
				int			len );

extern int docSaveSpecialParticule(	BufferItem *		bi,
					TextAttribute		ta,
					int			kind );

extern unsigned int docParaPropertyDifference(
				const ParagraphProperties *	pp1,
				const ParagraphProperties *	pp2,
				unsigned int			updMask );

extern void docLogRectangle(	const char *			label,
				const DocumentRectangle *	dr );

extern void docLogRectangles(	const char *			label1,
				const DocumentRectangle *	dr1,
				const char *			label2,
				const DocumentRectangle *	dr2 );

extern void docInitLayoutPosition(	LayoutPosition *	lp );

extern void docLinePositions(	BufferPosition *	bpLineBegin,
				BufferPosition *	bpLineEnd,
				const BufferItem *	bi,
				int			line );

extern void docInitDocumentStatistics(		DocumentStatistics *	ds );

extern void docCollectDocumentStatistics(	DocumentStatistics *	ds,
						const BufferDocument *	bd );

extern int docUpdParaProperties( unsigned int *		pPpChgPask,
				BufferItem *		paraBi,
				unsigned int		ppUpdMask,
				const ParagraphProperties *	ppNew );

extern int docUpdSectProperties( unsigned int *			pSpChgMask,
				SectionProperties *		to,
				unsigned int			spUpdMask,
				const SectionProperties *	from );

extern int docWhatPageHeader(	HeaderFooter **			pHf,
				BufferItem *			sectBi,
				int				page,
				const DocumentProperties *	dp );

extern int docWhatPageFooter(	HeaderFooter **			pHf,
				BufferItem *			sectBi,
				int				page,
				const DocumentProperties *	dp );

extern BufferItem * docGetSelectionRoot(	BufferDocument *	bd,
						int			nearest,
						const SelectionScope *	ss,
						const BufferSelection *	bs );

extern void docInitParagraphNumber(	ParagraphNumber *	pn );

extern TextParticule *	docInsertParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind );

extern int docCopyFieldRelative(	BufferDocument *	bdTo,
					const BufferDocument *	bdFrom,
					int *			fieldMap,
					unsigned int *		pFieldUpd,
					BufferItem *		biTo,
					const BufferItem *	biFrom,
					TextParticule *		tpTo,
					const TextParticule *	tpFrom,
					const char *		refFileName,
					int			refFileSize );

extern void docCleanObject(	InsertedObject *	io );

extern void docFieldRefreshFlags(	BufferItem *		bi,
					const BufferDocument *	bd );

extern void docAdaptBookmarkName(	int *		pLength,
					char *		mark );

extern int docParaSaveTabs(	ParagraphProperties *		to,
				const ParagraphProperties *	from );

extern void docInitDocumentList(	DocumentList *	dl );
extern void docCleanDocumentList(	DocumentList *	dl );

extern int docUpdDocProperties(	unsigned int *			pChgPask,
				DocumentProperties *		dp,
				unsigned int			updMask,
				const DocumentProperties *	dpNew );

extern int docCopyDocumentProperties(	DocumentProperties *		to,
					const DocumentProperties *	from );

extern int docAddToObjectData(		ObjectData *		od,
					const unsigned char *	bytes,
					int			size );

extern int docSetObjectData(		ObjectData *		od,
					const unsigned char *	bytes,
					int			size );

extern void docInitObjectData(		ObjectData *	od );
extern void docCleanObjectData(		ObjectData *	od );

extern int docRecalculateTextFields(	int *			pChanged,
					BufferDocument *	bd,
					BufferItem *		bi,
					unsigned int		whenMask );

extern int docRedivideStringInParticules(	BufferItem *	bi,
						int		strOff,
						int		strLen,
						int		part,
						int		partsFree,
						TextAttribute	ta );

extern int docBalanceFieldSelection(	int *			pBeginMoved,
					int *			pEndMoved,
					const BufferDocument *	bd,
					BufferPosition *	bpBegin,
					BufferPosition *	bpEnd );

extern void docInitSelectionScope(	SelectionScope *	ss );

extern int docSplitFieldInstructions(	const ObjectData *	od,
				FieldInstructionsComponent *	fic,
				int				count );

extern int docGetHeaderFooter(	HeaderFooter **			pHf,
				BufferItem **			pSectBi,
				const SelectionScope *		ss,
				const BufferSelection *		bs,
				const BufferDocument *		bd,
				int				which );

extern void docInitHeaderFooter(	HeaderFooter *		hf );
extern void docCleanHeaderFooter(	BufferDocument *	bd,
					HeaderFooter *		hf );

extern BufferItem * docMakeExternalItem( BufferDocument *		bd,
					const SectionProperties *	sp );

extern BufferItem * docInsertEmptyParagraph(	BufferDocument *	bd,
						BufferItem *		bi,
						TextAttribute		ta );

extern int docIntersectTableRectangle(	TableRectangle *		tr,
					const TableRectangle *		tr1,
					const TableRectangle *		tr2 );

extern void docExpandTableRectangleToWholeTable( TableRectangle *	tr );
extern void docExpandTableRectangleToWholeRows(	TableRectangle *	tr );
extern void docExpandTableRectangleToWholeColumns( TableRectangle *	tr );

extern int docShiftTableRectangleByRows( TableRectangle *	tr,
					int			rows );
extern int docShiftTableRectangleByColumns( TableRectangle *	tr,
					int			columns );

extern int docTableRectangleSelection(	BufferSelection *	bs,
					BufferItem **		pSectBi,
					BufferDocument *	bd,
					const SelectionScope *	ss,
					const TableRectangle *	tr );

extern int docMergeDocumentFontsIntoList( DocumentFontList *	dflTo,
					BufferDocument *	bdFrom );

extern int docFindBookmarkInDocument(	SelectionScope *	ss,
					BufferSelection *	bs,
					BufferDocument *	bd,
					const char *		markName,
					int			markSize );

extern int docSurroundTextSelectionByField( BufferSelection *	bsField,
					const BufferSelection *	bs,
					int			fieldNumber );

extern void docConstrainSelectionToOneParagraph( int *		pBeginMoved,
					int *			pEndMoved,
					BufferSelection *	bs );

extern void docUnionParaSelections(	BufferSelection *	bs,
					const BufferSelection *	bs1,
					const BufferSelection *	bs2 );

extern int *	docAllocateFieldMap(	const BufferDocument *	bdFrom );

#   endif
