/************************************************************************/
/*									*/
/*  Buffer administration routines.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	"docBuf.h"

/************************************************************************/
/*									*/
/*  Style administration.						*/
/*									*/
/************************************************************************/

void docInitStyle(	DocumentStyle *	ds )
    {
    ds->dsStyleNumber= -1;
    ds->dsBasedOn= -1;
    ds->dsIsCharacterStyle= 0;
    ds->dsBusy= 0;

    docInitParagraphProperties( &(ds->dsParagraphProperties) );

    docInitTextAttribute( &(ds->dsTextAttribute) );

    ds->dsName= (char *)0;
    }

void docCleanStyle(	DocumentStyle *	ds )
    {
    docCleanParagraphProperties( &(ds->dsParagraphProperties) );

    if  ( ds->dsName )
	{ free( ds->dsName );	}
    }

DocumentStyle * docInsertStyle(	BufferDocument *	bd,
				int			n,
				const char *		name )
    {
    DocumentStyle *	ds;
    char *		s= (char *)0;

    if  ( n > bd->bdStyleCount )
	{
	ds= (DocumentStyle *)realloc( bd->bdStyles,
		    ( n + 1 ) * sizeof( DocumentStyle ) );
	}
    else{
	ds= (DocumentStyle *)realloc( bd->bdStyles,
		    ( bd->bdStyleCount + 1 ) * sizeof( DocumentStyle ) );
	}
    if  ( ! ds )
	{ LLDEB(bd->bdStyleCount,ds); return ds; }
    bd->bdStyles= ds;

    if  ( name && name[0] )
	{
	s= strdup( name );
	if  ( ! s )
	    { XDEB(s); return (DocumentStyle *)0;	}
	}

    if  ( n == -1 )
	{
	for ( n= 0; n < bd->bdStyleCount; n++ )
	    {
	    if  (  ds[n].dsStyleNumber < 0 )
		{ break;	}
	    }
	}
    else{
	if  ( n < bd->bdStyleCount && ds[n].dsStyleNumber >= 0 )
	    { LLDEB(n,ds[n].dsStyleNumber);	}
	/* No!!
	for ( i= bd->bdStyleCount; i > n; i-- )
	    { ds[i]= ds[i-1];	}
	*/
	}

    while( bd->bdStyleCount < n )
	{ docInitStyle( ds+ bd->bdStyleCount ); bd->bdStyleCount++;	}

    ds += n;
    docInitStyle( ds );

    ds->dsStyleNumber= n;
    ds->dsName= s;

    if  ( n == bd->bdStyleCount )
	{ bd->bdStyleCount++;	}

    return ds;
    }

/************************************************************************/
/*									*/
/*  Free a BufferItem.							*/
/*									*/
/************************************************************************/

void docCleanItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    int			i;
    TextParticule *	tp;

    for ( i= 0; i < bi->biChildCount; i++ )
	{ docFreeItem( bd, bi->biChildren[i] ); }
    if  ( bi->biChildren )
	{ free( bi->biChildren );	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	    break;

	case DOClevSECT:
	    docCleanHeaderFooter( bd, &(bi->biSectHeader) );
	    docCleanHeaderFooter( bd, &(bi->biSectFirstPageHeader) );
	    docCleanHeaderFooter( bd, &(bi->biSectLeftPageHeader) );
	    docCleanHeaderFooter( bd, &(bi->biSectRightPageHeader) );

	    docCleanHeaderFooter( bd, &(bi->biSectFooter) );
	    docCleanHeaderFooter( bd, &(bi->biSectFirstPageFooter) );
	    docCleanHeaderFooter( bd, &(bi->biSectLeftPageFooter) );
	    docCleanHeaderFooter( bd, &(bi->biSectRightPageFooter) );

	    docCleanSectionProperties( &(bi->biSectProperties) );
	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    docCleanRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    tp= bi->biParaParticules;
	    for ( i= 0; i < bi->biParaParticuleCount; tp++, i++ )
		{
		if  ( tp->tpKind == DOCkindFIELDSTART	||
		      tp->tpKind == DOCkindFIELDEND	)
		    {
		    if  ( tp->tpObjectNumber < 0			||
			  tp->tpObjectNumber >=
					bd->bdFieldList.dflFieldCount	)
			{
			LDEB(bi->biParaParticules[i].tpObjectNumber);
			LDEB(bd->bdFieldList.dflFieldCount);
			continue;
			}

		    docCleanField( bd, bd->bdFieldList.dflFields+
							tp->tpObjectNumber );
		    docInitField( bd->bdFieldList.dflFields+
							tp->tpObjectNumber );
		    }
		}

	    for ( i= 0; i < bi->biParaObjectCount; i++ )
		{ docCleanObject( bi->biParaObjects+ i ); }

	    if  ( bi->biParaString )
		{ free( bi->biParaString );	}
	    if  ( bi->biParaParticules )
		{ free( bi->biParaParticules );	}
	    if  ( bi->biParaObjects )
		{ free( bi->biParaObjects );	}
	    if  ( bi->biParaLines )
		{ free( bi->biParaLines );	}
	    if  ( bi->biParaShapes )
		{ free( bi->biParaShapes );	}

	    docCleanParagraphProperties( &(bi->biParaProperties) );

	    break;
	default:
	    /*FALLTHROUGH*/
	case DOClevOUT:
	    LDEB(bi->biLevel);
	    break;
	}

    bi->biLevel= DOClevOUT;
    }

void docFreeItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    docCleanItem( bd, bi );
    free( bi );
    }

void docFreeDocument( BufferDocument *	bd )
    {
    DocumentStyle *	ds;
    int			i;

    docCleanItem( bd, &(bd->bdItem) );

    ds= bd->bdStyles;
    for ( i= 0; i < bd->bdStyleCount; ds++, i++ )
	{
	if  ( ds->dsStyleNumber < 0 )
	    { continue;	}

	docCleanStyle( ds );
	}

    if  ( bd->bdStyles )
	{ free( bd->bdStyles );	}

    docCleanFieldList( bd, &(bd->bdFieldList) );

    docCleanDocumentProperties( &(bd->bdProperties) );

    free( bd );
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferItem.						*/
/*									*/
/************************************************************************/

void docInitHeaderFooter(	HeaderFooter *	hf )
    {
    hf->hfItem= (BufferItem *)0;
    hf->hfPageUsedFor= -1;

    return;
    }

void docCleanHeaderFooter(	BufferDocument *	bd,
				HeaderFooter *		hf )
    {
    if  ( hf->hfItem )
	{ docFreeItem( bd, hf->hfItem );	}

    return;
    }

void docInitItem(	BufferItem *		bi,
			BufferItem *		parent,
			const BufferDocument *	bd,
			int			numberInParent,
			int			level )
    {
    bi->biChildren= (BufferItem **)0;
    bi->biChildCount= 0;

    switch( level )
	{
	case DOClevDOC:
	    break;

	case DOClevSECT:
	    docInitHeaderFooter( &(bi->biSectHeader) );
	    docInitHeaderFooter( &(bi->biSectFirstPageHeader) );
	    docInitHeaderFooter( &(bi->biSectLeftPageHeader) );
	    docInitHeaderFooter( &(bi->biSectRightPageHeader) );

	    docInitHeaderFooter( &(bi->biSectFooter) );
	    docInitHeaderFooter( &(bi->biSectFirstPageFooter) );
	    docInitHeaderFooter( &(bi->biSectLeftPageFooter) );
	    docInitHeaderFooter( &(bi->biSectRightPageFooter) );

	    docInitSectionProperties( &(bi->biSectProperties) );

	    if  ( bd )
		{ bi->biSectDocumentGeometry= bd->bdProperties.dpGeometry; }
	    break;

	case DOClevCELL:
	    break;

	case DOClevROW:
	    docInitRowProperties( &(bi->biRowProperties) );
	    break;

	case DOClevPARA:
	    bi->biParaStrlen= 0;
	    bi->biParaString= (unsigned char *)0;

	    bi->biParaParticuleCount= 0;
	    bi->biParaParticules= (TextParticule *)0;

	    bi->biParaLineCount= 0;
	    bi->biParaLines= (TextLine *)0;

	    bi->biParaObjectCount= 0;
	    bi->biParaObjects= (InsertedObject *)0;

	    bi->biParaShapeCount= 0;
	    bi->biParaShapes= (DrawingShape *)0;

	    bi->biParaAscentTwips= 0;
	    bi->biParaDescentTwips= 0;
	    bi->biParaLeadingTwips= 0;

	    bi->biParaSpaceAboveLinesTwips= 0;
	    bi->biParaBorderAboveParagraph= (const BorderProperties *)0;
	    bi->biParaSpaceBelowLinesTwips= 0;
	    bi->biParaBorderBelowParagraph= (const BorderProperties *)0;

	    docInitParagraphProperties( &(bi->biParaProperties) );

	    break;
	default:
	    bi->biLevel= DOClevOUT;
	    bi->biParent= (BufferItem *)0;
	    LDEB(level); return;
	}

    bi->biLevel= (ItemLevel)level;
    bi->biParent= parent;
    bi->biNumberInParent= numberInParent;

    docInitLayoutPosition( &(bi->biTopPosition) );
    docInitLayoutPosition( &(bi->biBelowPosition) );

    return;
    }

void docInitLayoutPosition(	LayoutPosition *	lp )
    {
    lp->lpPage= 0;
    lp->lpPageYTwips= 0;
    lp->lpYPixels= 0;
    }

/************************************************************************/
/*									*/
/*  Initialise a BufferDocument.					*/
/*									*/
/************************************************************************/

void docInitDocument(	BufferDocument *	bd )
    {
    docInitDocumentProperties( &(bd->bdProperties) );

    docInitItem( &(bd->bdItem), (BufferItem *)0, bd, 0, DOClevDOC );

    bd->bdStyles= (DocumentStyle *)0;
    bd->bdStyleCount= 0;

    docInitFieldList( &(bd->bdFieldList) );

    return;
    }


/************************************************************************/
/*									*/
/*  Make a new document consisting of one paragraph with one empty	*/
/*  particule.								*/
/*									*/
/************************************************************************/

BufferDocument * docNewFile(	const char *			fontFamilyName,
				int				fontSize,
				const DocumentGeometry *	dg )
    {
    BufferDocument *	bd;
    TextAttribute	ta;

    BufferItem *	bi;

    const char *	fontFamilyStyle= docFontFamilyStyle( fontFamilyName );

    docInitTextAttribute( &ta );
    ta.taFontNumber= 0;
    ta.taFontSizeHalfPoints= fontSize/ 10;

    bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docInitDocument( bd );

    bd->bdProperties.dpGeometry= *dg;

    if  ( ! docInsertFont( &(bd->bdProperties.dpFontList), 0,
					    fontFamilyStyle, fontFamilyName ) )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0;	}

    bi= docInsertItem( bd, &(bd->bdItem), -1, DOClevSECT );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;   }
    bi->biSectDocumentGeometry= *dg;

    bi= docInsertEmptyParagraph( bd, bi, ta );
    if  ( ! bi )
	{ XDEB(bi); docFreeDocument( bd ); return (BufferDocument *)0;   }

    return bd;
    }

/************************************************************************/
/*									*/
/*  Make sure there is enough space in the string of a paragraph.	*/
/*									*/
/************************************************************************/

int docInflateTextString(	BufferItem *    bi,
				int		by	)
    {
    unsigned char *	freshString;
    int			newSize;

    if  ( bi->biLevel != DOClevPARA )
	{ LLDEB(bi->biLevel,DOClevPARA); return -1;	}

    if  ( bi->biParaStrlen % 50 )
	{ newSize= bi->biParaStrlen+ by+  1; }
    else{ newSize= bi->biParaStrlen+ by+ 51; }

    freshString= (unsigned char *)realloc( bi->biParaString, newSize );
    if  ( ! freshString )
	{ LXDEB(bi->biParaStrlen,freshString); return -1; }

    bi->biParaString= freshString;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Add a Particule to a paragraph.					*/
/*									*/
/************************************************************************/

TextParticule * docCopyParticule(	BufferItem *		bi,
					int			part,
					int			off,
					int			len,
					int			kind,
					const TextParticule *	from	)
    {
    TextParticule *	tp;

    TextParticule	scratch;

    scratch= *from;

    tp= docInsertTextParticule( bi, part, off, len, kind,
						    from->tpTextAttribute );

    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    tp->tpPhysicalFont= scratch.tpPhysicalFont;

    return tp;
    }

TextParticule *	docInsertParticule(	BufferItem *	bi,
					int		n,
					int		off,
					int		len,
					int		kind )
    {
    TextParticule *	tp;

    tp= (TextParticule *)realloc( bi->biParaParticules,
		( bi->biParaParticuleCount + 1 ) * sizeof( TextParticule ) );
    if  ( ! tp )
	{ LLDEB(bi->biParaParticuleCount,tp); return tp; }
    bi->biParaParticules= tp;

    if  ( n == -1 )
	{ n= bi->biParaParticuleCount;	}
    else{
	int		i;

	for ( i= bi->biParaParticuleCount; i > n; i-- )
	    { tp[i]= tp[i-1];	}
	}

    tp += n;

    docInitTextAttribute( &tp->tpTextAttribute );

    tp->tpStroff= off;
    tp->tpStrlen= len;
    tp->tpKind= kind;
    tp->tpPhysicalFont= -1;

    tp->tpX0= 0;
    tp->tpPixelsWide= 0;

    bi->biParaParticuleCount++;

    return tp;
    }

/************************************************************************/
/*									*/
/*  Insert a text particule.						*/
/*									*/
/************************************************************************/

TextParticule *	docInsertTextParticule(	BufferItem *	bi,
					int		n,
					int		off,
					int		len,
					int		kind,
					TextAttribute	ta )
    {
    TextParticule *	tp;

    tp= docInsertParticule( bi, n, off, len, kind );
    if  ( ! tp )
	{ XDEB(tp); return tp;	}

    tp->tpTextAttribute= ta;

    return tp;
    }

/************************************************************************/
/*									*/
/*  Delete a series of particules.					*/
/*									*/
/************************************************************************/

void docDeleteParticules(	BufferItem *	bi,
				int		first,
				int		count	)
    {
    if  ( first > bi->biParaParticuleCount )
	{
	LLDEB(first,bi->biParaParticuleCount);
	first= bi->biParaParticuleCount;
	}

    if  ( first+ count > bi->biParaParticuleCount )
	{
	LLDEB(first+count,bi->biParaParticuleCount);
	count= bi->biParaParticuleCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biParaParticuleCount -= count;

    while( first < bi->biParaParticuleCount )
	{
	bi->biParaParticules[first]= bi->biParaParticules[first+ count];
	first++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Delete a series of items.					*/
/*  2)  Delete an item from its parent.					*/
/*									*/
/************************************************************************/

/*  1  */
void docDeleteItems(	BufferDocument *	bd,
			BufferItem *		bi,
			int			first,
			int			count	)
    {
    int		f;
    int		c;

    if  ( first > bi->biChildCount )
	{
	LLDEB(first,bi->biChildCount);
	first= bi->biChildCount;
	}

    if  ( first+ count > bi->biChildCount )
	{
	LLDEB(first+count,bi->biChildCount);
	count= bi->biChildCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biChildCount -= count;

    f= first; c= count;
    while( c > 0 )
	{ docFreeItem( bd, bi->biChildren[f] ); f++; c--; }

    while( first < bi->biChildCount )
	{
	bi->biChildren[first]= bi->biChildren[first+ count];
	bi->biChildren[first]->biNumberInParent= first;

	first++;
	}

    return;
    }

/*  2  */
void docDeleteItem(	BufferDocument *	bd,
			BufferItem *		bi )
    {
    if  ( bi->biParent )
	{ docDeleteItems( bd, bi->biParent, bi->biNumberInParent, 1 );	}
    else{ docFreeItem( bd, bi );					}
    }

/************************************************************************/
/*									*/
/*  Delete a series of lines.						*/
/*									*/
/************************************************************************/

void docDeleteLines	(	BufferItem *	bi,
				int		first,
				int		count	)
    {
    if  ( first > bi->biParaLineCount )
	{
	LLDEB(first,bi->biParaLineCount);
	first= bi->biParaLineCount;
	}

    if  ( first+ count > bi->biParaLineCount )
	{
	LLDEB(first+count,bi->biParaLineCount);
	count= bi->biParaLineCount- first;
	}

    if  ( count <= 0 )
	{ LDEB(count); return;	}

    bi->biParaLineCount -= count;

    while( first < bi->biParaLineCount )
	{
	bi->biParaLines[first]= bi->biParaLines[first+ count];
	first++;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Remember about a line in a Text Item.				*/
/*									*/
/************************************************************************/

void docInitTextLine(		TextLine *	tl )
    {
    tl->tlStroff= 0;
    tl->tlFirstParticule= 0;
    tl->tlStrlen= 0;
    tl->tlParticuleCount= 0;
    tl->tlWordCount= 0;

    docInitLayoutPosition( &(tl->tlTopPosition) );

    tl->tlLineAscentTwips= 0;
    tl->tlLineHeightTwips= 0;
    tl->tlLineSpacingTwips= 0;

    tl->tlLineAscentPixels= 0;
    tl->tlLineHeightTwips= 0;
    tl->tlLineSpacingTwips= 0;

    tl->tlHasPageBreak= 0;

    return;
    }

TextLine * docInsertTextLine(	BufferItem *	bi,
				int		line )
    {
    TextLine *		tl;
    int			newSize;

    if  ( bi->biParaLineCount % 10 )
	{ newSize= bi->biParaLineCount+  1; }
    else{ newSize= bi->biParaLineCount+ 11; }

    newSize *= sizeof(TextLine);

    tl= (TextLine *)realloc( bi->biParaLines, newSize );
    if  ( ! tl )
	{ LXDEB(bi->biParaLineCount,tl); return (TextLine *)0; }
    bi->biParaLines= tl;

    if  ( line == -1 )
	{ line= bi->biParaLineCount; }
    else{
	int		i;

	for ( i= bi->biParaLineCount; i > line; i-- )
	    { tl[i]= tl[i-1];	}
	}

    tl += line;

    docInitTextLine( tl );

    bi->biParaLineCount++; return tl;
    }

/************************************************************************/
/*									*/
/*  Add a new child to a parent.					*/
/*									*/
/************************************************************************/

BufferItem * docInsertItem(	const BufferDocument *	bd,
				BufferItem *		parent,
				int			n,
				ItemLevel		level	)
    {
    int			i;

    int *		countTarget;
    BufferItem ***	childTarget;
    int			newSize;

    BufferItem **	freshChildren;
    BufferItem *	bi;

    if  ( parent->biLevel != level- 1 )
	{
	SSDEB(docLevelStr(parent->biLevel),docLevelStr(level));
	return (BufferItem *)0;
	}

    switch( parent->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    countTarget= &(parent->biChildCount);
	    childTarget= &(parent->biChildren);
	    break;

	default:
	    LDEB(parent->biLevel); return (BufferItem *)0;
	}

    newSize= *countTarget;

    if  ( newSize % 10 )
	{ newSize ++;		}
    else{ newSize += 10;	}

    newSize *= sizeof(BufferItem *);

    freshChildren= (BufferItem **)realloc( *childTarget, newSize );
    if  ( ! freshChildren )
	{ LXDEB(newSize,freshChildren); return (BufferItem *)0; }
    *childTarget= freshChildren;

    bi= (BufferItem *)malloc( sizeof(BufferItem) );
    if  ( ! bi )
	{ XDEB(bi); return bi;	}


    if  ( n == -1 )
	{
	n= *countTarget;

	docInitItem( bi, parent, bd, n, level );
	}
    else{
	docInitItem( bi, parent, bd, n, level );

	for ( i= *countTarget; i > n; i-- )
	    {
	    freshChildren[i]= freshChildren[i-1];

	    freshChildren[i]->biNumberInParent= i;
	    }
	}

    freshChildren[n]= bi;

    (*countTarget)++;

    return bi;
    }

/************************************************************************/
/*									*/
/*  Make a new 'External Item' E.G. A section Header/Footer or a	*/
/*  Footnote.								*/
/*									*/
/************************************************************************/

BufferItem *	docMakeExternalItem(	BufferDocument *		bd,
					const SectionProperties *	sp )
    {
    BufferItem *	bi= (BufferItem *)malloc( sizeof(BufferItem) );

    if  ( ! bi )
	{ XDEB(bi); return (BufferItem *)0;	}

    docInitItem( bi, (BufferItem *)0, bd, 0, DOClevSECT );

    if  ( docCopySectionProperties( &(bi->biSectProperties), sp ) )
	{
	LDEB(1);
	docFreeItem( bd, bi );
	return (BufferItem *)0;
	}

    bi->biSectBreakKind= DOCsbkNONE;

    return bi;
    }

/************************************************************************/
/*									*/
/*  Make an empty paragraph. Needed at several locations.		*/
/*									*/
/************************************************************************/

BufferItem * docInsertEmptyParagraph(		BufferDocument *	bd,
						BufferItem *		bi,
						TextAttribute		ta )
    {
    if  ( bi->biLevel < DOClevSECT )
	{ LDEB(bi->biLevel); return (BufferItem *)0;	}

    if  ( bi->biLevel < DOClevROW )
	{
	bi= docInsertItem( bd, bi, -1, DOClevROW );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( bi->biLevel < DOClevCELL )
	{
	bi= docInsertItem( bd, bi, -1, DOClevCELL );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( bi->biLevel < DOClevPARA )
	{
	bi= docInsertItem( bd, bi, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}
    else{
	bi= docInsertItem( bd, bi->biParent, -1, DOClevPARA );
	if  ( ! bi )
	    { XDEB(bi); return (BufferItem *)0;   }
	}

    if  ( ! docInsertTextParticule( bi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); return (BufferItem *)0;	}

    return bi;
    }

/************************************************************************/
/*									*/
/*  Copy a paragraph to make a new one.					*/
/*									*/
/************************************************************************/

BufferItem * docCopyParaItem(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		biCellTo,
				int			n,
				BufferItem *		biParaFrom,
				const char *		refFileName )
    {
    BufferItem *	insBi;

    int			partTo= 0;
    int			partFrom= 0;
    int			particulesInserted= 0;
    int			charactersCopied= 0;

    insBi= docInsertItem( bdTo, biCellTo, n, DOClevPARA );
    if  ( ! insBi )
	{ XDEB(insBi); return insBi;	}

    if  ( docCopyParagraphProperties( &insBi->biParaProperties,
					&biParaFrom->biParaProperties ) )
	{ LDEB(1); return (BufferItem *)0;	}

    if  ( docCopyParticules( bdTo, bdFrom, fieldMap, pFieldUpd,
			insBi, biParaFrom,
			partTo, partFrom, biParaFrom->biParaParticuleCount,
			&particulesInserted, &charactersCopied, refFileName ) )
	{
	LDEB(biParaFrom->biParaParticuleCount);
	docDeleteItem( bdTo, insBi ); return (BufferItem *)0;
	}

    return insBi;
    }

BufferItem * docCopyRowItem(	BufferDocument *	bdTo,
				const BufferDocument *	bdFrom,
				int *			fieldMap,
				unsigned int *		pFieldUpd,
				BufferItem *		biSectTo,
				int			n,
				BufferItem *		biRowFrom,
				const char *		fileName )
    {
    BufferItem *	biRowTo;
    int			col;

    biRowTo= docInsertItem( bdTo, biSectTo, n, DOClevROW );
    if  ( ! biRowTo )
	{ XDEB(biRowTo); return biRowTo;	}

    if  ( docCopyRowProperties( &biRowTo->biRowProperties,
						&biRowFrom->biRowProperties ) )
	{ LDEB(1); return (BufferItem *)0;	}

    for ( col= 0; col < biRowFrom->biChildCount; col++ )
	{
	BufferItem *	biCellFrom;
	BufferItem *	biCellTo;
	int		para;

	biCellFrom= biRowFrom->biChildren[col];
	biCellTo= docInsertItem( bdTo, biRowTo, col, DOClevCELL );

	if  ( ! biCellTo )
	    { XDEB(biCellTo); return biCellTo;	}

	for ( para= 0; para < biCellFrom->biChildCount; para++ )
	    {
	    BufferItem *	biParaFrom= biCellFrom->biChildren[para];

	    if  ( ! docCopyParaItem( bdTo, bdFrom, fieldMap, pFieldUpd,
				    biCellTo, para, biParaFrom, fileName ) )
		{ LDEB(para); return (BufferItem *)0;	}
	    }
	}

    return biRowTo;
    }

/************************************************************************/
/*									*/
/*  Set a tab stop in a paragraph.					*/
/*									*/
/************************************************************************/

int docParaAddTab(		ParagraphProperties *	pp,
				const TabStop *		ts	)
    {
    TabStop *		tabs;
    int			newSize;
    int			i;

    if  ( pp->ppTabCount % 10 )
	{ newSize= pp->ppTabCount+  1; }
    else{ newSize= pp->ppTabCount+ 11; }

    newSize *= sizeof(TabStop);

    tabs= (TabStop *)realloc( pp->ppTabStops, newSize );
    if  ( ! tabs )
	{ LXDEB(newSize,tabs); return -1; }
    pp->ppTabStops= tabs;

    for ( i= pp->ppTabCount; i > 0; i-- )
	{
	if  ( tabs[i-1].tsTwips < ts->tsTwips )
	    { break;	}

	tabs[i]= tabs[i-1];
	}

    tabs[i]= *ts;

    pp->ppTabCount++; return 0;
    }

/************************************************************************/
/*									*/
/*  Find the line and particule number for a certain string position.	*/
/*									*/
/*  NOTE:	This expects the paragraph to be formatted.		*/
/*									*/
/************************************************************************/

int docFindLineAndParticule(	BufferItem *		bi,
				int			stroff,
				BufferPosition *	bp,
				int			lastOne )
    {
    int			line= 0;
    int			part= 0;

    TextLine *		tl= bi->biParaLines;
    TextParticule *	tp;

    while( line < bi->biParaLineCount		&&
    	   tl->tlStroff+ tl->tlStrlen < stroff	)
	{ line++; tl++; }

    if  ( line >= bi->biParaLineCount )
	{
	if  ( stroff == bi->biParaStrlen )
	    { line= bi->biParaLineCount- 1;	}
	else{
	    LLDEB(stroff,bi->biParaStrlen);
	    LLDEB(line,bi->biParaLineCount); return -1;
	    }
	}

    while( lastOne				&&
	   line < bi->biParaLineCount -1	&&
	   tl->tlStroff+ tl->tlStrlen == stroff	)
	{ line++; tl++; }

    part= tl->tlFirstParticule; tp= bi->biParaParticules+ part;
    while( tp->tpStroff+ tp->tpStrlen < stroff	&&
	   part < bi->biParaParticuleCount	)
	{ part++; tp++;	}

    if  ( part >= bi->biParaParticuleCount )
	{
	LLDEB(line,bi->biParaLineCount);
	LLDEB(stroff,bi->biParaStrlen);
	LLDEB(part,bi->biParaParticuleCount);
	docListItem( 0, bi );
	return -1;
	}

    while( lastOne				&&
	   part < bi->biParaParticuleCount -1	&&
	   tp->tpStroff+ tp->tpStrlen == stroff	)
	{ part++; tp++;	}

    bp->bpBi= bi;
    bp->bpLine= line;
    bp->bpParticule= part;
    bp->bpStroff= stroff;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the particule number for a certain string position.		*/
/*									*/
/*  NOTE:	This does not expect the paragraph to be formatted.	*/
/*									*/
/************************************************************************/

int docFindParticule(		const BufferItem *	bi,
				int			stroff,
				int *			pPart,
				int			lastOne )
    {
    int				part= 0;

    const TextParticule *	tp;

    part= 0; tp= bi->biParaParticules+ part;
    while( tp->tpStroff+ tp->tpStrlen < stroff	&&
	   part < bi->biParaParticuleCount	)
	{ part++; tp++;	}

    if  ( part >= bi->biParaParticuleCount )
	{
	LLDEB(stroff,bi->biParaStrlen);
	LLDEB(part,bi->biParaParticuleCount);
	docListItem( 0, bi );
	return -1;
	}

    while( lastOne				&&
	   part < bi->biParaParticuleCount -1	&&
	   tp->tpStroff+ tp->tpStrlen == stroff	)
	{ part++; tp++;	}

    *pPart= part;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the line number for a certain string particule Number.		*/
/*									*/
/*  NOTE:	This does not expect the paragraph to be formatted.	*/
/*									*/
/************************************************************************/

int docFindLineOfParticule(	const BufferItem *	bi,
				int			part,
				int *			pLine )
    {
    int			line= 0;

    const TextLine *	tl;

    line= 0; tl= bi->biParaLines+ line;
    while( tl->tlFirstParticule+ tl->tlParticuleCount <= part	&&
	   line < bi->biParaLineCount				)
	{ line++; tl++;	}

    if  ( line >= bi->biParaLineCount )
	{
	LLDEB(part,bi->biParaParticuleCount);
	docListItem( 0, bi );
	return -1;
	}

    *pLine= line;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Derive the frame for a paragraph from the page rectangle and the	*/
/*  paragraph properties.						*/
/*									*/
/*  For paragraphs inside a table cell, geometry is derived from the	*/
/*  table column.							*/
/*									*/
/************************************************************************/

void docParagraphFrameTwips(	FormattingFrame *		ff,
				int				bottom,
				int				stripHigh,
				const BufferItem *		paraBi )
    {
    const BufferItem *		sectBi;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    sectBi= paraBi;
    while( sectBi				&&
	   sectBi->biLevel != DOClevSECT	)
	{ sectBi= sectBi->biParent;	}

    if  ( ! sectBi )
	{ SXDEB(docLevelStr(paraBi->biLevel),sectBi); return;	}

    sp= &(sectBi->biSectProperties);
    dg= &(sp->spDocumentGeometry);

    ff->ffX0Geometry= dg->dgLeftMarginTwips;
    ff->ffX1Geometry= dg->dgPageWideTwips- dg->dgRightMarginTwips;

    ff->ffPageHigh= dg->dgPageHighTwips-
			    dg->dgTopMarginTwips- dg->dgBottomMarginTwips;
    ff->ffStripHigh= stripHigh;

    if  ( bottom > 0							&&
	  bottom <= dg->dgPageHighTwips- dg->dgBottomMarginTwips	)
	{ ff->ffY1= bottom;						}
    else{ ff->ffY1= dg->dgPageHighTwips- dg->dgBottomMarginTwips;	}

    if  ( paraBi->biParaInTable )
	{
	int			col0;
	int			col1;
	const BufferItem *	cellBi= paraBi->biParent;
	const BufferItem *	rowBi= cellBi->biParent;
	const RowProperties *	rp= &(rowBi->biRowProperties);

	col0= col1= cellBi->biNumberInParent;

	if  ( col1 >= rowBi->biRowCellCount )
	    { LLDEB(col1,rowBi->biRowCellCount);	}
	else{
	    const CellProperties *	cp1= rp->rpCells+ col1;

	    if  ( cp1->cpLeftInMergedRow )
		{
		while( col1 < rowBi->biRowCellCount- 1	&&
		       cp1[1].cpMergedWithLeft		)
		    { col1++; cp1++;	}
		}

	    ff->ffX1Geometry= dg->dgLeftMarginTwips+ cp1->cpRightBoundaryTwips;

	    if  ( col0 == 0 )
		{ ff->ffX0Geometry += rowBi->biRowLeftIndentTwips;	}
	    else{
		CellProperties *	cpp;

		cpp= rp->rpCells+ col0- 1;
		ff->ffX0Geometry += cpp->cpRightBoundaryTwips;
		}
	    }

	ff->ffX0Geometry += rowBi->biRowHalfGapWidthTwips;
	}

    ff->ffX0TextLines= ff->ffX0Geometry+ paraBi->biParaLeftIndentTwips;
    ff->ffX1TextLines= ff->ffX1Geometry- paraBi->biParaRightIndentTwips;

    ff->ffX0FirstLine= ff->ffX0TextLines+ paraBi->biParaFirstIndentTwips;

    return;
    }

/************************************************************************/
/*									*/
/*  Copy particules plus contents from one paragraph to another.	*/
/*									*/
/*  1)  Claim sufficient memory for the data and for the particules.	*/
/*  2)  Insert the new text into the paragraph.				*/
/*  3)  Insert the new particules into the paragraph.			*/
/*									*/
/*  4)  As this routine is the engine of the 'Paste' mechanism, make	*/
/*	links to the text itself relative.				*/
/*									*/
/************************************************************************/

int docCopyParticules(	BufferDocument *	bdTo,
			const BufferDocument *	bdFrom,
			int *			fieldMap,
			unsigned int *		pFieldUpd,
			BufferItem *		biTo,
			const BufferItem *	biFrom,
			int			partTo,
			int			partFrom,
			int			countFrom,
			int *			pParticulesInserted,
			int *			pCharactersCopied,
			const char *		refFileName )
    {
    TextParticule *		tp;
    TextParticule *		tpTo;
    const TextParticule *	tpFrom;

    int				size;
    int				stroffTo;

    int				i;

    int				replacedEmpty= 0;

    int				refFileSize= 0;

    if  ( refFileName )
	{ refFileSize= strlen( refFileName );	}

    /*  1  */
    size= ( biTo->biParaParticuleCount+ countFrom ) * sizeof( TextParticule );
    tp= (TextParticule *)realloc( biTo->biParaParticules, size );
    if  ( ! tp )
	{ LXDEB(size,tp); return -1;	}
    biTo->biParaParticules= tp;

    /*  2  */
    if  ( biTo->biParaStrlen == 0 && biTo->biParaParticuleCount == 1 )
	{ biTo->biParaParticuleCount--; replacedEmpty= 1;	}

    tpFrom= biFrom->biParaParticules+ partFrom;
    size= tpFrom[countFrom-1].tpStroff+ tpFrom[countFrom-1].tpStrlen-
							    tpFrom->tpStroff;

    if  ( docInflateTextString( biTo, size ) )
	{ LLDEB(biTo->biParaStrlen,size); return -1;	}

    if  ( partTo >= biTo->biParaParticuleCount )
	{
	partTo= biTo->biParaParticuleCount;
	stroffTo= biTo->biParaStrlen;
	}
    else{
	tp= biTo->biParaParticules+ partTo;
	stroffTo= tp->tpStroff;
	}

    if  ( biTo->biParaStrlen > 0 )
	{
	memmove( biTo->biParaString+ stroffTo+ size,
			    biTo->biParaString+ stroffTo,
			    biTo->biParaStrlen- stroffTo );
	}
    memcpy( biTo->biParaString+ stroffTo,
			    biFrom->biParaString+ tpFrom->tpStroff, size );
    biTo->biParaString[biTo->biParaStrlen+ size]= '\0';
			    
    /*  3  */
    tp= biTo->biParaParticules;
    for ( i= biTo->biParaParticuleCount- 1; i >= partTo; i-- )
	{
	tp[i+countFrom]= tp[i];
	tp[i+countFrom].tpStroff += size;
	}

    tpTo= tp+ partTo;
    for ( i= 0; i < countFrom; tpTo++, tpFrom++, i++ )
	{
	*(tpTo)= *(tpFrom);
	tpTo->tpStroff= stroffTo;

	/*  4  */
	if  ( ! refFileName || tpFrom->tpKind != DOCkindFIELDSTART )
	    {
	    if  ( docCopyParticuleData(
				bdTo, bdFrom, fieldMap, pFieldUpd,
				biTo, biFrom, tpTo, tpFrom ) )
		{ LLDEB(partTo,i); return -1;	}
	    }
	else{
	    if  ( docCopyFieldRelative( bdTo, bdFrom, fieldMap, pFieldUpd,
					    biTo, biFrom, tpTo, tpFrom,
					    refFileName, refFileSize ) )
		{ LLDEB(partTo,i); return -1;	}
	    }

	stroffTo += tpTo->tpStrlen;
	}

    biTo->biParaParticuleCount += countFrom;
    biTo->biParaStrlen += size;

    *pParticulesInserted += countFrom- replacedEmpty;
    *pCharactersCopied += size;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Save paragraph contents for readers.				*/
/*									*/
/************************************************************************/

int docSaveParticules(	BufferItem *		bi,
			TextAttribute		ta,
			const unsigned char	inputMapping[256],
			const unsigned char *	text,
			int			len	)
    {
    while( len > 0 )
	{
	const unsigned char *	particule= text;
	int			particuleLength= 0;
	unsigned char *		to;
	int			i;

	int	off= bi->biParaStrlen;

	while( len > 0 && text[0] != ' ' )
	    { text++; particuleLength++; len--; }
	while( len > 0 && text[0] == ' ' )
	    { text++; particuleLength++; len--; }

	/*
	appDebug( "%s(%3d): \"%.*s\"\n",
			    __FILE__, __LINE__, particuleLength, particule );
	*/

	if  ( docInflateTextString( bi, particuleLength ) )
	    { LLDEB(bi->biParaStrlen,particuleLength); return -1; }

	to= bi->biParaString+ off;
	for ( i= 0; i < particuleLength; i++ )
	    { *(to++)= inputMapping[*(particule++)];	}
	*to= '\0';

	bi->biParaStrlen += particuleLength;

	if  ( ! docInsertTextParticule( bi, -1, off, bi->biParaStrlen- off,
							DOCkindTEXT, ta ) )
	    { LDEB(bi->biParaParticuleCount); return -1;	}
	}

    return 0;
    }

int docSaveSpecialParticule(		BufferItem *		bi,
					TextAttribute		ta,
					int			kind )
    {
    if  ( docInflateTextString( bi, 1 ) )
	{ LDEB(bi->biParaStrlen); return -1;	}

    if  ( ! docInsertTextParticule( bi, -1, bi->biParaStrlen, 1,
							    kind, ta ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    bi->biParaString[bi->biParaStrlen++]= ' ';
    bi->biParaString[bi->biParaStrlen  ]= '\0';

    return 0;
    }

/************************************************************************/
/*									*/
/*  Statistics about a document. Used in the 'Document Properties'	*/
/*  dialog.								*/
/*									*/
/************************************************************************/

static void docCollectItemStatistics(	DocumentStatistics *	ds,
					const BufferItem *	bi )
    {
    int		i;

    switch( bi->biLevel )
	{
	case DOClevSECT:
	case DOClevDOC:
	case DOClevCELL:
	case DOClevROW:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{ docCollectItemStatistics( ds, bi->biChildren[i] ); }
	    break;

	case DOClevPARA:
	    ds->dsParagraphCount++;
	    ds->dsCharacterCount += bi->biParaStrlen;
	    ds->dsLineCount += bi->biParaLineCount;

	    for ( i= 0; i < bi->biParaLineCount; i++ )
		{ ds->dsWordCount += bi->biParaLines[i].tlWordCount;	}

	    break;

	default:
	    LDEB(bi->biLevel); return;
	}

    return;
    }


void docCollectDocumentStatistics(	DocumentStatistics *	ds,
					const BufferDocument *	bd )
    {
    docInitDocumentStatistics( ds );

    docCollectItemStatistics( ds, &(bd->bdItem) );

    ds->dsPageCount= bd->bdItem.biBelowPosition.lpPage+ 1;

    return;
    }

/************************************************************************/
/*									*/
/*  The contents of the 'listtable'					*/
/*									*/
/************************************************************************/

void docInitDocumentList(	DocumentList *	dl )
    {
    dl->dlListId= -1;
    dl->dlListTemplateId= -1;
    dl->dlListIsSimple= 0;
    dl->dlRestartForEverySection= 0;
    dl->dlListName= (char *)0;
    }

void docCleanDocumentList(	DocumentList *	dl )
    {
    if  ( dl->dlListName )
	{ free( dl->dlListName ); }
    }

