#   include	"appFrameConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>

#   include	<appWinMeta.h>
#   include	<utilPs.h>
#   include	<sioEndian.h>

#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<appDebugon.h>

#   ifndef	M_PI
#	define	M_PI	3.14159265358979323846
#   endif

# if 0
#    define	WMFDEB(x)	(x)
# else
#    define	WMFDEB(x)	/*nothing*/
# endif

# if 0
#    define	WMFLOG(x)	(x)
# else
#    define	WMFLOG(x)	/*nothing*/
# endif

/************************************************************************/
/*									*/
/*  Collect a series of points.						*/
/*									*/
/************************************************************************/

static int appMetaGetPointsPs(	DeviceContext *		dc,
				int			count,
				SimpleInputStream *	sis )
    {
    APP_POINT *		xp;

    int			done;

    xp= (APP_POINT *)realloc( dc->dcPoints, (count+ 1)* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(count,xp); return -1;	}
    dc->dcPoints= xp;

    for ( done= 0; done < count; xp++, done++ )
	{
	int	x= sioEndianGetLeInt16( sis );
	int	y= sioEndianGetLeInt16( sis );

	xp->x= DC_xViewport( x, dc );
	xp->y= DC_yViewport( y, dc );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Play a windows metafile to PostScript.				*/
/*									*/
/************************************************************************/

static void appMetaSetColorPs(		FILE *			f,
					DeviceContext *		dc,
					RGB8Color *		rgb )
    {
    if  ( dc->dcRedSet == rgb->rgb8Red		&&
	  dc->dcGreenSet == rgb->rgb8Green	&&
	  dc->dcBlueSet == rgb->rgb8Blue	)
	{ return;	}

    if  ( rgb->rgb8Red == rgb->rgb8Green	&&
	  rgb->rgb8Red == rgb->rgb8Blue		)
	{
	fprintf( f, "%g setgray\n", rgb->rgb8Red/255.0 );
	}
    else{
	fprintf( f, "%g %g %g setrgbcolor\n",
				    rgb->rgb8Red/255.0,
				    rgb->rgb8Green/255.0,
				    rgb->rgb8Blue/255.0 );
	}

    dc->dcRedSet= rgb->rgb8Red;
    dc->dcGreenSet= rgb->rgb8Green;
    dc->dcBlueSet= rgb->rgb8Blue;

    return;
    }

static void appMetaSetFontPs(		FILE *			f,
					DeviceContext *		dc,
					TextAttribute		ta )
    {
    if  ( docEqualTextAttributes( &ta, &(dc->dcTextAttributeSet) ) )
	{ return;	}

    appPsSetFont( f, "pf", ta );

    dc->dcTextAttributeSet= ta;
    }

static void appMetaSetPenPs(		FILE *			f,
					DeviceContext *		dc,
					const LogicalPen *	lp )
    {
    if  ( dc->dcPenSet.lpStyle == lp->lpStyle		&&
	  dc->dcPenSet.lpWidth == lp->lpWidth		)
	{ return;	}

    switch( lp->lpStyle )
	{
	case PS_SOLID:
	  solid:
	    fprintf( f, "%d setlinewidth ", DC_wViewport( lp->lpWidth, dc ) );
	    fprintf( f, "[] 0 setdash\n" );
	    break;

	case PS_INSIDEFRAME:
	    fprintf( f, "%d setlinewidth ", DC_wViewport( lp->lpWidth, dc ) );
	    fprintf( f, "[] 0 setdash\n" );
	    break;

	case PS_DASH:
	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    fprintf( f, "10 setlinewidth " );
	    fprintf( f, "[60 40] 0 setdash\n" );
	    break;

	case PS_DOT:
	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    fprintf( f, "10 setlinewidth " );
	    fprintf( f, "[10 20] 0 setdash\n" );
	    break;

	case PS_DASHDOT:
	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    fprintf( f, "10 setlinewidth " );
	    fprintf( f, "[60 20 10 20] 0 setdash\n" );
	    break;

	case PS_DASHDOTDOT:
	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    fprintf( f, "10 setlinewidth " );
	    fprintf( f, "[60 20 10 20 10 20] 0 setdash\n" );

	    break;

	case PS_NULL:
	    break;

	default:
	    WMFDEB(appDebug( "style= %d\n", lp->lpStyle ));
	    WMFLOG(fprintf( f, "style= %d\n", lp->lpStyle ));
	    break;
	}

    dc->dcPenSet= *lp;

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a bitmap image in the PostScript output for a metafile.	*/
/*									*/
/************************************************************************/

static int appMetaBitmapImagePs(	SimpleInputStream *	sis,
					FILE *			f,
					int			expectBytes,
					DeviceContext *		dc,
					int			srcX,
					int			srcY,
					int			srcXExt,
					int			srcYExt,
					int			dstX,
					int			dstY,
					int			dstXExt,
					int			dstYExt )
    {
    AppBitmapImage *		abi;
    BitmapDescription *		bd;
    double			twipsWide;
    double			twipsHigh;

    int				dstYExtAbs;

    if  ( srcX != 0 || srcY != 0 )
	{ LLDEB(srcX,srcY); return 0;	}

    if  ( appMetaBitmapImage( &abi, sis, expectBytes ) )
	{ LDEB(expectBytes); return -1;	}

    bd= &(abi->abiBitmap);

    if  ( srcXExt != bd->bdPixelsWide	||
	  srcYExt != bd->bdPixelsHigh	)
	{
	LLDEB(srcXExt,bd->bdPixelsWide);
	LLDEB(srcYExt,bd->bdPixelsHigh);
	appCleanBitmapImage( abi ); free( abi );
	return 0;
	}

    dstYExtAbs= dstYExt;
    if  ( dstYExtAbs < 0 )
	{ dstYExtAbs= -dstYExtAbs;	}

    switch( bd->bdUnit )
	{
	case BMunM:
	    twipsWide= ( dstXExt* TWIPS_PER_M )/ bd->bdXResolution;
	    twipsHigh= ( dstYExtAbs* TWIPS_PER_M )/ bd->bdYResolution;
	    break;

	case BMunINCH:
	    twipsWide= ( dstXExt* 20* 72 )/ bd->bdXResolution;
	    twipsHigh= ( dstYExtAbs* 20* 72 )/ bd->bdYResolution;
	    break;

	case BMunPOINT:
	    twipsWide= ( dstXExt* 20 )/ bd->bdXResolution;
	    twipsHigh= ( dstYExtAbs* 20 )/ bd->bdYResolution;
	    break;

	case BMunPIXEL:
	    twipsWide= DC_wViewport( dstXExt, dc );
	    twipsHigh= DC_hViewport( dstYExtAbs, dc );
	    break;

	default:
	    SDEB(bmunIntToString(bd->bdUnit));
	    twipsWide= DC_wViewport( dstXExt, dc );
	    twipsHigh= DC_hViewport( dstYExtAbs, dc );
	    break;
	}

    dstX= DC_xViewport( dstX, dc );
    dstY= DC_yViewport( dstY, dc );

    if  ( bmPsPrintBitmapImage( f, 1,
			    twipsWide, -twipsHigh,
			    dstX, ( dstY+ twipsHigh ), 0, 0,
			    bd->bdPixelsWide, bd->bdPixelsHigh,
			    bd, abi->abiBuffer ) )
	{
	LDEB(1);
	appCleanBitmapImage( abi ); free( abi );
	return -1;
	}

    appCleanBitmapImage( abi ); free( abi );

    return 0;
    }

static int appMetaSelectPenObjectPs(	DeviceContext *		dc,
					FILE *			f,
					int			ob )
    {
    LogicalPen *	lp= &(dc->dcObjects[ob].mfoLogicalPen);

    WMFDEB(appDebug( "SelectObject( ob= %d ) PEN Width= %d ",
							ob, lp->lpWidth ));

    WMFLOG(fprintf( f, "%% SelectObject( ob= %d ) PEN Width= %d ",
							ob, lp->lpWidth ));

    switch( lp->lpStyle )
	{
	case PS_SOLID:
	    WMFDEB(appDebug( "SOLID\n" ));
	    WMFLOG(fprintf( f, "SOLID\n" ));
	  solid:
	    dc->dcDrawBorders= 1;
	    break;

	case PS_INSIDEFRAME:
	    WMFDEB(appDebug( "INSIDEFRAME\n" ));
	    WMFLOG(fprintf( f, "INSIDEFRAME\n" ));

	    dc->dcDrawBorders= 1;
	    break;

	case PS_DASH:
	    WMFDEB(appDebug( "DASH\n" ));
	    WMFLOG(fprintf( f, "DASH\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;
	    break;

	case PS_DOT:
	    WMFDEB(appDebug( "DOT\n" ));
	    WMFLOG(fprintf( f, "DOT\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;
	    break;

	case PS_DASHDOT:
	    WMFDEB(appDebug( "DASHDOT\n" ));
	    WMFLOG(fprintf( f, "DASHDOT\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;
	    break;

	case PS_DASHDOTDOT:
	    WMFDEB(appDebug( "DASHDOTDOT\n" ));
	    WMFLOG(fprintf( f, "DASHDOTDOT\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;
	    break;

	case PS_NULL:
	    WMFDEB(appDebug( "NULL\n" ));
	    WMFLOG(fprintf( f, "NULL\n" ));
	    dc->dcDrawBorders= 0;
	    break;

	default:
	    WMFDEB(appDebug( "style= %d\n", lp->lpStyle ));
	    WMFLOG(fprintf( f, "style= %d\n", lp->lpStyle ));
	    dc->dcDrawBorders= 0;
	    break;
	}

    dc->dcPen= *lp;

    return 0;
    }

static int appMetaSelectPatternBrushObjectPs(	DeviceContext *		dc,
						FILE *			f,
						int			ob )
    {
    PatternBrush *	pb= &(dc->dcObjects[ob].mfoPatternBrush);
    AppBitmapImage *	abi= &(pb->pbAbi);
    int			bytesPerRow;

    WMFDEB(appDebug( "SelectObject( ob= %d ) ", ob ));
    WMFDEB(appDebug( "PATTERNBRUSH\n" ));

    bytesPerRow= bmPsRowStringSize( &(abi->abiBitmap),
						abi->abiBitmap.bdPixelsWide );
    if  ( bytesPerRow < 1 )
	{ LDEB(bytesPerRow); return -1;	}

    fprintf( f, "currentfile %d string readhexstring\n",
				    abi->abiBitmap.bdPixelsHigh* bytesPerRow );

    if  ( bmPsWriteBitmapData( f, 0, 0,
				abi->abiBitmap.bdPixelsWide,
				abi->abiBitmap.bdPixelsHigh,
				&(abi->abiBitmap), abi->abiBuffer ) )
	{ LDEB(1); return -1;	}

    fprintf( f, "%%\n" );
    fprintf( f, "pop /fill-data exch store\n" );

    fprintf( f, "/fill-wide %d store\n", abi->abiBitmap.bdPixelsWide );
    fprintf( f, "/fill-high %d store\n", abi->abiBitmap.bdPixelsHigh );

    fprintf( f, "/fill-cell " );
    fprintf( f, "{ " );
    fprintf( f, "gsave " );
    fprintf( f, "%d %d scale\n" ,
			    abi->abiBitmap.bdPixelsWide,
			    abi->abiBitmap.bdPixelsHigh );

    fprintf( f, "1 setgray 0 0 1 1 rectfill\n" );

    bmPsWriteImageInstructions( f,
			    &(abi->abiBitmap),
			    abi->abiBitmap.bdPixelsWide,
			    abi->abiBitmap.bdPixelsHigh,
			    "/fill-data load" );

    fprintf( f, "grestore " );
    fprintf( f, "} bind def\n" );

    dc->dcFillHatched= 0;
    dc->dcFillPattern= 1;
    dc->dcFillInsides= 1;

    return 0;
    }

static int appMetaSelectBrushObjectPs(	DeviceContext *		dc,
					FILE *			f,
					int			ob )
    {
    LogicalBrush *		lb= &(dc->dcObjects[ob].mfoLogicalBrush);

    WMFDEB(appDebug( "SelectObject( ob= %d ) ", ob ));
    WMFDEB(appDebug( "BRUSH\n" ));

    switch( lb->lbStyle )
	{
	case BS_SOLID:
	    dc->dcFillInsides= 1;
	    dc->dcFillHatched= 0;
	    dc->dcFillPattern= 0;
	    break;

	case BS_HOLLOW:
	    dc->dcFillInsides= 0;
	    dc->dcFillHatched= 0;
	    dc->dcFillPattern= 0;
	    break;

	case BS_HATCHED:
	    dc->dcFillInsides= 1;
	    dc->dcFillHatched= 1;
	    dc->dcFillPattern= 0;
	    break;

	case BS_PATTERN:
	default:
	    dc->dcFillInsides= 1;
	    dc->dcFillHatched= 0;
	    dc->dcFillPattern= 0;
	    LDEB(lb->lbStyle);
	    break;
	}

    dc->dcBrush= *lb;

    return 0;
    }

static int appMetaSelectFontObjectPs( DeviceContext *		dc,
				const char *			afmDirectory,
				FILE *				f,
				int				ob )
    {
    LogicalFont *	lf= &(dc->dcObjects[ob].mfoLogicalFont);

    int			family;
    int			face;

    AppFontFamily *	aff;
    AppFontTypeface *	aft;
    int			fontEncoding;

    DocumentFont *	df;

    df= dc->x_dcFontList.dflFonts+ lf->lfTextAttribute.taFontNumber;

    if  ( utilFontGetPsFont( &family, &face, &aff, &aft, &fontEncoding,
				    afmDirectory, df, lf->lfTextAttribute ) )
	{
	LDEB(lf->lfTextAttribute.taFontNumber);
	dc->dcAfi= (AfmFontInfo *)0;
	dc->dcFontEncoding= -1;
	return -1;
	}

    dc->dcAfi= (AfmFontInfo *)aft->aftPrintingData;
    dc->dcFontEncoding= fontEncoding;

    dc->dcFont= *lf;

    return 0;
    }

static int appMetaSelectObjectPs( DeviceContext *		dc,
				const char *			afmDirectory,
				FILE *				f,
				int				recordSize,
				SimpleInputStream *		sis )
    {
    int			ob;

    ob= sioEndianGetLeInt16( sis );

    if  ( ob < 0 || ob >= dc->dcObjectCount )
	{ LLDEB(ob,dc->dcObjectCount); return -1;	}

    switch( dc->dcObjects[ob].mfoType )
	{
	case MFtypePEN:
	    if  ( appMetaSelectPenObjectPs( dc, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;

	case MFtypeBRUSH:
	    if  ( appMetaSelectBrushObjectPs( dc, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;

	case MFtypePATTERNBRUSH:
	    if  ( appMetaSelectPatternBrushObjectPs( dc, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;

	case MFtypeFONT:
	    WMFDEB(appDebug( "SelectObject( ob= %d ) ", ob ));
	    WMFDEB(appDebug( "FONT\n" ));
	    if  ( appMetaSelectFontObjectPs( dc, afmDirectory, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;

	default:
	    WMFDEB(appDebug( "SelectObject( ob= %d ) ", ob ));
	    WMFDEB(appDebug( "type= %d\n", dc->dcObjects[ob].mfoType ));
	    break;
	}

    return 0;
    }

static int appMeta_FillPath(	FILE *			f,
				DeviceContext *		dc )
    {
    if  ( dc->dcFillPattern )
	{
	fprintf( f, "fill-pattern\n" );
	return 0;
	}

    appMetaSetColorPs( f, dc, &(dc->dcBrush.lbColor) );

    if  ( dc->dcFillHatched )
	{
	switch( dc->dcBrush.lbHatch )
	    {
	    case HS_HORIZONTAL:
		fprintf( f, "fill-horizontal\n" );
		break;
	    case HS_VERTICAL:
		fprintf( f, "fill-vertical\n" );
		break;
	    case HS_FDIAGONAL:
		fprintf( f, "fill-fdiagonal\n" );
		break;
	    case HS_BDIAGONAL:
		fprintf( f, "fill-bdiagonal\n" );
		break;
	    case HS_CROSS:
		fprintf( f, "fill-cross\n" );
		break;
	    case HS_DIAGCROSS:
		fprintf( f, "fill-diagcross\n" );
		break;
	    default:
		LDEB(dc->dcBrush.lbHatch);
		fprintf( f, "%% HATCH %d !!!!!!!!!!!\n",
						    dc->dcBrush.lbHatch );
	    }
	}
    else{ fprintf( f, "fill\n" );	}

    return 0;
    }


static int appMeta_PolyLinePs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		count;
    int		done;

    int		x0;
    int		y0;

    char *	command;

    appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
    appMetaSetPenPs( f, dc, &(dc->dcPen) );

    count= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("PolyLine( count=%d, ... )\n", count ));

    if  ( 2* count != recordSize- 4 )
	{ LLDEB(2*count,recordSize-4); return -1;	}

    command= "bp"; x0= y0= 0;

    for ( done= 0; done < count; done++ )
	{
	int	x= sioEndianGetLeInt16( sis );
	int	y= sioEndianGetLeInt16( sis );

	x= DC_xViewport( x, dc );
	y= DC_yViewport( y, dc );

	fprintf( f, "%d %d %s", x- x0, y- y0, command );

	if  ( done % 8 == 7 )
	    { putc( '\n', f );		}
	else{ putc( ' ', f );		}

	command= "rl"; x0= x; y0= y;
	}

    fprintf( f, "stroke\n" );

    return 0;
    }

static void appMetaDrawPolygonPs(	FILE *			f,
					DeviceContext *		dc,
					int			count )
    {
    int		done;
    char *	command;
    int		x0;
    int		y0;


    if  ( dc->dcFillInsides )
	{
	command= "bp"; x0= 0; y0= 0;

	for ( done= 0; done < count; done++ )
	    {
	    fprintf( f, "%d %d %s",
				dc->dcPoints[done].x- x0,
				dc->dcPoints[done].y- y0,
				command );

	    if  ( done % 8 == 7 )
		{ putc( '\n', f );		}
	    else{ putc( ' ', f );		}

	    command= "rl"; x0= dc->dcPoints[done].x; y0= dc->dcPoints[done].y;
	    }

	fprintf( f, "closepath\n" );

	appMeta_FillPath( f, dc );
	}

    if  ( dc->dcDrawBorders )
	{
	command= "bp"; x0= 0; y0= 0;

	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	for ( done= 0; done < count; done++ )
	    {
	    fprintf( f, "%d %d %s",
				dc->dcPoints[done].x- x0,
				dc->dcPoints[done].y- y0,
				command );

	    if  ( done % 8 == 7 )
		{ putc( '\n', f );		}
	    else{ putc( ' ', f );		}

	    command= "rl"; x0= dc->dcPoints[done].x; y0= dc->dcPoints[done].y;
	    }

	fprintf( f, "stroke\n" );
	}

    return;
    }

static int appMeta_PolygonPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		count;

    count= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("Polygon( count=%d, ... )\n", count ));

    if  ( appMetaGetPointsPs( dc, count, sis ) )
	{ LDEB(count); return -1;	}

    appMetaDrawPolygonPs( f, dc, count );

    return 0;
    }

static void appMeta_RectPathPs(		FILE *		f,
					int		x0,
					int		y0,
					int		x1,
					int		y1 )
    {
    int		dx= x1- x0;
    int		dy= y1- y0;

    fprintf( f, "%d %d bp ",  x0,  y0 );
    fprintf( f, "%d %d rl ",  dx,   0 );
    fprintf( f, "%d %d rl ",   0,  dy );
    fprintf( f, "%d %d rl ", -dx,   0 );

    fprintf( f, "closepath\n" );

    return;
    }

static int appMeta_RectanglePs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    LogicalPen *	lp= &(dc->dcPen);

    int			y0;
    int			x0;
    int			y1;
    int			x1;

    int			swap;

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("Rectangle(%d..%d,%d..%d)\n", x0,x1,y0,y1));
    WMFLOG(fprintf( f, "%% Rectangle(%d..%d,%d..%d)\n", x0,x1,y0,y1));

    x0= DC_xViewport( x0, dc );
    y0= DC_yViewport( y0, dc );
    x1= DC_xViewport( x1, dc );
    y1= DC_yViewport( y1, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    if  ( lp->lpStyle == PS_INSIDEFRAME )
	{
	int	width= 1;

	x0 += ( width+ 1 )/ 2;
	x1 -= ( width+ 1 )/ 2;
	y0 += ( width+ 1 )/ 2;
	y1 -= ( width+ 1 )/ 2;
	}

    if  ( dc->dcFillInsides )
	{
	appMeta_RectPathPs( f, x0, y0, x1, y1 );
	appMeta_FillPath( f, dc );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	appMeta_RectPathPs( f, x0, y0, x1, y1 );
	fprintf( f, "stroke\n" );
	}

    return 0;
    }

static int appMeta_LineToPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		x0= dc->dcX;
    int		y0= dc->dcY;
    int		y1;
    int		x1;

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("LineTo( x:%d-> %d, y:%d-> %d )\n", x0, x1, y0, y1 ));
    WMFLOG(fprintf( f, "%% LineTo( x:%d-> %d, y:%d-> %d )\n", x0, x1, y0, y1 ));

    dc->dcX= x1;
    dc->dcY= y1;

    x0= DC_xViewport( x0, dc );
    y0= DC_yViewport( y0, dc );
    x1= DC_xViewport( x1, dc );
    y1= DC_yViewport( y1, dc );

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	fprintf( f, "%d %d bp ", x0, y0 );
	fprintf( f, "%d %d rl stroke\n", x1- x0, y1- y0 );
	}

    return 0;
    }

static void appMetaShowStringPs(	FILE *			f,
					int			x0,
					int			y0,
					int			count,
					DeviceContext *		dc )
    {
    AfmFontInfo *	afi= dc->dcAfi;
    int			encoding= dc->dcFontEncoding;
    LogicalFont *	lf= &(dc->dcFont);

    int			width;

    AfmBBox		abb;

    const int		withKerning= 0;
    int			fontSizeTwips;

    if  ( ! afi )
	{ XDEB(afi); return;	}

    fontSizeTwips= 10* lf->lfTextAttribute.taFontSizeHalfPoints;
    width= psCalculateStringExtents( &abb, dc->dcString, count,
				fontSizeTwips, withKerning, encoding, afi );

    switch( dc->dcTextAlignment & 0x01 )
	{
	case TA_NOUPDATECP:
	    break;

	case TA_UPDATECP:
	    x0= DC_xViewport( dc->dcX, dc );
	    y0= DC_yViewport( dc->dcY, dc );
	    break;

	default:
	    XDEB(dc->dcTextAlignment & 0x01 );
	    break;
	}

    switch( dc->dcTextAlignment & 0x06 )
	{
	case TA_LEFT:
	    break;

	case TA_RIGHT:
	    x0 -= width;
	    break;

	case TA_CENTER:
	    x0 -= width/ 2;
	    break;

	default:
	    XDEB(dc->dcTextAlignment & 0x06);
	    break;
	}

    switch( dc->dcTextAlignment & 0x18 )
	{
	case TA_TOP:
	    y0 += ( fontSizeTwips* afi->afiAscender+ 500 )/ 1000;
	    break;

	case TA_BOTTOM:
	    y0 -= ( fontSizeTwips* afi->afiDescender+ 500 )/ 1000;
	    break;

	case TA_BASELINE:
	    break;

	default:
	    WMFDEB(appDebug(" 0x%x", dc->dcTextAlignment & 0x18 ));
	    break;
	}

    appMetaSetFontPs( f, dc, dc->dcFont.lfTextAttribute );
    appMetaSetColorPs( f, dc, &(dc->dcTextColor) );

    if  ( dc->dcFont.lfOrientation != 0 )
	{
	fprintf( f, "gsave %d %d translate %g rotate\n",
				x0, y0, -0.1* dc->dcFont.lfOrientation );

	x0= y0= 0;
	}

    fprintf( f, "(" );
    appPsPrintString( f, dc->dcString, count );
    fprintf( f, ") %d %d mvs\n", x0, y0 );

    if  ( dc->dcFont.lfTextAttribute.taIsUnderlined )
	{
	int		h;

	y0 -= (int)( ( fontSizeTwips* afi->afiUnderlinePosition+ 500 )/ 1000 );
	h= (int)( ( fontSizeTwips* afi->afiUnderlineThickness+ 500 )/ 1000 );

	fprintf( f, "%d %d %d %d rectfill\n", x0, y0, width, h );
	}

    if  ( dc->dcFont.lfOrientation != 0 )
	{ fprintf( f, "grestore\n" );	}

    return;
    }

static int appMeta_ExtTextOutPs(	SimpleInputStream *	sis,
					FILE *			f,
					int			recordSize,
					DeviceContext *		dc )
    {
    int		x0;
    int		y0;
    int		count;
    int		style;

    int		h1;
    int		w1;
    int		y1;
    int		x1;

    if  ( appMeta_ExtTextOut( sis, recordSize, &x0, &y0, &count, &style,
						&x1, &y1, &w1, &h1, dc ) )
	{ LDEB(1); return -1;	}

    x0= DC_xViewport( x0, dc );
    y0= DC_yViewport( y0, dc );

    x1= DC_xViewport( x1, dc );
    y1= DC_yViewport( y1, dc );

    w1= DC_wViewport( w1, dc );
    h1= DC_hViewport( h1, dc );

    appMetaShowStringPs( f, x0, y0, count, dc );

    return 0;
    }

static int appMeta_TextOutPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		x0;
    int		y0;
    int		count;

    if  ( appMeta_TextOut( sis, recordSize, &x0, &y0, &count, dc ) )
	{ LDEB(1); return -1;	}

    x0= DC_xViewport( x0, dc );
    y0= DC_yViewport( y0, dc );

    appMetaShowStringPs( f, x0, y0, count, dc );

    return 0;
    }

static int appMeta_PatBltPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    long	rop;

    int		h0;
    int		w0;
    int		y0;
    int		x0;

    int		x1;
    int		y1;

    int		swap;

    rop= sioEndianGetLeInt32( sis );
    h0= sioEndianGetLeInt16( sis );
    w0= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("PatBlt([%d+%d,%d+%d], rop=0x%lx )\n", x0,w0,y0,h0, rop ));

    x1= x0+ w0;
    y1= y0+ h0;

    x0= DC_xViewport( x0, dc );
    y0= DC_yViewport( y0, dc );

    x1= DC_xViewport( x1, dc );
    y1= DC_yViewport( y1, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    if  ( dc->dcFillInsides )
	{
	appMeta_RectPathPs( f, x0, y0, x1, y1 );
	appMeta_FillPath( f, dc );
	}

    return 0;
    }

static void appMetaGetArcPs(	SimpleInputStream *	sis,
				DeviceContext *		dc,
				char *			fun,
				int *			pY0,
				int *			pX0,
				int *			pY1,
				int *			pX1,
				int *			pYs,
				int *			pXs,
				double *		pas,
				double *		pae )

    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;
    int		ye;
    int		xe;

    int		ym;
    int		xm;
    int		h;
    int		w;

    double	as;
    double	ae;

    int		swap;

    ye= sioEndianGetLeInt16( sis );
    xe= sioEndianGetLeInt16( sis );
    ys= sioEndianGetLeInt16( sis );
    xs= sioEndianGetLeInt16( sis );

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("%s(%d..%d,%d..%d, (%d,%d), (%d,%d)\n", fun,
					    x0,x1,y0,y1, xs,ys,xe,ye));
    y0= DC_yViewport( y0, dc );
    x0= DC_xViewport( x0, dc );
    y1= DC_yViewport( y1, dc );
    x1= DC_xViewport( x1, dc );

    ys= DC_yViewport( ys, dc );
    xs= DC_xViewport( xs, dc );
    ye= DC_yViewport( ye, dc );
    xe= DC_xViewport( xe, dc );

    ym= ( y1+ y0 )/2;
    xm= ( x1+ x0 )/2;

    if  ( x1 < x0 )
	{
	xs= 2* xm- xs;
	xe= 2* xm- xe;
	swap= x0; x0= x1; x1= swap;
	}
    if  ( y1 < y0 )
	{
	ys= 2* ym- ys;
	ye= 2* ym- ye;
	swap= y0; y0= y1; y1= swap;
	}

    h= y1- y0;
    w= x1- x0;

    as= -atan2(	(double) w* ( ys- ym ), (double) h* ( xs- xm ) );
    ae= -atan2(	(double) w* ( ye- ym ), (double) h* ( xe- xm ) );

    xs= (int)( ( ( x1- x0 )/ 2 )* cos( as )+ xm );
    ys= (int)( ( ( x1- x0 )/ 2 )* sin( as ) );

    as= ( 180* as )/ M_PI;
    ae= ( 180* ae )/ M_PI;

    *pY0= y0;
    *pX0= x0;
    *pY1= y1;
    *pX1= x1;
    *pYs= ys;
    *pXs= xs;
    *pas= as;
    *pae= ae;

    return;
    }

static int appMeta_ArcPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;

    double	as;
    double	ae;

    appMetaGetArcPs( sis, dc, "Arc", 
			    &y0, &x0, &y1, &x1, &ys, &xs, &as, &ae );

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y0- y1)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	fprintf( f, "%d %d bp ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc stroke\n",
					( x0+ x1 )/ 2, (x1- x0)/2, as, ae );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_PiePs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;

    double	as;
    double	ae;

    appMetaGetArcPs( sis, dc, "Pie", 
			    &y0, &x0, &y1, &x1, &ys, &xs, &as, &ae );

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y0- y1)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcFillInsides )
	{
	fprintf( f, "%d %d bp ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc\n",
					( x0+ x1 )/ 2, (x1- x0)/2, as, ae );
	fprintf( f, "%d  0 lineto ", ( x0+ x1 )/ 2 );
	fprintf( f, "%d %d lineto closepath\n", xs, ys );

	appMeta_FillPath( f, dc );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	fprintf( f, "%d %d bp ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc ",
					( x0+ x1 )/ 2, (x1- x0)/2, as, ae );
	fprintf( f, "%d  0 lineto ", ( x0+ x1 )/ 2 );
	fprintf( f, "%d %d lineto stroke\n", xs, ys );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_ChordPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;

    double	as;
    double	ae;

    appMetaGetArcPs( sis, dc, "Chord", 
			    &y0, &x0, &y1, &x1,
			    &ys, &xs, &as, &ae );

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y0- y1)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	fprintf( f, "%d %d bp ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc %d %d lineto stroke\n",
					( x0+ x1 )/ 2, (x1- x0)/2,
					as, ae, xs, ys );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_EllipsePs(	SimpleInputStream *	sis,
				FILE *			f,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		swap;

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("Ellipse(%d..%d,%d..%d)\n", x0,x1,y0,y1));

    x0= DC_xViewport( x0, dc );
    y0= DC_yViewport( y0, dc );
    x1= DC_xViewport( x1, dc );
    y1= DC_yViewport( y1, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y1- y0)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcFillInsides )
	{
	fprintf( f, "%d 0 bp ", x1 );
	fprintf( f, "%d 0 %d 0 360 arc closepath\n",
					    ( x0+ x1 )/ 2, (x1- x0)/2 );

	appMeta_FillPath( f, dc );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	fprintf( f, "%d 0 bp ", x1 );
	fprintf( f, "%d 0 %d 0 360 arc stroke\n",
					    ( x0+ x1 )/ 2, (x1- x0)/2 );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_RoundRectPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		h;
    int		w;

    int		swap;

    int		top;
    int		bottom;
    int		radius;

    h= sioEndianGetLeInt16( sis );
    w= sioEndianGetLeInt16( sis );

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(appDebug("RoundRect(%d..%d,%d..%d, w= %d,h= %d\n",
					    x0,x1,y0,y1, w, h ));

    y0= DC_yViewport( y0, dc );
    x0= DC_xViewport( x0, dc );
    y1= DC_yViewport( y1, dc );
    x1= DC_xViewport( x1, dc );

    h= DC_hViewport( h, dc );
    w= DC_wViewport( w, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    if  ( h < 0 )
	{ h= -h;	}
    if  ( w < 0 )
	{ w= -w;	}

    top=    w* ( y1- y0 )/ ( 2* h );
    bottom= w* ( y0- y1 )/ ( 2* h );
    radius= w/ 2;

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
					    (double)h/ w, ( y0+ y1 )/ 2 );

    if  ( dc->dcFillInsides )
	{
	fprintf( f, "%d %d bp\n", x0, bottom+ radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x0, top, x0+ radius, top, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, top, x1, top- radius, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, bottom, x1- radius, bottom, radius );
	fprintf( f, "%d %d %d %d %d arct closepath\n",
				    x0, bottom, x0, bottom+ radius, radius );

	appMeta_FillPath( f, dc );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, dc, &(dc->dcPen.lpColor) );
	appMetaSetPenPs( f, dc, &(dc->dcPen) );

	fprintf( f, "%d %d bp\n", x0, bottom+ radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x0, top, x0+ radius, top, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, top, x1, top- radius, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, bottom, x1- radius, bottom, radius );
	fprintf( f, "%d %d %d %d %d arct stroke\n",
				    x0, bottom, x0, bottom+ radius, radius );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_PolyPolygonPs(	SimpleInputStream *	sis,
					FILE *			f,
					int			recordSize,
					DeviceContext *		dc )
    {
    int			count;
    int			i;

    int *		counts;

    if  ( appMeta_GetCounts( sis, recordSize, &count, &counts, dc ) )
	{ LDEB(1); return -1;	}

    WMFDEB(appDebug("PolyPolygon( count=%d, ... )\n", count ));

    for ( i= 0; i < count; i++ )
	{
	if  ( appMetaGetPointsPs( dc, counts[i], sis ) )
	    { LDEB(counts[i]); return -1;	}

	appMetaDrawPolygonPs( f, dc, counts[i] );
	}

    return 0;
    }

int appMetaPlayFilePs(	FILE *				f,
			SimpleInputStream *		sis,
			const char *			afmDirectory,
			int				mapMode,
			int				xWinExt,
			int				yWinExt,
			int				twipsWide,
			int				twipsHigh )
    {
    int			ob;

    {
    int			fileType;
    int			headerSize;
    int			windowsVersion;
    long		fileSize;
    int			objectCount;
    long		maxRecordSize;
    int			parameterCount;

    DeviceContext	dc;

    if  ( sioInGetCharacter( sis ) == EOF )
	{ return 0;	}
    sioInUngetLastRead( sis );

    fileType= sioEndianGetLeInt16( sis );
    headerSize= sioEndianGetLeInt16( sis );
    windowsVersion= sioEndianGetLeInt16( sis );
    fileSize= sioEndianGetLeInt32( sis );
    objectCount= sioEndianGetLeInt16( sis );
    maxRecordSize= sioEndianGetLeInt32( sis );
    parameterCount= sioEndianGetLeInt16( sis );

    if  ( appMetaInitDeviceContext( &dc, objectCount, mapMode,
				    xWinExt, yWinExt,
				    twipsWide, twipsHigh,
				    twipsWide, twipsHigh ) )
	{ LDEB(objectCount); return -1;	}

    WMFDEB(appDebug( "PS METAFILE ...\n" ));

    for (;;)
	{
	long		rop;

	int		x0;
	int		y0;
	int		w0;
	int		h0;

	int		x1;
	int		y1;
	int		w1;
	int		h1;

	int		done;
	int		ignore;

	long		recordSize;
	int		function;

	recordSize= sioEndianGetLeInt32( sis );
	function= sioEndianGetLeInt16( sis );

	if  ( recordSize < 0 || function < 0 )
	    { LLDEB(recordSize,function); return -1;	}

	switch( function )
	    {
	    case WINMETA_OffsetWindowOrg:
		if  ( appMetaOffsetWindowOrg( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetWindowOrg:
		if  ( appMetaSetWindowOrg( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetWindowExt:
		if  ( appMetaSetWindowExt( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetViewportOrg:
		if  ( appMetaSetViewportOrg( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetViewportExt:
		if  ( appMetaSetViewportExt( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SaveDC:
		if  ( appMetaSaveDC( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_RestoreDC:
		if  ( appMetaRestoreDC( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_OffsetClipRgn:
		if  ( appMetaOffsetClipRgn( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_ExcludeClipRect:
		if  ( appMetaExcludeClipRect( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_IntersectClipRect:
		if  ( appMetaIntersectClipRect( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetMapperFlags:
		rop= sioEndianGetLeInt32( sis );
		WMFDEB(appDebug("WINMETA_SetMapperFlags(0x%lx)\n",rop));
		continue;

	    case WINMETA_StretchBlt:
		rop= sioEndianGetLeInt32( sis );
		h0= sioEndianGetLeInt16( sis );
		w0= sioEndianGetLeInt16( sis );
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );
		h1= sioEndianGetLeInt16( sis );
		w1= sioEndianGetLeInt16( sis );
		y1= sioEndianGetLeInt16( sis );
		x1= sioEndianGetLeInt16( sis );

		WMFDEB(appDebug("StretchBlt([%d+%d,%d+%d]->[%d+%d,%d+%d],..)\n",
				x0,w0,y0,h0, x1,w1,y1,h1 ));

		if  ( appMetaBitmapImagePs( sis, f, 2*(recordSize-3-2-8*1),
					&dc, x0, y0, w0, h0, x1, y1, w1, h1 ) )
		    { LDEB(1); return -1; }

		continue;
	    case WINMETA_StretchDIBits:
		rop= sioEndianGetLeInt32( sis );
		ignore= sioEndianGetLeInt16( sis );
		/*  source	*/
		h0= sioEndianGetLeInt16( sis );
		w0= sioEndianGetLeInt16( sis );
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );
		/*  destination	*/
		h1= sioEndianGetLeInt16( sis );
		w1= sioEndianGetLeInt16( sis );
		y1= sioEndianGetLeInt16( sis );
		x1= sioEndianGetLeInt16( sis );

		WMFDEB(appDebug(
			"StretchDIBits([%d+%d,%d+%d]->[%d+%d,%d+%d],..)\n",
			x0,w0,y0,h0, x1,w1,y1,h1 ));

		if  ( appMetaBitmapImagePs( sis, f, 2*(recordSize-3-2-9*1),
					&dc, x0, y0, w0, h0, x1, y1, w1, h1 ) )
		    { LDEB(1); return -1; }
		continue;

	    case WINMETA_SelectObject:
		if  ( appMetaSelectObjectPs( &dc, afmDirectory, f,
							recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreateBrushIndirect:
		if  ( appMetaCreateBrushIndirect( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreatePenIndirect:
		if  ( appMetaCreatePenIndirect( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreatePatternBrush:
		if  ( appMetaCreatePatternBrush( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreatePalette:
		if  ( appMetaCreatePalette( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreateFontIndirect:
		if  ( appMetaCreateFontIndirect( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_DeleteObject:

		ob= sioEndianGetLeInt16( sis );

		if  ( ob < 0 || ob >= dc.dcObjectCount )
		    { LLDEB(ob,dc.dcObjectCount); return -1;	}

		appMetaCleanObject( &(dc.dcObjects[ob]) );

		dc.dcObjects[ob].mfoType= MFtypeFREE;
		continue;

	    case WINMETA_SetBkColor:
		if  ( appMetaSetBkColor( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}

		continue;

	    case WINMETA_SetTextColor:
		if  ( appMetaSetTextColor( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}

		continue;

	    case WINMETA_SetBkMode:
		WMFDEB(appDebug("SetBkMode(...)\n"));
		if  ( recordSize == 4 )
		    {
		    dc.dcBkMode= sioEndianGetLeInt16( sis );
		    continue;
		    }
		if  ( recordSize == 5 )
		    {
		    dc.dcBkMode= sioEndianGetLeInt32( sis );
		    continue;
		    }
		XLDEB(function,recordSize); return -1;

	    case WINMETA_SetTextCharExtra:
		WMFDEB(appDebug("SetTextCharExtra(...)\n"));
		dc.dcExtraTextSpacing= sioEndianGetLeInt16( sis );
		continue;

	    case WINMETA_SetTextAlign:
		if  ( appMetaSetTextAlign( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_SetTextJustification:
		WMFDEB(appDebug("SetTextJustification(...)\n"));

		dc.dcJustificationAmount= sioEndianGetLeInt16( sis );
		dc.dcJustificationSpaces= sioEndianGetLeInt16( sis );
		continue;

	    case WINMETA_SetStretchBltMode:
		switch( recordSize )
		    {
		    case 4:
			dc.dcStretchBltMode= sioEndianGetLeInt16( sis );
			break;
		    case 5:
			dc.dcStretchBltMode= sioEndianGetLeInt32( sis );
			break;
		    default:
			LDEB(recordSize); return -1;
		    }

		WMFDEB(appDebug("SetStretchBltMode(%ld)\n",
						    dc.dcStretchBltMode ));
		continue;

	    case WINMETA_PolyLine:
		if  ( appMeta_PolyLinePs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Polygon:
		if  ( appMeta_PolygonPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Rectangle:
		if  ( appMeta_RectanglePs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_LineTo:
		if  ( appMeta_LineToPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_MoveTo:
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );

		WMFDEB(appDebug("MoveTo( x=%d, y=%d )\n", x0, y0 ));

		dc.dcX= x0;
		dc.dcY= y0;

		continue;

	    case WINMETA_ExtTextOut:
		if  ( appMeta_ExtTextOutPs( sis, f, recordSize, &dc ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_TextOut:
		if  ( appMeta_TextOutPs( sis, f, recordSize, &dc ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_Escape:
		WMFDEB(appDebug("Escape(...)\n"));
		goto skipArguments;

	    case WINMETA_PatBlt:
		if  ( appMeta_PatBltPs( sis, f, recordSize, &dc ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_SetPolyFillMode:
		WMFDEB(appDebug("SetPolyFillMode(...)\n"));
		goto skipArguments;

	    case WINMETA_Pie:
		if  ( appMeta_PiePs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Arc:
		if  ( appMeta_ArcPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Chord:
		if  ( appMeta_ChordPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Ellipse:
		if  ( appMeta_EllipsePs( sis, f, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_RoundRect:
		if  ( appMeta_RoundRectPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetMapMode:
		appMetaSetMapMode( &dc, recordSize, sis );
		continue;

	    case WINMETA_SetROP2:
		if  ( recordSize == 5 )
		    {
		    dc.dcROP2= sioEndianGetLeInt32( sis );
		    WMFDEB(appDebug("SetROP2(%ld)\n", (long)dc.dcROP2 ));
		    continue;
		    }
		if  ( recordSize == 4 )
		    {
		    dc.dcROP2= sioEndianGetLeInt16( sis );
		    WMFDEB(appDebug("SetROP2(%ld)\n", (long)dc.dcROP2 ));
		    continue;
		    }
		XLDEB(function,recordSize); return -1;

	    case WINMETA_SelectPalette:
		ob= sioEndianGetLeInt16( sis );
		WMFDEB(appDebug("SelectPalette(%d)\n", ob ));
		continue;

	    case WINMETA_RealizePalette:
		WMFDEB(appDebug("RealizePalette()\n"));
		continue;

	    case WINMETA_PolyPolygon:
		if  ( appMeta_PolyPolygonPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_ScaleViewportExt:
		appMetaScaleViewportExt( &dc, recordSize, sis );
		continue;

	    case WINMETA_ScaleWindowExt:
		appMetaScaleViewportExt( &dc, recordSize, sis );
		continue;

	    case 0:
		if  ( recordSize != 3 )
		    { XLDEB(function,recordSize); return -1;	}
		break;
	    skipArguments:
		for ( done= 3; done < recordSize; done++ )
		    { ignore= sioEndianGetLeInt16( sis ); }
		continue;
	    default:
		XLDEB(function,recordSize);
		/*
		for ( done= 3; done < recordSize; done++ )
		    { ignore= sioEndianGetLeInt16( sis ); }
		continue;
		*/
		return -1;
	    }

	break;
	}

    appMetaCleanDeviceContext( &dc );
    }

    return 0;
    }

/************************************************************************/
/*									*/
/*  List the fonts in a windows metafile.				*/
/*									*/
/*  The geometry parameters are absolutely irrelevant, but required by	*/
/*  appMetaInitDeviceContext().						*/
/*									*/
/************************************************************************/

int appMetaListFontsPs( PostScriptFaceList *	psfl,
			SimpleInputStream *	sis,
			const char *		afmDirectory,
			const char *		prefix,
			int			mapMode,
			int			xWinExt,
			int			yWinExt,
			int			twipsWide,
			int			twipsHigh )
    {
    int			fileType;
    int			headerSize;
    int			windowsVersion;
    long		fileSize;
    int			objectCount;
    long		maxRecordSize;
    int			parameterCount;

    DeviceContext	dc;

    if  ( sioInGetCharacter( sis ) == EOF )
	{ return 0;	}
    sioInUngetLastRead( sis );

    fileType= sioEndianGetLeInt16( sis );
    headerSize= sioEndianGetLeInt16( sis );
    windowsVersion= sioEndianGetLeInt16( sis );
    fileSize= sioEndianGetLeInt32( sis );
    objectCount= sioEndianGetLeInt16( sis );
    maxRecordSize= sioEndianGetLeInt32( sis );
    parameterCount= sioEndianGetLeInt16( sis );

    if  ( appMetaInitDeviceContext( &dc, objectCount, mapMode,
						    xWinExt, yWinExt,
						    twipsWide, twipsHigh,
						    twipsWide, twipsHigh ) )
	{ LDEB(objectCount); return -1;	}

    WMFDEB(appDebug( "PS METAFILE FONTS ...\n" ));

    for (;;)
	{
	int			done;
	int			ignore;

	long			recordSize;
	int			function;

	LogicalFont *		lf;
	int			family;
	int			face;

	AppFontFamily *		aff;
	AppFontTypeface *	aft;

	DocumentFont *		df;
	AfmFontInfo *		afi;
	int			encoding;

	const int		appearsInText= 0;

	MetaFileObject *	mfo;

	recordSize= sioEndianGetLeInt32( sis );
	function= sioEndianGetLeInt16( sis );

	if  ( recordSize < 0 || function < 0 )
	    { LLDEB(recordSize,function); return -1;	}

	switch( function )
	    {
	    case WINMETA_SetWindowExt:
		if  ( appMetaSetWindowExt( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetViewportExt:
		if  ( appMetaSetViewportExt( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_CreateFontIndirect:
		if  ( appMetaCreateFontIndirect( &dc, recordSize, sis ) )
		    { LDEB(1); return -1;	}

		mfo= &(dc.dcObjects[0]);
		lf= &(mfo->mfoLogicalFont);

		df= dc.x_dcFontList.dflFonts+ lf->lfTextAttribute.taFontNumber;

		if  ( utilFontGetPsFont( &family, &face, &aff, &aft, &encoding,
				    afmDirectory, df, lf->lfTextAttribute ) )
		    { LDEB(lf->lfTextAttribute.taFontNumber); return -1; }

		afi= (AfmFontInfo *)aft->aftPrintingData;

		if  ( utilRememberPostsciptFace( psfl, encoding, afi,
				lf->lfTextAttribute, prefix, appearsInText ) )
		    { SDEB(afi->afiFontName); return -1;	}

		appMetaCleanObject( mfo );
		mfo->mfoType= MFtypeFREE;

		continue;

	    default:
		for ( done= 3; done < recordSize; done++ )
		    { ignore= sioEndianGetLeInt16( sis ); }
		continue;

	    case 0:
		break;
	    }

	break;
	}

    appMetaCleanDeviceContext( &dc );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Include the procedures that are used for WMF rendering in		*/
/*  PostScript.								*/
/*									*/
/*  sed -e 's/^/"/' -e 's/$/",/' file					*/
/*									*/
/************************************************************************/

static const char *	APPMETAPS_FillPrep[]=
{
"/fill-prep",
"    {",
"    pathbbox",
"    clip newpath",
"			% llx-u  lly-u  urx-u  ury-u",
"    [ 0 0 0 0 0 0 ] currentmatrix",
"    transform",
"    [ 0 0 0 0 0 0 ] defaultmatrix [ 0 0 0 0 0 0 ] invertmatrix",
"    transform",
"			% llx-u  lly-u  urx-d  ury-d",
"    4 2 roll",
"			% urx-d  ury-d  llx-u  lly-u",
"    [ 0 0 0 0 0 0 ] currentmatrix",
"    transform",
"    [ 0 0 0 0 0 0 ] defaultmatrix [ 0 0 0 0 0 0 ] invertmatrix",
"    transform",
"			% urx-d  ury-d  llx-d  lly-d",
"    3 -1 roll",
"			% urx-d  llx-d  lly-d  ury-d",
"    exch",
"			% urx-d  llx-d  ury-d  lly-d",
"    2 copy lt { exch } if",
"			% urx-d  llx-d  ury=d  lly=d",
"    /fill-high load div floor   /fill-high load mul",
"			% urx-d  llx-d  ury=d  lly=d-",
"    exch",
"    /fill-high load div ceiling /fill-high load mul",
"			% urx-d  llx-d  lly=d- ury=d+",
"    4 2 roll",
"			% lly=d- ury=d+ urx-d  llx-d",
"    2 copy lt { exch } if",
"			% lly=d- ury=d+ urx=d  llx=d",
"    /fill-wide load div floor   /fill-wide load mul",
"			% lly=d- ury=d+ urx=d  llx=d-",
"    exch",
"    /fill-wide load div ceiling /fill-wide load mul",
"			% lly=d- ury=d+ llx=d- urx=d+",
"    4 2 roll",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    [ 0 0 0 0 0 0 ] defaultmatrix setmatrix",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillPattern[]=
{
"/fill-pattern",
"    {",
"    gsave fill-prep",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    /fill-high load exch",
"			% llx=d- urx=d+ lly=d- <fill-high> ury=d+",
"	{",
"	gsave",
"			% llx=d- urx=d+ y",
"	0 exch translate",
"			% llx=d- urx=d+",
"	2 copy /fill-wide load exch",
"	    {",
"	    gsave",
"	    0 translate",
"	    fill-cell",
"	    grestore",
"	    } for",
"",
"	grestore",
"	} for",
"",
"    pop pop",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_HatchHorizontal[]=
{
"/hatch-horizontal",
"    {",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    /fill-high load exch",
"			% llx=d- urx=d+ lly=d- <fill-high> ury=d+",
"	{",
"			% llx=d- urx=d+ y",
"	2 copy moveto",
"			% llx=d- urx=d+ y",
"	3 copy exch pop",
"	lineto pop stroke",
"	} for",
"",
"    pop pop",
"    } bind def",
"",
};

static const char *	APPMETAPS_HatchVertical[]=
{
"/hatch-vertical",
"    {",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    4 2 roll",
"			% lly=d- ury=d+ llx=d- urx=d+",
"    /fill-wide load exch",
"			% lly=d- ury=d+ llx=d- <fill-wide> urx=d+",
"	{",
"			% lly=d- ury=d+ x",
"	2 copy exch moveto",
"			% lly=d- ury=d+ x",
"	3 copy exch pop",
"	exch lineto pop stroke",
"	} for",
"",
"    pop pop",
"    } bind def",
"",
};

static const char *	APPMETAPS_HatchFdiagonal[]=
{
"/hatch-fdiagonal",
"    {",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    4 2 roll 2 copy exch sub 1 copy",
"			% lly=d- ury=d+ llx=d- urx=d+ <dx> <dx>",
"    4 2 roll",
"			% lly=d- ury=d+ <dx> <dx> llx=d- urx=d+",
"    6 3 roll",
"			% <dx> llx=d- urx=d+ lly=d- ury=d+ <dx>",
"    3 -1 roll",
"			% <dx> llx=d- urx=d+ ury=d+ <dx> lly=d-",
"    exch sub exch",
"",
"    /fill-high load exch",
"			% <dx> llx=d- urx=d+ <dx> lly=d-- <fill-high> ury=d+",
"	{",
"			% <dx> llx=d- urx=d+ y",
"	2 copy moveto",
"			% <dx> llx=d- urx=d+ y",
"	4 copy",
"			% <dx> llx=d- urx=d+ y <dx> llx=d- urx=d+ y",
"	exch pop",
"			% <dx> llx=d- urx=d+ y <dx> llx=d- y",
"	3 -1 roll",
"			% <dx> llx=d- urx=d+ y llx=d- y <dx>",
"	add",
"			% <dx> llx=d- urx=d+ y llx=d- y--",
"	lineto pop stroke",
"	} for",
"",
"    pop pop pop",
"    } bind def",
"",
};

static const char *	APPMETAPS_HatchBdiagonal[]=
{
"/hatch-bdiagonal",
"    {",
"			% llx=d- urx=d+ lly=d- ury=d+",
"    4 2 roll 2 copy exch sub 1 copy",
"			% lly=d- ury=d+ llx=d- urx=d+ <dx> <dx>",
"    4 2 roll",
"			% lly=d- ury=d+ <dx> <dx> llx=d- urx=d+",
"    6 3 roll",
"			% <dx> llx=d- urx=d+ lly=d- ury=d+ <dx>",
"    add /fill-high load exch",
"			% <dx> llx=d- urx=d+ <dx> lly=d- <fill-high> ury=d++",
"	{",
"			% <dx> llx=d- urx=d+ y",
"	2 copy moveto",
"			% <dx> llx=d- urx=d+ y",
"	4 copy",
"			% <dx> llx=d- urx=d+ y <dx> llx=d- urx=d+ y",
"	exch pop",
"			% <dx> llx=d- urx=d+ y <dx> llx=d- y",
"	3 -1 roll",
"			% <dx> llx=d- urx=d+ y llx=d- y <dx>",
"	sub",
"			% <dx> llx=d- urx=d+ y llx=d- y--",
"	lineto pop stroke",
"	} for",
"",
"    pop pop pop",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillHorizontal[]=
{
"/fill-horizontal",
"    {",
"    gsave",
"",
"    1 setlinewidth",
"",
"    /fill-wide 8 store",
"    /fill-high 8 store",
"",
"    fill-prep",
"    hatch-horizontal",
"",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillVertical[]=
{
"/fill-vertical",
"    {",
"    gsave",
"",
"    1 setlinewidth",
"",
"    /fill-wide 8 store",
"    /fill-high 8 store",
"",
"    fill-prep",
"    hatch-vertical",
"",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillCross[]=
{
"/fill-cross",
"    {",
"    gsave",
"",
"    1 setlinewidth",
"",
"    /fill-wide 8 store",
"    /fill-high 8 store",
"",
"    fill-prep",
"    4 copy",
"    hatch-horizontal",
"    hatch-vertical",
"",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillBdiagonal[]=
{
"/fill-bdiagonal",
"    {",
"    gsave",
"",
"    1 setlinewidth",
"",
"    /fill-wide 8 store",
"    /fill-high 8 store",
"",
"    fill-prep",
"    hatch-bdiagonal",
"",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillFdiagonal[]=
{
"/fill-fdiagonal",
"    {",
"    gsave",
"",
"    1 setlinewidth",
"",
"    /fill-wide 8 store",
"    /fill-high 8 store",
"",
"    fill-prep",
"    hatch-fdiagonal",
"",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_FillDiagcross[]=
{
"/fill-diagcross",
"    {",
"    gsave",
"",
"    1 setlinewidth",
"",
"    /fill-wide 8 store",
"    /fill-high 8 store",
"",
"    fill-prep",
"    4 copy",
"    hatch-bdiagonal",
"    hatch-fdiagonal",
"",
"    grestore",
"    } bind def",
"",
};

static const char *	APPMETAPS_Rl[]=
{
"/rl",
"    {",
"    rlineto",
"    } bind def",
"",
};

static const char *	APPMETAPS_Bp[]=
{
"/bp",
"    {",
"    newpath moveto",
"    } bind def",
"",
};

int appMetaDefineProcsetPs(	FILE *		f )
    {
    fprintf( f, "%%%%BeginProcSet AppWmfToPs 1.0 1\n\n" );

    utilPsDefineProcedure( f, APPMETAPS_Bp,
		    sizeof(APPMETAPS_Bp)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_Rl,
		    sizeof(APPMETAPS_Rl)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillPrep,
		    sizeof(APPMETAPS_FillPrep)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillPattern,
		    sizeof(APPMETAPS_FillPattern)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_HatchHorizontal,
		    sizeof(APPMETAPS_HatchHorizontal)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_HatchVertical,
		    sizeof(APPMETAPS_HatchVertical)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_HatchFdiagonal,
		    sizeof(APPMETAPS_HatchFdiagonal)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_HatchBdiagonal,
		    sizeof(APPMETAPS_HatchBdiagonal)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillHorizontal,
		    sizeof(APPMETAPS_FillHorizontal)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillVertical,
		    sizeof(APPMETAPS_FillVertical)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillCross,
		    sizeof(APPMETAPS_FillCross)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillBdiagonal,
		    sizeof(APPMETAPS_FillBdiagonal)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillFdiagonal,
		    sizeof(APPMETAPS_FillFdiagonal)/sizeof(const char *) );

    utilPsDefineProcedure( f, APPMETAPS_FillDiagcross,
		    sizeof(APPMETAPS_FillDiagcross)/sizeof(const char *) );

    fprintf( f, "%%%%EndProcSet\n\n" );
    return 0;
    }

