/************************************************************************/
/*									*/
/*  Printing related functionality.					*/
/*									*/
/************************************************************************/

#   include	"appFrameConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<signal.h>
#   include	<locale.h>
#   include	<unistd.h>

#   include	<appSystem.h>
#   include	<appPaper.h>
#   include	"appFrame.h"

#   include	<appDebugon.h>

#   ifdef HAVE_MKSTEMP
#	define USE_MKSTEMP 1
#   else
#	define USE_MKSTEMP 0
#   endif

/************************************************************************/
/*									*/
/*  Print a document.							*/
/*									*/
/*  1)  Avoid user oriented numeric formatting in a PostScript file,	*/
/*	that is meant to be read by computers.				*/
/*  2)  Use mkstemp to make the name of a temporary file. Though there	*/
/*	is no reason here, we simply conform to the fashionable		*/
/*	paranoia to make 'gcc' shut up.					*/
/*									*/
/************************************************************************/

int appCallPrintFunction(	FILE *				f,
				const PrintJob *		pj,
				const PrintGeometry *		pg,
				int				firstPage,
				int				lastPage )
    {
    EditApplication *	ea= pj->pjApplication;
    int			rval= 0;

    /*  1  */
    setlocale( LC_NUMERIC, "C" );

    if  ( (*ea->eaPrintDocument)( f, pj, pg, firstPage, lastPage ) )
	{ LDEB(1); rval= -1;	}

    /*  1  */
    setlocale( LC_NUMERIC, "" );

    return rval;
    }

int appPrintDocument(	int				printer,
			const PrintJob *		pj,
			const PrintGeometry *		pg,
			int				firstPage,
			int				lastPage )
    {
    EditApplication *	ea= pj->pjApplication;
    PrintDestination *	pd= ea->eaPrintDestinations+ printer;
    FILE *		f;
    char		scratchName[L_tmpnam+1];
    char *		command= (char *)0;
    int			fileLen;

    const char *	s;
    char *		to;

    int			fd= -1;

    switch( pd->pdPrintKind )
	{
	case APPprinterPIPE:
	    {
	    void (*prevSigPipeHandler)( int );

	    prevSigPipeHandler= signal( SIGPIPE, SIG_IGN );

	    f= popen( pd->pdCommand, "w" );

	    if  ( ! f )
		{ SXDEB(pd->pdCommand,f); return -1;	}

	    appCallPrintFunction( f, pj, pg, firstPage, lastPage );

	    if  ( pclose( f ) )
		{ SDEB(pd->pdCommand); return -1;	}

	    signal( SIGPIPE, prevSigPipeHandler );
	    }

	    return 0;

	case APPprinterTMPFILE:

	    /*  2  */
#	    if USE_MKSTEMP

	    strcpy( scratchName, "/tmp/tprXXXXXX" );
	    fd= mkstemp( scratchName );

	    if  ( fd < 0 )
		{ SLDEB(scratchName,fd); return -1;	}

#	    else

	    tmpnam( scratchName );
	    fd= -1;

#	    endif

	    fileLen= strlen( scratchName );

	    to= command= malloc( pd->pdCommandLength+ 
					    pd->pdPercentCount* fileLen+ 1 );
	    if  ( ! command )
		{ LXDEB(pd->pdCommandLength,command); return -1; }
	    s= pd->pdCommand;

	    while( *s )
		{
		if  ( s[0] == '%' && s[1] == 'f' )
		    {
		    strcpy( to, scratchName ); to += fileLen;
		    s += 2; continue;
		    }

		*(to++)= *(s++);
		}
	    *to= '\0';

#	    if USE_MKSTEMP

	    f= fdopen( fd, "w" );
	    if  ( ! f )
		{
		SXDEB(scratchName,f);
		close( fd ); free( command );
		return -1;
		}

#	    else

	    f= fopen( scratchName, "w" );
	    if  ( ! f )
		{ SXDEB(scratchName,f); free( command ); return -1;	}

#	    endif

	    appCallPrintFunction( f, pj, pg, firstPage, lastPage );

	    fclose( f );
	    system( command );
	    free( command );
	    return 0;

	case APPprinterTOFILE:
	default:
	    LDEB(pd->pdPrintKind);
	    return -1;
	}
    }

static char * appDocBuildFaxCommand(		const char *	faxCommand,
						int		commandLen,
						const char *	fileName,
						const char *	faxNumber,
						char *		title,
						int		fileCount,
						int		faxCount,
						int		titleCount )
    {
    char *		command;
    const char *	s;
    char *		to;

    int			fileLen;
    int			faxLen;
    int			titleLen;

    if  ( ! title )
	{ title= "";	}

    fileLen= strlen( fileName );
    faxLen= strlen( faxNumber );
    titleLen= strlen( title );

    to= command= (char *)malloc( commandLen+
					fileCount* fileLen+
					faxCount* faxLen+
					titleCount* titleLen+ 1 );
    if  ( ! command )
	{ XDEB(command); return (char *)0;	}

    s= faxCommand;
    while( *s )
	{
	if  ( s[0] == '%' && s[1] == 'f' )
	    {
	    strcpy( to, fileName ); to += fileLen;
	    s += 2; continue;
	    }

	if  ( s[0] == '%' && s[1] == 'n' )
	    {
	    strcpy( to, faxNumber ); to += faxLen;
	    s += 2; continue;
	    }

	if  ( s[0] == '%' && s[1] == 't' )
	    {
	    strcpy( to, title ); to += titleLen;
	    s += 2; continue;
	    }

	*(to++)= *(s++);
	}
    *to= '\0';

    return command;
    }

/************************************************************************/
/*									*/
/*  Fill a PrintJob for an EditDocument.				*/
/*									*/
/************************************************************************/

void appPrintJobForEditDocument(	PrintJob *		pj,
					EditDocument *		ed )
    {
    pj->pjPrivateData= ed->edPrivateData;
    pj->pjFormat= ed->edFormat;
    pj->pjDrawingData= &(ed->edDrawingData);
    pj->pjApplication= ed->edApplication;
    pj->pjTitle= ed->edTitle;

    return;
    }

/************************************************************************/
/*									*/
/*  Fax a document.							*/
/*									*/
/************************************************************************/

int appFaxDocument(	EditDocument *			ed,
			const char *			faxNumber,
			const PrintGeometry *		pg,
			int				firstPage,
			int				lastPage )
    {
    EditApplication *	ea= ed->edApplication;
    int			fileCount= 0;
    int			faxCount= 0;
    int			titleCount= 0;

    int			commandLen;

    char		scratchName[L_tmpnam+1];

    char *		s;
    char *		command;

    int			fd= -1;
    FILE *		f;

    PrintJob		pj;

    if  ( ! ea->eaFaxCommand )
	{ XDEB(ea->eaFaxCommand); return -1;	}

    appPrintJobForEditDocument( &pj, ed );

    s= ea->eaFaxCommand;
    while( *s )
	{
	if  ( s[0] == '%' && s[1] == 'f' )
	    { fileCount++; s += 2; continue;	}

	if  ( s[0] == '%' && s[1] == 'n' )
	    { faxCount++; s += 2; continue;	}

	if  ( s[0] == '%' && s[1] == 't' )
	    { titleCount++; s += 2; continue;	}

	s++;
	}

    commandLen= s- ea->eaFaxCommand;

    if  ( faxCount == 0 )
	{ LDEB(faxCount); return -1;	}

    if  ( fileCount > 0 )
	{
#	if USE_MKSTEMP

	strcpy( scratchName, "/tmp/tprXXXXXX" );
	fd= mkstemp( scratchName );

	if  ( fd < 0 )
	    { SLDEB(scratchName,fd); return -1;	}

#	else

	tmpnam( scratchName );
	fd= -1;

#	endif

	command= appDocBuildFaxCommand( ea->eaFaxCommand, commandLen,
					scratchName, faxNumber, ed->edTitle,
					fileCount, faxCount, titleCount );

	if  ( ! command )
	    { XDEB(command); return -1;	}

#	if USE_MKSTEMP

	f= fdopen( fd, "w" );
	if  ( ! f )
	    { SXDEB(scratchName,f); close( fd ); free( command ); return -1; }

#	else

	f= fopen( scratchName, "w" );
	if  ( ! f )
	    { SXDEB(scratchName,f); free( command ); return -1;	}

#	endif

	appCallPrintFunction( f, &pj, pg, firstPage, lastPage );

	fclose( f );

	system( command );
	}
    else{
	scratchName[0]= '\0';

	command= appDocBuildFaxCommand( ea->eaFaxCommand, commandLen,
					scratchName, faxNumber, ed->edTitle,
					fileCount, faxCount, titleCount );

	f= popen( command, "w" );
	if  ( ! f )
	    { SXDEB(command,f); free( command ); return -1;	}

	appCallPrintFunction( f, &pj, pg, firstPage, lastPage );

	pclose( f );
	}

    free( command );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Start a print job for a command line call.				*/
/*									*/
/************************************************************************/

static int appPrintJobForCommand(	PrintJob *		pj,
					AppDrawingData *	add,
					EditApplication *	ea,
					const char *		fromName )
    {
    double		screenPixelsPerMM;
    double		verPixPerMM;
    double		xfac;
    double		yfac;

    if  ( ea->eaMakePrivateData	)
	{
	pj->pjPrivateData= (*ea->eaMakePrivateData)();
	if  ( ! pj->pjPrivateData )
	    { XDEB(pj->pjPrivateData); return -1; }
	}

    if  ( (*ea->eaOpenDocument)( ea, pj->pjPrivateData, &(pj->pjFormat),
					    ea->eaToplevel.atTopWidget,
					    (APP_WIDGET)0, fromName ) )
	{ SDEB(fromName); return -1; }

    if  ( ea->eaToplevel.atTopWidget )
	{
	appGetFactors( ea, &screenPixelsPerMM, &verPixPerMM, &xfac, &yfac );

	appSetDrawingEnvironment( add, ea->eaMagnification, xfac,
			    screenPixelsPerMM,
			    ea->eaAfmDirectory,
			    ea->eaGhostscriptFontmap,
			    ea->eaGhostscriptFontToXmapping,
			    ea->eaToplevel.atTopWidget );

	add->addPageGapPixels= (int)( ea->eaPageGapMM* verPixPerMM );
	}
    else{
	add->addPhysicalFontList.apflAfmDirectory= ea->eaAfmDirectory;
	add->addPhysicalFontList.apflGhostscriptFontmap=
					    ea->eaGhostscriptFontmap;
	add->addPhysicalFontList.apflGhostscriptFontToXmapping=
					    ea->eaGhostscriptFontToXmapping;
	}

    pj->pjDrawingData= add;
    pj->pjApplication= ea;
    pj->pjTitle= fromName;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initial setup for command line printing.				*/
/*									*/
/************************************************************************/

static void appPrintFinishCommandRun(	EditApplication *	ea,
					PrintJob *		pj,
					AppDrawingData *	add )
    {
    (*ea->eaFreeDocument)( pj->pjPrivateData, pj->pjFormat, add );

    appCleanDrawingData( add );

    return;
    }

static int appPrintStartCommandRun(	EditApplication *	ea,
					PrintJob *		pj,
					PrintGeometry *		pg,
					AppDrawingData *	add,
					const char *		paperString,
					const char *		fromName )
    {
    pg->pgSheetGeometry= ea->eaDefaultDocumentGeometry;

    if  ( paperString )
	{
	DocumentGeometry *	dg= &(pg->pgSheetGeometry);
	int			paperFormat;

	if  ( appPaperFormatFromString( &paperFormat,
					    &(dg->dgPageWideTwips),
					    &(dg->dgPageHighTwips),
					    ea->eaUnitInt, paperString ) )
	    { SDEB(paperString); return -1;	}
	}

    if  ( appPrintJobForCommand( pj, add, ea, fromName ) )
	{ SDEB(fromName); return -1;	}

    if  ( (*ea->eaLayoutDocument)( pj->pjPrivateData, pj->pjFormat, add,
						    &(pg->pgSheetGeometry) ) )
	{
	SDEB(fromName);
	appPrintFinishCommandRun( ea, pj, add );
	return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Command line Print To File entrypoint.				*/
/*									*/
/************************************************************************/

int appPrintToFile(	EditApplication *	ea,
			const char *		fromName,
			const char *		toName,
			const char *		paperString )
    {
    int			rval= 0;
    AppDrawingData	add;

    FILE *		f;

    PrintGeometry	pg;
    PrintJob		pj;

    utilPsInitPrintGeometry( &pg );
    appInitDrawingData( &add );

    if  ( ! strcmp( fromName, toName ) )
	{ SSDEB(fromName,toName); return -1;	}

    if  ( appPrintStartCommandRun( ea, &pj, &pg, &add, paperString, fromName ) )
	{ SDEB(fromName); return -1;	}

    f= fopen( toName, "w" );
    if  ( ! f )
	{
	SXDEB(toName,f);
	appPrintFinishCommandRun( ea, &pj, &add );
	return -1;
	}

    {
    const int	firstPage= -1;
    const int	lastPage= -1;

    if  ( appCallPrintFunction( f, &pj, &pg, firstPage, lastPage ) )
	{ SDEB(fromName); rval= -1;	}
    }

    fclose( f );

    appPrintFinishCommandRun( ea, &pj, &add );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Command line Print To Printer entrypoint.				*/
/*									*/
/************************************************************************/

int appPrintToPrinter(	EditApplication *	ea,
			const char *		fromName,
			const char *		toName,
			const char *		paperString )
    {
    int			rval= 0;
    AppDrawingData	add;

    int			printer;

    PrintGeometry	pg;
    PrintJob		pj;

    utilPsInitPrintGeometry( &pg );
    appInitDrawingData( &add );

    if  ( ea->eaPrintDestinationCount == 0				&&
	  utilPrinterGetPrinters( &(ea->eaPrintDestinationCount),
				    &(ea->eaDefaultPrintDestination),
				    &(ea->eaPrintDestinations),
				    ea->eaCustomPrintCommand,
				    ea->eaCustomPrinterName )		)
	{ LDEB(1); return -1;	}

    printer= ea->eaDefaultPrintDestination;

    if  ( toName )
	{
	int				i;
	const PrintDestination *	pd;

	pd= ea->eaPrintDestinations;
	for ( i= 0; i < ea->eaPrintDestinationCount; pd++, i++ )
	    {
	    if  ( ! strcmp( fromName, pd->pdPrinterName ) )
		{ printer= i; break;	}
	    }
	}

    if  ( printer < 0 )
	{ LDEB(printer); return -1;	}

    if  ( appPrintStartCommandRun( ea, &pj, &pg, &add, paperString, fromName ) )
	{ SDEB(fromName); return -1;	}

    {
    const int	firstPage= -1;
    const int	lastPage= -1;

    if  ( appPrintDocument( printer, &pj, &pg, firstPage, lastPage ) )
	{ SDEB(fromName); rval= -1;	}
    }

    appPrintFinishCommandRun( ea, &pj, &add );

    return rval;
    }

