/************************************************************************/
/*  Editor functionality, indepentent of the application.		*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"
#   include	"docPs.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Insert a table in the document.					*/
/*									*/
/*  0)  Also expose the table borders.					*/
/*  1)  If a range of data was selected, first discard it.		*/
/*  2)  If the selection was at the end of a paragraph, try to move to	*/
/*	the beginning of the next one.					*/
/*  3)  If the IBar is in the middle of a paragraph, split it.		*/
/*	tedSplitParaContents() does the layout of both halves.		*/
/*  4)  If the paragraph is not the first one in the 'cell' that	*/
/*	contains it, split the cell.					*/
/*  5)  If the 'cell' is not the first one in the 'row' that contains	*/
/*	it, split the row.						*/
/*									*/
/************************************************************************/

static int tedInsertRows(	BufferItem *		sectBi,
				const BufferItem *	refRowBi,
				BufferItem **		pNewBi,
				const RowProperties *	rp,
				TextAttribute		ta,
				const BufferDocument *	bd,
				AppDrawingData *	add,
				int			pos,
				int			rows,
				int			inHeaderFooter,
				DocumentRectangle *	drChanged )
    {
    BufferItem *	newBi= (BufferItem *)0;

    int			row;
    int			col;

    newBi= (BufferItem *)0;

    for ( row= 0; row < rows; pos++, row++ )
	{
	BufferItem *		rowBi= (BufferItem *)0;

	rowBi= docInsertItem( bd, sectBi, pos, DOClevROW );
	if  ( ! rowBi )
	    { XDEB(rowBi); return -1;	}

	for ( col= 0; col < rp->rpCellCount; col++ )
	    {
	    BufferItem *	cellBi;
	    BufferItem *	paraBi;

	    const BufferItem *	refParaBi= (BufferItem *)0;
	    const BufferItem *	refCellBi= (BufferItem *)0;

	    if  ( refRowBi && col < refRowBi->biChildCount )
		{ refCellBi= refRowBi->biChildren[col];	}

	    cellBi= docInsertItem( bd, rowBi, col, DOClevCELL );
	    if  ( ! cellBi )
		{ XDEB(cellBi); return -1;	}

	    paraBi= docInsertItem( bd, cellBi, 0, DOClevPARA );
	    if  ( ! paraBi )
		{ XDEB(paraBi); return -1;	}

	    if  ( refCellBi && refCellBi->biChildCount > 0 )
		{ refParaBi= refCellBi->biChildren[0];	}

	    if  ( refParaBi )
		{
		docCopyParagraphProperties( &paraBi->biParaProperties,
						&refParaBi->biParaProperties );
		}

	    if  ( refParaBi )
		{ ta= refParaBi->biParaParticules[0].tpTextAttribute; }

	    if  ( ! docInsertTextParticule( paraBi, 0, 0, 0,
							DOCkindTEXT, ta ) )
		{ LDEB(1); return -1;	}

	    if  ( row == 0 && col == 0 )
		{ newBi= paraBi;	}

	    paraBi->biParaInTable= 1;
	    }

	if  ( docCopyRowProperties( &(rowBi->biRowProperties), rp ) )
	    { LDEB(1); return -1;	}

	tedLayoutItem( rowBi, bd, inHeaderFooter, add, drChanged );
	}

    *pNewBi= newBi; return 0;
    }

int tedInsertTable(		EditDocument *		ed,
				int			rows,
				int			columns )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    DocumentProperties *	dp= &(bd->bdProperties);
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;

    BufferItem *		newBi;
    BufferItem *		sectBi;
    BufferItem *		refRowBi;
    int				col;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    TextAttribute		ta;

    int				wide;

    RowProperties		rp;

    BufferSelection *		bs= &(td->tdSelection);
    SelectionScope *		ss= &(td->tdSelectionScope);

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    /*  0  */
    drChanged.drY0--;

    /*  1  */
    if  ( ! tedHasIBarSelection( td ) )
	{ LDEB(1); return -1;	}

    /*  2  */
    if  ( bs->bsBegin.bpStroff == bi->biParaStrlen )
	{
	tedNextPosition( add, &bs->bsBegin );
	bi= bs->bsBegin.bpBi;
	}

    /*  3  */
    if  ( bs->bsBegin.bpStroff != 0 )
	{
	int		onNewPage= 0;

	if  ( tedSplitParaContents( &newBi, bi, &drChanged,
						ed, DOClevROW, onNewPage ) )
	    { LDEB(1); return -1;	}

	bi= newBi;
	}

    /*  4  */
    if  ( bi->biNumberInParent > 0 )
	{
	if  ( docSplitGroupItem( bd, &newBi,
					bi->biParent, bi->biNumberInParent ) )
	    { LDEB(bi->biNumberInParent); return -1;	}

	if  ( bi->biParent->biNumberInParent == 0 )
	    {
	    tedLayoutItem( newBi,
		    bd, ss->ssInHeaderFooter, add, &drChanged );
	    tedLayoutItem( bi->biParent, bd,
		    ss->ssInHeaderFooter, add, &drChanged );
	    }
	}

    /*  5  */
    if  ( bi->biParent->biNumberInParent > 0 )
	{
	if  ( docSplitGroupItem( bd, &newBi,
		    bi->biParent->biParent, bi->biParent->biNumberInParent ) )
	    { LDEB(bi->biParent->biNumberInParent); return -1;	}

	tedLayoutItem( newBi, bd,
			ss->ssInHeaderFooter, add, &drChanged );
	tedLayoutItem( bi->biParent->biParent, bd,
			ss->ssInHeaderFooter, add, &drChanged );
	}


    ta= bi->biParaParticules[0].tpTextAttribute;

    wide=   dp->dpGeometry.dgPageWideTwips;
    wide -= dp->dpGeometry.dgLeftMarginTwips;
    wide -= dp->dpGeometry.dgRightMarginTwips;

    docInitRowProperties( &rp );
    rp.rpHalfGapWidthTwips= 5* ta.taFontSizeHalfPoints;
    rp.rpLeftIndentTwips=  -5* ta.taFontSizeHalfPoints;

    for ( col= 0; col < columns; col++ )
	{
	CellProperties	cp;

	docInitCellProperties( &cp );
	cp.cpRightBoundaryTwips= ( ( col+ 1 )* wide )/ columns;

	if  ( docInsertRowColumn( &rp, col, &cp ) )
	    { LDEB(col); return -1;	}

	rp.rpHasTableParagraphs= 1;
	}

    newBi= (BufferItem *)0;
    refRowBi= (BufferItem *)0;
    sectBi= bi->biParent->biParent->biParent;

    if  ( tedInsertRows( sectBi, refRowBi, &newBi,
			    &rp, ta, bd, add,
			    bi->biParent->biParent->biNumberInParent, rows,
			    ss->ssInHeaderFooter,
			    &drChanged ) )
	{ LDEB(rows); return -1;	}

    docCleanRowProperties( &rp );

    tedEditFinishIBarSelection( ed, newBi, &drChanged, oldBackY1, 0 );

    return 0;
    }

int tedInsertRowsInTable(	BufferItem *		sectBi,
				const RowProperties *	rp,
				EditDocument *		ed,
				int			row,
				int			rows )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		refParaBi= td->tdSelection.bsBegin.bpBi;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    BufferItem *		newBi;
    BufferItem *		refRowBi;

    TextAttribute		ta;

    SelectionScope *		ss= &(td->tdSelectionScope);

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    ta= refParaBi->biParaParticules[0].tpTextAttribute;
    newBi= (BufferItem *)0;
    refRowBi= (BufferItem *)0;

    if  ( refParaBi->biParaInTable )
	{ refRowBi= refParaBi->biParent->biParent;	}

    if  ( tedInsertRows( sectBi, refRowBi, &newBi, rp, ta, bd, add, row, rows,
					ss->ssInHeaderFooter, &drChanged ) )
	{ LDEB(rows); return -1;	}

    tedEditFinishIBarSelection( ed, newBi, &drChanged, oldBackY1, 0 );

    return 0;
    }

int tedInsertRowInTable(	EditDocument *		ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    const RowProperties *	rp;

    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    rp= &(sectBi->biChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, rp, ed, row, /*rows*/ 1 ) )
	{ LDEB(row); return -1;	}

    return 0;
    }

int tedAppendRowToTable(	EditDocument *		ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    const RowProperties *	rp;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    rp= &(sectBi->biChildren[row]->biRowProperties);

    if  ( tedInsertRowsInTable( sectBi, rp, ed, row+ 1, /*rows*/ 1 ) )
	{ LDEB(row); return -1;	}
    
    return 0;
    }

int tedDeleteRowsFromTable(	EditDocument *		ed,
				int			delRow0,
				int			delRow1 )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    DocumentProperties *	dp= &(bd->bdProperties);
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    AppDrawingData *		add= &(ed->edDrawingData);

    const SectionProperties *	sp;
    int				page;

    BufferItem *		sectBi;
    BufferItem *		rowBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    BufferPosition		bpNew;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    SelectionScope *		ss= &(td->tdSelectionScope);
    HeaderFooter *		hf= (HeaderFooter *)0;

    BufferItem *		docSectBi;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    rowBi= sectBi->biChildren[delRow0];
    drChanged.drY0= BI_TOP_PIXELS( add, rowBi )- 1;

    rowBi= sectBi->biChildren[delRow0];
    drChanged.drY1= BI_BELOW_PIXELS( add, rowBi )- 1;

    docInitPosition( &bpNew );
    if  ( docLastPosition( sectBi->biChildren[delRow0], &bpNew )	||
	  docNextPosition( &bpNew )					)
	{
	docInitPosition( &bpNew );
	if  ( docFirstPosition( sectBi->biChildren[delRow1], &bpNew )	||
	      docPrevPosition( &bpNew, /*lastOne=*/ 0 )			)
	    { docInitPosition( &bpNew ); }
	}

    tedDeleteItems( bd, sectBi, delRow0, delRow1- delRow0+ 1, add );

    switch( ss->ssInHeaderFooter )
	{
	int		inHeaderFooter;

	case DOCinBODY:
	    tedLayoutItem( sectBi, bd, ss->ssInHeaderFooter, add, &drChanged );
	    break;

	case DOCinSECT_HEADER:
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	    page= ss->ssHeaderFooterPage;
	    docSectBi= bd->bdItem.biChildren[ss->ssHeaderFooterSection];
	    sp= &(docSectBi->biSectProperties);

	    inHeaderFooter= docWhatPageHeader( &hf, docSectBi, page, dp );

	    if  ( ! hf						||
		  inHeaderFooter != ss->ssInHeaderFooter	||
		  hf->hfItem != sectBi				)
		{ LDEB(1); return -1;	}

	    tedLayoutExternal( hf, page, sp, bd, add, &drChanged );
	    break;

	case DOCinSECT_FOOTER:
	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    page= ss->ssHeaderFooterPage;
	    docSectBi= bd->bdItem.biChildren[ss->ssHeaderFooterSection];
	    sp= &(docSectBi->biSectProperties);

	    inHeaderFooter= docWhatPageFooter( &hf, docSectBi, page, dp );

	    if  ( ! hf						||
		  inHeaderFooter != ss->ssInHeaderFooter	||
		  hf->hfItem != sectBi				)
		{ LDEB(1); return -1;	}

	    tedLayoutExternal( hf, page, sp, bd, add, &drChanged );
	    break;

	default:
	    LDEB(ss->ssInHeaderFooter); return -1;
	}

    tedEditFinishIBarSelection( ed, bpNew.bpBi, &drChanged, oldBackY1,
							    bpNew.bpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert cells/columns in tables.					*/
/*									*/
/************************************************************************/

static BufferItem * tedInsertCellInRow(	BufferDocument *	bd,
					BufferItem *		rowBi,
					int			col,
					const CellProperties *	cp,
					TextAttribute		ta )
    {
    RowProperties *	rp;

    BufferItem *	cellBi;
    BufferItem *	paraBi;

    rp= &(rowBi->biRowProperties);

    if  ( docInsertRowColumn( rp, col, cp ) )
	{ LDEB(col); return (BufferItem *)0;	}

    cellBi= docInsertItem( bd, rowBi, col, DOClevCELL );
    if  ( ! cellBi )
	{ LXDEB(col,cellBi); return (BufferItem *)0;	}

    paraBi= docInsertEmptyParagraph( bd, cellBi, ta );
    if  ( ! paraBi )
	{ XDEB(paraBi); return (BufferItem *)0;	}

    paraBi->biParaInTable= 1;

    return paraBi;
    }

static int tedSplitColumnInRows(	EditDocument *	ed,
					BufferItem *	sectBi,
					int		row0,
					int		row,
					int		row1,
					int		col,
					int		after )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		newBi= (BufferItem *)0;

    BufferItem *		rowBi;

    TextAttribute		ta;

    DocumentRectangle		drChanged;
    int				oldBackY1= add->addBackRect.drY1;

    SelectionScope *		ss= &(td->tdSelectionScope);

    if  ( after )
	{ after= 1;	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    rowBi= sectBi->biChildren[row0];
    drChanged.drY0= BI_TOP_PIXELS( add, rowBi )- 1;

    rowBi= sectBi->biChildren[row1];
    drChanged.drY1= BI_BELOW_PIXELS( add, rowBi )- 1;

    while( row0 <= row1 )
	{
	RowProperties *		rp;

	CellProperties *	oldCp;
	CellProperties		newCp;

	int			left;
	int			middle;

	BufferItem *		cellBi;
	BufferItem *		paraBi;

	rowBi= sectBi->biChildren[row0];
	rp= &(rowBi->biRowProperties);

	oldCp= rp->rpCells+ col;

	if  ( col == 0 )
	    { left= rp->rpLeftIndentTwips;			}
	else{ left= rp->rpCells[col-1].cpRightBoundaryTwips;	}

	middle= ( rp->rpCells[col].cpRightBoundaryTwips+ left )/ 2;

	newCp= *oldCp;
	if  ( after )
	    { oldCp->cpRightBoundaryTwips= middle;	}
	else{ newCp. cpRightBoundaryTwips= middle;	}

	cellBi= rowBi->biChildren[col];
	paraBi= cellBi->biChildren[0];
	ta= paraBi->biParaParticules[0].tpTextAttribute;

	paraBi= tedInsertCellInRow( bd, rowBi, col+ after, &newCp, ta );
	if  ( row0 == row )
	    { newBi= paraBi;	}

	tedLayoutItem( rowBi, bd, ss->ssInHeaderFooter, add, &drChanged );

	row0++;
	}

    tedEditFinishIBarSelection( ed, newBi, &drChanged, oldBackY1, 0 );

    return 0;
    }

int tedInsertColumnInTable(	EditDocument *		ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsBegin.bpBi;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    int			after= 0;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( ed, sectBi, row0, row, row1, col, after );
    }

int tedAppendColumnToTable(	EditDocument *		ed )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	bi= td->tdSelection.bsEnd.bpBi;

    BufferItem *	sectBi;

    int			col;
    int			row;
    int			row0;
    int			row1;

    int			after= 1;

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( ed, sectBi, row0, row, row1, col, after );
    }

int tedDeleteColumnsFromRows(	EditDocument *	ed,
				int		delRow0,
				int		delRow1,
				int		delCol0,
				int		delCol1 )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsEnd.bpBi;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		sectBi;
    BufferItem *		rowBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    BufferPosition		bpNew;

    DocumentRectangle		drChanged;
    int				oldBackY1= add->addBackRect.drY1;

    int				count= delCol1- delCol0+ 1;

    SelectionScope *		ss= &(td->tdSelectionScope);

    if  ( docDelimitTable( bi, &sectBi, &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    rowBi= sectBi->biChildren[row0];
    drChanged.drY0= BI_TOP_PIXELS( add, rowBi );

    rowBi= sectBi->biChildren[row1];
    drChanged.drY1= BI_BELOW_PIXELS( add, rowBi )- 1;

    docInitPosition( &bpNew );
    rowBi= sectBi->biChildren[delRow1];
    if  ( delCol1 >= rowBi->biChildCount- 1			||
	  docLastPosition( rowBi->biChildren[delCol1], &bpNew )	||
	  docNextPosition( &bpNew )					)
	{
	docInitPosition( &bpNew );
	rowBi= sectBi->biChildren[delRow0];
	if  ( delCol0 <= 0						||
	      docFirstPosition( rowBi->biChildren[delCol0], &bpNew ) ||
	      docPrevPosition( &bpNew, /*lastOne=*/ 0 )			)
	    { docInitPosition( &bpNew ); }
	}

    while( delRow0 <= delRow1 )
	{
	int		wide;
	RowProperties *	rp;

	rowBi= sectBi->biChildren[delRow0];
	rp= &(rowBi->biRowProperties);

	if  ( delCol0 == 0 )
	    { wide= rp->rpLeftIndentTwips;				}
	else{ wide= rp->rpCells[delCol0-1].cpRightBoundaryTwips;	}
	wide= rp->rpCells[delCol1].cpRightBoundaryTwips- wide;

	rp->rpCellCount -= count;

	for ( col= delCol0; col < rp->rpCellCount; col++ )
	    {
	    docCleanCellProperties( &(rp->rpCells[col]) );
	    rp->rpCells[col]= rp->rpCells[col+ count];
	    rp->rpCells[col].cpRightBoundaryTwips -= wide;
	    }

	tedDeleteItems( bd, rowBi, delCol0, count, add );

	tedLayoutItem( rowBi, bd, ss->ssInHeaderFooter, add, &drChanged );

	delRow0++;
	}

    tedEditFinishIBarSelection( ed, bpNew.bpBi, &drChanged, oldBackY1,
							    bpNew.bpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adapt format tool to the circumstances.				*/
/*									*/
/************************************************************************/

void tedAdaptFormatToolToDocument(	EditDocument *	ed )
    {
    EditApplication *		ea= ed->edApplication;
    TedAppResources *		tar= (TedAppResources *)ea->eaResourceData;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferDocument *	bd= td->tdDocument;

    if  ( ! tar->tarInspector )
	{ return;	}

    tedFormatToolAdaptToSelection( tar->tarInspector, bd,
	    &(td->tdSelectionScope), &(td->tdSelection), ed->edFileReadOnly );

    return;
    }

/************************************************************************/
/*									*/
/*  'Set' event handler of the table tool.				*/
/*									*/
/************************************************************************/

void tedAppSetTableProperties(	EditApplication *	ea,
				const TableRectangle *	trChange,
				const RowProperties *	rpFrom )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td;
    BufferDocument *		bd;

    BufferItem *		sectBi;

    RowProperties *		rpTo;
    const CellProperties *	cpFrom;
    CellProperties *		cpTo;

    int				rowPropertiesAlso= 0;

    BufferSelection		bs;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;
    bd= td->tdDocument;

    if  ( docTableRectangleSelection( &bs, &sectBi,
				    bd, &(td->tdSelectionScope), trChange ) )
	{ LDEB(1); return;	}

    if  ( trChange->trCol0 >= 0 && trChange->trCol0 == trChange->trCol1 )
	{
	int			row;

	cpFrom= rpFrom->rpCells+ trChange->trCol0;

	for ( row= trChange->trRow0; row <= trChange->trRow1; row++ )
	    {
	    rpTo= &(sectBi->biChildren[row]->biRowProperties);
	    cpTo= rpTo->rpCells+ trChange->trCol0;

	    cpTo->cpLeftBorder= cpFrom->cpLeftBorder;
	    cpTo->cpRightBorder= cpFrom->cpRightBorder;
	    }
	}

    if  ( trChange->trRow0 >= 0 && trChange->trRow0 == trChange->trRow1 )
	{
	int			col;

	rpTo= &(sectBi->biChildren[trChange->trRow0]->biRowProperties);
	cpFrom= rpFrom->rpCells;
	cpTo= rpTo->rpCells;

	for ( col= trChange->trCol0; col <= trChange->trCol1; cpFrom++, cpTo++, col++ )
	    {
	    cpTo->cpTopBorder= cpFrom->cpTopBorder;
	    cpTo->cpBottomBorder= cpFrom->cpBottomBorder;
	    }

	rowPropertiesAlso= 1;
	}

    tedChangeTableLayout( ed,
		td->tdSelection.bsBegin.bpBi, sectBi,
		rpFrom, rowPropertiesAlso, trChange->trRow0, trChange->trRow1 );

    return;
    }

void tedAppSetTableSelection(	EditDocument *		ed,
				const SelectionScope *	ss,
				const TableRectangle *	trSet )
    {
    TedDocument *	td;

    TableRectangle	tr;

    int			tableRectangle= 0;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    if  ( ! tedGetTableRectangle( &tr,
				&(td->tdSelectionScope), &(td->tdSelection) ) )
	{ tableRectangle= 1;	}

    if  ( ! tableRectangle )
	{ LDEB(tableRectangle); return;	}

    docExpandTableRectangleToWholeTable( &tr );

    if  ( ! docIntersectTableRectangle( &tr, &tr, trSet ) )
	{ return;	}

    tedDocTableSelectTableRectangle( ed, ss, &tr );

    return;
    }

/************************************************************************/
/*									*/
/*  Update the of a table in response to a user action.			*/
/*									*/
/************************************************************************/

void tedChangeTableLayout(	EditDocument *		ed,
				BufferItem *		bi,
				BufferItem *		sectBi,
				const RowProperties *	rp,
				int			rowPropertiesAlso,
				int			row0,
				int			row1 )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		rowBi= sectBi->biChildren[row0];

    DocumentRectangle		drChanged;

    SelectionScope *		ss= &(td->tdSelectionScope);

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    rowBi= sectBi->biChildren[row0];
    drChanged.drY0= BI_TOP_PIXELS( add, rowBi );

    rowBi= sectBi->biChildren[row1];
    drChanged.drY1= BI_BELOW_PIXELS( add, rowBi )- 1;

    while( row0 <= row1 )
	{
	CellProperties *	cp;
	int			col;

	rowBi= sectBi->biChildren[row0];

	rowBi->biRowLeftIndentTwips= rp->rpLeftIndentTwips;

	if  ( rowPropertiesAlso )
	    {
	    rowBi->biRowHeightTwips= rp->rpHeightTwips;
	    rowBi->biRowKeepOnPage= rp->rpKeepOnPage;
	    rowBi->biRowIsTableHeader= rp->rpIsTableHeader;
	    }

	rowBi->biRowHalfGapWidthTwips= rp->rpHalfGapWidthTwips;
	rowBi->biRowLeftIndentTwips= rp->rpLeftIndentTwips;

	cp= rp->rpCells;
	for ( col= 0; col < rp->rpCellCount; cp++, col++ )
	    {
	    rowBi->biRowCells[col].cpRightBoundaryTwips=
						cp->cpRightBoundaryTwips;
	    }

	tedLayoutItem( rowBi, bd, ss->ssInHeaderFooter, add, &drChanged );

	row0++;
	}

    appDocumentChanged( ed, 1 );

    tedDelimitCurrentSelection( td, add );

    tedExposeRectangle( ed, (const DocumentRectangle *)0, 0,0 );

    tedDocAdaptHorizontalRuler( ed, bi );

    return;
    }

/************************************************************************/
/*									*/
/*  Callback for the 'DrawTableGrid' toggle.				*/
/*									*/
/************************************************************************/

void tedDocTableDrawGrid(	APP_WIDGET	option,
				void *		voided,
				void *		voidcbs )
    {
    EditDocument *		ed= (EditDocument *)voided;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    int				set;

    set= appGuiGetToggleStateFromCallback( option, voidcbs );

    td->tdDrawTableGrid= set;

    tedExposeRectangle( ed, (const DocumentRectangle *)0, 0,0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Get a suggestion about the line height: Used for the initial value	*/
/*  for space before/after in the format tool.				*/
/*									*/
/************************************************************************/

int tedGetParaLineHeight(	int *			pLineHeight,
				EditDocument *		ed )
    {
    TedDocument *	td;
    AppDrawingData *	add;

    BufferItem *	bi;

    add= &(ed->edDrawingData);
    td= (TedDocument *)ed->edPrivateData;

    bi= td->tdSelection.bsBegin.bpBi;

    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), bi ) )
	{ LDEB(1); return -1;	}

    *pLineHeight= bi->biParaAscentTwips- bi->biParaDescentTwips; return 0;
    }

