/************************************************************************/
/*									*/
/*  Ted: Management of the selection and the current position.		*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>
#   include	<limits.h>

#   include	"docPs.h"
#   include	"tedApp.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Is the selection a position between two chars?			*/
/*									*/
/************************************************************************/

int tedHasIBarSelection(	const TedDocument *   td )
    {
    const BufferSelection *	bs= &(td->tdSelection);

    if  ( bs->bsBegin.bpBi				&&
	  bs->bsEnd.bpBi == bs->bsBegin.bpBi		&&
	  bs->bsEnd.bpStroff == bs->bsBegin.bpStroff	)
	{ return 1;	}

    return 0;
    }

int tedHasSelection(	const TedDocument *   td )
    {
    const BufferSelection *	bs= &(td->tdSelection);

    if  ( bs->bsBegin.bpBi )
	{ return 1;	}

    return 0;
    }

int tedGetSelection(	SelectionScope *	ss,
			BufferSelection *	bs,
			TedDocument *		td )
    {
    if  ( ! td->tdSelection.bsBegin.bpBi )
	{ return 1;	}

    *ss= td->tdSelectionScope;
    *bs= td->tdSelection;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Determine the derived properties such as line number and		*/
/*  coordinates of the current selection.				*/
/*									*/
/************************************************************************/

void tedDelimitCurrentSelection(	TedDocument *		td,
					AppDrawingData *	add )
    {
    BufferDocument *		bd= td->tdDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);

    docFindLineAndParticule( td->tdSelection.bsBegin.bpBi,
					td->tdSelection.bsBegin.bpStroff,
					&(td->tdSelection.bsBegin), 1 );
    docFindLineAndParticule( td->tdSelection.bsEnd.bpBi,
					td->tdSelection.bsEnd.bpStroff,
					&(td->tdSelection.bsEnd), 0 );

    tedSelectionCoordinates( &(td->tdSelection), add );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
					     add, dp, &(td->tdSelection) );
    }

/************************************************************************/
/*									*/
/*  Is the selection exactly an object.					*/
/*									*/
/************************************************************************/

int tedGetObjectSelection(	TedDocument *   	td,
				BufferPosition *	bpObject,
				InsertedObject **	pIo	)
    {
    const BufferSelection *	bs= &(td->tdSelection);

    if  ( bs->bsBegin.bpBi					&&
	  bs->bsEnd.bpBi == bs->bsBegin.bpBi	)
	{
	BufferItem *		bi= bs->bsBegin.bpBi;
	int			part= bs->bsBegin.bpParticule;
	TextParticule *		tp= bi->biParaParticules+ part;

	if  ( tp->tpKind == DOCkindOBJECT			&&
	      bs->bsBegin.bpStroff == tp->tpStroff		&&
	      bs->bsEnd.bpStroff == tp->tpStroff+ tp->tpStrlen	)
	    {
	    *bpObject= bs->bsBegin;
	    *pIo= bi->biParaObjects+ tp->tpObjectNumber;

	    return 0;
	    }
	}

    return -1;
    }

static void tedScrollToPosition(	EditDocument *		ed,
					const BufferPosition *	bp,
					int *			pScrolledX,
					int *			pScrolledY )
    {
    AppDrawingData *		add= &(ed->edDrawingData);

    appScrollToRectangle( ed,
			bp->bpXPixels, BP_TOP_PIXELS( add, bp ),
			bp->bpXPixels, bp->bpY1Pixels,
			pScrolledX, pScrolledY );

    return;
    }


/************************************************************************/
/*									*/
/*  Extend a selection that begins outside a table cell and ends inside	*/
/*  that cell.								*/
/*									*/
/*  The speccial cases that occur when the selection begins in a table	*/
/*  are covered by tedExtendSelectionFromTable(), that has priority	*/
/*  over tedExtendSelectionIntoTable().					*/
/*									*/
/************************************************************************/

static int tedExtendSelectionIntoTable(	EditDocument *		ed,
					BufferPosition *	bpEnd )
    {
    BufferItem *	cellBi= bpEnd->bpBi->biParent;
    BufferItem *	rowBi=	cellBi->biParent;

    int			col1= rowBi->biChildCount -1;

    docLastPosition( rowBi->biChildren[col1], bpEnd );
    tedPositionCoordinates( bpEnd, &(ed->edDrawingData) );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Extend a selection that begins inside a table cell and ends outside	*/
/*  that cell.								*/
/*									*/
/*  1)  Not in same section.						*/
/*  2)  Look for selection end in same table.				*/
/*  3)  Found?								*/
/*									*/
/************************************************************************/

static int tedExtendSelectionFromTable(	EditDocument *		ed,
					BufferPosition *	bpBegin,
					BufferPosition *	bpEnd,
					int *			pCol0,
					int *			pCol1 )
    {
    BufferItem *	cellBi= bpBegin->bpBi->biParent;
    BufferItem *	rowBi=	cellBi->biParent;
    BufferItem *	sectBi= rowBi->biParent;

    BufferItem *	endRowBi= bpEnd->bpBi->biParent->biParent;

    int			row1= rowBi->biNumberInParent;

    int			col0= -1;
    int			col1= -1;

    /*  1  */
    if  ( bpEnd->bpBi->biParent->biParent->biParent != sectBi )
	{
	docFirstPosition( rowBi->biChildren[0], bpBegin );
	tedPositionCoordinates( bpBegin, &(ed->edDrawingData) );

	if  ( bpEnd->bpBi->biParaInTable )
	    { tedExtendSelectionIntoTable( ed, bpEnd ); }

	return 0;
	}

    /*  2  */
    while( row1 < sectBi->biChildCount- 1				&&
	   docAlignedColumns( &(rowBi->biRowProperties),
	       &(sectBi->biChildren[row1+1]->biRowProperties) )	)
	{
	if  ( endRowBi == sectBi->biChildren[row1] )
	    { break;	}

	row1++;
	}

    if  ( endRowBi == sectBi->biChildren[row1] )
	{
	if  ( bpEnd->bpBi->biParent->biNumberInParent <
					    cellBi->biNumberInParent )
	    {
	    col0= bpEnd->bpBi->biParent->biNumberInParent;
	    col1= bpBegin->bpBi->biParent->biNumberInParent;

	    docFirstPosition( rowBi->biChildren[col0], bpBegin );
	    rowBi= bpEnd->bpBi->biParent->biParent;
	    docLastPosition( rowBi->biChildren[col1], bpEnd );
	    }
	else{
	    docFirstPosition( bpBegin->bpBi->biParent, bpBegin );
	    docLastPosition( bpEnd->bpBi->biParent, bpEnd );
	    }

	tedPositionCoordinates( bpBegin, &(ed->edDrawingData) );
	tedPositionCoordinates( bpEnd, &(ed->edDrawingData) );
	col0= bpBegin->bpBi->biParent->biNumberInParent;
	col1= bpEnd->bpBi->biParent->biNumberInParent;
	}
    else{
	docFirstPosition( rowBi->biChildren[0], bpBegin );
	tedPositionCoordinates( bpBegin, &(ed->edDrawingData) );

	if  ( bpEnd->bpBi->biParaInTable )
	    { tedExtendSelectionIntoTable( ed, bpEnd ); }

	return 0;
	}

    *pCol0= col0; *pCol1= col1; return 0;
    }

/************************************************************************/
/*									*/
/*  Extend the selection upon subsequent MotionNotify events.		*/
/*									*/
/************************************************************************/

static void tedSetExtendedSelection(	EditDocument *		ed,
					const SelectionScope *	ss,
					int			exposeSelection,
					int			col0,
					int			col1,
					int			dir,
					const BufferPosition *	bpExposeBegin,
					const BufferPosition *	bpExposeEnd,
					const BufferPosition *	bpTo,
					const BufferPosition *	bpBegin,
					const BufferPosition *	bpEnd )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    BufferSelection		bsExpose;

    int				scrolledX= 0;
    int				scrolledY= 0;

    bsExpose.bsBegin= *bpExposeBegin;
    bsExpose.bsEnd= *bpExposeEnd;
    bsExpose.bsCol0= bsExpose.bsCol1= -1;

    tedScrollToPosition( ed, bpTo, &scrolledX, &scrolledY );

    if  ( exposeSelection )
	{ tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY ); }

    tedExposeSelection( ed, &bsExpose, scrolledX, scrolledY );

    td->tdSelection.bsBegin= *bpBegin;
    td->tdSelection.bsEnd= *bpEnd;
    td->tdSelection.bsCol0= col0;
    td->tdSelection.bsCol1= col1;
    td->tdSelection.bsDirection= dir;

    td->tdSelectionScope= *ss;

    if  ( exposeSelection )
	{ tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY ); }

    return;
    }

static void tedBalanceFieldSelection(	EditDocument *		ed,
					int *			pBalanced,
					BufferPosition *	bpBegin,
					BufferPosition *	bpEnd )
    {
    int				beginMoved= 0;
    int				endMoved= 0;

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const AppDrawingData *	add= &(ed->edDrawingData);

    docBalanceFieldSelection( &beginMoved, &endMoved, td->tdDocument,
							    bpBegin, bpEnd );

    if  ( beginMoved )
	{
	*pBalanced= 1;
	tedPositionCoordinates( bpBegin, add );
	}

    if  ( endMoved )
	{
	*pBalanced= 1;
	tedPositionCoordinates( bpEnd, add );
	}

    return;
    }

int tedExtendSelectionToPosition(	EditDocument *		ed,
					const SelectionScope *	ss,
					const BufferPosition *	bpAnchor,
					const BufferPosition *	bpFound )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    int				col0= -1;
    int				col1= -1;

    BufferPosition		bpFrom;
    BufferPosition		bpTo;

    int				directionToAnchor;
    int				directionBeginAnchor;
    int				directionEndAnchor;
    int				dir;

    int				scrolledX= 0;
    int				scrolledY= 0;

    const int			mindLine= 1;

    int				cellChanged= 0;
    int				balanced= 0;
    int				exposeSelection;

    bpFrom= *bpAnchor;
    bpTo= *bpFound;

    directionToAnchor= docComparePositions( &bpTo, bpAnchor, mindLine );
    directionBeginAnchor= docComparePositions(
			    &(td->tdSelection.bsBegin), bpAnchor, mindLine );
    directionEndAnchor= docComparePositions(
			    &(td->tdSelection.bsEnd), bpAnchor, mindLine );

    /********************/
    /*  Before anchor.	*/
    /********************/
    if  ( directionToAnchor < 0 )
	{
	int		directionToBegin;

	if  ( docPositionsInsideCell( &bpTo, &bpFrom ) )
	    {
	    if  ( bpFrom.bpBi->biParaInTable )
		{ col0= col1= bpFrom.bpBi->biParent->biNumberInParent;	}
	    }
	else{
	    if  ( bpTo.bpBi->biParaInTable )
		{
		if  ( tedExtendSelectionFromTable( ed, &bpTo, &bpFrom,
							    &col0, &col1 ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( bpFrom.bpBi->biParaInTable )
		    {
		    if  ( tedExtendSelectionIntoTable( ed, &bpFrom ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }

	tedBalanceFieldSelection( ed, &balanced, &bpTo, &bpFrom );

	directionToBegin= docComparePositions( &bpTo,
				    &(td->tdSelection.bsBegin), mindLine );
	cellChanged= ! docPositionsInsideCell(
					&bpTo, &(td->tdSelection.bsBegin) );

	/****************************************/
	/*  Undraw selection after the anchor.	*/
	/****************************************/
	if  ( directionEndAnchor > 0	||
	      balanced			||
	      cellChanged		)
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	/************************/
	/*  Extended Left.	*/
	/************************/
	if  ( directionToBegin < 0 )
	    {
	    dir= -1;
	    exposeSelection=	directionEndAnchor > 0	||
				balanced		||
				cellChanged		;

	    tedSetExtendedSelection( ed, ss, exposeSelection, col0, col1, dir,
				&bpTo, &(td->tdSelection.bsBegin),
				&bpTo, &bpTo, &bpFrom );

	    return 0;
	    }

	/************************/
	/*  Shrunk Left.	*/
	/************************/
	if  ( directionToBegin > 0 )
	    {
	    dir= -1;
	    exposeSelection=	directionEndAnchor > 0	||
				balanced		||
				cellChanged		;

	    tedSetExtendedSelection( ed, ss, exposeSelection, col0, col1, dir,
				    &(td->tdSelection.bsBegin), &bpTo,
				    &bpTo, &bpTo, &bpFrom );

	    return 0;
	    }

	return 0;
	}

    /********************/
    /*  After anchor.	*/
    /********************/
    if  ( directionToAnchor > 0 )
	{
	int		directionToEnd;

	if  ( docPositionsInsideCell( &bpTo, &bpFrom ) )
	    {
	    if  ( bpFrom.bpBi->biParaInTable )
		{ col0= col1= bpFrom.bpBi->biParent->biNumberInParent;	}
	    }
	else{
	    if  ( bpFrom.bpBi->biParaInTable )
		{
		if  ( tedExtendSelectionFromTable( ed, &bpFrom, &bpTo,
							    &col0, &col1 ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		if  ( bpTo.bpBi->biParaInTable )
		    {
		    if  ( tedExtendSelectionIntoTable( ed, &bpTo ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }

	tedBalanceFieldSelection( ed, &balanced, &bpFrom, &bpTo );

	directionToEnd= docComparePositions( &bpTo,
					&(td->tdSelection.bsEnd), mindLine );
	cellChanged= ! docPositionsInsideCell(
					&bpTo, &(td->tdSelection.bsEnd) );

	/****************************************/
	/*  Undraw selection before the anchor.	*/
	/****************************************/
	if  ( directionBeginAnchor < 0	||
	      balanced			||
	      cellChanged		)
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	/************************/
	/*  Extended Right.	*/
	/************************/
	if  ( directionToEnd > 0 )
	    {
	    dir= 1;
	    exposeSelection=	directionBeginAnchor < 0	||
				balanced			||
				cellChanged			;

	    tedSetExtendedSelection( ed, ss, exposeSelection, col0, col1, dir,
				    &(td->tdSelection.bsEnd), &bpTo,
				    &bpTo, &bpFrom, &bpTo );

	    return 0;
	    }

	/************************/
	/*  Shrunk Right.	*/
	/************************/
	if  ( directionToEnd < 0 )
	    {
	    dir= 1;
	    exposeSelection=	directionBeginAnchor < 0	||
				balanced			||
				cellChanged			;

	    tedSetExtendedSelection( ed, ss, exposeSelection, col0, col1, dir,
				    &bpTo, &(td->tdSelection.bsEnd),
				    &bpTo, &bpFrom, &bpTo );

	    return 0;
	    }

	return 0;
	}

    /********************/
    /*  At anchor.	*/
    /********************/
	{
	bpFrom= *bpAnchor;
	bpTo= *bpAnchor;

	tedBalanceFieldSelection( ed, &balanced, &bpFrom, &bpTo );

	/****************************************/
	/*  Undraw selection before the anchor.	*/
	/*  Undraw selection after the anchor.	*/
	/****************************************/
	if  ( directionBeginAnchor < 0	||
	      directionEndAnchor > 0	)
	    {
	    tedExposeSelection( ed, &(td->tdSelection), scrolledX, scrolledY );
	    }

	dir= 0;
	exposeSelection=	directionBeginAnchor < 0	||
				directionEndAnchor > 0		||
				balanced			;

	tedSetExtendedSelection( ed, ss, exposeSelection, col0, col1, dir,
				&bpFrom, &bpTo,
				&bpFrom, &bpFrom, &bpTo );
	}

    return 0;
    }

int tedExtendSelectionToXY(	EditDocument *		ed,
				BufferItem *		rootBi,
				const BufferPosition *	bpAnchor,
				const SelectionScope *	ss,
				int			mouseX,
				int			mouseY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);

    BufferPosition		bpTo;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    bpTo= *bpAnchor;

    if  ( tedFindPosition( rootBi, add, mouseX+ ox, mouseY+ oy, &bpTo ) )
	{ /*LLDEB(mouseX,mouseY);*/ return 0; }

    /*  superfluous: done by tedFindPosition()
    tedPositionCoordinates( &bpTo, &(ed->edDrawingData) );
    */

    return tedExtendSelectionToPosition( ed, ss, bpAnchor, &bpTo );
    }

/************************************************************************/
/*									*/
/*  Move to the next/previous position.					*/
/*									*/
/************************************************************************/

int tedFirstPosition(	const AppDrawingData *	add,
			BufferDocument *	bd,
			BufferPosition *	bp	)
    {
    if  ( docFirstPosition( &(bd->bdItem), bp ) )
	{ return -1;	}

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedLastPosition(	const AppDrawingData *	add,
			BufferDocument *	bd,
			BufferPosition *	bp	)
    {
    if  ( docLastPosition( &(bd->bdItem), bp ) )
	{ return -1;	}

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedNextPosition(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    if  ( docNextPosition( bp ) )
	{ return -1;	}

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedPrevPosition(	const AppDrawingData *	add,
			BufferPosition *	bp,
			int			lastOne )
    {
    if  ( docPrevPosition( bp, lastOne ) )
	{ return -1;	}

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedBeginOfLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    BufferItem *	bi= bp->bpBi;
    TextLine *		tl= bi->biParaLines+ bp->bpLine;
    int			part= tl->tlFirstParticule;
    TextParticule *	tp= bi->biParaParticules+ part;

    bp->bpParticule= part;
    bp->bpStroff= tp->tpStroff;

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedEndOfLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    BufferItem *	bi= bp->bpBi;
    TextLine *		tl= bi->biParaLines+ bp->bpLine;
    int			part= tl->tlFirstParticule+ tl->tlParticuleCount- 1;
    TextParticule *	tp= bi->biParaParticules+ part;

    bp->bpParticule= part;
    bp->bpStroff= tp->tpStroff+ tp->tpStrlen;

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedPrevLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    int			part;
    int			off;

    int			x= bp->bpXPixels;
    int			baseline;

    if  ( docPrevLine( &tp, &tl, bp ) )
	{ return -1;	}

    baseline= TL_BASE_PIXELS( add, tl );

    part= tedFindParticule( tl, bp->bpBi->biParaParticules, x, baseline );
    if  ( part < 0 )
	{ LDEB(part); return -1;	}

    off= tedFindStringOffset( bp->bpBi->biParaParticules+ part,
				bp->bpBi->biParaString, add,
				&x, x, baseline );
    if  ( off < 0 )
	{ LDEB(off); return -1;	}

    bp->bpStroff= off;
    bp->bpParticule= part;

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedNextLine(	const AppDrawingData *	add,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    int			part;
    int			off;

    int			x= bp->bpXPixels;
    int			baseline;

    if  ( docNextLine( &tp, &tl, bp ) )
	{ return -1;	}

    baseline= TL_BASE_PIXELS( add, tl );

    part= tedFindParticule( tl, bp->bpBi->biParaParticules, x, baseline );
    if  ( part < 0 )
	{ LDEB(part); return -1;	}

    off= tedFindStringOffset( bp->bpBi->biParaParticules+ part,
				bp->bpBi->biParaString, add,
				&x, x, baseline );
    if  ( off < 0 )
	{ LDEB(off); return -1;	}

    bp->bpStroff= off;
    bp->bpParticule= part;

    tedPositionCoordinates( bp, add );

    return 0;
    }

int tedPageUp(	BufferPosition *	bp,
		BufferDocument *	bd,
		const AppDrawingData *	add,
		int			pageHeight )
    {
    int			x= bp->bpXPixels;
    int			y;

    y= bp->bpBaselinePixels- pageHeight;

    if  ( y < 0 )
	{ y= 0;	}

    if  ( tedFindPosition( &(bd->bdItem), add, x, y, bp ) )
	{ return -1;	}

    return 0;
    }

int tedPageDown(	BufferPosition *	bp,
			BufferDocument *	bd,
			const AppDrawingData *	add,
			int			docHeight,
			int			pageHeight )
    {
    int			x= bp->bpXPixels;
    int			y;

    y= bp->bpBaselinePixels+ pageHeight;

    if  ( y >= docHeight )
	{ y= docHeight- 1;	}

    if  ( tedFindPosition( &(bd->bdItem), add, x, y, bp ) )
	{ return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Scroll a selection into view.					*/
/*									*/
/************************************************************************/

void tedScrollToSelection(	EditDocument *		ed,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    const AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferSelection *	bs= &(td->tdSelection);

    appScrollToRectangle( ed,
		    bs->bsBegin.bpXPixels,
		    BP_TOP_PIXELS( add, &(bs->bsBegin) ),
		    bs->bsEnd.bpXPixels, bs->bsEnd.bpY1Pixels,
		    pScrolledX, pScrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Remember the text attribute of the beginning of the selection.	*/
/*									*/
/*  NOTE that for I-Bar selections, in case of ambiguity, there is a	*/
/*	preference for the attribute of the particule before the	*/
/*	current position.						*/
/*									*/
/************************************************************************/

static void tedSetCurrentTextAttribute(		TedDocument *		td,
						int			IBar,
						const BufferPosition *	bp )
    {
    BufferItem *		bi= bp->bpBi;
    TextParticule *		tp= bi->biParaParticules+ bp->bpParticule;

    if  ( IBar				&&
	  bp->bpParticule > 0		&&
	  bp->bpStroff == tp->tpStroff	&&
	  tp[-1].tpKind == DOCkindTEXT	)
	{ tp--; }

    td->tdCurrentTextAttribute= tp->tpTextAttribute;
    td->tdCurrentPhysicalFont= tp->tpPhysicalFont;

    return;
    }

/************************************************************************/
/*									*/
/*  A selection was made, do bookkeeping and provoke drawing.		*/
/*									*/
/*  0)  If an object (Picture) was selected, remove the special window.	*/
/*  1)  Provoke the old selection to be redrawn. (Before scrolling)	*/
/*  2)  Stop Cursor blinking.						*/
/*  3)  Scroll to the new selection.					*/
/*  4)  Provoke the old selection to be redrawn. (It might have moved)	*/
/*  5)  Bookkeeping.							*/
/*  6)  Provoke the new selection to be redrawn.			*/
/*									*/
/*  NOTE:	As scrolling to the new selection can provoke a redraw	*/
/*		the selection needs to be set before the scrollbar is	*/
/*		adapted. Otherwise the selection administration can	*/
/*		still refer to deleted text.				*/
/*									*/
/*  A)  If the selection is partially inside and partially outside a	*/
/*	text level field, force it to contain the whole field. This is	*/
/*	to enforce proper text field nesting.				*/
/*									*/
/************************************************************************/

void tedSetSelection(	EditDocument *		ed,
			const SelectionScope *	ssSet,
			const BufferSelection *	bsSet,
			int *			pScrolledX,
			int *			pScrolledY )
    {
    EditApplication *		ea= ed->edApplication;

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    AppDrawingData *		add= &(ed->edDrawingData);
    int				hadSelection;

    SelectionScope		ssOld;
    BufferSelection		bsOld;

    BufferSelection		bsNew;

    int				beginMoved= 0;
    int				endMoved= 0;

    DocumentRectangle		drExpose;

    DocumentRectangle		drOldHeadFoot;
    DocumentRectangle		drNewHeadFoot;
    int				movedFromHeaderFooter= 0;
    int				movedToHeaderFooter= 0;

    HeaderFooter *		hf= (HeaderFooter *)0;
    BufferItem *		sectBi;

    /*  0  */
    if  ( td->tdObjectSelected )
	{ tedHideObjectWindows( ed );	}

    if  ( tedGetSelection( &ssOld, &bsOld, td ) )
	{ LDEB(1); return;	}

    if  ( ssOld.ssInHeaderFooter != DOCinBODY )
	{
	if  ( docGetHeaderFooter( &hf, &sectBi,
				&ssOld, &bsOld, bd, ssOld.ssInHeaderFooter ) )
	    { LDEB(ssOld.ssInHeaderFooter);	}
	else{
	    if  ( ssOld.ssInHeaderFooter != ssSet->ssInHeaderFooter )
		{ movedFromHeaderFooter= 1;	}
	    if  ( ssOld.ssHeaderFooterSection != ssSet->ssHeaderFooterSection )
		{ movedFromHeaderFooter= 1;	}
	    if  ( ssOld.ssHeaderFooterPage != ssSet->ssHeaderFooterPage )
		{ movedFromHeaderFooter= 1;	}

	    if  ( movedFromHeaderFooter )
		{
		if  ( tedGetHeaderFooterBox( &drOldHeadFoot, sectBi, dp, add,
						ssOld.ssHeaderFooterPage,
						ssOld.ssInHeaderFooter ) )
		    {
		    LDEB(ssOld.ssInHeaderFooter);
		    movedFromHeaderFooter= 0;
		    }
		}
	    }
	}

    if  ( ssSet->ssInHeaderFooter != DOCinBODY )
	{
	if  ( docGetHeaderFooter( &hf, &sectBi,
				ssSet, &bsNew, bd, ssSet->ssInHeaderFooter ) )
	    { LDEB(ssSet->ssInHeaderFooter);	}
	else{
	    int		done= 0;

	    if  ( ssOld.ssInHeaderFooter != ssSet->ssInHeaderFooter )
		{ movedToHeaderFooter= 1;	}
	    if  ( ssOld.ssHeaderFooterSection != ssSet->ssHeaderFooterSection )
		{ movedToHeaderFooter= 1;	}
	    if  ( ssOld.ssHeaderFooterPage != ssSet->ssHeaderFooterPage )
		{ movedToHeaderFooter= 1;	}

	    if  ( movedToHeaderFooter )
		{
		if  ( tedGetHeaderFooterBox( &drNewHeadFoot, sectBi, dp, add,
						ssSet->ssHeaderFooterPage,
						ssSet->ssInHeaderFooter ) )
		    { LDEB(ssSet->ssInHeaderFooter); movedToHeaderFooter= 0; }
		}

	    if  ( tedRedoHeaderFooterLayout( &done, hf,
						ssSet->ssHeaderFooterPage,
						sectBi, bd, add,
						&drNewHeadFoot ) )
		{ LDEB(1);	}
	    }
	}

    /*  A  */
    bsNew= *bsSet;
    docBalanceFieldSelection( &beginMoved, &endMoved, td->tdDocument,
					    &(bsNew.bsBegin), &(bsNew.bsEnd) );

    hadSelection= tedHasSelection( td );
    if  ( hadSelection )
	{
	/*  1  */
	int	hadIBarSelection= tedHasIBarSelection( td );

	drExpose= td->tdSelectedRectangle;

	if  ( movedFromHeaderFooter )
	    { docUnionRectangle( &drExpose, &drExpose, &drOldHeadFoot ); }

	tedExposeRectangle( ed, &drExpose, *pScrolledX, *pScrolledY );

	/*  2  */
	if  ( hadIBarSelection )
	    { tedStopCursorBlink( ed ); }
	}

    /*  5  */
    if  ( bsNew.bsBegin.bpBi->biParaInTable			&&
	  bsNew.bsEnd.bpBi->biParaInTable			&&
	  bsNew.bsBegin.bpBi->biParent->biParent ==
			bsNew.bsEnd.bpBi->biParent->biParent	)
	{
	bsNew.bsCol0= bsNew.bsBegin.bpBi->biParent->biNumberInParent;
	bsNew.bsCol1= bsNew.bsEnd.bpBi->biParent->biNumberInParent;
	}

    /*  5  */
    tedSelectionCoordinates( &bsNew, add );
    td->tdVisibleSelectionCopied= 0;
    td->tdSelection= bsNew;
    td->tdSelectionScope= *ssSet;

    if  ( td->tdSelectionScope.ssInHeaderFooter == DOCinBODY )
	{
	td->tdSelectionScope.ssHeaderFooterSection= -1;
	td->tdSelectionScope.ssHeaderFooterPage= -1;
	}

    /*  3  */
    tedScrollToSelection( ed, pScrolledX, pScrolledY );

    if  ( hadSelection )
	{
	/*  4  */
	drExpose= td->tdSelectedRectangle;

	if  ( movedFromHeaderFooter )
	    { docUnionRectangle( &drExpose, &drExpose, &drOldHeadFoot ); }

	tedExposeRectangle( ed, &drExpose, *pScrolledX, *pScrolledY );
	}

    /*  6  */
    tedSelectionRectangle( &(td->tdSelectedRectangle),
						add, dp, &(td->tdSelection) );

    drExpose= td->tdSelectedRectangle;

    if  ( movedToHeaderFooter )
	{ docUnionRectangle( &drExpose, &drExpose, &drNewHeadFoot ); }

    tedExposeRectangle( ed, &drExpose, *pScrolledX, *pScrolledY );

    if  ( tedHasIBarSelection( td ) )
	{ tedSetCurrentTextAttribute( td, 1, &(td->tdSelection.bsBegin) ); }
    else{ tedSetCurrentTextAttribute( td, 0, &(td->tdSelection.bsBegin) ); }

    if  ( ea->eaFindTool )
	{ appFindToolDisableReplace( ea->eaFindTool );	}

    return;
    }

void tedSetSelectedPosition(	EditDocument *		ed,
				const SelectionScope *	ssSet,
				const BufferPosition *	bpSet,
				int *			pScrolledX,
				int *			pScrolledY )
    {
    BufferSelection		bsNew;

    bsNew.bsBegin= *bpSet;
    bsNew.bsEnd= *bpSet;
    bsNew.bsAnchor= *bpSet;
    bsNew.bsEnd= *bpSet;
    bsNew.bsCol0= -1;
    bsNew.bsCol1= -1;
    bsNew.bsDirection= 0;

    tedSetSelection( ed, ssSet, &bsNew, pScrolledX, pScrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  'Select All from the 'Edit' menu.					*/
/*									*/
/************************************************************************/

void tedDocSelAll(		EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    SelectionScope		ssNew;
    BufferSelection		bsNew;

    int				scrolledX= 0;
    int				scrolledY= 0;

    docInitSelectionScope( &ssNew );
    docInitSelection( &bsNew );

    if  ( docFirstPosition( &(bd->bdItem), &(bsNew.bsBegin) ) )
	{ LDEB(1); return;	}

    if  ( docLastPosition( &(bd->bdItem), &(bsNew.bsEnd) ) )
	{ LDEB(1); return;	}

    bsNew.bsDirection= 1; bsNew.bsAnchor= bsNew.bsBegin;
    tedSetSelection( ed, &ssNew, &bsNew, &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return;
    }


/************************************************************************/
/*									*/
/*  Change the selection to cover whole paragraphs.			*/
/*									*/
/*  Depending on the 'direction' argument:				*/
/*	<  0:	The previous one.					*/
/*	== 0:	The current one.					*/
/*	>  0:	The next one.						*/
/*									*/
/************************************************************************/

int tedSelectWholeParagraph(	EditApplication *	ea,
				int			direction )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;

    BufferSelection		bsNew;

    int				scrolledX= 0;
    int				scrolledY= 0;

    if  ( ! bi )
	{ XDEB(bi); return -1;	}

    if  ( direction > 0 )
	{ bi= docNextParagraph( bi );	}
    if  ( direction < 0 )
	{ bi= docPrevParagraph( bi );	}

    if  ( ! bi )
	{ return -1;	}

    if  ( direction == 0 )
	{ direction= 1;	}

    docInitSelection( &bsNew );

    docSetParaSelection( &bsNew, bi, direction, 0, bi->biParaStrlen );

    tedSetSelection( ed, &(td->tdSelectionScope), &bsNew, &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Change the selection to cover whole sections.			*/
/*									*/
/*  Depending on the 'direction' argument:				*/
/*	<  0:	The previous one.					*/
/*	== 0:	The current one.					*/
/*	>  0:	The next one.						*/
/*									*/
/************************************************************************/

int tedSelectWholeSection(	EditApplication *	ea,
				int			direction )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferItem *		biBegin= td->tdSelection.bsBegin.bpBi;
    BufferItem *		biEnd= td->tdSelection.bsEnd.bpBi;

    BufferSelection		bsNew;

    int				scrolledX= 0;
    int				scrolledY= 0;

    if  ( ! biBegin || ! biEnd )
	{ XXDEB(biBegin,biEnd); return -1;	}

    while( biBegin && biBegin->biLevel > DOClevSECT )
	{ biBegin= biBegin->biParent;	}
    while( biEnd && biEnd->biLevel > DOClevSECT )
	{ biEnd= biEnd->biParent;	}

    if  ( ! biBegin || ! biEnd )
	{ XXDEB(biBegin,biEnd); return -1;	}

    if  ( direction > 0 )
	{
	biBegin= docNextSection( biBegin );
	biEnd= docNextSection( biEnd );
	}
    if  ( direction < 0 )
	{
	biBegin= docPrevSection( biBegin );
	biEnd= docPrevSection( biEnd );
	}

    if  ( ! biBegin || ! biEnd )
	{ return -1;	}

    if  ( direction == 0 )
	{ direction= 1;	}

    docInitSelection( &bsNew );

    if  ( docFirstPosition( biBegin, &(bsNew.bsBegin) ) )
	{ LDEB(1); return -1;	}

    if  ( docLastPosition( biEnd, &(bsNew.bsEnd) ) )
	{ LDEB(1); return -1;	}

    bsNew.bsDirection= direction;

    tedSetSelection( ed, &(td->tdSelectionScope), &bsNew,
						    &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Determine the rectangular area as it is selected in a table.	*/
/*									*/
/*  This funtion fails if the whole selection is not inside ONE table.	*/
/*									*/
/************************************************************************/

int tedGetTableRectangle(	TableRectangle *	tr,
				const SelectionScope *	ss,
				const BufferSelection *	bs )
    {
    BufferItem *	sectBi0;
    BufferItem *	sectBi1;

    TableRectangle	trBegin;
    TableRectangle	trEnd;

    if  ( docDelimitTable( bs->bsBegin.bpBi, &sectBi0,
						&(trBegin.trCol0),
						&(trBegin.trRow00),
						&(trBegin.trRow0),
						&(trBegin.trRow11) ) )
	{ /* LDEB(1); */ return -1;	}

    if  ( docDelimitTable( bs->bsEnd.bpBi, &sectBi1,
						&(trEnd.trCol0),
						&(trEnd.trRow00),
						&(trEnd.trRow0),
						&(trEnd.trRow11) ) )
	{ /* LDEB(1); */ return -1;	}

    if  ( sectBi0 != sectBi1 )
	{ /* XXDEB(sectBi0,sectBi1); */ return -1;	}

    if  ( trBegin.trRow00 != trEnd.trRow00	||
	  trBegin.trRow11 != trEnd.trRow11	)
	{ return -1;	}

    if  ( trBegin.trCol0 > trEnd.trCol0 )
	{ return -1;	}
    if  ( trBegin.trRow0 > trEnd.trRow0 )
	{ return -1;	}

    if  ( ss->ssInHeaderFooter == DOCinBODY )
	{ tr->trSectionNumber= sectBi0->biNumberInParent;	}
    else{ tr->trSectionNumber= ss->ssHeaderFooterSection;	}

    tr->trCol0= trBegin.trCol0;
    tr->trCol1= trEnd.trCol0;
    tr->trCol11= sectBi0->biChildren[trBegin.trRow0]->biChildCount- 1;

    tr->trRow00= trBegin.trRow00;
    tr->trRow0= trBegin.trRow0;
    tr->trRow1= trEnd.trRow0;
    tr->trRow11= trBegin.trRow11;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Table related menu option callbacks.				*/
/*									*/
/************************************************************************/

void tedDocTableSelectTableRectangle(	EditDocument *		ed,
					const SelectionScope *	ss,
					const TableRectangle *	tr )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;

    BufferSelection	bs;
    BufferItem *	sectBi;

    int			scrolledX= 0;
    int			scrolledY= 0;

    if  ( docTableRectangleSelection( &bs, &sectBi, bd, ss, tr ) )
	{ LDEB(1); return;	}

    tedSetSelection( ed, ss, &bs, &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return;
    }

void tedDocTableSelectTable(	APP_WIDGET	option,
				void *		voided,
				void *		voidpbcbs )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    TableRectangle	tr;

    if  ( tedGetTableRectangle( &tr,
			    &(td->tdSelectionScope), &(td->tdSelection) ) )
	{ LDEB(1); return;	}

    docExpandTableRectangleToWholeTable( &tr );

    tedDocTableSelectTableRectangle( ed, &(td->tdSelectionScope), &tr );

    return;
    }

void tedDocTableSelectRow(	APP_WIDGET	option,
				void *		voided,
				void *		voidpbcbs )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    TableRectangle	tr;

    if  ( tedGetTableRectangle( &tr,
			    &(td->tdSelectionScope), &(td->tdSelection) ) )
	{ LDEB(1); return;	}

    docExpandTableRectangleToWholeColumns( &tr );

    tedDocTableSelectTableRectangle( ed, &(td->tdSelectionScope), &tr );

    return;
    }

void tedDocTableSelectColumn(	APP_WIDGET	option,
				void *		voided,
				void *		voidpbcbs )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    TableRectangle	tr;

    if  ( tedGetTableRectangle( &tr,
			    &(td->tdSelectionScope), &(td->tdSelection) ) )
	{ LDEB(1); return;	}

    docExpandTableRectangleToWholeRows( &tr );

    tedDocTableSelectTableRectangle( ed, &(td->tdSelectionScope), &tr );

    return;
    }

/************************************************************************/
/*									*/
/*  Move the current selection to a certain header/Footer. If the	*/
/*  header or footer does not exist, make it.				*/
/*									*/
/************************************************************************/

static int TED_TryPageOffsets[]=
    {
     0,
     1,
    -1,
     2
    };

void tedAppEditHeaderFooter(		EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					int			which )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    AppDrawingData *		add;
    TedDocument *		td;
    BufferDocument *		bd;
    const DocumentProperties *	dp;

    HeaderFooter *		hf;
    HeaderFooter *		hfTry;
    BufferItem *		sectBi;

    SelectionScope		ss;
    BufferSelection		bs;
    BufferPosition		bp;

    int				page= -1;
    int				pg;
    int				i;

    int				scrolledX= 0;
    int				scrolledY= 0;

    SelectionScope		ssHeadFoot;

    DocumentRectangle		drChanged;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    add= &(ed->edDrawingData);
    td= (TedDocument *)ed->edPrivateData;
    bd= td->tdDocument;
    dp= &(bd->bdProperties);

    drChanged= td->tdSelectedRectangle;

    if  ( tedGetSelection( &ss, &bs, td ) )
	{ LDEB(1); return;	}

    if  ( docGetHeaderFooter( &hf, &sectBi, &ss, &bs, bd, which ) )
	{ LDEB(which); return;	}

    if  ( ss.ssInHeaderFooter == which				&&
	  ss.ssHeaderFooterSection == sectBi->biNumberInParent	)
	{ return;	}

    if  ( ! hf->hfItem )
	{
	BufferItem *	paraBi;

	hf->hfItem= docMakeExternalItem(
				bd, &(sectBi->biSectProperties), which );
	if  ( ! hf->hfItem )
	    { XDEB(hf->hfItem); return;	}

	hf->hfPageUsedFor= -1;

	paraBi= docInsertEmptyParagraph( bd, hf->hfItem,
						td->tdCurrentTextAttribute );
	if  ( ! paraBi )
	    {
	    XDEB(paraBi);
	    docCleanHeaderFooter( bd, hf );
	    docInitHeaderFooter( hf );
	    return;
	    }

	appDocumentChanged( ed, 1 );
	}

    switch( which )
	{
	case DOCinFIRST_HEADER:
	case DOCinFIRST_FOOTER:
	    page= sectBi->biTopPosition.lpPage;
	    break;

	case DOCinSECT_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:
	    for ( i= 0; i < sizeof(TED_TryPageOffsets)/sizeof(int); i++ )
		{
		pg= bs.bsBegin.bpTopPosition.lpPage+ TED_TryPageOffsets[i];

		if  ( pg < sectBi->biTopPosition.lpPage		||
		      pg > sectBi->biBelowPosition.lpPage	)
		    { continue;	}

		if  ( docWhatPageHeader( &hfTry, sectBi, pg, dp ) == which )
		    { page= pg; break;	}
		}
	    break;

	case DOCinSECT_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:
	    for ( i= 0; i < sizeof(TED_TryPageOffsets)/sizeof(int); i++ )
		{
		pg= bs.bsBegin.bpTopPosition.lpPage+ TED_TryPageOffsets[i];

		if  ( pg < sectBi->biTopPosition.lpPage		||
		      pg > sectBi->biBelowPosition.lpPage	)
		    { continue;	}

		if  ( docWhatPageFooter( &hfTry, sectBi, pg, dp ) == which )
		    { page= pg; break;	}
		}
	    break;

	default:
	    LDEB(which); return;
	}

    if  ( page < sectBi->biTopPosition.lpPage		||
	  page > sectBi->biBelowPosition.lpPage		)
	{
	TedAppResources *	tar= (TedAppResources *)ea->eaResourceData;

	appQuestionRunErrorDialog( ea, relative, option,
					    tar->tarHeadFootNotReachable );
	return;
	}

    if  ( docFirstPosition( hf->hfItem, &bp ) )
	{ LDEB(1); return;	}

    docInitSelectionScope( &ssHeadFoot );
    ssHeadFoot.ssInHeaderFooter= which;
    ssHeadFoot.ssHeaderFooterSection= sectBi->biNumberInParent;
    ssHeadFoot.ssHeaderFooterPage= page;

    tedSetSelectedPosition( ed, &ssHeadFoot, &bp, &scrolledX, &scrolledY );

    tedAdaptToolsToPosition( ed );

    return;
    }

/************************************************************************/
/*									*/
/*  Is a posintion in a hyperlink,bookmark,field?			*/
/*									*/
/************************************************************************/

static void tedPositionEnvironment(	const BufferDocument *	bd,
					const BufferPosition *	bp,
					int *			pInHyperlink,
					int *			pInField,
					int *			pInBookmark )
    {
    int				startPart;
    int				endPart;
    const char *		markName;
    int				markSize;

    const char *		fileName;
    int				fileSize;

    *pInHyperlink= 0;
    *pInField= 0;
    *pInBookmark= 0;

    if  ( ! docGetHyperlinkForPosition( bd, bp, &startPart, &endPart,
				&fileName, &fileSize, &markName, &markSize ) )
	{ *pInField= *pInHyperlink= 1;	}
    else{
	*pInField= docParticuleInField( bp->bpBi, bp->bpParticule );
	}


    if  ( ! docGetBookmarkForPosition( bd, bp, &startPart, &endPart,
						    &markName, &markSize ) )
	{ *pInBookmark= 1;	}

    return;
    }


/************************************************************************/
/*									*/
/*  Adapt tools and rulers to the current position.			*/
/*									*/
/************************************************************************/

void tedAdaptToolsToPosition(	EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;
    TedAppResources *		tar= (TedAppResources *)ea->eaResourceData;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferDocument *	bd= td->tdDocument;
    const BufferPosition *	bp= &(td->tdSelection.bsBegin);

    int				inHyperlink;
    int				inField;
    int				inTable;
    int				inBookmark;
    int				canReplace;
    int				inBody;

    if  ( ! ed->edIsReadonly )
	{ td->tdCanReplaceSelection= 1;	}

    canReplace= td->tdCanReplaceSelection;
    inTable= bp->bpBi->biParaInTable= bp->bpBi->biParaInTable != 0;
    inBody= td->tdSelectionScope.ssInHeaderFooter == DOCinBODY;

    tedPositionEnvironment( bd, bp, &inHyperlink, &inField, &inBookmark );

    tedDocAdaptHorizontalRuler( ed, bp->bpBi );

    tedAdaptFontIndicatorsToSelection( ea, ed );

    appGuiEnableWidget( td->tdCopyWidget, 0 );
    appGuiEnableWidget( td->tdCutWidget, 0 );
    appGuiEnableWidget( td->tdPasteWidget, canReplace );
    appGuiEnableWidget( td->tdInsPictOption, canReplace );
    appGuiEnableWidget( td->tdInsFileOption, canReplace );

    appGuiEnableWidget( td->tdInsSymbolOption, canReplace );
    appGuiEnableWidget( td->tdFormatOneParaOption, 0 );
    appGuiEnableWidget( td->tdInsHyperlinkOption, inHyperlink );
    appGuiEnableWidget( td->tdInsBookmarkOption, inBookmark || ! inField );
    appGuiEnableWidget( td->tdInsInsertTableOption, canReplace && ! inTable );

    appGuiEnableWidget( td->tdInsInsertPageNumberOption, 
					    canReplace && ! inBody );

    appGuiEnableWidget( td->tdInsInsertLineBreakOption, canReplace );
    appGuiEnableWidget( td->tdInsInsertPageBreakOption,
					canReplace && inBody && ! inTable );
    appGuiEnableWidget( td->tdInsInsertSectBreakOption,
					canReplace && inBody && ! inTable );

    appGuiEnableWidget( td->tdTabInsertTableOption, ! inTable );
    appGuiEnableWidget( td->tdTabAddRowOption, inTable );
    appGuiEnableWidget( td->tdTabAddColumnOption, inTable );

    appGuiEnableWidget( td->tdSelectTableWidget, inTable );
    appGuiEnableWidget( td->tdSelectRowWidget, inTable );
    appGuiEnableWidget( td->tdSelectColumnOption, inTable );

    if  ( tar->tarInspector )
	{
	tedFormatToolAdaptToSelection( tar->tarInspector, bd,
				&(td->tdSelectionScope), &(td->tdSelection),
				ed->edFileReadOnly );
	}

    return;
    }

void tedAdaptToolsToSelection(	EditDocument *		ed )
    {
    EditApplication *		ea= ed->edApplication;
    TedAppResources *		tar= (TedAppResources *)ea->eaResourceData;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferDocument *	bd= td->tdDocument;

    int				inHyperlink;
    int				inField;
    int				inBookmark;

    const BufferPosition *	bpBegin= &(td->tdSelection.bsBegin);
    const BufferPosition *	bpEnd= &(td->tdSelection.bsEnd);

    TableRectangle		tr;

    int				inBody;
    int				canReplace;
    int				tableRectangle= 0;
    int				oneParagraph;

    inBody= td->tdSelectionScope.ssInHeaderFooter == DOCinBODY;
    oneParagraph= bpEnd->bpBi == bpBegin->bpBi;

    if  ( tedHasIBarSelection( td ) )
	{ tedAdaptToolsToPosition( ed ); return; }

    tedPositionEnvironment( bd, bpBegin, &inHyperlink, &inField, &inBookmark );

    if  ( ! tedGetTableRectangle( &tr,
			    &(td->tdSelectionScope), &(td->tdSelection) ) )
	{ tableRectangle= 1;	}

    tedDocAdaptHorizontalRuler( ed, bpBegin->bpBi );

    tedAdaptFontIndicatorsToSelection( ea, ed );

    appGuiEnableWidget( td->tdFormatOneParaOption,
			    ! oneParagraph &&
			    docSelectionInsideCell( &(td->tdSelection) ) );

    appGuiEnableWidget( td->tdCopyWidget, 1 );

    if  ( ed->edIsReadonly )
	{ td->tdCanReplaceSelection= 0;	}
    else{
	td->tdCanReplaceSelection=
		! tableRectangle ||
		( tr.trCol0 == tr.trCol1 && tr.trRow0 == tr.trRow1 );
	}

    canReplace= td->tdCanReplaceSelection;

    appGuiEnableWidget( td->tdCutWidget, canReplace );
    appGuiEnableWidget( td->tdPasteWidget, canReplace );

    appGuiEnableWidget( td->tdInsPictOption, canReplace );
    appGuiEnableWidget( td->tdInsFileOption, canReplace );
    appGuiEnableWidget( td->tdInsSymbolOption, canReplace );
    appGuiEnableWidget( td->tdInsHyperlinkOption, oneParagraph );
    appGuiEnableWidget( td->tdInsBookmarkOption,
				    oneParagraph &&
				    ( inBookmark || ! inField ) );
    appGuiEnableWidget( td->tdInsInsertTableOption, 0 );

    appGuiEnableWidget( td->tdInsInsertPageNumberOption, 
				    canReplace && oneParagraph && ! inBody );

    appGuiEnableWidget( td->tdTabInsertTableOption, 0 );

    appGuiEnableWidget( td->tdInsInsertLineBreakOption, 0 );
    appGuiEnableWidget( td->tdInsInsertPageBreakOption, 0 );
    appGuiEnableWidget( td->tdInsInsertSectBreakOption, 0 );

    appGuiEnableWidget( td->tdTabAddColumnOption, tableRectangle );
    appGuiEnableWidget( td->tdTabAddRowOption, tableRectangle );

    appGuiEnableWidget( td->tdSelectTableWidget, tableRectangle );
    appGuiEnableWidget( td->tdSelectRowWidget, tableRectangle );
    appGuiEnableWidget( td->tdSelectColumnOption, tableRectangle );

    if  ( tar->tarInspector )
	{
	tedFormatToolAdaptToSelection( tar->tarInspector, bd,
				&(td->tdSelectionScope), &(td->tdSelection),
				ed->edFileReadOnly );
	}
    }
