/************************************************************************/
/*									*/
/*  Print Postscript.							*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<appImage.h>
#   include	<appWinMeta.h>
#   include	<appMacPict.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>

#   include	"docDraw.h"
#   include	"docPs.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Printing of borders.						*/
/*									*/
/************************************************************************/

static void psPrintHorizontalBorder(	const BorderProperties *	bp,
					void *				vps,
					int				x0,
					int				x1,
					int				y )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( bp->bpStyle == DOCbsNONE )
	{ return;	}

    fprintf( ps->psFile, "%d %d %d %d rectfill\n",
					x0, y, x1- x0, bp->bpPenWideTwips );

    return;
    }

static void psPrintVerticalBorder(	const BorderProperties *	bp,
					void *				vps,
					int				x,
					int				y0,
					int				y1 )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( bp->bpStyle == DOCbsNONE )
	{ return;	}

    fprintf( ps->psFile, "%d %d %d %d rectfill\n",
					x, y0, bp->bpPenWideTwips, y1- y0 );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a series of particules with the same attributes.		*/
/*									*/
/************************************************************************/

typedef int (*PLAY_METAFILE)(	FILE *				f,
				SimpleInputStream *		sis,
				const char *			afmDirectory,
				int				xExt,
				int				yExt,
				int				twipsWide,
				int				twipsHigh );

static int psPrintMetafile(	FILE *				f,
				const InsertedObject *		io,
				const char *			afmDirectory,
				int				x0,
				int				baseline,
				int				scaleX,
				int				scaleY,
				int				xExt,
				int				yExt,
				int				twipsWide,
				int				twipsHigh )
    {
    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisMeta;
    const MemoryBuffer *	mb;

    PostScriptFont *		fontList= (PostScriptFont *)0;
    int				fontCount= 0;

    int				y0;

    PLAY_METAFILE		playMetafile;

    if  ( docPsListObjectFonts( io, afmDirectory,
					    "pf", &fontList, &fontCount ) )
	{ LDEB(1); return -1;	}

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	    mb= &(io->ioObjectData);
	    playMetafile= appMetaPlayFilePs;
	    break;

	case DOCokMACPICT:
	    mb= &(io->ioObjectData);
	    playMetafile= appMacPictPlayFilePs;
	    break;

	case DOCokOLEOBJECT:
	    mb= &(io->ioResultData);
	    playMetafile= appMetaPlayFilePs;
	    break;

	case DOCokPICTJPEGBLIP:
	case DOCokPICTPNGBLIP:
	default:
	    LDEB(io->ioKind); return 0;
	}

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); sioInClose( sisMem ); return -1;	}

    y0= baseline- ( ( scaleY/100.0 )* twipsHigh );

    fprintf( f, "100 dict begin\n" );

    appPsFontNames( f, fontList, fontCount, /*allFonts=*/ 1 );

    fprintf( f, "gsave %d %d translate\n", x0, y0 );

    if  ( scaleX != 100 || scaleY != 100 )
	{ fprintf( f, "%f %f scale\n", scaleX/100.0, scaleY/100.0 ); }

    if  ( (*playMetafile)( f, sisMeta,
			    afmDirectory, xExt, yExt, twipsWide, twipsHigh ) )
	{ LDEB(1);	}

    fprintf( f, "grestore end\n" );

    sioInClose( sisMeta );
    sioInClose( sisMem );

    if  ( fontList )
	{ free( fontList );	}

    return 0;
    }

static int psPrintIncludeEpsObject(	PrintingState *		ps,
					InsertedObject *	io,
					int			x0,
					int			baseLine )
    {
    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisHex;

    char			line[512+2];

    sisMem= sioInMemoryOpen( &(io->ioResultData) );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisHex= sioInHexOpen( sisMem );
    if  ( ! sisHex )
	{ XDEB(sisHex); sioInClose( sisMem ); return -1;	}

    appPsBeginEpsObject( ps, x0, baseLine,
				0, 0, io->ioTwipsWide, io->ioTwipsHigh,
				io->ioObjectData.mbBytes );

    while( sioInGetString( line, 512+1, sisHex ) )
	{
	int		emit= 1;
	const char *	s= line;

	while( isspace( *s ) )
	    { s++;	}

	if  ( ! *s || *s == '%' )
	    { emit= 0;	}

	if  ( emit )
	    { fputs( line, ps->psFile ); }

	while( ! strchr( line, '\n' ) )
	    {
	    if  ( ! sioInGetString( line, 512+1, sisHex ) )
		{ break;	}

	    if  ( emit )
		{ fputs( line, ps->psFile ); }
	    }
	}

    sioInClose( sisHex );
    sioInClose( sisMem );

    appPsEndEpsObject( ps );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Write links to be picked up by the distiller.			*/
/*									*/
/************************************************************************/

static void psUriLinkDestination(	PrintingState *	ps )
    {
    appPsPrintString( ps->psFile,
	(const unsigned char *)ps->psLinkFile, ps->psLinkFileSize );

    if  ( ps->psLinkMarkSize > 0 )
	{
	putc( '#', ps->psFile );
	appPsPrintString( ps->psFile,
			    (const unsigned char *)ps->psLinkMark,
			    ps->psLinkMarkSize );
	}
    }

static void psWebLinkDestination(	PrintingState *	ps )
    {
    fprintf( ps->psFile, "  /Action << /Subtype /URI /URI (" );

    psUriLinkDestination( ps );

    fprintf( ps->psFile, ") >>\n" );

    return;
    }

static void psFileLinkDestination(	PrintingState *		ps )
    {
    const unsigned char *	file;
    int				size;

    file= (const unsigned char *)ps->psLinkFile;
    size= ps->psLinkFileSize;

    if  ( size > 5 && ! strncmp( (const char *)file, "file:", 5 ) )
	{ file += 5; size -= 5; }
    else{
	while( size > 0 && isalpha( *file ) )
	    { file++; size--;	}

	if  ( size > 0 && *file == ':' )
	    { psWebLinkDestination( ps ); return; }

	file= (const unsigned char *)ps->psLinkFile;
	size= ps->psLinkFileSize;
	}

    fprintf( ps->psFile, "  /Action /Launch /File (" );

    appPsPrintString( ps->psFile, file, size );

    fprintf( ps->psFile, ")\n" );

    if  ( ps->psLinkMarkSize )
	{
	fprintf( ps->psFile, "  /URI (" );
	psUriLinkDestination( ps );
	fprintf( ps->psFile, ")\n" );
	}

    return;
    }

static void psFlushLink(	PrintingState *		ps,
				const ParticuleData *	pd,
				int			lineTop,
				int			lineHeight )
    {
    if  ( ps->psLinkParticulesDone > 0 )
	{
	fprintf( ps->psFile, "[ /Rect [ %d %d %d %d ]\n",
				    ps->psLinkRectLeft, lineTop+ lineHeight, 
				    pd->pdX0+ pd->pdVisibleWidth,
				    lineTop );

	fprintf( ps->psFile, "  /Border [ 0 0 0 ]\n" );

	if  ( ps->psLinkFileSize == 0 )
	    {
	    fprintf( ps->psFile, "  /Action /Goto\n" );
	    fprintf( ps->psFile, "  /Dest /%.*s\n",
					ps->psLinkMarkSize, ps->psLinkMark );
	    }
	else{
	    psFileLinkDestination( ps );
	    }


	fprintf( ps->psFile, "  /Subtype /Link\n" );
	fprintf( ps->psFile, "/LNK pdfmark\n" );

	ps->psLinkParticulesDone= 0;
	ps->psLinkRectLeft= pd->pdX0;
	}

    return;
    }

static int psPrintTab(	PrintingState *			ps,
			DrawingContext *		dc,
			const BufferItem *		bi,
			const TextParticule *		tp,
			ParticuleData *			pd,
			int				baseLine,
			int				lineHeight )
    {
    TabStop *	ts= bi->biParaTabStops+ pd->pdTabNumber;

    int		x0= pd->pdX0+ lineHeight/ 4;
    int		x1= pd->pdX0+ pd->pdWidth- lineHeight/2;

    switch( ts->tsLeader )
	{
	case DOCtlNONE:
	    break;

	case DOCtlDOTS:

	    x0= 60* ( ( x0+ 59 )/ 60 );
	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( ps->psInLink )
		{ appPsSetLinkColor( ps );	}
	    else{ appPsSetTextColor( ps );	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		fprintf( ps->psFile, "%d %d %d dot-tab-bold\n",
						    x1- x0, x0, baseLine );
		}
	    else{
		fprintf( ps->psFile, "%d %d %d dot-tab\n",
						    x1- x0, x0, baseLine );
		}
	    break;

	case DOCtlHYPH:
	    LDEB(ts->tsLeader);
	    break;

	case DOCtlUNDERLINE:

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( ps->psInLink )
		{ appPsSetLinkColor( ps );	}
	    else{ appPsSetTextColor( ps );	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		fprintf( ps->psFile, "%d %d %d ul-tab-bold\n",
						    x1- x0, x0, baseLine );
		}
	    else{
		fprintf( ps->psFile, "%d %d %d ul-tab\n",
						    x1- x0, x0, baseLine );
		}
	    break;

	case DOCtlTHICK:
	    LDEB(ts->tsLeader);
	    break;

	case DOCtlEQUAL:
	    LDEB(ts->tsLeader);
	    break;

	default:
	    LDEB(ts->tsLeader);
	    break;
	}

    return 0;
    }

static int psPrintParticules(	PrintingState *			ps,
				DrawingContext *		dc,
				const BufferItem *		bi,
				const DocumentFontList *	dfl,
				const char *			afmDirectory,
				const TextParticule *		tp,
				ParticuleData *			pd,
				int				count,
				int				baseLine,
				int				lineTop,
				int				lineHeight )
    {
    BufferDocument *		bd= dc->dcDocument;

    int				drawn;
    int				i;
    int				len;
    int				high;
    InsertedObject *		io;

    DocumentField *		df;
    int				fontSizeTwips;

    /*  1  */
    switch( tp->tpKind )
	{
	case DOCkindTAB:
	    if  ( pd->pdTabNumber >= 0			&&
		  pd->pdTabNumber < bi->biParaTabCount	)
		{
		if  ( psPrintTab( ps, dc, bi, tp, pd, baseLine, lineHeight ) )
		    { LDEB(1);	}
		}

	    return drawn= 1;

	case DOCkindTEXT:
	    break;

	case DOCkindFIELDSTART:
	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    if  ( df->dfKind == DOCfkBOOKMARK )
		{
		const char *	mark;
		int		markSize;

		if  ( ! docFieldGetBookmark( df, &mark, &markSize ) )
		    {
		    NupTransform *	nt= &(ps->psCurrentTransform);
		    int			top;
		    int			x= 0;
		    int			y= lineTop;

		    top= nt->ntAxy* x+ nt->ntAyy* y+ nt->ntTy;

		    fprintf( ps->psFile,
				    "[ /Dest" );
		    fprintf( ps->psFile,
				    "  /%.*s\n", markSize, mark );

		    fprintf( ps->psFile,
				    "  /View [ /XYZ null %d null ]\n", top );

		    fprintf( ps->psFile,
				    "/DEST pdfmark\n" );
		    }

		return drawn= 1;
		}

	    if  ( df->dfKind == DOCfkHYPERLINK )
		{
		if  ( ! docFieldGetHyperlink( df,
				&(ps->psLinkFile), &(ps->psLinkFileSize),
				&(ps->psLinkMark), &(ps->psLinkMarkSize) ) )
		    {
		    ps->psInLink= 1;
		    ps->psLinkParticulesDone= 0;
		    ps->psLinkRectLeft= pd->pdX0;

		    appPsSetLinkColor( ps );
		    }

		return drawn= 1;
		}

	    return drawn= 1;

	case DOCkindFIELDEND:
	    if  ( ps->psInLink )
		{
		psFlushLink( ps, pd, lineTop, lineHeight );
		ps->psInLink= 0;
		}

	    appPsSetTextColor( ps );

	    return drawn= 1;

	case DOCkindXE:
	case DOCkindTC:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	    return drawn= 1;

	case DOCkindOBJECT:
	    io= bi->biParaObjects+ tp->tpObjectNumber;

	    switch( io->ioKind )
		{
		case DOCokPICTWMETAFILE:
		case DOCokMACPICT:
		    if  ( psPrintMetafile( ps->psFile, io,
				    afmDirectory, pd->pdX0, baseLine,
				    io->ioScaleX, io->ioScaleY,
				    io->ioXExtent, io->ioYExtent,
				    io->ioTwipsWide, io->ioTwipsHigh ) )
			{ LDEB(1); break;	}

		    ps->psCurrentPhysicalFont= -1;
		    ps->psLinkParticulesDone++;
		    return 1;

		case DOCokPICTJPEGBLIP:
		case DOCokPICTPNGBLIP:
		    if  ( io->ioPrivate )
			{
			AppBitmapImage *	abi;
			BitmapDescription *	bmd;

			abi= (AppBitmapImage *)io->ioPrivate;
			bmd= &abi->abiBitmap;

			if  ( bmPsPrintBitmap( ps->psFile, 1,
				    ( 20.0* io->ioScaleX )/ 100.0,
				    ( -20.0* io->ioScaleY )/ 100.0,
				    pd->pdX0, baseLine, 0, 0,
				    bmd->bdPixelsWide, bmd->bdPixelsHigh,
				    bmd, abi->abiBuffer ) )
			    { LDEB(1); return -1; }

			ps->psLinkParticulesDone++;
			return 1;
			}
		    break;

		case DOCokOLEOBJECT:
		    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
			{
			if  ( psPrintMetafile( ps->psFile, io,
				    afmDirectory, pd->pdX0, baseLine,
				    io->ioScaleX, io->ioScaleY,
				    io->ioXExtent, io->ioYExtent,
				    io->ioTwipsWide, io->ioTwipsHigh ) )
			    { LDEB(1); break;	}

			ps->psCurrentPhysicalFont= -1;
			ps->psLinkParticulesDone++;
			return 1;
			}
		    break;

		case DOCokINCLUDEPICTURE:
		    if  ( io->ioResultKind == DOCokEPSF )
			{
			if  ( psPrintIncludeEpsObject( ps,
						io, pd->pdX0, baseLine ) )
			    { LDEB(1); break;	}

			ps->psCurrentPhysicalFont= -1;
			ps->psLinkParticulesDone++;
			return 1;
			}
		    else{ LLDEB(io->ioKind,io->ioResultKind); break;	}

		default:
		    LDEB(io->ioKind); return 0;
		}

	    appPsSetTextColor( ps );

	    high= ( io->ioScaleY* io->ioTwipsHigh )/ 100;
	    fprintf( ps->psFile, "%d %d moveto ",
				    pd->pdX0, baseLine- high );
	    fprintf( ps->psFile, "%d %d lineto ",
				    pd->pdX0+ pd->pdWidth, baseLine- high );
	    fprintf( ps->psFile, "%d %d lineto ",
				    pd->pdX0+ pd->pdWidth, baseLine );
	    fprintf( ps->psFile, "%d %d lineto ", pd->pdX0, baseLine );
	    fprintf( ps->psFile, "closepath stroke\n" );

	    ps->psLinkParticulesDone++;
	    return 1;
	default:
	    LDEB(tp->tpKind); return -1;
	}

    for ( drawn= 1; drawn < count; drawn++ )
	{
	if  ( tp[drawn].tpPhysicalFont != tp->tpPhysicalFont	||
	      tp[drawn].tpKind != DOCkindTEXT			)
	    { break;	}

	if  ( bi->biParaAlignment == DOCiaJUSTIFIED )
	    { break;	}
	}

    len= tp[drawn-1].tpStroff+ tp[drawn-1].tpStrlen- tp->tpStroff;

    if  ( len == 0 )
	{ return drawn;	}

    while( len > 0						&&
	   bi->biParaString[tp->tpStroff+ len- 1] == ' '	)
	{ len--;	}

    fontSizeTwips= 10* tp->tpTextAttribute.taFontSizeHalfPoints;

    if  ( len > 0 )
	{
	int			y;
	int			capHeight;

	if  ( tp->tpPhysicalFont != ps->psCurrentPhysicalFont )
	    {
	    appPsSetFont( ps->psFile, "f", tp->tpTextAttribute );
	    ps->psCurrentPhysicalFont= tp->tpPhysicalFont;
	    }

	if  ( ps->psInLink )
	    { appPsSetLinkColor( ps );	}
	else{ appPsSetTextColor( ps );	}

	putc( '(', ps->psFile );
	appPsPrintString( ps->psFile, bi->biParaString+ tp->tpStroff, len );
	putc( ')', ps->psFile );

	capHeight= ( fontSizeTwips* pd->pdAfi->afiCapHeight+ 500 )/ 1000;
	if  ( capHeight == 0 )
	    { capHeight= ( fontSizeTwips* pd->pdAfi->afiAscender+ 500 )/ 1000; }
	y= baseLine;

	if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT )
	    { y -= ( 4* capHeight )/ 10; }

	if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT )
	    { y += ( 4* capHeight )/ 10; }

	fprintf( ps->psFile, " %d %d mvs\n", pd->pdX0, y );
	}

    i= 0; while( i < drawn )
	{
	int	x0;
	int	x1;
	int	y0;
	int	h;

	if  ( ! tp[i].tpTextAttribute.taIsUnderlined )
	    { i++; continue;	}

	x1= x0= pd[i].pdX0;
	y0= baseLine- ( fontSizeTwips*
			pd[i].pdAfi->afiUnderlinePosition+ 500 )/ 1000;
	h= ( fontSizeTwips*
			pd[i].pdAfi->afiUnderlineThickness+ 500 )/ 1000;

	while( i < drawn && tp[i].tpTextAttribute.taIsUnderlined )
	    { x1= pd[i].pdX0+ pd[i].pdWidth; i++; }

	if  ( ps->psInLink )
	    { appPsSetLinkColor( ps );	}
	else{ appPsSetTextColor( ps );	}

	fprintf( ps->psFile, "%d %d %d %d rectfill\n", x0, y0, x1- x0, h );
	}

    ps->psLinkParticulesDone += drawn;

    return drawn;
    }

/************************************************************************/
/*									*/
/*  Print one line of output.						*/
/*									*/
/************************************************************************/

static int psPrintTextLine(	PrintingState *			ps,
				DrawingContext *		dc,
				const DocumentFontList *	dfl,
				const char *			afmDirectory,
				const BufferItem *		bi,
				const TextParticule *		tp,
				int				particuleCount,
				ParticuleData *			pd,
				int				baseLine,
				int				lineTop,
				int				lineHeight )
    {
    int				done= 0;

    while( done < particuleCount )
	{
	int		drawn;

#	if 0
	if  ( tp->tpKind == DOCkindTEXT		||
	      tp->tpKind == DOCkindTAB		)
	    {
	    if  ( tp->tpPhysicalFont != ps->psCurrentPhysicalFont )
		{
		appPsSetFont( ps->psFile, "f", tp->tpTextAttribute );
		ps->psCurrentPhysicalFont= tp->tpPhysicalFont;
		}
	    }
	else{ ps->psCurrentPhysicalFont= -1;	}
#	endif

	drawn= psPrintParticules( ps, dc, bi, dfl, afmDirectory, tp, pd,
			particuleCount- done, baseLine, lineTop, lineHeight );

	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn; pd += drawn;
	}

    if  ( done > 0 && ps->psInLink )
	{ psFlushLink( ps, pd- 1, lineTop, lineHeight ); }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout and print successive lines of a paragraph.			*/
/*									*/
/************************************************************************/

static int docPsPrintTextLine(	const BufferItem *		bi,
				int				line,
				const FormattingFrame *		ff,
				void *				vps,
				DrawingContext *		dc )
    {
    BufferDocument *		bd= dc->dcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    AppDrawingData *		add= dc->dcDrawingData;
    PrintingState *		ps= (PrintingState *)vps;
    AppPhysicalFontList *	apfl= &(add->addPhysicalFontList);

    const TextLine *		tl= bi->biParaLines+ line;
    int				part= tl->tlFirstParticule;
    const TextParticule *	tp= bi->biParaParticules+ part;

    int				accepted;
    TextLine			boxLine;
    int				baseline;

    ParticuleData *		pd;

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    boxLine= *tl;
    accepted= docPsLineBox( &boxLine, bi, part,
				    dp->dpTabIntervalTwips, &dp->dpFontList,
				    apfl, tp, pd, ff );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( ps->psInLink )
	{ ps->psLinkRectLeft= pd->pdX0;	}

    baseline= tl->tlTopPosition.lpPageYTwips+ tl->tlLineAscentTwips;

    psPrintTextLine( ps, dc, &(dp->dpFontList), apfl->apflAfmDirectory,
			bi, tp, accepted, pd,
			baseline, tl->tlTopPosition.lpPageYTwips,
			tl->tlLineHeightTwips );

    return accepted;
    }

static int docPsPrintParaTop(	const BufferItem *		bi,
				const BorderProperties *	bp,
				const FormattingFrame *		ff,
				void *				vps,
				DrawingContext *		dc )
    {
    PrintingState *	ps= (PrintingState *)vps;
    int			x0= ff->ffX0TextLinesTwips;

    if  ( ff->ffX0FirstLineTwips < x0 )
	{ x0= ff->ffX0FirstLineTwips;	}

    appPsSetTextColor( ps );

    psPrintHorizontalBorder( bi->biParaBorderAboveParagraph,
					vps, x0, ff->ffX1TextLinesTwips,
					bi->biTopPosition.lpPageYTwips+
					bi->biParaSpaceBeforeTwips );

    return 0;
    }

static int docPsPrintParaBottom( const BufferItem *		bi,
				const BorderProperties *	bp,
				const FormattingFrame *		ff,
				void *				vps,
				DrawingContext *		dc )
    {
    PrintingState *	ps= (PrintingState *)vps;
    int			x0= ff->ffX0TextLinesTwips;

    if  ( ff->ffX0FirstLineTwips < x0 )
	{ x0= ff->ffX0FirstLineTwips;	}

    appPsSetTextColor( ps );

    psPrintHorizontalBorder( bi->biParaBorderBelowParagraph,
					vps, x0, ff->ffX1TextLinesTwips,
					bi->biBelowPosition.lpPageYTwips-
					bi->biParaSpaceAfterTwips );

    return 0;
    }

static int docPsPrintCellTop(	const BorderProperties *	bp,
				int				asGrid,
				int				x0Twips,
				int				x0Pixels,
				int				x1Twips,
				int				x1Pixels,
				void *				vps,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	appPsSetTextColor( ps );

	psPrintHorizontalBorder( bp, vps,
				    x0Twips, x1Twips, lpTop->lpPageYTwips );
	}

    return 0;
    }

static int docPsPrintCellBottom( const BorderProperties *	bp,
				int				asGrid,
				int				x0Twips,
				int				x0Pixels,
				int				x1Twips,
				int				x1Pixels,
				void *				vps,
				DrawingContext *		dc,
				const LayoutPosition *		lpBottom )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	appPsSetTextColor( ps );

	psPrintHorizontalBorder( bp, vps,
				    x0Twips, x1Twips, lpBottom->lpPageYTwips );
	}

    return 0;
    }

static int docPsPrintCellLeft(	const BorderProperties *	bp,
				int				asGrid,
				void *				vps,
				DrawingContext *		dc,
				int				x0Twips,
				int				x0Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	appPsSetTextColor( ps );

	psPrintVerticalBorder( bp, vps, x0Twips,
			    lpTop->lpPageYTwips, lpBelow->lpPageYTwips );
	}

    return 0;
    }

static int docPsPrintCellRight(	const BorderProperties *	bp,
				int				asGrid,
				void *				vps,
				DrawingContext *		dc,
				int				x1Twips,
				int				x1Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    PrintingState *	ps= (PrintingState *)vps;

    if  ( ! asGrid && bp->bpStyle != DOCbsNONE )
	{
	appPsSetTextColor( ps );

	psPrintVerticalBorder( bp, vps, x1Twips,
			    lpTop->lpPageYTwips, lpBelow->lpPageYTwips );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/************************************************************************/

static int docPsFinishPage(	void *				vps,
				DrawingContext *		dc,
				int				page,
				int				asLast )
    {
    PrintingState *	ps= (PrintingState *)vps;

    appPsFinishPage( ps, page, asLast );

    return 0;
    }

static int docPsStartPage(	void *				vps,
				const DocumentGeometry *	dg,
				DrawingContext *		dc,
				int				page )
    {
    PrintingState *	ps= (PrintingState *)vps;

    ps->psCurrentPhysicalFont= -1;

    appPsStartPage( ps, page, dg );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Leave a trace in the PDF document information.			*/
/*									*/
/************************************************************************/

static void psSaveInfo(		const char *		tag,
				const unsigned char *	info,
				FILE *			f )
    {
    if  ( ! info )
	{ return;	}

    fprintf( f, "  %s (", tag );
    appPsPrintString( f, info, strlen( (const char *)info ) );
    fprintf( f, ")\n" );
    }

static void psSaveDate(		const char *		tag,
				const struct tm *	tm,
				FILE *			f )
    {
    char	scratch[40+1];

    if  ( tm->tm_mday == 0 )
	{ return;	}

    if  ( strftime( scratch, sizeof(scratch)- 1, "D:%Y%m%d%H%M", tm ) < 1 )
	{ LDEB(1); return;	}

    psSaveInfo( tag, (const unsigned char *)scratch, f );

    return;
    }

/************************************************************************/
/*									*/
/*  Print a document.							*/
/*									*/
/************************************************************************/

int docPsPrintDocument(	FILE *				f,
			const char *			title,
			const char *			applicationName,
			const char *			applicationReference,
			AppDrawingData *		add,
			BufferDocument *		bd,
			int				nup,
			int				horizontal,
			int				firstPage,
			int				lastPage,
			const DocumentGeometry *	dgPrinter,
			LAYOUT_EXTERNAL			layoutExternal,
			DOC_CLOSE_OBJECT		closeObject )
    {
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentGeometry *		dg= &(dp->dpGeometry);
    BufferItem *		docBi= &(bd->bdItem);

    PostScriptFont *		fontList= (PostScriptFont *)0;
    int				fontCount= 0;
    int				i;

    DrawingContext		dc;
    PrintingState		ps;

    docInitDrawingContext( &dc );

    dc.dcDrawTextLine= docPsPrintTextLine;
    dc.dcDrawParaTop= docPsPrintParaTop;
    dc.dcDrawParaBottom= docPsPrintParaBottom;
    dc.dcDrawCellTop= docPsPrintCellTop;
    dc.dcDrawCellBottom= docPsPrintCellBottom;
    dc.dcDrawCellLeft= docPsPrintCellLeft;
    dc.dcDrawCellRight= docPsPrintCellRight;
    dc.dcFinishPage= docPsFinishPage;
    dc.dcStartPage= docPsStartPage;
    dc.dcLayoutExternal= layoutExternal;
    dc.dcCloseObject= closeObject;

    dc.dcDrawingData= add;
    dc.dcDocument= bd;

    dc.dcFirstPage= firstPage;
    dc.dcLastPage= lastPage;
    dc.dcDrawHeadersFooters= 1;

    appPsInitPrintingState( &ps );
    ps.psPrinterGeometry= *dgPrinter;
    ps.psFile= f;

    if  ( dp->dpTitle && dp->dpTitle[0] )
	{ title= (char *)dp->dpTitle;	}

    if  ( appPsSetNup( &ps, dg, nup, horizontal ) )
	{ LDEB(nup); return -1;	}

    if  ( docPsPrintGetItemFonts( docBi,
			(&dp->dpFontList), &(add->addPhysicalFontList),
			&fontList, &fontCount ) )
	{ LDEB(1); return -1;	}

    fprintf( f, "%%!PS-Adobe-2.0\n" );

    if  ( title )
	{ fprintf( f, "%%%%Title: %s\n", title ); }

    appPsBoundingBoxComment( &ps, dg, "BoundingBox", "Orientation" );

    fprintf( f, "%%%%Creator: %s: %s\n",
				applicationName, applicationReference );
    fprintf( f, "%%%%Pages: (atend)\n" );
    fprintf( f, "%%%%PageOrder: Ascend\n" );
    if  ( fontCount > 0 )
	{
	fprintf( f, "%%%%DocumentFonts: %s\n",
					    fontList[0].psfAfi->afiFontName );
	for ( i= 1; i < fontCount; i++ )
	    {
	    int		done;

	    for ( done= 0; done < i; done++ )
		{
		if  ( ! strcmp( fontList[done].psfAfi->afiFontName,
					fontList[i].psfAfi->afiFontName ) )
		    { break;	}
		}

	    if  ( done >= i )
		{
		fprintf( f, "%%%%+ %s\n", fontList[i].psfAfi->afiFontName );
		}
	    }
	}
    fprintf( f, "%%%%EndComments\n" );

    fprintf( f, "%%%%BeginProlog\n" );

    fprintf( f, "\n/mvs { moveto show } bind def\n" );

    fprintf( f, "\n/dot-tab\n" );
    fprintf( f, "  {\n" );
    fprintf( f, "  gsave\n" );
    fprintf( f, "  10 setlinewidth [ 1 59 ] 0 setdash 1 setlinecap\n" );
    fprintf( f, "  newpath moveto 0 rlineto stroke\n" );
    fprintf( f, "  grestore\n" );
    fprintf( f, "  } bind def\n" );

    fprintf( f, "\n/dot-tab-bold\n" );
    fprintf( f, "  {\n" );
    fprintf( f, "  gsave\n" );
    fprintf( f, "  16 setlinewidth [ 1 59 ] 0 setdash 1 setlinecap\n" );
    fprintf( f, "  newpath moveto 0 rlineto stroke\n" );
    fprintf( f, "  grestore\n" );
    fprintf( f, "  } bind def\n" );

    fprintf( f, "\n/ul-tab\n" );
    fprintf( f, "  {\n" );
    fprintf( f, "  gsave\n" );
    fprintf( f, "  10 setlinewidth\n" );
    fprintf( f, "  newpath moveto 0 rlineto stroke\n" );
    fprintf( f, "  grestore\n" );
    fprintf( f, "  } bind def\n" );

    fprintf( f, "\n/ul-tab-bold\n" );
    fprintf( f, "  {\n" );
    fprintf( f, "  gsave\n" );
    fprintf( f, "  16 setlinewidth\n" );
    fprintf( f, "  newpath moveto 0 rlineto stroke\n" );
    fprintf( f, "  grestore\n" );
    fprintf( f, "  } bind def\n" );

    fprintf( f, "\n%%pdfmark emulation\n" );
    fprintf( f, "/pdfmark where\n" );
    fprintf( f, "    { pop }\n" );
    fprintf( f, "    { userdict /pdfmark /cleartomark load put }\n" );
    fprintf( f, "ifelse\n" );

    appPsSetRectfillEmulation( f );
    appPsSetSelectfontEmulation( f );

    appPsDefineEpsProcs( f );

    appPsFontNames( f, fontList, fontCount, /*allFonts=*/ 1 );

    appMetaDefineProcsetPs( f );

    fprintf( f, "%%%%EndProlog\n" );
    fprintf( f, "%%%%BeginSetup\n" );

    if  ( docHasDocumentInfo( dp ) )
	{
	unsigned char *			scratch;

	scratch= malloc( strlen( applicationName )+ 2+
					strlen( applicationReference )+ 2 );
	if  ( ! scratch )
	    { XDEB(scratch); return -1;	}
	sprintf( (char *)scratch,
			"%s: %s", applicationName, applicationReference );

	fprintf( f, "[\n" );

	psSaveInfo( "/Title",		dp->dpTitle, f );
	psSaveInfo( "/Author",		dp->dpAuthor, f );
	psSaveInfo( "/Subject",		dp->dpSubject, f );
	psSaveInfo( "/Keywords",	dp->dpKeywords, f );
	psSaveInfo( "/Creator",		scratch, f );
	psSaveInfo( "/Producer",	scratch, f );

	psSaveDate( "/ModDate",		&(dp->dpRevtim), f );
	psSaveDate( "/CreationDate",	&(dp->dpCreatim), f );

	fprintf( f, "/DOCINFO pdfmark\n\n" );

	free( scratch );
	}

    fprintf( f, "%%%%EndSetup\n" );

    if  ( firstPage < 0 )
	{ firstPage= 0;	}

    for ( i= 0; i < docBi->biChildCount; i++ )
	{
	if  ( docBi->biChildren[i]->biBelowPosition.lpPage >= firstPage )
	    { break;	}
	}

    if  ( i >= docBi->biChildCount )
	{ LDEB(i); return -1; }

    appPsStartPage( &ps, firstPage, dg );
    docDrawPageHeader( docBi->biChildren[i],
						(void *)&ps, &dc, firstPage );

    docDrawItem( docBi, (void *)&ps, &dc );

    if  ( lastPage < 0 )
	{ lastPage= docBi->biBelowPosition.lpPage;	}

    for ( i= docBi->biChildCount- 1; i >= 0; i-- )
	{
	if  ( docBi->biChildren[i]->biTopPosition.lpPage <= lastPage )
	    { break;	}
	}

    if  ( i < 0 )
	{ LDEB(i); return -1;	}

    docDrawPageFooter( docBi->biChildren[i],
						(void *)&ps, &dc, lastPage );
    appPsFinishPage( &ps, lastPage, /*asLast*/ 1 );

    appPsCleanPrintingState( &ps );

    if  ( fontList )
	{ free( fontList );	}

    return 0;
    }
