/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<appImage.h>
#   include	"docPs.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Initialisation for paragraph layout.				*/
/*									*/
/************************************************************************/

static int docPsOpenParaFonts(	BufferItem *			paraBi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    int				part;
    TextParticule *		tp;

    const DocumentFontList *	dfl= &(dp->dpFontList);

    tp= paraBi->biParaParticules;
    for ( part= 0; part < paraBi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpPhysicalFont < 0 )
	    {
	    tp->tpPhysicalFont= appOpenDocumentFont( add, dfl,
							tp->tpTextAttribute );

	    if  ( tp->tpPhysicalFont < 0 )
		{
		APP_DEB(appDebug( "\"%.*s\"\n",
		    (int)tp->tpStrlen, paraBi->biParaString+ tp->tpStroff ));
		LLDEB(part,tp->tpPhysicalFont);
		return -1;
		}
	    }
	}

    return 0;
    }

static int docPsLayoutParaInit(	BufferItem *			paraBi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    if  ( docPsOpenParaFonts( paraBi, add, dp ) )
	{ LDEB(1); return -1;	}

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), paraBi ) )
	{ LDEB(1); return -1;	}

    paraBi->biParaLineCount= 0;

    return 0;
    }

static int docPsPlaceParaInit(	BufferItem *			paraBi,
				AppDrawingData *		add,
				const DocumentProperties *	dp )
    {
    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/*  1)  Cope with the output of jade: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static void docPsAboveLine(	LayoutPosition *		lp,
				int				fromLinePos,
				const BufferItem *		paraBi,
				int				part,
				const DocumentGeometry *	dg,
				AppDrawingData *		add )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;

    int			spaceAboveLineTwips= 0;
    int			spaceAboveLinePixels= 0;

    /*  1  */
    if  ( part == 0 && ! fromLinePos && paraBi->biInHeaderFooter == DOCinBODY )
	{
	int		isFirst= 0;

	if  ( paraBi->biParaInTable )
	    {
	    if  ( paraBi->biNumberInParent == 0 )
		{ isFirst= 1;	}
	    }
	else{
	    if  ( lp->lpPageYTwips == dg->dgTopMarginTwips )
		{ isFirst= 1;	}
	    }

	spaceAboveLineTwips= paraBi->biParaSpaceAboveLinesTwips;

	if  ( ! isFirst )
	    { spaceAboveLineTwips += paraBi->biParaSpaceBeforeTwips;	}

	spaceAboveLinePixels= TWIPStoPIXELS( xfac, spaceAboveLineTwips );
	}

    lp->lpPageYTwips += spaceAboveLineTwips;
    /* still at top */

    return;
    }

static int docPsBelowLine(	const LayoutPosition *		lpTop,
				LayoutPosition *		lp,
				const TextLine *		tl,
				const BufferItem *		paraBi,
				int				past,
				const FormattingFrame *		ff,
				AppDrawingData *		add )
    {
    int			spaceBelowLineTwips= 0;
    int			lineBottom;
    int			lineHeight;

    int			isLast= 0;

    if  ( past == paraBi->biParaParticuleCount )
	{
	if  ( paraBi->biParaInTable )
	    {
	    if  ( paraBi->biNumberInParent ==
					paraBi->biParent->biChildCount- 1 )
		{ isLast= 1;	}
	    }

	if  ( ! isLast )
	    {
	    spaceBelowLineTwips= paraBi->biParaSpaceBelowLinesTwips;
	    spaceBelowLineTwips += paraBi->biParaSpaceAfterTwips;
	    }
	}

    lineBottom= lp->lpPageYTwips+ tl->tlLineHeightTwips+ spaceBelowLineTwips;
    lineHeight= lineBottom- lpTop->lpPageYTwips;

    if  ( paraBi->biInHeaderFooter == DOCinBODY			&&
	  lineBottom > ff->ffY1					&&
	  lineHeight < ff->ffPageHigh				&&
	  ( ff->ffStripHigh < 0			||
	    lineHeight < ff->ffStripHigh	)		)
	{
	lp->lpPageYTwips= ff->ffY1+ 1;
	lp->lpAtTopOfColumn= 0;
	return 1;
	}

    lp->lpPageYTwips += tl->tlLineSpacingTwips;
    lp->lpAtTopOfColumn= 0;

    if  ( past == paraBi->biParaParticuleCount /* && ! isLast */ )
	{
	spaceBelowLineTwips= paraBi->biParaSpaceBelowLinesTwips;

	lp->lpPageYTwips += spaceBelowLineTwips;
	lp->lpAtTopOfColumn= 0;
	}

    return 0;
    }

static int docPsPlace_Line(	TextLine *			resTl,
				BufferItem *			bi,
				int				part,
				TextParticule *			tp,
				ParticuleData *			pd,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ff,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				const ScreenLayout *		sl )
    {
    int				accepted;
    int				res;
    int				i;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lp;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    const int			fromLinePos= 0;

    lp= *lpTop;
    sp= &(bi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    workTl= *resTl;
    /*
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;
    */

    /*  1  */
    docPsAboveLine( &lp, fromLinePos, bi, part, dg, add );

    tl->tlTopPosition= lp;
    accepted= tl->tlParticuleCount;

    if  ( accepted < 1 )
	{ LDEB(accepted); docListItem(0,bi); return -1; }

    res= docPsBelowLine( lpTop, &lp, tl, bi, part+ accepted, ff, add );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ *lpBottom= lp; return 0; }

    if  ( sl && sl->slLayoutLine					&&
	  (*sl->slLayoutLine)( tl, bi, part, accepted, add,
						    ff, tp, pd ) < 0	)
	{ LDEB(accepted); return -1;	}

    for ( i= 0; i < accepted; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    df->dfPage= lp.lpPage;
	    }
	}

    *lpBottom= lp;
    *resTl= *tl;

    return accepted;
    }

static int docPsPlaceLines(	const ParagraphLayoutContext *	plc,
				BufferItem *			bi,
				int				part,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ff,
				int *				pLinesDone,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom )
    {
    int				done= 0;
    int				line= *pLinesDone;

    TextParticule *		tp= bi->biParaParticules+ part;
    TextLine *			tl= bi->biParaLines+ line;

    ParticuleData *		pd= (ParticuleData *)0;

    LayoutPosition		lp;

    lp= *lpTop;

    while( part < bi->biParaParticuleCount )
	{
	int			accepted;
	TextLine		boxLine;

	DocumentRectangle *	drChanged= plc->plcChangedRectanglePixels;

	if  ( line >= bi->biParaLineCount )
	    { done= bi->biParaParticuleCount- part; break;	}

	boxLine= *tl;

	accepted= docPsPlace_Line( &boxLine, bi, part, tp, pd, add, bd,
			ff, &lp, &lp, &(plc->plcScreenLayout) );
	
	if  ( drChanged	)
	    {
	    int		tlYPixels;
	    int		boxYPixels;

	    tlYPixels= TL_TOP_PIXELS( add, tl );
	    boxYPixels= TL_TOP_PIXELS( add, &boxLine );

	    if  ( tlYPixels != boxYPixels )
		{
		int		oldY1Pixels;
		int		newY1Pixels;

		if  ( drChanged->drY0 > boxYPixels )
		    { drChanged->drY0=  boxYPixels;	}
		if  ( drChanged->drY0 > tlYPixels )
		    { drChanged->drY0=  tlYPixels;	}

		oldY1Pixels= TL_BELOW_PIXELS( add, tl );
		newY1Pixels= LP_YPIXELS( add, &lp );

		if  ( drChanged->drY1 < oldY1Pixels )
		    { drChanged->drY1=  oldY1Pixels; }

		if  ( drChanged->drY1 < newY1Pixels )
		    { drChanged->drY1=  newY1Pixels; }
		}
	    }

	if  ( accepted == 0 )
	    { break;	}

	*tl= boxLine;

	if  ( accepted < 0 )
	    { LLDEB(line,accepted); return -1; }

	part += accepted; tp += accepted; done += accepted; line++; tl++;

	if  ( boxLine.tlHasPageBreak )
	    {
	    lp.lpPageYTwips= ff->ffY1+ 1;
	    lp.lpAtTopOfColumn= 0;
	    break;
	    }
	}

    *pLinesDone= line;
    *lpBottom= lp;

    return done;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/*  1)  Cope with the output of sgmls: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static int docPsLayout_Line(	TextLine *			resTl,
				int				fromLinePos,
				BufferItem *			bi,
				int				part,
				TextParticule *			tp,
				ParticuleData *			pd,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ff,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom,
				const ScreenLayout *		sl )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);
    int				accepted;
    int				res;
    int				i;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lp;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    lp= *lpTop;
    sp= &(bi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    docInitTextLine( tl );
    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;

    /*  1  */
    docPsAboveLine( &lp, fromLinePos, bi, part, dg, add );

    tl->tlTopPosition= lp;
    accepted= docPsLineBox( tl, bi, part,
			    dp->dpTabIntervalTwips, &dp->dpFontList,
			    &(add->addPhysicalFontList), tp, pd, ff );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    res= docPsBelowLine( lpTop, &lp, tl, bi, part+ accepted, ff, add );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ *lpBottom= lp; return 0; }

    tl->tlStroff= tp->tpStroff;
    tl->tlFirstParticule= part;
    tl->tlStrlen=
	    tp[accepted-1].tpStroff+ tp[accepted-1].tpStrlen- tp->tpStroff;
    tl->tlParticuleCount= accepted;

    if  ( sl && sl->slLayoutLine					&&
	  (*sl->slLayoutLine)( tl, bi, part, accepted, add,
						    ff, tp, pd ) < 0	)
	{ LDEB(accepted); return -1;	}

    for ( i= 0; i < accepted; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    DocumentField *	df;

	    df= bd->bdFieldList.dflFields+ tp->tpObjectNumber;

	    df->dfPage= lp.lpPage;
	    }
	}

    *lpBottom= lp;
    *resTl= *tl;

    return accepted;
    }

static int docPsLayoutLines(	const ParagraphLayoutContext *	plc,
				BufferItem *			bi,
				int				part,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				const FormattingFrame *		ff,
				int *				pLinesDone,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom )
    {
    int				done= 0;
    int				line= *pLinesDone;

    TextParticule *		tp= bi->biParaParticules+ part;

    ParticuleData *		pd;

    LayoutPosition		lp;

    lp= *lpTop;

    if  ( docPsClaimParticuleData( bi, &pd ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    while( part < bi->biParaParticuleCount )
	{
	int		accepted;
	TextLine	boxLine;
	TextLine *	tl;
	const int	fromLinePos= 0;

	accepted= docPsLayout_Line( &boxLine, fromLinePos, bi,
			part, tp, pd, add, bd,
			ff, &lp, &lp, &(plc->plcScreenLayout) );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	if  ( accepted == 0 )
	    { break;	}

	if  ( line >= bi->biParaLineCount )
	    {
	    tl= docInsertTextLine( bi, -1 );
	    if  ( ! tl )
		{ XDEB(tl); return -1;		}
	    }
	else{
	    tl= bi->biParaLines+ line;
	    }

	*tl= boxLine;

	part += accepted; tp += accepted; done += accepted; line++;

	if  ( boxLine.tlHasPageBreak )
	    {
	    lp.lpPageYTwips= ff->ffY1+ 1;
	    lp.lpAtTopOfColumn= 0;
	    break;
	    }
	}

    *pLinesDone= line;
    *lpBottom= lp;

    return done;
    }

static int docPsLayoutParaItem(	BufferItem *			bi,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				LayoutJob *			lj,
				const ParagraphLayoutContext *	plc )
    {
    ParagraphLayoutJob		plj;

    docPsBeginParagraphLayoutProgress( &plj,
			bi->biNumberInParent, 0, 0, bi->biNumberInParent+ 1,
			&(lj->ljPosition) );

    if  ( docPsLayoutParagraphs( plc, bi->biParent, add, bd, &plj ) )
	{ LDEB(1); return -1;	}

    lj->ljPosition= plj.pljPos.plpPos;

    if  ( bi->biParaLineCount < 1 )
	{ LDEB(bi->biParaLineCount); docListItem(0,bi); return -1; }

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the position of the top of a section, depending on the kind	*/
/*  of break								*/
/*									*/
/*  1)  Note that even and odd are swapped as from the users point of	*/
/*	view page numbers count from one.				*/
/*  2)  As this routine is only called for the body of the document,	*/
/*	page wraps can be performed unconditionally.			*/
/*									*/
/************************************************************************/

static void docPsPlaceSectTop(		BufferItem *			sectBi,
					const DocumentProperties *	dp,
					AppDrawingData *		add,
					LayoutJob *			lj )
    {
    if  ( sectBi->biSectBreakKind != DOCsbkNONE		||
	  ( sectBi->biParent			&&
	    sectBi->biNumberInParent == 0	)	)
	{
	if  ( sectBi->biNumberInParent == 0 )
	    {
	    docPsStartPageLayout( 0, sectBi, dp, add, &(lj->ljPosition) );
	    }
	else{ docPsToNextPage( sectBi, dp, add, &(lj->ljPosition) );	}

	/*  1  */
	if  ( ( sectBi->biSectBreakKind == DOCsbkEVEN	&&
		! ( lj->ljPosition.lpPage % 2 )		)	||
	      ( sectBi->biSectBreakKind == DOCsbkODD	&&
		( lj->ljPosition.lpPage % 2 )		)	)
	    { docPsToNextPage( sectBi, dp, add, &(lj->ljPosition) );	}
	}

    sectBi->biTopPosition= lj->ljPosition;

    if  ( sectBi->biNumberInParent == 0	&&
	  sectBi->biParent			)
	{ sectBi->biParent->biTopPosition= lj->ljPosition; }

    return;
    }

/************************************************************************/
/*									*/
/*  Adjust the bottom of an item to changes inside.			*/
/*									*/
/************************************************************************/

static void docPsPlaceAdjustBottom(	int *			pChanged,
					BufferItem *		bi,
					const LayoutPosition *	lp,
					AppDrawingData *	add,
					DocumentRectangle *	drChanged )
    {
    int			changed= 0;

    int			oldY1Pixels;
    int			newY1Pixels;

    if  ( bi->biBelowPosition.lpPage != lp->lpPage		||
	  bi->biBelowPosition.lpPageYTwips != lp->lpPageYTwips	)
	{ bi->biBelowPosition= *lp; changed= 1; }

    oldY1Pixels= BI_BELOW_PIXELS( add, bi );
    newY1Pixels= LP_YPIXELS( add, lp );

    if  ( oldY1Pixels < newY1Pixels )
	{
	if  ( drChanged					&&
	      drChanged->drY1 < newY1Pixels -1	)
	    { drChanged->drY1=  newY1Pixels -1;	}
	}

    if  ( oldY1Pixels > newY1Pixels )
	{
	if  ( drChanged							&&
	      drChanged->drY1 < oldY1Pixels- 1	)
	    { drChanged->drY1=  oldY1Pixels- 1;	}
	}

    if  ( changed )
	{ *pChanged= changed;	}

    return;
    }

static void docPsLayoutAdjustBottom(	int *			pChanged,
					BufferItem *		bi,
					const LayoutPosition *	lp,
					AppDrawingData *	add,
					DocumentRectangle *	drChanged )
    {
    docPsPlaceAdjustBottom( pChanged, bi, lp, add, drChanged );

    *pChanged= 1;
    }

/************************************************************************/
/*									*/
/*  Place successive items, after the predecessor.			*/
/*									*/
/************************************************************************/

static int docPsPlaceChildren(	BufferItem *			biParent,
				int				from,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				LayoutJob *			lj,
				const ParagraphLayoutContext *	plc )
    {
    int				i;
    int				changed;
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    const DocumentProperties *	dp= &(bd->bdProperties);

    for ( i= from; i < biParent->biChildCount; i++ )
	{
	BufferItem *	child= biParent->biChildren[i];

	child->biTopPosition= lj->ljPosition;

	switch( child->biLevel )
	    {
	    case DOClevSECT:
		docPsPlaceSectTop( child, dp, add, lj );

		if  ( docPsPlaceChildren( child, 0, add, bd, lj, plc ) )
		    { LDEB(1); return -1;	}
		break;

	    case DOClevDOC:
	    rowAsGroup:
		if  ( docPsPlaceChildren( child, 0, add, bd, lj, plc ) )
		    { LDEB(1); return -1;	}
		break;

	    case DOClevCELL:
		{
		ParagraphLayoutJob	plj;

		docPsBeginParagraphLayoutProgress( &plj,
						0, 0, 0, child->biChildCount,
						&(lj->ljPosition) );

		if  ( docPsLayoutParagraphs( plc, child, add, bd, &plj ) )
		    { LDEB(1); return -1;	}

		lj->ljPosition= plj.pljPos.plpPos;
		}
		break;

	    case DOClevROW:
		if  ( ! child->biRowHasTableParagraphs )
		    { goto rowAsGroup;	}

		if  ( docPsLayoutRowItem( child, add, bd, lj, plc ) )
		    { LDEB(1); return -1;	}

		break;

	    case DOClevPARA:
		if  ( child->biParaLineCount == 0 )
		    { break;	}

		if  ( docPsLayoutParaItem( child, add, bd, lj, plc ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(biParent->biLevel); return -1;
	    }

	docPsPlaceAdjustBottom( &changed, child,
					&(lj->ljPosition), add, drChanged );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Adjust the geometry of a parent item to changes in a child.		*/
/*									*/
/************************************************************************/

static int docPsFixupParentGeometry(	BufferItem *		bi,
					BufferItem *		biParent )
    {
    LayoutPosition *	lpBi= &(bi->biTopPosition);
    LayoutPosition *	lpPa= &(biParent->biTopPosition);

    if  ( lpPa->lpPage != lpBi->lpPage			||
	  lpPa->lpPageYTwips != lpBi->lpPageYTwips	)
	{
	SDEB(docLevelStr(biParent->biLevel));
	SDEB(docLevelStr(bi->biLevel));
	LLDEB(lpPa->lpPage,lpBi->lpPage);
	LLDEB(lpPa->lpPageYTwips,lpBi->lpPageYTwips);

	biParent->biTopPosition= *lpBi;

	return -1;
	}

    return 0;
    }

static int docPsAdjustParentGeometry(	BufferItem *		bi,
					AppDrawingData *	add,
					const BufferDocument *	bd,
					LayoutJob *		lj )
    {
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    BufferItem *		parentBi;
    int				from;

    ParagraphLayoutContext	plc;

    plc.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;

    plc.plcScreenLayout= lj->ljPlaceScreen;
    plc.plcStartParagraph= docPsPlaceParaInit;
    plc.plcLayoutLines= docPsPlaceLines;
    plc.plcAdjustBottom= docPsPlaceAdjustBottom;

    parentBi= bi->biParent;

    if  ( bi->biLevel == DOClevPARA )
	{
	ParagraphLayoutJob	plj;

	docPsInitParagraphLayoutJob( &plj );

	docPsBeginParagraphLayoutProgress( &plj,
				    bi->biNumberInParent+ 1, 0, 0,
				    parentBi->biChildCount,
				    &(lj->ljPosition) );

	docPsAdvanceParagraphLayout( &(plj.pljPos0.plpProgress),
				    &(plj.pljPos.plpProgress),
				    lj->ljPosition.lpPage, parentBi );

	if  ( plj.pljPos0.plpProgress.clpPara >= bi->biNumberInParent+ 1 )
	    {
	    if  ( plj.pljPos0.plpProgress.clpPara > bi->biNumberInParent+ 1 )
		{ LLDEB(plj.pljPos0.plpProgress.clpPara,bi->biNumberInParent); }

	    from= bi->biNumberInParent+ 1;
	    }
	else{
	    from= plj.pljPos0.plpProgress.clpPara;

	    if  ( from == 0 )
		{
		lj->ljPosition= parentBi->biChildren[from]->biTopPosition;
		}
	    else{
		lj->ljPosition=
			    parentBi->biChildren[from- 1]->biBelowPosition;
		}
	    }
	}
    else{ from= bi->biNumberInParent+ 1;	}

    while( parentBi )
	{
	int		changed= 0;

	switch( parentBi->biLevel )
	    {
	    case DOClevDOC:
	    case DOClevSECT:
	    parentRowAsGroup:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { LDEB(from); from= 0; }

		if  ( from <= parentBi->biChildCount- 1 )
		    { docPsPlaceChildren( parentBi, from, add, bd, lj, &plc ); }
		else{ lj->ljPosition= bi->biBelowPosition;		}

		break;

	    case DOClevCELL:
		if  ( bi->biNumberInParent == 0			&&
		      docPsFixupParentGeometry( bi, parentBi )	)
		    { LDEB(from); from= 0; }

		if  ( from <= parentBi->biChildCount- 1 )
		    {
		    ParagraphLayoutJob	plj;

		    docPsInitParagraphLayoutJob( &plj );

		    docPsBeginParagraphLayoutProgress( &plj,
					    from, 0, 0, parentBi->biChildCount,
					    &(lj->ljPosition) );

		    if  ( docPsLayoutParagraphs( &plc,
						parentBi, add, bd, &plj ) )
			{ LDEB(1); return -1;	}

		    lj->ljPosition= plj.pljPos.plpPos;
		    }
		else{ lj->ljPosition= bi->biBelowPosition;		}

		break;

	    case DOClevROW:
		if  ( ! parentBi->biRowHasTableParagraphs )
		    { goto parentRowAsGroup;	}

		lj->ljPosition= parentBi->biTopPosition;

		if  ( docPsLayoutRowItem( parentBi, add, bd, lj, &plc ) )
		    { LDEB(1); return -1;	}

		break;

	    default:
		LDEB(parentBi->biLevel); return -1;
	    }

	docPsPlaceAdjustBottom( &changed, parentBi,
					&(lj->ljPosition), add, drChanged );

	if  ( ! changed )
	    { break; }

	bi= parentBi; parentBi= bi->biParent;
	from= bi->biNumberInParent+ 1;
	}

    if  ( bi->biInHeaderFooter == DOCinBODY && ! parentBi )
	{
	int	extendRectangle;

	int	y1;

	y1= lj->ljPosition.lpPage* add->addPageStepPixels+
			    add->addPaperRect.drY1- add->addPaperRect.drY0;

	extendRectangle= add->addBackRect.drY1 != y1;

	if  ( drChanged					&&
	      extendRectangle				&&
	      drChanged->drY1 < add->addBackRect.drY1 )
	    { drChanged->drY1=  add->addBackRect.drY1; }

	add->addBackRect.drY1= y1;
	add->addDocRect.drY1= y1- add->addBottomMarginPixels;

	if  ( drChanged					&&
	      extendRectangle				&&
	      drChanged->drY1 < add->addBackRect.drY1 )
	    { drChanged->drY1=  add->addBackRect.drY1; }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a section.						*/
/*									*/
/************************************************************************/

static int docPsLayoutHdFt(		HeaderFooter *			hf,
					const SectionProperties *	sp,
					AppDrawingData *		add,
					const BufferDocument *		bd,
					LayoutJob *			lj )
    {
    LayoutJob			headerLj;
    LayoutPosition		headerPosition;

    const DocumentGeometry *	dgSect= &(sp->spDocumentGeometry);

    int				high;
    int				y1;

    hf->hfPageUsedFor= -1;

    if  ( ! hf->hfItem )
	{ return 0;	}

    headerPosition.lpPage= 0;
    headerPosition.lpPageYTwips= dgSect->dgHeaderPositionTwips;
    headerPosition.lpAtTopOfColumn= 1; /* not really */

    headerLj= *lj;
    headerLj.ljChangedRectanglePixels= (DocumentRectangle *)0;
    headerLj.ljPosition= headerPosition;

    if  ( docPsLayoutItemAndParents( hf->hfItem, add, bd, &headerLj, 0 ) )
	{ LDEB(1); return -1;	}

    y1= hf->hfItem->biBelowPosition.lpPageYTwips;

    switch( hf->hfItem->biInHeaderFooter )
	{
	case DOCinSECT_HEADER:
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	    hf->hfY0UsedTwips= dgSect->dgHeaderPositionTwips;
	    hf->hfY1UsedTwips= y1;
	    hf->hfY0ReservedTwips= hf->hfY0UsedTwips;
	    if  ( y1 > dgSect->dgTopMarginTwips )
		{ hf->hfY1ReservedTwips= y1;				}
	    else{ hf->hfY1ReservedTwips= dgSect->dgTopMarginTwips;	}
	    break;

	case DOCinSECT_FOOTER:
	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    high= y1- dgSect->dgHeaderPositionTwips;
	    hf->hfY1UsedTwips= dgSect->dgPageHighTwips-
					    dgSect->dgFooterPositionTwips;
	    hf->hfY0UsedTwips= hf->hfY1UsedTwips- high;

	    hf->hfY1ReservedTwips= hf->hfY1UsedTwips;

	    if  ( hf->hfY0UsedTwips >
			dgSect->dgPageHighTwips- dgSect->dgBottomMarginTwips )
		{
		hf->hfY0ReservedTwips= dgSect->dgPageHighTwips-
						dgSect->dgBottomMarginTwips;
		}
	    else{
		hf->hfY0ReservedTwips= hf->hfY0UsedTwips;
		}
	    break;

	default:
	    LDEB(hf->hfItem->biInHeaderFooter); return -1;
	}

    return 0;
    }

static int docPsLayoutSectionItem(	BufferItem *		sectBi,
					AppDrawingData *	add,
					const BufferDocument *	bd,
					LayoutJob *		lj )
    {
    const SectionProperties *	sp= &(sectBi->biSectProperties);
    const DocumentGeometry *	dgSect= &(sp->spDocumentGeometry);
    const DocumentProperties *	dp= &(bd->bdProperties);

    int				i;

    /**/

    sectBi->biSectHeader.hfY0ReservedTwips=
    sectBi->biSectFirstPageHeader.hfY0ReservedTwips=
    sectBi->biSectLeftPageHeader.hfY0ReservedTwips=
    sectBi->biSectRightPageHeader.hfY0ReservedTwips=
					dgSect->dgHeaderPositionTwips;

    sectBi->biSectHeader.hfY1ReservedTwips=
    sectBi->biSectFirstPageHeader.hfY1ReservedTwips=
    sectBi->biSectLeftPageHeader.hfY1ReservedTwips=
    sectBi->biSectRightPageHeader.hfY1ReservedTwips=
					dgSect->dgTopMarginTwips;

    if  ( docPsLayoutHdFt( &(sectBi->biSectHeader), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHdFt( &(sectBi->biSectFirstPageHeader), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHdFt( &(sectBi->biSectLeftPageHeader), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHdFt( &(sectBi->biSectRightPageHeader), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    /**/

    sectBi->biSectFooter.hfY0ReservedTwips=
    sectBi->biSectFirstPageFooter.hfY0ReservedTwips=
    sectBi->biSectLeftPageFooter.hfY0ReservedTwips=
    sectBi->biSectRightPageFooter.hfY0ReservedTwips=
					dgSect->dgBottomMarginTwips;

    sectBi->biSectFooter.hfY1ReservedTwips=
    sectBi->biSectFirstPageFooter.hfY1ReservedTwips=
    sectBi->biSectLeftPageFooter.hfY1ReservedTwips=
    sectBi->biSectRightPageFooter.hfY1ReservedTwips=
		dgSect->dgPageHighTwips- dgSect->dgFooterPositionTwips;

    if  ( docPsLayoutHdFt( &(sectBi->biSectFooter), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHdFt( &(sectBi->biSectFirstPageFooter), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHdFt( &(sectBi->biSectLeftPageFooter), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docPsLayoutHdFt( &(sectBi->biSectRightPageFooter), sp, add, bd, lj ) )
	{ LDEB(1); return -1;	}

    /**/

    docPsPlaceSectTop( sectBi, dp, add, lj );

    for ( i= 0; i < sectBi->biChildCount; i++ )
	{
	const int	andparents= 0;

	if  ( docPsLayoutItemAndParents( sectBi->biChildren[i],
						add, bd, lj, andparents ) )
	    { LDEB(i); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a document item.					*/
/*									*/
/*  This is the main entry poin of the formatter.			*/
/*									*/
/************************************************************************/

int docPsLayoutItemAndParents(	BufferItem *		bi,
				AppDrawingData *	add,
				const BufferDocument *	bd,
				LayoutJob *		lj,
				int			andParents )
    {
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    int				i;
    int				y1;

    ParagraphLayoutContext	plc;

    plc.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;

    plc.plcScreenLayout= lj->ljLayoutScreen;
    plc.plcStartParagraph= docPsLayoutParaInit;
    plc.plcLayoutLines= docPsLayoutLines;
    plc.plcAdjustBottom= docPsLayoutAdjustBottom;

    y1= BI_BELOW_PIXELS( add, bi )- 1;

    i= BI_TOP_PIXELS( add, bi );

    if  ( drChanged				&&
	  drChanged->drY0 > i	)
	{ drChanged->drY0=  i;	}

    bi->biTopPosition= lj->ljPosition;

    if  ( drChanged )
	{
	i= BI_TOP_PIXELS( add, bi );

	if  ( drChanged->drY0 > i )
	    { drChanged->drY0=  i;			}
	if  ( drChanged->drY1 < y1 )
	    { drChanged->drY1=  y1;	}
	}

    switch( bi->biLevel )
	{
	case DOClevDOC:
	rowAsGroup:
	sectAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docPsLayoutItemAndParents( bi->biChildren[i],
							    add, bd, lj, 0 ) )
		    { LDEB(1); return -1;	}
		}
	    break;

	case DOClevCELL:
	    {
	    ParagraphLayoutJob	plj;

	    docPsInitParagraphLayoutJob( &plj );

	    docPsBeginParagraphLayoutProgress( &plj,
					    0, 0, 0, bi->biChildCount,
					    &(lj->ljPosition) );

	    if  ( docPsLayoutParagraphs( &plc, bi, add, bd, &plj ) )
		{ LDEB(1); return -1;	}

	    lj->ljPosition= plj.pljPos.plpPos;

	    docFieldRefreshFlags( bi, bd );
	    }
	    break;

	case DOClevSECT:
	    if  ( ! bi->biParent )
		{ goto sectAsGroup;	}

	    if  ( docPsLayoutSectionItem( bi, add, bd, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevROW:
	    if  ( ! bi->biRowHasTableParagraphs )
		{ goto rowAsGroup;	}

	    docFieldRefreshFlags( bi, bd );

	    if  ( docPsLayoutRowItem( bi, add, bd, lj, &plc ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevPARA:
	    docFieldRefreshFlags( bi, bd );

	    if  ( docPsLayoutParaItem( bi, add, bd, lj, &plc ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    bi->biBelowPosition= lj->ljPosition;

    y1= BI_BELOW_PIXELS( add, bi )- 1;

    if  ( drChanged				&&
	  drChanged->drY1 < y1	)
	{ drChanged->drY1=  y1;	}

    if  ( andParents					&&
	  docPsAdjustParentGeometry( bi, add, bd, lj )	)
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Finish formatting a paragraph, either because additional lines are	*/
/*  to be added to the paragraph, or because its tail is to be shifted	*/
/*  (up or) down after some lines have been reformatted.		*/
/*									*/
/************************************************************************/

static int docPsLayoutFinishParaAdjust(
				const ParagraphLayoutContext *	plc,
				BufferItem *			paraBi,
				ParagraphLayoutPosition *	plp,
				AppDrawingData *		add,
				const BufferDocument *		bd )
    {
    int				done= 0;
    const DocumentProperties *	dp= &(bd->bdProperties);
    CellLayoutPosition *	clp= &(plp->plpProgress);

    while( clp->clpPart < paraBi->biParaParticuleCount )
	{
	int	accepted;

	accepted= (*plc->plcLayoutLines)( plc, paraBi, clp->clpPart, add, bd,
			    &(plp->plpFormattingFrame),
			    &(clp->clpLine), &(plp->plpPos), &(plp->plpPos) );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	plp->plpProgress.clpPart += accepted;
	done += accepted;

	/*  1  */
	if  ( clp->clpPart < paraBi->biParaParticuleCount		&&
	      ( accepted > 0 || ! plp->plpPos.lpAtTopOfColumn )		)
	    {
	    docPsToNextPage( paraBi, dp, add, &(plp->plpPos) );

	    if  ( clp->clpPart == 0 )
		{ paraBi->biTopPosition= plp->plpPos;	}
	    }
	}

    return done;
    }

/************************************************************************/
/*									*/
/*  Adjust the layout of a paragraph after a modification.		*/
/*									*/
/*  0)  Do not start with the first line of the page.			*/
/*  1)  Initialisation for paragraph formatting.			*/
/*  2)  Recalculate the layout of the first line that was changed.	*/
/*  3)  Did not fit on this page.. Try the next one.			*/
/*  4)  Layout subsequent lines of the paragraph, until the layout	*/
/*	process reaches the point outside the modified region where it	*/
/*	starts to cut lines at the same point where they were		*/
/*	originally cut.							*/
/*  5)  More lines than the paragraph originally contained are needed.	*/
/*	Just layout the rest.						*/
/*  6)  We are beyond the end of the modified region, and the new line	*/
/*	is to start where the old line starts. Just place the		*/
/*	subsequent lines from here.					*/
/*  7)  Recalculate the layout of a subsequent line that was changed.	*/
/*  8)  Remove superfluous lines from the administration.		*/
/*  9)  If the paragraph has changed height.. Adjust the redraw region.	*/
/*  10) Adjust the geometry of the rest of the document to the changes.	*/
/*									*/
/************************************************************************/

/*  3  */
static int docPsLayoutOneLine(	TextLine *			tl,
				BufferItem *			paraBi,
				ParagraphLayoutPosition *	plp,
				TextParticule *			tp,
				ParticuleData *			pd,
				AppDrawingData *		add,
				const BufferDocument *		bd,
				ScreenLayout *			sl,
				DocumentRectangle *		drChanged )
    {
    CellLayoutPosition *	clp= &(plp->plpProgress);
    const DocumentProperties *	dp= &(bd->bdProperties);

    const int			fromLinePos= 1;
    int				accepted;

    TextLine			boxLine;

    if  ( drChanged )
	{
	int	x0Pixels= TL_TOP_PIXELS( add, tl );
	int	x1Pixels= TL_BELOW_PIXELS( add, tl );

	if  ( drChanged->drY0 > x0Pixels	)
	    { drChanged->drY0=  x0Pixels;	}

	if  ( drChanged->drY1 < x1Pixels	)
	    { drChanged->drY1=  x1Pixels;	}
	}

    /*  2,7  */
    accepted= docPsLayout_Line( &boxLine, fromLinePos,
				paraBi, clp->clpPart, tp, pd, add, bd,
				&(plp->plpFormattingFrame),
				&(plp->plpPos), &(plp->plpPos), sl );

    if  ( accepted < 0 )
	{ LDEB(accepted); return -1;	}

    /*  3  */
    if  ( accepted == 0 )
	{
	if  ( paraBi->biInHeaderFooter != DOCinBODY )
	    { LDEB(paraBi->biInHeaderFooter);	}

	docPsToNextPage( paraBi, dp, add, &(plp->plpPos) );

	accepted= docPsLayout_Line( &boxLine, fromLinePos,
				paraBi, clp->clpPart, tp, pd, add, bd,
				&(plp->plpFormattingFrame),
				&(plp->plpPos), &(plp->plpPos), sl );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}
	}

    if  ( drChanged )
	{
	int	x1= LP_YPIXELS( add, &(plp->plpPos) );

	if  ( drChanged->drY1 < x1	)
	    { drChanged->drY1=  x1;	}
	}

    *tl= boxLine;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Redo the layout of a paragraph after an editing action.		*/
/*									*/
/*  To avoid screen flashing, a minimalistic approach is used: As	*/
/*  little of the layout is redone, and an administration is kept of	*/
/*  what has been changed, in order to redraw a minimal screen		*/
/*  rectangle.								*/
/*									*/
/************************************************************************/

int docPsAdjustParaLayout(	BufferItem *		paraBi,
				int			fromLine,
				int			stroffShift,
				int			upto,
				AppDrawingData *	add,
				const BufferDocument *	bd,
				LayoutJob *		lj )
    {
    const DocumentProperties *	dp= &(bd->bdProperties);

    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;

    TextLine			boxLine;
    TextLine *			tl;
    int				oneMore;

    TextParticule *		tp;

    ParticuleData *		pd;

    int				accepted;
    int				off;

    const int			bottomTwips= -1;
    const int			stripHigh= -1;

    int				paragraphHeightChanged;
    int				firstLineHeightChanged;

    ParagraphLayoutContext	plcL;
    ParagraphLayoutContext	plcP;

    ParagraphLayoutPosition	plp;

    plcL.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;
    plcP.plcChangedRectanglePixels= lj->ljChangedRectanglePixels;

    plcL.plcScreenLayout= lj->ljLayoutScreen;
    plcL.plcStartParagraph= docPsLayoutParaInit;
    plcL.plcLayoutLines= docPsLayoutLines;
    plcL.plcAdjustBottom= docPsLayoutAdjustBottom;

    plcP.plcScreenLayout= lj->ljPlaceScreen;
    plcP.plcStartParagraph= docPsPlaceParaInit;
    plcP.plcLayoutLines= docPsPlaceLines;
    plcP.plcAdjustBottom= docPsPlaceAdjustBottom;

    if  ( docPsClaimParticuleData( paraBi, &pd ) )
	{ LDEB(paraBi->biParaParticuleCount); return -1;	}

    /*  1  */
    if  ( docPsOpenParaFonts( paraBi, add, dp ) )
	{ LDEB(1); return -1;	}

    /*  0  */
    tl= paraBi->biParaLines+ fromLine;
    oneMore= 0;
    if  ( fromLine > 0							&&
	  tl[-1].tlTopPosition.lpPage != tl[ 0].tlTopPosition.lpPage	)
	{ fromLine--; tl--; oneMore= 1;	}
    tp= paraBi->biParaParticules+ tl->tlFirstParticule;

    plp.plpProgress.clpPara= paraBi->biNumberInParent;
    plp.plpProgress.clpLine= fromLine;
    plp.plpProgress.clpPart= tl->tlFirstParticule;
    plp.plpPos= tl->tlTopPosition;

    lj->ljPosition= tl->tlTopPosition;

    docParagraphFrameTwips( &(plp.plpFormattingFrame),
					bottomTwips, stripHigh, paraBi, dp,
					tl[ 0].tlTopPosition.lpPage,
					tl[ 0].tlTopPosition.lpColumn );

    if  ( docPsParagraphLineExtents( &(add->addPhysicalFontList), paraBi ) )
	{ LDEB(1); return -1;	}

    if  ( lj->ljLayoutScreen.slScreenFrame )
	{
	(*lj->ljLayoutScreen.slScreenFrame)( &(plp.plpFormattingFrame),
								add, paraBi );
	}

    if  ( lj->ljLayoutScreen.slStartParagraph				&&
	  (*lj->ljLayoutScreen.slStartParagraph)( paraBi, add, dp )	)
	{ LDEB(1); return -1;	}

    if  ( drChanged )
	{
	int	y0= TL_TOP_PIXELS( add, tl );

	if  ( drChanged->drX0 > add->addBackRect.drX0 )
	    { drChanged->drX0=  add->addBackRect.drX0;	}
	if  ( drChanged->drX1 < add->addBackRect.drX1 )
	    { drChanged->drX1=  add->addBackRect.drX1;	}

	if  ( drChanged->drY0 > y0 )
	    { drChanged->drY0=  y0;	}
	}

    /*  2,3  */
    boxLine= *tl;
    accepted= docPsLayoutOneLine( &boxLine, paraBi, &plp, tp, pd, add, bd,
					    &(lj->ljLayoutScreen), drChanged );

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    firstLineHeightChanged= 0;
    if  ( boxLine.tlTopPosition.lpPage != tl->tlTopPosition.lpPage	||
	  boxLine.tlTopPosition.lpPageYTwips !=
				    tl->tlTopPosition.lpPageYTwips	||
	  boxLine.tlLineHeightTwips != tl->tlLineHeightTwips		)
	{ firstLineHeightChanged= 1;	}

    *tl= boxLine;

    if  ( tl->tlHasPageBreak )
	{
	if  ( paraBi->biInHeaderFooter != DOCinBODY )
	    { LDEB(paraBi->biInHeaderFooter);				}
	else{ docPsToNextPage( paraBi, dp, add, &(plp.plpPos) );	}
	}

    off= tl->tlStroff+ tl->tlStrlen;
    plp.plpProgress.clpLine++; tl++;
    plp.plpProgress.clpPart += accepted; tp += accepted; pd += accepted;

    /*  4  */
    while( plp.plpProgress.clpPart < paraBi->biParaParticuleCount )
	{
	/*  5  */
	if  ( plp.plpProgress.clpLine >= paraBi->biParaLineCount )
	    {
	    accepted= docPsLayoutFinishParaAdjust( &plcL,
						    paraBi, &plp, add, bd );

	    if  ( accepted < 1 )
		{ LDEB(accepted); return -1;	}

	    break;
	    }

	/*  6  */
	if  ( ! oneMore						&&
	      tl->tlStroff + stroffShift == off && off >= upto )
	    {
	    int		ll;
	    int		partShift;

	    partShift= plp.plpProgress.clpPart- tl->tlFirstParticule;

	    for ( ll= plp.plpProgress.clpLine;
					ll < paraBi->biParaLineCount; ll++ )
		{
		paraBi->biParaLines[ll].tlFirstParticule += partShift;
		paraBi->biParaLines[ll].tlStroff += stroffShift;
		}

	    accepted= docPsLayoutFinishParaAdjust( &plcP,
						    paraBi, &plp, add, bd );

	    if  ( accepted < 1 )
		{ LDEB(accepted); return -1;	}

	    break;
	    }

	/*  7,3  */
	boxLine= *tl;
	accepted= docPsLayoutOneLine( &boxLine, paraBi, &plp, tp, pd, add, bd,
					    &(lj->ljLayoutScreen), drChanged );

	if  ( accepted < 1 )
	    { LDEB(accepted); return -1;	}

	*tl= boxLine;

	if  ( tl->tlHasPageBreak )
	    {
	    if  ( paraBi->biInHeaderFooter != DOCinBODY )
		{ LDEB(paraBi->biInHeaderFooter);			}
	    else{ docPsToNextPage( paraBi, dp, add, &(plp.plpPos) );	}
	    }

	off= tl->tlStroff+ tl->tlStrlen;
	plp.plpProgress.clpLine++; tl++; oneMore= 0;
	plp.plpProgress.clpPart += accepted; tp += accepted; pd += accepted;
	}

    /*  8  */
    if  ( plp.plpProgress.clpLine < paraBi->biParaLineCount )
	{
	docDeleteLines( paraBi, plp.plpProgress.clpLine,
			    paraBi->biParaLineCount- plp.plpProgress.clpLine );
	}

    if  ( plp.plpProgress.clpPart != paraBi->biParaParticuleCount )
	{
	LLDEB(plp.plpProgress.clpPart,paraBi->biParaParticuleCount);
	return -1;
	}
    if  ( plp.plpProgress.clpLine !=  paraBi->biParaLineCount )
	{
	LLDEB(plp.plpProgress.clpLine,paraBi->biParaLineCount);
	return -1;
	}

    /*  9  */
    paragraphHeightChanged= 0;

    docPsPlaceAdjustBottom( &paragraphHeightChanged, paraBi,
					    &(plp.plpPos), add, drChanged );

    lj->ljPosition= plp.plpPos;

    if  ( firstLineHeightChanged || paragraphHeightChanged )
	{
	if  ( firstLineHeightChanged	&&
	      paraBi->biNumberInParent > 0	)
	    {
	    paraBi= paraBi->biParent->biChildren[paraBi->biNumberInParent- 1];
	    lj->ljPosition= paraBi->biBelowPosition;
	    }

	/*  10  */
	if  ( docPsAdjustParentGeometry( paraBi, add, bd, lj ) )
	    { LDEB(1); return -1;	}
	}

    return 0;
    }

