/*
 *  $Id: write_ip.c,v 1.1.1.1 1999/04/20 15:58:15 dugsong Exp $
 *
 *  libnet
 *  write_ip.c - writes a prebuilt IP packet to the network
 *
 *  Copyright (c) 1998, 1999 Mike D. Schiffman <mike@infonexus.com>
 *                           route|daemon9 <route@infonexus.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#if (HAVE_CONFIG_H)
#include "../include/config.h"
#endif
#include "../include/libnet.h"

int
write_ip(int sock, u_char *buf, int len)
{
    int c, s;
    u_char *local_buf;
    struct sockaddr_in sin;
    struct ip *ip_hdr;
    struct udphdr *udp_hdr;
    struct tcphdr *tcp_hdr;

#if (BSD_BYTE_SWAP)
    /*
     *  It is non-intuitive for write_ip to change the packet it is writing.
     *  Indeed, it should not do this as it can confuse upper layer functions.
     *  Consider the case: for (i = 10; i; i--) write_ip(sock, buf, len);
     *  This would fail on every even call if BSD_BYTE_SWAP was defined
     *  and we didn't do have a local copy.
     *  XXX - local_buf could be made static to obviate the need for malloc
     *  calls during successive write_ip calls.
     */
    local_buf = (u_char *)malloc(len);
    if (!local_buf)
    {
#if (__DEBUG)
        fprintf(stderr, "write_ip: malloc failed (%s)\n", strerror(errno));
#endif
        return (-1);
    }

    memcpy(local_buf, buf, len);
#else
    local_buf = buf;
#endif

    ip_hdr = (struct ip *)(local_buf);

    memset(&sin, 0, sizeof(struct sockaddr_in));
    sin.sin_family      = AF_INET;
    sin.sin_addr.s_addr = ip_hdr->ip_dst.s_addr;

    /*
     *  Figure out which transport (if any) we are using and fill in the 
     *  socket address port information accordingly.  The relative complexity
     *  here is the price we pay for the simple interface.
     */
    switch (ip_hdr->ip_p)
    {
        case IPPROTO_TCP:
            tcp_hdr = (struct tcphdr *)(local_buf + IP_H);
            sin.sin_port = tcp_hdr->th_sport; 
            break;
        case IPPROTO_UDP:
            udp_hdr = (struct udphdr *)(local_buf + IP_H);
            sin.sin_port = udp_hdr->uh_sport; 
            break;
        case IPPROTO_ICMP:
        case IPPROTO_IP:
            sin.sin_port = 0;   /* Unused in this case */
            break;
        default:
#if (__DEBUG)
            fprintf(stderr, "write_ip: unknown transport\n");
#endif
            return (-1);
    }

    s = sizeof(struct sockaddr);
#if (BSD_BYTE_SWAP)
    /*
     *  For link access, we don't need to worry about the inconsistencies of
     *  certain BSD kernels.  However, raw socket nuances abound.  Certain
     *  BSD implmentations require the ip_len and ip_off fields to be in host
     *  byte order.  It's MUCH easier to change it here than inside the bpf
     *  writing routine.
     */
    ip_hdr->ip_len = FIX(ip_hdr->ip_len);
    ip_hdr->ip_off = FIX(ip_hdr->ip_off);
#endif

    c = sendto(sock, local_buf, len, 0, (struct sockaddr *)&sin, s);
    if (c != len)
    {
#if (__DEBUG)
        fprintf(stderr, "write_ip: %d bytes written (%s)\n", c, strerror(errno));
#endif
    }
    return (c);
}

/* EOF */
