/*
 * tcpreen.c - TCP connection reverse engineering tool.
 * Last modified: version 0.9.1, Aug 05 2002
 */

/**********************************************************************
 *  Copyright (C) 2002 Rmi Denis-Courmont.                           *
 *  This program is free software; you can redistribute and/or modify *
 *  it under the terms of the GNU General Public License as published *
 *  by the Free Software Foundation; version 2 of the license.        *
 *                                                                    *
 *  This program is distributed in the hope that it will be useful,   *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              *
 *  See the GNU General Public License for more details.              *
 *                                                                    *
 *  You should have received a copy of the GNU General Pulic License  *
 *  along with this program; if not, you can get it from:             *
 *  http://www.gnu.org/copyleft/gpl.html                              *
 *                                                                    *
 *  You are highly encouraged to submit your modifications to         *
 *  remi@simphalempin.com for possible inclusion in the official      *
 *  distribution. By doing so, and unless otherwise stated, you give  *
 *  Rmi Denis-Courmont an unlimited, non-exclusive right to modify,  *
 *  reuse and/or relicense the code.                                  *
 **********************************************************************/

/*
 * This program was tested under Linux Mandrake 8.1 (i586) and compiled
 * with GCC 2.96, but it should compile fine on most Unix flavor.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h> /* strncpy(), memset(), memcmp() */
#include <stdlib.h> /* exit(), calloc(), free(), abort() */
#include <stddef.h> /* size_t */
#ifdef HAVE_LIMITS_H
# include <limits.h> /* ULONG_MAX */
#endif
#ifdef HAVE_UNISTD_H
# include <sys/types.h> /* pid_t, uid_t */
# include <unistd.h> /* close(), fork(), select() */
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#if defined(HAVE_SYS_SOCKET_H)
# include <sys/socket.h>
#elif defined(_Windows)
# include <winsock.h>
# define perror( str ) stub_perror(str)
# define close( fd ) closesocket(fd)
void stub_perror(const char *str);
#endif
#ifdef HAVE_NETDB_H
# include <netdb.h> /* NI_* */
#endif
#ifdef HAVE_SYSLOG_H
# include <syslog.h>
#endif

#include "tcpreen.h"
#include "bridge.h"
#include "log.h"
#include "socketinfo.h"
#include "solve.h"

#ifndef HAVE_WORKING_FORK
# define fork( ) -1
# define wait( retval_ptr ) -1
#endif
#ifndef HAVE_SETUID
# define setuid( uid ) 0
#endif

#if 0
static void chg_table_elem(int *table, int newelem, int oldelem);
#define register_pid( table, pid ) chg_table_elem(table, pid, 0)
#define unregister_pid( table, pid ) chg_table_elem(table, 0, pid)
#else
# define register_pid( table, pid ) (void)0
# define unregister_pid( table, pid ) (void)0
#endif

/*
 * Establishes a bridge between a TCP listener and a TCP active
 * connection.
 * The listening socket is dropped as soon as the first incoming
 * connection is established.
 *
 * If <conf->maxclients> is nul, bridge_main will not fork, and will
 * treat each client connection one by one (this is the only possible
 * mode if you don't have fork()). Otherwise, it defines the maximum
 * number of child processes that can run at a time.
 *
 * NOTE: it is assumed that "pid_t" is "int" (or perhaps "unsigned").
 *
 * Returns 0 on success, true on error.
 */
int bridge_main(const struct bridgeconf *conf)
{
	int *in, *out, retval = -1, addrflags, numclients = 0, countdown;
	const struct logfile *logp;
	/*pid_t *clients; */

	/* explicitly bufferized settings */
	int mode;
	long maxclients;
	const struct logfile *logs;
	mode = conf->mode;
	logs = conf->logs;
	maxclients = conf->maxclients;

	/* Hostname lookup flags for getsockaddr(ie. getnameinfo()) */
	addrflags = (mode & tcpreen_numeric) ? NI_NUMERICHOST | NI_NUMERICSERV : 0;
	/* Number of connections to be accepted left */
	countdown = conf->totalclients;

	/* Clients PID table */
#if 0
	if (maxclients) {
		if ((clients = (uid_t *)calloc(maxclients, sizeof(uid_t))) == NULL) {
			perror(_("Memory allocation error"));
			return -1;
		}
		memset(clients, 0, sizeof(uid_t) * maxclients);
	}
#endif

	/* Opens system log */
	if (mode & tcpreen_syslog) {
		openlog("tcpreen", LOG_PID, LOG_DAEMON);
		syslog(LOG_NOTICE, _("starting\n"));
	}

	/* Server socket(s) */
	if (!(mode & tcpreen_connect_both)) {
		if (seteuid(conf->priv)) {
			perror("seteuid");
			if (mode & tcpreen_syslog)
				syslog(LOG_CRIT, _("seteuid failure (%m)"));
			goto end;
		}
		if ((retval = sockhostinfo_listen(&in, &conf->bridge)) != 0) {
			sockhostinfo_perror(retval, &conf->bridge);
			if (mode & tcpreen_syslog)
				syslog(LOG_CRIT, _("cannot bind to %s port %s (%m)"),
					conf->bridge.hostname, conf->bridge.service);
			goto end;
		}
		if (seteuid(conf->unpriv)) {
			perror("seteuid");
			if (mode & tcpreen_syslog)
				syslog(LOG_CRIT, _("seteuid failure (%m)"));
			goto drop_in;
		}

		if (logs != NULL) {
			int fd, *fdp;

			for (fdp = in; (fd = *fdp) != -1; fdp++) {
				char addr[NI_MAXADDRESS];

				getsockaddr(fd, addrflags, 0, addr, sizeof(addr));
				for (logp = logs; logp != NULL; logp = logp->next) {
					FILE *log;

					log = logp->stream;
					if (!fprintf(log, _("Listening on: %s\n"), addr)) {
						perror(_("log file writing error"));
						if (mode & tcpreen_syslog)
							syslog(LOG_CRIT, _("log file writing error (%m)"));
						goto drop_in;
					}
				}
			}
		}
	}

	if (mode & tcpreen_listen_both) {
		if (seteuid(conf->priv)) {
			perror("seteuid");
			if (mode & tcpreen_syslog)
				syslog(LOG_CRIT, _("seteuid failure (%m)"));
			goto drop_in;
		}
		if ((retval = sockhostinfo_listen(&out, &conf->server)) != 0) {
			sockhostinfo_perror(retval, &conf->server);
			if (mode & tcpreen_syslog)
				syslog(LOG_CRIT, _("cannot bind to %s port %s (%m)"),
					conf->server.hostname, conf->server.service);
			goto drop_in;
		}
		if (seteuid(conf->unpriv))
			abort(); /* cannot fail at this point */

		if (logs != NULL) {
			int fd, *fdp;

			for (fdp = out; (fd = *fdp) != -1; fdp++) {
				char addr[NI_MAXADDRESS];

				getsockaddr(fd, addrflags, 0, addr, sizeof(addr));
				for (logp = logs; logp != NULL; logp = logp->next) {
					FILE *log;

					log = logp->stream;
					if (!fprintf(log, _("Listening for server on: %s\n"), addr)) {
						perror(_("log file writing error"));
						if (mode & tcpreen_syslog)
							syslog(LOG_CRIT, _("log file writing error (%m)"));
						goto drop;
					}
				}
			}
		}
	}

	/* Definitely drops priviledges */
	if (seteuid(conf->priv) || setuid(conf->unpriv)) {
		perror("setuid");
		if (mode & tcpreen_syslog)
			syslog(LOG_CRIT, _("setuid failure (%m)"));
		goto drop;
	}

/*
 * Server loop (one client processed at a time).
 */
	retval = 0;
	do {
		int fd[4];

		if (countdown > 0)
			countdown--; /* one connection occurs (even if it fails) */
		
		/* settinp up client socket... */
		if (mode & tcpreen_connect_both) {
			if ((retval = sockhostinfo_connect(fd, &conf->bridge)) != 0) {
				sockhostinfo_perror(retval, &conf->bridge);
				continue;
			}
		}
		else
			if ((fd[0] = accept_array(in)) == -1) {
				perror(_("listening socket error"));
				if (mode & tcpreen_syslog)
					syslog(LOG_ERR, _("listening socket error (%m)"));
				continue;
			}

		if (logs != NULL) {
			char addr[NI_MAXADDRESS];

			getsockaddr(fd[0], addrflags, 1, addr, sizeof(addr));
			for (logp = logs; logp != NULL; logp = logp->next) {
				FILE *log;

				log = logp->stream;
				if (!fprintf(log, _("Connected from: %s\n"), addr)) {
					perror(_("log file writing error"));
					if (mode & tcpreen_syslog)
						syslog(LOG_CRIT, _("log file writing error (%m)"));
					close(fd[0]);
					retval = -1;
					continue;
				}
			}
		}
		if (mode & tcpreen_syslog) {
			if (mode & tcpreen_connect_both) {
				char addr[NI_MAXADDRESS];

				getsockaddr(fd[0], NI_NUMERICHOST, 0, addr, sizeof(addr));
				syslog(LOG_INFO, _("connection to client; local address: %s\n"), addr);
			} else {
				char addr[NI_MAXADDRESS];

				getsockaddr(fd[0], NI_NUMERICHOST, 1, addr, sizeof(addr));
				syslog(LOG_INFO, _("connection to client; remote address: %s\n"), addr);
			}
		}
		fd[3] = fd[0];

		/* setting up server socket... */
		if (mode & tcpreen_listen_both) {
			if ((fd[1] = accept_array(out)) == -1) {
				perror(_("server-side listening socket error"));
				if (mode & tcpreen_syslog)
					syslog(LOG_ERR, _("cannot connect to server (%m)\n"));
				close(fd[0]);
				continue;
			}
		}
		else
			if ((retval = sockhostinfo_connect(fd + 1, &conf->server)) != 0) {
				sockhostinfo_perror(retval, &conf->server);
				if (mode & tcpreen_syslog)
					syslog(LOG_ERR, _("cannot connect to server (%m)\n"));
				close(fd[0]);
				continue;
			}

		if (logs != NULL) {
			char addr[NI_MAXADDRESS];

			getsockaddr(fd[1], addrflags, 1, addr, sizeof(addr));
			for (logp = logs; logp != NULL; logp = logp->next) {
				FILE *log;

				log = logp->stream;
				if (!fprintf(log, _("Connected to: %s\n"), addr)) {
					perror(_("log file writing error"));
					if (mode & tcpreen_syslog)
						syslog(LOG_CRIT, _("log file writing error (%m)"));
					retval = -1;
					close(fd[0]);
					close(fd[1]);
				}
			}
		}
		if (mode & tcpreen_syslog) {
			if (mode & tcpreen_listen_both) {
				char addr[NI_MAXADDRESS];

				getsockaddr(fd[1], NI_NUMERICHOST, 1, addr, sizeof(addr));
				syslog(LOG_INFO, _("connection to server; remote address: %s\n"), addr);
			} else {
				char addr[NI_MAXADDRESS];

				getsockaddr(fd[1], NI_NUMERICHOST, 0, addr, sizeof(addr));
				syslog(LOG_INFO, _("connection to server; local address: %s\n"), addr);
			}
		}
		fd[2] = fd[1];

		/* Both sides of the bridge are now properly connected!	*/
		switch (retval = ((maxclients) ? fork() : 0)) {
			case -1:
				perror("fork");
				if (mode & tcpreen_syslog)
					syslog(LOG_ERR, _("fork() error (%m)"));
				close(fd[0]);
				close(fd[1]);
				retval = 0; /* won't stop */
				break;

			case 0: /* child process */
				retval = monitor_bridge(fd, logs, conf->bytelimit);

				if (fd[0] != -1)
					close(fd[0]);
				else if (fd[2] != -1)
					close(fd[2]);

				if (fd[1] != -1)
					close(fd[1]);
				else if (fd[3] != -1)
					close(fd[3]);

				if (maxclients)
					exit(retval);

			default: /* parent process */
				if (maxclients) {
					close(fd[0]);
					close(fd[1]);
					numclients++;
					register_pid(clients, retval);
				}
				retval = 0;
		}

	/* Make room for future clients */
	if (maxclients)
		while (maxclients == numclients) {
			pid_t pid;
			int val;

			if ((pid = wait(&val)) != -1) {
				unregister_pid(clients, pid);
				numclients--;
			}
		}

	} while (countdown && !retval);

	if (mode & tcpreen_syslog) {
		syslog(LOG_NOTICE, _("stopping\n"));
		closelog();
	}
drop:
	if (mode & tcpreen_listen_both)
		fd_closearray(out);
drop_in:
	if (!(mode & tcpreen_connect_both))
		fd_closearray(in);

	/*
	 * Graceful termination for pending clients
	 */
	while (numclients) {
		pid_t pid;
		int val;

		if ((pid = wait(&val)) != -1) {
			unregister_pid(clients, pid);
			numclients--;
		}
	}
end:
#if 0
	if (maxclients)
		free(clients);
#endif
	return retval;
}

#if 0
/*
 * Replace the first occurence of <oldelem> in table <table> with
 * <newelem>.
 * Always succeed (in less gentle words, this causes a segfault or
 * something like that if <oldelem> is not found).
 */
static void chg_table_elem(int *table, int newelem, int oldelem)
{
	while (*table != oldelem)
		table++;
	*table = newelem;
}
#endif

