/*
 * main.c - main() function for tcpreen - command line parsing and
 * basic security checks.
 */

/**********************************************************************
 *  Copyright (C) 2002 Rmi Denis-Courmont.                           *
 *  This program is free software; you can redistribute and/or modify *
 *  it under the terms of the GNU General Public License as published *
 *  by the Free Software Foundation; version 2 of the license.        *
 *                                                                    *
 *  This program is distributed in the hope that it will be useful,   *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              *
 *  See the GNU General Public License for more details.              *
 *                                                                    *
 *  You should have received a copy of the GNU General Pulic License  *
 *  along with this program; if not, you can get it from:             *
 *  http://www.gnu.org/copyleft/gpl.html                              *
 *                                                                    *
 *  You are highly encouraged to submit your modifications to         *
 *  remi@simphalempin.com for possible inclusion in the official      *
 *  distribution. By doing so, and unless otherwise stated, you give  *
 *  Rmi Denis-Courmont an unlimited, non-exclusive right to modify,  *
 *  reuse and/or relicense the code.                                  *
 **********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h> /* strtol(), stroul(), on Windows: exit() */
#include <string.h> /* strchr(), memcpy(), strncpy(), memset() */
#include <limits.h> /* ULONG_MAX */
#ifdef HAVE_GETOPT_H
# include <getopt.h> /* getopt_long() */
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h> /* geteuid(), getuid(), getopt() */
#endif
#ifdef HAVE_PWD_H
# include <pwd.h> /* getpwnam(), getpwuid() */
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h> /* PF_*, SOCK_STREAM */
#endif

#define begin_socket() (void)0
#define end_socket() (void)0

#ifdef HAVE_NETDB_H
# include <netdb.h> /* struct addrinfo */
#else
# ifdef _Windows
#  undef begin_socket
#  undef end_socket
void begin_socket(void);
void end_socket(void);
# endif
#endif

#ifdef HAVE_LOCALE_H
# include <locale.h>
#endif

#include "tcpreen.h"
#include "log.h"
#include "output.h"
#include "solve.h"

/* Possible return values */
#define MAIN_NOERR	0
#define MAIN_SHORTCIRCUIT -1 /* for internal use */
#define MAIN_PARMPROB	1 /* parameter problem */
#define MAIN_IOERR	2 /* I/O error */

int usage(void)
{
	puts(_(
"Usage: tcpreen [OPTION]... SERVER_ADDRESS [LOCAL_ADDRESS]\n"
"Establishes a bridge between two TCP ports then monitors a TCP session.\n"
"SERVER_ADDRESS is the real address of the server to connect to;\n"
"e.g.: `[www.myserver.com]:80' or just `80' for localhost, port 80.\n"
"LOCAL_ADDRESS is the address to which the client will connect.\n"
"\n"
"  -4, --inet     use IPv4 for the following host address\n"
"  -6, --inet6    use IPv6 for the following host address\n"
"  -b, --bytes    limit maximum TCP session length in bytes; e.g.: -b 1024\n"
"  -c, --connect  connect to the client instead of listening for it\n"
"  -C, --C        encode log file a bit like a C source file (default)\n"
"  -f, --force    enforce unusual options combination\n"
"  -F, --fork     enable multi-process operation; e.g.: -F 10 (10 processes)\n"
"  -h, --help     display this help and exit\n"
"  -H, --hex      encode log file entirely in hexadecimal\n"
"  -l, --listen   listen for the server (ouch!) instead of connecting to it\n"
"  -L, --syslog   log connections to syslog (as daemon/info)\n"
"  -m, --multi    monitor multiple subsequent connections\n"
"  -n, --numeric  disable reverse DNS lookup\n"
"  -N, --null     do not monitor data, only connections themselves\n"
"  -o, --output   open a log file; e.g.: -o mylog.txt\n"
"  -q, --quiet    do not write to stdout (default)\n"
"  -R, --raw      do not encode the log at all\n"
"  -s, --strip    strip non-printable characters when writing to log file\n"
"  -u, --uid      specify an unprivilieged UID/username to be used\n"
/*"  -U, --unix     use Unix socket whose path follows\n"*/
"  -v, --verbose  increase verbosity -- cumulative\n"
"  -V, --version  display program version and exit\n"));
	printf(_("Report any bug to: <%s>\n"), PACKAGE_BUGREPORT);
#ifndef HAVE_GETOPT_LONG
	puts(_("Long options are not supported on your system."));
#endif
	return MAIN_SHORTCIRCUIT;
}

int version(void)
{
#ifndef VERSION
# define VERSION "unknown"
#endif
	puts(
"tcpreen "VERSION"\n"
"\n"
"Copyright (C) 2002 Rmi Denis-Courmont");
	puts(_(
"This is free software; see the source for copying conditions.\n"
"There is NO warranty; not even for MERCHANTABILITY or\n"
"FITNESS FOR A PARTICULAR PURPOSE.\n"));
	printf(_("Written by %s\n"), "Rmi Denis-Courmont <remi@simphalempin.com>");
	return MAIN_SHORTCIRCUIT;
}

/*
 * Generic error handling
 */
static int error_gen(const char *path, const char *msg)
{
	fprintf(stderr, "%s: %s.\n", path, _(msg));
	return MAIN_PARMPROB;
}

static int error_host(const char *hostname)
{
	return error_gen(hostname, N_("invalid host specification\n"
		"Expected format: [hostname]:service\n"
		"Examples:        [localhost]:smtp\n"
		" (IPv4 only:)    [192.168.0.1]:80\n"
		" (IPv6 only:)    [::1]:telnet"));
}

static int error_qty(const char *quantity)
{
	return error_gen(quantity, N_("invalid number (or capacity exceeded)"));
}

static int error_extra(const char *extra)
{
	return error_gen(extra, N_("unexpected extra parameter"));
}

/*
 * Command line options parsing.
 * argc, argv and loglist must be kept valid at all cost.
 * servername and bridgename MUST be NI_MAXHOST bytes long,
 * serverservice and bridgeservice MUST be NI_MAXSERV bytes long.
 */
int parse_args(int argc, char *argv[], struct bridgeconf *conf)
{
	int (*logfunc)(const void *, int, FILE *) = fwrite_C;
	int check, enforce = 0, verbose = 0, count = 0;
	struct sockhostinfo *curinfo;
	struct option longopts[] = {
		{ "inet",	1, NULL, '4' },
		{ "inet6",	1, NULL, '6' },
		{ "bytes",	1, NULL, 'b' },
		{ "connect",	0, NULL, 'c' },
		{ "C",		0, NULL, 'C' },
		{ "force",	0, NULL, 'f' },
		{ "fork",	1, NULL, 'F' },
		{ "help",	0, NULL, 'h' },
		{ "hex",	0, NULL, 'H' },
		{ "listen",	0, NULL, 'l' },
		{ "syslog",     0, NULL, 'L' },
		{ "multi",	2, NULL, 'm' },
		{ "numeric",	0, NULL, 'n' },
		{ "null",	0, NULL, 'N' },
		{ "output",	1, NULL, 'o' },
		{ "quiet",	0, NULL, 'q' },
		{ "raw",	0, NULL, 'R' },
		{ "strip",	0, NULL, 's' },
		{ "uid",	1, NULL, 'u' },
		/*{ "unix",	1, NULL, 'U' }, -- add "U:" to getopt_long */
		{ "verbose",	0, NULL, 'v' },
		{ "version",	0, NULL, 'V' },
		{ NULL, 	0, NULL, 0 }
	};
	/* No parameters at all? */
	if (argc <= 1)
		return usage();

	curinfo = &conf->server;

	while ((check = getopt_long(argc, argv, "4:6:b:cCfF:hHlLm::nNo:qRsu:vV", longopts, NULL)) != EOF) {
		switch (check) {
			case '4':
				if (count > 1)
					return error_extra(optarg);
				curinfo->family = PF_INET;
				if (parse_host(optarg, curinfo->hostname, curinfo->service))
					return error_host(optarg);
				count++;
				break;

			case '6':
				if (count > 1)
					return error_extra(optarg);
				curinfo->family = PF_INET6;
				if (parse_host(optarg, curinfo->hostname, curinfo->service))
					return error_host(optarg);
				count++;
				break;

			case 'b': {
				char *end;
				long lim;

				lim = strtol(optarg, &end, 0);
				if ((*end) || (lim < 0) || (lim == LONG_MAX))
					return error_qty(optarg);
				conf->bytelimit = lim;
			}
				break;

			case 'c':
				conf->mode |= tcpreen_connect_both;
				break;

			case 'C':
				logfunc = fwrite_C;
				break;

			case 'f':
				enforce = 1;
				break;

			case 'F': {
				char *end;
				long num;

				num = strtoul(optarg, &end, 0);
				if (*end || (num > INT_MAX))
					return error_qty(optarg);
				conf->maxclients = (int)num;
			}
				break;

			case 'h': /* help */
				return usage();

			case 'H':
				logfunc = fwrite_hex;
				break;

			case 'l':
				conf->mode |= tcpreen_listen_both;
				break;

			case 'L':
				conf->mode |= tcpreen_syslog;
				break;

			case 'm':
				if (optarg != NULL) {
					char *end;
					conf->totalclients = strtol(optarg, &end, 0);
					if (*end)
						return error_qty(optarg);
				}
				else
					conf->totalclients = -1;
				break;

			case 'n':
				conf->mode |= tcpreen_numeric;
				break;

			case 'N':
				logfunc = NULL;
				break;

			case 'o': {
				int is_not_stdout;
				FILE *secure_append_fopen(const char *path);

				is_not_stdout = strcmp(optarg, "-");

				if (logfile_add(&conf->logs,
						is_not_stdout ? secure_append_fopen(optarg) : stdout,
						is_not_stdout, logfunc)) {
					perror(is_not_stdout ? optarg: _("Standard output"));
					return MAIN_IOERR;
				}
			}
				break;

			case 'q':
				verbose = 0;
				break;

			case 'R':
				logfunc = fwrite_raw;
				break;

			case 's':
				logfunc = fwrite_strip;
				break;

			case 'u': {
				char *end;

				conf->unpriv = (uid_t)strtol(optarg, &end, 0);
				if (*end) { /* optarg is not a number */
					struct passwd *pwd;
					
					if ((pwd = getpwnam(optarg)) == NULL)
						return error_gen(optarg, N_("unrecognized user name"));
					conf->unpriv = pwd->pw_uid;
					break;
				}
#if HAVE_GETPWUID
				else {
					struct passwd *pwd;
					
					if ((pwd = getpwuid(conf->unpriv)) == NULL)
						return error_gen(optarg, N_("invalid UID"));
				}
#endif
			}
#if 0
			case 'U':
				if (count > 1)
					return error_extra(optarg);
				curinfo->family = PF_UNIX;
				*curinfo->hostname = 0;
				strncpy(curinfo->service, optarg, sizeof(curinfo->service));
				curinfo->service[sizeof(curinfo->service)] = 0;
				count++;
				break;
#endif
			case 'v':
				verbose ++;
				break;

			case 'V':
				return version();

			default: /* never happens */
			case '?': /* error: unrecognized option */
				usage();
				return MAIN_PARMPROB;
		}
		if (count == 1)
				curinfo = &conf->bridge;
	}

	if ((count == 0) && (optind < argc)) {
		if (parse_host(argv[optind], conf->server.hostname,
				conf->server.service))
			return error_host(argv[optind]);
		count++;
		optind++;
	}
	if ((count == 1) && (optind < argc)) {
		if (parse_host(argv[optind], conf->bridge.hostname,
				conf->bridge.service))
			return error_host(argv[optind]);
		/* count++; */
		optind++;
	}
	if (optind < argc)
		return error_extra(argv[optind]);

	if (verbose)
		if (logfile_add(&conf->logs, stdout, 0, (verbose == 1) ? NULL : logfunc)) {
			perror(argv[0]);
			return MAIN_IOERR;
		}

	/*
	 * Some sanity checks
	 */
	if (!(*conf->server.service) && !(conf->mode & tcpreen_listen_both))
		return error_gen(argv[0], N_("no server port specified"));
	if (!(*conf->bridge.service) && (conf->mode & tcpreen_connect_both))
		return error_gen(argv[0], N_("no client port specified with -c option"));
	
	if (!enforce) {
		const char *warn_msg = NULL;

#ifdef HAVE_GETUID
		if (conf->unpriv == 0)
			warn_msg = N_("insecure use with root privileges (you should use -u)");
#endif
		if ((conf->mode & tcpreen_reversed) == tcpreen_reversed)
			warn_msg = N_("probably erroneous use of both -c and -l options");
		else if ((conf->mode & tcpreen_connect_both) && (conf->totalclients != 1))
			warn_msg = N_("dangerous use of both -c and -m options");
		else if ((conf->logs == NULL) && (!(*conf->bridge.service) || !(*conf->server.service)))
			warn_msg = N_("dynamically allocated port but nowhere to log");
		else if (conf->maxclients > conf->totalclients)
			warn_msg = N_("option -F should be used together with option -m,\n"
					"(with a value higher than or equal to that of -m)");

		if (warn_msg != NULL) {
			fprintf(stderr, _("%s: %s;\nUse -f to override.\n"),
				argv[0], warn_msg);
			return MAIN_PARMPROB;
		}
	}
	if ((conf->maxclients) && (conf->logs != NULL))
		return error_gen(argv[0], N_("cannot use log files in multi-process mode (-F)"));
	return MAIN_NOERR;
}


/*
 * Main function
 */
int main(int argc, char *argv[])
{
	struct bridgeconf conf;
	int val;

	/* Default settings */
	memset(&conf, 0, sizeof (conf));
	conf.bytelimit = -1;
	conf.totalclients = 1;
	conf.priv = geteuid();
	conf.unpriv = getuid();
	conf.server.socktype = conf.bridge.socktype = SOCK_STREAM;

	/* Use low privileges during initialization *
	 * (in particular to open log files)	    */
	seteuid(conf.unpriv);

	/* Initialization */
	begin_socket();

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	/* Command line parsing and checking */
	val = parse_args(argc, argv, &conf);

	/* Let's come to serious things... */
	if (!val)
		val = bridge_main(&conf);

	logfile_closeall(conf.logs);
	end_socket();

	return (val >= 0 ) ? val : MAIN_IOERR;
}


#ifdef _WINSOCKAPI_
/*
 * Additionnal functions for Winsock support
 */
# undef perror
# include <windows.h>
# include <wincon.h> /* SetConsoleTitle() */
# include <winsock.h>
# ifndef WINSOCK_VERSION
#  define WINSOCK_VERSION 0x0101
# endif
void begin_socket(void)
{
	WSADATA wsaData;

	SetConsoleTitle(_("TCP re-engineering tool"));

	if(!WSAStartup(WINSOCK_VERSION, &wsaData)) {
		if(wsaData.wVersion == WINSOCK_VERSION)
			return;
		else
			WSACleanup();
	}

	fputs(_("Winsock unavailable or unsupported version. Aborting.\n"), stderr);
	exit(1);
}


void end_socket(void)
{
	WSACleanup();
}

/*
 * There might be some problems here:
 * if both errno and WSAGetLastError() are non nul, we cannot guess
 * which one to report.
 * Lets not use M$ Windows if this is a real problem for you.
 */
void stub_perror(const char *str)
{
	int err;

	if((err = WSAGetLastError()) != 0) {
		fprintf(stderr, _("%s: Winsock error number %d.\n"), str, err);
	}
	else
		perror(str);
}
#endif

