/*
 * bridge.c - Monitored file (socket, pipe...) bridge
 */

/**********************************************************************
 *  Copyright (C) 2002 Rmi Denis-Courmont.                           *
 *  This program is free software; you can redistribute and/or modify *
 *  it under the terms of the GNU General Public License as published *
 *  by the Free Software Foundation; version 2 of the license.        *
 *                                                                    *
 *  This program is distributed in the hope that it will be useful,   *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.              *
 *  See the GNU General Public License for more details.              *
 *                                                                    *
 *  You should have received a copy of the GNU General Pulic License  *
 *  along with this program; if not, you can get it from:             *
 *  http://www.gnu.org/copyleft/gpl.html                              *
 *                                                                    *
 *  You are highly encouraged to submit your modifications to         *
 *  remi@simphalempin.com for possible inclusion in the official      *
 *  distribution. By doing so, and unless otherwise stated, you give  *
 *  Rmi Denis-Courmont an unlimited, non-exclusive right to modify,  *
 *  reuse and/or relicense the code.                                  *
 **********************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h> /* memcpy() */
#ifdef HAVE_LIMITS_H
# include <limits.h> /* SHRT_MAX */
#endif
#ifdef HAVE_UNISTD_H
# include <sys/time.h> /* struct timeval (unused, but needed for select) */
# include <unistd.h> /* close(), select(), read(), write() */
#endif
#ifdef HAVE_SYS_SOCKET_H
# include <sys/socket.h> /* shutdown() */
#else
# ifdef _Windows
#  include <winsock.h>
#  define perror( str ) stub_perror(str)
void stub_perror(const char *str);
#  define close( fd ) closesocket(fd)
#  define read( fd, buf, len ) recv(fd, buf, len, 0)
#  define write( fd, buf, len ) send(fd, buf, len, 0)
# else
#  define shutdown( fd, how ) (void)0
# endif
#endif

#include "log.h"

/*
 * Finds a pair of fds (one for reading, one for writing) in fd vectors
 * readfds and writefds, both of length <len>.
 *
 * Loops for ever if len is nul (that's not a bug, that's logic).
 * Returns a non-negative value (`val') on success, -1 on select() error.
 *
 * On return, readfds[index] is readable, writefds[index] is writeable,
 * exceptflag is true if an exceptionnal condition occured on
 * readfds[index] rather than a readability event.
 *
 * Function undefined if len is negative or exceeds MAXFILE.
 */
inline
int select_fd_pair(const int *readfds, const int *writefds, int len,
			int *exceptflag)
{
	while (1) {
		fd_set readset, writeset, exceptset;
		int maxreadfd = -1, maxwritefd = -1, i, s2;

		FD_ZERO(&readset);
		FD_ZERO(&writeset);

		for (i = 0; i < len; i++) {
			int fd;

			if ((fd = readfds[i]) != -1) {
				FD_SET(fd, &readset);
				if (maxreadfd < fd)
					maxreadfd = fd;
			}

			if ((fd = writefds[i]) != -1) {
				FD_SET(fd, &writeset);
				if (maxwritefd < fd)
					maxwritefd = fd;
			}
		}
		memcpy(&exceptset, &readset, sizeof(exceptset));


		/*  Finds readable stream(s)... */
		if ((select(++maxreadfd, &readset, NULL, &exceptset, NULL) == -1)
		/* ...and then writeable stream(s). */
		 || ((s2 = select(++maxwritefd, NULL, &writeset, NULL, NULL)) == -1))
			return -1;

		for (i = 0; s2 > 0; i++) {
			int fd;

			if ((fd = writefds[i]) != -1)
				if (FD_ISSET(fd, &writeset)) {
					s2--;
					fd = readfds[i];
					if (FD_ISSET(fd, &exceptset)) {
						*exceptflag = 1;
						return i;
					} else if (FD_ISSET(fd, &readset)) {
						*exceptflag = 0;
						return i;
					}
				}
		}
	}
}

int times_in_array(int *array, int len, int elem)
{
	int c = 0, i;

	for (i = 0; i < len; i++)
		if (array[i] == elem)
			c++;
	return c;
}

static int spare_close(int *readfds, int *writefds, int len, int fd, int how)
{
	if (times_in_array(how ? writefds : readfds, len, fd) == 1) {
		if (times_in_array(how ? readfds : writefds, len, fd) == 0)
			return close(fd);
		else
			return shutdown(fd, how);
	}
	return 0;
}


#define MAX_PACKET_SIZE (SHRT_MAX > 65535) ? 65535 : SHRT_MAX /* (bytes) */

/*
 * Operates a bridge between fd[0](input)/fd[2](output) and another
 * between fd[1](input)/fd[3](output) and display any transmitted data
 * to each streams in the NULL-terminated stream list <logs>.
 *
 * Note: no assumption is made about the transport protocol used,
 * but the use of shutdown() assumes we work with sockets
 * (shutdown() will otherwise silently fail -- not a big problem).
 *
 * fd is modified: closed descriptors are replaced by (-1).
 * In case of error, some of them might not have been closed.
 * Do it yourself.
 */

#define blen 2 /* number of half-duplex bridges */
#define readfds fds
#define writefds (fds+blen)
int monitor_bridge(int fds[4], const struct logfile *logs,
		   long limit)
{
	long count[blen] = { 0, 0 }, totalcount = 0;
	const struct logfile *logp;
	int (*logwrite)(const void *, int, FILE *), bcount = blen;

	for (logp = logs; logp != NULL; logp = logp->next)
		if (fputs(_("Transmission started...\n"), logp->stream) < 0)
			goto log_error;

	do {
		int index, oobflag;

		/* What should we do? */
		if ((index = select_fd_pair(readfds, writefds, blen, &oobflag)) < 0) {
			perror("select");
			return -1;
		} else {
		/* Processes data */
			int len, rfd;
			char buf[MAX_PACKET_SIZE];
			
			/* Reads data */
			rfd = readfds[index];
			switch (len = (oobflag) ? recv(rfd, &buf, sizeof(buf), MSG_OOB)
						: read(rfd, &buf, sizeof(buf))) {
				case -1:
					perror("read/recv");
				case 0: /* end-of-file */
					spare_close(readfds, writefds, blen, rfd, 0);
					readfds[index] = -1;
					spare_close(readfds, writefds, blen, writefds[index], 1);
					writefds[index] = -1;
					bcount--;

					for (logp = logs; logp != NULL; logp = logp->next)
						if (fputs(_((index & 1) ? "End of output.\n" : "End of input.\n"), logp->stream) < 0)
							goto log_error;
					break;

				default:
					for (logp = logs; logp != NULL; logp = logp->next)
						if ((logwrite = logp->write) != NULL) {
							FILE *log;
							log = logp->stream;
							if (((oobflag) && (fputs("(OOB)", log) < 0))
							 || (fputs((index & 1) ? ">>> " : "<<< ", log) < 0)
							 || (logwrite(buf, len, log) != len))
								goto log_error;
						}

					if (count[index] >= 0)
						count[index] += len;
					totalcount += len;
					
					/* Sends data to the real destination */
					if (((oobflag) ? send(writefds[index], buf, len, MSG_OOB)
							: write(writefds[index], buf, len)) != len) {
						perror("write/send");

						spare_close(readfds, writefds, blen, rfd, 0);
						readfds[index] = -1;
						spare_close(readfds, writefds, blen, writefds[index], 1);
						writefds[index] = -1;
						bcount--;
					}
			}
		}
		if (totalcount < 0)
			totalcount = LONG_MAX;
	} while (bcount && ((limit == -1) || (limit > totalcount)));

	for (logp = logs; logp != NULL; logp = logp->next) {
		FILE *log;

		log = logp->stream;
		/* Displays final statistics */
		if (fputs(_("End of transmission:\n"), log) < 0)
			goto log_error;
		if (count[0] >= 0)
			fprintf(log, _(" %ld byte(s) received.\n"), count[0]);
		if (count[1] >= 0)
			fprintf(log, _(" %ld byte(s) sent.\n"), count[1]);
		if ((limit != -1) && (limit <= totalcount))
			fputs(_(" transmission length limit exceeded.\n"), log);
	}
	return 0;

log_error:
	perror(_("log file writing error"));
	return -1;
}

