/***************************************************************************
                          teleporter.cpp  -  description
                             -------------------
    begin                : Tue Oct 30 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"
#include "teleporter.h"

#include <qcanvas.h>
#include <qdom.h>
#include <qstring.h>

#include "level.h"
#include "taxi.h"
#include "movingobject.h"
#include "cdp.h"
#include "event.h"
#include "state.h"

Teleporter::Teleporter(const QDomElement *description, Level *parent, QCanvas *canvas) {

	my_canvas = canvas;
	level = parent;
	cdp = parent->cdp;
	sisters = level->teleporters;

	visible = true;
	has_object = false;

	read_teleporter (description);
}

Teleporter::~Teleporter(){

	if (!has_object) {
		delete object;		// delete the constant sprite. True objects deleted by level
	}

}

/** Checks, whether the taxi would currently get teleported by this teleporter and if so,
teleports the taxi there, decreasing the score if appropriate.
Returns true, if teleportation took place */
bool Teleporter::check_teleportation (Taxi *taxi, double *score){
	if (destination_tp) {	// if no destination_tp, this teleporter won't take us anywhere, anyway
		if (visible) {		// if not visible, tp can not function
			if (active_leave ()) {	// if not active, tp can not function
				if (*score >= fare) {
					if (sisters[destination_tp-1]->visible && sisters[destination_tp-1]->active_arrive ()) {
						bool collides = false;
						QRect taxi_rect = taxi->boundingRect ();
						if (fixed_size) {
							if (has_object) {
								calculate_region ();
							}
							if (taxi_rect.intersects (region)) {
								collides = true;
							}
						} else {
							if (taxi->collidesWith (object)) {
								collides = true;
							}
						}
						if (collides) {
							if (!blocked) {
								if (has_object) {
									hot_area.moveTopLeft (QPoint (object->x () + hot_area_off_x, object->y () + hot_area_off_y));
								}
								if (taxi_rect.contains(hot_area, true)) {
			                                         	*score -= fare;
									// trigger leave-event if available (trigger before arrive_event!)
									if (leave_event) {
										leave_event->trigger ();
									}
									sisters[destination_tp-1]->teleport_to_you (taxi, factor_x, factor_y);
									return (true);
								}
							}
						} else {
							blocked = false;
						}				
					}
				}
			}
		}
	}

	return (false);		// else return not teleporting
}

/** The taxi is to be teleported to this teleporter */
void Teleporter::teleport_to_you (Taxi *taxi, double f_x, double f_y){
	if (has_object) {
		taxi->teleport_to(0, 0, f_x, f_y, false, this);
	} else {
		taxi->teleport_to(x, y, f_x, f_y);
	}
	blocked = true;

	// trigger arrive-event if available
	if (arrive_event) {
		arrive_event->trigger ();
	}
}

/** Tells the teleporter, whether it is currently
visible */
void Teleporter::set_visible(bool visibility){
	visible = visibility;
}

/** Read the teleporter's description and creates the teleporter */
void Teleporter::read_teleporter (const QDomElement *description) {

	destination_tp = cdp->get_int_attribute("destination", *description, 0, level->num_teleporters, 1, 1);
	fare = cdp->get_double_attribute("fare", *description, -100, 100, 0, 1);
	factor_x = cdp->get_double_attribute("factor_x", *description, -100, 100, 1, 1);
	factor_y = cdp->get_double_attribute("factor_y", *description, -100, 100, 1, 1);

	if (description->hasAttribute ("ident")) {
		object_ident = cdp->get_string_attribute ("ident", *description, "#bad value#", 1);
		object_promised = true;
	} else {
		object_promised = false;

		x = cdp->get_int_attribute("x", *description, 0, 3000, 100, 1);
		y = cdp->get_int_attribute("y", *description, 0, 3000, 100, 1);

		object = new QCanvasSprite(new QCanvasPixmapArray(cdp->prefix_check(cdp-> get_string_attribute("image", *description, "teleporter1.png",
																					   2), true), 1), my_canvas);

		object->move(x, y);
		object->moveBy(-object->boundingRect().width() / 2, -object->boundingRect().height() /2);
		calculate_hot_area ();

		object->setAnimated (false);
		if (cdp->for_real()) {
			object->show ();
		}
	}

	if (description->hasAttribute("width") || description->hasAttribute("height")) {
		region.setWidth (cdp->get_int_attribute("width", *description, 10, 100, 20, 2));
		region.setHeight (cdp->get_int_attribute("height", *description, 10, 100, 20, 2));
		calculate_region ();
		fixed_size = true;
	} else {
		fixed_size = false;
	}

// events
	QString dummy;

	if ((dummy = cdp->get_string_attribute ("arrive_event", *description, "NOTHING", 3)) == "NOTHING") {
		arrive_event = 0;
	} else {
		arrive_event = level->resolve_event (dummy);
	}
	if ((dummy = cdp->get_string_attribute ("leave_event", *description, "NOTHING", 3)) == "NOTHING") {
		leave_event = 0;
	} else {
		leave_event = level->resolve_event (dummy);
	}

// states
	if ((dummy = cdp->get_string_attribute ("arrive_state", *description, "NOTHING", 3)) == "NOTHING") {
		arrive_state = 0;
	} else {
		arrive_state = level->resolve_state (dummy);
	}
	if ((dummy = cdp->get_string_attribute ("leave_state", *description, "NOTHING", 3)) == "NOTHING") {
		leave_state = 0;
	} else {
		leave_state = level->resolve_state (dummy);
	}
}

/** Calculated the (inital) region for this teleporter (if fixed_size) */
void Teleporter::calculate_region () {

	QPoint center = object->boundingRect().center ();
	region.moveCenter(center);

}

/** Calculates the (initial) hot_area for this teleporter */
void Teleporter::calculate_hot_area () {

	hot_area = object->boundingRect();
	QPoint point = hot_area.center();
	hot_area.setWidth(10);
	hot_area.setHeight(10);
	hot_area.moveCenter(point);

	center_off_x = object->boundingRect ().width () / 2;
	center_off_y = object->boundingRect ().height () / 2;

	hot_area_off_x = center_off_x - 5;
	hot_area_off_y = center_off_y - 5;
}

/** Fulfills the promise (assigns the promised object to this teleporter) */
void Teleporter::promise_fulfilled(MovingObject *obj){

	object = obj;
	calculate_hot_area ();
	has_object = true;

}

/** Returns the current x-destination for teleporting to this teleporter */
int Teleporter::destX (){
	return ((int) (object->x() + center_off_x));
}

/** Returns the current y-destination for teleporting to this teleporter */
int Teleporter::destY(){
	return ((int) (object->y() + center_off_y));
}

/** Returns whether the teleporter is currently in a state that it will transport taxis that
fly onto them (as opposed to active_arrive (), which returns, whether the teleporter will accept "incoming traffic") */
bool Teleporter::active_leave (){
	if (leave_state) {
		return leave_state->isSet ();
	} else {
		return true;
	}
}

/** Returns whether the teleporter is currently in a state that it will accept "incoming traffic" (as opposed to active_leave (), which returns, whether the teleporter will accept "outgoing traffic") */
bool Teleporter::active_arrive(){
	if (arrive_state) {
		return arrive_state->isSet ();
	} else {
		return true;
	}
}
