/***************************************************************************
                          taxi.cpp  -  description
                             -------------------
    begin                : Fri May 18 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"

#include <stdlib.h>

#include <qdom.h>

#include "taxi.h"

#include "taxipilot.h"
#include "cdp.h"
#include "teleporter.h"
#include "platform.h"
#include "movingobject.h"

Taxi::Taxi(const QString & file, QCanvas * canvas, int base_x, int base_y, Taxipilot * parent,
		   bool for_credits_only):QCanvasSprite(0, canvas)
{

	cdp = parent->cdp;
	tm = parent->tm;

	idle_timer = anim_timer = flaps_anim_timer =-1;
	lr = 0;
	fuel_low = fuel_out = false;
	flaps_state = 0;
	flaps_prev_state = false;
	landed = false;

	flaps = new QCanvasSprite(0, canvas);
	flaps->setZ(TAXI_FLAPS_Z);
	flaps->setAnimated (false);

	// prepare the flying-animations
	flying_lr = new QCanvasSprite(0, canvas);
	flying_lr->setAnimated (false);
	flying_ud = new QCanvasSprite(0, canvas);
	flying_ud->setAnimated (false);
	flying_lr->setZ(TAXI_FLYING_Z);
	flying_ud->setZ(TAXI_FLYING_Z);

	read_taxi(file);

	if (!cdp->isOk()) return;		// serious parsing error!

	// intialize flying-animation with a dummy (just to be sure)
	if (cdp->for_real()) {
		flying_lr->setSequence(flying_left.anim[lr]);
		flying_lr->setFrame(flying_lr_frame = 0);
		flying_ud->setSequence(flying_up.anim[lr]);
		flying_ud->setFrame(flying_ud_frame = 0);
        }

/* calculate offset x and y */
	setSequence(still.anim[lr]);
	setFrame(0);
	off_x = width() / 2;
	if (flaps_avail && cdp->for_real()) {
		flaps->setFrame(0);
		double_off_y = flaps->boundingRect().height();
	} else {
		double_off_y = height();
	}
	off_y = double_off_y / 2;

//	paused = false;
	setZ(TAXI_Z);
	dynamic_teleport = false;

// Initialize pointers in order to prevent crashes on udefined pointers
	current_anim = &still;
	cflaps_anim = &flaps_anim;

	// appear!
	if (!for_credits_only) {
		teleport_to(base_x, base_y, 0, 0, true);

		show();					// here we go! ( Note that teleport_to only triggers teleporting! )
	} else {
		go_idle ();
	}

// the taxi is the major moving object. Therefor we adjust the canvas to it.
	int x = width ();
	if (height () > x) {		// don't wanna include math.h only for max ()!
		x = height ();
	}
	canvas->retune (16 + ((int) (x / 16)) * 16);

	config_changed ();
}

Taxi::~Taxi()
{

	tm->killTimers (this);		// probably not necessary to do this explicitly, but who knows
	hide ();
	flaps->hide ();
	flying_lr->hide ();
	flying_ud->hide ();

	for (int i = 0; i < num_idle_animations; i++) {
		delete_animation (idle[i]);
	}

	delete_animation (teleporting);
	delete_animation (still);

	if (cdp->for_real () || cdp->thorough_check ()) {
		delete_animation (flying_up);
		delete_animation (flying_down);
		delete_animation (flying_left);
		delete_animation (flying_right);
		if (flaps_avail) {
			delete_animation (flaps_anim);
		}
	}	

	delete[]idle;
	delete flying_lr;
	delete flying_ud;
	delete flaps;

}

/** Signals we are going up. The reason this is not simply done via setVelocity is
that taxi is eventually to do animation stuff. The double acceleration is passed as
a parameter instead of using some constant amount, so we may eventually
abstract from keyboard input, and allow (analog) joystick input */
void Taxi::up(double acc)
{
	if (ready) {
		if (fuel > 0) {

			flying_ud->setSequence(flying_up.anim[lr]);
			if (flying_ud_frame >= flying_up.frames) {
				flying_ud_frame = 0;
			}
			flying_ud->move(x(), y());
			flying_ud->setFrame(flying_ud_frame++);
			flying_ud->show();

			if (landed) {
				landed = false;
				platform_p->detach_taxi ();
				setXVelocity ((platform_p->current_plat_vx () * (1000 / OBJECTS_AP) / cdp->fps ()));
				setYVelocity ((platform_p->current_plat_vy () * (1000 / OBJECTS_AP) / cdp->fps ()));
				wrap_moveBy (0, -2);		// To avoid getting "caught" again right away
			}

			setYVelocity(yVelocity() + acc);

			fuel -= fuel_con;			
		}
	}
}

/** Signals we are going down. See up () for details. */
void Taxi::down(double acc)
{
	if (ready && !landed) {
		if (fuel > 0) {

			flying_ud->setSequence(flying_down.anim[lr]);
			if (flying_ud_frame >= flying_down.frames) {
				flying_ud_frame = 0;
			}
			flying_ud->move(x(), y());
			flying_ud->setFrame(flying_ud_frame++);
			flying_ud->show();

			setYVelocity(yVelocity() + acc);

			fuel -= fuel_con;
		}
	}
}

/** See above */
void Taxi::left(double acc)
{
	if (ready && !(flaps_block && flaps_state) && !landed) {
		if (fuel > 0) {

			if (lr != 0) {
				if (flaps_avail) {
					flaps->setSequence (cflaps_anim->anim[0]);
				}
				setSequence (current_anim->anim[0]);
				lr = 0;
			}

			flying_lr->setSequence(flying_left.anim[0]);
			if (flying_lr_frame >= flying_left.frames) {
				flying_lr_frame = 0;
			}
			flying_lr->move(x(), y());
			flying_lr->setFrame(flying_lr_frame++);
			flying_lr->show();

			setXVelocity(xVelocity() + acc);

			fuel -= fuel_con;
		}
	}
}

/** See above */
void Taxi::right(double acc)
{
	if (ready && !(flaps_block && flaps_state) && !landed) {
		if (fuel > 0) {

			if (lr != 1) {
				if (flaps_avail) {
					flaps->setSequence (cflaps_anim->anim[1]);
				}
				setSequence (current_anim->anim[1]);
				lr = 1;
			}

			flying_lr->setSequence(flying_right.anim[1]);
			if (flying_lr_frame >= flying_right.frames) {
				flying_lr_frame = 0;
			}
			flying_lr->move(x(), y());
			flying_lr->setFrame(flying_lr_frame++);
			flying_lr->show();

			setXVelocity(xVelocity() + acc);

			fuel -= fuel_con;
		}
	}
}

/** This function reads a taxi-description ( specs for the animations ) from a file.
read_taxi is only called by the constructor. */
void Taxi::read_taxi(const QString & file)
{

	QDomElement e;				// just some element
	QDomNode n;					// just some node
	QDomNodeList list;			// just a list of nodes

	cdp->init_debug(file, false);

	QDomElement xdocElement = cdp->open_config_file(file);

	author = cdp->get_string_attribute("author", xdocElement, i18n("Anonymous"), 3);
	title = cdp->get_string_attribute("title", xdocElement, i18n("Unnamed"), 3);

	e = cdp->get_element("teleporting", xdocElement, 0);	// find the teleporting-animation
	read_animation(&teleporting, e);

	e = cdp->get_element("still", xdocElement, 0);	// find the standing-still-image/animation
	read_animation(&still, e);
	still.continuous = true;

	e = cdp->get_element("idle", xdocElement, 0);	// find the idle_animations-section
	list = cdp->get_node_list("data", e, 1, -1, 1, 0);	// get all idle-animations

	num_idle_animations = 0;
	idle = new Anim[list.length()];
	n = list.item(0);

	while (!n.isNull()) {
		e = n.toElement();

		read_animation(&idle[num_idle_animations], e);

		idle[num_idle_animations].repetitions = cdp->get_int_attribute("repetitions", e, 1, 10000, 1, 2);

		n = n.nextSibling();
		num_idle_animations++;
	}

	if (!(cdp->for_real () || cdp->thorough_check ())) {
		return;		// We don't need all the other stuff for the start-screen
	}

	e = cdp->get_element("fuel", xdocElement, 0);	// find the fuel-section
	max_fuel = cdp->get_int_attribute("max", e, 50, 64000, 800, 2);
	fuel = cdp->get_int_attribute("initial", e, 50, max_fuel, max_fuel, 2);

	e = cdp->get_element("flying_left", xdocElement, 0);	// find the flying_left-animation
	read_animation(&flying_left, e);

	e = cdp->get_element("flying_right", xdocElement, 0);	// find the flying_right-animation
	read_animation(&flying_right, e);

	e = cdp->get_element("flying_up", xdocElement, 0);	// find the flying_up-animation
	read_animation(&flying_up, e);

	e = cdp->get_element("flying_down", xdocElement, 0);	// find the flying_down-animation
	read_animation(&flying_down, e);

	e = cdp->get_element("landing_flaps", xdocElement, 2);
	setSequence(still.anim[lr]);
	if (e.hasAttribute("offset_x1_left") || e.hasAttribute("offset_x1_right")) {
		landing_flaps_off_x1[0] = cdp->get_int_attribute("offset_x1_left", e, -20, width() + 20, 0, 2);
		landing_flaps_off_x1[1] = cdp->get_int_attribute("offset_x1_right", e, -20, width() + 20, 0, 2);
	} else {
		landing_flaps_off_x1[0] = landing_flaps_off_x1[1] =
			cdp->get_int_attribute("offset_x1", e, -20, width() + 20, 0, 2);
	}
	if (e.hasAttribute("offset_x2_left") || e.hasAttribute("offset_x2_right")) {
		landing_flaps_off_x2[0] = cdp->get_int_attribute("offset_x2_left", e, -20, width() + 20, width(), 2);
		landing_flaps_off_x2[1] = cdp->get_int_attribute("offset_x2_right", e, -20, width() + 20, width(), 2);
	} else {
		landing_flaps_off_x2[0] = landing_flaps_off_x2[1] =
			cdp->get_int_attribute("offset_x2", e, -20, width() + 20, width(), 2);
	}

	if ((flaps_avail = cdp->get_bool_attribute("moveable", e, false, 3))) {
		read_animation(&flaps_anim, e);
		flaps->setSequence(flaps_anim.anim[lr]);
		flaps_block = cdp->get_bool_attribute("blocking", e, false, 3);
		cflaps_frame = flaps_anim.frames -1;
	} else {
		flaps_block = false;
		flaps_state = 2;		// if flaps are not moveable, they are always out.
	}

}

/** Stops the currently running flying animation, if any (called from Taxipilot::do_chores ()) */
void Taxi::stop_flying_animation()
{
	flying_lr->hide();
	flying_ud->hide();
}

  /** Takes care of teleportation (appearing, disappearing animations), setting new position
and speed. If appear_only is given, does not do disappearing animation. */
void Taxi::teleport_to(int x, int y, double speedfactor_x, double speedfactor_y, bool appear_only, Teleporter *dynamic)
{

	if (landed) platform_p->detach_taxi ();
	ready = landed = false;

	if (!flaps_state) {			// make sure, flaps show up correctly after teleporting
		flaps_prev_state = false;
	} else if (flaps_state == 2) {
		flaps_prev_state = true;
	} else {
		if (flaps_anim.direction > 0) {
			flaps_prev_state = false;
		} else {
			flaps_prev_state = true;
		}
	}

	if (dynamic) {
		dynamic_teleporter = dynamic;
		dynamic_teleport = true;
	} else {
		dynamic_teleport = false;
		new_x = x - off_x;
		new_y = y - off_y;
	}

	new_speed_x = xVelocity() * speedfactor_x;
	new_speed_y = yVelocity() * speedfactor_y;
	setVelocity(0, 0);

	if (idle_timer != -1) {
		tm->killTimer(idle_timer);
		idle_timer = -1;
	}

	if (appear_only) {
		reappear ();
	} else {
		do_animation (&teleporting, &Taxi::reappear);
		do_animation (&flaps_anim, &Taxi::flaps_in, false, true);
	}

}

/** Returns the current fuel level in percent of the taxi (Used by Taxipilot to set the
FuelDial) */
int Taxi::fuel_state()
{
	int return_val;
	return_val = (int) (((double) fuel / max_fuel) * 100);
	if (return_val < 10) {
		fuel_low = true;
		if (fuel <= 0) {
			fuel_out = true;
		}
	} else {
		fuel_low = fuel_out = false;
	}
	return (return_val);
}

/** Increases the taxi's fuel supply by step (max up to max_fuel). Returns actual
increase in fuel (so that's 0 if taxi is already full). */
double Taxi::fuel_up(int step)
{
	double prev_fuel;				// a dummy storing the previous fuel state

	if (step <= 0)
		return (0);				// no negative fueling, please

	prev_fuel = fuel;
	fuel += (step * fuel_con);
	if (fuel > max_fuel)
		fuel = max_fuel;
	return (fuel - prev_fuel);
}

/** Toggles the taxis landing-flaps */
void Taxi::toggle_flaps()
{

	if (flaps_avail && ready) {	// if not, we simply ignore this
		if (flaps_state == 1) {	// flaps are already moving some way
			if (cflaps_anim->direction == -1) {		// so we need to reverse this
				do_animation (&flaps_anim, &Taxi::flaps_in, false, true);
			} else {
				do_animation (&flaps_anim, &Taxi::flaps_out, true, true);
			}
		} else {
			if (flaps_state == 0) {	// flaps are in, going out
				do_animation (&flaps_anim, &Taxi::flaps_out, true, true);
			} else {			// flaps are out, going in
				if (landed) {
					platform_p->detach_taxi ();
					landed = false;
				}
				do_animation (&flaps_anim, &Taxi::flaps_in, false, true);
			}
		}
	}

}

/** Repositions the taxi. This also takes care of repositioning the landing flaps. */
void Taxi::reposition(int x, int y, double xv, double yv, bool align_bottom)
{

	int temp_off_y;

	if (align_bottom) {
		temp_off_y = double_off_y;
	} else {
		temp_off_y = off_y;
	}

	if (x < 0) {				// position parameter = -1 means don't change position.
		if (y >= 0) {
			moveBy(0, (y - temp_off_y) - Taxi::y());
			flaps->moveBy(0, (y - temp_off_y) - flaps->y());
		}
	} else {
		if (y < 0) {
			moveBy(x - Taxi::x(), 0);
			flaps->moveBy(x - flaps->x(), 0);
		} else {
			move(x, (y - temp_off_y));
			flaps->move(x, (y - temp_off_y));
		}
	}

	setVelocity(xv, yv);
}

/** Tells the taxi it crashed. Does or will do stuff
like setting of an appropriate animation,
sound and other things to do. */
void Taxi::crash()
{
	setVelocity(0, 0);
	do_animation (&flaps_anim, &Taxi::flaps_in, false, true);	
	if (landed) {
		landed = false;
		platform_p->detach_taxi ();
	}
}

/** Returns the (absolute) x-position the left landing-flaps is to be found at */
int Taxi::landing_flaps_x1()
{
	return ((int) x() + landing_flaps_off_x1[lr]);
}

/** Returns the (absolute) x-position the right landing-flap is to be found at */
int Taxi::landing_flaps_x2()
{
	return ((int) x() + landing_flaps_off_x2[lr]);
}

/** Returns the (absolute) y-position the landing-flaps are to be found at */
int Taxi::landing_flaps_y(){
	if (!flaps_avail) return (BOTTOMEDGE());
	return (flaps->boundingRect().bottom());
}


/** This function is only used by read_taxi, and takes care of reading in
(direction-specific or non-specific) animations. */
void Taxi::read_animation(Anim * anim, QDomElement e)
{
	anim->frames = cdp->get_int_attribute("frames", e, 1, 10000, 1, 3);
	if (e.hasAttribute("file_pattern_left") || e.hasAttribute("file_pattern_right")) {
		anim->anim[0] = new QCanvasPixmapArray(cdp->
											   prefix_check_animation(cdp->
																	  get_string_attribute("file_pattern_left", e, "",
																						   0), anim->frames),
											   anim->frames);
		anim->anim[1] =
			new QCanvasPixmapArray(cdp->
								   prefix_check_animation(cdp->get_string_attribute("file_pattern_right", e, "", 0),
														  anim->frames), anim->frames);
	} else {
		anim->anim[0] = anim->anim[1] =
			new QCanvasPixmapArray(cdp->
								   prefix_check_animation(cdp->get_string_attribute("file_pattern", e, "", 0),
														  anim->frames), anim->frames);
	}

	anim->frame_period = cdp->get_int_attribute("anim_frame_period", e, 5, 1000, 50, 3);
	
	// Initialization only:
	anim->continuous = false;
	anim->repetitions = 1;

}

/** A wrapper for moveBy, taking care of the flaps if appropriate */
void Taxi::wrap_moveBy (double delta_x, double delta_y){
	moveBy (delta_x, delta_y);
	flaps->moveBy (delta_x, delta_y);
}

/** Adjusts the taxi to a fractional position, so it doesn't wiggle on a moving platform */
void Taxi::adjust (double fraction_x, double fraction_y) {
	move (((int) x()) + fraction_x, ((int) y()) + fraction_y);
	flaps->move (((int) x()) + fraction_x, ((int) y()) + fraction_y);
}

/** Sets of the animation anim for this taxi or it's flaps,  and triggers
a function-call when completed. Reverse means: play in reverse order; flaps means, the animation is for the flaps */
void Taxi::do_animation (Anim * anim, void (Taxi::*call_back) (), bool reverse, bool flaps) {

	if (flaps) {
		if (flaps_avail) {				// no need to bother otherwise
			cflaps_anim = anim;
			flaps_call_back = call_back;
			if (flaps_anim_timer != -1) {
				tm->killTimer (flaps_anim_timer);
			}	
			flaps_anim_timer = tm->startTimer (this, cflaps_anim->frame_period);
			flaps_state = 1;
			Taxi::flaps->show ();
		}
	} else {
		current_anim = anim;
		taxi_call_back = call_back;
		repetition = 1;
		if (anim_timer != -1) {
			tm->killTimer (anim_timer);
		}	
		anim_timer = tm->startTimer (this, current_anim->frame_period);
	}		

	if (reverse) {
		anim->direction = -1;
		if (!flaps) {
			ctaxi_frame = anim->frames -1;
		}
	} else {
		anim->direction = +1;
		if (!flaps) {
			ctaxi_frame = 0;
		}
	}

	if (flaps) {
		if (flaps_avail) {
			Taxi::flaps->setFrame (0);		// so as not to crash next command
			Taxi::flaps->setSequence (cflaps_anim->anim[lr]);
			Taxi::flaps->setFrame (cflaps_frame);
		}
	} else {
		setFrame (0);		// so as not to crash next command
		setSequence (current_anim->anim[lr]);
		setFrame (ctaxi_frame);
	}

}

/** Stuff to do when the flaps just finished going out */
void Taxi::flaps_out(){
	flaps_state = 2;
}

/** Stuff to do when flaps have just finished going in */
void Taxi::flaps_in (){
	flaps_state = 0;
	flaps->hide();
}

/** Taxi is to reappear (at global new_x, new_y) */
void Taxi::reappear (){

	if (dynamic_teleport) {
		new_x = dynamic_teleporter->destX() - off_x;
		new_y = dynamic_teleporter->destY() - off_y;
	}

	move(new_x, new_y);
	flaps->move(new_x, new_y);

	do_animation (&teleporting, &Taxi::done_appearing, true);

	if (flaps_avail && flaps_prev_state) {
		do_animation (&flaps_anim, &Taxi::flaps_out, true, true);
	}

}

/** Starts idle_timer (not idle_animation) */
void Taxi::go_idle (){
	if (idle_timer != -1) {
		tm->killTimer (idle_timer);		//just in case
	}

	idle_timer = tm->startTimer (this, (rand() % 3000) + 2000);

	do_animation (&still, &Taxi::do_nothing);         // this is a continuous animation. no call_back needed
}

/** Does nothing */
void Taxi::do_nothing(){
}

/** Stuff to do when done appearing */
void Taxi::done_appearing (){

	ready = true;	// we're ready to go
	setVelocity(new_speed_x, new_speed_y);
	go_idle ();

}

/** Takes care of deleting the one or two QCanvasPixmapArrays associated with this
animation */
void Taxi::delete_animation (Anim &anim){
	if (anim.anim[0] == anim.anim[1]) {	// both point to the same mem. delete only one!
	} else {
		delete anim.anim[1];
	}
	delete anim.anim[0];
}

/** The config has changed. Recalculate. */
void Taxi::config_changed(){
	fuel_con = 20 / (double) cdp->cps ();
}

/** Mostly everyting will be done inside: timing for all animations */
void Taxi::timerTick(int id)
{

	if (id == idle_timer) {	// this only sets off the actual animation
		tm->killTimer (idle_timer);
		idle_timer = -1;
		do_animation (&idle[(rand() % num_idle_animations)], &Taxi::go_idle);
	} else if (id == anim_timer) {
		if (current_anim->direction > 0) {
			if (ctaxi_frame < (current_anim->frames-1)) {
				setFrame (++ctaxi_frame);
			} else if (current_anim->continuous) {
				setFrame (ctaxi_frame = 0);				
			} else if (repetition < current_anim->repetitions) {
				setFrame (ctaxi_frame = 0);
				repetition++;
			} else {
				tm->killTimer (anim_timer);
				anim_timer = -1;
				(this->*taxi_call_back) ();
			}
		} else if (current_anim->direction < 0) {
			if (ctaxi_frame > 0) {
				setFrame (--ctaxi_frame);
			} else if (current_anim->continuous) {
				setFrame (ctaxi_frame = (current_anim->frames -1));
			} else if (repetition < current_anim->repetitions) {
				setFrame (ctaxi_frame = (current_anim->frames -1));
				repetition++;
			} else {
				tm->killTimer (anim_timer);
				anim_timer = -1;
				(this->*taxi_call_back) ();
			}
		}
	} else if (id == flaps_anim_timer) {
		if (cflaps_anim->direction > 0) {
			if (cflaps_frame < (cflaps_anim->frames-1)) {
				flaps->setFrame (++cflaps_frame);
			} else {                             // flaps' animations are never continous or repetitous
				tm->killTimer (flaps_anim_timer);
				flaps_anim_timer = -1;
				(this->*flaps_call_back) ();
			}
		} else if (cflaps_anim->direction < 0) {
			if (cflaps_frame > 0) {
				flaps->setFrame (--cflaps_frame);
			} else {
				tm->killTimer (flaps_anim_timer);
				flaps_anim_timer = -1;
				(this->*flaps_call_back) ();
			}
		}
	}
}

/** Reimplementation of the advance function to make sure that the flaps get moved
accordingly. */
void Taxi::advance (int phase){
	if (phase == 1) {
		moveBy (xVelocity (), yVelocity ());
		flaps->moveBy (xVelocity (), yVelocity ());
	}
}

/** Tell the taxi it has landed on the specified paltform */
void Taxi::land (Platform *on) {
	landed = true;
	platform_p = on;
	platform_p->attach_taxi (this);
}