/***************************************************************************
                          cdp.h  -  description
                             -------------------
    begin                : Sun Aug 26 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CDP_H
#define CDP_H

/**Common debugging and prefixing class.
Takes care of file-operations that are needed in several of the other classes, most prominently of
prefixing paths and checking for existance of files. Also covers the debugging functionality.
  *@author Thomas Friedrichsmeier
  */

#include <qobject.h>
#include <qmap.h>

class QString;
class QDomElement;
class QDomNode;
class QDomNodeList;
class QWidget;
class Taxipilot;

class Cdp {
  public:
	Cdp(Taxipilot * parent);
	~Cdp();
/** Initializes the function that will take care of debugging messages and potential errors. in_file
is the filename which is currently to be checked. If exit_on_error == false, does not kill
the app on errors. This is useful, if even critical errors are not necessarly fatal, such as
when reading the mission.  */
	void init_debug(QString in_file, bool exit_on_error);
/** Appends an arbitrary string to the debug or error messages. If level == 0, this is a critical
error and the program will abort in do_debug (). If *in_node is specified a backtrace is shown */
	void debug_msg(QString message, int level, QDomNode const *in_node = 0);
/** Add a directory prefix to a filename if appropriate (i.e. not an absolute path) and checks,
whether the file exists. image specifies, whether this is an image (use image-prefix instead of
data-prefix) */
	QString prefix_check(QString filename, bool image);
/** Add a directory prefix to a filename_pattern if appropriate (i.e. not an absolute path) and checks,
whether all the files needed fro the animation exist */
	QString prefix_check_animation(QString filename_pattern, int frames);
/** prefix of the resource-dir, where the configuration-files are to be found */
	QString data_prefix;
/** Alternative prefix to the configuration-files */
	QString alt_data_prefix;
/** prefix of the resource-dir, where the image-files are to be found */
	QString image_prefix;
/** alternative prefix to the image-files */
	QString alt_image_prefix;
/** Checks for the existence of a certain integer XML-attribute in element. If given, returns its
value. If not given or not an int or not between min and max, return def and writes debugging message of level level */
	int get_int_attribute(QString attribute, const QDomElement & element, int min, int max, int def, int level);
  /** Checks for the existence of a certain double XML-attribute in element. If given, returns its
value. If not given or not a double number or not between min and max, returns def and writes debugging message of level level */
	double get_double_attribute(QString attribute, const QDomElement & element, double min, double max, double def,
								int level);
  /** Checks for the existence of a certain string XML-attribute in element. If given, returns its
value. If not given or not existent, returns def and writes debugging message of level level */
	QString get_string_attribute(QString attribute, const QDomElement & element, QString def, int level, bool trans=true);
  /** Checks for the existence of a certain bool XML-attribute in element. If given, returns its
value. If not given or not existent, returns def and writes debugging message of level level */
	bool get_bool_attribute(QString attribute, const QDomElement & element, bool def, int level);
  /** Opens the config-file file, parses it into XML, and returns document element */
	QDomElement open_config_file(QString const &file);
  /** Gets all child-nodes named name of element parent, checking whether there are at least
min, at most max nodes and min_if_any nodes if any, otherwise stating an error at
level level. (if name = "", don't check for name, but return all child nodes, if max < 0, there is no maximum) */
	QDomNodeList get_node_list(QString const &name, QDomElement const &parent, int min, int max, int min_if_any,
							   int level);
  /** Get the child-element named name of element parent. If no such element exists
	or there are more than one element, outputs error at level level. */
	QDomElement get_element(QString const &name, QDomElement const &parent, int level);
/** Returns inverse of errors_occurred, displaying a Dialog if errors_occured */
	bool ok();
/** Checks for the existence of a certain string XML-attribute in element. If given, checks
whether the attribute's value matches any of possible_values, which is a ';' seperated
list of allowed strings. If not given, not valid or not existent, returns def and writes
debugging message of level level. */
	QString get_multi_choice_attribute(QString attribute, const QDomElement & element, QString possible_values, QString def, int level);
/** Checks, whether the given element is really of type/tagName "name" and returns false otherwise
+ outputs error-message at level level */
	bool verify_tag (QString const &name, QDomElement const & element, int level);
/** Returns true, if the commandline-option highlight was set, false otherwise */
	bool draw_highlight ();
/** Returns true, if the commandline-option thorough_check was set, false otherwise */
	bool thorough_check();
  /** Returns whether we're reading for real (not just for the credits/startscreen) */
  bool for_real ();
  /** Sets whether reading is going to be for real */
  void set_for_real (bool real);
  /** Return the default mission path */
  QString defaultPath ();
  /** Return the startup-Mission */
  QString startupMission ();
  /** Return calculations-per-second setting */
  int cps ();
  /** Return frames-per-second setting */
  int fps ();
  /** Signals, that the mission mission has just been opened (and therefore is the one
last chosen) */
  void setLastLoaded (QString &mission);
  /** Saves the current config (and emits a config_changed ()) */
  void set_new_config ();
  /** Like ok, but does not yet give feedback to the user, if errors occured. Used in
constructors to prevent crashes, while still making it possible to continue parsing. */
  bool isOk ();

  private:
	bool draw_highlights;
	bool opt_thorough_check;
	bool read_for_real;
/** The filenmae of the currently debugged config file */
	 QString debug_in_file;
/** The debug_level */
	int debug_level;
/** Traces back the nodes in the current file, so as to display them in the debugging-output */
	QString backtrace(QDomNode const &node);
/** If die_on_error == false, does not kill the app on errors. This is useful, if
even critical errors are not necessarly fatal, such as when reading the mission */
	bool die_on_error;
/** Makes sense only when die_on_error == false. Keeps track of critical errors*/
	bool errors_occurred;
/** Stores a pointer to the parent */
	Taxipilot *tp;
/** Stores translation-table for the config-file currently opened */
	QMap <QString, QString> translations;

friend class Preferences;

/** Preferences */
	QString pref_DefaultMission;
/** Preferences */
	QString pref_LastMission;
/** Preferences */
	QString pref_SelectedMission;
/** Preferences */
	bool pref_useDefaultMission;	
/** Preferences */
	bool pref_useLastMission;
/** Preferences */
	bool pref_useSelectedMission;
/** Preferences */
	int pref_fps;
/** Preferences */
	int pref_cps;
};

#endif
