/***************************************************************************
                          platform.h  -  description
                             -------------------
    begin                : Tue Nov 6 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PLATFORM_H
#define PLATFORM_H

/**A generalized platform, which may either be a regular
platform (i.e. one which is part of the static image of the level)
or a moving object.
  *@author Thomas Friedrichsmeier
  */

#include <qstring.h>

class Taxipilot;
class Taxi;
class Passenger;
class MovingObject;
class QDomElement;
class Cdp;
class QCanvasLine;
class Level;
class Event;

class Platform {
public:
	Platform(const QDomElement *description, Level *parent, bool draw_lines);
	~Platform();
/** Fulfills the promise (assigns the promised object to this platform) */
	void promise_fulfilled(MovingObject * obj);
  /** Current y-position of the actual platform (i.e. taking into account any offset) */
  int current_plat_y ();
  /** Checks whether the given taxi is landed on this platform and if so, takes appropriate
actions. */
  bool check_landed (Taxi *taxi_p);
  /** Current x-position of the actual platform's passenger_base_x (i.e. taking into account any offset) */
  int current_plat_base_x();
  /** Current x2-position of the actual platform (i.e. taking into account any offset) */
  int current_plat_x2();
  /** Current x1-position of the actual platform (i.e. taking into account any offset) */
  int current_plat_x1();
/** Return current x-velocity of platform (if it is an object) */
	double current_plat_vx ();
/** Return current y-velocity of platform (if it is an object) */
	double current_plat_vy ();
/** Detaches the taxi when it takes off */
	void detach_taxi();
/** A taxi has landed on top of this paltform (moving platforms will use the pointer to
move the taxi with them) */
	void attach_taxi(Taxi * taxi_p);
/** Tells the platform, whether it is currently visible */
	void set_visible(bool visibility);
/** Tells the platform, it has moved (so it can then move the taxi / passenger) */
	void movedBy (double dx, double dy);
/** Same as above, but absolute positioning */
	void movedTo (double x, double y);
/** Drop a passenger on top of this paltform (moving platforms will use the pointer to
move the passenger with them) */
	void attach_passenger (Passenger *pass);
/** Detaches the passenger when it has boarded the taxi or disappeared */
	void detach_passenger();
friend class Taxipilot;
friend class Level;
private:
/** The pointer to the common debugging and prefixing class */
	Cdp *cdp;
/** Pointer to the parent TaxiPilot */
	Taxipilot *tp;
/** Pointer to the parent Level */
	Level *level;

/** Whether platform-lines will be drawn */
	bool draw_platform_lines;
/** Pointer to this platform's line (if any) */
	QCanvasLine *line;	

/** Whether this platform has been defined as an object, which will have to be connected */
	bool object_promised;
/** (redundant:) Whether the platform has a moving object */
	bool has_object;
/** The identification-string of the promised object */
	QString object_ident;
/** Pointer to the moving object */
	MovingObject *object;

/** Whether the platform is currently visible (can not be active otherwise) */
	bool visible;

/** static y-position or y-offset of Platform */
	int y;
/** static x-range or -offsets of Platform, x1 must be < x2 */
	int x1, x2;
/** Last known reference position of the object (Need to signal differences to the passenger) */
	double last_x, last_y;
/** is this a passenger platform? */
	bool accepts_passenger;
/** static x-position (center) or offset where passenger will appear on this platform */
	int passenger_base_x;
/** price of fuel on this platform. */
	double fuel_price;
/** speed of fueling. 0 if no fuel available */
	int fuel_step;
/** Label of this platform. Will be numbers starting with one, unless explicitly set in the level-decpription file */
	QString label;

/** Whether the taxi is currently landed on this platform */
	bool taxi_ontop;
/** Pointer to a taxi landed on this platform */
	Taxi *taxi;

/** Whether a passenger is currently on top of this platform */
	bool passenger_ontop;
/** Pointer to a passenger on this platform */
	Passenger *passenger;

/** A timer used to advance any taxis or passengers on this platform (if it is a moving platform) */
	int advance_timer;

/** Events (arrive and leave are for passengers) */
	Event *land_event, *take_off_event, *arrive_event, *leave_event;

private: // Private methods
/** Read the platforms's description and creates it */
	void read_platform(const QDomElement * description);
};

#endif
