/***************************************************************************
                          passenger.h  -  description
                             -------------------
    begin                : Mon May 28 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PASSENGER_H
#define PASSENGER_H

#include <qcanvas.h>

#include "timedobject.h"

/**
  *@author Thomas Friedrichsmeier
  */

class Taxipilot;				// forward reference
class Cdp;
class QDomElement;
class QString;
class Platform;

/** Passenger is the most complete class of the project so far. It handles the entire behavior of
a passenger. In the constructor a number of animations are read. These will be handled inside the
timerEvent-function. Passenger communicates with Taxipilot through a number a signals ans slots,
mostly needed for boarding and leaving the taxi. What I'm currently wondering is, whether I should
define "static" passenger, instead of creating and deleting them over and over again. This shouldn't
be too hard to change, though. The TODO list for the code of Passenger is rather short: 1) add sound.
More importantly, however, we need more and better animation. Any artists out there? */
class Passenger: public QCanvasSprite, public TimedObject {
  public:
	Passenger(const QString & file, QCanvas * canvas, Taxipilot * parent);
	~Passenger();
/** Moves the passenger and the reference points by the given offset-values. Used
by moving platforms to move an attached passenger. */
	void move_passenger_by (double delta_x, double delta_y);
/** Initializes the passenger (use on=0 for start_screen-passengers) */
	void initialize (int base_x, int base_y, Platform *on, bool for_credits=false);
/** Discards this passenger (but does not delete it). Call initialize in oder to reuse the
passenger */
	void discard ();
  /** Adjusts the passenger to a fractional position, so it doesn't wiggle on a moving platform */
  void adjust(double fraction_x, double fraction_y);
/** impatience of the passenger: range 1 through 4. Used to determine a number of things such as
speed of tip-countdown */
	int impatience;
/** name of this passenger */
	QString title;
/** author(s) of this passenger */
	QString author;
friend class Taxipilot;
  private:
/** taxi has landed at taxi_x, we can start boarding now. */
	void pick_up(int taxi_x);
/** Timer ids */
	int idle_timer, boarding_timer, leaving_timer, animation_timer;
/** Current frame */
	int frame;
/** where does the sprite start out? */
	double sprite_base_x;
/** where is it heading? */
	double sprite_dest_x;
/** how fast is it heading (into which direction?) */
	double step;
/** set to true, when the passenger has been abandoned */
	bool got_abandoned;
/** Whether the passenger is sitting on a platform */
	bool on_platform;
/** The platform the passenger is sitting on */
	Platform *platform_p;
/** needed in order to reference the parent ( function-call to st->onActivePlatform ) */
	Taxipilot *tp;
/** The pointer to the common debugging and prefixing class */
	Cdp *cdp;
/** Whether we're here for real (not just for the startscreen) */
	bool for_real;
/** This function reads a passenger-description ( specs for the animations ) from a file. It's
supposed to be XML-based. Only I don't know a thing about XML. You might want to second-check the
code. Note that there are no security catches. You misspell a single filename -> probably you'll get
a crash, once it is needed. read_passenger is only called by the constructor. */
	void read_passenger(const QString & file);
/** The number of different idle_animations this passenger has. Must be at least one. */
	int num_idle_animations;
/** This struct describes animations */
	struct Anim {
/** The actual animation as used by QT */
		QCanvasPixmapArray *anim;
/** This is the number of frames in the animation */
		int frames;
/** Frame-period for this animation */
		int frame_period;
/** Is the animation continuous (i.e. looping)? */
		bool continuous;
/** Simple things like waving made be repeated. */
		int repetitions;
	} still, teleporting, walking_right, walking_left, *walking, *current_anim, *idle;
/** The walking-speed in pixels per frame */
	double walking_speed;
/** This function is only used by read_passenger, and takes care of reading in
animations. */
	void read_animation(Anim * anim, const QDomElement * e);
/** Sets of the animation anim for this passenger,  and trigger
a function-call when completed. Reverse means: play in reverse order; dep_on_impatience means animation
speed depends on passenger impatience */
	void do_animation (Anim * anim, void (Passenger::*call_back) (), bool reverse=false, bool dep_on_impatience=false);
// The following needed for do_animation:
/** Current repetition-cycle */
	int repetition;
/** Pointer to the call-back-function */
	void (Passenger::*next_call_back) ();
/** +1 if animation is normal, -1 if it is reversed (for appearing) */
	int animation_direction;
/** Whether the taxipilot has already been notified, that we are on screen */
	bool emitted_im_here;
/** Function called when the passenger is ready teleporting into the taxi. */
	void boarded ();
/** Function called when the passenger has disappeared completely. */
	void done ();
/** Function called when the passenger as been delivered (and is done teleporting
from the taxi) and is to leave the platform. */
	void leave ();
/** Function called when the passenger has just appeared on the platform, or
when it had been abandoned and is back to its base-position. */
	void wait_for_pickup ();
/** Does nothing */
	void do_nothing ();
 /** You got me, where I wanted to go. Thanks */
	void got_delivered(int base_x, Platform *on);
  protected:				// Protected methods
/** Mostly everyting will be done inside: timing for all animations */
	void timerTick (int id);
};

#endif
