/***************************************************************************
                          passenger.cpp  -  description
                             -------------------
    begin                : Mon May 28 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"

#include <qwidget.h>
#include <qdom.h>
#include <qstring.h>

#include <stdlib.h>

#include "passenger.h"

#include "taxipilot.h"
#include "cdp.h"
#include "platform.h"


Passenger::Passenger(const QString & file, QCanvas * canvas, Taxipilot * parent):QCanvasSprite(0,
			  canvas)
{

	tp = parent;
	tm = parent->tm;
	cdp = parent->cdp;
	idle_timer = boarding_timer = leaving_timer =  animation_timer = -1;

	read_passenger(file);
	setZ (PASSENGER_Z);

	on_platform = false;
}

Passenger::~Passenger()
{
	tm->killTimers (this);
	discard ();

	delete still.anim;
	delete teleporting.anim;

	for (int i=0; i < num_idle_animations; i++) {
		delete idle[i].anim;
	}
	delete[]idle;

	if (cdp->for_real () || cdp->thorough_check ()) {
		delete walking_left.anim;
		delete walking_right.anim;
	}

}

/** Initializes the passenger (use on=0 for start_screen-passengers) */
void Passenger::initialize (int base_x, int base_y, Platform *on, bool for_credits) {

	if (!((on == 0) || for_credits)) {
		for_real = true;
		platform_p = on;
		on_platform = true;
		platform_p->attach_passenger (this);
	} else {
		for_real = false;
		on_platform = false;
	}

       	impatience = 1 + (rand() % 4);

	emitted_im_here = false;

	setSequence (still.anim); // for width and heigth
	setFrame (0);
	move(((sprite_base_x = base_x - (width() / 2))), (base_y - height()));

	if (for_credits) {
		do_animation (&still, &Passenger::do_nothing);
		idle_timer = tm->startTimer(this, (int) (5000 / impatience));
	} else {
		do_animation (&teleporting, &Passenger::wait_for_pickup, true);
		show();
	}

}

/** Discards this passenger (but does not delete it). Call initialize in oder to reuse the
passenger */
void Passenger::discard (){

	hide ();
	if (on_platform) platform_p->detach_passenger ();
	on_platform = false;

	tm->killTimers (this);
	idle_timer = boarding_timer = leaving_timer =  animation_timer = -1;

}

/** No descriptions */
void Passenger::pick_up(int taxi_x)
{

	sprite_dest_x = taxi_x - (int) (width() / 2);
	if (LEFTEDGE() < sprite_dest_x) {
		step = walking_speed;				// never do steps > 1! You might miss your destination.
		walking = &walking_right;
	} else if (LEFTEDGE() > sprite_dest_x) {
		step = -walking_speed;
		walking = &walking_left;
	} else {					// the way to be on spot here (without crashing) is the passenger appearing right where we already are
		step = 0;
	}
	do_animation (walking, &Passenger::do_nothing, false, true);

	if (idle_timer != -1) {
		tm->killTimer(idle_timer);	// stop any idle animation running
		idle_timer = -1;
	}
	if (boarding_timer == -1) {	// in case we got abandoned and timer is still running
		boarding_timer = tm->startTimer(this, (int) (PASSENGER_AP / impatience));	// start boarding-animation
	}
	
	got_abandoned = false;
}

/** You got me, where I wanted to go. Thanks */
void Passenger::got_delivered(int base_x, Platform *on)
{

	platform_p = on;
	on_platform = true;
	platform_p->attach_passenger (this);

	sprite_base_x = base_x - (width() / 2);
	move(sprite_base_x, platform_p->current_plat_y() - height());
	show();

	sprite_dest_x = platform_p->current_plat_base_x ()- (width() / 2);

	if (sprite_base_x < sprite_dest_x) {
		step = 0.6;				// never do steps > 1! You might miss your destination.
		walking = &walking_right;
	} else if (sprite_dest_x < sprite_base_x) {
		step = -0.6;
		walking = &walking_left;
	} else {					// who knows...
		step = 0;
	}

	// before we can leave, we have to appear, though
	do_animation (&teleporting, &Passenger::leave, true);
}

void Passenger::read_passenger(const QString & file)
{								// initialization only

	QDomElement e;				// just some element
	QDomNode n;					// just some node
	QDomNodeList list;			// just a list of nodes

	cdp->init_debug(file, true);

	QDomElement xdocElement = cdp->open_config_file(file);

	author = cdp->get_string_attribute("author", xdocElement, i18n("Anonymous"), 3);
	title = cdp->get_string_attribute("title", xdocElement, i18n("Unnamed"), 3);

	e = cdp->get_element("idle", xdocElement, 0);	// find the idle_animations-section
	list = cdp->get_node_list("data", e, 1, -1, 1, 0);	// get all idle-animations

	num_idle_animations = 0;
	idle = new Anim [list.length()];
	n = list.item(0);

	while (!n.isNull()) {
		e = n.toElement();

		read_animation (&idle[num_idle_animations], &e);
		idle[num_idle_animations].repetitions = cdp->get_int_attribute("repetitions", e, 1, 10000, 1, 2);

		n = n.nextSibling();
		num_idle_animations++;
	}

	e = cdp->get_element("teleporting", xdocElement, 0);	// find the teleporting-animation
	read_animation (&teleporting, &e);

	e = cdp->get_element("still", xdocElement, 0);	// find the standing-still-image
	// not sure, we'll ever use different frames on this, though
	read_animation (&still, &e);
	still.continuous = true;

	// the following are only needed for the actual game
	if (cdp->for_real() || cdp->thorough_check ()) {
		e = cdp->get_element("walking", xdocElement, 2);
		walking_speed = cdp->get_double_attribute ("speed", e, 0.05, 5, 0.6, 3);

		e = cdp->get_element("walking_left", xdocElement, 0);	// find the walking_left-animation
		read_animation (&walking_left, &e);
		walking_left.continuous = true;

		e = cdp->get_element("walking_right", xdocElement, 0);	// find the walking_right-animation
		read_animation (&walking_right, &e);
		walking_right.continuous = true;
       }
}

/** Moves the passenger and the reference points by the given offset-values. Used
by moving platforms to move an attached passenger. */
void Passenger::move_passenger_by (double delta_x, double delta_y){

	sprite_base_x += delta_x;
	sprite_dest_x += delta_x;
	// passenger does not change y-position on its own and therefore does never store it

	moveBy (delta_x, delta_y);

}

/** This function is only used by read_passenger, and takes care of reading in
animations. */
void Passenger::read_animation(Anim * anim, const QDomElement * e) {

	anim->frames = cdp->get_int_attribute("frames", *e, 1, 10000, 1, 2);
	anim->anim = new QCanvasPixmapArray(cdp->prefix_check_animation(cdp->get_string_attribute("file_pattern", *e, "", 0),
													  anim->frames), anim->frames);
	anim->frame_period = cdp->get_int_attribute("anim_frame_period", *e, 5, 1000, PASSENGER_AP, 3);
	
	// Initialization only:
	anim->continuous = false;
	anim->repetitions = 1;

}

/** Sets of the animation anim for this passenger,  and trigger
a function-call when completed. Reverse means: play in reverse order; dep_on_impatience means animation
speed depends on passenger impatience */
void Passenger::do_animation (Anim * anim, void (Passenger::*call_back) (), bool reverse, bool dep_on_impatience) {
	current_anim = anim;
	this->next_call_back = call_back;
	if (reverse) {
		animation_direction = -1;
		frame = anim->frames -1;
	} else {
		animation_direction = +1;
		frame = 0;
	}
	repetition = 1;
	if (animation_timer != -1) {
		tm->killTimer (animation_timer);
	}	
	setFrame (0);		// so as not to crash next command
	setSequence (current_anim->anim);
	setFrame (frame);
	if (!dep_on_impatience) {
		animation_timer = tm->startTimer (this, current_anim->frame_period);
	} else {
		animation_timer = tm->startTimer (this, current_anim->frame_period / impatience);		
	}
}

/** Function called when the passenger has just appeared on the platform, or
when it had been abandoned and is back to its base-position. */
void Passenger::wait_for_pickup (){

	do_animation (&still, &Passenger::do_nothing);         // this is a continuous animation. no call_back needed
	idle_timer = tm->startTimer(this, (rand() % 1000) + (int) (4000 / impatience));
	if (!emitted_im_here) {
		if (for_real) tp->passenger_ready ();
		emitted_im_here = true;
	}

}

/** Function called when the passenger has been delivered (and is done teleporting
from the taxi) and is to leave the platform. */
void Passenger::leave (){

	do_animation (walking, &Passenger::do_nothing, false, true);
	leaving_timer = tm->startTimer(this, (int) (PASSENGER_AP / impatience));

}

/** Function called when the passenger has disappeared completely. */
void Passenger::done (){

	hide ();
	tp->delivery_complete ();
	// passenger will subsequently get discarded -> no call to platform_p->detach ();
	
}

/** Function called when the passenger is ready teleporting into the taxi. */
void Passenger::boarded (){

	hide();
	platform_p->detach_passenger();
	on_platform = false;
	tp->picked_up ();

}

/** Does nothing */
void Passenger::do_nothing (){
}

void Passenger::timerTick (int id) {

	if (id == boarding_timer) {
		moveBy(step, 0);
		if (tp->onActivePlatform()) {
			if ((LEFTEDGE() - sprite_dest_x) * step >= 0) {

				tm->killTimer(boarding_timer);
				boarding_timer = -1;

				// teleport to vanish inside taxi
				do_animation (&teleporting, &Passenger::boarded);
			}
		} else {			// Hey, where did you go?!
			if (!got_abandoned) {	// we've only just been abandoned
				sprite_dest_x = sprite_base_x;	// head back for base
				step = (step * -1);

				if (step > 0) {                         	// turn around
					walking = &walking_right;
				} else {
					walking = &walking_left;
				}
				do_animation (walking, &Passenger::do_nothing, false, true);
	
				got_abandoned = true;
				tp->passenger_abandoned ();
			} else {		// we've long since been abandoned
				moveBy(step, 0);
				if ((LEFTEDGE() - sprite_dest_x) * step >= 0) {
					tm->killTimer(boarding_timer);	// back to square 1
					boarding_timer = -1;
					wait_for_pickup ();
				}
			}
		}
	} else if (id == leaving_timer) {
		if ((LEFTEDGE() - sprite_dest_x) * step >= 0) {
			tm->killTimer(leaving_timer);
			leaving_timer = -1;

			// teleport to vanish
			do_animation (&teleporting, &Passenger::done);
		} else {
			moveBy(step, 0);
		}
	} else if (id == idle_timer) {	// this only sets off the actual animation
		tm->killTimer (idle_timer);
		do_animation (&idle[(rand() % num_idle_animations)], &Passenger::wait_for_pickup);
	} else if (id == animation_timer) {
		if (animation_direction > 0) {
			if (frame < (current_anim->frames-1)) {
				setFrame (++frame);
			} else if (current_anim->continuous) {
				setFrame (frame = 0);				
			} else if (repetition < current_anim->repetitions) {
				setFrame (frame = 0);
				repetition++;
			} else {
				tm->killTimer (animation_timer);
				animation_timer = -1;
				(this->*next_call_back) ();
			}
		} else if (animation_direction < 0) {
			if (frame > 0) {
				setFrame (--frame);
			} else if (current_anim->continuous) {
				setFrame (frame = (current_anim->frames -1));				
			} else if (repetition < current_anim->repetitions) {
				setFrame (frame = (current_anim->frames -1));
				repetition++;
			} else {
				tm->killTimer (animation_timer);
				animation_timer = -1;
				(this->*next_call_back) ();
			}
		}
	}
}

/** Adjusts the passenger to a fractional position, so it doesn't wiggle on a moving platform */
void Passenger::adjust(double fraction_x, double fraction_y){
	double old_x = x();
	move (((int) x()) + fraction_x, ((int) y()) + fraction_y);
	sprite_dest_x += x () - old_x;
	sprite_base_x += x () - old_x;
}
