/***************************************************************************
                          movingobject.cpp  -  description
                             -------------------
    begin                : Tue Oct 23 2001
    copyright            : (C) 2001, 2002 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "defines.h"

#include "movingobject.h"

#include <qdom.h>
#include <qstring.h>
#include <qlist.h>
#include <qbitmap.h>
#include <qpainter.h>

#include "taxipilot.h"
#include "level.h"
#include "cdp.h"
#include "platform.h"
#include "teleporter.h"
#include "conditional.h"
#include "state.h"
#include "level.h"
#include "event.h"

MovingObject::MovingObject(const QDomElement *description, Level *parent, QCanvas *canvas):QCanvasSprite(0, canvas)
{
	cdp = parent->cdp;
	tm = parent->tp->tm;
	level = parent;

	advance_timer = animation_timer = -1;
	setAnimated (false);

	read_object (description);

	current_segment = 0;
	frame = 0;
	step = 0;
	setSequence (path[0].anim);		// so other classes can already access width and height and stuff
	if (cdp->for_real ()) {
		advance_timer = tm->startTimer (this, OBJECTS_AP);         // sets everything off
		setVisible (last_visible = true);	// child-objects will assume visiblity, until told otherwise.
	}
}

MovingObject::~MovingObject()
{
        hide ();
	tm->killTimers (this);
	
	QList <QCanvasPixmapArray> deleted;		// to prevent multiple deletion of shared animations
	deleted.setAutoDelete (false);

	QList <Action> deleted_actions;
	deleted_actions.setAutoDelete (false);

	for (int i=0; i < segments; i++) {
		if (deleted.find (path[i].anim) == -1) {
			deleted.append (path[i].anim);
			delete path[i].anim;			
		}

		if (path[i].action) {
			if (deleted_actions.find (path[i].action) == -1) {
				deleted_actions.append (path[i].action);
				Action *action_a;
				Action *action_b;
				action_a = path[i].action;
				do {
					action_b = action_a->next;
					delete action_a;
					action_a = action_b;
				} while (action_b);
			}
		}
	}
	deleted.clear ();

	if (segments) {
		delete [] path;
	}
}

/** This function reads a moving-object-description ( specs for the animations ) from a
QDomNode (of a level-description) read_object is only called by the constructor. */
void MovingObject::read_object(const QDomElement *description)
{
	QCanvasPixmapArray *globalanim = 0;
	int global_anim_frames = 1;

	int frames_dl = 1; // debug-levels if frames / anim missing in individual segments
	int anim_dl = 0;

	int global_z;
	int global_frame_period;
	QDomElement e;
	QDomNodeList list;
	QDomNode n;
	QString dummy;
	bool looping;
	int max_segment;

	if (description->hasAttribute ("ident")) {
		has_ident = true;
		ident = cdp->get_string_attribute ("ident", *description, "#not given#", 1);
	} else {
		has_ident = false;
	}

	teleporter = center = platform = false;
	harmful_if_visible = true;

	QString classification = cdp->get_multi_choice_attribute ( "class", *description, "harmful;background;teleporter;harmful center;harmless center;platform", "harmful", 2 );
	if (classification == "harmful") {

	} else if (classification == "background") {
		harmful_if_visible = false;
	} else if (classification == "platform") {
		platform = true;
		harmful_if_visible = true;		// you can crash into platforms
	} else {
		if (classification == "teleporter") {
			teleporter = true;
			harmful_if_visible = false;		
		} else if (classification == "harmful center") {
			center = true;			
		} else if (classification == "harmless center") {
			harmful_if_visible = false;
			center = true;
		}
		if (!has_ident) {
			cdp->debug_msg (i18n ("Object classified as '") + classification + i18n ("teleporter but no ident given"), 1, description);
		}
	}
	global_z = cdp->get_int_attribute ( "z", *description, 0, 300, 150, 3 );
	if (description->hasAttribute ( "text" )) {
		global_anim_frames = 1;
		globalanim = create_text_qcpma ( *description );
		frames_dl = anim_dl = 3;
	} else {
		if (description->hasAttribute ("frames")) {
			global_anim_frames = cdp->get_int_attribute ( "frames", *description, 1, 10000, 1, 1 );
			frames_dl = 3;
		}
		if (description->hasAttribute ( "file_pattern" )) {
			globalanim = new QCanvasPixmapArray(cdp->prefix_check_animation(cdp->get_string_attribute("file_pattern", *description, "", 3),
																global_anim_frames), global_anim_frames);
			if (globalanim) anim_dl = 3;
		}
	}
	global_frame_period = cdp->get_int_attribute ( "anim_frame_period", *description, 5, 10000, 50, 3 );
	looping = cdp->get_bool_attribute ( "looping", *description, true, 2 );

	list = cdp->get_node_list ( "path", *description, 1, -1, 1, 1 );	// now get the path-segments

	segments = 0;

	double previous_vx, previous_vy;

	previous_vx = cdp->get_double_attribute ( "initial_vx", *description, 0, 1000, 0, 3 ) / ( (int) (1000 / OBJECTS_AP));
	previous_vy = cdp->get_double_attribute ( "initial_vy", *description, 0, 1000, 0, 3 ) / ( (int) (1000 / OBJECTS_AP));

	if ( looping ) {
		max_segment = list.length ();
		path = new PathSegment[list.length()];
	} else {
		max_segment = list.length ()*2;
		path = new PathSegment[list.length()*2];
	}

	n = list.item(0);

	while (!n.isNull()) {
		e = n.toElement();

		// first get segment-specific animation, if available
		path[segments].z = cdp->get_int_attribute ( "z", e, 0, 300, global_z, 3 );
		path[segments].timer_period = cdp->get_int_attribute ( "anim_frame_period", e, 5, 1000, global_frame_period, 3 );
		if (e.hasAttribute ( "text" )) {
			path[segments].frames = 1;
			path[segments].anim = create_text_qcpma ( *description );
		} else {
			path[segments].frames = cdp->get_int_attribute ( "frames", e, 1, 10000, global_anim_frames, frames_dl );
// TODO: Make sure, identical animations are only loaded once
			if (e.hasAttribute ( "file_pattern" )) {
				path[segments].anim = new QCanvasPixmapArray(cdp->prefix_check_animation(cdp->get_string_attribute("file_pattern", e, "", 3),
															path[segments].frames), path[segments].frames);
			} else {
				cdp->debug_msg ( "No animation specified. Assuming global animation (if any)", anim_dl, description );
				path[segments].anim = globalanim;
			}
		}

		path[segments].visible = cdp->get_bool_attribute ( "visible", e, true, 3 );

		// get action-stuff (things like halt/freeze/jump)
		path[segments].action = 0;
		QDomNode n_b;
		QDomElement e_b;
		n_b = e.firstChild ();
		bool first_action = true;

		while (!n_b.isNull ()) {
			e_b = n_b.toElement ();			
			QString tagName = e_b.tagName ();

			if ((tagName == "stop") || (tagName == "freeze") || (tagName == "trigger") || (tagName == "jump")) {
				Action *new_action = new Action;
				if (tagName == "stop") {
					new_action->stop = true;
					new_action->freeze = false;
					new_action->trigger = false;					
					new_action->jump = false;					
				} else if (tagName == "freeze") {
					new_action->stop = false;
					new_action->freeze = true;
					new_action->trigger = false;					
					new_action->jump = false;					
				} else if (tagName == "trigger") {
					new_action->stop = false;
					new_action->freeze = false;
					new_action->trigger = true;
					new_action->jump = false;					
					QString ev_dummy = cdp->get_string_attribute ("event", e_b, "NOT GIVEN", 1);
					new_action->event_to_trigger = level->resolve_event (ev_dummy);
				} else if (tagName == "jump") {
					new_action->stop = false;
					new_action->freeze = false;
					new_action->trigger = false;					
					new_action->jump = true;
					new_action->jump_target = 	cdp->get_int_attribute ("target", e_b, 1, max_segment, 1, 1) -1;	// start counting from 1 instead of 0
				}
				new_action->conditional = new Conditional (&cdp->get_element ("condition", e_b, 0), level);
				new_action->next = 0;

				if (!first_action) {		// find place to insert the action
					Action *action_dummy = path[segments].action;
					while (action_dummy->next) {
						action_dummy = action_dummy->next;
					}
					action_dummy->next = new_action;
				} else {
					 path[segments].action = new_action;
				}
				first_action = false;
			} else {
				cdp->debug_msg (i18n ("Tag '") + tagName + i18n ("' not supported. Ignored"), 2, &e);		
			}
        		n_b = n_b.nextSibling ();
		}

		// get initial position
// TODO: Set sensible upper and lower limits
		path[segments].start_x = cdp->get_double_attribute ( "x", e, -100, 10000, 20, 1 );		
		path[segments].start_y = cdp->get_double_attribute ( "y", e, -100, 10000, 20, 1 );

		path[segments].steps = cdp->get_int_attribute ( "time", e, OBJECTS_AP, 100000, 600, 1 ) / OBJECTS_AP;
	
		// and now the tricky part: do path calculations
		int mode_dummy;		// 0 is abrupt (easy), 1 is linear, 2 is custom
		dummy = cdp->get_multi_choice_attribute ( "accel", e, "linear;abrupt;custom", "abrupt", 2 );
		if ( dummy == "linear" ) {
			mode_dummy = 1;
		} else if ( dummy == "abrupt" ) {
			mode_dummy = 0;
		} else if ( dummy == "custom" ) {
			mode_dummy = 2;

			int r = (int) (1000 / OBJECTS_AP);

			// no need to know about target, Do everything right here.
			path[segments].vx = cdp->get_double_attribute ( "vx", e, -10000, 10000, 0, 2 ) / r;
			path[segments].vy = cdp->get_double_attribute ( "vy", e, -10000, 10000, 0, 2 ) / r;

			path[segments].ax = cdp->get_double_attribute ( "ax", e, -10000, 10000, 0, 2 ) / (r * r);
			path[segments].ay = cdp->get_double_attribute ( "ay", e, -10000, 10000, 0, 2 ) / (r * r);

			path[segments].dax = cdp->get_double_attribute ( "dax", e, -10000, 10000, 0, 3 ) / (r * r * r);
			path[segments].day = cdp->get_double_attribute ( "day", e, -10000, 10000, 0, 3 ) / (r * r * r);

		}

		double target_x, target_y;

		n = n.nextSibling();		// we need to know about the starting-points of the next frame			

		if (n.isNull()) {
			if (looping) {
				target_x = path[0].start_x;
				target_y = path[0].start_y;
			} else {
/*				int seg = segments -1;
				if (seg < 0) { 		// max (0, segments)
					seg = 0;				
				}
				target_x = path[seg].start_x;
				target_y = path[seg].start_y; */
				target_x = path[segments].start_x;
				target_y = path[segments].start_y;
			}
		} else {
			e = n.toElement ();
			target_x = cdp->get_double_attribute ( "x", e, -100, 10000, 20, 1 );
			target_y = cdp->get_double_attribute ( "y", e, -100, 10000, 20, 1 );		
		}

		if ( mode_dummy == 0 ) {		// accel-mode "abrupt"
			previous_vx = path[segments].vx = (target_x - path[segments].start_x) / path[segments].steps;				        	
			previous_vy = path[segments].vy = (target_y - path[segments].start_y) / path[segments].steps;				        	
			path[segments].ax =	path[segments].ay = path[segments].dax = path[segments].day = 0;
		} else if ( mode_dummy == 1 ) {
			path[segments].vx = previous_vx;		// Starting with end-velocity of last segment		
			path[segments].vy = previous_vy;

			double target_av_vx, target_av_vy;		// the average velocity we have to achieve
			target_av_vx = (target_x - path[segments].start_x) / path[segments].steps;
			target_av_vy = (target_y - path[segments].start_y) / path[segments].steps;
			path[segments].ax = ((target_av_vx - previous_vx) * 2) / path[segments].steps;
			path[segments].ay = ((target_av_vy - previous_vy) * 2) / path[segments].steps;

			path[segments].dax = path[segments].day = 0;

			previous_vx += path[segments].ax * path[segments].steps;
			previous_vy += path[segments].ay * path[segments].steps;
		}  /* else if ( mode_dummy == 2 ) {
		// already done
		} */
			
		segments++; 		// already advanced n to next element in list
	}
		
	if (!looping) {		// bouncing object: calculate way back
		for (int i = 0; i < segments; i++) {

			if (i < (segments -1)) {
				path[segments + i].action = path[segments-i-2].action;
			} else {
				path[segments + i].action = 0;
			}

			path[segments + i].anim = path[segments-i-1].anim;
			path[segments + i].visible = path[segments-i-1].visible;
			path[segments + i].z = path[segments-i-1].z;
			path[segments + i].steps = path[segments-i-1].steps;
			path[segments + i].frames = path[segments-i-1].frames;
			path[segments + i].timer_period = path[segments-i-1].timer_period;	

			// I'd be surprised if the following section always worked as expected!

			path[segments + i].start_x = (path[segments-i-1].start_x + path[segments-i-1].steps*path[segments-i-1].vx + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].ax/2) + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].dax/6));
			path[segments + i].start_y = (path[segments-i-1].start_y + path[segments-i-1].steps*path[segments-i-1].vy + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].ay/2) + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].day/6));

			path[segments + i].vx = - (path[segments-i-1].vx + path[segments-i-1].steps*path[segments-i-1].ax + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].dax/2));
			path[segments + i].vy = - (path[segments-i-1].vy + path[segments-i-1].steps*path[segments-i-1].ay + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].day/2));

			path[segments + i].ax = (path[segments-i-1].ax + path[segments-i-1].steps*path[segments-i-1].dax);
			path[segments + i].ay = (path[segments-i-1].ay + path[segments-i-1].steps*path[segments-i-1].day);

			path[segments + i].dax = - path[segments-i-1].dax;
			path[segments + i].day = - path[segments-i-1].day;

		}
		segments *= 2;
	}
}

/** Whether the object would currently be harmful upon collision with the taxi */
bool MovingObject::harmful ()
{
	return (visible () && harmful_if_visible);
}

/** Returns the current x-velocity of the object */
double MovingObject::current_vx (){
	return (vx);
}

/** Returns the current y-Velocity of the object */
double MovingObject::current_vy(){
	return (vy);
}

/** Parses the element e for a description of font, color, text and returns a QCanvasPixmapArray
with 1 frame, containing the rendered text. */
QCanvasPixmapArray * MovingObject::create_text_qcpma (const QDomElement & e){

	int lines = 1;
	int pos = -1;
	int old_pos = 0;
	int width = 0;

	QFont fnt (cdp->get_string_attribute ( "font", e, "helvetica", 3 ), cdp->get_int_attribute ( "size", e, 4, 200, 14, 3 ));
	QColor color (cdp->get_int_attribute ( "r", e, 0, 255, 255, 3 ), cdp->get_int_attribute ( "g", e, 0, 255, 255, 3 ), cdp->get_int_attribute ( "b", e, 0, 255, 255, 3 ));
	QFontMetrics fm (fnt);
	QRect brect;

	QString text = cdp->get_string_attribute ( "text", e, "Hi there!", 2 );
	while ((pos = text.find ( "\\n" )) >= 0) {		// convert \n to newline
		text.replace ( pos, 2, "\n" );
		brect = fm.boundingRect (text.mid (old_pos, pos - old_pos));
		if (width < brect.width ()) {
			width = brect.width ();
		}
		lines++;
		old_pos = pos+1;
	}
	brect = fm.boundingRect (text.right (text.length() - old_pos));
	if (width < brect.width ()) {
		width = brect.width ();
	} else {
		brect.setWidth (width);
	}
	brect.moveTopLeft (QPoint (0, 0));
	brect.setHeight (fm.lineSpacing () * lines);      // somehow boundingRect does not give the correct size (too small, even with only single line texts)
	QPixmap * text_pm = new QPixmap (brect.size ());
	QList<QPixmap> dummy;
	QList<QPoint> dummyb;

	QString align = cdp->get_multi_choice_attribute ( "align", e, "left;center;right", "center", 3 );
	int al_flags = Qt::AlignRight;
	if (align == "left") {
		al_flags = Qt::AlignLeft;
	} else if (align == "center") {
		al_flags = Qt::AlignHCenter;
	}

	QPainter painter (text_pm);
	painter.setFont(fnt);
	painter.setPen(color);
	painter.setBackgroundColor (QColor (0, 0, 0));
	painter.eraseRect (brect);
	painter.drawText(brect, al_flags, text);
	painter.end ();

// create a mask
	QBitmap mask (brect.width (), brect.height (), true);
	QPainter maskpainter (&mask);
	maskpainter.setFont(fnt);
	maskpainter.setPen(Qt::color1);
	maskpainter.setBackgroundColor (Qt::color0);
	maskpainter.eraseRect (brect);
	maskpainter.drawText(brect, al_flags, text);
	maskpainter.end ();

	text_pm->setMask (mask);

	dummy.append (text_pm);
	dummyb.append (new QPoint (0,0));

	return (new QCanvasPixmapArray (dummy, dummyb));
}

void MovingObject::timerTick(int id)
{
	if (id == advance_timer) {
		if ( step == 0 ) {		// Segments has just begun
			processing_action = actions_done = false;			
			move ( path[current_segment].start_x, path[current_segment].start_y );
			if (platform) platform_p->movedTo (x (), y());
			setZ (path[current_segment].z);
			if (animation_timer != -1) tm->killTimer (animation_timer);
			if ( path[current_segment].frames > 1 ) {
				animation_timer = tm->startTimer (this, path[current_segment].timer_period);
			} else {
				animation_timer = -1;
			}
			if (frame > (path[current_segment].frames -1)) setFrame (frame = 0);
			setSequence (path[current_segment].anim);
			if (last_visible != path[current_segment].visible) {
				setVisible (last_visible = path[current_segment].visible);
				if (platform) platform_p->set_visible (last_visible);
				if (teleporter) teleporter_p->set_visible (last_visible);
			}
			vx = path[current_segment].vx;
			vy = path[current_segment].vy;
			ax = path[current_segment].ax;
			ay = path[current_segment].ay;
			step++;
		} else {		// normal move or action
			if (processing_action) {
				process_actions ();
				if (actions_done) {
					step = 0;
					if ( ++current_segment >= segments ) {
						current_segment = 0;
					}
					return; 		// start over with step == 0!
				}
			}
			if (!processing_action) {			// this is not an else!
				ax += path[current_segment].dax;
				ay += path[current_segment].day;
				vx += ax;
				vy += ay;
				moveBy (vx, vy);
				if (platform) platform_p->movedBy (vx, vy);
				if ( ++step >= (path[current_segment].steps -1)) { // finished this segment
					if (path[current_segment].action && (!actions_done)) {
						current_action = path[current_segment].action;
						process_actions ();						
					}
					if (!processing_action) {
						step = 0;
						if ( ++current_segment >= segments ) {
							current_segment = 0;
						}
					}				
				}
			}
		}
	} else if (id == animation_timer) {
		if ( ++frame >= path[current_segment].frames ) {
			frame = 0;
		}
		setFrame ( frame );
	}
}

/** Process a (chain of) actions (stop/freeze/jump) */
void MovingObject::process_actions () {
	bool processing = true;
	while (processing) {
		if (!current_action->conditional->fulfilled ()) {
			if (current_action->next) {
				current_action = current_action->next;
			} else {
				processing = false;
				processing_action = false;
				actions_done = true;
			}
		} else {
			if (initialize_action ()) {
				processing = false;
			} else {
				if (current_action->next) {
					current_action = current_action->next;
				} else {
					processing = false;
					processing_action = false;
					actions_done = true;
				}
			}
		}
	}
}

/** Does anything that needs to be done when first encountering the current_action. Returns
true, if the action is enduring until its condition fails, false, if we can proceed. */
bool MovingObject::initialize_action () {
	processing_action = true;
	if (current_action->freeze) {
		if (animation_timer != -1) {
			tm->killTimer (animation_timer);
			animation_timer = -1;
		}
		return true;
	} else if (current_action->stop) {
		return true;
	} else if (current_action->trigger) {
		current_action->event_to_trigger->trigger ();
		return false;
	} else if (current_action->jump) {
		processing_action = false;
		actions_done = true;
		current_segment = current_action->jump_target -1;		// (current_segment will be increased by one after this!)
/*		QString dummy;
		cdp->debug_msg ("jumped to " + dummy.setNum (current_segment+2) + " !", 2, 0); */
		return true;
	} else {
		return false;
	}
}