/***************************************************************************
                          platform.cpp  -  description
                             -------------------
    begin                : Tue Nov 6 2001
    copyright            : (C) 2001 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "platform.h"

#include <qcanvas.h>
#include <qstring.h>
#include <qdom.h>

#include "passenger.h"
#include "taxi.h"
#include "taxipilot.h"
#include "level.h"
#include "movingobject.h"
#include "cdp.h"

#include "defines.h"

Platform::Platform(const QDomElement *description, Level *parent, bool draw_lines){

	draw_platform_lines = draw_lines;

	level = parent;
	tp = level->tp;
	cdp = level->cdp;

	visible = true;
	has_object = false;
	
	advance_timer = -1;

	passenger_ontop = taxi_ontop = false;

	read_platform (description);

}

Platform::~Platform(){

	if (draw_platform_lines) {
		delete line;
	}

}

/** Read the platforms's description and creates it */
void Platform::read_platform(const QDomElement * description){

	QString convert;

	if (description->hasAttribute ("ident")) {
		object_ident = cdp->get_string_attribute ("ident", *description, "#bad value#", 1);
		object_promised = true;
	} else {
		last_x = last_y = 0;
		object_promised = false;
	}

	y = cdp->get_int_attribute("y", *description, 0, 3000, 100, 1);
	x1 = cdp->get_int_attribute("x1", *description, 0, 3000, 100, 1);
	x2 =	cdp->get_int_attribute("x2", *description, x1, 3000, x1 + 100, 1);

	if (cdp->get_bool_attribute("has_fuel", *description, false, 3)) {
		fuel_price = cdp->get_double_attribute("fuel_price", *description, 0, 100, 0.05, 2);
		fuel_step = cdp->get_int_attribute("fuel_step", *description, 1, 8000, 1, 3);
	} else {
		fuel_step = 0;
	}

	if ((accepts_passenger = cdp->get_bool_attribute("passengers", *description, false, 3))) {
		passenger_base_x = cdp->get_int_attribute("passenger_base_x", *description, 0, 3000, x1, 1);
	}

	label = cdp->get_string_attribute("label", *description, i18n("Platform ") + convert.setNum(level->num_platforms + 1), 3);

	if (draw_platform_lines) {
		line = new QCanvasLine(level->canvas());
		line->setPen(yellow);
		line->setPoints(x1, y, x2, y);
		line->setZ(LEVEL_Z + 1);
		line->show();
	}

}

/** Tells the platform, whether it is currently visible */
void Platform::set_visible(bool visibility){
	if (passenger_ontop) {
		passenger->setVisible (visibility);
	}
	if (taxi_ontop) {
		taxi->landed = false;
	}

	visible = visibility;
}

/** Fulfills the promise (assigns the promised object to this platform) */
void Platform::promise_fulfilled(MovingObject * obj){

	object = obj;
	has_object = true;
	
	connect (obj, SIGNAL (am_visible (bool)), this, SLOT (set_visible (bool)));

	last_x = (int) object->x();
	last_y = (int) object->y();

	advance_timer = startTimer (OBJECTS_AP);

}

/** Drop a passenger on top of this paltform (moving platforms will use the pointer to
move the passenger with them) */
void Platform::attach_passenger (Passenger *pass){

	passenger_ontop = true;
	passenger = pass;
	connect (pass, SIGNAL (im_gone ()), this, SLOT (detach_passenger ()));
	connect (pass, SIGNAL (boarding_complete ()), this, SLOT (detach_passenger ()));	
	connect (passenger, SIGNAL (deleted ()), this, SLOT (detach_passenger ()));
	connect (tp, SIGNAL (gameOver ()), this, SLOT (detach_passenger ()));	

}

/** Detaches the passenger when it has boarded the taxi or disappeared */
void Platform::detach_passenger(){

	disconnect (passenger, SIGNAL (im_gone ()), this, SLOT (detach_passenger ()));
	disconnect (passenger, SIGNAL (boarding_complete ()), this, SLOT (detach_passenger ()));	
	disconnect (passenger, SIGNAL (deleted ()), this, SLOT (detach_passenger ()));	
	disconnect (tp, SIGNAL (gameOver ()), this, SLOT (detach_passenger ()));	

	passenger_ontop = false;

}

/** Takes care of moving the passenger and taxi on this platform (if they are on the
platform and if the platform is a moving object) */
void Platform::timerEvent(QTimerEvent * e){

	int o_x = (int) object->x();
	int o_y = (int) object->y();
	
	if (taxi_ontop) {
		taxi->wrap_moveBy (o_x - last_x, o_y - last_y);
	}
	if (passenger_ontop) {
		passenger->move_passenger_by (o_x - last_x, o_y - last_y);
	}
	if (draw_platform_lines) {
		line->move (current_plat_x1 (), current_plat_y ());
	}

	// set new reference position
	last_x = o_x;
	last_y = o_y;

}

/** Checks whether the given taxi is landed on this platform and if so, takes appropriate
actions. */
bool Platform::check_landed (Taxi *taxi_p){

	double tolerance_down;
	
	tolerance_down = (taxi_p->yVelocity() * ( (int) ((double) cdp->fps() / (double) cdp->cps ()) + 1));

	if (has_object) {
		tolerance_down -= (object->current_vy () * ( (int) ((double) cdp->fps () / (double) cdp->cps ()) + 1));
	}
	
	tolerance_down += 5;

	if ((taxi_p->landing_flaps_y () >= (current_plat_y () - taxi_p->yVelocity () - 2)) && ((taxi_p->landing_flaps_y () - current_plat_y ()) < tolerance_down)) {
		if ((taxi_p->landing_flaps_x1() >= current_plat_x1 ()) && (taxi_p->landing_flaps_x2() <= current_plat_x2 ())) {
			attach_taxi (taxi_p);
			taxi_p->landed = true;
			return (true);
		}
	}
	return (false);
}

/** Current y-position of the actual platform (i.e. taking into account any offset) */
int Platform::current_plat_y (){
	return (last_y + y);
}

/** Current x1-position of the actual platform (i.e. taking into account any offset) */
int Platform::current_plat_x1(){
	return (last_x + x1);
}

/** Current x2-position of the actual platform (i.e. taking into account any offset) */
int Platform::current_plat_x2(){
	return (last_x + x2);
}

/** Current x-position of the actual platform's passenger_base_x (i.e. taking into account any offset) */
int Platform::current_plat_base_x(){
	return (last_x + passenger_base_x);
}

/** A taxi has landed on top of this paltform (moving platforms will use the pointer to
move the taxi with them) */
void Platform::attach_taxi(Taxi * taxi_p){

	if (!taxi_ontop) {	// this function will be called over and over again
		taxi_ontop = true;
		taxi = taxi_p;

		connect (taxi, SIGNAL (take_off ()), this, SLOT (detach_taxi ()));
		connect (tp, SIGNAL (taxiCrashed ()), this, SLOT (detach_taxi ()));	
        }

}

/** Detaches the taxi when it takes off */
void Platform::detach_taxi(){

	disconnect (taxi, SIGNAL (take_off ()), this, SLOT (detach_taxi ()));
	disconnect (tp, SIGNAL (taxiCrashed ()), this, SLOT (detach_taxi ()));	

	taxi_ontop = false;

}
