/***************************************************************************
                          movingobject.cpp  -  description
                             -------------------
    begin                : Tue Oct 23 2001
    copyright            : (C) 2001 by Thomas Friedrichsmeier
    email                : Thomas.Friedrichsmeier@ruhr-uni-bochum.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "movingobject.h"

#include <qdom.h>
#include <qstring.h>
#include <qlist.h>

#include "taxipilot.h"
#include "level.h"
#include "cdp.h"

#include "defines.h"

MovingObject::MovingObject(const QDomElement *description, Taxipilot *parent, QCanvas *canvas):QCanvasSprite(0, canvas)
{
	cdp = parent->cdp;

	advance_timer = animation_timer = -1;
	setAnimated (false);

	if (cdp->for_real ()) {
		QObject::connect(parent, SIGNAL(gamePaused()), this, SLOT(pause()));
		QObject::connect(parent, SIGNAL(gameResumed()), this, SLOT(resume()));
	}

	read_object (description);

	paused = false;
	current_segment = 0;
	frame = 0;
	step = 0;
	setSequence (path[0].anim);		// so other classes can already access width and height and stuff
	if (cdp->for_real ()) {
		advance_timer = startTimer (OBJECTS_AP);         // sets everything off
		emit (am_visible (last_visible=path[current_segment].visible));
		setVisible (last_visible);
	}
}

MovingObject::~MovingObject()
{
        hide ();
	
	QList <QCanvasPixmapArray> deleted;		// to prevent multiple deletion of shared animations
	deleted.setAutoDelete (false);

	for (int i=0; i < segments; i++) {
		if (deleted.find (path[i].anim) == -1) {
			deleted.append (path[i].anim);
			delete path[i].anim;			
		}
	}

	deleted.clear ();
}

/** This function reads a moving-object-description ( specs for the animations ) from a
QDomNode (of a level-description) read_object is only called by the constructor. */
void MovingObject::read_object(const QDomElement *description)
{
	QCanvasPixmapArray *globalanim = 0;
	int global_anim_frames = 1;

	int frames_dl = 1; // debug-levels if frames / anim missing in individual segments
	int anim_dl = 0;

	int global_z;
	int global_frame_period;
	QDomElement e;
	QDomNodeList list;
	QDomNode n;
	QString dummy;
	bool looping;

	if (description->hasAttribute ("ident")) {
		has_ident = true;
		ident = cdp->get_string_attribute ("ident", *description, "#not given#", 1);
	} else {
		has_ident = false;
	}

	teleporter = center = platform = false;
	harmful_if_visible = true;

	QString classification = cdp->get_multi_choice_attribute ( "class", *description, "harmful;background;teleporter;harmful center;harmless center;platform", "harmful", 2 );
	if (classification == "harmful") {

	} else if (classification == "background") {
		harmful_if_visible = false;
	} else if (classification == "platform") {
		platform = true;
		harmful_if_visible = true;		// you can crash into platforms
	} else {
		if (classification == "teleporter") {
			teleporter = true;
			harmful_if_visible = false;		
		} else if (classification == "harmful center") {
			center = true;			
		} else if (classification == "harmless center") {
			harmful_if_visible = false;
			center = true;
		}
		if (!has_ident) {
			cdp->debug_msg (i18n ("Object classified as '") + classification + i18n ("teleporter but no ident given"), 1, description);
		}
	}
	global_z = cdp->get_int_attribute ( "z", *description, 0, 300, 150, 3 );
	if (description->hasAttribute ("frames")) {
		global_anim_frames = cdp->get_int_attribute ( "frames", *description, 1, 10000, 1, 1 );
		frames_dl = 3;
	}
	if (description->hasAttribute ( "file_pattern" )) {
		globalanim = new QCanvasPixmapArray(cdp->prefix_check_animation(cdp->get_string_attribute("file_pattern", *description, "", 3),
															global_anim_frames), global_anim_frames);
		if (globalanim) anim_dl = 3;
	}
	global_frame_period = cdp->get_int_attribute ( "anim_frame_period", *description, 5, 1000, 50, 3 );
	looping = cdp->get_bool_attribute ( "looping", *description, true, 2 );

	list = cdp->get_node_list ( "path", *description, 1, -1, 1, 1 );	// now get the path-segments

	segments = 0;

	double previous_vx, previous_vy;

	previous_vx = cdp->get_double_attribute ( "initial_vx", *description, 0, 1000, 0, 3 ) / ( (int) (1000 / OBJECTS_AP));
	previous_vy = cdp->get_double_attribute ( "initial_vy", *description, 0, 1000, 0, 3 ) / ( (int) (1000 / OBJECTS_AP));

	if ( looping ) {
		path = new PathSegment[list.length()];
	} else {
		path = new PathSegment[list.length()*2];
	}

	n = list.item(0);

	while (!n.isNull()) {
		e = n.toElement();

		// first get segment-specific animation, if available
		path[segments].z = cdp->get_int_attribute ( "z", e, 0, 300, global_z, 3 );
		path[segments].frames = cdp->get_int_attribute ( "frames", e, 1, 10000, global_anim_frames, frames_dl );
		path[segments].timer_period = cdp->get_int_attribute ( "anim_frame_period", e, 5, 1000, global_frame_period, 3 );
// TODO: Make sure, identical animations are only loaded once
		if (e.hasAttribute ( "file_pattern" )) {
			path[segments].anim = new QCanvasPixmapArray(cdp->prefix_check_animation(cdp->get_string_attribute("file_pattern", e, "", 3),
														path[segments].frames), path[segments].frames);
		} else {
			cdp->debug_msg ( "No animation specified. Assuming global animation (if any)", anim_dl, description );
			path[segments].anim = globalanim;
		}

		path[segments].visible = cdp->get_bool_attribute ( "visible", e, true, 3 );

		// get initial position
// TODO: Set sensible upper and lower limits
		path[segments].start_x = cdp->get_double_attribute ( "x", e, -100, 10000, 20, 1 );		
		path[segments].start_y = cdp->get_double_attribute ( "y", e, -100, 10000, 20, 1 );

		path[segments].steps = cdp->get_int_attribute ( "time", e, OBJECTS_AP, 100000, 600, 1 ) / OBJECTS_AP;
	
		// and now the tricky part: do path calculations
		int mode_dummy;		// 0 is abrupt (easy), 1 is linear, 2 is custom
		dummy = cdp->get_multi_choice_attribute ( "accel", e, "linear;abrupt;custom", "abrupt", 2 );
		if ( dummy == "linear" ) {
			mode_dummy = 1;
		} else if ( dummy == "abrupt" ) {
			mode_dummy = 0;
		} else if ( dummy == "custom" ) {
			mode_dummy = 2;

			int r = (int) (1000 / OBJECTS_AP);

			// no need to know about target, Do everything right here.
			path[segments].vx = cdp->get_double_attribute ( "vx", e, -1000, 1000, 0, 1 ) / r;
			path[segments].vy = cdp->get_double_attribute ( "vy", e, -1000, 1000, 0, 1 ) / r;

			path[segments].ax = cdp->get_double_attribute ( "ax", e, -1000, 1000, 0, 2 ) / (r * r);
			path[segments].ay = cdp->get_double_attribute ( "ay", e, -1000, 1000, 0, 2 ) / (r * r);

			path[segments].dax = cdp->get_double_attribute ( "dax", e, -1000, 1000, 0, 3 ) / (r * r * r);
			path[segments].day = cdp->get_double_attribute ( "day", e, -1000, 1000, 0, 3 ) / (r * r * r);

		}

		double target_x, target_y;

		n = n.nextSibling();		// we need to know about the starting-points of the next frame			

		if (n.isNull()) {
			target_x = path[segments].start_x;
			target_y = path[segments].start_y;
		} else {
			e = n.toElement ();
			target_x = cdp->get_double_attribute ( "x", e, -100, 10000, 20, 1 );
			target_y = cdp->get_double_attribute ( "y", e, -100, 10000, 20, 1 );		
		}

		if ( mode_dummy == 0 ) {		// accel-mode "abrupt"
			previous_vx = path[segments].vx = (target_x - path[segments].start_x) / path[segments].steps;				        	
			previous_vy = path[segments].vy = (target_y - path[segments].start_y) / path[segments].steps;				        	
			path[segments].ax =	path[segments].ay = path[segments].dax = path[segments].day = 0;
		} else if ( mode_dummy == 1 ) {
			path[segments].vx = previous_vx;		// Starting with end-velocity of last segment		
			path[segments].vy = previous_vy;

			double target_av_vx, target_av_vy;		// the average velocity we have to achieve
			target_av_vx = (target_x - path[segments].start_x) / path[segments].steps;
			target_av_vy = (target_y - path[segments].start_y) / path[segments].steps;
			path[segments].ax = ((target_av_vx - previous_vx) * 2) / path[segments].steps;
			path[segments].ay = ((target_av_vy - previous_vy) * 2) / path[segments].steps;

			path[segments].dax = path[segments].day = 0;

			previous_vx += path[segments].ax * path[segments].steps;
			previous_vy += path[segments].ay * path[segments].steps;
		}  /* else if ( mode_dummy == 2 ) {
	// already done
		} */
			
		segments++; 		// already advanced n to next element in list
	}
		
	if (!looping) {		// bouncing object: calculate way back
		for (int i = 0; i < segments; i++) {

			path[segments + i].anim = path[segments-i-1].anim;
			path[segments + i].visible = path[segments-i-1].visible;
			path[segments + i].z = path[segments-i-1].z;
			path[segments + i].steps = path[segments-i-1].steps;
			path[segments + i].frames = path[segments-i-1].frames;
			path[segments + i].timer_period = path[segments-i-1].timer_period;	

			// I'd be surprised if the following section always worked as expected!

			path[segments + i].start_x = (path[segments-i-1].start_x + path[segments-i-1].steps*path[segments-i-1].vx + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].ax/2) + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].dax/6));
			path[segments + i].start_y = (path[segments-i-1].start_y + path[segments-i-1].steps*path[segments-i-1].vy + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].ay/2) + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].day/6));

			path[segments + i].vx = - (path[segments-i-1].vx + path[segments-i-1].steps*path[segments-i-1].ax + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].dax/2));
			path[segments + i].vy = - (path[segments-i-1].vy + path[segments-i-1].steps*path[segments-i-1].ay + (path[segments-i-1].steps*path[segments-i-1].steps*path[segments-i-1].day/2));

			path[segments + i].ax = (path[segments-i-1].ax + path[segments-i-1].steps*path[segments-i-1].dax);
			path[segments + i].ay = (path[segments-i-1].ay + path[segments-i-1].steps*path[segments-i-1].day);

			path[segments + i].dax = - path[segments-i-1].dax;
			path[segments + i].day = - path[segments-i-1].day;
						
		}
		segments *= 2;
	}
}

/** Whether the object would currently be harmful upon collision with the taxi */
bool MovingObject::harmful ()
{
	return (visible () && harmful_if_visible);
}

/** Advancing and animating */
void MovingObject::timerEvent(QTimerEvent * e)
{
	if (!paused) {
		if (e->timerId () == advance_timer) {
			if ( step == 0 ) {		// Segments has just begun
				move ( path[current_segment].start_x, path[current_segment].start_y );
				setZ (path[current_segment].z);
				if (animation_timer != -1) killTimer (animation_timer);
				if ( path[current_segment].frames > 1 ) {
					animation_timer = startTimer (path[current_segment].timer_period);
				} else {
					animation_timer = -1;
				}
				if (frame > (path[current_segment].frames -1)) setFrame (frame = 0);
				setSequence (path[current_segment].anim);
				if (last_visible != path[current_segment].visible) {
					emit (am_visible (last_visible));
					setVisible (last_visible = path[current_segment].visible);
				}
				vx = path[current_segment].vx;
				vy = path[current_segment].vy;
				ax = path[current_segment].ax;
				ay = path[current_segment].ay;
				step++;
			} else {		// normal move
				ax += path[current_segment].dax;
				ay += path[current_segment].day;
				vx += ax;
				vy += ay;
				moveBy (vx, vy);
				if ( ++step >= (path[current_segment].steps -1)) { // finished this segment
					step = 0;
					if ( ++current_segment >= segments ) {
						current_segment = 0;
					}
				}
			}
		} else if (e->timerId () == animation_timer) {
			if ( ++frame >= path[current_segment].frames ) {
				frame = 0;
			}
			setFrame ( frame );
		}
	}	
}

/** Pauses the object */
void MovingObject::pause(){
	paused = true;
}

/** Resumes movement for a paused object */
void MovingObject::resume(){
	paused = false;
}

/** Returns the current x-velocity of the object */
double MovingObject::current_vx (){
	return (vx);
}

/** Returns the current y-Velocity of the object */
double MovingObject::current_vy(){
	return (vy);
}
