/*
 * Sylpheed -- a GTK+ based, lightweight, and fast e-mail client
 * Copyright (C) 1999,2000 Hiroyuki Yamamoto
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "xml.h"
#include "utils.h"

static void xml_free_tag	(XMLTag		*tag);
static gint xml_get_next_char	(XMLFile	*file);
static void xml_get_parenthesis	(XMLFile	*file,
				 gchar		*buf);

XMLFile *xml_open_file(const gchar *path)
{
	XMLFile *newfile;

	g_return_val_if_fail(path != NULL, NULL);

	newfile = g_new(XMLFile, 1);

	newfile->fp = fopen(path, "r");
	if (!newfile->fp) {
		//debug_print("can't open %s\n", path);
		g_free(newfile);
		return NULL;
	}

	newfile->buf[0] = '\0';
	newfile->bufp = newfile->buf;

	newfile->dtd = NULL;
	newfile->tag_stack = NULL;
	newfile->level = 0;
	newfile->is_single_element = FALSE;

	return newfile;
}

void xml_close_file(XMLFile *file)
{
	g_return_if_fail(file != NULL);

	if (file->fp) fclose(file->fp);

	g_free(file->dtd);

	while (file->tag_stack != NULL)
		xml_pop_tag(file);

	g_free(file);
}

static GNode *xml_build_tree(XMLFile *file, GNode *parent, guint level)
{
	GNode *node = NULL;
	XMLNode *xmlnode;
	XMLTag *tag;

	for (;;) {
		xml_parse_next_tag(file);
		if (file->level < level) break;
		if (file->level == level) {
			g_warning("xml_build_tree(): Parse error\n");
			break;
		}

		tag = xml_get_current_tag(file);
		if (!tag) break;
		xmlnode = g_new(XMLNode, 1);
		xmlnode->tag = xml_copy_tag(tag);
		xmlnode->element = xml_get_element(file);
		if (!parent)
			node = g_node_new(xmlnode);
		else
			node = g_node_append_data(parent, xmlnode);

		xml_build_tree(file, node, file->level);
		if (file->level == 0) break;
	}

	return node;
}

GNode *xml_parse_file(const gchar *path)
{
	XMLFile *file;
	GNode *node;

	file = xml_open_file(path);
	g_return_val_if_fail(file != NULL, NULL);

	xml_get_dtd(file);

	node = xml_build_tree(file, NULL, file->level);

	xml_close_file(file);
	return node;
}

void xml_get_dtd(XMLFile *file)
{
	gchar buf[XMLBUFSIZE];
	gchar *bufp = buf;

	xml_get_parenthesis(file, buf);

	if ((*bufp++ == '?') &&
	    (bufp = strcasestr(bufp, "xml")) &&
	    (bufp = strcasestr(bufp + 3, "version")) &&
	    (bufp = strchr(bufp + 7, '?')))
		file->dtd = g_strdup(buf);
	else {
		g_warning("Can't get xml dtd\n");
		return;
	}
}

void xml_parse_next_tag(XMLFile *file)
{
	gchar buf[XMLBUFSIZE];
	gchar *bufp = buf;
	XMLTag *tag;
	gint len;

	if (file->is_single_element == TRUE) {
		file->is_single_element = FALSE;
		xml_pop_tag(file);
		return;
	}

	xml_get_parenthesis(file, buf);

	g_strstrip(buf);
	if (buf[0] == '\0') {
		g_warning("Can't parse next tag\n");
		return;
	}

	/* end-tag */
	if (buf[0] == '/') {
		if (strcmp(xml_get_current_tag(file)->tag, buf + 1) != 0) {
			g_warning("xml_parse_next_tag(): Tag name mismatch: %s\n", buf);
			return;
		}
		xml_pop_tag(file);
		return;
	}

	tag = g_new0(XMLTag, 1);
	xml_push_tag(file, tag);

	len = strlen(buf);
	if (len > 0 && buf[len - 1] == '/') {
		file->is_single_element = TRUE;
		buf[len - 1] = '\0';
		g_strchomp(buf);
	}
	if (strlen(buf) == 0) {
		g_warning("xml_parse_next_tag(): Tag name is empty\n");
		return;
	}

	while (*bufp != '\0' && !isspace(*bufp)) bufp++;
	if (*bufp == '\0') {
		tag->tag = g_strdup(buf);
		return;
	} else {
		*bufp++ = '\0';
		tag->tag = g_strdup(buf);
	}

	/* parse attributes ( name=value ) */
	while (*bufp) {
		XMLAttr *attr;
		gchar *attr_name;
		gchar *attr_value;
		gchar *p;

		while (isspace(*bufp)) bufp++;
		attr_name = bufp;
		if ((p = strchr(attr_name, '=')) == NULL) {
			g_warning("xml_parse_next_tag(): Syntax error in tag\n");
			return;
		}
		bufp = p;
		*bufp++ = '\0';
		while (isspace(*bufp)) bufp++;

		if (*bufp != '"' && *bufp != '\'') {
			g_warning("xml_parse_next_tag(): Syntax error in tag\n");
			return;
		}
		bufp++;
		attr_value = bufp;
		if ((p = strchr(attr_value, '"')) == NULL &&
		    (p = strchr(attr_value, '\'')) == NULL) {
			g_warning("xml_parse_next_tag(): Syntax error in tag\n");
			return;
		}
		bufp = p;
		*bufp++ = '\0';

		g_strchomp(attr_name);

		attr = g_new(XMLAttr, 1);
		attr->name  = g_strdup(attr_name);
		attr->value = g_strdup(attr_value);
		tag->attr = g_list_append(tag->attr, attr);
	}
}

void xml_push_tag(XMLFile *file, XMLTag *tag)
{
	g_return_if_fail(tag != NULL);

	file->tag_stack = g_list_prepend(file->tag_stack, tag);
	file->level++;
}

void xml_pop_tag(XMLFile *file)
{
	XMLTag *tag;

	if (!file->tag_stack) return;

	tag = (XMLTag *)file->tag_stack->data;

	xml_free_tag(tag);
	file->tag_stack = g_list_remove(file->tag_stack, tag);
	file->level--;
}

XMLTag *xml_get_current_tag(XMLFile *file)
{
	if (file->tag_stack)
		return (XMLTag *)file->tag_stack->data;
	else
		return NULL;
}

GList *xml_get_current_tag_attr(XMLFile *file)
{
	XMLTag *tag;

	tag = xml_get_current_tag(file);
	if (!tag) return NULL;

	return tag->attr;
}

gchar *xml_get_element(XMLFile *file)
{
	gchar buf[XMLBUFSIZE];
	gchar *end;

	if (xml_read_line(file) == NULL) return NULL;

	if ((end = strchr(file->bufp, '<')) && end != file->bufp) {
		strncpy(buf, file->bufp, end - file->bufp);
		buf[end - file->bufp] = '\0';
		g_strstrip(buf);
		//debug_print("xml: element = %s\n", buf);
		file->bufp = end;
		return g_strdup(buf);
	}

	return NULL;
}

gchar *xml_read_line(XMLFile *file)
{
	while (*file->bufp == '\n' || *file->bufp == '\0') {
		if (fgets(file->buf, XMLBUFSIZE, file->fp) == NULL)
			return NULL;
		g_strstrip(file->buf);
		file->bufp = file->buf;
	}

	return file->bufp;
}

gboolean xml_compare_tag(XMLFile *file, const gchar *name)
{
	XMLTag *tag;

	tag = xml_get_current_tag(file);

	if (tag && strcmp(tag->tag, name) == 0)
		return TRUE;
	else
		return FALSE;
}

XMLTag *xml_copy_tag(XMLTag *tag)
{
	XMLTag *new_tag;
	XMLAttr *attr;
	GList *list;

	new_tag = g_new(XMLTag, 1);
	new_tag->tag = g_strdup(tag->tag);
	new_tag->attr = NULL;
	for (list = tag->attr; list != NULL; list = list->next) {
		attr = xml_copy_attr((XMLAttr *)list->data);
		new_tag->attr = g_list_append(new_tag->attr, attr);
	}

	return new_tag;
}

XMLAttr *xml_copy_attr(XMLAttr *attr)
{
	XMLAttr *new_attr;

	new_attr = g_new(XMLAttr, 1);
	new_attr->name  = g_strdup(attr->name);
	new_attr->value = g_strdup(attr->value);

	return new_attr;
}

void xml_free_node(XMLNode *node)
{
	if (!node) return;

	xml_free_tag(node->tag);
	g_free(node->element);
	g_free(node);
}

static gboolean xml_free_func(GNode *node, gpointer data)
{
	XMLNode *xmlnode = node->data;

	xml_free_node(xmlnode);
	return FALSE;
}

void xml_free_tree(GNode *node)
{
	g_return_if_fail(node != NULL);

	g_node_traverse(node, G_PRE_ORDER, G_TRAVERSE_ALL, -1, xml_free_func,
			NULL);

	g_node_destroy(node);
}

static void xml_free_tag(XMLTag *tag)
{
	if (!tag) return;

	g_free(tag->tag);
	while (tag->attr != NULL) {
		XMLAttr *attr = (XMLAttr *)tag->attr->data;
		g_free(attr->name);
		g_free(attr->value);
		g_free(attr);
		tag->attr = g_list_remove(tag->attr, tag->attr->data);
	}
	g_free(tag);
}

static gint xml_get_next_char(XMLFile *file)
{
	gint retval;

	if (xml_read_line(file) == NULL) return EOF;

	retval = *file->bufp;
	file->bufp++;

	return retval;
}

static void xml_get_parenthesis(XMLFile *file, gchar *buf)
{
	gchar *start, *end;

	if (xml_read_line(file) == NULL) {
		buf[0] = '\0';
		return;
	}

	if ((start = strchr(file->bufp, '<'))) {
		start++;
		if ((end = strchr(start, '>'))) {
			strncpy(buf, start, end - start);
			buf[end - start] = '\0';
			g_strstrip(buf);
			file->bufp = end + 1;
		} else {
			g_warning("xml_get_parenthesis(): Parse error\n");
			buf[0] = '\0';
		}
	} else
		buf[0] = '\0';
}
