/*
 * Sylpheed -- a GTK+ based, lightweight, and fast e-mail client
 * Copyright (C) 2001 Werner Koch (dd9jn)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#ifdef USE_GPGME

#include "defs.h"

#include <glib.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>
#include <ctype.h>

#include <gpgme.h>

#include "intl.h"
#include "procmime.h"
#include "procheader.h"
#include "base64.h"
#include "uuencode.h"
#include "unmime.h"
#include "codeconv.h"
#include "utils.h"
#include "prefs_common.h"
#include "passphrase.h"
#include "select-keys.h"
#include "rfc2015.h"

#define DIM(v)     (sizeof(v)/sizeof((v)[0]))

static char *content_names[] = {
    "Content-Type",
    "Content-Disposition",
    "Content-Transfer-Encoding",
    NULL
};

static char *mime_version_name[] = {
    "Mime-Version",
    NULL
};


struct passphrase_cb_info_s {
    GpgmeCtx c;
    int did_it;
};


static void dump_mimeinfo ( const char *text, MimeInfo *x )
{

    g_message ("** MimeInfo[%s] %p  level=%d",
               text, x, x? x->level:0 );
    if ( !x )
        return;

    g_message ("**      enc=`%s' enc_type=%d mime_type=%d",
               x->encoding, x->encoding_type, x->mime_type );
    g_message ("**      cont_type=`%s' cs=`%s' name=`%s' bnd=`%s'",
               x->content_type, x->charset, x->name, x->boundary );
    g_message ("**      cont_disp=`%s' fname=`%s' fpos=%ld size=%u, lvl=%d",
               x->content_disposition, x->filename, x->fpos, x->size,
               x->level );
    dump_mimeinfo (".main", x->main );
    dump_mimeinfo (".sub", x->sub );
    dump_mimeinfo (".next", x->next );
    g_message ("** MimeInfo[.parent] %p", x ); 
    dump_mimeinfo (".children", x->children );
    dump_mimeinfo (".plaintext", x->plaintext );
}

static void dump_part ( MimeInfo *mimeinfo, FILE *fp )
{
    unsigned int size = mimeinfo->size;
    int c;

    if ( fseek ( fp, mimeinfo->fpos, SEEK_SET) ) {
        g_warning ("dump_part: fseek error");
        return;
    }

    g_message ("** --- begin dump_part ----");
    while ( size-- && (c = getc (fp)) != EOF ) 
        putc (c, stderr);
    if ( ferror (fp) ) 
        g_warning ("dump_part: read error");
    g_message ("** --- end dump_part ----");
}


void
rfc2015_secure_remove ( const char *fname )
{
    if (!fname)
        return;
    /* fixme: overwrite the file first */
    remove (fname);
}




static void check_signature ( MimeInfo *mimeinfo, MimeInfo *partinfo,
                              FILE *fp )
{
    GpgmeCtx ctx;
    GpgmeError err;
    GpgmeData sig, text;
    GpgmeSigStat status;
    const char *result=NULL;

    err = gpgme_new (&ctx);
    g_assert (!err);
    
    /* don't include the last character (LF). It does not belong to the
     * signed text */
    err = gpgme_data_new_from_filepart ( &text, NULL, fp,
                                         mimeinfo->children->fpos,
                                         mimeinfo->children->size?
                                         (mimeinfo->children->size-1): 0 );
    g_assert (!err);

    err = gpgme_data_new_from_filepart ( &sig, NULL, fp,
                                         partinfo->fpos,
                                         partinfo->size );
    g_assert (!err);

    err = gpgme_op_verify (ctx, sig, text, &status );
    switch ( status ) {
      case GPGME_SIG_STAT_NONE:
        g_message ("Verification Status: None");
        result = _("Oops: Signature not verified");
        break;
      case GPGME_SIG_STAT_NOSIG:
        g_message ("Verification Status: No Signature");
        result = _("No signature found");
        break;
      case GPGME_SIG_STAT_GOOD:
        g_message ("Verification Status: Good");
        result = _("Good signature");
        break;
      case GPGME_SIG_STAT_BAD:
        g_message ("Verification Status: Bad");
        result = _("BAD signature");
        break;
      case GPGME_SIG_STAT_NOKEY:
        g_message ("Verification Status: No Key");
        result = _("No public key to verify the signature");
        break;
      case GPGME_SIG_STAT_ERROR:
        g_message ("Verification Status: Error");
        result = _("Error verifying the signature");
        break;
    }
    
    g_assert (!err);
    g_free (partinfo->sigstatus);
    partinfo->sigstatus = g_strdup (result);

    gpgme_data_release (sig);
    gpgme_data_release (text);
    gpgme_release (ctx);
}


static const char *
passphrase_cb ( void *opaque, const char *desc, void *r_hd )
{
    /* GpgmeCtx ctx = opaque; */
    struct passphrase_cb_info_s *info = opaque;
    GpgmeCtx ctx = info->c;
    const char *pass;

    if ( !desc ) {
        /* FIXME: cleanup by looking at *r_hd */

        
        return NULL;
    }

    gpgmegtk_set_passphrase_grab (prefs_common.passphrase_grab);
    g_message ("%% requesting passphrase for `%s': ", desc );
    pass = gpgmegtk_passphrase_mbox (desc);
    if ( !pass ) {
        g_message ("%% cancel passphrase entry");
        gpgme_cancel (ctx);
    }
    else   
        g_message ("%% sending passphrase");

    return pass;
}

/*
 * Copy a gpgme data object to a temporary file and
 * return this filename 
 */
#if 0
static char *
copy_gpgmedata_to_temp ( GpgmeData data, guint *length )
{
    static int id;
    char *tmp;
    FILE *fp;
    char buf[100];
    size_t nread;
    GpgmeError err;
    
    tmp = g_strdup_printf("%s%cgpgtmp.%08x",
                          get_mime_tmp_dir(), G_DIR_SEPARATOR, ++id );

    if ((fp = fopen(tmp, "w")) == NULL) {
        FILE_OP_ERROR(tmp, "fopen");
        g_free(tmp);
        return NULL;
    }

    err = gpgme_data_rewind ( data );
    g_assert (!err);
    while ( !(err = gpgme_data_read ( data, buf, 100, &nread )) ) {
        fwrite ( buf, nread, 1, fp );
    }
    if (err != GPGME_EOF) {
        g_warning ("** gpgme_data_read failed: %s", gpgme_strerror (err));
    }
    fclose (fp);
    *length = nread;

    return tmp;
}
#endif /* 0 */

static GpgmeData
pgp_decrypt ( MimeInfo *partinfo, FILE *fp )
{
    GpgmeCtx ctx;
    GpgmeError err;
    GpgmeData cipher, plain;
    struct passphrase_cb_info_s info;

    memset ( &info, 0, sizeof info );
    
    err = gpgme_new (&ctx);
    g_assert (!err);
    
    err = gpgme_data_new_from_filepart ( &cipher, NULL, fp,
                                         partinfo->fpos,
                                         partinfo->size );
    g_assert (!err);

    err = gpgme_data_new ( &plain );
    g_assert (!err);

    if ( !getenv("GPG_AGENT_INFO") ) {
        info.c = ctx;
        gpgme_set_passphrase_cb ( ctx, passphrase_cb, &info );
    } 

    err = gpgme_op_decrypt (ctx, cipher, plain );
    gpgme_data_release (cipher);
    if ( err ) {
        g_warning ("** decryption failed: %s", gpgme_strerror (err) );
        gpgme_data_release (plain);
        plain = NULL;
    }
    else {
        g_message ("** decryption succeeded");
    }
    gpgme_release (ctx);
    return plain;
}



void rfc2015_check_signature ( MimeInfo *mimeinfo, FILE *fp )
{
    MimeInfo *partinfo;
    int n;

    if (!mimeinfo)
        return;
    if ( strcmp (mimeinfo->content_type, "multipart/signed") ) 
        return;

    g_message ("** multipart/signed encountered");
    /* check that we have at least 2 parts of tjhe correct type */
    n = 0;
    for (partinfo = mimeinfo->children; partinfo;
         partinfo = partinfo->next ) {
        if ( ++n > 1  && !strcmp (partinfo->content_type,
                                  "application/pgp-signature") ) 
            break;
    }
    if ( !partinfo )
        return;
    g_message ("** yep, it is a pgp signature");
    dump_mimeinfo ("gpg-signature", partinfo );
    dump_part (partinfo, fp );
    dump_mimeinfo ("signed text", mimeinfo->children );
    dump_part (mimeinfo->children, fp);
    check_signature (mimeinfo, partinfo, fp);
}


int rfc2015_is_encrypted ( MimeInfo *mimeinfo )
{
    if (!mimeinfo)
        return 0;
    if ( strcasecmp (mimeinfo->content_type, "multipart/encrypted") )
        return 0;
    /* fixme: we should schek the protocol parameter */
    return 1;
}

static int
name_cmp( const char *a, const char *b )
{
    for( ; *a && *b; a++, b++ ) {
        if( *a != *b
            && toupper(*(unsigned char *)a) != toupper(*(unsigned char *)b) )
            return 1;
    }
    return *a != *b;
}

static int
headerp( char *p, char **names )
{
    int i, c;
    char *p2;

    p2 = strchr(p, ':');
    if( !p2 || p == p2 ) {
        return 0;
    }
    if( p2[-1] == ' ' || p2[-1] == '\t' ) {
        return 0;
    }

    if( !names[0] )
        return 1;  
    c = *p2;
    *p2 = 0;
    for(i=0 ; names[i]; i++ ) {
        if( !name_cmp( names[i], p ) )
            break;
    }
    *p2 = c;
    return !!names[i];
}


void rfc2015_decrypt_message ( MsgInfo *msginfo, MimeInfo *mimeinfo, FILE *fp )
{
    static int id;
    MimeInfo *partinfo;
    int n, found;
    int ver_okay=0;
    char *fname;
    GpgmeData plain;
    FILE *dstfp;
    size_t nread;
    char buf[BUFFSIZE];
    GpgmeError err;

    g_message ("** multipart/encrypted encountered");

    g_assert (mimeinfo->mime_type == MIME_MULTIPART);

    /* skip headers */
    if (fseek(fp, mimeinfo->fpos, SEEK_SET) < 0)
        perror("fseek");
    while (fgets(buf, sizeof(buf), fp) != NULL)
        if (buf[0] == '\r' || buf[0] == '\n') break;
    
    procmime_scan_multipart_message(mimeinfo, fp);

    /* check that we have the 2 parts */
    n = found = 0;
    for (partinfo = mimeinfo->children; partinfo;
         partinfo = partinfo->next ) {
        if ( ++n == 1 && !strcmp (partinfo->content_type,
                                  "application/pgp-encrypted") ) {
            /* Fixme: check that the version is 1 */
            ver_okay = 1;
        }
        else if ( n == 2 && !strcmp (partinfo->content_type,
                                     "application/octet-stream") ) {
            if ( partinfo->next )
                g_warning ("** oops: pgp_encrypted with more than 2 parts");
            break;
        }
    }
    if ( !ver_okay || !partinfo ) {
        msginfo->decryption_failed = 1;
        /* fixme: remove the stuff, that the above procmime_scan_multiparts() 
         * has appended to mimeino */
        return;
    }
    g_message ("** yep, it is pgp encrypted");
    plain = pgp_decrypt ( partinfo, fp );
    if (!plain) {
        msginfo->decryption_failed = 1;
        return;
    }
    
    fname = g_strdup_printf("%s%cplaintext.%08x",
                          get_mime_tmp_dir(), G_DIR_SEPARATOR, ++id );

    if ((dstfp = fopen(fname, "w")) == NULL) {
        FILE_OP_ERROR(fname, "fopen");
        g_free(fname);
        msginfo->decryption_failed = 1;
        return;
    }

    /* write the orginal header to the new file */
    if (fseek(fp, mimeinfo->fpos, SEEK_SET) < 0)
        perror("fseek");
    while ( fgets(buf, sizeof(buf), fp) ) {
        if ( headerp ( buf, content_names ) )
            continue;
        if (buf[0] == '\r' || buf[0] == '\n')
            break;
        fputs (buf, dstfp);
    }
    

    err = gpgme_data_rewind ( plain );
    g_assert (!err);
    while ( !(err = gpgme_data_read ( plain, buf, sizeof(buf), &nread )) ) {
        fwrite ( buf, nread, 1, dstfp );
    }
    if (err != GPGME_EOF) {
        g_warning ("** gpgme_data_read failed: %s", gpgme_strerror (err));
    }
    fclose (dstfp);

    msginfo->plaintext_file = fname;
    msginfo->decryption_failed = 0;
}


/* 
 * plain contains an entire mime object.
 * Encrypt it and return an GpgmeData object with the encrypted version of
 * the file or NULL in case of error.
 */
static GpgmeData
pgp_encrypt ( GpgmeData plain, GpgmeRecipients rset )
{
    GpgmeCtx ctx;
    GpgmeError err;
    GpgmeData cipher;

    err = gpgme_new (&ctx);
    g_assert (!err);
    
    err = gpgme_data_new ( &cipher );
    g_assert (!err);

    err = gpgme_op_encrypt (ctx, rset, plain, cipher );
    if ( err ) {
        g_warning ("** encryption failed: %s", gpgme_strerror (err) );
        gpgme_data_release (cipher);
        cipher = NULL;
    }
    else {
        g_message ("** encryption succeeded");
    }
    gpgme_release (ctx);
    return cipher;
}


/*
 * Encrypt the file by extracting all recipients and finding the
 * encryption keys for all of them.  The file content is then replaced
 * by the encrypted one.  */
int
rfc2015_encrypt ( const char *file, GSList *recp_list)
{
    FILE *fp = NULL;
    char buf[BUFFSIZE];
    int i, clineidx;
    char *clines[3] = {NULL};
    GpgmeError err;
    GpgmeData header = NULL;
    GpgmeData plain = NULL;
    GpgmeData cipher = NULL;
    GpgmeRecipients rset = NULL;
    size_t nread;
    int mime_version_seen = 0;

    /* Create the list of recipients */
    rset = gpgmegtk_recipient_selection (recp_list);
    if (!rset) {
        g_warning ("error creating recipient list" );
        goto failure;
    }

    /* Open the source file */
    if ((fp = fopen(file, "r+b")) == NULL) {
        FILE_OP_ERROR(file, "fopen");
        goto failure;
    }

    err = gpgme_data_new ( &header );
    g_assert (!err);
    err = gpgme_data_new ( &plain );
    g_assert (!err);
    
    /* get the content header lines from the source */
    clineidx=0;
    while ( !err && fgets(buf, sizeof(buf), fp) ) {
        /* fixme: check for overlong lines */
        if ( headerp ( buf, content_names ) ) {
            if (clineidx >= DIM (clines) ) {
                g_warning ("Oops|");
                goto failure;
            }
            clines[clineidx++] = g_strdup (buf);
            continue; 
        }
        if ( headerp ( buf, mime_version_name ) ) 
            mime_version_seen = 1;

        if (buf[0] == '\r' || buf[0] == '\n')
            break;
        err = gpgme_data_write (header, buf, strlen (buf));
    }
    if (ferror (fp) ) {
        FILE_OP_ERROR (file, "fgets");
        goto failure;
    }

    /* write them to the temp data and add the rest of the message */
    for (i=0; !err && i <clineidx; i++ ) {
        g_message ("%% %s:%d: cline=`%s'", __FILE__ ,__LINE__, clines[i] );
        err = gpgme_data_write (plain, clines[i], strlen (clines[i]));
    }
    if (!err)
        err = gpgme_data_write (plain, "\r\n", 2 );
    while ( !err && fgets(buf, sizeof(buf), fp) ) {
        err = gpgme_data_write (plain, buf, strlen (buf) );
    }
    if (ferror (fp) ) {
        FILE_OP_ERROR (file, "fgets");
        goto failure;
    }
    if ( err ) {
        g_warning ("** gpgme_data_write failed: %s", gpgme_strerror (err));
        goto failure;
    }

    cipher = pgp_encrypt (plain, rset);
    gpgme_data_release (plain); plain = NULL;
    gpgme_recipients_release (rset); rset = NULL;
    if ( !cipher ) 
        goto failure;

    /* we have the encrypted message available in cipher and now we 
     * are going to rewrite the source file */
    rewind (fp);

    /* Write the header, append new content lines, part 1 and part 2 header */
    err = gpgme_data_rewind ( header );
    g_assert (!err);
    while ( !(err = gpgme_data_read ( header, buf, BUFFSIZE, &nread )) ) {
        fwrite ( buf, nread, 1, fp );
    }
    if (err != GPGME_EOF) {
        g_warning ("** gpgme_data_read failed: %s", gpgme_strerror (err));
        goto failure;
    }
    if ( ferror (fp) ) {
        FILE_OP_ERROR (file, "fwrite");
        goto failure;
    }
    gpgme_data_release (header); header = NULL;
    
    if (!mime_version_seen) 
        fputs ( "MIME-Version: 1\r\n", fp );
        
    fputs ( "Content-Type: multipart/encrypted; "
                           "protocol=\"application/pgp-encrypted\";\r\n"
            " boundary=\"x42x42x42x42x42x\"\r\n"
            "\r\n"
            "--x42x42x42x42x42x\r\n"
            "Content-Type: application/pgp-encrypted\r\n"
            "\r\n"
            "Version: 1\r\n"
            "\r\n"
            "--x42x42x42x42x42x\r\n"
            "Content-Type: application/octet-stream\r\n"
            "\r\n"
            , fp );

    /* append the encrypted stuff */
    err = gpgme_data_rewind ( cipher );
    if (err) {
        g_warning ("** gpgme_data_rewind on cipher failed: %s",
                   gpgme_strerror (err));
        goto failure;
    }
                   
    while ( !(err = gpgme_data_read ( cipher, buf, BUFFSIZE, &nread )) ) {
        fwrite ( buf, nread, 1, fp );
    }
    if (err != GPGME_EOF) {
        g_warning ("** gpgme_data_read failed: %s", gpgme_strerror (err));
        goto failure;
    }

    /* and the final boundary */
    fputs ( "\r\n"
            "--x42x42x42x42x42x--\r\n"
            "\r\n"
            , fp );
    fflush (fp);
    if ( ferror (fp) ) {
        FILE_OP_ERROR (file, "fwrite");
        goto failure;
    }
    fclose (fp);
    gpgme_data_release (cipher);
    return 0;

  failure:
    if (fp) 
        fclose (fp);
    gpgme_data_release (header);
    gpgme_data_release (plain);
    gpgme_data_release (cipher);
    gpgme_recipients_release (rset);
    return -1; /* error */
}

#endif /*USE_GPGME*/
