/*
 * Sylpheed -- a GTK+ based, lightweight, and fast e-mail client
 *
 * Copyright (c) 2000 by Alfons Hoogervorst <alfons@proteus.demon.nl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "intl.h"
#include "defs.h"

#include <glib.h> 
#include <gtk/gtkmain.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkeditable.h>

#include <string.h>
#include <ctype.h>
#if (HAVE_WCTYPE_H && HAVE_WCHAR_H)
#  include <wchar.h>
#  include <wctype.h>
#endif

#include "xml.h"
#include "addr_compl.h"
#include "utils.h"
#include "addressbook.h"
#include "main.h"

#define LOG_MESSAGE \
	debug_mode == 0 ? (debug_mode == debug_mode) : (void)debug_print 

/* How it works:
 *
 * The address book is read into memory. We set up an address list
 * containing all address book entries. Next we make the completion
 * list, which contains all the completable strings, and store a
 * reference to the address entry it belongs to.
 * After calling the g_completion_complete(), we get a reference
 * to a valid email address.  
 *
 * Completion is very simplified. We never complete on another prefix,
 * i.e. we neglect the next smallest possible prefix for the current
 * completion cache. This is simply done so we might break up the
 * addresses a little more (e.g. break up alfons@proteus.demon.nl into
 * something like alfons, proteus, demon, nl; and then completing on
 * any of those words).
 */ 
	
/* address_entry - structure which refers to the original address entry in the
 * address book 
 */
typedef struct
{
	gchar *name;
	gchar *address;
} address_entry;

/* completion_entry - structure used to complete addresses, with a reference
 * the the real address information.
 */
typedef struct
{
	gchar		*string; /* string to complete */
	address_entry	*ref;	 /* address the string belongs to  */
} completion_entry;

/*******************************************************************************/

static gint	    g_ref_count;	/* list ref count */
static GList 	   *g_completion_list;	/* list of strings to be checked */
static GList 	   *g_address_list;	/* address storage */
static GCompletion *g_completion;	/* completion object */

/* To allow for continuing completion we have to keep track of the state
 * using the following variables. No need to create a context object. */

static gint	    g_completion_count;		/* nr of addresses incl. the prefix */
static gint	    g_completion_next;		/* next prev address */
static GSList	   *g_completion_addresses;	/* unique addresses found in the
						   completion cache. */
static gchar	   *g_completion_prefix;	/* last prefix. (this is cached here
						 * because the prefix passed to g_completion
						 * is g_strdown()'ed */

/*******************************************************************************/

/* completion_func() - used by GTK to find the string data to be used for 
 * completion 
 */
static gchar *completion_func(gpointer data)
{
	g_assert(data);
	return ((completion_entry *) data)->string;
} 

static void init_all(void)
{
	g_completion = g_completion_new(completion_func);
	g_assert(g_completion);
}

static void free_all(void)
{
	GList *walk;
	
	walk = g_list_first(g_completion_list);
	for ( ; walk != NULL; walk = g_list_next(walk) ) {
		completion_entry *ce = (completion_entry *) walk->data;
		g_free(ce->string);
		g_free(walk->data);
	}
	g_list_free(g_completion_list);
	g_completion_list = NULL;
	
	walk = g_address_list;
	for ( ; walk != NULL; walk = g_list_next(walk) ) {
		address_entry *ae = (address_entry *) walk->data;
		g_free(ae->name);
		g_free(ae->address);
		g_free(walk->data);
	}
	g_list_free(g_address_list);
	g_address_list = NULL;
	
	g_completion_free(g_completion);
	g_completion = NULL;
}

/* add_address() - adds address to the completion list. this function looks
 * complicated, but it's only allocation checks.
 */
static gint add_address(const gchar *name, const gchar *address)
{
	address_entry    *ae;
	completion_entry *ce1;
	completion_entry *ce2;

	if (!name || !address) return -1;

	ae = g_new0(address_entry, 1);
	ce1 = g_new0(completion_entry, 1),
	ce2 = g_new0(completion_entry, 1);

	g_return_val_if_fail(ae != NULL, -1);
	g_return_val_if_fail(ce1 != NULL && ce2 != NULL, -1);	

	ae->name    = g_strdup(name);
	ae->address = g_strdup(address);		
	ce1->string = g_strdup(name);
	ce2->string = g_strdup(address);

	if ( ae->name && ae->address && ce1->string && ce2->string ) {
		/* GCompletion list is case sensitive */
		g_strdown(ce2->string);
		g_strdown(ce1->string);
		ce1->ref = ce2->ref = ae;
		g_completion_list = g_list_append(g_completion_list, (gpointer)ce1);
		g_completion_list = g_list_append(g_completion_list, (gpointer)ce2);
		g_address_list 	  = g_list_append(g_address_list,    (gpointer)ae);
		return 0;
	}

	g_warning( "%s(%d) - Error allocating memory\n", __FILE__, __LINE__ );

	/* need to clean up any valid allocated string */
	if (ae) {
		g_free(ae->name);
		g_free(ae->address);
		g_free(ae);
	}
	if (ce1) {
		g_free(ce1->string);
		g_free(ce1);
	}
	if (ce2) {
		g_free(ce2->string);
		g_free(ce2);
	}

	return -1;
}

static gboolean get_all_addresses(GNode *node, gpointer ae)
{
	XMLNode *xmlnode = (XMLNode *) node->data;
	address_entry *addr = (address_entry *) ae;

	/* this simply checks if tag is "item". in that case, it
	 * verifies it has already seen an item. if it did, an
	 * address retrieval was complete */
	if ( strcmp( xmlnode->tag->tag, "item" ) == 0 ) {
		/* see if a previous item was complete */
		/* TODO: does sylpheed address book allow empty names to be entered?
		 * if so, addr->name *AND* addr->address should be checked. */
		/* add address to our database */
		add_address(addr->name, addr->address);
		g_free(addr->name);
		g_free(addr->address);
		addr->name = NULL;
		addr->address = NULL;
	}
	else if ( strcmp( xmlnode->tag->tag, "name" ) == 0 ) {
		addr->name = g_strdup(xmlnode->element);
	}
	else if ( strcmp( xmlnode->tag->tag, "address" ) == 0 ) {
		addr->address = g_strdup(xmlnode->element);
	}

	return FALSE;
}

/* read_address_book()
 */ 
static void read_address_book(void)
{	
	gchar *path;
	GNode *tree; 
	address_entry ad = {NULL, NULL};

	LOG_MESSAGE( _("%s%d entering read_address_book\n"), __FILE__, __LINE__);
	path = g_strconcat(get_rc_dir(), G_DIR_SEPARATOR_S, ADDRESS_BOOK, NULL);
	tree = xml_parse_file(path);
	g_free(path);
	if (!tree) {
		LOG_MESSAGE( _("%s(%d) no addressbook\n"), __FILE__, __LINE__);
		return;
	}

	/* retrieve all addresses */
	g_node_traverse(tree, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
			(GNodeTraverseFunc) get_all_addresses, &ad);
	/* still one pending? */
	if ( ad.name ) {
		add_address(ad.name, ad.address);
		if ( ad.name )
			g_free(ad.name);
		if ( ad.address ) 
			g_free(ad.address);
	}

	xml_free_tree(tree);

	LOG_MESSAGE( _("%s(%d) leaving read_address_book - OK\n"), __FILE__, __LINE__);
}

/* start_address_completion() - returns the number of addresses 
 * that should be matched for completion.
 */
gint start_address_completion(void)
{
	clear_completion_cache();
	if ( !g_ref_count ) {
		init_all();
		/* open the address book */
		read_address_book();
		/* merge the completion entry list into g_completion */
		g_completion_add_items(g_completion, g_completion_list);
	}
	g_ref_count++;
	LOG_MESSAGE( "start_address_completion ref count %d\n", g_ref_count );

	return g_list_length(g_completion_list);
}

/* get_address_from_edit() - returns a possible address (or a part)
 * from an entry box. To make life easier, we only look at the last valid address 
 * component; address completion only works at the last string component in
 * the entry box. 
 */ 
gchar *get_address_from_edit(GtkEntry *entry, gint *start_pos)
{
	const gchar *edit_text;
	gint cur_pos;
	wchar_t *wtext;
	wchar_t *wp;
	wchar_t rfc_mail_sep;
	gchar *str;

	edit_text = gtk_entry_get_text(entry);
	if (edit_text == NULL) return NULL;
	wtext = strdup_mbstowcs(edit_text);
	g_return_val_if_fail(wtext != NULL, NULL);
	cur_pos = gtk_editable_get_position(GTK_EDITABLE(entry));

	if (mbtowc(&rfc_mail_sep, ",", 1) < 0) {
		g_free(wtext);
		return NULL;
	}

	/* scan for a separator. doesn't matter if walk points at null byte. */
	for (wp = wtext + cur_pos; wp > wtext && *wp != rfc_mail_sep; wp--)
		;

	/* have something valid */
	if (wcslen(wp) == 0) {
		g_free(wtext);
		return NULL;
	}

#define IS_VALID_CHAR(x)	(iswalnum(x) || ((x) > 0x7f))

	/* now scan back until we hit a valid character */
	for ( ; *wp && !IS_VALID_CHAR(*wp); wp++ )
		;

#undef IS_VALID_CHAR

	if (wcslen(wp) == 0) {
		g_free(wtext);
		return NULL;
	}

	if (start_pos) *start_pos = wp - wtext;

	str = strdup_wcstombs(wp);
	g_free(wtext);

	return str;
} 

/* replace_address_in_edit() - replaces an incompleted address with a completed one.
 */
void replace_address_in_edit(GtkEntry *entry, const gchar *newtext,
			     gint start_pos)
{
	gtk_editable_delete_text(GTK_EDITABLE(entry), start_pos, -1);
	gtk_editable_insert_text(GTK_EDITABLE(entry), newtext, strlen(newtext),
				 &start_pos);
	gtk_editable_set_position(GTK_EDITABLE(entry), -1);
}

/* complete_address() - tries to complete an addres, and returns the
 * number of addresses found. use get_complete_address() to get one.
 * returns zero if no match was found, otherwise the number of addresses,
 * with the original prefix at index 0. 
 */
guint complete_address(const gchar *str)
{
	GList *result;
	gchar *d = g_strdup(str);
	guint  count, cpl;
	completion_entry *ce;
	
	g_assert(d);
	if ( d == NULL ) {
		LOG_MESSAGE("%s(%d) - allocation error\n", __FILE__, __LINE__);
		return 0;
	}
	
	clear_completion_cache();
	g_completion_prefix = g_strdup(str);
	if ( g_completion_prefix == NULL ) {
		LOG_MESSAGE("%s(%d) - allocation error\n", __FILE__, __LINE__);
		g_free(d);
		return 0;
	}
	
	/* g_completion is case sensitive */
	g_strdown(d);
	result = g_completion_complete(g_completion, d, NULL);
	g_free(d);	
	count = g_list_length(result);
	if ( count ) {
		/* create list with unique addresses  */
		for ( cpl = 0, result = g_list_first(result)
			; result != NULL
			; result = g_list_next(result)  ) {
			ce = (completion_entry *)(result->data);
			if ( NULL == g_slist_find(g_completion_addresses, (gpointer)(ce->ref)) ) {
				cpl++;
				g_completion_addresses = g_slist_append(g_completion_addresses, (gpointer)(ce->ref));
			}
		}
		count = cpl + 1;		/* index 0 is the original prefix */
		g_completion_next = 1;	/* we start at the first completed one */
	}
	else {
		g_free(g_completion_prefix);
		g_completion_prefix = NULL;
	}
	
	g_completion_count = count;
	return count;
}

/* get_complete_address() - returns a complete address. the returned
 * string should be freed 
 */
gchar *get_complete_address(gint index)
{
	const address_entry *p;
	
	if ( index < g_completion_count ) {
		if ( index == 0 ) {
			return g_strdup(g_completion_prefix);
		}
		else {
			/* get something from the unique addresses */
			p = (address_entry *)g_slist_nth_data(g_completion_addresses, index - 1);
			if ( p == NULL ) {
				return NULL;
			}
			else {
				return g_strdup_printf("%s <%s>", p->name, p->address);
			}
		}
	}
	else {
		return NULL;
	}
}

gchar *get_next_complete_address(void)
{
	if ( is_completion_pending() ) {
		gchar *res = get_complete_address(g_completion_next);
		g_completion_next += 1;
		if ( g_completion_next >= g_completion_count )
			g_completion_next = 0;
		return res;
	}
	else {
		return NULL;
	}
}

gchar *get_prev_complete_address(void)
{
	if ( is_completion_pending() ) {
		int n = g_completion_next - 2;

		/* real previous */
		n = (n + (g_completion_count * 5)) % g_completion_count;

		/* real next */
		g_completion_next = n + 1;
		if ( g_completion_next >=  g_completion_count ) {
			g_completion_next = 0;
		}			
		return get_complete_address(n);
	}
	else {
		return NULL;
	}
}

/* should clear up anything after complete_address() */
void clear_completion_cache(void)
{
	if ( is_completion_pending() ) {
		if ( g_completion_prefix ) 
			g_free(g_completion_prefix);
		if ( g_completion_addresses ) {	
			g_slist_free(g_completion_addresses);
			g_completion_addresses = NULL;
		}			
		g_completion_count = g_completion_next = 0;
	}
}

gboolean is_completion_pending(void)
{
	/* check if completion pending, i.e. we might satisfy a request for the next
	 * or previous address */
	 return g_completion_count;
}

/* invalidate_address_completion() - should be called if address book
 * changed; 
 */
gint invalidate_address_completion(void)
{
	if ( g_ref_count ) {
		/* simply the same as start_address_completion() */
		LOG_MESSAGE("Invalidation request for address completion\n");
		free_all();
		init_all();
		read_address_book();
		g_completion_add_items(g_completion, g_completion_list);			
		clear_completion_cache();
	}
	else {
	}
	return g_list_length(g_completion_list);
}

/* end_address_completion_end()
 */
gint end_address_completion(void)
{
	clear_completion_cache();
	if ( 0 == --g_ref_count )
		free_all();
	LOG_MESSAGE("end_address_completion ref count %d\n", g_ref_count);
	return g_ref_count; 
} /* sylpheed_address_completion_end() */
