/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007-2009 Jens Baaran, Germany.
 ******************************************************************************/

#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <libgen.h>
#include <pthread.h>
#include "sudoku.h"

extern int action;
extern int max_rating;
extern int verbosity;
extern int nthread;
extern char fn_out[512];
extern char progname[100];
extern FILE *explFile;
extern FILE *devurandom;
extern FILE *inFile;

/*
 ******************************************************************************/
int main (int argc, char *argv[]) {
	
	clock_t t0;
	float ct1;
	int i, elim[NUM_ET], ret, d0, d1;
	t_sudoku sudoku, foursudokus[4];
	extern int num_sudokus;

	void ***targv;
	pthread_t *thread;
	pthread_attr_t *attr_p;
	int ithread, nsud, rv, trv;
	
	// not tested for SIZEs != 3
	if (SIZE != 3) {
		fprintf(stderr,"SIZE must be 3, please recompile. Aborting ...\n");
		abort();
	}
	
	// progname
	strncpy(progname,basename(argv[0]),100);
	
	// explanations go to stdout, input of sudokus from stdin by default
	explFile = stdout;
	inFile = stdin;

	// parse options
	// -------------
	if ((argc == 1) || (getoptions(argc, argv) != 0)) {
		fflush(stderr);
		print_usage(stdout);
		return(0);
	}
  	
	// command -c creates sudokus
	// --------------------------
	if (action == 1) {
		devurandom = fopen("/dev/urandom","rb");
		fprintf(stderr,"\nCreating %d sudokus\n",num_sudokus);

		if (nthread < 1) {
			
			for (i=0; i<num_sudokus; i++) {
				reset_sudoku(&(foursudokus[i%4]));
				foursudokus[i%4].rating = create_sudoku(&(foursudokus[i%4]),verbosity);
			}
			if (num_sudokus == 4) sudokus_to_pdf(fn_out,foursudokus);
			sudokus_to_svg(fn_out,foursudokus);
		} else {

			// allocate
			thread = (pthread_t*) calloc(nthread,sizeof(pthread_t));
			targv  = (void***)   calloc(nthread,sizeof(void**));
			for (ithread=0; ithread<nthread; ithread++) {
				targv[ithread] = (void**) calloc(1,sizeof(void*));
				targv[ithread][0] = &verbosity;
			}
			attr_p = NULL;
			
			// start threads
			for (ithread=0; ithread<nthread; ithread++) {
				if ((rv = pthread_create(&(thread[ithread]), attr_p, (void *) execute_thread_create, (void *) targv[ithread])) != 0) {
					fprintf(stderr,"Error %d!\n%s\n",rv,strerror(rv));
				}
			}

			// wait for threads to terminate
			nsud = 0;
			for (ithread=0; ithread<nthread; ithread++) {
				if ((rv = pthread_join(thread[ithread], (void **) &trv)) != 0) {
					fprintf(stderr,"Error %d!\n%s\n",rv,strerror(rv));
				}
			}

			// message
			for (ithread=0; ithread<nthread; ithread++) {
				fprintf(stderr,"Thread %d produced %d sudokus.\n",ithread,trv);
				nsud += trv;
			}
			fprintf(stderr,"created %d sudokus in %d threads.\n",nsud,nthread);
		
			// free
			free(thread);
		}
		fclose(devurandom);
	}

	// comand -p solves & prints up to 4 sudokus on a
	// DIN A4 page (pdf), if they have a unique solution
	// -------------------------------------------------
	if (action == 2) {

		for (i=0; i<4; i++) {
			
			// read from stdin & initialize
			if (read_sudoku_from_file(inFile,&(foursudokus[i])) != 0) {
				reset_sudoku(&(foursudokus[i]));
				break;
			}

			// solve
			printf("\nSolving sudoku ");
			fprint_sudoku(stdout,foursudokus[i]);
			t0 = clock();
			copy_sudoku(foursudokus[i],&sudoku);
			ret = solve_sudoku(&sudoku,elim,NUM_ET-1,verbosity,1);
			ct1 = (clock() - t0) * (float)(1.0 / CLOCKS_PER_SEC);

			// check uniqueness & terminal output
			if (ret == 0) {
				if (check_uniqueness1(foursudokus[i],sudoku) == 0) {
					foursudokus[i].rating = sudoku.rating;
					printf("\nFound unique solution\nsolution time: %d msec\n",(int)(ct1*1000));
					// if sudoku is symmetric, beautify it
					if (check_sudoku_symmetry(foursudokus[i], &d0, &d1) != symm_none) {
						beautify_symmetric_sudoku(&(foursudokus[i]));
					}
				} else {
					printf("\nThis sudoku has more than one solution, resetting grid.\n");
					reset_sudoku(&(foursudokus[i]));
				}
			} else {
				printf("\nNo solution found, resetting grid.\n");
				reset_sudoku(&(foursudokus[i]));
			}
		}
		
		// print svg and pdf
		if (DEBUG) fprintf(stderr,"printing ...\n");
		sudokus_to_pdf(fn_out,foursudokus);
		sudokus_to_svg(fn_out,foursudokus);
		if (DEBUG) fprintf(stderr,"done printing\n");	
	}
	
	// command -s solves sudoku
	// ------------------------
	if (action == 3) {

		fprintf(stderr,"\nSolving sudoku\n");
		
		if (nthread < 1) {

			solve_sudoku_and_check_uniqueness(inFile,verbosity);

		} else {
			
			// allocate & initialize
			thread       = (pthread_t*) calloc(nthread,sizeof(pthread_t));
			targv        = (void***)    calloc(nthread,sizeof(void**));
			for (ithread=0; ithread<nthread; ithread++)
				targv[ithread] = (void**) calloc(2,sizeof(void*));
			attr_p = NULL;

			// start threads
			for (ithread=0; ithread<nthread; ithread++) {
				targv[ithread][0] = inFile;
				targv[ithread][1] = &verbosity;
				if ((rv = pthread_create(&(thread[ithread]), attr_p, (void *) execute_thread_solve, (void *) targv[ithread])) != 0) {
					fprintf(stderr,"Error %d!\n%s\n",rv,strerror(rv));
				}
			}

			// wait for threads to terminate
			nsud = 0;
			for (ithread=0; ithread<nthread; ithread++) {
				if ((rv = pthread_join(thread[ithread], (void **) &trv)) != 0) {
					fprintf(stderr,"Error %d!\n%s\n",rv,strerror(rv));
				}
				fprintf(stderr,"Thread %d solved %d sudokus.\n",ithread,trv);
				nsud += trv;
			}

			// message
			fprintf(stderr,"solved %d sudokus in %d threads.\n",nsud,nthread);
		
			// free
			for (ithread=0; ithread<nthread; ithread++) free(targv[ithread]);
			free(targv);
			free(thread);

		}
		
	}

	// command -w prints html page of sudoku with candidates and values
	// ----------------------------------------------------------------
	if (action == 4) {
		if (read_sudoku_and_candidates_from_file(inFile,&sudoku) != 0) {
			return(0);
		}
		print_sudoku_and_candidates_as_html(fn_out,sudoku);
	}

	// command -N prints canonized sudokus to stdout
	// ---------------------------------------------
	if (action == 5) {
		while (read_sudoku_from_file(inFile,&sudoku) == 0) {
			fprint_sudoku(stdout,sudoku);
			fprintf(stdout,"\t");
			canoni(sudoku,&(foursudokus[0]));
			fprint_sudoku(stdout,foursudokus[0]);
			fprintf(stdout,"\n");
		}
	}
	
	// done
	if ((inFile != NULL) && (inFile != stdin)) fclose(inFile);
	return 0;
}

