/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007-2009 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "sudoku.h"

/******************************************************************************
 * 
 * 
 * functions for adding text and graphics to pdf page
 * 
 * 
 ******************************************************************************/ 

/* draw pdf line, returns number of characters printed to pdf file
 ******************************************************************************/
int pdf_draw_line(FILE *fn, float lw, int xs, int ys, int xe, int ye) {
	char s[1000];
	int len = snprintf(s,1000,"%3.1f w %d %d m %d %d l S\n",lw,xs,ys,xe,ye);
	fprintf(fn,"%s",s);
	return(len);
}

/* draw pdf text, returns number of characters written to pdf file
 ******************************************************************************/
int pdf_draw_text(FILE *fn, int x, int y, char *t) {
	char s[1100];
	int len = 0;
	if (strlen(t) > 1000) {
		fprintf(stderr,"ERROR: pdf string too long!\n");
	} else {
		len = snprintf(s,1100,"BT %d %d Td (%s) Tj ET\n",x,y,t);
		fprintf(fn,"%s",s);
	}
	return(len);
}

/* set pdf font
 ******************************************************************************/
int pdf_set_font(FILE *fn, char *name, int size) {
	char s[1000];
	int len = snprintf(s,1000,"%s %d Tf\n",name,size);
	fprintf(fn,"%s",s);
	return(len);
}

/* set grayscale color
 ******************************************************************************/
int pdf_set_gray_stroke(FILE *fn, float g) {
	char s[1000];
	int len = snprintf(s,1000,"%4.2f g\n",g);
	fprintf(fn,"%s",s);
	return(len);
}

/******************************************************************************
 * 
 * 
 * functions for printing pdf-objects to pdf file
 * 
 * 
 ******************************************************************************/ 

/* print pdf header
 ******************************************************************************/
int pdf_print_head(FILE *fn) {
	char s[1000];
	int len = snprintf(s,1000,"%%PDF-1.4\n");
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_head\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print catalog object
 ******************************************************************************/
int pdf_print_obj_catalog(FILE *fn, int oid, int pid) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n<<\n/Type /Catalog\n/Pages %d 0 R\n>>\nendobj\n",oid,pid);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_catalog\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print pages object
 ******************************************************************************/
int pdf_print_obj_pages(FILE *fn, int oid, int kid) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n<<\n/Type /Pages\n/Kids [ %d 0 R ]\n/Count 1\n>>\nendobj\n",oid,kid);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_pages\n");
	fprintf(fn,"%s",s);
	return(len);
}
	
/* print info object
 ******************************************************************************/
int pdf_print_obj_info(FILE *fn, int oid) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n<<\n/Producer (sudognu)\n>>\nendobj\n",oid);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_info\n");
	fprintf(fn,"%s",s);
	return(len);
}
	
/* print page object, including bounding box
 ******************************************************************************/
int pdf_print_obj_page(FILE *fn, int oid, int cid, char *fname, int fid, int pid, int bb[4]) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n<<\n/Type /Page\n/MediaBox [ %d %d %d %d ]\n/Contents %d 0 R\n/Resources <<\n/ProcSet [ /PDF /Text /ImageB /ImageC /ImageI ]\n/Font <<\n%s %d 0 R\n>>\n>>\n/Parent %d 0 R\n>>\nendobj\n",oid,bb[0],bb[1],bb[2],bb[3],cid,fname,fid,pid);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_page\n");
	fprintf(fn,"%s",s);
	return(len);
}
	
/* print length object
 ******************************************************************************/
int pdf_print_obj_length(FILE *fn, int oid, int l) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n%d\nendobj\n",oid,l);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_length\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print font object
 ******************************************************************************/
int pdf_print_obj_font(FILE *fn, int oid, char *bf, char *st, char *e) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n<<\n/Type /Font\n/BaseFont %s\n/Subtype %s\n/Encoding %s\n>>\nendobj\n",oid,bf,st,e);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_font\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print header of content object
 ******************************************************************************/
int pdf_print_obj_conthead(FILE *fn, int oid, int lid) {
	char s[1000];
	int len = snprintf(s,1000,"%d 0 obj\n<<\n/Length %d 0 R\n>>\nstream\n",oid,lid);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_conthead\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print footer of content object
 ******************************************************************************/
int pdf_print_obj_contfoot(FILE *fn) {
	char s[1000];
	int len = snprintf(s,1000,"endstream\nendobj\n");
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_obj_contfoot\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print crossref table
 ******************************************************************************/
int pdf_print_xref(FILE *fn, int l_obj[8]) {
	char s[1000], st[100];
	int i, len;
	snprintf(s,100,"xref\n0 8\n%010d %05d f \n",0,65535);
	for (i=0; i<7; i++) {
		snprintf(st,100,"%010d %05d n \n",l_obj[i],0);
		strncat(s,st,100);
	}
	len = strlen(s);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_xref\n");
	fprintf(fn,"%s",s);
	return(len);
}

/* print trailer
 ******************************************************************************/
int pdf_print_trailer(FILE *fn, int start_crossref) {
	char s[1000];
	int len = snprintf(s,1000,"trailer\n<<\n/Root 1 0 R\n/Info 3 0 R\n/Size 8\n>>\nstartxref\n%d\n%%%%EOF",start_crossref);
	if (len == 999)
		fprintf(stderr,
				"WARNING: string buffer may be too short in pdf_print_trailer\n");
	fprintf(fn,"%s",s);
	return(len);
}

/******************************************************************************
 * 
 * 
 * print sudoku to pdf file
 * 
 * 
 ******************************************************************************/ 

int grid_to_pdf(
	t_field grid[SIZE2][SIZE2], int size, char *title,
	FILE *f, char* font, int x0, int y0, int sqsize, float linewidth) {

	int fontsize  = (2 * (sqsize + 1)) / 3;
	int fontoffsy = (2 * (sqsize - fontsize + 1)) / 3;
	int fontoffsx = (3 * fontoffsy) / 2;
	int sudsize   = sqsize*size, x, y, origin[2]={x0, y0};
	int i, j;
	char buf[10];
	int l = 0;

	// set line colour
	l += pdf_set_gray_stroke(f,0.0);
	
	// print sudoku title a little above sudoku
	if (DEBUG) fprintf(stderr,"  draw title ...\n");
	l += pdf_set_font(f, font, fontsize/2);
	l += pdf_draw_text(f, origin[0] + fontoffsx, origin[1] + fontoffsy + sudsize, title);

	// print horizontal lines
	if (DEBUG) fprintf(stderr,"  draw horizontal lines and numbers ...\n");
	y = origin[1];
	for (i=0; i<10; i++) {
		if ((i%3) == 0) {
			l += pdf_draw_line(f, 3*linewidth, origin[0], y, origin[0] + sudsize, y);
		} else {
			l += pdf_draw_line(f,   linewidth, origin[0], y, origin[0] + sudsize, y);
		}
		y += sqsize;
	}
	
	// print vertical lines
	if (DEBUG) fprintf(stderr,"  draw vertical lines ...\n");
	x = origin[0];
	for (i=0; i<10; i++) {
		if ((i%3) == 0) {
			l += pdf_draw_line(f, 3*linewidth, x, origin[1], x, origin[1] + sudsize);
		} else {
			l += pdf_draw_line(f,   linewidth, x, origin[1], x, origin[1] + sudsize);
		}
		x += sqsize;
	}
	
	// print numbers from top to bottom and from left to right
	if (DEBUG) fprintf(stderr,"  draw numbers ...\n");
	l += pdf_set_font(f, font, fontsize);
	for (i=0; i<9; i++) {
		for (j=0; j<9; j++) {
			if (grid[i][j].value > 0) {
				snprintf(buf, 12, "%d", grid[i][j].value);
				l += pdf_draw_text(f, origin[0] + j*sqsize + fontoffsx, origin[1] - i*sqsize + fontoffsy + sudsize - sqsize, buf);
			}
		}
	}
	if (DEBUG) fprintf(stderr,"  ... done printing\n");
	
	return l;
}

/******************************************************************************
 * 
 * 
 * print pdf document
 * 
 * 
 ******************************************************************************/ 

void sudokus_to_pdf(char *filename, t_sudoku sudoku[4]) {
	/* progname for printing info */
	extern char progname[100];

	/* define object ids */
	int id_cat    = 1;
	int id_pages  = 2;
	int id_info   = 3;
	int id_page   = 4;
	int id_cont   = 5;
	int id_length = 6;
	int id_font   = 7;
	
	/* define bounding box */
	int bb[4] = { 0, 0, 604, 841 };
	
	/* define a font id string, define font properties */
	char font_str[]  = "/F1";
	char font_name[] = "/Helvetica";
	char font_type[] = "/Type1";
	char font_enc[]  = "/StandardEncoding";
	
	/* need to keep track of number of chars of each object and the number of
	 *  chars of the content stream */
	int l_cont, l_obj[8];
	
	/* some useful identifiers */
	char buf[4][60], dstr[1000];
	int i;
	int size = sudoku[0].size;
	float eps = 1.e-2;
	int height = bb[3] - bb[1]; 	/* default resolution is 72 dpi */
	int width  = bb[2] - bb[0];
	int sqsize = (int) (width * 5.0 / 9.0 / 11.0);
	int fontsize;
	int x, y, dx, dy;
		
	/* pdf file name */
	char fname[512] = "sudoku";
	if (strlen(filename) != 0) strncpy(fname,filename,511);
	strcat(fname,".pdf");
	FILE *f;
	
	/* open pdf-file */
	f = fopen(fname,"w");

	/* print head, document structure information, etc. up to beginning of
	 * the content stream */
	l_obj[0]        = pdf_print_head(f);
	l_obj[id_cat]   = l_obj[0]        + pdf_print_obj_catalog(f, id_cat, id_pages);
	l_obj[id_pages] = l_obj[id_cat]   + pdf_print_obj_pages(f, id_pages, id_page);
	l_obj[id_info]  = l_obj[id_pages] + pdf_print_obj_info(f, id_info);
	l_obj[id_page]  = l_obj[id_info]  + pdf_print_obj_page(f, id_page, id_cont, font_str, id_font, id_pages, bb);
	l_obj[id_cont]  = l_obj[id_page]  + pdf_print_obj_conthead(f, id_cont, id_length);
	
	/* print content stream
	 ************************/

	// offset parameters for the sudokus
	y = (int) ((height - 2.0 * size * sqsize)/6.0);
	x = (int) ((width  - 2.0 * size * sqsize)/3.0);
	dx = size * sqsize + x;
	dy = size * sqsize + 2 * sqsize;

	// strings for rating of sudokus
	for (i=0; i<4; i++) {
		if      (sudoku[i].rating > 200-eps) strncpy(dstr,"fiendish",10);
		else if (sudoku[i].rating > 100-eps) strncpy(dstr,"evil",10);
		else if (sudoku[i].rating >  10-eps) strncpy(dstr,"hard",10);
		else if (sudoku[i].rating >   2-eps) strncpy(dstr,"moderate",10);
		else if (sudoku[i].rating >   1-eps) strncpy(dstr,"easy",10);
		else strncpy(dstr,"",10);
		snprintf(buf[i], 60, "Sudoku %d, difficulty %d - %s",i+1,(int)sudoku[i].rating,dstr);
	}

	// print the 4 grids
	if (sudoku[0].rating > eps) l_cont  = grid_to_pdf(sudoku[0].grid, size, buf[0], f, font_str, x   , y+dy, sqsize, 0.5);
	if (sudoku[1].rating > eps) l_cont += grid_to_pdf(sudoku[1].grid, size, buf[1], f, font_str, x+dx, y+dy, sqsize, 0.5);
	if (sudoku[2].rating > eps) l_cont += grid_to_pdf(sudoku[2].grid, size, buf[2], f, font_str, x   , y   , sqsize, 0.5);
	if (sudoku[3].rating > eps) l_cont += grid_to_pdf(sudoku[3].grid, size, buf[3], f, font_str, x+dx, y   , sqsize, 0.5);
	
	// print a nice header
	if (DEBUG) fprintf(stderr,"  print header ...\n");
	strcpy(dstr,"SUDOKU");
	fontsize = (int) (height / 12.0);
	y        = (int) (height - fontsize - fontsize / 2);
	x        = (int) (width / 2.0 - (fontsize * strlen(dstr) * 2) / 6.0);
	l_cont += pdf_set_font(f, font_str, fontsize);
	l_cont += pdf_draw_text(f, x, y, dstr);
	if (DEBUG) fprintf(stderr,"  done printing header\n");
	
	// print small program info at bottom
	if (DEBUG) fprintf(stderr,"  print info ...\n");
	snprintf(dstr,999,"printed by %s version %s -- http://sudognu.sourceforge.net",progname,VERSION);
	fontsize = (int) (fontsize / 10);
	y        = (int) ((height - 2.0 * size * sqsize)/6.0 - 1.5 * fontsize);
	x        = (int) ((width  - 2.0 * size * sqsize)/3.0 + 0.5 * fontsize);
	l_cont += pdf_set_font(f, font_str, fontsize);
	l_cont += pdf_draw_text(f, x, y, dstr);
	if (DEBUG) fprintf(stderr,"  done printing info\n");

	/* done printing content stream
	 ******************************/
	
	/* print footer of the content stream, its length, and define font */
	l_obj[id_cont]  += pdf_print_obj_contfoot(f);
	l_obj[id_cont]  += l_cont;
	l_obj[id_length] = l_obj[id_cont] + pdf_print_obj_length(f, id_length, l_cont);
	l_obj[id_font]   = l_obj[id_length] + pdf_print_obj_font(f, id_font, font_name, font_type, font_enc);

	/* print cross reference table and trailer */
	pdf_print_xref(f, l_obj);
	pdf_print_trailer(f, l_obj[id_font]);

	/* done. close file */
	fclose(f);
}
