/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>
#include "sudoku.h"

extern pthread_mutex_t mut_input;
extern pthread_mutex_t mut_stdout;

/*
 ******************************************************************************/
int execute_thread_solve(void *arg[]) {
	
	FILE *in       = (FILE *) arg[0];
	int *verbosity = (int *) arg[1];

	// set verbosity to 0 for threads, because mutex protection for output after
	// every solution step doesn't seem feasible. maybe each thread needs its
	// own output file.
	if (*verbosity > 0) {
		return(solve_sudoku_and_check_uniqueness(in,1));
	} else {
		return(solve_sudoku_and_check_uniqueness(in,0));
	}
}

/*
 ******************************************************************************/
int solve_sudoku_and_check_uniqueness(FILE *in, int verbosity) {
	
	int nsolved=0;
	int elim[NUM_ET], ret;
	t_sudoku sudoku, dsudoku;
	
	reset_sudoku(&sudoku);

	while(1) {

		// read from input
		pthread_mutex_lock(&mut_input);
		if (read_sudoku_from_file(in,&sudoku) != 0) {
			pthread_mutex_unlock(&mut_input);
			return(nsolved);
		}
		pthread_mutex_unlock(&mut_input);

		// solve sudoku
		copy_sudoku(sudoku,&dsudoku);
		ret = solve_sudoku(&dsudoku,elim,NUM_ET-1,verbosity,1);

		// check uniqueness & terminal output
		if (ret == 0) {
			if (check_uniqueness1(sudoku,dsudoku) == 0) {
				pthread_mutex_lock(&mut_stdout);
				switch (verbosity) {
					case 1:
						fprint_sudoku(stdout, dsudoku);
						printf("\n");
//						fflush(stdout);
						break;
					case 2:
						printf("\t%d\t%d\t%s\n",get_num_cell_with_stat1(sudoku,stat_ggiven) + get_num_cell_with_stat1(sudoku,stat_given),(int)dsudoku.rating,VERSION);
						break;
				}
				pthread_mutex_unlock(&mut_stdout);
			} else {
//				fflush(stdout);
				fprintf(stderr,"\n\nSudoku has more than one solution\n");
				if (DEBUG) fprint_sudoku(stderr,sudoku);
				exit(4);
			}
		} else if (ret != 0) {
//			fflush(stdout);
			fprintf(stderr,"\n\nNo solution found for sudoku\n");
			fprint_sudoku(stderr,sudoku);
			exit(3);
		}
		nsolved++;
	}
	return(nsolved);
}

/*
 ******************************************************************************/
int solve_sudoku(t_sudoku *sudoku, int elim[NUM_ET], int depth, int verbosity, int doRate) {
	
	int i, ret;
	t_sudoku tmpSudoku, origSudoku;
	
	for (i=0; i<NUM_ET; i++) elim[i] = 0;
	get_candidates(sudoku);
	copy_sudoku(*sudoku,&tmpSudoku);
	copy_sudoku(*sudoku,&origSudoku);
	if (verbosity > 9) printf("\n");
	ret = solve_sudoku_rec(sudoku,&tmpSudoku,elim,0,0,0,depth,verbosity, doRate);
	if (verbosity == 2) {
		printf("\t");
		fprint_sudoku(stdout,origSudoku);
	}
	if (ret == 0) {
		if (verbosity == 2) {
			canoni(origSudoku,&tmpSudoku);
			printf("\t");
			fprint_sudoku(stdout,tmpSudoku);
		}
		if (doRate == 1) {
			sudoku->rating += get_basic_rating(*sudoku,elim);
			if ((verbosity > 2) && (verbosity < 11)) {
				fprintf(stderr,"\n basic rating: \t%8.1f\n",(float)get_basic_rating(*sudoku,elim));
				fprintf(stderr,"sudoku rating: \t%8.1f\n",sudoku->rating);
			}
		}
	}
	return(ret);
}

/*
 ******************************************************************************/
int check_uniqueness1(t_sudoku sudoku, t_sudoku ssudoku) {
	
	int size = sudoku.size;
	int i, r, c, ret, elim[NUM_ET];
	t_sudoku csudoku;
	t_sudoku dsudoku;
	t_sudoku esudoku;
	
	// work on copy of grid
	copy_sudoku(sudoku,&csudoku);
	get_candidates(&csudoku);

	// loop over all cells
	for (i=0; i<size*size; i++) {
		r = i / size;
		c = i % size;
		// for all cells with status stat_guess
		if (ssudoku.grid[r][c].stat == stat_guess) {
			copy_sudoku(csudoku,&dsudoku);
			// remove their solution from candidates
			dsudoku.grid[r][c].cand[ssudoku.grid[r][c].value-1] = 0;
			dsudoku.grid[r][c].ncand -= 1;
			// try to solve
			copy_sudoku(dsudoku,&esudoku);
			ret = solve_sudoku_rec(&dsudoku,&esudoku,elim,et_d,0,0,et_s,0,0);
			if (ret == 0) {
				if (DEBUG) printf("\nsolution is not unique.\n");
				return(-1);
			}
		}
	}
	if (DEBUG) printf("\nsolution is unique.\n");
	
	// no solution found: solution is unique
	return(0);
}

/*
 ******************************************************************************/
int solve_sudoku_rec(t_sudoku *sudoku, t_sudoku *dsudoku,
		int elim[NUM_ET], int r, int c, int cc, int depth, int verbosity, int doRate) {
	
	t_sudoku esudoku;
	int ret = 1, r1=-1, c1=-1, cc1=-1;
	int size = sudoku->size;
	int backtrack_depth;
	extern int dfcs[];
	
	// forcing chains don't work during backtracking
	backtrack_depth = 1;
	while ( (dfcs[backtrack_depth] != et_fc)
			&& (backtrack_depth <= depth)
			&& (backtrack_depth < (NUM_ET-1)) ) {
		backtrack_depth++;
	}
	backtrack_depth--;

	// dgrid is temp grid, if there have been guesses it contains latest guess
	// grid is original grid before last guess
	
	while(ret != 0) {

		ret = eliminate(dsudoku, elim, depth, verbosity, doRate); 

		if (ret > 0) {

			// find the most promising cell & candidate
			find_cell_and_cand_for_guess(dsudoku->grid,size,&r1,&c1,&cc1);
			// make sure indices are ok
			if ( (r1 >= size) || (r1 < 0) || (c1 >= size) || (c1 < 0) || (cc1 >= size) || (cc1 < 0) ) {
				fprintf(stderr,"\n*** Error, could not find a cell for guess (r%dc%d cc%d.\n",r1+1,c1+1,cc1+1);
				print_sudoku_as_grid(*dsudoku);
				print_candidates_as_grid(*dsudoku);
				return(0);	
			}

			// guess
			if (verbosity == 2) printf("g");
			if (verbosity > 9)  printf("g  r%dc%d (%d) guess\n",r1+1,c1+1,cc1+1);
			// for every new guess make another copy of the grid
			dsudoku->grid[r1][c1].stat  = stat_guess;
			copy_sudoku(*dsudoku,&esudoku);
			// insert guess into new grid
			esudoku.grid[r1][c1].value = cc1+1;
			elim[et_g] += 1;
			// update candidates for temp grid
			remove_candidates(&esudoku,r1,c1);
			// try to solve temp grid
			ret = solve_sudoku_rec(dsudoku, &esudoku, elim, r1, c1, cc1, backtrack_depth, verbosity, doRate);
			dsudoku->rating = esudoku.rating;
			
		} else if (ret < 0) {

			// make sure indices are ok
			if ( (r >= size) || (r  < 0) || (c >= size) || (c  < 0) || (cc >= size) || (cc < 0) ) {
				fprintf(stderr,"\n*** Error, indices are not ok.\n");
				print_sudoku_as_grid(*dsudoku);
				print_candidates_as_grid(*dsudoku);
				return(0);	
			}
			
			if (sudoku->grid[r][c].stat == stat_guess) {
				// wrong guess, remove candidate from original grid
				if (verbosity == 2) printf("w");
				if (verbosity > 9)  printf("w  r%dc%d wrong guess, remove candidate %d\n",r+1,c+1,cc+1);
				sudoku->grid[r][c].cand[cc] = 0;
				sudoku->grid[r][c].ncand -= 1;
				elim[et_wg] += 1;
			}
			// return to previous level
			return(ret);

		}
		
		if (ret == 0) copy_sudoku(*dsudoku,sudoku);

	}
	
	return(ret);
}

/*
 ******************************************************************************/
int eliminate(t_sudoku *sudoku, int elim[NUM_ET], int depth, int verbosity, int doRate) {

	int i, j, e, ne = 2; // averaged difficulties of the (ne) easiest techniques determine difficulty of a solution step
	char str[1024] = "";
	extern int dfcs[];
	
	do {
		i = e = 0;
		if (doRate) {
//			for (j = et_hs; ((j <= et_jf) && (i < ne)); j++) {
			//				switch(j) {
			for (j=1; ( (j < NUM_ET-1) && (i < ne)); j++) {
				switch(dfcs[j]) {
				case et_hs:
					e = hidden_single(sudoku,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),".%d,",e);
					break;
				case et_s:
					e = single(sudoku,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"-%d,",e);
					break;
				case et_hd:
					e = hidden_tuple(sudoku,2,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"d%d,",e);
					break;
				case et_d:
					e = tuple(sudoku,2,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"D%d,",e);
					break;
				case et_b:
					e = block_line(sudoku,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"b%d,",e);
					break;
				case et_l:
					e = line_block(sudoku,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"B%d,",e);
					break;
				case et_ht:
					e = hidden_tuple(sudoku,3,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"t%d,",e);
					break;
				case et_t:
					e = tuple(sudoku,3,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"T%d,",e);
					break;
				case et_hq:
					e = hidden_tuple(sudoku,4,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"q%d,",e);
					break;
				case et_q:
					e = tuple(sudoku,4,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"Q%d,",e);
					break;
				case et_xy:
					e = xy_wing(sudoku,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"W%d,",e);
					break;
				case et_xyz:
					e = xyz_wing(sudoku,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"Y%d,",e);
					break;
				case et_xw:
					e = fish(sudoku,2,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"X%d,",e);
					break;
				case et_sf:
					e = fish(sudoku,3,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"S%d,",e);
					break;
				case et_jf:
					e = fish(sudoku,4,0,0);
					if ((verbosity > 2) && (verbosity < 11)) sprintf(str+strlen(str),"J%d,",e);
					break;
				case et_fc:
					e = 0;
					break;
				case -1:
					e = 0;
					break;
				}
				e = ( (e+i) > ne ? (ne-i) : e );
				i += e;
				sudoku->rating += (e * j) / ((float) ne);
			}
			if ((i < ne) && (get_num_cell_with_stat1(*sudoku,stat_free) > 0)) {
				sudoku->rating += ((ne - i) * (et_jf + 1)) / ((float) ne);
			}
			if ((verbosity > 2) && (verbosity < 11) && (strlen(str) > 0) ) {
				sprintf(str+strlen(str)-1,")");
				fprintf(stderr," rating %7.1f\t\t(%s\n",sudoku->rating,str);
				strcpy(str,"");
			}
		}
	} while (apply_solution_technique1(sudoku,elim,depth,verbosity,stdout) > 0);
	return (check_sudoku(*sudoku));
}

/*
 ******************************************************************************/
int apply_solution_technique1(t_sudoku *sudoku, int elim[NUM_ET], int depth, int verbosity, FILE *out) {
	
	int new=0, fcl, iet;
	extern int dfcs[];

	if (check_sudoku(*sudoku) == 0) return(-1);

	// apply solution techniques starting from easy ones. difficulties are stored
	// in dfcs[]. dfcs[1] to dfcs[NUM_ET-2] contains IDs for elimination
	// techniques as integers from 1 (et_hs, hidden single) through et_fc (forcing
	// chain), as defined in header file sudoku.h.
	// dfcs[0] is wrong guess, dfcs[NUM_ET-1] is guess
	// dfcs[i] = -1 means: do nothing.
	for (iet=1; iet<NUM_ET-1; iet++) {
		if (depth >= iet) {
			switch(dfcs[iet]) {
				case et_hs:
					if (hidden_single(sudoku,1,verbosity) > 0) {
						new = 1;
						if (verbosity == 2) fprintf(out,".");
						elim[dfcs[iet]] += 1;
					}
					break;
				case et_s:
					if (single(sudoku,1,verbosity) > 0) {
						new = 1;
						elim[dfcs[iet]] += 1;
						if (verbosity == 2) fprintf(out,"-");
					}
					break;
				case et_hd:
					if ((new = hidden_tuple(sudoku,2,1,verbosity)) > 0) {
						elim[et_hd] += 1;
						if (verbosity == 2) fprintf(out,"d");
					}
					break;
				case et_d:
					if ((new = tuple(sudoku,2,1,verbosity)) > 0) {
						elim[et_d] += 1;
						if (verbosity == 2) fprintf(out,"D");
					}
					break;
				case et_b:
					if ((new = block_line(sudoku,1,verbosity)) > 0) {
						elim[dfcs[iet]] += 1;
						if (verbosity == 2) fprintf(out,"b");
					}
					break;
				case et_l:
					if ((new = line_block(sudoku,1,verbosity)) > 0) {
						elim[dfcs[iet]] += 1;
						if (verbosity == 2) fprintf(out,"B");
					}
					break;
				case et_ht:
					if ((new = hidden_tuple(sudoku,3,1,verbosity)) > 0) {
						elim[et_ht] += 1;
						if (verbosity == 2) fprintf(out,"t");
					}
					break;
				case et_t:
					if ((new = tuple(sudoku,3,1,verbosity)) > 0) {
						elim[et_t] += 1;
						if (verbosity == 2) fprintf(out,"T");
					}
					break;
				case et_hq:
					if ((new = hidden_tuple(sudoku,4,1,verbosity)) > 0) {
						elim[et_hq] += 1;
						if (verbosity == 2) fprintf(out,"q");
					}
					break;
				case et_q:
					if ((new = tuple(sudoku,4,1,verbosity)) > 0) {
						elim[et_q] += 1;
						if (verbosity == 2) fprintf(out,"Q");
					}
					break;
				case et_xy:
					if ((new = xy_wing(sudoku,1,verbosity)) > 0) {
						elim[et_xy] += 1;
						if (verbosity == 2) fprintf(out,"W");
					}
					break;
				case et_xyz:
					if ((new = xyz_wing(sudoku,1,verbosity)) > 0) {
						elim[et_xyz] += 1;
						if (verbosity == 2) fprintf(out,"Y");
					}
					break;
				case et_xw:
					if ((new = fish(sudoku,2,1,verbosity)) > 0) {
						elim[et_xw] += 1;
						if (verbosity == 2) fprintf(out,"X");
					}
					break;
				case et_sf:
					if ((new = fish(sudoku,3,1,verbosity)) > 0) {
						elim[et_sf] += 1;
						if (verbosity == 2) fprintf(out,"S");
					}
					break;
				case et_jf:
					if ((new = fish(sudoku,4,1,verbosity)) > 0) {
						elim[et_jf] += 1;
						if (verbosity == 2) fprintf(out,"J");
					}
					break;
				case et_fc:
					if ((new = forcing_chain(sudoku,verbosity,&fcl)) != 0) {
						switch(new) {
							case 1:
								elim[et_g] += 1;
								if (verbosity == 2) fprintf(out,"e%d",fcl);
								break;
							case 2:
								elim[et_fc] += 1;
								if (verbosity == 2) fprintf(out,"f%d",fcl);
								break;
							case 3:
								elim[et_fc] += 1;
								if (verbosity == 2) fprintf(out,"F%d",fcl);
								break;
						}
					}
					break;
				case -1:
					break;
				default:
					fprintf(out,"WARNING: unknown elim type: %d\n",dfcs[iet]);
					break;
			}
		}
		if (new > 0) return(new);
	}
	return(new);
}

/*
 ******************************************************************************/
void find_cell_and_cand_for_guess(t_field grid[SIZE2][SIZE2], int size, int *r, int *c, int *cc) {
	int nc[SIZE2], i, ir, ic;
	int min_ncand = size+1, min_nc = (size*size)+1;

	// if we are not yet done with cell of previous guess, try out next candidate
	// of that cell
	if ( (*r > -1) && (*c > -1) && (grid[*r][*c].stat == stat_guess) && (grid[*r][*c].ncand > 0) ) {
		if (DEBUG) printf(" *** look for next candidate at (%d,%d):",(*r)+1,(*c)+1);
		*cc = 0;
		while ( (*cc < size) && (grid[*r][*c].cand[*cc] == 0) ) (*cc) += 1;
		if (DEBUG) printf(" *** trying %d\n",(*cc) + 1);
		return;
	}
	
	// determine number of candidates present in mesh for each value
	for (i=0; i<size; i++) nc[i] = 0;
	for (ir=0; ir<size; ir++) {
		for (ic=0; ic<size; ic++) {
			for (i=0; i<size; i++) {
				if (grid[ir][ic].cand[i] != 0) nc[i]++;
			}
		}
	}
	for (i=0; i<size; i++) {
		if ((nc[i] < min_nc) && (nc[i] > 0)) {
			min_nc = nc[i];
			*cc = i;
		}
	}
	if (DEBUG) printf(" *** %d candidates for %d\n",min_nc,(*cc)+1);
//	if (DEBUG) print_cands_as_grid(grid,size);
	
	// select cell with smallest number of candidates, one of them is least
	// common candidate
	for (ir=0; ir<size; ir++) {
		for (ic=0; ic<size; ic++) {
			if ( (grid[ir][ic].cand[*cc] != 0) && (grid[ir][ic].ncand < min_ncand) ) {
				min_ncand = grid[ir][ic].ncand;
				*r = ir;
				*c = ic;
			}
		}
	}
	if (DEBUG) printf(" *** selecting r%dc%d cc%d for guess\n",(*r)+1,(*c)+1,(*cc)+1);
	return;
}

