/*
 * This file is part of sudognu.
 *
 * Copyright (C) 2007 Jens Baaran, Germany.
 ******************************************************************************/

#include <stdio.h>
#include <sys/times.h>
#include <time.h>
#include <stdlib.h>
#include <limits.h>
#include <pthread.h>
#include "sudoku.h"

extern pthread_mutex_t mut_input;
extern pthread_mutex_t mut_stdout;
extern pthread_mutex_t mut_nsud;

int nsud = 0;

/*
 ******************************************************************************/
int execute_thread_create(void *arg[]) {
	
	extern int num_sudokus;
	int tsud = 0;
	t_sudoku sudoku;

	reset_sudoku(&sudoku);
	while (nsud < num_sudokus) {
		pthread_mutex_lock(&mut_nsud);
		nsud++;
		pthread_mutex_unlock(&mut_nsud);
		tsud++;
		create_sudoku(&sudoku);
	}
	return tsud;
}

/*
 ******************************************************************************/
int create_sudoku(t_sudoku *sudoku) {

	int size = sudoku->size;
	int r, c, cc, ng, ngiv, gg, gr, gc;
	int elim[NUM_ET], ret;
	t_sudoku csudoku, dsudoku;
	extern int symmgrid;
	
	reset_sudoku(sudoku);
	
	// repeat until symmetric grid with no extra givens is found
	do {
	
		// repeat until solvable grid is found
		do {
			// reset grid
			reset_sudoku(&csudoku);
			gr=gc=-1;
			ng=0;
			ret=1;
			
			// put in givens
			do {
				if (symmgrid == 0) {
					if (find_next_given_unsymm(csudoku,&r,&c,&cc) < 0) {
						ret = -1;
						break;
					}
				} else {
					if (find_next_given_symm(csudoku,&r,&c,&cc,&gr,&gc) < 0) {
						ret = -1;
						break;
					}
				}
				csudoku.grid[r][c].stat = stat_given;
				csudoku.grid[r][c].value = cc+1;
				ng++;
				get_candidates(&csudoku);
				// do NOT use forcing chains or guess for elimination, because
				// uniqueness is not checked here!!
				if (ng > 8) ret = eliminate(&csudoku,elim,et_d,0,0);
			} while(ret > 0);
		} while (ret < 0);

		// remove all solved cells, only givens remain	
		reset_sudoku(sudoku);
		for (ng=r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				if (csudoku.grid[r][c].stat == stat_given) {
					sudoku->grid[r][c].value = csudoku.grid[r][c].value;
					sudoku->grid[r][c].stat = csudoku.grid[r][c].stat;
					ng++;
				}
			}
		}
	
		// check, if any given can be eliminated
		for (gg=r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				// for all givens on symmetry axes
				if (((r == ((size-1)/2)) || (c == ((size-1)/2))) && (sudoku->grid[r][c].stat == stat_given)) {
					copy_sudoku(*sudoku,&csudoku);
					// turn status to stat_free
					csudoku.grid[r][c].value = 0;
					csudoku.grid[r][c].stat = stat_free;
					copy_sudoku(csudoku,&dsudoku);
					// solve
					ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
					if (check_uniqueness1(csudoku,dsudoku) == 0) {
						gg++;
						sudoku->grid[r][c].value = 0;
						sudoku->grid[r][c].stat = stat_free;
					}
				}
			}
		}
		for (r=0; r<size; r++) {
			for (c=0; c<size; c++) {
				// for all givens not on symmetry axes
				if (((r != ((size-1)/2)) && (c != ((size-1)/2))) && (sudoku->grid[r][c].stat == stat_given)) {
					copy_sudoku(*sudoku,&csudoku);
					// turn status to stat_free
					csudoku.grid[r][c].value = 0;
					csudoku.grid[r][c].stat = stat_free;
					copy_sudoku(csudoku,&dsudoku);
					// solve
					ret = solve_sudoku(&dsudoku,elim,et_d,0,0);
					if (check_uniqueness1(csudoku,dsudoku) == 0) {
						gg++;
						sudoku->grid[r][c].value = 0;
						sudoku->grid[r][c].stat = stat_free;
					}
				}
			}
		}
		if (DEBUG) if (gg > 0) fprintf(stderr,"removed %d gratuituous givens\n",gg);

	} while ((symmgrid == 1) && (check_sudoku_symmetry(*sudoku) < 1));

	// check symmetry
	if (check_sudoku_symmetry(*sudoku) == 1) {
		transpose_sudoku(sudoku);
	}

	// determine number of givens
	for (r=ngiv=0; r<size; r++) {
		for (c=0; c<size; c++) {
			if ((sudoku->grid[r][c].stat == stat_given) || (sudoku->grid[r][c].stat == stat_ggiven)) ngiv++;
		}
	}
	
	// archive sudoku
	copy_sudoku(*sudoku,&dsudoku);
	pthread_mutex_lock(&mut_stdout);                   // beginning of mutex protection
	fprint_sudoku(stdout,*sudoku);    // print sudoku
	printf("\t");
	ret = solve_sudoku(&dsudoku,elim,NUM_ET-1,1,1); // print solution steps
	printf("\t%d\t%d\t%s\n",ngiv,(int)dsudoku.rating,VERSION); // print additional info
	pthread_mutex_unlock(&mut_stdout);                 // end of mutex protection
	
	sudoku->rating = dsudoku.rating;
	return(sudoku->rating);
}

/* select cells, sudoku is generally not gonna be symmetric
 ******************************************************************************/
int find_next_given_unsymm(t_sudoku sudoku, int *r, int *c, int *cc) {
	
	int size = sudoku.size;
	int nc[SIZE2], i, ir, ic;
	int max_ncand = -1, max_nc = -1;
	int ng, index;

	ng = get_num_cell_with_stat1(sudoku,stat_given);
	if (DEBUG) fprintf(stderr,"%d givens found\n",ng);
	
	// pick random cells, set its status to stat_given and value to ng+1
	if (ng < 8) {
		index = get_random_cell2(sudoku.grid,size,stat_free);
		*r = index / size;
		*c = index % size;
		*cc = ng;
		if (DEBUG) fprintf(stderr,"%d. given: %d at r%dc%d\n",ng,*cc,*r,*c);
		return(0);
	}
	
	// determine number of candidates present in mesh for each value
	for (i=0; i<size; i++) nc[i] = 0;
	for (ir=0; ir<size; ir++) {
		for (ic=0; ic<size; ic++) {
			for (i=0; i<size; i++) {
				if (sudoku.grid[ir][ic].cand[i] != 0) nc[i]++;
			}
		}
	}
	for (i=0; i<size; i++) {
		if ((nc[i] > max_nc) && (nc[i] > 0)) {
			max_nc = nc[i];
			*cc = i;
		}
	}
	if (max_nc < 1) return(-1);
	if (DEBUG) printf(" *** %d candidates for %d\n",max_nc,(*cc)+1);
	if (DEBUG) print_candidates_as_grid(sudoku);
	
	// select cell with maximum number of candidates, one of them is most
	// common candidate
	for (ir=0; ir<size; ir++) {
		for (ic=0; ic<size; ic++) {
			if ( (sudoku.grid[ir][ic].cand[*cc] != 0) && (sudoku.grid[ir][ic].ncand > max_ncand) ) {
				max_ncand = sudoku.grid[ir][ic].ncand;
				*r = ir;
				*c = ic;
			}
		}
	}
	return(0);
}

/* select cells for next given in a way that makes sudoku symmetric
 ******************************************************************************/
int find_next_given_symm(t_sudoku sudoku, int *r, int *c, int *cc, int *gr, int * gc) {

	int size = sudoku.size;
	int nc[SIZE2], i, ir, ic;
	int max_ncand = -1;
	int ng, index;

	// determine number of candidates present in mesh for each value
	// -------------------------------------------------------------
	for (i=0; i<size; i++) nc[i] = 0;
	for (ir=0; ir<size; ir++) {
		for (ic=0; ic<size; ic++) {
			for (i=0; i<size; i++) {
				if (sudoku.grid[ir][ic].cand[i] != 0) nc[i]++;
			}
		}
	}

	// get number of givens, if too many givens necessary, return error
	// ------------------------------------------
	ng = get_num_cell_with_stat1(sudoku,stat_given);
	if (ng > 28) {
		return(-1);
	}
	
	// select positions for next 4 candidates
	// --------------------------------------
	if ((ng % 4) == 0) {
		if (ng < 20) {
			// pick random cell in upper left 4x4 matrix
			index = get_random_cell2(sudoku.grid,4,stat_free);
			*r = *gr = index / 4;
			*c = *gc = index % 4;

		// insert candidates #21 to 28 on symmetry axes
		} else if (ng < 28) {
			if ((ng == 20) && (check_sudoku_symmetry(sudoku) == 1))
				transpose_sudoku(&sudoku);
			// in top 4 rows of middle column find cell with maximum number of candidates
			for (ir=0; ir<4; ir++) {
				if ((sudoku.grid[ir][4].stat != stat_given) && (sudoku.grid[ir][4].ncand > max_ncand)) {
					max_ncand = sudoku.grid[ir][4].ncand;
					*r = *gr = ir;
					*c = *gc = 4;
				}
			}
		}
	}
			
	// insert candidate #29 in the middle
	// ----------------------------------
	if (ng == 28) {
		*r = *c = 4;
	}

	// mirror position for next candidate
	// ----------------------------------
	if (ng < 20) {
		// this is for cells not on the symmetry axes
		switch(ng % 4) {
		case 1:
			*r = *gr;
			*c = (size-1) - *gc;
			break;
		case 2:
			*r = (size-1) - *gr;
			*c = *gc;
			break;
		case 3:
			*r = (size-1) - *gr;
			*c = (size-1) - *gc;
			break;
		}
	} else if (ng < 28) {
		// this is for cells on the symmetry axes
		switch(ng % 4) {
		case 1:
			*r = (size-1) - *gr;
			*c = *gc;
			break;
		case 2:
			*r = *gc;
			*c = *gr;
			break;
		case 3:
			*r = *gc;
			*c = (size-1) - *gr;
			break;
		}
	}
	
	// select random candidate
	// -----------------------
	if (sudoku.grid[*r][*c].value > 0) {
		*cc = sudoku.grid[*r][*c].value-1; // if cell is solved, there is no choice
	} else {
		*cc = get_random_cand(sudoku.grid[*r][*c]);
	}
	
	if (DEBUG) {
		printf("%2d r%dc%d (%d), val %d, cand",ng+1,*r+1,*c+1,*cc+1,sudoku.grid[*r][*c].value);
		for (i=0; i<size; i++) if (sudoku.grid[*r][*c].cand[i] != 0) printf(" %d",sudoku.grid[*r][*c].cand[i]);
		printf("\n");
	}
	
	// return success
	// --------------
	return(0);
}

