/* $Id$ */
#ifndef lint
static char *sccsid = "@(#)stat.c	1.3 (UKC) 17/10/85";
#endif  lint
/***

* program name:
	stat
* function:
	Display the contents of an i-node.
* switches:
	-f fmt	Display the i-node according to the format characters
		in "fmt".  Syntax is like that of printf(3s) with the
		following formatting characters:-
		%f		File type.
		%d		Device inode lives on.
		%i		Inode itself.
		%p		Protection bits.
		%l		# of links.
		%u		The user id
		%g		The group id.
		%t		Type of device.
		%s		Total size of file.
		%a		Access time.
		%m		Modified time.
		%c		Status change time.
		%b		# of blocks allocated.
* libraries used:
	standard + getopt
* compile time parameters:
	cc -O -o stat stat.c -lgetopt
* history:
	pur-ee!pucc-j!rsk, rsk@purdue-asc.arpa
	Original version by Hoch in the dark days of v6;
	this one by Rsk for 4.2 BSD.

***/

		/*
		 *	System - wide header files
		 */

#include	<stdio.h>
#include	<sys/time.h>
#include	<sys/types.h>
#include	<ctype.h>
#include	<sys/stat.h>
#include	<pwd.h>
#include	<grp.h>
#include        <stdlib.h>
#include        <unistd.h>

			/*
			 *	Global structures and definitions
			 */


#define	FAIL	-1		/* Failure return code from call */	
#define OKAY	0		/* Success return code from call */

char	*fmt_str;		/* Format string for displaying i-nodes */
char	*progname;		/* Name we're called as */

char *fmt_app __P((char *, char *));
void FmtInode __P((char *, char*, struct stat *));
void Usage __P((void));


/*
**  STAT_IT -- Display an i-node.
**
**	Print the contents of an i-node.
**
**	Parameters:
**		ip    --  Reference to a stat structure.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

int
stat_it(filename)
char	*filename;
{
	int		count;
	char		buf [BUFSIZ];
	struct 	stat	Sbuf;		/* for return values from stat() call */
#define	LBUFSIZ		256	/* Length of symbolic link translation buffer */
	char		Lbuf [LBUFSIZ];	/* Symbolic link translation buffer */

	if( lstat(filename,&Sbuf) == FAIL) {
		fprintf(stderr,"Can't lstat %s\n",filename); 
		return(FAIL);
	}

	if( (Sbuf.st_mode & S_IFMT) == S_IFLNK) {
		if( (count = readlink(filename, Lbuf, sizeof Lbuf)) == FAIL) {
			fprintf(stderr, "Can't readlink %s\n", filename);
			return(FAIL);
		}
		if( count < LBUFSIZ)
			Lbuf[count] = '\0';
		printf("  File: \"%s\" -> \"%s\"\n", filename, Lbuf);
	} else
		printf("  File: \"%s\"\n", filename);


	FmtInode(buf, fmt_str, &Sbuf);

	printf("%s\n", buf);

	return(OKAY);
}

/*
**  UTOS  -- Uid to string mapping
**
**	Return the login of the given
**	uid.
**
**	Parameters:
**		buf    --  Buffer to fill,
**		ip     --  Reference to a stat structure.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

void
UtoS(buf, ip)
char		buf [];
struct stat	*ip;
{
	struct passwd	*pwent;
	extern struct passwd	*getpwuid();

	(void) setpwent();
	if ((pwent = getpwuid(ip -> st_uid)) == NULL) {
		fprintf(stderr, "getpwuid() failed\n");
		exit(1);
	}
	(void) sprintf(buf, "%lu/%s",
		ip -> st_uid,
		pwent->pw_name);
}
/*
**  GTOS -- Group to string mapping
**
**	Given a numeric group-id, return
**	the ASCII name of that group.
**
**	Parameters:
**		buf    --  Buffer to fill,
**		ip      --  Reference to a stat structure.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

void
GtoS(buf, ip)
char		buf [];
struct stat	*ip;
{
	struct group	*grent;
	extern struct group	*getgrid();

	(void) setgrent();
	if( (grent = getgrgid(ip -> st_gid)) == NULL) {
		fprintf(stderr, "getgrgid(%lu) failed\n", ip -> st_gid);
		exit(1);
	}
	(void) sprintf(buf, "%lu/%s",
		ip -> st_gid,
		grent -> gr_name);
}

/*
** DEVTOS --  Device to string mapping
**
**	Return a string containing the type
**	of the device, if the file is a device.
**	Else return a null string.
**
**	Parameters:
**		buf    --  Buffer to fill,
**		ip      --  Reference to a stat structure.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

void
DevToS(buf, ip)
char		buf [];
struct stat	*ip;
{
	if ((ip -> st_mode & S_IFMT) == S_IFCHR
	|| (ip -> st_mode & S_IFMT) == S_IFBLK)
		sprintf(buf, "Device Type: %lu", ip -> st_rdev);
	else
		strcpy(buf, "");
}
/*
**  MODETOS -- Mode word to string mapping
**
**	Display the contents of the mode field
**	in a readable format.
**
**	Parameters:
**		buf    --  Buffer to write on,
**		mode   --  Mode field to interpret.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

void
ModeToS(buf, ip)
char		buf [];
struct stat	*ip;
{
	sprintf(buf, "%04o/----------", ip -> st_mode & 07777);

	if(ip -> st_mode & 0000001)	/* Other execute */
		buf[5 + 9] = 'x';
	if(ip -> st_mode & 0000002)	/* Other write */
		buf[5 + 8] = 'w';
	if(ip -> st_mode & 0000004)	/* Other read */
		buf[5 + 7] = 'r';
	if(ip -> st_mode & 0000010)	/* Group execute */
		buf[5 + 6] = 'x';
	if(ip -> st_mode & 0000020)	/* Group write */
		buf[5 + 5] = 'w';
	if(ip -> st_mode & 0000040)	/* Group read */
		buf[5 + 4] = 'r';
	if(ip -> st_mode & 0000100)	/* User execute */
		buf[5 + 3] = 'x';
	if(ip -> st_mode & 0000200)	/* User write */
		buf[5 + 2] = 'w';
	if(ip -> st_mode & 0000400)	/* User read */
		buf[5 + 1] = 'r';
	if(ip -> st_mode & 0001000)	/* Sticky bit */
		buf[5 + 9] = 't';
	if(ip -> st_mode & 0002000)	/* Set group id */
		buf[5 + 6] = 's';
	if(ip -> st_mode & 0004000)	/* Set user id */
		buf[5 + 4] = 's';
	switch( ip -> st_mode & S_IFMT) {
		case	S_IFDIR:	buf[5 + 0] = 'd';
					break;
		case	S_IFCHR:	buf[5 + 0] = 'c';
					break;
		case	S_IFBLK:	buf[5 + 0] = 'b';
					break;
		case	S_IFREG:	buf[5 + 0] = '-';
					break;
		case	S_IFLNK:	buf[5 + 0] = 'l';
					break;
		case	S_IFSOCK:	buf[5 + 0] = 's';
					break;
		default		:	buf[5 + 0] = '?';
	}
}
/*
**  TYPETOS -- Filetype to string mapping
**
**	Given a filetype, return a string
**	representation of that type.
**
**	Parameters:
**		buf    --  Buffer to fill,
**		ip     --  Reference to a stat structure.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

void
TypeToS(buf, ip)
char		buf [];
struct stat	*ip;
{
	switch( ip -> st_mode & S_IFMT ) {
		case	S_IFDIR:
			strcpy(buf, "Directory");
			break;

		case	S_IFCHR:
			strcpy(buf, "Character Device");
			break;

		case	S_IFBLK:
			strcpy(buf, "Block Device");
			break;

		case	S_IFREG:
			strcpy(buf, "Regular File");
			break;

		case	S_IFLNK:
			strcpy(buf, "Symbolic Link");
			break;

		case	S_IFSOCK:
			strcpy(buf, "Socket");
			break;

		default:
			strcpy(buf, "Unknown");
			break;
	}
}
/*
**  FMT_APP -- Append a format string
**
**	Copy the first string argument
**	to the second, returning a pointer
**	to the next free byte.
**
**	Parameters:
**		src  --  Source string,
**		dest --  Destination string.
**
**	Returns:
**		Reference to the next free byte
**		in the destination string.
**
**	Notes:
**		This routine is auxiliary to
**		FmtInode() below.
**
*/

char	*
fmt_app(src, dest)
register char	*src,
		*dest;
{

	while ((*dest++ = *src++) != '\0')
		;	/* NULL BODY */

	return dest - 1;		/* Back over the NUL byte */
}


/*
**  FMTINODE -- Format an Inode
**
**	Build a textual representation of
**	the Inode into the specified buffer
**	according to the printf-like format
**	string.
**
**	Parameters:
**		buf  --  Buffer to fill,
**		fmt  --  Printf-like format string.
**		rp   --  Reference to Inode cell,
**
**	Returns:
**		None.
**
**
**	Notes:
**		The format characters recognised here are:
**
**	%f		File type.
**	%d		Device inode lives on.
**	%i		Inode itself.
**	%p		Protection bits.
**	%l		# of links.
**	%u		The user id
**	%g		The group id.
**	%t		Type of device.
**	%s		Total size of file.
**	%a		Access time.
**	%m		Modified time.
**	%c		Status change time.
**	%b		# of blocks allocated.
**
**	Use a backslash to 'hide' any special character, e.g. '%'
**
**	The default format string is:
**
**		"%r,,%u,%t,%o"
*/

void
FmtInode(buf, fmt, ip)
char		*buf,
		*fmt;
struct stat	*ip;
{
	char		auxbuf [64];
	register char	*bufp;
	register int	c;
	extern char	*ctime();

	for (bufp = buf; (c = *fmt++) != '\0'; ) {
		if (c == '\\')
			c = *fmt++;
		
		if (c != '%') {
			*bufp++ = c;
		} else {
			switch (c = *fmt++) {
				case 'f':	/* Filetype */
					TypeToS(auxbuf, ip);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'i':	/* The actual Inode	*/
					sprintf(auxbuf, "%ld", ip -> st_ino);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'd':	/* The Device	*/
					sprintf(auxbuf, "%ld", ip -> st_dev);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 't':	/* Device Type */
					DevToS(auxbuf, ip);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'p':	/* Protection bits */
					ModeToS(auxbuf, ip);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'l':	/* # of links */
					sprintf(auxbuf, "%d", ip -> st_nlink);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'u':	/* User id */
					UtoS(auxbuf, ip);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'g':	/* Group id */
					GtoS(auxbuf, ip);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 's':	/* Total size of file */
					sprintf(auxbuf, "%qd", ip -> st_size);
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'a':	/* Access time */
					sprintf(auxbuf, "%24.24s", ctime(&ip -> st_atime));
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'm':	/* Modified time */
					sprintf(auxbuf, "%24.24s", ctime(&ip -> st_mtime));
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'c':	/* Status change time */
					sprintf(auxbuf, "%24.24s", ctime(&ip -> st_ctime));
					bufp = fmt_app(auxbuf, bufp);
					break;

				case 'b':	/* # of blocks allocated */
					sprintf(auxbuf, "%qd",ip -> st_blocks);
					bufp = fmt_app(auxbuf, bufp);
					break;

				default:	/* Just copy it over */
					*bufp++ = c;
					break;
			}
		}
	}
}
/*
**  USAGE -- Display a reminder
**
**	Print a reminder of the correct usage
**	of the command.
**
**	Parameters:
**		None.
**
**	Returns:
**		None.
**
**	Notes:
**		None.
**
*/

void
Usage()
{
	fprintf(stderr, "Usage: %s [-f fmt] file1 file2 ...\n", progname);
}

int
main(argc, argv)
int argc;
char *argv[];
{
	extern int	optind;
	extern char	*optarg;
	int		c;


	/*
	 *	Note how we're called ...
	 */

	progname = argv [0];

	if (argc == 1) {
		Usage();
		exit(1);
	}

	/*
	 *	Set some defaults ...
	 */

	fmt_str = "  Size: %s\tAllocated Blocks: %b\tFiletype: %f\n  Mode: (%p)\tUid: (%u) Gid: (%g)\nDevice: %d\tInode: %i\tLinks: %l\t%t\nAccess: %a\nModify: %m\nChange: %c\n";

	/*
	 *	Check for flag arguments ...
	 */

	while ((c = getopt(argc, argv, "f:")) != EOF)
		switch (c) {
			default:
				Usage();
				exit(1);

			case 'f':		/* Format specifier */
				fmt_str = optarg;
				break;
		}

	/*
	 *	Ok, now for the file arguments ...
	 */

	for (c = 0; optind < argc; ++optind) {
		c += stat_it(argv [optind]);
	}

	/* check for errors, e.g. file does not exists */
	if (c == 0)
		exit(0);
	else
		exit(2);
}
