/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * desktop.c - functions to deal with "desktop" window.
 */

#include <vncviewer.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xmu/Converters.h>
#ifdef MITSHM
#include <X11/extensions/XShm.h>
#endif

#include <X11/cursorfont.h>

GC gc;
GC srcGC, dstGC; /* used for debugging copyrect */
Window desktopWin;
Cursor dotCursor3 = None;
Cursor dotCursor4 = None;
Cursor bogoCursor = None;
Widget form, viewport, desktop;

static Bool modifierPressed[256];

XImage *image = NULL;
XImage *image_ycrop = NULL;

static Cursor CreateDotCursor();
static Cursor CreateBogoCursor();
static void CopyBGR233ToScreen(CARD8 *buf, int x, int y, int width,int height);
static void HandleBasicDesktopEvent(Widget w, XtPointer ptr, XEvent *ev,
				    Boolean *cont);

static void CopyBGR565ToScreen(CARD16 *buf, int x, int y, int width,int height);

static XtResource desktopBackingStoreResources[] = {
  {
    XtNbackingStore, XtCBackingStore, XtRBackingStore, sizeof(int), 0,
    XtRImmediate, (XtPointer) Always,
  },
};

void create_image() {
	image = NULL;
	image_ycrop = NULL;

//fprintf(stderr, "useShm: %d\n", appData.useShm);

#ifdef MITSHM
	if (appData.useShm) {
		image = CreateShmImage(0);
		if (!image) {
			if (appData.yCrop > 0) {
				image_ycrop = CreateShmImage(1);
				if (!image_ycrop) {
					appData.useShm = False;
				} else {
					fprintf(stderr, "created smaller image_ycrop "
					    "shm image\n");
				}
			} else {
				appData.useShm = False;
			}
		}
	}
#endif

	if (!image) {
		image = XCreateImage(dpy, vis, visdepth, ZPixmap, 0, NULL,
		    si.framebufferWidth, si.framebufferHeight, BitmapPad(dpy), 0);

		image->data = malloc(image->bytes_per_line * image->height);
		if (!image->data) {
			fprintf(stderr,"malloc failed\n");
			exit(1);
		} else {
			fprintf(stderr, "created non-shm image\n");
		}
	}
}

int old_width = 0;
int old_height = 0;

int guessCrop(void) {
	int w = si.framebufferWidth;

	if (w == 320) {
		return 240;
	} else if (w == 400) {
		return 300;
	} else if (w == 640) {
		return 480;
	} else if (w == 800) {
		return 600;
	} else if (w == 1024) {
		return 768;
	} else if (w == 1152) {
		return 864;
	} else if (w == 1280) {
		return 1024;
	} else if (w == 1600) {
		return 1200;
	} else if (w == 1920) {
		return 1200;
	} else {
		int h = (3 * w) / 4;
		return h;
	}
}

void check_tall(void) {
	if (! appData.yCrop) {
		int w = si.framebufferWidth;
		int h = si.framebufferHeight;
		if (h > 2 * w) {
			fprintf(stderr, "Tall display (%dx%d) suspect 'x11vnc -ncache' mode,\n", w, h);
			fprintf(stderr, "  setting auto -ycrop detection.\n", w, h);
			appData.yCrop = -1;
		}
	}
}

/*
 * DesktopInitBeforeRealization creates the "desktop" widget and the viewport
 * which controls it.
 */

void
DesktopInitBeforeRealization()
{
	int i;

	form = XtVaCreateManagedWidget("form", formWidgetClass, toplevel,
	    XtNborderWidth, 0, XtNdefaultDistance, 0, NULL);

	viewport = XtVaCreateManagedWidget("viewport", viewportWidgetClass, form,
	    XtNborderWidth, 0, NULL);

	desktop = XtVaCreateManagedWidget("desktop", coreWidgetClass, viewport,
	    XtNborderWidth, 0, NULL);

	XtVaSetValues(desktop, XtNwidth, si.framebufferWidth,
	    XtNheight, si.framebufferHeight, NULL);

	XtAddEventHandler(desktop, LeaveWindowMask|ExposureMask,
	    True, HandleBasicDesktopEvent, NULL);


	check_tall();
	
	if (appData.yCrop) {
		int wm, hm;
		if (appData.yCrop < 0) {
			appData.yCrop = guessCrop();
			fprintf(stderr, "Set -ycrop to: %d\n", appData.yCrop);
		}
		hm = appData.yCrop;
		if (0 && appData.sbWidth <= 6 && appData.sbWidth > 0) {
			hm += appData.sbWidth;
		}
		XtVaSetValues(toplevel, XtNmaxHeight, hm, NULL);
		XtVaSetValues(form,     XtNmaxHeight, hm, NULL);
		XtVaSetValues(viewport, XtNforceBars, False, NULL);
	}
	old_width  = si.framebufferWidth;
	old_height = si.framebufferHeight;

	for (i = 0; i < 256; i++) {
		modifierPressed[i] = False;
	}

	create_image();
}

static Widget scrollbar_y = NULL;

static int xsst = 2;
#include <X11/Xaw/Scrollbar.h>

static XtCallbackProc Scrolled(Widget w, XtPointer closure, XtPointer call_data) {
	Position x, y;
	XtVaGetValues(desktop, XtNx, &x, XtNy, &y, NULL);
	if (0) fprintf(stderr, "scrolled by %d pixels x=%d y=%d\n", (int) call_data, x, y);
	if (xsst == 2) {
		x = 0;
		y = 0;
		XtVaSetValues(desktop, XtNx, x, XtNy, y, NULL);
	} else if (xsst) {
		XawScrollbarSetThumb(w, 0.0, 0.0);
	} else {
		float t = 0.0;
		XtVaSetValues(w, XtNtopOfThumb, &t, NULL);
	}
}
static XtCallbackProc Jumped(Widget w, XtPointer closure, XtPointer call_data) {
	float top = *((float *) call_data);
	Position x, y;
	XtVaGetValues(desktop, XtNx, &x, XtNy, &y, NULL);
	if (0) fprintf(stderr, "thumb value: %.4f x=%d y=%d\n", top, x, y);
	if (top > 0.01) {
		if (xsst == 2) {
			x = 0;
			y = 0;
			XtVaSetValues(desktop, XtNx, x, XtNy, y, NULL);
		} else if (xsst) {
			XawScrollbarSetThumb(w, 0.0, 0.0);
		} else {
			float t = 0.0, s = 1.0;
			XtVaSetValues(w, XtNtopOfThumb, *(XtArgVal*)&t, XtNshown, *(XtArgVal*)&s, NULL);
		}
	}
}


extern double dnow(void);

void check_things() {
	static int installed_callback = 0;
	static int first = 1;
	static double last_scrollbar = 0.0;
	int w = si.framebufferWidth;
	int h = si.framebufferHeight;
	double now = dnow();
	static double last = 0;

	if (first) {
		first = 0;
		SendFramebufferUpdateRequest(0, 0, si.framebufferWidth, si.framebufferHeight, False);
	}
	if (appData.yCrop > 0 && appData.yCrop < dpyHeight && h > 2*w && now > last_scrollbar + 0.25) {
		Widget wv, wh, wc;
		Position x0, y0;
		Position x1, y1;
		Dimension w0, h0, b0;
		Dimension w1, h1, b1;
		Dimension w2, h2, b2;
		
		wc = XtNameToWidget(viewport, "clip");
		wv = XtNameToWidget(viewport, "vertical");
		wh = XtNameToWidget(viewport, "horizontal");
		if (wc && wv && wh) {
			int doit = 1;
			int sb = appData.sbWidth;
			XtVaGetValues(wv, XtNwidth, &w0, XtNheight, &h0, XtNborderWidth, &b0, XtNx, &x0, XtNy, &y0, NULL);
			XtVaGetValues(wh, XtNwidth, &w1, XtNheight, &h1, XtNborderWidth, &b1, XtNx, &x1, XtNy, &y1, NULL);
			XtVaGetValues(wc, XtNwidth, &w2, XtNheight, &h2, XtNborderWidth, &b2, NULL);
			if (!sb) {
				sb = 2;
			}
			if (w0 != sb || h1 != sb) {
				fprintf(stderr, "Very tall (-ncache) fb, setting scrollbar thickness to: %d pixels\n", sb);
				
				XtUnmanageChild(wv);
				XtUnmanageChild(wh);
				XtUnmanageChild(wc);

				XtVaSetValues(wv, XtNwidth, sb,  XtNx, x0 + (w0 - sb), NULL);
				XtVaSetValues(wh, XtNheight, sb, XtNy, y1 + (h1 - sb), NULL);
				w2 = w2 + (w0 - sb);
				h2 = h2 + (h1 - sb);
				if (w2 > 10 && h2 > 10) {
					XtVaSetValues(wc, XtNwidth, w2, XtNheight, h2, NULL);
				}

				XtManageChild(wv);
				XtManageChild(wh);
				XtManageChild(wc);
			}
		}
		last_scrollbar = dnow();
	}
	
	if (now <= last + 1.0) {
		return;
	}

	dpyWidth  = WidthOfScreen(DefaultScreenOfDisplay(dpy));
	dpyHeight = HeightOfScreen(DefaultScreenOfDisplay(dpy));

	last = dnow();
}

/*
 * DesktopInitAfterRealization does things which require the X windows to
 * exist.  It creates some GCs and sets the dot cursor.
 */

void Xcursors(int set) {
	if (dotCursor3 == None) {
		dotCursor3 = CreateDotCursor(3);
	}
	if (dotCursor4 == None) {
		dotCursor4 = CreateDotCursor(4);
	}
	if (set) {
		XSetWindowAttributes attr;
		unsigned long valuemask = 0;

		if (!appData.useX11Cursor) {
			if (appData.viewOnly) {
				attr.cursor = dotCursor4;    
			} else {
				attr.cursor = dotCursor3;    
			}
			valuemask |= CWCursor;
			XChangeWindowAttributes(dpy, desktopWin, valuemask, &attr);
		}
	}
}

void
DesktopInitAfterRealization()
{
	XGCValues gcv;
	XSetWindowAttributes attr;
	XWindowAttributes gattr;
	unsigned long valuemask = 0;

	desktopWin = XtWindow(desktop);

	gc = XCreateGC(dpy,desktopWin,0,NULL);

	gcv.function = GXxor;
	gcv.foreground = 0x0f0f0f0f;
	srcGC = XCreateGC(dpy,desktopWin,GCFunction|GCForeground,&gcv);
	gcv.foreground = 0xf0f0f0f0;
	dstGC = XCreateGC(dpy,desktopWin,GCFunction|GCForeground,&gcv);

	XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
	    NULL, 0);

	if (appData.useBackingstore) {
		Screen *s = DefaultScreenOfDisplay(dpy);
		if (DoesBackingStore(s) != Always) {
			fprintf(stderr, "X server does not do backingstore, disabling it.\n");
			appData.useBackingstore = False;
		}
	}

	if (appData.useBackingstore) {
		XtVaGetApplicationResources(desktop, (XtPointer)&attr.backing_store,
		    desktopBackingStoreResources, 1, NULL);
		valuemask |= CWBackingStore;
	} else {
		attr.background_pixel = BlackPixel(dpy, DefaultScreen(dpy));
		valuemask |= CWBackPixel;
	}

	Xcursors(0);
	if (!appData.useX11Cursor) {
		if (appData.viewOnly) {
			attr.cursor = dotCursor4;    
		} else {
			attr.cursor = dotCursor3;    
		}
		valuemask |= CWCursor;
	}
	bogoCursor = XCreateFontCursor(dpy, XC_bogosity);

	XChangeWindowAttributes(dpy, desktopWin, valuemask, &attr);

	if (XGetWindowAttributes(dpy, desktopWin, &gattr)) {
#if 0
		fprintf(stderr, "desktopWin backingstore: %d save_under: %d\n", gattr.backing_store, gattr.save_under);
#endif
	}
	fprintf(stderr, "\n");
}

extern void FreeX11Cursor(void);
extern void FreeSoftCursor(void);

void
DesktopCursorOff()
{
	XSetWindowAttributes attr;
	unsigned long valuemask;

	if (dotCursor3 == None) {
		dotCursor3 = CreateDotCursor(3);
		dotCursor4 = CreateDotCursor(4);
	}
	if (appData.viewOnly) {
		XDefineCursor(dpy, desktopWin, dotCursor4);
	} else {
		XDefineCursor(dpy, desktopWin, dotCursor3);
	}
	FreeX11Cursor();
	FreeSoftCursor();
}

void put_image(int src_x, int src_y, int dst_x, int dst_y, int width,
    int height) {
	
#ifdef MITSHM
	if (appData.useShm) {
		if (image_ycrop == NULL) {
//fprintf(stderr, "shm not image_ycrop\n");
			XShmPutImage(dpy, desktopWin, gc, image, src_x, src_y,
			    dst_x, dst_y, width, height, False);
		} else if ((width < 32 && height < 32) || height > appData.yCrop) {
//fprintf(stderr, "non-shmB image %d %d %d %d %d %d\n", src_x, src_y, dst_x, dst_y, width, height);
			XPutImage(dpy, desktopWin, gc, image, src_x, src_y,
			    dst_x, dst_y, width, height);
		} else {
			char *src, *dst;
			int Bpp = image->bits_per_pixel / 8;
			int Bpl  = image->bytes_per_line, h;
			int Bpl2 = image_ycrop->bytes_per_line;
			src = image->data + src_y * Bpl + src_x * Bpp;
			dst = image_ycrop->data;
			for (h = 0; h < height; h++) {
				memcpy(dst, src, width * Bpp);
				src += Bpl;
				dst += Bpl2;
			}
//fprintf(stderr, "shm image_ycrop %d %d %d %d %d %d\n", 0, 0, dst_x, dst_y, width, height);
			XShmPutImage(dpy, desktopWin, gc, image_ycrop, 0, 0,
			    dst_x, dst_y, width, height, False);
		}
	} else
#endif
	{
//fprintf(stderr, "non-shmA image %d %d %d %d %d %d\n", src_x, src_y, dst_x, dst_y, width, height);
		XPutImage(dpy, desktopWin, gc, image, src_x, src_y,
		   dst_x, dst_y, width, height);
	}
}


/*
 * HandleBasicDesktopEvent - deal with expose and leave events.
 */

static void
HandleBasicDesktopEvent(Widget w, XtPointer ptr, XEvent *ev, Boolean *cont)
{
	int i, x, y, width, height;


  switch (ev->type) {
  case Expose:
  case GraphicsExpose:
    /* sometimes due to scrollbars being added/removed we get an expose outside
       the actual desktop area.  Make sure we don't pass it on to the RFB
       server. */
	x = ev->xexpose.x;
	y = ev->xexpose.y;
	width  = ev->xexpose.width;
	height = ev->xexpose.height;

//fprintf(stderr, "Expose: %dx%d+%d+%d\n", width, height, x, y);
	if (x + width > si.framebufferWidth) {
		width = si.framebufferWidth - x;
		if (width <= 0) {
			break;
		}
	}

	if (y + height > si.framebufferHeight) {
		height = si.framebufferHeight - y;
		if (height <= 0) {
			break;
		}
	}

	if (appData.useBackingstore) {
		SendFramebufferUpdateRequest(x, y, width, height, False);
	} else {
		put_image(x, y, x, y, width, height);
		XSync(dpy, False);
	}
	break;

  case LeaveNotify:
	for (i = 0; i < 256; i++) {
		if (modifierPressed[i]) {
			SendKeyEvent(XKeycodeToKeysym(dpy, i, 0), False);
			modifierPressed[i] = False;
		}
	}
	break;
  }
	check_things();
}


/*
 * SendRFBEvent is an action which sends an RFB event.  It can be used in two
 * ways.  Without any parameters it simply sends an RFB event corresponding to
 * the X event which caused it to be called.  With parameters, it generates a
 * "fake" RFB event based on those parameters.  The first parameter is the
 * event type, either "fbupdate", "ptr", "keydown", "keyup" or "key"
 * (down&up).  The "fbupdate" event requests full framebuffer update. For a
 * "key" event the second parameter is simply a keysym string as understood by
 * XStringToKeysym().  For a "ptr" event, the following three parameters are
 * just X, Y and the button mask (0 for all up, 1 for button1 down, 2 for
 * button2 down, 3 for both, etc).
 */

extern Bool selectingSingleWindow;

extern Cursor dotCursor3;
extern Cursor dotCursor4;

extern void set_server_scale(int);

void
SendRFBEvent(Widget w, XEvent *ev, String *params, Cardinal *num_params)
{
  KeySym ks;
  char keyname[256];
  int buttonMask, x, y;

	if (appData.fullScreen && ev->type == MotionNotify) {
		if (BumpScroll(ev)) {
			return;
		}
	}

	if (selectingSingleWindow && ev->type == ButtonPress) {
		selectingSingleWindow = False;
		SendSingleWindow(ev->xbutton.x, ev->xbutton.y);
		if (appData.viewOnly) {
			XDefineCursor(dpy, desktopWin, dotCursor4);
		} else {
			XDefineCursor(dpy, desktopWin, dotCursor3);
		}
		return;
	}

	if (appData.viewOnly) {
		int W = si.framebufferWidth;
		int H = si.framebufferHeight;
		if (*num_params != 0) {
		    if (strcasecmp(params[0],"fbupdate") == 0) {
			SendFramebufferUpdateRequest(0, 0, W, H, False);
		    }
		}
		if (ev->type == ButtonRelease) {
			XButtonEvent *b = (XButtonEvent *) ev;
			if (b->state & Button3Mask) {
				ShowPopup(w, ev, params, num_params);
			}
		} else if (ev->type == KeyRelease) {
			XLookupString(&ev->xkey, keyname, 256, &ks, NULL);
			if (ks == XK_1 || ks == XK_KP_1 || ks == XK_KP_End) {
				set_server_scale(1);
			} else if (ks == XK_2 || ks == XK_KP_2 || ks == XK_KP_Down) {
				set_server_scale(2);
			} else if (ks == XK_3 || ks == XK_KP_3 || ks == XK_KP_Next) {
				set_server_scale(3);
			} else if (ks == XK_4 || ks == XK_KP_4) {
				set_server_scale(4);
			} else if (ks == XK_5 || ks == XK_KP_5) {
				set_server_scale(5);
			} else if (ks == XK_6 || ks == XK_KP_6) {
				set_server_scale(6);
			} else if (ks == XK_r || ks == XK_R) {
				SendFramebufferUpdateRequest(0, 0, W, H, False);
			} else if (ks == XK_b || ks == XK_B) {
				ToggleBell(w, ev, params, num_params);
			} else if (ks == XK_f || ks == XK_F) {
				Toggle8bpp(w, ev, params, num_params);
			} else if (ks == XK_V) {
				ToggleViewOnly(w, ev, params, num_params);
			}
		}
		return;
	}

  if (*num_params != 0) {
    if (strncasecmp(params[0],"key",3) == 0) {
      if (*num_params != 2) {
	fprintf(stderr,
		"Invalid params: SendRFBEvent(key|keydown|keyup,<keysym>)\n");
	return;
      }
      ks = XStringToKeysym(params[1]);
      if (ks == NoSymbol) {
	fprintf(stderr,"Invalid keysym '%s' passed to SendRFBEvent\n",
		params[1]);
	return;
      }
      if (strcasecmp(params[0],"keydown") == 0) {
	SendKeyEvent(ks, 1);
      } else if (strcasecmp(params[0],"keyup") == 0) {
	SendKeyEvent(ks, 0);
      } else if (strcasecmp(params[0],"key") == 0) {
	SendKeyEvent(ks, 1);
	SendKeyEvent(ks, 0);
      } else {
	fprintf(stderr,"Invalid event '%s' passed to SendRFBEvent\n",
		params[0]);
	return;
      }
    } else if (strcasecmp(params[0],"fbupdate") == 0) {
      if (*num_params != 1) {
	fprintf(stderr, "Invalid params: SendRFBEvent(fbupdate)\n");
	return;
      }
      SendFramebufferUpdateRequest(0, 0, si.framebufferWidth,
				   si.framebufferHeight, False);
    } else if (strcasecmp(params[0],"ptr") == 0) {
      if (*num_params == 4) {
	x = atoi(params[1]);
	y = atoi(params[2]);
	buttonMask = atoi(params[3]);
	SendPointerEvent(x, y, buttonMask);
      } else if (*num_params == 2) {
	switch (ev->type) {
	case ButtonPress:
	case ButtonRelease:
	  x = ev->xbutton.x;
	  y = ev->xbutton.y;
	  break;
	case KeyPress:
	case KeyRelease:
	  x = ev->xkey.x;
	  y = ev->xkey.y;
	  break;
	default:
	  fprintf(stderr,
		  "Invalid event caused SendRFBEvent(ptr,<buttonMask>)\n");
	  return;
	}
	buttonMask = atoi(params[1]);
	SendPointerEvent(x, y, buttonMask);
      } else {
	fprintf(stderr,
		"Invalid params: SendRFBEvent(ptr,<x>,<y>,<buttonMask>)\n"
		"             or SendRFBEvent(ptr,<buttonMask>)\n");
	return;
      }

    } else {
      fprintf(stderr,"Invalid event '%s' passed to SendRFBEvent\n", params[0]);
    }
    return;
  }

  switch (ev->type) {

  case MotionNotify:
    while (XCheckTypedWindowEvent(dpy, desktopWin, MotionNotify, ev))
      ;	/* discard all queued motion notify events */

    SendPointerEvent(ev->xmotion.x, ev->xmotion.y,
		     (ev->xmotion.state & 0x1f00) >> 8);
    return;

  case ButtonPress:
    SendPointerEvent(ev->xbutton.x, ev->xbutton.y,
		     (((ev->xbutton.state & 0x1f00) >> 8) |
		      (1 << (ev->xbutton.button - 1))));
    return;

  case ButtonRelease:
    SendPointerEvent(ev->xbutton.x, ev->xbutton.y,
		     (((ev->xbutton.state & 0x1f00) >> 8) &
		      ~(1 << (ev->xbutton.button - 1))));
    return;

  case KeyPress:
  case KeyRelease:
    XLookupString(&ev->xkey, keyname, 256, &ks, NULL);

    if (IsModifierKey(ks)) {
      ks = XKeycodeToKeysym(dpy, ev->xkey.keycode, 0);
      modifierPressed[ev->xkey.keycode] = (ev->type == KeyPress);
    }

    SendKeyEvent(ks, (ev->type == KeyPress));
    return;

  default:
    fprintf(stderr,"Invalid event passed to SendRFBEvent\n");
  }
}


/*
 * CreateDotCursor.
 */

#ifndef very_small_dot_cursor
static Cursor
CreateDotCursor(int which)
{
	Cursor cursor;
	Pixmap src, msk;
	static char srcBits3[] = { 0x00, 0x02, 0x00 };
	static char mskBits3[] = { 0x02, 0x07, 0x02 };
	static char srcBits4[] = { 0x00, 0x06, 0x06, 0x00 };
	static char mskBits4[] = { 0x06, 0x0f, 0x0f, 0x06 };
	XColor fg, bg;

	if (which == 3) {
		src = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), srcBits3, 3, 3);
		msk = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), mskBits3, 3, 3);
	} else {
		src = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), srcBits4, 4, 4);
		msk = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), mskBits4, 4, 4);
	}
	XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "black",
	    &fg, &fg);
	XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "white",
	    &bg, &bg);
	cursor = XCreatePixmapCursor(dpy, src, msk, &fg, &bg, 1, 1);
	XFreePixmap(dpy, src);
	XFreePixmap(dpy, msk);

	return cursor;
}
#else
static Cursor
CreateDotCursor()
{
	Cursor cursor;
	Pixmap src, msk;
	static char srcBits[] = { 0, 14, 0 };
	static char mskBits[] = { 14,31,14 };
	XColor fg, bg;

	src = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), srcBits, 3, 3);
	msk = XCreateBitmapFromData(dpy, DefaultRootWindow(dpy), mskBits, 3, 3);
	XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "black",
	    &fg, &fg);
	XAllocNamedColor(dpy, DefaultColormap(dpy,DefaultScreen(dpy)), "white",
	    &bg, &bg);
	cursor = XCreatePixmapCursor(dpy, src, msk, &fg, &bg, 1, 1);
	XFreePixmap(dpy, src);
	XFreePixmap(dpy, msk);

	return cursor;
}
#endif

int skip_maybe_sync = 0;
void maybe_sync(int width, int height) {
	static int singles = 0;
	if (skip_maybe_sync) {
		return;
	}
	if (width > 1 || height > 1) {
		XSync(dpy, False);
		singles = 0;
	} else {
		if (++singles >= 32) {
			singles = 0;
			XSync(dpy, False);
		}
	}
}
/*
 * FillImage.
 */

void
FillScreen(int x, int y, int width, int height, unsigned long fill)
{
	int bpp = image->bits_per_pixel;
	int Bpp = image->bits_per_pixel / 8;
	int Bpl = image->bytes_per_line;
	int h, widthInBytes = width * Bpp;
	static char *buf = NULL;
	static int buflen = 0;
	unsigned char  *ucp;
	unsigned short *usp;
	unsigned int   *uip;
	char *scr;
	int b0, b1, b2;

//fprintf(stderr, "FillImage bpp=%d %04dx%04d+%04d+%04d -- 0x%x\n", bpp, width, height, x, y, fill);

	if (widthInBytes > buflen || !buf)  {
		if (buf) {
			free(buf);
		}
		buflen = widthInBytes * 2;
		buf = (char *)malloc(buflen); 
	}
	ucp = (unsigned char*) buf;
	usp = (unsigned short*) buf;
	uip = (unsigned int*) buf;

	if (isLSB) {
		b0 = 0; b1 = 1; b2 = 2;
	} else {
		b0 = 2; b1 = 1; b2 = 0;
	}

	for (h = 0; h < width; h++) {
		if (bpp == 8) {
			*(ucp+h) = (unsigned char)  fill;
		} else if (bpp == 16) {
			*(usp+h) = (unsigned short) fill;
		} else if (bpp == 24) {
			*(ucp + 3*h + b0) = (unsigned char) ((fill & 0x0000ff) >> 0);
			*(ucp + 3*h + b1) = (unsigned char) ((fill & 0x00ff00) >> 8);
			*(ucp + 3*h + b2) = (unsigned char) ((fill & 0xff0000) >> 16);
		} else if (bpp == 32) {
			*(uip+h) = (unsigned int)   fill;
		}
	}

	scr = image->data + y * Bpl + x * Bpp;

	for (h = 0; h < height; h++) {
		memcpy(scr, buf, widthInBytes);
		scr += Bpl;
	}
	put_image(x, y, x, y, width, height);
	maybe_sync(width, height);
}

void copy_rect(int x, int y, int width, int height, int src_x, int src_y) {
	char *src, *dst;
	int i;
	int Bpp = image->bits_per_pixel / 8;
	int Bpl = image->bytes_per_line;

//fprintf(stderr, "copy_rect: %04dx%04d+%04d+%04d -- %04d %04d Bpp=%d Bpl=%d\n", width, height, x, y, src_x, src_y, Bpp, Bpl);
	if (y < src_y) {
		src = image->data + src_y * Bpl + src_x * Bpp;
		dst = image->data +     y * Bpl +     x * Bpp;
		for (i = 0; i < height; i++)  {
			memmove(dst, src, Bpp * width); 
			src += Bpl;
			dst += Bpl;
		}
	} else {
		src = image->data + (src_y + height - 1) * Bpl + src_x * Bpp;
		dst = image->data + (y     + height - 1) * Bpl +     x * Bpp;
		for (i = 0; i < height; i++)  {
			memmove(dst, src, Bpp * width); 
			src -= Bpl;
			dst -= Bpl;
		}
	}
}


/*
 * CopyDataToScreen.
 */

void
CopyDataToScreen(char *buf, int x, int y, int width, int height)
{
	if (appData.rawDelay != 0) {
		XFillRectangle(dpy, desktopWin, gc, x, y, width, height);
		XSync(dpy,False);
		usleep(appData.rawDelay * 1000);
	}

	if (appData.useBGR233) {
		CopyBGR233ToScreen((CARD8 *)buf, x, y, width, height);
	} else if (appData.useBGR565) {
		CopyBGR565ToScreen((CARD16 *)buf, x, y, width, height);
	} else {
		int h;
		int widthInBytes = width * myFormat.bitsPerPixel / 8;
		int scrWidthInBytes = si.framebufferWidth * myFormat.bitsPerPixel / 8;

		char *scr = (image->data + y * scrWidthInBytes
		    + x * myFormat.bitsPerPixel / 8);

//fprintf(stderr, "CopyDataToScreen %dx%d+%d+%d\n", width, height, x, y);

		for (h = 0; h < height; h++) {
			memcpy(scr, buf, widthInBytes);
			buf += widthInBytes;
			scr += scrWidthInBytes;
		}
	}

	put_image(x, y, x, y, width, height);
	maybe_sync(width, height);
}


/*
 * CopyBGR233ToScreen.
 */

static void
CopyBGR233ToScreen(CARD8 *buf, int x, int y, int width, int height)
{
	int p, q;
	int xoff = 7 - (x & 7);
	int xcur;
	int fbwb = si.framebufferWidth / 8;
	CARD8 *scr1 = ((CARD8 *)image->data) + y * fbwb + x / 8;
	CARD8 *scrt;
	CARD8  *scr8  = ( (CARD8 *)image->data) + y * si.framebufferWidth + x;
	CARD16 *scr16 = ((CARD16 *)image->data) + y * si.framebufferWidth + x;
	CARD32 *scr32 = ((CARD32 *)image->data) + y * si.framebufferWidth + x;
	int b0, b1, b2;

	switch (visbpp) {

    /* thanks to Chris Hooper for single bpp support */

	case 1:
		for (q = 0; q < height; q++) {
			xcur = xoff;
			scrt = scr1;
			for (p = 0; p < width; p++) {
				*scrt = ((*scrt & ~(1 << xcur))
					 | (BGR233ToPixel[*(buf++)] << xcur));

				if (xcur-- == 0) {
					xcur = 7;
					scrt++;
				}
			}
			scr1 += fbwb;
		}
		break;

	case 8:
		for (q = 0; q < height; q++) {
			for (p = 0; p < width; p++) {
				*(scr8++) = BGR233ToPixel[*(buf++)];
			}
			scr8 += si.framebufferWidth - width;
		}
		break;

	case 16:
		for (q = 0; q < height; q++) {
			for (p = 0; p < width; p++) {
				*(scr16++) = BGR233ToPixel[*(buf++)];
			}
			scr16 += si.framebufferWidth - width;
		}
		break;

	case 24:
		if (isLSB) {
			b0 = 0; b1 = 1; b2 = 2;
		} else {
			b0 = 2; b1 = 1; b2 = 0;
		}
		scr8  = ((CARD8 *)image->data) + (y * si.framebufferWidth + x) * 3;
		for (q = 0; q < height; q++) {
			for (p = 0; p < width; p++) {
				CARD32 v = BGR233ToPixel[*(buf++)];
				*(scr8 + b0) = (unsigned char) ((v & 0x0000ff) >> 0);
				*(scr8 + b1) = (unsigned char) ((v & 0x00ff00) >> 8);
				*(scr8 + b2) = (unsigned char) ((v & 0xff0000) >> 16);
				scr8 += 3;
			}
			scr8 += (si.framebufferWidth - width) * 3;
		}
		break;

	case 32:
		for (q = 0; q < height; q++) {
			for (p = 0; p < width; p++) {
				*(scr32++) = BGR233ToPixel[*(buf++)];
			}
			scr32 += si.framebufferWidth - width;
		}
		break;
	}
}

static void
BGR565_24bpp(CARD16 *buf, int x, int y, int width, int height)
{
	int p, q;
	int b0, b1, b2;
	unsigned char *scr= (unsigned char *)image->data + (y * si.framebufferWidth + x) * 3;

	if (isLSB) {
		b0 = 0; b1 = 1; b2 = 2;
	} else {
		b0 = 2; b1 = 1; b2 = 0;
	}

	/*  case 24: */
	for (q = 0; q < height; q++) {
		for (p = 0; p < width; p++) {
			CARD32 v = BGR565ToPixel[*(buf++)];
			*(scr + b0) = (unsigned char) ((v & 0x0000ff) >> 0);
			*(scr + b1) = (unsigned char) ((v & 0x00ff00) >> 8);
			*(scr + b2) = (unsigned char) ((v & 0xff0000) >> 16);
			scr += 3;
		}
		scr += (si.framebufferWidth - width) * 3;
	}
}

static void
CopyBGR565ToScreen(CARD16 *buf, int x, int y, int width, int height)
{
	int p, q;
	CARD32 *scr32 = ((CARD32 *)image->data) + y * si.framebufferWidth + x;

	if (visbpp == 24) {
		BGR565_24bpp(buf, x, y, width, height);
		return;
	}

	/*  case 32: */
	for (q = 0; q < height; q++) {
		for (p = 0; p < width; p++) {
			*(scr32++) = BGR565ToPixel[*(buf++)];
		}
		scr32 += si.framebufferWidth - width;
	}
}

static void reset_image(void) {
	if (UsingShm()) {
		ShmCleanup();
	}
	if (image && image->data) {
		XDestroyImage(image);
	}
	image = NULL;
	if (image_ycrop && image_ycrop->data) {
		XDestroyImage(image_ycrop);
	}
	image_ycrop = NULL;
	create_image();
	XFlush(dpy);
}

void ReDoDesktop(void) {
	int w, h, x, y, dw, dh;

fprintf(stderr, "ReDoDesktop: ycrop: %d\n", appData.yCrop);

	check_tall();
	if (appData.yCrop) {
		if (appData.yCrop < 0 || old_width <= 0) {
			appData.yCrop = guessCrop();
			fprintf(stderr, "Set -ycrop to: %d\n", appData.yCrop);
		} else {
			int w1 = si.framebufferWidth;
			int w0 = old_width;
			appData.yCrop = (w1 * appData.yCrop) / old_width;
			if (appData.yCrop <= 100)  {
				appData.yCrop = guessCrop();
				fprintf(stderr, "Set small -ycrop to: %d\n", appData.yCrop);
			}
		}
		fprintf(stderr, "Using -ycrop: %d\n", appData.yCrop);
	}
	old_width  = si.framebufferWidth;
	old_height = si.framebufferHeight;

	if (appData.fullScreen) {
		if (image && image->data) {
			int len;
			int h = image->height;
			int w = image->width;
			len = image->bytes_per_line * image->height;
			/* black out window first: */
			memset(image->data, 0, len);
  			XPutImage(dpy, XtWindow(desktop), gc, image, 0, 0, 0, 0, w, h);
			XFlush(dpy);
		}
		XtResizeWidget(desktop, si.framebufferWidth, si.framebufferHeight, 0);
		XSync(dpy, False);
		usleep(100*1000);
		FullScreenOn();
		XSync(dpy, False);
		usleep(100*1000);
		reset_image();
		return;
	}

	dw = appData.wmDecorationWidth;
	dh = appData.wmDecorationHeight;

	w = si.framebufferWidth;
	h = si.framebufferHeight;
	if (appData.yCrop > 0) {
		h = appData.yCrop;
		if (0 && appData.sbWidth <= 6 && appData.sbWidth > 0) {
			h += appData.sbWidth;
		}
	}

	if (w + dw >= dpyWidth) {
		w = dpyWidth - dw;
	}
	if (h + dh >= dpyHeight) {
		h = dpyHeight - dh;
	}

	XtVaSetValues(toplevel, XtNmaxWidth, w, XtNmaxHeight, h, NULL);

	XtVaSetValues(desktop, XtNwidth, si.framebufferWidth,
	    XtNheight, si.framebufferHeight, NULL);

	x = (dpyWidth  - w - dw)/2;
	y = (dpyHeight - h - dh)/2;

	XtResizeWidget(desktop, si.framebufferWidth, si.framebufferHeight, 0);

	if (appData.yCrop > 0) {
		XtVaSetValues(toplevel, XtNmaxHeight, appData.yCrop, NULL);
		XtVaSetValues(form,     XtNmaxHeight, appData.yCrop, NULL);
	}

	XtConfigureWidget(toplevel, x + dw, y + dh, w, h, 0);

	reset_image();
}
