
/*
 * The Real SoundTracker - main user interface handling
 *
 * Copyright (C) 1998-2001 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <unistd.h>

#include "poll.h"

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <glib.h>
#ifdef USE_GNOME
#include <gnome.h>
#endif
#ifndef NO_GDK_PIXBUF
#include <gdk-pixbuf/gdk-pixbuf.h>
#endif

#include "i18n.h"
#include "gui.h"
#include "gui-subs.h"
#include "xm.h"
#include "st-subs.h"
#include "audio.h"
#include "xm-player.h"
#include "tracker.h"
#include "main.h"
#include "keys.h"
#include "instrument-editor.h"
#include "sample-editor.h"
#include "track-editor.h"
#include "scope-group.h"
#include "module-info.h"
#include "preferences.h"
#include "menubar.h"
#include "time-buffer.h"
#include "tips-dialog.h"
#include "gui-settings.h"
#include "file-operations.h"
#include "playlist.h"
#include "extspinbutton.h"

int gui_playing_mode = 0;
int notebook_current_page = NOTEBOOK_PAGE_FILE;
GtkWidget *editing_toggle;
GtkWidget *gui_curins_name, *gui_cursmpl_name;
GtkWidget *mainwindow;
ScopeGroup *scopegroup;

static GtkWidget *gui_splash_window = NULL;
#ifndef NO_GDK_PIXBUF
static GdkPixbuf *gui_splash_logo = NULL;
static GtkWidget *gui_splash_logo_area;
#endif
static GtkWidget *gui_splash_label;
static GtkWidget *gui_splash_close_button;

static gint pipetag = -1;
static GtkWidget *mainwindow_upper_hbox, *mainwindow_second_hbox;
static GtkWidget *notebook;
static GtkWidget *spin_editpat, *spin_patlen, *spin_numchans;
static GtkWidget *cursmpl_spin;
static GtkAdjustment *adj_amplification, *adj_pitchbend;
static GtkWidget *spin_jump, *curins_spin, *spin_octave;
static Playlist *playlist;

guint statusbar_context_id;
GtkWidget *status_bar;
GtkWidget *st_clock;

static void gui_tempo_changed (int value);
static void gui_bpm_changed (int value);

gui_subs_slider tempo_slider = {
    N_("Tempo"), 1, 31, gui_tempo_changed, GUI_SUBS_SLIDER_SPIN_ONLY
};
gui_subs_slider bpm_slider = {
    "BPM", 32, 255, gui_bpm_changed, GUI_SUBS_SLIDER_SPIN_ONLY
};

static GdkColor gui_clipping_led_on, gui_clipping_led_off;
static GtkWidget *gui_clipping_led;
static gboolean gui_clipping_led_status;

static int editing_pat = 0;

static int gui_ewc_startstop = 0;

/* gui event handlers */
static void file_selected(GtkWidget *w, GtkFileSelection *fs);
static void current_instrument_changed(GtkSpinButton *spin);
static void current_instrument_name_changed(void);
static void current_sample_changed(GtkSpinButton *spin);
static void current_sample_name_changed(void);
static int keyevent(GtkWidget *widget, GdkEventKey *event, gpointer data);
static void gui_editpat_changed(GtkSpinButton *spin);
static void gui_patlen_changed(GtkSpinButton *spin);
static void gui_numchans_changed(GtkSpinButton *spin);
static void notebook_page_switched(GtkNotebook *notebook, GtkNotebookPage *page, int page_num);
static void gui_adj_amplification_changed(GtkAdjustment *adj);
static void gui_adj_pitchbend_changed(GtkAdjustment *adj);

/* mixer / player communication */
static void read_mixer_pipe(gpointer data, gint source, GdkInputCondition condition);
static void wait_for_player(void);
static void play_song(void);
static void play_pattern(void);
static void play_current_pattern_row(void);

/* gui initialization / helpers */
static void gui_enable(int enable);
static void offset_current_pattern(int offset);
static void offset_current_instrument(int offset);
static void offset_current_sample(int offset);

static void gui_auto_switch_page (void);

static void
gui_mixer_play_pattern (int pattern,
			int row,
			int stop_after_row)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_PLAY_PATTERN;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &pattern, sizeof(pattern));
    write(audio_ctlpipe, &row, sizeof(row));
    write(audio_ctlpipe, &stop_after_row, sizeof(stop_after_row));
}

static void
gui_mixer_stop_playing (void)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_STOP_PLAYING;
    write(audio_ctlpipe, &i, sizeof(i));
}

static void
gui_mixer_set_songpos (int songpos)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_SET_SONGPOS;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &songpos, sizeof(songpos));
}

static void
gui_mixer_set_pattern (int pattern)
{
    audio_ctlpipe_id i = AUDIO_CTLPIPE_SET_PATTERN;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &pattern, sizeof(pattern));
}

static void
gui_load_callback (gint reply,
		   gpointer data)
{
    if(reply == 0) {
	gui_load_xm((gchar*)data);
        gui_auto_switch_page();
    }
}

static void
gui_save_callback (gint reply,
		   gpointer data)
{
    if(reply == 0) {
	statusbar_update(STATUS_SAVING_MODULE, TRUE);
	if(XM_Save(xm, (gchar*)data, FALSE)) {
	    xm->modified = 0;
	    gui_auto_switch_page();
	    statusbar_update(STATUS_MODULE_SAVED, FALSE);
	} else {
	    statusbar_update(STATUS_IDLE, FALSE);
	}
    }
}

static void
gui_save_song_callback (gint reply,
	                gpointer data)
{
    if(reply == 0) {
	statusbar_update(STATUS_SAVING_SONG, TRUE);
	if(XM_Save(xm, (gchar*)data, TRUE)) {
	    gui_auto_switch_page();
	    statusbar_update(STATUS_SONG_SAVED, FALSE);
	} else {
	    statusbar_update(STATUS_IDLE, FALSE);
	}
    }
}

static void
gui_save_wav_callback (gint reply,
		       gpointer data)
{
    if(reply == 0) {
	int l = strlen(data);
	audio_ctlpipe_id i = AUDIO_CTLPIPE_RENDER_SONG_TO_FILE;

	gui_play_stop();

	write(audio_ctlpipe, &i, sizeof(i));
	write(audio_ctlpipe, &l, sizeof(l));
	write(audio_ctlpipe, data, l + 1);

	gui_set_current_pattern(xm->pattern_order_table[0]);
	wait_for_player();
    }
}

static void
file_selected (GtkWidget *w,
	       GtkFileSelection *fs)
{
    gchar *fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));

    gtk_widget_hide(GTK_WIDGET(fs));

    if(fs == GTK_FILE_SELECTION(fileops_dialogs[DIALOG_LOAD_MOD])) {
	file_selection_save_path(fn, gui_settings.loadmod_path);
	if(xm->modified) {
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to free the current project?\nAll changes will be lost!"),
				      gui_load_callback,
				      fn);
	} else {
	    gui_load_callback(0, fn);
	}
    } else if(fs == GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_MOD])) {
	FILE *f = fopen(fn, "r");

	file_selection_save_path(fn, gui_settings.savemod_path);

	if(f != NULL) {
	    fclose(f);
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to overwrite the file?"),
				      gui_save_callback,
				      fn);
	} else {
	    gui_save_callback(0, fn);
	}
    } else if(fs == GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_SONG_AS_XM])) {
	FILE *f = fopen(fn, "r");

	file_selection_save_path(fn, gui_settings.savesongasxm_path);

	if(f != NULL) {
	    fclose(f);
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to overwrite the file?"),
				      gui_save_song_callback,
				      fn);
	} else {
	    gui_save_song_callback(0, fn);
	}
    } else {
	FILE *f = fopen(fn, "r");

	file_selection_save_path(fn, gui_settings.savemodaswav_path);

	if(f != NULL) {
	    fclose(f);
	    gnome_app_ok_cancel_modal(GNOME_APP(mainwindow),
				      _("Are you sure you want to overwrite the file?"),
				      gui_save_wav_callback,
				      fn);
	} else {
	    gui_save_wav_callback(0, fn);
	}
    }
}

static void
current_instrument_changed (GtkSpinButton *spin)
{
    int m = xm_get_modified();
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];
    STSample *s = &i->samples[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))];

    instrument_editor_set_instrument(i);
    sample_editor_set_sample(s);
    modinfo_set_current_instrument(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin)) - 1);
    xm_set_modified(m);
}

static void
current_instrument_name_changed (void)
{
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];

    strncpy(i->name, gtk_entry_get_text(GTK_ENTRY(gui_curins_name)), 22);
    i->name[22] = 0;
    modinfo_update_instrument(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1);
    xm_set_modified(1);
}

static void
current_sample_changed (GtkSpinButton *spin)
{
    int m = xm_get_modified();
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];
    STSample *s = &i->samples[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))];

    gtk_entry_set_text(GTK_ENTRY(gui_cursmpl_name), s->name);
    sample_editor_set_sample(s);
    modinfo_set_current_sample(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin)));
    xm_set_modified(m);
}

static void
current_sample_name_changed (void)
{
    STInstrument *i = &xm->instruments[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))-1];
    STSample *s = &i->samples[gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))];

    strncpy(s->name, gtk_entry_get_text(GTK_ENTRY(gui_cursmpl_name)), 22);
    s->name[22] = 0;
    modinfo_update_sample(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin)));
    xm_set_modified(1);
}

static gboolean
gui_handle_standard_keys (int shift,
			  int ctrl,
			  int alt,
			  guint32 keyval)
{
    gboolean handled = FALSE, b;

    switch (keyval) {
    case GDK_F1 ... GDK_F7:
	if(!shift && !ctrl && !alt) {
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_octave), keyval - GDK_F1);
	    handled = TRUE;
	}
	break;
    case '1' ... '8':
	if(ctrl) {
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_jump), keyval - '0');
	    handled = TRUE;
        }
        if(alt) {
            switch(keyval)
            {
	    case '1' ... '5':
                gtk_notebook_set_page(GTK_NOTEBOOK(notebook), keyval - '1');
                break;
	    default:
		break;
            }
	    handled = TRUE;
        }
	break;
    case GDK_Left:
	if(ctrl) {
	    /* previous instrument */
	    offset_current_instrument(shift ? -5 : -1);
	    handled = TRUE;
	} else if(alt) {
	    /* previous pattern */
	    offset_current_pattern(shift ? -10 : -1);
	    handled = TRUE;
	}
	break;
    case GDK_Right:
	if(ctrl) {
	    /* next instrument */
	    offset_current_instrument(shift ? 5 : 1);
	    handled = TRUE;
	} else if(alt) {
	    /* next pattern */
	    offset_current_pattern(shift ? 10 : 1);
	    handled = TRUE;
	}
	break;
    case GDK_Up:
	if(ctrl) {
	    /* next sample */
	    offset_current_sample(shift ? 4 : 1);
	    handled = TRUE;
	}
	break;
    case GDK_Down:
	if(ctrl) {
	    /* previous sample */
	    offset_current_sample(shift ? -4 : -1);
	    handled = TRUE;
	}
	break;
    case GDK_Alt_R:
    case GDK_Meta_R:
    case GDK_Super_R:
    case GDK_Hyper_R:
    case GDK_Mode_switch: /* well... this is X :D */
	play_pattern();
	handled = TRUE;
	break;
    case GDK_Control_R:
    case GDK_Multi_key:
	play_song();
	handled = TRUE;
	break;
    case GDK_Menu:
	play_current_pattern_row();
	break;
    case ' ':
        if(ctrl || alt || shift)
            break;
	b = GUI_ENABLED;
	gui_play_stop();
    if(notebook_current_page != NOTEBOOK_PAGE_SAMPLE_EDITOR && notebook_current_page != NOTEBOOK_PAGE_INSTRUMENT_EDITOR) {
		if(b) {
        	    /* toggle editing mode (only if we haven't been in playing mode) */
	            gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(editing_toggle), !GUI_EDITING);
    	        if (GTK_TOGGLE_BUTTON(editing_toggle)->active)
					show_editmode_status();
	            else
					statusbar_update(STATUS_IDLE, FALSE);
        }
    }
	handled = TRUE;
	break;
    case GDK_Escape:
        if(ctrl || alt || shift)
            break;
	/* toggle editing mode, even if we're in playing mode */
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(editing_toggle), !GUI_EDITING);
        if (GTK_TOGGLE_BUTTON(editing_toggle)->active)
	    show_editmode_status();
        else
	    statusbar_update(STATUS_IDLE, FALSE);
	handled = TRUE;
	break;
    }

    return handled;
}

static int
keyevent (GtkWidget *widget,
	  GdkEventKey *event,
	  gpointer data)
{
    static gboolean (*handle_page_keys[])(int,int,int,guint32,gboolean) = {
	fileops_page_handle_keys,
	track_editor_handle_keys,
	instrument_editor_handle_keys,
	sample_editor_handle_keys,
	modinfo_page_handle_keys,
    };
    gboolean pressed = (gboolean)data;
    gboolean handled = FALSE;
    gboolean entry_focus = GTK_IS_ENTRY(GTK_WINDOW(mainwindow)->focus_widget);

    if(!entry_focus && GTK_WIDGET_VISIBLE(notebook)) {
	int shift = event->state & GDK_SHIFT_MASK;
	int ctrl = event->state & GDK_CONTROL_MASK;
	int alt = event->state & GDK_MOD1_MASK;

	if(pressed)
	    handled = gui_handle_standard_keys(shift, ctrl, alt, event->keyval);
	handled = handled || handle_page_keys[notebook_current_page](shift, ctrl, alt, event->keyval, pressed);

	if(!handled) switch(event->keyval) {
	    /* from gtk+-1.2.8's gtkwindow.c. These keypresses need to
	       be stopped in any case. */
	case GDK_Up:
	case GDK_Down:
	case GDK_Left:
	case GDK_Right:
	case GDK_KP_Up:
	case GDK_KP_Down:
	case GDK_KP_Left:
	case GDK_KP_Right:
	case GDK_Tab:
	case GDK_ISO_Left_Tab:
	    handled = TRUE;
	}

	if(handled) {
	    if(pressed) {
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
	    } else {
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_release_event");
	    }
	}
    } else {
	if(pressed) {
	    switch(event->keyval) {
	    case GDK_Tab:
	    case GDK_Return:
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		gtk_window_set_focus(GTK_WINDOW(mainwindow), NULL);
		break;
	    }
	}
    }

    return TRUE;
}

static void
playlist_position_changed (Playlist *p,
			   int newpos)
{
    if(gui_playing_mode) {
	// This will only be executed when the user changes the song position manually
	gui_mixer_set_songpos(newpos);
	event_waiter_start(audio_songpos_ew);
    } else {
	gui_set_current_pattern(playlist_get_nth_pattern(p, newpos));
    }
}

static void
playlist_restart_position_changed (Playlist *p,
				   int pos)
{
    xm->restart_position = pos;
    xm_set_modified(1);
}

static void
playlist_entry_changed (Playlist *p,
			int pos,
			int pat)
{
    int i;

    if(pos != -1) {
	xm->pattern_order_table[pos] = pat;
    } else {
	for(i = 0; i < xm->song_length; i++) {
	    xm->pattern_order_table[i] = playlist_get_nth_pattern(p, i);
	}
    }

    xm_set_modified(1);
}

static void
playlist_song_length_changed (Playlist *p,
			      int length)
{
    xm->song_length = length;
    playlist_entry_changed(p, -1, -1);
}

static void
gui_editpat_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    if(n != editing_pat)
	gui_set_current_pattern(n);
}

static void
gui_patlen_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);
    XMPattern *pat = &xm->patterns[editing_pat];

    if(n != pat->length) {
	st_set_pattern_length(pat, n);
	tracker_set_pattern(tracker, NULL);
	tracker_set_pattern(tracker, pat);
	xm_set_modified(1);
    }
}

static void
gui_numchans_changed (GtkSpinButton *spin)
{
    int n = gtk_spin_button_get_value_as_int(spin);

    g_assert((n & 1) == 0);

    if(xm->num_channels != n) {
	gui_play_stop();
	tracker_set_pattern(tracker, NULL);
	st_set_num_channels(xm, n);
	gui_init_xm(0);
	xm_set_modified(1);
    }
}

static void
gui_tempo_changed (int value)
{
    audio_ctlpipe_id i;
    xm->tempo = value;
    xm_set_modified(1);
    if(gui_playing_mode) {
	event_waiter_start(audio_tempo_ew);
    }
    i = AUDIO_CTLPIPE_SET_TEMPO;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &value, sizeof(value));
}

static void
gui_bpm_changed (int value)
{
    audio_ctlpipe_id i;
    xm->bpm = value;
    xm_set_modified(1);
    if(gui_playing_mode) {
	event_waiter_start(audio_bpm_ew);
    }
    i = AUDIO_CTLPIPE_SET_BPM;
    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &value, sizeof(value));
}

static void
gui_adj_amplification_changed (GtkAdjustment *adj)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_SET_AMPLIFICATION;
    float b = 8.0 - adj->value;

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &b, sizeof(b));
}

static void
gui_adj_pitchbend_changed (GtkAdjustment *adj)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_SET_PITCHBEND;
    float b = adj->value;

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &b, sizeof(b));
}

static void
gui_reset_pitch_bender (void)
{
    gtk_adjustment_set_value(adj_pitchbend, 0.0);
}

static void
notebook_page_switched (GtkNotebook *notebook,
			GtkNotebookPage *page,
			int page_num)
{
    notebook_current_page = page_num;
}

/* gui_update_player_pos() is responsible for updating various GUI
   features according to the current player position. ProTracker and
   FastTracker scroll the patterns while the song is playing, and we
   need the time-buffer and event-waiter interfaces here for correct
   synchronization (we're called from
   track-editor.c::tracker_timeout() which hands us time-correct data)

   We have an ImpulseTracker-like editing mode as well ("asynchronous
   editing"), which disables the scrolling, but still updates the
   current song position spin buttons, for example. */

void
gui_update_player_pos (const audio_player_pos *p)
{
    int m = xm_get_modified();

    if(gui_playing_mode == PLAYING_NOTE)
	return;

    if(gui_playing_mode == PLAYING_SONG) {
	if(event_waiter_ready(audio_songpos_ew, p->time)) {
	    /* The following check prevents excessive calls of set_position() */
	    if(p->songpos != playlist_get_position(playlist)) {
		playlist_freeze_signals(playlist);
		playlist_set_position(playlist, p->songpos);
		playlist_thaw_signals(playlist);
	    }
	}
	if(!ASYNCEDIT) {
	    /* The following is a no-op if we're already in the right pattern */
	    gui_set_current_pattern(xm->pattern_order_table[p->songpos]);
	}
    }

    if(!ASYNCEDIT) {
	tracker_set_patpos(tracker, p->patpos);

	if(notebook_current_page == 0) {
	    /* Prevent accumulation of X drawing primitives */
	    gdk_flush();
	}
    }

    if(gui_settings.tempo_bpm_update) {
	if(event_waiter_ready(audio_tempo_ew, p->time)) {
	    tempo_slider.update_without_signal = TRUE;
	    gui_subs_set_slider_value(&tempo_slider, p->tempo);
	    tempo_slider.update_without_signal = FALSE;
	}

	if(event_waiter_ready(audio_bpm_ew, p->time)) {
	    bpm_slider.update_without_signal = TRUE;
	    gui_subs_set_slider_value(&bpm_slider, p->bpm);
	    bpm_slider.update_without_signal = FALSE;
	}
    }

    xm_set_modified(m);
}

void
gui_clipping_indicator_update (double songtime)
{
    if(songtime < 0.0) {
	gui_clipping_led_status = 0;
    } else {
	audio_clipping_indicator *c = time_buffer_get(audio_clipping_indicator_tb, songtime);
	gui_clipping_led_status = c && c->clipping;
    }
    gtk_widget_draw(gui_clipping_led, NULL);
}

static void
read_mixer_pipe (gpointer data,
		 gint source,
		 GdkInputCondition condition)
{
    audio_backpipe_id a;
    struct pollfd pfd = { source, POLLIN, 0 };
    int x;

    static char *msgbuf = NULL;
    static int msgbuflen = 0;

  loop:
    if(poll(&pfd, 1, 0) <= 0)
	return;

    read(source, &a, sizeof(a));
//    printf("read %d\n", a);

    switch(a) {
    case AUDIO_BACKPIPE_PLAYING_STOPPED:
        statusbar_update(STATUS_IDLE, FALSE);
#ifdef USE_GNOME
        gtk_clock_stop(GTK_CLOCK(st_clock));
#endif

        if(gui_ewc_startstop > 0) {
	    /* can be equal to zero when the audio subsystem decides to stop playing on its own. */
	    gui_ewc_startstop--;
	}
	gui_playing_mode = 0;
	scope_group_stop_updating(scopegroup);
	tracker_stop_updating();
	sample_editor_stop_updating();
	gui_enable(1);
	break;

    case AUDIO_BACKPIPE_PLAYING_STARTED:
        statusbar_update(STATUS_PLAYING_SONG, FALSE);
	/* fall through */

    case AUDIO_BACKPIPE_PLAYING_PATTERN_STARTED:
        if(a == AUDIO_BACKPIPE_PLAYING_PATTERN_STARTED)
	    statusbar_update(STATUS_PLAYING_PATTERN, FALSE);
#ifdef USE_GNOME
        gtk_clock_set_seconds(GTK_CLOCK(st_clock), 0);
        gtk_clock_start(GTK_CLOCK(st_clock));
#endif

        gui_ewc_startstop--;
	gui_playing_mode = (a == AUDIO_BACKPIPE_PLAYING_STARTED) ? PLAYING_SONG : PLAYING_PATTERN;
	if(!ASYNCEDIT) {
	    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(editing_toggle), FALSE);
	}
	gui_enable(0);
	scope_group_start_updating(scopegroup);
	tracker_start_updating();
	sample_editor_start_updating();
	break;

    case AUDIO_BACKPIPE_PLAYING_NOTE_STARTED:
	gui_ewc_startstop--;
	if(!gui_playing_mode) {
	    gui_playing_mode = PLAYING_NOTE;
	    scope_group_start_updating(scopegroup);
	    tracker_start_updating();
	    sample_editor_start_updating();
	}
	break;

    case AUDIO_BACKPIPE_DRIVER_OPEN_FAILED:
	gui_ewc_startstop--;
        break;

    case AUDIO_BACKPIPE_ERROR_MESSAGE:
    case AUDIO_BACKPIPE_WARNING_MESSAGE:
        statusbar_update(STATUS_IDLE, FALSE);
        readpipe(source, &x, sizeof(x));
	if(msgbuflen < x + 1) {
	    g_free(msgbuf);
	    msgbuf = g_new(char, x + 1);
	    msgbuflen = x + 1;
	}
	readpipe(source, msgbuf, x + 1);
	if(a == AUDIO_BACKPIPE_ERROR_MESSAGE)
            gnome_error_dialog(msgbuf);
	else
            gnome_warning_dialog(msgbuf);
	break;

    default:
	fprintf(stderr, "\n\n*** read_mixer_pipe: unexpected backpipe id %d\n\n\n", a);
	g_assert_not_reached();
	break;
    }

    goto loop;
}

static void
wait_for_player (void)
{
    struct pollfd pfd = { audio_backpipe, POLLIN, 0 };

    gui_ewc_startstop++;
    while(gui_ewc_startstop != 0) {
	g_return_if_fail(poll(&pfd, 1, -1) > 0);
	read_mixer_pipe(NULL, audio_backpipe, 0);
    }
}

static void
play_song (void)
{
    int sp = playlist_get_position(playlist);
    int pp = 0;
    audio_ctlpipe_id i = AUDIO_CTLPIPE_PLAY_SONG;

    g_assert(xm != NULL);

    gui_play_stop();

    write(audio_ctlpipe, &i, sizeof(i));
    write(audio_ctlpipe, &sp, sizeof(sp));
    write(audio_ctlpipe, &pp, sizeof(pp));

    gui_set_current_pattern(xm->pattern_order_table[sp]);
    wait_for_player();
}

static void
play_pattern (void)
{
    gui_play_stop();
    gui_mixer_play_pattern(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat)), 0, 0);
    wait_for_player();
}

static void
play_current_pattern_row (void)
{
    gui_play_stop();
    gui_mixer_play_pattern(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat)), tracker->patpos, 1);
    gui_ewc_startstop++;
}

void
gui_play_stop (void)
{
    gui_mixer_stop_playing();
    wait_for_player();
}

void
gui_init_xm (int new_xm)
{
    int m = xm_get_modified();
    audio_ctlpipe_id i;

    i = AUDIO_CTLPIPE_INIT_PLAYER;
    write(audio_ctlpipe, &i, sizeof(i));
    tracker_reset(tracker);
    if(new_xm) {
	gui_playlist_initialize();
	editing_pat = -1;
	gui_set_current_pattern(xm->pattern_order_table[0]);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(curins_spin), 1);
	current_instrument_changed(GTK_SPIN_BUTTON(curins_spin));
	modinfo_set_current_instrument(0);
	modinfo_set_current_sample(0);
	modinfo_update_all();
    } else {
	i = editing_pat;
	editing_pat = -1;
	gui_set_current_pattern(i);
    }
    gui_subs_set_slider_value(&tempo_slider, xm->tempo);
    gui_subs_set_slider_value(&bpm_slider, xm->bpm);
    track_editor_set_num_channels(xm->num_channels);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_numchans), xm->num_channels);
    scope_group_set_num_channels(scopegroup, xm->num_channels);
    xm_set_modified(m);
}

void
gui_free_xm (void)
{
    gui_play_stop();
    instrument_editor_set_instrument(NULL);
    sample_editor_set_sample(NULL);
    tracker_set_pattern(tracker, NULL);
    XM_Free(xm);
    xm = NULL;
}

void
gui_new_xm (void)
{
    xm = XM_New();

    if(!xm) {
	fprintf(stderr, "Whooops, having memory problems?\n");
	exit(1);
    }
    gui_init_xm(1);
}

void
gui_load_xm (const char *filename)
{
    statusbar_update(STATUS_LOADING_MODULE, TRUE);

    gui_free_xm();
    xm = File_Load(filename);

    if(!xm) {
	gui_new_xm();
	statusbar_update(STATUS_IDLE, FALSE);
    } else {
	gui_init_xm(1);
	statusbar_update(STATUS_MODULE_LOADED, FALSE);
    }
}

void
gui_play_note (int channel,
	       int note)
{
    int instrument = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin));
    audio_ctlpipe_id a = AUDIO_CTLPIPE_PLAY_NOTE;

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &channel, sizeof(channel));
    write(audio_ctlpipe, &note, sizeof(note));
    write(audio_ctlpipe, &instrument, sizeof(instrument));
    gui_ewc_startstop++;
}

void
gui_play_note_full (unsigned channel,
		    unsigned note,
		    STSample *sample,
		    guint32 offset,
		    guint32 count)
{
    int x;
    audio_ctlpipe_id a = AUDIO_CTLPIPE_PLAY_NOTE_FULL;

    g_assert(sizeof(int) == sizeof(unsigned));

    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &channel, sizeof(channel));
    write(audio_ctlpipe, &note, sizeof(note));
    write(audio_ctlpipe, &sample, sizeof(sample));
    x = offset; write(audio_ctlpipe, &x, sizeof(x));
    x = count; write(audio_ctlpipe, &x, sizeof(x));
    gui_ewc_startstop++;
}

void
gui_play_note_keyoff (int channel)
{
    audio_ctlpipe_id a = AUDIO_CTLPIPE_PLAY_NOTE_KEYOFF;
    write(audio_ctlpipe, &a, sizeof(a));
    write(audio_ctlpipe, &channel, sizeof(channel));
}

static void
gui_enable (int enable)
{
    if(!ASYNCEDIT) {
	gtk_widget_set_sensitive(vscrollbar, enable);
	gtk_widget_set_sensitive(spin_patlen, enable);
    }
    playlist_enable(playlist, enable);
}

void
gui_set_current_pattern (int p)
{
    int m;

    if(editing_pat == p)
	return;

    m = xm_get_modified();

    editing_pat = p;
    tracker_set_pattern(tracker, &xm->patterns[p]);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_editpat), p);
    gui_update_pattern_data();

    if(!GUI_ENABLED && !ASYNCEDIT) {
	gui_mixer_set_pattern(p);
    }

    xm_set_modified(m);
}

void
gui_update_pattern_data (void)
{
    int p = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat));

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_patlen), xm->patterns[p].length);
}

int
gui_get_current_pattern (void)
{
    return editing_pat;
}

static void
offset_current_pattern (int offset)
{
    int nv;

    nv = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_editpat)) + offset;

    if(nv < 0)
	nv = 0;
    else if(nv > 255)
	nv = 255;

    gui_set_current_pattern(nv);
}

void
gui_set_current_instrument (int n)
{
    int m = xm_get_modified();
    g_return_if_fail(n >= 1 && n <= 128);
    if(n != gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin))) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(curins_spin), n);
    }
    xm_set_modified(m);
}

void
gui_set_current_sample (int n)
{
    int m = xm_get_modified();
    g_return_if_fail(n >= 0 && n <= 15);
    if(n != gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin))) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(cursmpl_spin), n);
    }
    xm_set_modified(m);
}

int
gui_get_current_sample (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin));
}

int
gui_get_current_octave_value (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_octave));
}

int
gui_get_current_jump_value (void)
{
    if(!GUI_ENABLED && !ASYNCEDIT)
	return 0;
    else
	return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin_jump));
}

int
gui_get_current_instrument (void)
{
    return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin));
}

static void
offset_current_instrument (int offset)
{
    int nv, v;

    v = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(curins_spin));
    nv = v + offset;

    if(nv < 1)
	nv = 1;
    else if(nv > 128)
	nv = 128;

    gui_set_current_instrument(nv);
}

static void
offset_current_sample (int offset)
{
    int nv, v;

    v = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cursmpl_spin));
    nv = v + offset;

    if(nv < 0)
	nv = 0;
    else if(nv > 15)
	nv = 15;

    gui_set_current_sample(nv);
}

void
gui_get_text_entry (int length,
		    void(*changedfunc)(),
		    GtkWidget **widget)
{
    GtkWidget *thing;

    thing = gtk_entry_new_with_max_length(length);

    gtk_signal_connect_after(GTK_OBJECT(thing), "changed",
			     GTK_SIGNAL_FUNC(changedfunc), NULL);

    *widget = thing;
}

void
gui_playlist_initialize (void)
{
    int i;

    playlist_freeze_signals(playlist);
    playlist_freeze(playlist);
    playlist_set_position(playlist, 0);
    playlist_set_length(playlist, xm->song_length);
    for(i = 0; i < xm->song_length; i++) {
	playlist_set_nth_pattern(playlist, i, xm->pattern_order_table[i]);
    }
    playlist_set_restart_position(playlist, xm->restart_position);
    playlist_thaw(playlist);
    playlist_thaw_signals(playlist);
}

static gint
gui_clipping_led_event (GtkWidget *thing,
			GdkEvent *event)
{
    static GdkGC *clipping_led_gc = NULL;

    if(!clipping_led_gc)
	clipping_led_gc = gdk_gc_new(thing->window);

    switch (event->type) {
    case GDK_MAP:
    case GDK_EXPOSE:
	gdk_gc_set_foreground(clipping_led_gc, gui_clipping_led_status ? &gui_clipping_led_on : &gui_clipping_led_off);
	gdk_draw_rectangle(thing->window, clipping_led_gc, 1, 0, 0, -1, -1);
    default:
	break;
    }
    
    return 0;
}

void
gui_go_to_fileops_page (void)
{
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			  0);
}

void
gui_auto_switch_page (void)
{
    if(gui_settings.auto_switch)
        gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
                              1);
}

#ifndef NO_GDK_PIXBUF
static gint
gui_splash_logo_expose (GtkWidget *widget,
			GdkEventExpose *event,
			gpointer data)
{
    gdk_pixbuf_render_to_drawable (gui_splash_logo,
				   widget->window,
				   widget->style->black_gc,
				   event->area.x, event->area.y,
				   event->area.x, event->area.y,
				   event->area.width, event->area.height,
				   GDK_RGB_DITHER_NORMAL,
				   0, 0);

    return TRUE;
}
#endif

static void
gui_splash_close (void)
{
    gtk_widget_destroy(gui_splash_window);
    gui_splash_window = NULL;
}

static void
gui_splash_set_label (const gchar *text,
		      gboolean update)
{
    char buf[256];

    strcpy(buf, "SoundTracker v" VERSION " - ");
    strncat(buf, text, 255-sizeof(buf));

    gtk_label_set_text(GTK_LABEL(gui_splash_label), buf);

    while(update && gtk_events_pending ()) {
	gtk_main_iteration ();
    }
}

int
gui_splash (int argc,
	    char *argv[])
{
    GtkWidget *vbox, *thing;
#ifndef NO_GDK_PIXBUF
    GtkWidget *hbox, *logo_area, *frame;
#endif

#ifdef USE_GNOME
    gnome_init("SoundTracker", VERSION, argc, argv);
#else
    gtk_init(&argc, &argv);
    gdk_rgb_init();
#endif

    gui_splash_window = gtk_window_new (GTK_WINDOW_DIALOG);

    gtk_window_set_title (GTK_WINDOW(gui_splash_window), _("SoundTracker Startup"));
//    gtk_window_set_wmclass (GTK_WINDOW (gui_splash_window), "soundtracker_startup", "SoundTracker");
    gtk_window_set_position (GTK_WINDOW (gui_splash_window), GTK_WIN_POS_CENTER);
    gtk_window_set_policy (GTK_WINDOW (gui_splash_window), FALSE, FALSE, FALSE);
    gtk_window_set_modal(GTK_WINDOW(gui_splash_window), TRUE);

    gtk_signal_connect(GTK_OBJECT (gui_splash_window), "delete_event",
		       GTK_SIGNAL_FUNC (gui_splash_close), NULL);

    vbox = gtk_vbox_new (FALSE, 4);
    gtk_container_add (GTK_CONTAINER (gui_splash_window), vbox);

    gtk_container_border_width(GTK_CONTAINER(vbox), 4);

    /* Show splash screen if enabled and image available. */

#ifndef NO_GDK_PIXBUF
    gui_splash_logo = gdk_pixbuf_new_from_file(PREFIX"/share/soundtracker/soundtracker_splash.png");
    if(gui_splash_logo) {
	thing = gtk_hseparator_new();
	gtk_widget_show(thing);
	gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	add_empty_vbox(hbox);

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, TRUE, 0);
	gtk_widget_show(frame);

	gui_splash_logo_area = logo_area = gtk_drawing_area_new ();
	gtk_container_add (GTK_CONTAINER(frame), logo_area);
	gtk_widget_show(logo_area);

	add_empty_vbox(hbox);

	gtk_signal_connect (GTK_OBJECT (logo_area), "expose_event",
			    GTK_SIGNAL_FUNC (gui_splash_logo_expose),
			    NULL);

	gtk_drawing_area_size (GTK_DRAWING_AREA (logo_area),
			       gdk_pixbuf_get_width(gui_splash_logo),
			       gdk_pixbuf_get_height(gui_splash_logo));
    }
#endif

    /* Show version number. */

    thing = gtk_hseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);

    gui_splash_label = thing = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    gui_splash_set_label(_("Loading..."), FALSE);

    /* Show tips if enabled. */

    if(tips_dialog_show_tips) {
	GtkWidget *tipsbox = tips_dialog_get_vbox();

	if(tipsbox) {
	    thing = gtk_hseparator_new();
	    gtk_widget_show(thing);
	    gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);

	    gtk_box_pack_start(GTK_BOX(vbox), tipsbox, FALSE, FALSE, 0);
	    gtk_widget_show(tipsbox);
	}
    }

    thing = gtk_hseparator_new();
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);

    if(
#ifndef NO_GDK_PIXBUF
       gui_splash_logo ||
#endif
       tips_dialog_show_tips) {
	gui_splash_close_button = thing = gtk_button_new_with_label(_("Use SoundTracker!"));
	gtk_widget_show(thing);
	gtk_box_pack_start(GTK_BOX(vbox), thing, FALSE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT (thing), "clicked",
			   GTK_SIGNAL_FUNC(gui_splash_close), NULL);
	gtk_widget_set_sensitive(thing, FALSE);
    }

    gtk_widget_show(vbox);
    gtk_widget_show(gui_splash_window);

    gui_splash_set_label(_("Loading..."), TRUE);

    return 1;
}

int
gui_final (int argc,
	   char *argv[])
{
    GtkWidget *thing, *mainvbox, *table, *hbox, *frame, *mainvbox0;
    GdkColormap *colormap;
#ifdef USE_GNOME
    GtkWidget *dockitem;
#endif

    pipetag = gdk_input_add(audio_backpipe, GDK_INPUT_READ, read_mixer_pipe, NULL);

#ifdef USE_GNOME
    mainwindow = gnome_app_new("SoundTracker", "SoundTracker " VERSION);
#else
    mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (mainwindow), "SoundTracker " VERSION);
#endif

    gtk_signal_connect (GTK_OBJECT (mainwindow), "delete_event",
			GTK_SIGNAL_FUNC (menubar_quit_requested), NULL);

    if(gui_splash_window) {
	gtk_window_set_transient_for(GTK_WINDOW(gui_splash_window),
				     GTK_WINDOW(mainwindow));
    }

    if(gui_settings.st_window_x != -666) {
	gtk_window_set_default_size (GTK_WINDOW (mainwindow),
				     gui_settings.st_window_w,
				     gui_settings.st_window_h);
	gtk_widget_set_uposition (GTK_WIDGET (mainwindow),
				  gui_settings.st_window_x,
				  gui_settings.st_window_y);
    }

    fileops_dialogs[DIALOG_LOAD_MOD] = file_selection_create(_("Load XM..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_LOAD_MOD]), gui_settings.loadmod_path);
    fileops_dialogs[DIALOG_SAVE_MOD] = file_selection_create(_("Save XM..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_MOD]), gui_settings.savemod_path);
#ifndef NO_AUDIOFILE
    fileops_dialogs[DIALOG_SAVE_MOD_AS_WAV] = file_selection_create(_("Render module as WAV..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_MOD_AS_WAV]), gui_settings.savemodaswav_path);
#endif
    fileops_dialogs[DIALOG_SAVE_SONG_AS_XM] = file_selection_create(_("Save song as XM..."), file_selected);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileops_dialogs[DIALOG_SAVE_SONG_AS_XM]), gui_settings.savesongasxm_path);

    mainvbox0 = gtk_vbox_new(FALSE, 0);
    gtk_container_border_width(GTK_CONTAINER(mainvbox0), 0);
#ifdef USE_GNOME
    gnome_app_set_contents(GNOME_APP(mainwindow), mainvbox0);
#else
    gtk_container_add(GTK_CONTAINER(mainwindow), mainvbox0);
#endif
    gtk_widget_show(mainvbox0);

    menubar_create(mainwindow, mainvbox0);
	
    mainvbox = gtk_vbox_new(FALSE, 4);
    gtk_container_border_width(GTK_CONTAINER(mainvbox), 4);
    gtk_box_pack_start(GTK_BOX(mainvbox0), mainvbox, TRUE, TRUE, 0);
    gtk_widget_show(mainvbox);

    /* The upper part of the window */
    mainwindow_upper_hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(mainvbox), mainwindow_upper_hbox, FALSE, TRUE, 0);
    gtk_widget_show(mainwindow_upper_hbox);

    /* Program List */
    thing = playlist_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    playlist = PLAYLIST(thing);
    gtk_signal_connect (GTK_OBJECT (playlist), "current_position_changed",
			GTK_SIGNAL_FUNC (playlist_position_changed), NULL);
    gtk_signal_connect (GTK_OBJECT (playlist), "restart_position_changed",
			GTK_SIGNAL_FUNC (playlist_restart_position_changed), NULL);
    gtk_signal_connect (GTK_OBJECT (playlist), "song_length_changed",
			GTK_SIGNAL_FUNC (playlist_song_length_changed), NULL);
    gtk_signal_connect (GTK_OBJECT (playlist), "entry_changed",
			GTK_SIGNAL_FUNC (playlist_entry_changed), NULL);

    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    /* Basic editing commands and properties */
    
    table = gtk_table_new(5, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 4);
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), table, FALSE, TRUE, 0);
    gtk_widget_show(table);

    thing = gtk_button_new_with_label (_("Play Song"));
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(play_song), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 0, 1);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label (_("Play Pattern"));
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(play_pattern), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 1, 2, 0, 1);
    gtk_widget_show(thing);

    thing = gtk_button_new_with_label (_("Stop"));
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(gui_play_stop), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 2, 1, 2);
    gtk_widget_show(thing);

    thing = gui_subs_create_slider(&tempo_slider);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 3, 4);

    gtk_widget_show(thing);

    thing = gui_subs_create_slider(&bpm_slider);
    gtk_table_attach_defaults(GTK_TABLE(table), thing, 0, 1, 4, 5);
    gtk_widget_show(thing);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 2, 2, 3);
    gtk_widget_show(hbox);

    thing = gtk_label_new(_("Number of Channels:"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    add_empty_hbox(hbox);

    spin_numchans = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(8, 2, 32, 2.0, 8.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(spin_numchans));
    gtk_box_pack_start(GTK_BOX(hbox), spin_numchans, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(spin_numchans), "changed",
		       GTK_SIGNAL_FUNC(gui_numchans_changed), NULL);
    gtk_widget_show(spin_numchans);

    hbox = gtk_hbox_new(FALSE, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 3, 4);
    gtk_widget_show(hbox);

    thing = gtk_label_new(_("Pattern"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    add_empty_hbox(hbox);

    spin_editpat = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(0, 0, 255, 1.0, 10.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(spin_editpat));
    gtk_box_pack_start(GTK_BOX(hbox), spin_editpat, FALSE, TRUE, 0);
    gtk_widget_show(spin_editpat);
    gtk_signal_connect(GTK_OBJECT(spin_editpat), "changed",
		       GTK_SIGNAL_FUNC(gui_editpat_changed), NULL);
    
    hbox = gtk_hbox_new(FALSE, 4);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 1, 2, 4, 5);
    gtk_widget_show(hbox);

    thing = gtk_label_new(_("PatLength"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    add_empty_hbox(hbox);

    spin_patlen = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(64, 1, 256, 1.0, 16.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(spin_patlen));
    gtk_box_pack_start(GTK_BOX(hbox), spin_patlen, FALSE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(spin_patlen), "changed",
		       GTK_SIGNAL_FUNC(gui_patlen_changed), NULL);
    gtk_widget_show(spin_patlen);

    /* Scopes Group or Instrument / Sample Listing */

    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    
    scopegroup = SCOPE_GROUP(scope_group_new());
    gtk_widget_show(GTK_WIDGET(scopegroup));
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), GTK_WIDGET(scopegroup), TRUE, TRUE, 0);

    /* Amplification and Pitchbender */

    thing = gtk_vseparator_new();
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);
    
    hbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), hbox, FALSE, TRUE, 0);

    adj_amplification = GTK_ADJUSTMENT(gtk_adjustment_new(7.0, 0, 8.0, 0.1, 0.1, 0.1));
    thing = gtk_vscale_new(adj_amplification);
    gtk_scale_set_draw_value(GTK_SCALE(thing), FALSE);
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(adj_amplification), "value_changed",
			GTK_SIGNAL_FUNC(gui_adj_amplification_changed), NULL);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
    gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, TRUE, 0);
    gtk_widget_show(frame);

    gui_clipping_led = thing = gtk_drawing_area_new();
    gtk_drawing_area_size(GTK_DRAWING_AREA(thing), 15, 15);
    gtk_widget_set_events(thing, GDK_EXPOSURE_MASK);
    gtk_container_add (GTK_CONTAINER(frame), thing);
    colormap = gtk_widget_get_colormap(thing);
    gui_clipping_led_on.red = 0xffff;
    gui_clipping_led_on.green = 0;
    gui_clipping_led_on.blue = 0;
    gui_clipping_led_on.pixel = 0;
    gui_clipping_led_off.red = 0;
    gui_clipping_led_off.green = 0;
    gui_clipping_led_off.blue = 0;
    gui_clipping_led_off.pixel = 0;
    gdk_color_alloc(colormap, &gui_clipping_led_on);
    gdk_color_alloc(colormap, &gui_clipping_led_off);
    gtk_signal_connect(GTK_OBJECT(thing), "event", GTK_SIGNAL_FUNC(gui_clipping_led_event), thing);
    gtk_widget_show (thing);

    hbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(mainwindow_upper_hbox), hbox, FALSE, TRUE, 0);

    adj_pitchbend = GTK_ADJUSTMENT(gtk_adjustment_new(0.0, -20.0, +20.0, 1, 1, 1));
    thing = gtk_vscale_new(adj_pitchbend);
    gtk_scale_set_draw_value(GTK_SCALE(thing), FALSE);
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT(adj_pitchbend), "value_changed",
			GTK_SIGNAL_FUNC(gui_adj_pitchbend_changed), NULL);

    thing = gtk_button_new_with_label("R");
    gtk_widget_show(thing);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (thing), "clicked",
			GTK_SIGNAL_FUNC(gui_reset_pitch_bender), NULL);

    /* Instrument, sample, editing status */

    mainwindow_second_hbox = hbox = gtk_hbox_new(FALSE, 4);
    gtk_box_pack_start(GTK_BOX(mainvbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);

    editing_toggle = thing = gtk_check_button_new_with_label("Editing");
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(thing), 0);
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    thing = gtk_label_new(_("Octave"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    spin_octave = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(5.0, 0.0, 6.0, 1.0, 1.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(spin_octave));
    gtk_box_pack_start(GTK_BOX(hbox), spin_octave, FALSE, TRUE, 0);
    gtk_widget_show(spin_octave);

    thing = gtk_label_new(_("Jump"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    spin_jump = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(1.0, 0.0, 16.0, 1.0, 1.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(spin_jump));
    gtk_box_pack_start(GTK_BOX(hbox), spin_jump, FALSE, TRUE, 0);
    gtk_widget_show(spin_jump);

    thing = gtk_label_new(_("Instr"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    curins_spin = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(1.0, 1.0, 128.0, 1.0, 16.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(curins_spin));
    gtk_box_pack_start(GTK_BOX(hbox), curins_spin, FALSE, TRUE, 0);
    gtk_widget_show(curins_spin);
    gtk_signal_connect (GTK_OBJECT(curins_spin), "changed",
			GTK_SIGNAL_FUNC(current_instrument_changed), NULL);

    gui_get_text_entry(22, current_instrument_name_changed, &gui_curins_name);
    gtk_box_pack_start(GTK_BOX(hbox), gui_curins_name, TRUE, TRUE, 0);
    gtk_widget_show(gui_curins_name);
    gtk_widget_set_usize(gui_curins_name, 100, gui_curins_name->requisition.height);

    thing = gtk_label_new(_("Sample"));
    gtk_box_pack_start(GTK_BOX(hbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    cursmpl_spin = extspinbutton_new(GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 15.0, 1.0, 4.0, 0.0)), 0, 0);
    extspinbutton_disable_size_hack(EXTSPINBUTTON(cursmpl_spin));
    gtk_box_pack_start(GTK_BOX(hbox), cursmpl_spin, FALSE, TRUE, 0);
    gtk_widget_show(cursmpl_spin);
    gtk_signal_connect (GTK_OBJECT(cursmpl_spin), "changed",
			GTK_SIGNAL_FUNC(current_sample_changed), NULL);

    gui_get_text_entry(22, current_sample_name_changed, &gui_cursmpl_name);
    gtk_box_pack_start(GTK_BOX(hbox), gui_cursmpl_name, TRUE, TRUE, 0);
    gtk_widget_show(gui_cursmpl_name);
    gtk_widget_set_usize(gui_cursmpl_name, 100, gui_cursmpl_name->requisition.height);

    /* The notebook */

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(mainvbox), notebook, TRUE, TRUE, 0);
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
    gtk_widget_show(notebook);
    gtk_container_border_width(GTK_CONTAINER(notebook), 0);
    gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
		       GTK_SIGNAL_FUNC(notebook_page_switched), NULL);

    fileops_page_create(GTK_NOTEBOOK(notebook));
    tracker_page_create(GTK_NOTEBOOK(notebook));
    instrument_page_create(GTK_NOTEBOOK(notebook));
    sample_editor_page_create(GTK_NOTEBOOK(notebook));
    modinfo_page_create(GTK_NOTEBOOK(notebook));

    // Activate tracker page
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook),
			  1);
    notebook_current_page = 1;

    /* Status Bar */

#define WELCOME_MESSAGE _("Welcome to SoundTracker!")

#ifdef USE_GNOME
    dockitem = gnome_dock_item_new("Status Bar", (GNOME_DOCK_ITEM_BEH_EXCLUSIVE | GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL));
    gnome_app_add_dock_item(GNOME_APP(mainwindow), GNOME_DOCK_ITEM(dockitem), GNOME_DOCK_BOTTOM, 0, 0, 0);
    gtk_widget_show(dockitem);

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_container_border_width(GTK_CONTAINER(hbox), 2);
    gtk_container_add(GTK_CONTAINER(dockitem), hbox);
    gtk_widget_show(hbox);

    status_bar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_NEVER);
    gtk_widget_show (status_bar);
    gtk_box_pack_start (GTK_BOX (hbox), status_bar, TRUE, TRUE, 0);
    gtk_widget_set_usize (status_bar, 300 , 20); /* so that it doesn't vanish when undocked */

    thing = gtk_frame_new (NULL);
    gtk_widget_show (thing);
    gtk_box_pack_start (GTK_BOX (hbox), thing, FALSE, FALSE, 0);
    gtk_widget_set_usize (thing, 48, 20);
    gtk_frame_set_shadow_type (GTK_FRAME (thing), GTK_SHADOW_IN);

    st_clock = gtk_clock_new (GTK_CLOCK_INCREASING);
    gtk_widget_show (st_clock);
    gtk_container_add (GTK_CONTAINER (thing), st_clock);
    gtk_widget_set_usize (st_clock, 48, 20);
    gtk_clock_set_format (GTK_CLOCK (st_clock), _("%M:%S"));
    gtk_clock_set_seconds(GTK_CLOCK (st_clock), 0);

    gnome_appbar_set_status(GNOME_APPBAR(status_bar), WELCOME_MESSAGE);
#else
    thing = gtk_hbox_new(FALSE, 1);
    gtk_box_pack_start(GTK_BOX(mainvbox), thing, FALSE, TRUE, 0);
    gtk_widget_show(thing);

    status_bar = gtk_statusbar_new();
    gtk_box_pack_start(GTK_BOX (thing), status_bar, TRUE, TRUE, 0);
    gtk_widget_show(status_bar);
    gtk_widget_set_usize(status_bar, -2, 20);

    statusbar_context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "ST Statusbar");
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), statusbar_context_id, WELCOME_MESSAGE);
#endif
    
    /* capture all key presses */
    gtk_widget_add_events(GTK_WIDGET(mainwindow), GDK_KEY_RELEASE_MASK);
    gtk_signal_connect(GTK_OBJECT(mainwindow), "key_press_event", GTK_SIGNAL_FUNC(keyevent), (gpointer)1);
    gtk_signal_connect(GTK_OBJECT(mainwindow), "key_release_event", GTK_SIGNAL_FUNC(keyevent), (gpointer)0);

    if(argc == 2) {
	gui_load_xm(argv[1]);
    } else {
	gui_new_xm();
    }

    menubar_init_prefs();

    gtk_widget_show (mainwindow);

    if(!keys_init()) {
	return 0;
    }

    if(gui_splash_window) {
	if(
#ifndef NO_GDK_PIXBUF
           gui_splash_logo ||
#endif
           tips_dialog_show_tips) {
	    gdk_window_raise(gui_splash_window->window);
//	    gtk_window_set_transient_for(GTK_WINDOW(gui_splash_window), GTK_WINDOW(mainwindow));
// (doesn't do anything on WindowMaker)
	    gui_splash_set_label(_("Ready."), TRUE);
#ifndef NO_GDK_PIXBUF
	    if(gui_splash_logo) {
		gtk_widget_add_events(gui_splash_logo_area,
				      GDK_BUTTON_PRESS_MASK);
		gtk_signal_connect (GTK_OBJECT (gui_splash_logo_area), "button_press_event",
				    GTK_SIGNAL_FUNC (gui_splash_close),
				    NULL);
	    }
#endif
	    gtk_widget_set_sensitive(gui_splash_close_button, TRUE);
	} else {
	    gui_splash_close();
	}
    }

    return 1;
}
