/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: */
/* spp_rpc_decode 
 * 
 * Purpose:
 *
 * This preprocessor normalizes the RPC requests from remote machines by
 * converting all fragments into one continous stream.
 * This is very useful for doing things like defeating hostile attackers
 * trying to stealth themselves from IDSs by fragmenting the request so the
 * string 0186A0 is broken up.
 *
 * Arguments:
 *   
 * This plugin takes a list of integers representing the TCP ports that the
 * user is interested in having normalized
 *
 * Effect:
 *
 * Changes the data in the packet payload and changes
 * p->dsize to reflect the new (smaller) payload size.
 *
 * Comments:
 *
 */
#include "spp_rpc_decode.h"
#include <ctype.h>

#define MODNAME "spp_rpc_decode"

extern char *file_name;
extern int file_line;

/* Instantiate the list of ports we're going to watch */
PortList RpcDecodePorts;

/*
 * Function: SetupRpcDecode()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupRpcDecode()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterPreprocessor("rpc_decode", RpcDecodeInit);

#ifdef DEBUG
    printf("Preprocessor: RpcDecode in setup...\n");
#endif
}


/*
 * Function: RpcDecodeInit(u_char *)
 *
 * Purpose: Processes the args sent to the preprocessor, sets up the
 *          port list, links the processing function into the preproc
 *          function list
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void RpcDecodeInit(u_char *args)
{
#ifdef DEBUG
    printf("Preprocessor: RpcDecode Initialized\n");
#endif

    /* parse the argument list into a list of ports to normalize */
    SetRpcPorts(args);

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(PreprocRpcDecode);
}

/*
 * Function: SetRpcPorts(char *)
 *
 * Purpose: Reads the list of port numbers from the argument string and
 *          parses them into the port list data struct
 *
 * Arguments: portlist => argument list
 *
 * Returns: void function
 *
 */
void SetRpcPorts(char *portlist)
{
    char **toks;
    int num_toks;
    int num_ports = 0;
    int num;

    if(*portlist == (char)NULL)
    {
        FatalError("ERROR %s (%d)=> No arguments to rpc_decode preprocessor!\n", file_name, file_line);
    }

    /* tokenize the argument list */
    toks = mSplit(portlist, " ", 31, &num_toks, '\\');

    /* convert the tokens and place them into the port list */
    for(num = 0; num < num_toks; num++)
    {
        if(isdigit((int)toks[num][0]))
        {
            RpcDecodePorts.ports[num_ports++] = atoi(toks[num]);
        }
        else
        {
            FatalError("ERROR %s(%d) => Unknown argument to rpc_decode preprocessor: \"%s\"\n", file_name, file_line, toks[num]);
        }
    }

    RpcDecodePorts.num_entries = num_ports;

#ifdef DEBUG
    printf("Decoding RPC on %d ports: ", RpcDecodePorts.num_entries);

    for(num_ports = 0; num_ports < RpcDecodePorts.num_entries; num_ports++)
    {
        printf("%d ", RpcDecodePorts.ports[num_ports]);
    }

    printf("\n");
#endif

}                                                                               
   

/*
 * Function: PreprocRpcDecode(Packet *)
 *
 * Purpose: Inspects the packet's payload for fragment records and 
 *          converts them into one infragmented record.
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void PreprocRpcDecode(Packet *p)
{
    int i = 0;           /* loop counter */

#ifdef DEBUG
    printf("rpc decoder init on %d bytes\n", p->dsize);
#endif

    /* check to make sure we're talking TCP and that the TWH has already
       completed before processing anything */
    if(!PacketIsTCP(p))
    {
#ifdef DEBUG
        printf("It isn't TCP session traffic\n");
#endif
        return;
    }

    if(!IsTcpSessionTraffic(p))
    {
#ifdef DEBUG
        printf("It isn't TCP session traffic\n");
#endif
        return;
    }

    /* check the port against the decode port list */
    for(i = 0; i < RpcDecodePorts.num_entries; i++)
    {
        if(RpcDecodePorts.ports[i] == p->dp)
        {
            ConvertRPC(p->data, p->dsize);
        }
    }
}



void ConvertRPC(u_int8_t *data, u_int16_t size)
{
    u_int8_t *rpc;       /* this is where the converted data will be written */
    u_int8_t *tmpptr;    /* this is where the converted data will be written */
    u_int8_t *index;     /* this is the index pointer to walk thru the data */
    u_int8_t *end;       /* points to the end of the payload for loop control */
    u_int16_t psize;     /* payload size */
    int i = 0;           /* loop counter */
    int length;          /* this is to store 32 bits of the data  */
    int total_len = 0;   /* this is to store 32 bits of the data  */
    u_int32_t *hdrptr;   /* points to 4 byte header...easier to grab this way */


    /* on match, normalize the data */
#ifdef DEBUG
    printf("Got RPC traffic (%d bytes)!\n", size);
#endif

    /* setup the pointers */
    hdrptr = (u_int32_t *)  data;
    rpc =   (u_int8_t *)    data;
    index = (u_int8_t *)    data;
    end =   (u_int8_t *)    data + size;
    psize = (u_int16_t)     size;

    if(size < 4)
        return;

#ifndef WORDS_BIGENDIAN
    if(((int)(htonl(*hdrptr) & 0x80000000) >> 31) == 1)
#else
        if(((int)(*hdrptr & 0x80000000) >> 31) == 1)
#endif
        {
            return;
        }

    /* now we know it's in fragmented records, 4 bytes of 
     * header(of which the most sig bit fragment (0=yes 1=no). 
     * The header is followed by the value move pointer up 4 
     * bytes, we need to stuff header in first 4 bytes.  
     * But the header has the total length...we don't know 
     * until the end 
     */
    tmpptr = rpc;
    rpc += 4;    

    while(index < end)
    {
        /* get the fragment length (31 bits) and move the pointer to the
           start of the actual data */
        hdrptr = (int *) index;

#ifndef WORDS_BIGENDIAN
        length = (int)(htonl(*hdrptr) & 0x7FFFFFFF);
#else
        length = (int)(*hdrptr & 0x7FFFFFFF);
#endif
        if((index + 4 + length) > end)
        {
            DebugMessage(DEBUG_FLOW, "WARNING: rpc_decode calculated bad "
                    "length: %d\n", length);
            return;
        }
        else
        {
            total_len += length;
            index += 4;
            for (i=0; i < length; i++,rpc++,index++,hdrptr++)
                *rpc = *index;
        }
    }

    /* point to beginning again  */
    rpc = tmpptr;        

    /* we need to add header to first 4 bytes  */
    *rpc++ = (char) 0x80;
    *rpc++ = (char) 0x0;
    *rpc++ = (char) 0x0;
    *rpc = total_len;

    /* set the payload size to reflect the new size */ 
    size = 4+total_len;

//#ifdef DEBUG
    printf("New size: %d\n", size);
    printf("converted data:\n");
    ClearDumpBuf();
    PrintNetData(stdout, data, size);
    ClearDumpBuf();
//#endif

    return;
}

