/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spp_bo.c,v 1.5.4.1 2002/03/15 14:42:32 chrisgreen Exp $ */
/* Snort Preprocessor Plugin Source File Bo */

/* spp_bo 
 * 
 * Purpose: Detects Back Orifice traffic by brute forcing the weak encryption
 *          of the program's network protocol and detects the magic cookie
 *          that it's servers and clients require to communicate with each 
 *          other.
 *
 * Arguments: "nobrute" => turn off brute forcing, just look for packets with
 *                         the default key 
 *            <number>  => set the default key to something other than 31337
 *   
 * Effect: Analyzes UDP traffic for the BO magic cookie, reports if it finds
 *         traffic matching the profile.
 *
 * Comments:
 *
 */

#include "spp_bo.h"

#define MODNAME "spp_bo"

int BoRand();
void ParseBoArgs(char *);

/* external globals from rules.c */
extern char *file_name;
extern int file_line;

/* global keyvalue for the BoRand() function */
static long holdrand = 1L;

/* brute forcing is on by default */
int brute_force_enable = 1;
int default_key;

/*
 * Function: SetupBo()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.  
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupBo()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterPreprocessor("bo", BoInit);

#ifdef DEBUG
    printf("Preprocessor: Back Orifice is setup...\n");
#endif
}


/*
 * Function: BoInit(u_char *)
 *
 * Purpose: Link the BO preprocessor to the preperocessor call chain.
 *
 * Arguments: args => ptr to argument string (spp_bo takes no args)
 *
 * Returns: void function
 *
 */
void BoInit(u_char *args)
{
#ifdef DEBUG
    printf("Preprocessor: Bo Initialized\n");
#endif

    if(args != NULL)
    {
        ParseBoArgs(args);
    }

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(BoProcess);
}


void ParseBoArgs(char *args)
{
    int num_toks;
    char **toks;
    int i;

    /* tokenize the argument list */
    toks = mSplit(args, " ", 2, &num_toks, '\\');
    
    for(i=0; i<num_toks; i++)
    {
        if(!strncasecmp("-nobrute", toks[0], strlen("-nobrute")))
        {
            brute_force_enable = 0;
            if(!pv.quiet_flag)
                printf("Back Orifice detection brute force: DISABLED\n");
        }
        else if(isdigit((int)toks[i][0]))
        {
            default_key = atoi(toks[i]);
        }
        else
        {
            FatalError("ERROR %s(%d) => Unknown argument to back orifice decoder: \"%s\"\n", file_name, file_line, toks[i]);
        }
    }
}

/*
 * Function: BoRand()
 *
 * Purpose: Back Orifice "encryption" algorithm
 *
 * Arguments: None.
 *
 * Returns: key to XOR with current char to be "encrypted"
 */
int BoRand()
{
    return(((holdrand = holdrand * 214013L + 2531011L) >> 16) & 0x7fff);
}


/*
 * Function: BoProcess(Packet *)
 *
 * Purpose: Look for the magic cookie, squawk if you find it.
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void BoProcess(Packet *p)
{
    char *pkt_data;
    char *magic_data;
    char *end;
    char *magic_cookie = "*!*QWTY?";
    int protoclown = 0;
    int detect = 1;
    char foo;
    char outstring[STD_BUF];
    Event event;

    DebugMessage(DEBUG_PLUGIN, "   -> spp_bo: Checking for BO traffic...\n");

    /* make sure it's UDP and that it's at least 19 bytes long */
    if(!PacketIsUDP(p))
    {
        DebugMessage(DEBUG_PLUGIN, 
                    "   -> spp_bo: Not UDP\n");
        return;
    }

    if(p->dsize < BACKORIFICE_MIN_SIZE)
    {
        return;
    }

    /* we're not going to analyze DNS or RPC traffic */
    if(p->dp == 111 || p->dp == 53 || p->dp == 32771 || 
       p->dp == 137 || p->dp == 138)
    {
        DebugMessage(DEBUG_PLUGIN, 
                     "   -> spp_bo: bailing for destination port...\n");
        return;
    }

    DebugMessage(DEBUG_PLUGIN, "  -> Checking possible BackOrifice Traffic!\n");

    /* set a pointer to the packet data */
    pkt_data = p->data;

    /* set a ptr to the beginning of the converted data accumulator */
    magic_data = magic_cookie;

    /* set the end ptr */
    end = pkt_data + BACKORIFICE_MAGIC_SIZE;

    /* clown SMASH! */
    if(!default_key)
    {
        protoclown = holdrand = BACKORIFICE_DEFAULT_KEY;
    }
    else
    {
        protoclown = holdrand = default_key;
    }

    /* convert the first 8 bytes of the packet looking for the BO protocol
     * magic cookie
     */ 
    while(pkt_data<end)
    {
        foo = (char) (*pkt_data ^ (BoRand()%256));

        if(*magic_data != foo)
        {
            detect = 0;
            DebugMessage(DEBUG_PLUGIN, 
                    "Failed check one on 0x%X : 0x%X\n", 
                    *magic_data, foo);

            goto check2;
        }

        magic_data++;
        pkt_data++;
    }

check2:
    /* check for a successful key match */
    if(!detect)
    {
        /* try dest port * 2 for the key, hey it works in the movies */
        protoclown = holdrand = p->dp << 1;;

        /* reset the buffer ptrs */
        pkt_data = p->data;
        magic_data = magic_cookie;
        detect = 1;

        /* crank on through to the other side */
        while(pkt_data<end)
        {
            foo = (char) (*pkt_data ^ (BoRand()%256));

            if(*magic_data != foo)
            {
                detect = 0;
                DebugMessage(DEBUG_PLUGIN, "Failed check two on 0x%X\n", 
                        *magic_data);
                if(brute_force_enable)
                    goto check3;
                else
                    return;
            }

            magic_data++;
            pkt_data++;
        }

check3:
        /* if we still didn't decrypt it properly, brute force the fucker */
        if(!detect)
        {
            /* this loop ought to take a while... */
            /* is there a mathematician in the house? */
            for(protoclown=0;protoclown<65536;protoclown++)
            {
                holdrand = protoclown;
                pkt_data = p->data;
                magic_data = magic_cookie;
                detect = 1;

                while(pkt_data<end)
                {
                    foo = (char) (*pkt_data ^ (BoRand()%256));

                    if(*magic_data != foo)
                    {
                        detect = 0;
                        DebugMessage(DEBUG_PLUGIN,
                                     "Failed brute force key(%d) on 0x%X\n", 
                                     protoclown, *magic_data);
                        goto loopbail;
                    }

                    magic_data++;
                    pkt_data++;
                }

loopbail:                
                if(detect)
                {
                    /* why have 1 goto when you can have 4...? */
                    goto finish;
                }
            }
        }
    }

finish:
    /* wow, what do you know, it was BO traffic after all... */
    if(detect)
    {
        DebugMessage(DEBUG_PLUGIN, "Detected Back Orifice Data!\n");
        DebugMessage(DEBUG_PLUGIN, "hash value: %d\n", protoclown);
        snprintf(outstring, STD_BUF-1, 
                "spp_bo: Back Orifice Traffic detected (key: %d)", protoclown);
        SetEvent(&event, GENERATOR_SPP_BO, BO_TRAFFIC_DETECT, 1, 0, 0, 0);
        CallAlertFuncs(p, outstring, NULL, &event);
    }

    return;
}


