/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: sp_pattern_match.c,v 1.22.4.3 2002/03/15 14:42:31 chrisgreen Exp $ */
#include "sp_pattern_match.h"

extern int file_line;
extern char *file_name;

int list_file_line;     /* current line being processed in the list
                 * file */
void SetupPatternMatch()
{
    RegisterPlugin("content", PayloadSearchInit);
    RegisterPlugin("content-list", PayloadSearchListInit);
    RegisterPlugin("offset", PayloadSearchOffset);
    RegisterPlugin("depth", PayloadSearchDepth);
    RegisterPlugin("nocase", PayloadSearchNocase);
    RegisterPlugin("regex", PayloadSearchRegex);
    RegisterPlugin("uricontent", PayloadSearchUri);

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "Plugin: PatternMatch Initialized!\n");
#endif
}


void PayloadSearchListInit(char *data, OptTreeNode * otn, int protocol)
{
    char *sptr;
    char *eptr;

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "In PayloadSearchListInit()\n");
#endif

    /* get the path/file name from the data */
    while(isspace((int) *data))
        data++;

    /* grab everything between the starting " and the end one */
    sptr = index(data, '"');
    eptr = strrchr(data, '"');

    if(sptr != NULL && eptr != NULL)
    {
        /* increment past the first quote */
        sptr++;

        /* zero out the second one */
        *eptr = 0;
    }
    else
    {
        sptr = data;
    }

    /* read the content keywords from the list file */
    ParseContentListFile(sptr, otn, protocol);


    /* link the plugin function in to the current OTN */
    AddOptFuncToList(CheckORPatternMatch, otn);

    return;
}


void PayloadSearchInit(char *data, OptTreeNode * otn, int protocol)
{
#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "In PayloadSearchInit()\n");
#endif

    /* whack a new node onto the list */
    NewNode(otn);

    /* set up the pattern buffer */
    ParsePattern(data, otn);

    /* link the plugin function in to the current OTN */
    AddOptFuncToList(CheckANDPatternMatch, otn);

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "OTN function PatternMatch Added to rule!\n");
#endif
}



void PayloadSearchUri(char *data, OptTreeNode * otn, int protocol)
{
#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "In PayloadSearchUri()\n");
#endif

    /* whack a new node onto the list */
    NewNode(otn);

    /* set up the pattern buffer */
    ParsePattern(data, otn);

    /* link the plugin function in to the current OTN */
    AddOptFuncToList(CheckUriPatternMatch, otn);

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "OTN function PatternMatch Added to rule!\n");
#endif
}




void PayloadSearchOffset(char *data, OptTreeNode * otn, int protocol)
{
    PatternMatchData *idx;

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "In PayloadSearch()\n");
#endif

    idx = otn->ds_list[PLUGIN_PATTERN_MATCH];

    if(idx == NULL)
    {
        FatalError("ERROR %s Line %d => Please place \"content\" rules before depth, nocase or offset modifiers.\n", file_name, file_line);
    }

    while(idx->next != NULL)
        idx = idx->next;

    while(isspace((int) *data))
        data++;

    idx->offset = strtol(data, NULL, 10);

    if(errno == ERANGE)
    {
        FatalError("ERROR %s Line %d => Range problem on offset value\n", file_name, file_line);
    }

    if(idx->offset > 65535)
    {
        FatalError("ERROR %s Line %d => Offset greater than max Ipv4 packet size\n", file_name, file_line);
    }

    DebugMessage(DEBUG_PARSER, "Pattern offset = %d\n", idx->offset);

    return;
}



void PayloadSearchDepth(char *data, OptTreeNode * otn, int protocol)
{
    PatternMatchData *idx;

    idx = (PatternMatchData *) otn->ds_list[PLUGIN_PATTERN_MATCH];

    if(idx == NULL)
    {
        FatalError("ERROR %s Line %d => Please place \"content\" rules "
                "before depth, nocase or offset modifiers.\n", 
                file_name, file_line);
    }

    while(idx->next != NULL)
        idx = idx->next;

    while(isspace((int) *data))
        data++;

    idx->depth = strtol(data, NULL, 10);

    if(errno == ERANGE)
    {
        FatalError("ERROR %s Line %d => Range problem on depth value\n", 
                file_name, file_line);
    }

    if(idx->offset > 65535)
    {
        FatalError("ERROR %s Line %d => Depth greater than max Ipv4 packet size\n", file_name, file_line);
    }

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "Pattern offset = %d\n", idx->offset);
#endif

    return;
}



void PayloadSearchNocase(char *data, OptTreeNode * otn, int protocol)
{
    PatternMatchData *idx;
    int i;

    idx = (PatternMatchData *) otn->ds_list[PLUGIN_PATTERN_MATCH];

    if(idx == NULL)
    {
        FatalError("ERROR Line %d => Please place \"content\" rules before"
                " depth, nocase or offset modifiers.\n", file_line);
    }
    while(idx->next != NULL)
        idx = idx->next;

    idx->search = mSearchCI;

    i = idx->pattern_size;

    while(--i >= 0)
        idx->pattern_buf[i] = toupper((unsigned char) idx->pattern_buf[i]);

    free(idx->skip_stride);
    idx->skip_stride = make_skip(idx->pattern_buf, idx->pattern_size);

    free(idx->shift_stride);
    idx->shift_stride = make_shift(idx->pattern_buf, idx->pattern_size);

    return;
}



void PayloadSearchRegex(char *data, OptTreeNode * otn, int protocol)
{
    PatternMatchData *idx;
    int i;

    idx = (PatternMatchData *) otn->ds_list[PLUGIN_PATTERN_MATCH];

    if(idx == NULL)
    {
        FatalError("ERROR %s Line %d => Please place \"content\" rules "
                   "before regex modifiers.\n", file_name, file_line);
    }

    while(idx->next != NULL)
        idx = idx->next;

    idx->search = mSearchREG;

    i = idx->pattern_size;

    /*while(--i >= 0)
    {
        printf("uppercasing 0x%X", idx->pattern_buf[i]);
        idx->pattern_buf[i] = toupper((unsigned char) idx->pattern_buf[i]);
        printf("to 0x%X (%c)\n", idx->pattern_buf[i], idx->pattern_buf[i]);
    }*/

    free(idx->skip_stride);
    idx->skip_stride = make_skip(idx->pattern_buf, idx->pattern_size);

    free(idx->shift_stride);
    idx->shift_stride = make_shift(idx->pattern_buf, idx->pattern_size);

    return;
}




void NewNode(OptTreeNode * otn)
{
    PatternMatchData *idx;

    idx = (PatternMatchData *) otn->ds_list[PLUGIN_PATTERN_MATCH];

    if(idx == NULL)
    {
        if((otn->ds_list[PLUGIN_PATTERN_MATCH] = 
                    (PatternMatchData *) calloc(sizeof(PatternMatchData), 
                                                sizeof(char))) == NULL)
        {
            FatalError("ERROR => sp_pattern_match NewNode() calloc failed!\n");
        }
    }
    else
    {
        idx = otn->ds_list[PLUGIN_PATTERN_MATCH];

        while(idx->next != NULL)
            idx = idx->next;

        if((idx->next = (PatternMatchData *) 
                    calloc(sizeof(PatternMatchData), sizeof(char))) == NULL)
        {
            FatalError("ERROR => sp_pattern_match NewNode() calloc failed!\n");
        }
    }
}



/****************************************************************************
 *
 * Function: ParsePattern(char *)
 *
 * Purpose: Process the application layer patterns and attach them to the
 *          appropriate rule.  My god this is ugly code.
 *
 * Arguments: rule => the pattern string
 *
 * Returns: void function
 *
 ***************************************************************************/
void ParsePattern(char *rule, OptTreeNode * otn)
{
    unsigned char tmp_buf[2048];

    /* got enough ptrs for you? */
    char *start_ptr;
    char *end_ptr;
    char *idx;
    char *dummy_idx;
    char *dummy_end;
    char hex_buf[9];
    u_int dummy_size = 0;
    u_int size;
    int hexmode = 0;
    int hexsize = 0;
    int pending = 0;
    int cnt = 0;
    int literal = 0;
    int exception_flag = 0;
    PatternMatchData *ds_idx;

    /* clear out the temp buffer */
    bzero(tmp_buf, 2048);

    while(isspace((int)*rule))
        rule++;

    if(*rule == '!')
    {
        exception_flag = 1;
    }

    /* find the start of the data */
    start_ptr = index(rule, '"');

    if(start_ptr == NULL)
    {
        FatalError("ERROR %s Line %d => Content data needs to be "
                    "enclosed in quotation marks (\")!\n", 
                    file_name, file_line);
    }

    /* move the start up from the beggining quotes */
    start_ptr++;

    /* find the end of the data */
    end_ptr = strrchr(start_ptr, '"');

    if(end_ptr == NULL)
    {
        FatalError("ERROR %s Line %d => Content data needs to be enclosed "
                   "in quotation marks (\")!\n", file_name, file_line);
    }

    /* set the end to be NULL */
    *end_ptr = 0;

    /* how big is it?? */
    size = end_ptr - start_ptr;

    /* uh, this shouldn't happen */
    if(size <= 0)
    {
        FatalError("ERROR %s Line %d => Bad pattern length!\n", 
                   file_name, file_line);
    }
    /* set all the pointers to the appropriate places... */
    idx = start_ptr;

    /* set the indexes into the temp buffer */
    dummy_idx = tmp_buf;
    dummy_end = (dummy_idx + size);

    /* why is this buffer so small? */
    bzero(hex_buf, 9);
    memset(hex_buf, '0', 8);

    /* BEGIN BAD JUJU..... */
    while(idx < end_ptr)
    {
        DebugMessage(DEBUG_PARSER, "processing char: %c\n", *idx);
        switch(*idx)
        {
            case '|':
                DebugMessage(DEBUG_PARSER, "Got bar... ");
                if(!literal)
                {
                    DebugMessage(DEBUG_PARSER, "not in literal mode... ");
                    if(!hexmode)
                    {
                        DebugMessage(DEBUG_PARSER, "Entering hexmode\n");
                        hexmode = 1;
                    }
                    else
                    {
                        DebugMessage(DEBUG_PARSER, "Exiting hexmode\n");
                        hexmode = 0;
                        pending = 0;
                    }

                    if(hexmode)
                        hexsize = 0;
                }
                else
                {
                    DebugMessage(DEBUG_PARSER, "literal set, Clearing\n");
                    literal = 0;
                    tmp_buf[dummy_size] = start_ptr[cnt];
                    dummy_size++;
                }

                break;

            case '\\':
                DebugMessage(DEBUG_PATTERN_MATCH, "Got literal char... ");

                if(!literal)
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, "Setting literal\n");

                    literal = 1;
                }
                else
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, "Clearing literal\n");
                    tmp_buf[dummy_size] = start_ptr[cnt];
                    literal = 0;
                    dummy_size++;
                }

                break;
            case '"':
                if (!literal) {
                    FatalError("ERROR %s Line %d => Non-escaped "
                   " '\"' character!\n", file_name, file_line);
                }
            /* otherwise process the character as default */
            default:
                if(hexmode)
                {
                    if(isxdigit((int) *idx))
                    {
                        hexsize++;

                        if(!pending)
                        {
                            hex_buf[7] = *idx;
                            pending++;
                        }
                        else
                        {
                            hex_buf[8] = *idx;
                            pending--;

                            if(dummy_idx < dummy_end)
                            {
                                tmp_buf[dummy_size] = (u_long) 
                                            strtol(hex_buf, (char **) NULL, 16);

                                dummy_size++;
                                bzero(hex_buf, 9);
                                memset(hex_buf, '0', 8);
                            }
                            else
                            {
                                FatalError("ERROR => ParsePattern() dummy "
                                        "buffer overflow, make a smaller "
                                        "pattern please! (Max size = 2048)\n");
                            }
                        }
                    }
                    else
                    {
                        if(*idx != ' ')
                        {
                            FatalError("ERROR %s Line %d => What is this "
                                    "\"%c\"(0x%X) doing in your binary "
                                    "buffer?  Valid hex values only please! "
                                    "(0x0 - 0xF) Position: %d\n",
                                    file_name, 
                                    file_line, (char) *idx, (char) *idx, cnt);
                        }
                    }
                }
                else
                {
                    if(*idx >= 0x1F && *idx <= 0x7e)
                    {
                        if(dummy_idx < dummy_end)
                        {
                            tmp_buf[dummy_size] = start_ptr[cnt];
                            dummy_size++;
                        }
                        else
                        {
                            FatalError("ERROR %s Line %d=> ParsePattern() "
                            "dummy buffer overflow!\n", file_name, file_line);
                        }

                        if(literal)
                        {
                            literal = 0;
                        }
                    }
                    else
                    {
                        if(literal)
                        {
                            tmp_buf[dummy_size] = start_ptr[cnt];
                            dummy_size++;
                            DebugMessage(DEBUG_PARSER, "Clearing literal\n");
                            literal = 0;
                        }
                        else
                        {
                            FatalError("ERROR %s Line %d=> character value out "
                                    "of range, try a binary buffer dude\n", 
                                    file_name, file_line);
                        }
                    }
                }

                break;
        }

        dummy_idx++;
        idx++;
        cnt++;
    }
    /* ...END BAD JUJU */

    /* error prunning */

    if (literal) {
        FatalError("ERROR %s Line %d=> backslash escape is not "
                    "completed\n", file_name, file_line);
    }
    if (hexmode) {
        FatalError("ERROR %s Line %d=> hexmode is not "
                    "completed\n", file_name, file_line);
    }

    ds_idx = (PatternMatchData *) otn->ds_list[PLUGIN_PATTERN_MATCH];

    while(ds_idx->next != NULL)
        ds_idx = ds_idx->next;

    if((ds_idx->pattern_buf = (char *) calloc(dummy_size+1, sizeof(char))) 
            == NULL)
    {
        FatalError("ERROR => ParsePattern() pattern_buf malloc filed!\n");
    }

    memcpy(ds_idx->pattern_buf, tmp_buf, dummy_size);

    ds_idx->pattern_size = dummy_size;

    ds_idx->search = mSearch;

    ds_idx->skip_stride = make_skip(ds_idx->pattern_buf, ds_idx->pattern_size);
    ds_idx->shift_stride = make_shift(ds_idx->pattern_buf, ds_idx->pattern_size);
    ds_idx->exception_flag = exception_flag;

    return;
}




int CheckORPatternMatch(Packet * p, struct _OptTreeNode * otn_idx, 
        OptFpList * fp_list)
{
    int sub_depth;
    int found = 0;
    PatternMatchData *idx;

    DebugMessage(DEBUG_PATTERN_MATCH, "CheckPatternORMatch: ");
    
    idx = otn_idx->ds_list[PLUGIN_PATTERN_MATCH];

    while(idx != NULL)
    {
        if(idx->offset > p->dsize)
        {
            DebugMessage(DEBUG_PATTERN_MATCH, "Initial offset larger than "
                    "payload!\n");

            goto sizetoosmall;
        }
        else
        {
            /* do some tests to make sure we stay in bounds */
            if((idx->depth + idx->offset) > p->dsize)
            {
                /* we want to check only depth bytes anyway */
                sub_depth = p->dsize - idx->offset; 

                if((sub_depth > 0) && (sub_depth >= (int)idx->pattern_size))
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, 
                            "testing pattern: %s\n", idx->pattern_buf);

                    found = (idx->search((char *)(p->data + idx->offset), 
                                        sub_depth, idx->pattern_buf,
                                        idx->pattern_size, idx->skip_stride, 
                                        idx->shift_stride)
                            ^ idx->exception_flag);
#ifdef DEBUG
                    if(!found)
                    {
                        DebugMessage(DEBUG_PATTERN_MATCH, 
                                "Pattern Match failed!\n");
                    }
#endif
                }
            }
            else
            {
                DebugMessage(DEBUG_PATTERN_MATCH, "Testing pattern (lower "
                        "section): %s\n", idx->pattern_buf);

                if(idx->depth && (p->dsize-idx->offset> idx->depth))
                {
                    found = (idx->search((char *)(p->data + idx->offset), 
                                        idx->depth, idx->pattern_buf,
                                        idx->pattern_size, idx->skip_stride, 
                                        idx->shift_stride)
                            ^ idx->exception_flag);
                }
                else
                {
                    found = (idx->search((char *)(p->data + idx->offset), 
                                        p->dsize - idx->offset ,
                                        idx->pattern_buf, idx->pattern_size, 
                                        idx->skip_stride, idx->shift_stride)
                            ^ idx->exception_flag);
                }
#ifdef DEBUG
                if(!found)
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, 
                            "Pattern Match failed! Exit the loop.\n");
                }
#endif
            }
        }

        DebugMessage(DEBUG_PATTERN_MATCH, "Checking the results\n");

        if(found)
        {
            DebugMessage(DEBUG_PATTERN_MATCH, "Pattern Match "
                    "successful: %s!\n", idx->pattern_buf);

            return fp_list->next->OptTestFunc(p, otn_idx, fp_list->next);

        }
#ifdef DEBUG
        else
        {
            DebugMessage(DEBUG_PATTERN_MATCH, "Pattern match failed\n");
        }
#endif

        DebugMessage(DEBUG_PATTERN_MATCH, "Stepping to next content keyword\n");

sizetoosmall:

        idx = idx->next;
    }

    DebugMessage(DEBUG_PATTERN_MATCH, "No more keywords, exiting... \n");

    return 0;
}



int CheckANDPatternMatch(Packet *p, struct _OptTreeNode *otn_idx, 
        OptFpList *fp_list)
{
    int sub_depth;
    int found = 0;
    PatternMatchData *idx;

    DebugMessage(DEBUG_PATTERN_MATCH, "CheckPatternANDMatch: ");

    idx = otn_idx->ds_list[PLUGIN_PATTERN_MATCH];

    while(idx != NULL)
    {
        if(idx->offset > p->dsize)
        {
            DebugMessage(DEBUG_PATTERN_MATCH, 
                    "Initial offset larger than payload!\n");
            return 0;
        }
        else
        {
            /* do some tests to make sure we stay in bounds */
            if((idx->depth + idx->offset) > p->dsize)
            {
                /* we want to match depth bytes anyway */
                sub_depth = p->dsize - idx->offset; 

                if((sub_depth > 0) && (sub_depth >= (int)idx->pattern_size))
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, "testing pattern: %s\n", 
                            idx->pattern_buf);

                    found = (idx->search((char *)(p->data+idx->offset), 
                                        sub_depth,idx->pattern_buf,
                                        idx->pattern_size, idx->skip_stride, 
                                        idx->shift_stride)
                            ^ idx->exception_flag);

                    if(!found)
                    {
                        DebugMessage(DEBUG_PATTERN_MATCH, 
                                "Pattern Match failed!\n");

                        return 0;
                    }
                } 
                else 
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, 
                            "Pattern Match failed -- sub_depth: %d < "
                            "(int)idx->pattern_size: %d!\n",
                            sub_depth, (int)idx->pattern_size);
                    return 0;
                }
		    
            }
            else
            {
                DebugMessage(DEBUG_PATTERN_MATCH, 
                        "Testing pattern (lower section): %s\n", 
                        idx->pattern_buf);

                /* if depth field is present and we don't go over the 
                 * dsize boundary with it 
                 */
                if(idx->depth && (p->dsize-idx->offset> idx->depth))
                {
                    found = (idx->search((char *)(p->data+idx->offset), 
                                        idx->depth, idx->pattern_buf,
                                        idx->pattern_size, idx->skip_stride, 
                                        idx->shift_stride) 
                            ^ idx->exception_flag);
                }
                else
                {
                    found = (idx->search((char *)(p->data+idx->offset), 
                                        p->dsize - idx->offset,
                                        idx->pattern_buf, idx->pattern_size, 
                                        idx->skip_stride, idx->shift_stride)
                            ^ idx->exception_flag);
                }

                if(!found)
                {
                    DebugMessage(DEBUG_PATTERN_MATCH, 
                            "Pattern Match failed!\n");

                    return 0;
                }
            }
        }

        idx = idx->next;

        DebugMessage(DEBUG_PATTERN_MATCH, 
                "Stepping to next content keyword...\n");
    }

    if(found)
    {
        DebugMessage(DEBUG_PATTERN_MATCH, "Pattern Match successful!\n");

        return fp_list->next->OptTestFunc(p, otn_idx, fp_list->next);

    }
#ifdef DEBUG
    else
    {
        DebugMessage(DEBUG_PATTERN_MATCH, "Pattern match failed\n");
    }
#endif

    return 0;
}


/************************************************************************/
/************************************************************************/
/************************************************************************/

int CheckUriPatternMatch(Packet *p, struct _OptTreeNode *otn_idx, 
        OptFpList *fp_list)
{
    int sub_depth;
    int found = 0;
    PatternMatchData *idx;
    int dsize = 0; 
    char *data = NULL;

    if(p->URI.uri != NULL)
    {
        data = p->URI.uri;
        dsize = p->URI.length;
    }
    else
    {
        data = p->data;
        dsize = p->dsize;
    }

    DebugMessage(DEBUG_PLUGIN, "CheckUriPatternMatch: ");

    idx = otn_idx->ds_list[PLUGIN_PATTERN_MATCH];

    while(idx != NULL)
    {
        if(idx->offset > dsize)
        {
            DebugMessage(DEBUG_PLUGIN, "Initial offset larger than payload!\n");

            return 0;
        }
        else
        {
            /* do some tests to make sure we stay in bounds */
            if((idx->depth + idx->offset) > dsize)
            {
                /* we want to match depth bytes anyway */
                sub_depth = dsize - idx->offset; 

                if((sub_depth > 0) && (sub_depth >= (int)idx->pattern_size))
                {
                    DebugMessage(DEBUG_PLUGIN, "testing pattern: %s\n", 
                            idx->pattern_buf);

                    found = (idx->search((char *)(data+idx->offset), 
                                sub_depth,idx->pattern_buf,
                                idx->pattern_size, idx->skip_stride, 
                                idx->shift_stride)
                            ^ idx->exception_flag);

                    if(!found)
                    {
                        DebugMessage(DEBUG_PLUGIN, "Pattern Match failed!\n");

                        return 0;
                    }
                }
                else
                {
                    return 0;
                }
            }
            else
            {
                DebugMessage(DEBUG_PLUGIN, "Testing pattern (lower "
                        "section): %s\n", idx->pattern_buf);
                
                /* if depth field is present and we don't go over the 
                 * dsize boundary with it 
                 */
                if(idx->depth && (dsize-idx->offset> idx->depth))
                {
                    found = (idx->search((char *)(data+idx->offset), 
                                idx->depth, idx->pattern_buf,
                                idx->pattern_size, idx->skip_stride, 
                                idx->shift_stride) 
                            ^ idx->exception_flag);
                }
                else
                {
                    found = (idx->search((char *)(data+idx->offset), 
                                dsize - idx->offset,
                                idx->pattern_buf, idx->pattern_size, 
                                idx->skip_stride, idx->shift_stride)
                            ^ idx->exception_flag);
                }

                if(!found)
                {
                    DebugMessage(DEBUG_PLUGIN, "Pattern Match failed!\n");

                    return 0;
                }
            }
        }

        idx = idx->next;

        DebugMessage(DEBUG_PLUGIN, "Stepping to next content keyword...\n");
    }

    if(found)
    {
        DebugMessage(DEBUG_PLUGIN, "Pattern Match successful!\n");

        return fp_list->next->OptTestFunc(p, otn_idx, fp_list->next);
    }

    DebugMessage(DEBUG_PLUGIN, "Pattern match failed\n");

    return 0;


}



/****************************************************************************
 *
 * Function: ParseContentListFile(char *, OptTreeNode *, int protocol)
 *
 * Purpose:  Read the content_list file a line at a time, put the content of
 *           the line into buffer
 *
 * Arguments:otn => rule including the list
 *           file => list file filename
 *	     protocol => protocol
 *
 * Returns: void function
 *
 ***************************************************************************/
void ParseContentListFile(char *file, OptTreeNode * otn, int protocol)
{
    FILE *thefp;                /* file pointer for the content_list file */
    char buf[STD_BUF+1];        /* file read buffer */
    char rule_buf[STD_BUF+1];   /* content keyword buffer */
    int frazes_count;           /* frazes counter */


#ifdef DEBUG
    PatternMatchData *idx;
    DebugMessage(DEBUG_PATTERN_MATCH, "Opening content_list file: %s\n", file);
#endif

    /* open the list file */
    if((thefp = fopen(file, "r")) == NULL)
    {
        FatalError("Unable to open list file: %s\n", file);
    }
    /* clear the line and rule buffers */
    bzero((char *) buf, STD_BUF);
    bzero((char *) rule_buf, STD_BUF);
    frazes_count = 0;

    /* loop thru each list_file line and content to the rule */
    while((fgets(buf, STD_BUF-2, thefp)) != NULL)
    {
        /* inc the line counter */
        list_file_line++;

        DebugMessage(DEBUG_PATTERN_MATCH, "Got line %d: %s", 
                list_file_line, buf);

        /* if it's not a comment or a <CR>, send it to the parser */
        if((buf[0] != '#') && (buf[0] != 0x0a) && (buf[0] != ';'))
        {
            DebugMessage(DEBUG_PATTERN_MATCH, 
                    "Adding content keyword: %s", buf);

            frazes_count++;
            strip(buf);

            NewNode(otn);

            /* check and add content keyword */
            ParsePattern(buf, otn);

            DebugMessage(DEBUG_PATTERN_MATCH, "Content keyword %s\" added!\n", 
                    buf);
        }
    }

#ifdef DEBUG
    DebugMessage(DEBUG_PATTERN_MATCH, "%d frazes read...\n", frazes_count);

    idx = (PatternMatchData *) otn->ds_list[PLUGIN_PATTERN_MATCH];
    
    if(idx == NULL)
    {
        DebugMessage(DEBUG_PATTERN_MATCH, "No patterns loaded\n");
    }
    else
    {
        while(idx != NULL)
        {
            DebugMessage(DEBUG_PATTERN_MATCH, "Pattern = %s\n", 
                    idx->pattern_buf);
            idx = idx->next;
        }
    }
#endif

    fclose(thefp);

    return;
}
