/* $Id: spp_frag2.c,v 1.25 2001/11/26 05:34:57 roesch Exp $ */

/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/*
 * Bugfixes;
 *
 *  Aug  1 15:28:36 MDT 2001, cpw: changed to use packet time for all time
 *  comparisons.  Modified calling sequence on a number subroutines to
 *  pass down current FragTracker and time to PruneFragCache.  The
 *  current FragTracker was getting released early by PruneFragCache
 *  which led to a segmentation fault.  I'd suggest a look see at
 *  PruneFragCache to see that is doing things in the right order.  Not
 *  knowing how it all works, I assumed that the loop was from oldest to
 *  newest.
 *  Using snort compiled with this version I was able to successfully
 *  process a gaggle of frags (GOF).  An individual packet looked like this:
 *  14:11:21.020324 a.11071 > b.11070: udp 8000 (frag 62829:1480@0+)
 *  14:11:21.020327 a > b: frag 1480 (frag 62829:1480@1480+)
 *  14:11:21.020329 a > b: frag 1480 (frag 62829:1480@2960+)
 *  14:11:21.020412 a > b: frag 1480 (frag 62829:1480@4440+)
 *  14:11:21.020416 a > b: frag 1480 (frag 62829:1480@5920+)
 *  14:11:21.020418 a > 134.253.164.21: frag 608 (frag 62829:608@7400)
 *  There were about 301066 full packets read from a 821947618 byte file.
 *
 *  Prior to this patch, I got the following seg fault:
 *  #0  ubi_btInsert (RootPtr=0x48, NewNode=0x8a94f10, ItemPtr=0x8a94f10,
 *      OldNode=0xbfffee7c) at ubi_BinTree.c:637
 *  #1  0x8077825 in ubi_sptInsert (RootPtr=0x48, NewNode=0x8a94f10,
 *      ItemPtr=0x8a94f10, OldNode=0x0) at ubi_SplayTree.c:317
 *  #2  0x807c08e in InsertFrag (p=0xbfffef5c, ft=0x874bdc8) at spp_frag2.c:534
 *  #3  0x807be82 in Frag2Defrag (p=0xbfffef5c) at spp_frag2.c:430
 *  #4  0x8058416 in Preprocess (p=0xbfffef5c) at rules.c:3427
 *  ...
 */

/*  I N C L U D E S  ************************************************/
#include "spp_frag2.h"
#include "ubi_SplayTree.h"

/*  D E F I N E S  **************************************************/
#define FRAG_GOT_FIRST      0x00000001
#define FRAG_GOT_LAST       0x00000002

#define FRAG_PRUNE_QUANTA   60
#define FRAG_MEMCAP         4194304  

#if defined (SOLARIS) || defined (SUNOS)
#define SPARC_TWIDDLE       2
#else
#define SPARC_TWIDDLE       0
#endif

/*  D A T A   S T R U C T U R E S  **********************************/
typedef struct _Frag2Data
{
    u_int8_t os_flags;
    u_int32_t memcap;
    u_int32_t frag_timeout;
    u_int32_t last_prune_time;
    u_int8_t stop_traverse;

    u_int16_t frag_pad;
} Frag2Data;


typedef struct _Frag2Frag
{
    ubi_trNode Node;
    
    u_int8_t *data;
    u_int16_t size;
    u_int16_t offset;
} Frag2Frag;

typedef struct _FragTracker
{
    ubi_trNode Node;

    u_int32_t sip;          /* src IP */
    u_int32_t dip;          /* dst IP */
    u_int16_t id;           /* IP ID */
    u_int8_t protocol;      /* IP protocol */
    
    u_int32_t frag_flags;
    u_int32_t last_frag_time;

    u_int32_t frag_bytes;
    u_int32_t calculated_size;
    u_int32_t frag_pkts;
    
    ubi_trRoot fraglist;
    ubi_trRootPtr fraglistPtr;
} FragTracker;

/*  G L O B A L S  **************************************************/
static ubi_trRoot f_cache;
static ubi_trRootPtr FragRootPtr = &f_cache;
    
extern char *file_name;
extern int file_line;

int frag_mem_usage;

u_int16_t next_offset;
u_int32_t frag2_alloc_faults;

Frag2Data f2data;
Packet *defrag_pkt;


/*  P R O T O T Y P E S  ********************************************/
void ParseFrag2Args(u_char *);
void Frag2Defrag(Packet *);
FragTracker *GetFragTracker(Packet *);
FragTracker *NewFragTracker(Packet *);
int InsertFrag(Packet *, FragTracker *);
int FragIsComplete(FragTracker *);
void RebuildFrag(FragTracker *, Packet *);
void Frag2DeleteFrag(FragTracker *);
int PruneFragCache(FragTracker *, u_int32_t, u_int32_t);
void ZapFrag(FragTracker *);
void Frag2InitPkt();
void Frag2CleanExit(int, void *);
void Frag2Restart(int, void *);

void *Frag2Alloc(FragTracker *cft, int tv_sec, u_int32_t size)
{
    void *tmp;

    frag_mem_usage += size;

    /* if we use up all of our RAM, try to free up some stale frags */
    if( (u_int32_t)frag_mem_usage > f2data.memcap)
    {

        frag2_alloc_faults++;
        pc.frag_mem_faults++;

        if(!PruneFragCache(cft, (u_int32_t)tv_sec, 0))
        {
            /* if we can't prune due to time, just nuke 5 random sessions */
            PruneFragCache(cft, 0, 5);
        }
    }

    tmp = (void *) calloc(size, sizeof(char));

    if(tmp == NULL)
    {
        FatalError("spp_defrag: Unable to allocate memory! "
                   "(%lu bytes in use)\n", frag_mem_usage);
    }

    return tmp;
}


static int Frag2CompareFunc(ubi_trItemPtr ItemPtr, ubi_trNodePtr NodePtr)
{
    FragTracker *nFt;
    FragTracker *iFt;

    nFt = (FragTracker *) NodePtr;
    iFt = (FragTracker *) ItemPtr;

    DebugMessage(DEBUG_FLOW,"NodePtr: sip: 0x%X  dip: 0x%X  ip: 0x%X  "
                 "proto: 0x%X\n", nFt->sip, nFt->dip, nFt->id, nFt->protocol);
    DebugMessage(DEBUG_FLOW,"ItemPtr: sip: 0x%X  dip: 0x%X  ip: 0x%X  "
                 "proto: 0x%X\n", iFt->sip, iFt->dip, iFt->id, iFt->protocol);

    if(nFt->sip < iFt->sip) return 1;
    if(nFt->sip > iFt->sip) return -1;
    if(nFt->dip < iFt->dip) return 1;
    if(nFt->dip > iFt->dip) return -1;
    if(nFt->id < iFt->id) return 1;
    if(nFt->id > iFt->id) return -1;
    if(nFt->protocol < iFt->protocol) return 1;
    if(nFt->protocol > iFt->protocol) return -1;

    return 0;
}


static int Frag2FragCompare(ubi_trItemPtr ItemPtr, ubi_trNodePtr NodePtr)
{
    Frag2Frag *nFrag;
    Frag2Frag *iFrag;

    nFrag = (Frag2Frag *) NodePtr;
    iFrag = (Frag2Frag *) ItemPtr;

    if(nFrag->offset < iFrag->offset) return 1;
    if(nFrag->offset > iFrag->offset) return -1;

    return 0;
}



static void CompletionTraverse(ubi_trNodePtr NodePtr, void *complete)
{
    Frag2Frag *frag;
    Event event;
    int *comp = (int *) complete;

    frag = (Frag2Frag *) NodePtr;
    
    if(frag->offset == next_offset)
    {
        next_offset = frag->offset + frag->size;
    }
    else if(frag->offset < next_offset)
    {
        if(frag->size < next_offset)
        {
            SetEvent(&event, GENERATOR_SPP_FRAG2, 
                    FRAG2_TEARDROP, 1, 0, 5, 0);
            CallAlertPlugins(NULL, "spp_frag2: Teardrop attack", 
                    NULL, &event);
            CallLogPlugins(NULL, "spp_frag2: Teardrop attack", 
                    NULL, &event);
        }
    }
    else if(frag->offset > next_offset)
    {
        DebugMessage(DEBUG_FLOW, "Holes in completion check...\n");
        *comp = 0;
    }

    return;
}

static void RebuildTraverse(ubi_trNodePtr NodePtr, void *buffer)
{
    Frag2Frag *frag;
    u_int8_t *buf = (u_int8_t *)buffer;

    if(f2data.stop_traverse)
        return;

    frag = (Frag2Frag *)NodePtr;

    DebugMessage(DEBUG_FLOW, "frag offset: 0x%X  size: %lu  pointer: %p\n", 
            (unsigned int) frag->offset, (unsigned long) frag->size,
            (buf+frag->offset));

    if((frag->offset + frag->size) < 65516)
    {
        memcpy(buf+frag->offset, frag->data, frag->size);
        pc.rebuild_element++;
    }
    else
    {
        DebugMessage(DEBUG_FLOW, "frag2: pkt rebuild size violation\n");
        f2data.stop_traverse = 1;
    }

    return;
}

static void KillFrag(ubi_trNodePtr NodePtr)
{
    Frag2Frag *frag;

    frag = (Frag2Frag *) NodePtr;

    frag_mem_usage -= frag->size;
    free(frag->data);

    frag_mem_usage -= sizeof(Frag2Frag);
    free(frag);
    
}


void SetupFrag2()
{
    RegisterPreprocessor("frag2", Frag2Init);
    DebugMessage(DEBUG_FLOW, "Preprocessor: frag2 is setup...\n");
}


void Frag2Init(u_char *args)
{

    DebugMessage(DEBUG_FLOW, "Initializing frag2\n");

    ParseFrag2Args(args);

    (void)ubi_trInitTree(FragRootPtr,        /* ptr to the tree head */
                         Frag2CompareFunc,   /* comparison function */
                         0);                 /* no dups */

    defrag_pkt = (Packet *)calloc(sizeof(Packet), sizeof(char));
    Frag2InitPkt();

    f2data.last_prune_time = 0;


    AddFuncToPreprocList(Frag2Defrag);
    AddFuncToCleanExitList(Frag2CleanExit, NULL);
    AddFuncToRestartList(Frag2Restart, NULL);
}


void ParseFrag2Args(u_char *args)
{
    char **toks;
    int num_toks;
    int i;
    char *index;
    char **stoks = NULL;
    int s_toks;

    if(args == NULL)
    {
        f2data.memcap = FRAG_MEMCAP;  /* 4MB */
        f2data.frag_timeout = FRAG_PRUNE_QUANTA; /* 60 seconds */

        if(!pv.quiet_flag)
        {
            LogMessage("No arguments to frag2 directive, "
                    "setting defaults to:\n");
            LogMessage("    Fragment timeout: %d seconds\n", 
                    f2data.frag_timeout);
            LogMessage("    Fragment memory cap: %lu bytes\n", f2data.memcap);
        }

        return;
    }
    else
    {
        f2data.memcap = FRAG_MEMCAP;  
        f2data.frag_timeout = FRAG_PRUNE_QUANTA; 

        toks = mSplit(args, ",", 12, &num_toks, 0);

        i=0;

        while(i < num_toks)
        {
            index = toks[i];

            while(isspace((int)*index)) index++;

            stoks = mSplit(index, " ", 4, &s_toks, 0);

            if(!strncasecmp(stoks[0], "timeout", 7))
            {
                if(isdigit((int)stoks[1][0]))
                {
                    f2data.frag_timeout = atoi(stoks[1]);
                }
                else
                {
                    LogMessage("WARNING %s(%d) => Bad timeout in config file, "
                            "defaulting to %d seconds\n", FRAG_PRUNE_QUANTA);

                    f2data.frag_timeout = FRAG_PRUNE_QUANTA;
                }
            }
            else if(!strncasecmp(stoks[0], "memcap", 6))
            {
                if(isdigit((int)stoks[1][0]))
                {
                    f2data.memcap = atoi(stoks[1]);

                    if(f2data.memcap < 16384)
                    {
                        LogMessage("WARNING %s(%d) => Ludicrous (<16k) memcap "
                                "size, setting to default (%d bytes)\n", 
                                file_name, file_line, FRAG_MEMCAP);

                        f2data.memcap = FRAG_MEMCAP;
                    }
                }
                else
                {
                    LogMessage("WARNING %s(%d) => Bad memcap in config file, "
                            "defaulting to %lu bytes\n", FRAG_MEMCAP);

                    f2data.memcap = FRAG_MEMCAP;
                }
            }

            do
            {
                s_toks--;
                free(stoks[s_toks]);
            } while(s_toks);

            i++;
        }

        do
        {
            num_toks--;
            free(toks[num_toks]);
        }while(num_toks);

        LogMessage("[*] Frag2 config:\n");
        LogMessage("    timeout: %d\n", f2data.frag_timeout);
        LogMessage("    memcap: %lu\n", f2data.memcap);

    }
}

void Frag2Defrag(Packet *p)
{
    FragTracker *ft;
    
    /* make sure we're interested in this packet */
    if(p == NULL || p->iph == NULL || !p->frag_flag) 
    {
        return;
    }

    if(p->packet_flags & PKT_REBUILT_FRAG)
    {
        return;
    }

    if (!f2data.last_prune_time)
         f2data.last_prune_time = p->pkth->ts.tv_sec;

    DebugMessage(DEBUG_FLOW, "Got frag packet (mem use: %lu frag "
            "trackers: %d  p->pkt_flags: 0x%X)\n", frag_mem_usage, 
            ubi_trCount(FragRootPtr), p->packet_flags);

    ft = GetFragTracker(p);
    
    if(ft == NULL)
    {
        DebugMessage(DEBUG_FLOW, "Didn't find frag packet\n");
        DebugMessage(DEBUG_FLOW, "Adding New FragTracker...\n");
        ft = NewFragTracker(p);

        if(ft == NULL)
        {
            DebugMessage(DEBUG_FLOW, "Discarding fragment, unable to "
                    "generate new FragTracker\n");
        }

        return;
    }

    DebugMessage(DEBUG_FLOW, "Found frag packet\n");

    if(InsertFrag(p, ft) == -1)
    {
        LogMessage("WARNING: Bad insert in fraglist for FragTracker %p\n", ft);
    }
    else
    {
        if(FragIsComplete(ft))
        {
            RebuildFrag(ft, p);
        }
    }

    if( (u_int32_t)(p->pkth->ts.tv_sec) > 
            (f2data.last_prune_time + f2data.frag_timeout))
    {
        DebugMessage(DEBUG_FLOW, "Prune time quanta for defrag code hit, "
                "pruning frag tree...\n");
        PruneFragCache(ft, p->pkth->ts.tv_sec, 0);
        DebugMessage(DEBUG_FLOW,  "Finished pruning, %lu frag trackers "
                "active, %lu alloc faults\n", ubi_trCount(FragRootPtr),
                (unsigned long) frag2_alloc_faults);

        f2data.last_prune_time = p->pkth->ts.tv_sec;
    }

    return;
}



FragTracker *GetFragTracker(Packet *p)
{
    FragTracker ft;
    FragTracker *returned;

    if(ubi_trCount(FragRootPtr) == 0)
        return NULL;

    if(frag_mem_usage == 0)
    {
        DebugMessage(DEBUG_FLOW, "trCount says nodes exist but "
                                 "frag_mem_usage = 0, something's fucked up...\n");
        return NULL;
    }
                
    ft.sip = p->iph->ip_src.s_addr;
    ft.dip = p->iph->ip_dst.s_addr;
    ft.id = p->iph->ip_id;
    ft.protocol = p->iph->ip_proto;

    returned = (FragTracker *) ubi_sptFind(FragRootPtr, (ubi_btItemPtr)&ft);

    return returned;
}


FragTracker *NewFragTracker(Packet *p)
{
    FragTracker *tmp;

    /* allocate space for a frag tracker */
    tmp = Frag2Alloc(NULL, p->pkth->ts.tv_sec, sizeof(FragTracker));

    /* set the frag tracker data fields */
    tmp->sip = p->iph->ip_src.s_addr;
    tmp->dip = p->iph->ip_dst.s_addr;
    tmp->id = p->iph->ip_id;
    tmp->protocol = p->iph->ip_proto;

    tmp->fraglistPtr = &tmp->fraglist;
    
    /* initialize the fragment list */
    (void)ubi_trInitTree(tmp->fraglistPtr,    /* ptr to the tree head */
                         Frag2FragCompare,    /* comparison function */
                         0);                  /* dups ok */

    /* insert the fragment into the frag list */
    if(InsertFrag(p, tmp) != -1)
    {
        /* insert the frag tracker into the fragment tree */
        if(ubi_sptInsert(FragRootPtr, (ubi_btNodePtr)tmp, 
                    (ubi_btNodePtr)tmp, NULL) == FALSE)
        {
            LogMessage("NewFragTracker: sptInsert failed, that sucks\n");
            Frag2DeleteFrag(tmp);
            return NULL;
        }
    }
    else
    {
        frag_mem_usage -= sizeof(FragTracker);
        free(tmp);
        return NULL;
    }

    pc.frag_trackers++;

    return tmp;
}


int InsertFrag(Packet *p, FragTracker *ft)
{
    Frag2Frag *returned;
    Frag2Frag *killme = NULL;
    Frag2Frag *newfrag;
    Event event;

    /* set the frag flag if this is the first fragment */
    if(p->mf && p->frag_offset == 0)
    {
        ft->frag_flags |= FRAG_GOT_FIRST;
    }
    else if(!p->mf && p->frag_offset > 0)
    {
        ft->frag_flags |= FRAG_GOT_LAST;
        ft->calculated_size = (p->frag_offset << 3) + p->dsize;

        if(ft->calculated_size > 65516)
        {
            SetEvent(&event, GENERATOR_SPP_FRAG2, 
                    FRAG2_OVERSIZE_FRAG, 1, 0, 5, 0);
            CallAlertPlugins(p, "spp_frag2: Oversized fragment, probable DoS", 
                    NULL, &event);
            CallLogPlugins(p, "spp_frag2: Oversized fragment, probably DoS",
                    NULL, &event);
        }
    }

    ft->last_frag_time = p->pkth->ts.tv_sec;
    ft->frag_pkts++;
    ft->frag_bytes += p->dsize;

    newfrag = (Frag2Frag *) Frag2Alloc(ft, p->pkth->ts.tv_sec, 
            sizeof(Frag2Frag));

    newfrag->data = (u_int8_t *) Frag2Alloc(ft, p->pkth->ts.tv_sec, p->dsize);

    memcpy(newfrag->data, p->data, p->dsize);

    newfrag->offset = p->frag_offset << 3;
    newfrag->size = p->dsize;

    returned = (Frag2Frag *) ubi_sptFind(ft->fraglistPtr, 
            (ubi_btItemPtr)newfrag);

    if(returned != NULL)
    {
        killme = (Frag2Frag *) ubi_sptRemove(ft->fraglistPtr, 
                (ubi_btNodePtr) returned);
        
        if(killme != NULL)
        {
            frag_mem_usage -= killme->size;
            free(killme->data);

            frag_mem_usage -= sizeof(Frag2Frag);
            free(killme);
        }
    }

    if(ubi_sptInsert(ft->fraglistPtr, (ubi_btNodePtr)newfrag, 
                (ubi_btNodePtr)newfrag, NULL) == ubi_trFALSE)
    {
        LogMessage("InsertFrag: sptInsert failed, that sucks\n");

        /* 
         * if there was a problem inserting the fragment into the fraglist
         * nuke it
         */
        frag_mem_usage -= killme->size;
        free(newfrag->data);
        frag_mem_usage -= sizeof(Frag2Frag);
        free(newfrag);
        return -1;
    }

    return 0;
}



int FragIsComplete(FragTracker *ft)
{
    int complete = 1;
    
    if((ft->frag_flags & (FRAG_GOT_FIRST|FRAG_GOT_LAST)) == (FRAG_GOT_FIRST|FRAG_GOT_LAST))
    {
        next_offset = 0;

        /* traverse the frag tree and see if they're all there.... */
        (void)ubi_trTraverse(ft->fraglistPtr, CompletionTraverse, &complete);

        return complete;
    }
    else
    {
        return 0;
    }
}


void RebuildFrag(FragTracker *ft, Packet *p)
{
    u_int32_t dlt_size;
    u_int16_t iph_size;
    u_int8_t *rebuild_ptr;

    DebugMessage(DEBUG_FLOW, "Rebuilding pkt [0x%X:%d  0x%X:%d]\n", 
            p->iph->ip_src.s_addr, p->sp, p->iph->ip_dst.s_addr, p->dp);
    DebugMessage(DEBUG_FLOW, "Calculated size: %d\n", ft->calculated_size);

    if(ft->calculated_size > 65516)
    {
        LogMessage("WARNING: Dumping oversized fragment\n");
        ZapFrag(ft);
    }

    /* copy the packet header from the last packet of the frag */
    memcpy(defrag_pkt->pkth, p->pkth, sizeof(SnortPktHeader));

    /* copy the packet data from the last packet of the frag */
    memcpy(defrag_pkt->pkt+SPARC_TWIDDLE, p->pkt, p->pkth->caplen);

    /* calculate how big the DLT is */
    dlt_size = ((u_int32_t)p->iph) - ((u_int32_t)p->pkt) + SPARC_TWIDDLE;

    /* calculate the IP header size from the packet data */
    iph_size = (p->iph->ip_hlen << 2);

    /* set pointer for the IP header in the rebuilt packet */
    defrag_pkt->iph = (IPHdr *) (defrag_pkt->pkt + dlt_size);

    /* clear the packet fragment fields */ 
    defrag_pkt->iph->ip_off = 0x0000;
    defrag_pkt->frag_flag = 0;

    /* check for a bad IP header size */
    if(iph_size > 60)
    {
        iph_size = 20;
    }
    
    /* set the new packet's capture length */
    defrag_pkt->pkth->caplen = dlt_size + iph_size + ft->calculated_size;
    defrag_pkt->pkth->len = defrag_pkt->pkth->caplen;

    /* set the pointer to the beginning of the transport layer of the rebuilt
     * packet
     */
    rebuild_ptr = defrag_pkt->pkt + dlt_size + iph_size;
    f2data.frag_pad = dlt_size+iph_size;

    /* walk the fragment list and rebuild the packet */
    f2data.stop_traverse = 0;
    (void)ubi_trTraverse(ft->fraglistPtr, RebuildTraverse, rebuild_ptr);
    f2data.stop_traverse = 0;
    
    /* reset the ip header pointer */
    defrag_pkt->iph = (IPHdr *) (defrag_pkt->pkt + dlt_size);
    defrag_pkt->iph->ip_off = 0x0000;

    /* set the ip dgm length */
    defrag_pkt->iph->ip_len = 
        htons((u_short)(defrag_pkt->pkth->caplen-dlt_size));

    /* tell the rest of the system that this is a rebuilt fragment */
    defrag_pkt->packet_flags |= PKT_REBUILT_FRAG;
    defrag_pkt->frag_flag = 0;
    
    defrag_pkt->iph->ip_csum = 0;
        
    /* calculate the ip checksum for the packet */
    defrag_pkt->iph->ip_csum = checksum((u_int16_t *)defrag_pkt->iph, 
                                     iph_size, (u_int16_t *) NULL, 0);

    pc.rebuilt_frags++;

#ifdef DEBUG
    ClearDumpBuf();
    PrintNetData(stdout, defrag_pkt->pkt, defrag_pkt->pkth->caplen);
    ClearDumpBuf();
#endif

    /* process the packet through the detection engine */
    DebugMessage(DEBUG_FLOW, "Processing rebuilt packet:\n");
    ProcessPacket(NULL, defrag_pkt->pkth, defrag_pkt->pkt+SPARC_TWIDDLE);
    DebugMessage(DEBUG_FLOW, "Done with rebuilt packet, deleting...\n");

    /* dump the packet frags */
    ZapFrag(ft);
}


void Frag2DeleteFrag(FragTracker *ft)
{
    if(ft == NULL)
    {
        return;
    }
        
    (void)ubi_trKillTree(ft->fraglistPtr, KillFrag);

    frag_mem_usage -= sizeof(FragTracker);
    free(ft);
}



int PruneFragCache(FragTracker *cft, u_int32_t time, u_int32_t mustdie)
{
    FragTracker *ft;
    u_int32_t pruned = 0;
    
    if(ubi_trCount(FragRootPtr) == 0)
        return 0;

    ft = (FragTracker *) ubi_btFirst((ubi_btNodePtr)FragRootPtr);

    if(ft == NULL)
        return 0;

    if(time)
    {
        do
        {
            if(ft->last_frag_time + f2data.frag_timeout < time)
            {
                FragTracker *savft = ft;

                ft = (FragTracker *) ubi_btNext((ubi_btNodePtr)ft);

                if ((int)savft != (int)cft)
                {
                    DebugMessage(DEBUG_FLOW, "Pruning stale fragment\n");
                    ZapFrag(savft);
                    pc.frag_timeout++;
                    pruned++;
                }
            }
            else
            {
                ft = (FragTracker *) ubi_btNext((ubi_btNodePtr)ft);
            }
        } while(ft != NULL);

        return pruned;
    }
    else
    {
        while(mustdie--)
        {
            /* 
             * this code pulls a random leaf node from the frag tree for
             * deletion, allowing us to unpredicatably zap nodes which are
             * pretty stale anyway
             */
            ft = (FragTracker *) ubi_btLeafNode((ubi_btNodePtr) FragRootPtr);
            if((int)ft != (int)cft)
            {
                ZapFrag(ft);
                pc.frag_incomp++;
            }
        }

        return mustdie;
    }

    return 0;
}



void ZapFrag(FragTracker *ft)
{
    FragTracker *killme;
    
    if(ft != NULL && FragRootPtr->count != 0)
    {
        killme = (FragTracker *) ubi_sptRemove(FragRootPtr, 
                (ubi_btNodePtr) ft);

        Frag2DeleteFrag(killme);
    }
}



void Frag2Restart(int signal, void *foo)
{
    return;
}



void Frag2CleanExit(int signal, void *foo)
{
    return;
}



void Frag2InitPkt()
{
    defrag_pkt->pkth =  calloc(sizeof(SnortPktHeader), 
            sizeof(char));

    defrag_pkt->pkt = (u_int8_t *) calloc(
            ETHERNET_HEADER_LEN+65536+SPARC_TWIDDLE, sizeof(char));
}
