/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spo_unified.c,v 1.4 2001/06/29 02:38:54 roesch Exp $ */

/* spo_unified 
 * 
 * Purpose:
 *
 * This plugin generates the new unified alert and logging formats
 *
 * Arguments:
 *   
 * filename of the alert and log spools
 *
 * Effect:
 *
 * Packet logs are written (quickly) to a unified output file
 *
 * Comments:
 *
 * The future...
 */

/* your output plugin header file goes here */
#include "spo_unified.h"

/* external globals from rules.c */
extern char *file_name;    /* current rules file filename */
extern int file_line;      /* current line being processed in rules file */
extern OptTreeNode *otn_tmp;
extern int thiszone;

/* If you need to instantiate the plugin's data structure, do it here */
UnifiedData UnifiedInfo;

/*
 * Function: SetupUnified()
 *
 * Purpose: Registers the output plugin keyword and initialization 
 *          function into the output plugin list.  This is the function that
 *          gets called from InitOutputPlugins() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupUnified()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterOutputPlugin("log_unified", NT_OUTPUT_LOG, UnifiedLogInit);
    RegisterOutputPlugin("alert_unified", NT_OUTPUT_ALERT, UnifiedAlertInit);

    DebugMessage(DEBUG_INIT, "Output plugin: Unified logging/alerting "
                             "is setup...\n");
}


/*
 * Function: UnifiedInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void UnifiedLogInit(u_char *args)
{
    DebugMessage(DEBUG_INIT, "Output: Unified Log Initialized\n");

    /* tell command line loggers to go away */
    pv.log_plugin_active = 1;

    /* parse the argument list from the rules file */
    ParseUnifiedLogArgs(args);

    UnifiedInitLogFile(&UnifiedInfo);

    pv.log_bitmap |= LOG_UNIFIED;

    /* Set the preprocessor function into the function list */
    AddFuncToOutputList(LogUnified, NT_OUTPUT_LOG, &UnifiedInfo);
    AddFuncToCleanExitList(SpoUnifiedLogCleanExitFunc, &UnifiedInfo);
    AddFuncToRestartList(SpoUnifiedLogRestartFunc, &UnifiedInfo);
}



void UnifiedAlertInit(u_char *args)
{
    DebugMessage(DEBUG_INIT, "Output: Unified Alert Initialized\n");

    /* parse the argument list from the rules file */
    ParseUnifiedAlertArgs(args);

    UnifiedInitAlertFile(&UnifiedInfo);

    /* Set the preprocessor function into the function list */
    AddFuncToOutputList(AlertUnified, NT_OUTPUT_ALERT, &UnifiedInfo);
    AddFuncToCleanExitList(SpoUnifiedAlertCleanExitFunc, &UnifiedInfo);
    AddFuncToRestartList(SpoUnifiedAlertRestartFunc, &UnifiedInfo);
}




/*
 * Function: ParseUnifiedLogArgs(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.  This function doesn't
 *          have to exist if it makes sense to parse the args in the init 
 *          function.
 *
 * Arguments: args => argument list
 *
 * Returns: void function
 *
 */
void ParseUnifiedLogArgs(char *args)
{
    DebugMessage(DEBUG_PLUGIN, "Args: %s\n", args);

    if(args != NULL)
    {
        while(isspace((int)*args)) args++;

        if(strncmp(args, "", 1) != 0)
            UnifiedInfo.log_filename = strdup(args);
        else
            UnifiedInfo.log_filename = strdup("snort-unified.log");
    }
    else
    {
        UnifiedInfo.log_filename = strdup("snort-unified.log");
    }
}

#define UNITIMESIZE 11
/*
 * Function: UnifiedInitLogFile()
 *
 * Purpose: Initialize the unified log file header
 *
 * Arguments: data => pointer to the plugin's reference data struct 
 *
 * Returns: void function
 */
void UnifiedInitLogFile(UnifiedData *data)
{
    time_t curr_time;      /* place to stick the clock data */
    struct tm *loc_time;   /* place to stick the adjusted clock data */
    char timebuf[UNITIMESIZE];
    char logdir[STD_BUF];
    int value;
    UnifiedLogFileHeader hdr;

    bzero(logdir, STD_BUF);
    bzero(timebuf, UNITIMESIZE);
    curr_time = time(NULL);
    loc_time = localtime(&curr_time);
    strftime(timebuf,UNITIMESIZE,"%m%d@%H%M",loc_time);

    if(data->log_filename[0] == '/')
        value = snprintf(logdir, STD_BUF, "%s%s", 
                chrootdir == NULL ? "" : chrootdir, data->log_filename);
    else
        value = snprintf(logdir, STD_BUF, "%s%s/%s-%s",
                chrootdir == NULL ? "" : chrootdir, pv.log_dir, timebuf, 
                data->log_filename);

    if(value == -1)
    {
        FatalError("ERROR: log file logging path and file name are "
                "too long, aborting!\n");
    }

    printf("Opening %s\n", logdir);

    if((data->log=fopen(logdir, "w")) == NULL)
    {
        FatalError("UnifiedInitLogFile(%s): %s\n", logdir, strerror(errno));
    }

    /* write the log file header */
    hdr.magic = SNORT_MAGIC;
    hdr.version_major = SNORT_VERSION_MAJOR;
    hdr.version_minor = SNORT_VERSION_MINOR;
    hdr.timezone = thiszone;
    hdr.snaplen = snaplen;
    hdr.sigfigs = 0;
    hdr.linktype = datalinks[0];

    if(fwrite((char *)&hdr, sizeof(hdr), 1, data->log) != 1)
    {
        FatalError("UnifiedLogInit(): %s", strerror(errno));
    }
        

    /* keep a copy of the filename for later reference */
    if(data->log_filename != NULL)
    {
        free(data->log_filename);

        data->log_filename = strdup(logdir);
    }

    return;
}





void ParseUnifiedAlertArgs(char *args)
{
    DebugMessage(DEBUG_PLUGIN, "Args: %s\n", args);

    if(args != NULL)
    {
        while(isspace((int)*args)) args++;

        if(strncmp(args, "", 1) != 0)
            UnifiedInfo.alert_filename = strdup(args);
        else
            UnifiedInfo.alert_filename = strdup("snort-unified.alert");
    }
    else
    {
        UnifiedInfo.alert_filename = strdup("snort-unified.alert");
    }

    printf("UnifiedAlertFilename = %s\n", UnifiedInfo.alert_filename);
}

/*
 * Function: UnifiedInitAlertFile()
 *
 * Purpose: Initialize the unified log alert file
 *
 * Arguments: data => pointer to the plugin's reference data struct 
 *
 * Returns: void function
 */
void UnifiedInitAlertFile(UnifiedData *data)
{
    time_t curr_time;      /* place to stick the clock data */
    struct tm *loc_time;   /* place to stick the adjusted clock data */
    char timebuf[UNITIMESIZE];
    char logdir[STD_BUF];
    int value;
    UnifiedAlertFileHeader hdr;

    bzero(logdir, STD_BUF);
    bzero(timebuf, UNITIMESIZE);
    curr_time = time(NULL);
    loc_time = localtime(&curr_time);
    strftime(timebuf,UNITIMESIZE,"%m%d@%H%M",loc_time);

    if(data->alert_filename[0] == '/')
        value = snprintf(logdir, STD_BUF, "%s%s", 
                chrootdir == NULL ? "" : chrootdir, data->alert_filename);
    else
        value = snprintf(logdir, STD_BUF, "%s%s/%s-%s",
                chrootdir == NULL ? "" : chrootdir, pv.log_dir, timebuf, 
                data->alert_filename);


    if(value == -1)
    {
        FatalError("ERROR: log file logging path and file name are "
                "too long, aborting!\n");
    }

#ifdef DEBUG
    printf("Opening %s\n", logdir);
#endif

    if((data->alert=fopen(logdir, "w+")) == NULL)
    {
        FatalError("UnifiedInitAlertFile(%s): %s\n", logdir, strerror(errno));
    }

    hdr.magic = SNORT_MAGIC;
    hdr.version_major = 1;
    hdr.version_minor = 8;
    hdr.timezone = thiszone;

    if(fwrite((char *)&hdr, sizeof(hdr), 1, data->alert) != 1)
    {
        FatalError("UnifiedAlertInit(): %s\n", strerror(errno));
    }
        
    /* keep a copy of the filename for later reference */
    if(data->alert_filename != NULL)
    {
        free(data->alert_filename);

        data->alert_filename = strdup(logdir);
    }

    return;
}


/**  R U N   T I M E   F U N C T I O N S  ************************************/

/*
 * Function: LogUnified(Packet *, char *msg, void *arg)
 *
 * Purpose: Perform the preprocessor's intended function.  This can be
 *          simple (statistics collection) or complex (IP defragmentation)
 *          as you like.  Try not to destroy the performance of the whole
 *          system by trying to do too much....
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 */
void LogUnified(Packet *p, char *msg, void *arg, Event *event)
{
    UnifiedLog logheader;
    UnifiedData *data = (UnifiedData *)arg;

    if(event != NULL)
    {
        logheader.event.sig_generator = event->sig_generator;
        logheader.event.sig_id = event->sig_id;
        logheader.event.sig_rev = event->sig_rev;
        logheader.event.classification = event->classification;
        logheader.event.priority = event->priority;
        logheader.event.event_reference = event->event_reference;
    }



    if(p)
    {
        logheader.flags = p->packet_flags;

        /* 
         * this will have to be fixed when we transition to the pa_engine
         * code (p->pkth is libpcap specific)
         */ 
        memcpy(&logheader.pkth, p->pkth, sizeof(SnortPktHeader));
    }
    else
    {
        logheader.flags = 0;
        memcpy(&logheader.pkth, 0, sizeof(SnortPktHeader));
    }
    
    fwrite((char*)&logheader, sizeof(UnifiedLog), 1, data->log);

    if(p)
    {
        fwrite((char*)p->pkt, p->pkth->caplen, 1, data->log);
    }
}


void AlertUnified(Packet *p, char *msg, void *arg, Event *event)
{
    UnifiedData *data = (UnifiedData *)arg;
    UnifiedAlert alertdata;

    bzero(&alertdata, sizeof(alertdata));

    if(event != NULL)
    {
        alertdata.event.sig_generator = event->sig_generator;
        alertdata.event.sig_id = event->sig_id;
        alertdata.event.sig_rev = event->sig_rev;
        alertdata.event.classification = event->classification;
        alertdata.event.priority = event->priority;
        alertdata.event.event_reference = event->event_reference;
    }

    if(p)
    {
        alertdata.ts.tv_sec = p->pkth->ts.tv_sec;
        alertdata.ts.tv_usec = p->pkth->ts.tv_usec;

        if(p->iph != NULL)
        {
            alertdata.sip = p->iph->ip_src.s_addr;
            alertdata.dip = p->iph->ip_dst.s_addr;
            alertdata.sp = p->sp;
            alertdata.dp = p->dp;
            alertdata.protocol = p->iph->ip_proto;
            alertdata.flags = p->packet_flags;
        }
    }

    fwrite((char *)&alertdata, sizeof(UnifiedAlert), 1, data->alert);
}


/*
 * Function: SpoUnifiedCleanExitFunc()
 *
 * Purpose: Cleanup at exit time
 *
 * Arguments: signal => signal that caused this event
 *            arg => data ptr to reference this plugin's data
 *
 * Returns: void function
 */
void SpoUnifiedLogCleanExitFunc(int signal, void *arg)
{
    /* cast the arg pointer to the proper type */
    UnifiedData *data = (UnifiedData *)arg;

    DebugMessage(DEBUG_FLOW, "SpoUnifiedCleanExitFunc\n");

    fclose(data->log);

    /* 
     * if we haven't written any data, dump the output file so there aren't
     * fragments all over the disk 
     */
    if(pc.alert_pkts==0 && pc.log_pkts==0)
    {
        unlink(data->log_filename);
    }

    /* free up initialized memory */
    free(data->log_filename);
    free(data);
}



/*
 * Function: SpoUnifiedRestartFunc()
 *
 * Purpose: For restarts (SIGHUP usually) clean up structs that need it
 *
 * Arguments: signal => signal that caused this event
 *            arg => data ptr to reference this plugin's data
 *
 * Returns: void function
 */
void SpoUnifiedLogRestartFunc(int signal, void *arg)
{
    UnifiedData *data = (UnifiedData *)arg;

    DebugMessage(DEBUG_FLOW, "SpoUnifiedRestartFunc\n");

    /* 
     * if we haven't written any data, dump the output file so there aren't
     * fragments all over the disk 
     */
    if(pc.alert_pkts==0 && pc.log_pkts==0)
    {
        unlink(data->log_filename);
    }

    fclose(data->log);
    free(data->log_filename);
    free(data);
}




/*
 * Function: SpoUnifiedAlertCleanExitFunc()
 *
 * Purpose: Cleanup at exit time
 *
 * Arguments: signal => signal that caused this event
 *            arg => data ptr to reference this plugin's data
 *
 * Returns: void function
 */
void SpoUnifiedAlertCleanExitFunc(int signal, void *arg)
{
    /* cast the arg pointer to the proper type */
    UnifiedData *data = (UnifiedData *)arg;

    DebugMessage(DEBUG_FLOW, "SpoUnifiedAlertCleanExitFunc\n");

    fclose(data->alert);

    /* 
     * if we haven't written any data, dump the output file so there aren't
     * fragments all over the disk 
     */
    if(pc.alert_pkts==0 && pc.log_pkts==0)
    {
        unlink(data->alert_filename);
    }

    /* free up initialized memory */
    free(data->alert_filename);
    free(data);
}



/*
 * Function: SpoUnifiedAlertRestartFunc()
 *
 * Purpose: For restarts (SIGHUP usually) clean up structs that need it
 *
 * Arguments: signal => signal that caused this event
 *            arg => data ptr to reference this plugin's data
 *
 * Returns: void function
 */
void SpoUnifiedAlertRestartFunc(int signal, void *arg)
{
    UnifiedData *data = (UnifiedData *)arg;

    DebugMessage(DEBUG_FLOW, "SpoUnifiedRestartFunc\n");

    /* 
     * if we haven't written any data, dump the output file so there aren't
     * fragments all over the disk 
     */
    if(pc.alert_pkts==0 && pc.log_pkts==0)
    {
        unlink(data->log_filename);
    }

    fclose(data->alert);
    free(data->alert_filename);
    free(data);
}

