#!/usr/bin/perl

# Snortsnarf, a utility to convert snort log files to HTML pages
# Authors: Stuart Staniford, Silicon Defense (stuart@SiliconDefense.com)
#          James Hoagland, Silicon Defense (hoagland@SiliconDefense.com)
# copyright (c) 2000 by Silicon Defense (http://www.silicondefense.com/)

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.  

# Snortsnarf description:
#
# Code to parse files of snort alerts and portscan logs, and produce
# HTML output intended to allow an intrusion detection analyst to
# engage in diagnostic inspection and tracking down problems.  
# The model is that one is using a cron job or similar to
# produce a daily/hourly/whatever file of snort alerts.	 This script
# can be run on each such file to produce a convenient HTML breakout
# of all the alerts.
#
# The idea is that the analyst can click around the alerts instead
# of wearily grepping and awking through them.

# Please send complaints, kudos, and especially improvements and bugfixes to
# hoagland@SiliconDefense.com. This is a quick hack with features added and
# may only be worth what you paid for it.  It is still under active
# development and may change at any time.

# this file (snortsnarf.pl) is part of Snortsnarf v111500.1

##############################################################################

# Usage:

# snortsnarf.pl <options> <file1 file2 ...>

# Any filenames with the string 'scan' in the name are assumed to be
# portscan files rather than alert files.  Similarly, filenames containing
# the string 'syslog' are assumed to be syslog files.

# The script will produce a directory snfout.file1 (by default) full of
# a large number of html files.	 These are placed under the current
# directory.  It also produces a file index.html in that directory.	 
# This is a good place to start with a browser.

# Options include:

# -d directory			directory is the path to the directory the HTML pages will
#							be generated in, overriding the default.

# -dns					This will cause the script to lookup the DNS name 
#								of each IP it writes a page for.  On a large alert 
#								file, this will take a very long time and will hammer 
#								your DNS server.

# -ldir URL				URL is the URL of a base directory in which the 
#								log files usually found in /var/log/snort are living,
#								With this option, snortsnarf will generate lots of 
#								links into those files based at that URL.  Eg: with
#								-ldir "http://host.name.here/logs" we get links like
#								http://host.name.here/logs/10.0.0.1/UDP-137-137
#								Note that the logs themselves aren't parsed.

# -homenet network		network is the network IP address or CIDR network notation
#								for your home network, as told to snort.  CIDR takes
#								that standard form of address/masksize (sizes 0-32
#								supported).  In the other form, 1-3 zeros at the end
#								of the address are assumed to be what varies within
#								your network.	 If this argument is not provided, the
#								home network is assumed to be 0.0.0.0 (no home).  This
#								is currently used only with the -ldir option.

# -color=<option>		option is 'yes' or 'no' or 'rotate'.  The style of coloring
#								is given by the option if any.  'rotate' is the only
#								style currently available.  This changes the background
#								color between an alert/portscan report and the next one
#								iff the next one has a different message (signature)
#								source host, or destination host.  If 'yes' or this not
#								given on the command line, the the default ('rotate' in
#								this version) is used. 

# -split=<threshold>	To speed up display of pages containing lots of alerts,
#							<threshold> is a maximum number of alerts to display on a
#							page unless explicitly requested.  Instead the alerts are
#							broken into several pages.  Default is 100.  Can be set to
#							0 to never split the list alerts being displayed.

# -onewindow			If this option is given, certain URs will not be targeted to
#							other browser windows as is the default.

# -db path				path is the full path to the annotation database (an XML file),
#							from the point of view of the server hosting the CGI scripts
#							that access it, or the empty string to not use an annotation
#							database.  The default is to not use it.

# -cgidir URL 			URL is the location of the cgi-bin directory that CGI scripts
#							than go along with the program are stored.  This may be
#							relative to the pages that are generated (so that when a
#							link appears on a page with URL as a prefix, it will be
#							valid).  "/cgi-bin" is the default.

# -sisr configfile		Generate links with SnortSnarf Incident Storage and Reporting 
#							(SISR).  The argument is the full path to the SISR
#							configuration file to use.  This file is not parsed by
#							snortsnarf.pl.

# -nmapurl URL			URL is the URL of a base URL directory in which the html
#							output of running nmap2html on nmap output is living. 
#							With this option, snortsnarf will generate links to that
#							output for IP addresses.  If the path to that directory
#							is given with the -nmapdir option, then only those pages
#							that actually exist (at the time snortsnarf is run) are
#							linked to.

# -nmapdir dir			dir is the directory on the file system in which nmap2html
#							output is stored.  When the -nmapurl option is given,
#							this is used to verify that a nmaplog.pl page actually
#							exists before linking to it. 

# -rulesfile file		file is the base rule file (e.g., snort.lib) that snort was
#							run with.  If this option is given, the rulesin that
#							file (and included files) generating a signature  are
#							included in the page for that signatures alerts.  Note
#							that the processing of the rule files are pretty rough.
#							E.g., variables are ignored. 

# -rulesdir dir			dir is a directry to use as the base path for rules files
#							rather than the one given or the paths listed in include
#							directives.  (Useful if the files have been relocated.)

# -refresh=<secs>		add a refresh tag to every HTML page generated, causing
#							browsers to refresh the page ever <secs> seconds.



# Snortsnarf is believed to work with Perl 5 on the following OS's
# OpenBSD 2.6
# RedHat Linux 6.1 and 6.2
# Windows NT 4.0 (NTFS only) (tweak the $os parameter below.  Wildcards won't).

# It probably works with most other Unixen/Linuxen, and is certain not to 
# work on the FAT filesystem.

# Believed to work with alert files generated with these snort command lines:
# snort -D -c oursnort.lib -o -d 
# snort -i eth0 -c oursnort.lib -o -e  -d

# This script is pretty memory intensive while it is running.  A previous
# version of the script was known to handle 5MB of alerts on a machine with
# 64MB of memory (no virtual memory).  It has also been known to eventually
# finish of huge alert files with Gigabytes of virtual memory available.  The
# script should be studied to figure out why it uses more memory than expected.

# HTML looks marginally ok with Netscape 4.7 and Explorer 4.72.	 
# Other browsers not tested.

# .tar.gz of the dir is about half the size of the alert file.
# YMMV!

##############################################################################

# Credits, etc

# Initial alert parsing code borrowed from Joey McAlerney, Silicon 
# Defense.
# A couple of ideas were stolen from Snort2html by Dan Swan.
# Thanks to SANS GIAC and Donald McLachlan for a paper with ideas 
# on how to look up IP addresses.

# Huge thanks to DARPA for supporting us for part of the time developing
# this code.  Major thanks to Paul Arabelo for being our test customer
# while we learned to do operational intrusion detection instead of being
# only a researcher.

# Shouts to Marty Roesch and Patrick Mullen and the rest of the snort
# community for developing snort and the snort portscan preprocessor.  

# Kudos to Steve Northcutt who has taught us all what an intrusion
# detection analyst should be.

# Version control info: $Id: snortsnarf.pl,v 1.35 2000/11/15 22:40:28 jim Exp $

$script = "<a href=\"http://www.silicondefense.com/snortsnarf/\">".
						"SnortSnarf</a>";
$version = "v111500.1";
$author_email = "hoagland\@SiliconDefense.com";
$author = "<a href=\"mailto:hoagland\@SiliconDefense.com\">Jim Hoagland</a> and <a href=\"mailto:stuart\@SiliconDefense.com\">Stuart Staniford</a>";

use lib qw(./include);
use Socket;
#use Sys::Hostname;
use Cwd;
require "snort_alert_parse.pl";
require "web_utils.pl";

if ($::sap_version ne $version) {die "snortsnarf.pl is version \"$version\" but snort_alert_parse.pl is version \"$::sap_version\"; did you remember to install the $version include directory someplace perl would find it?\n";}

##############################################################################

# portability stuff - toggle for Unix/Windows.

$os = 'unix';  # Either 'windows' or 'unix'
if($os eq 'windows')
{
 $dirsep = "\\";		
 #$root = "d:\\";		
 $root = "e:\\"; # Do not make this your system drive, don't want it to fill up
 $logfileext= '.ids';
 $logfileprototerm= '_';
 $alert_file = $root."util".$dirsep."snort".$dirsep."log".$dirsep."alert.ids"; # default input file
}
elsif($os eq 'unix')
{
 $dirsep = "\/";		# Unix
 $root = "\/";			# Unix
 $logfileext= '';
 $logfileprototerm= ':';
 $alert_file = $root."var".$dirsep."log".$dirsep."snort.alert"; # default input file
}

$html = 'html';			# usually html or htm

# Various global variables

$index_page = "index.$html";

%monthnum = ('Jan' => 1,
			 'Feb' => 2,
			 'Mar' => 3,
			 'Apr' => 4,
			 'May' => 5,
			 'Jun' => 6,
			 'Jul' => 7,
			 'Aug' => 8,
			 'Sep' => 9,
			 'Oct' => 10,
			 'Nov' => 11,
			 'Dec' => 12);

# maps the text for ICMP messages found in an alert message to its component
# in a file name for an snort log of the connection
%ICMP_text_to_filename= ( 
		'ECHO REPLY' => 'ICMP_ECHO_REPLY',
		'DESTINATION UNREACHABLE: NET UNREACHABLE' => 'ICMP_NET_UNRCH', 
		'DESTINATION UNREACHABLE: HOST UNREACHABLE' => 'ICMP_HST_UNRCH', 
		'DESTINATION UNREACHABLE: PROTOCOL UNREACHABLE' => 'ICMP_PROTO_UNRCH', 
		'DESTINATION UNREACHABLE: PORT UNREACHABLE' => 'ICMP_PORT_UNRCH', 
		'DESTINATION UNREACHABLE: FRAGMENTATION NEEDED' => 'ICMP_UNRCH_FRAG_NEEDED', 
		'DESTINATION UNREACHABLE: SOURCE ROUTE FAILED' => 'ICMP_UNRCH_SOURCE_ROUTE_FAILED', 
		'DESTINATION UNREACHABLE: NET UNKNOWN' => 'ICMP_UNRCH_NETWORK_UNKNOWN', 
		'DESTINATION UNREACHABLE: HOST UNKNOWN' => 'ICMP_UNRCH_HOST_UNKNOWN', 
		'DESTINATION UNREACHABLE: HOST ISOLATED' => 'ICMP_UNRCH_HOST_ISOLATED', 
		'DESTINATION UNREACHABLE: NET ANO' => 'ICMP_UNRCH_NET_ANO',
		'DESTINATION UNREACHABLE: HOST ANO' => 'ICMP_UNRCH_HOST_ANO',
		'DESTINATION UNREACHABLE: NET UNREACHABLE TOS' => 'ICMP_UNRCH_NET_UNR_TOS', 
		'DESTINATION UNREACHABLE: HOST UNREACHABLE TOS' => 'ICMP_UNRCH_HOST_UNR_TOS', 
		'DESTINATION UNREACHABLE: PACKET FILTERED' => 'ICMP_UNRCH_PACKET_FILT', 
		'DESTINATION UNREACHABLE: PREC VIOLATION' => 'ICMP_UNRCH_PREC_VIOL', 
		'DESTINATION UNREACHABLE: PREC CUTOFF' => 'ICMP_UNRCH_PREC_CUTOFF', 
		'DESTINATION UNREACHABLE: UNKNOWN' => 'ICMP_UNKNOWN', 
		'SOURCE QUENCH' => 'ICMP_SRC_QUENCH', 
		'REDIRECT' => 'ICMP_REDIRECT', 
		'ECHO' => 'ICMP_ECHO', 
		'TTL EXCEEDED' => 'ICMP_TTL_EXCEED', 
		'PARAMATER PROBLEM' => 'ICMP_PARAM_PROB', 
		'TIMESTAMP REQUEST'=> 'ICMP_TIMESTAMP', 
		'TIMESTAMP REPLY' => 'ICMP_TIMESTAMP_RPL', 
		'INFO REQUEST' => 'ICMP_INFO_REQ', 
		'INFO REPLY' => 'ICMP_INFO_RPL', 
		'ADDRESS REQUEST'=> 'ICMP_ADDR', 
		'ADDRESS REPLY' => 'ICMP_ADDR_RPL', 
		'UNKNOWN' => 'ICMP_UNKNOWN' 
);

# the colors to cycle through for displayed alerts
@color= ('#E7DEBD','#E0CDD0','#D5E2CE');

$normbgcol= '#E7DEBD'; # normal background color
$anombgcol= '#B0E0F0'; # background color for anomaly pages

$logo_filename= 'SDlogo.gif';
##############################################################################

# Main program

&process_options();
&initialize();

# gather alerts from input files
foreach $current_file (@ARGV)
{
	my $fh= &initialize_per_file($current_file);
	next unless defined($fh);
	
	while($alert = &get_alert($fh))
	 {
	 	$alert->{'anomscore'}= &is_anom_rept($alert);
	 	if (defined($alert->{'anomscore'})) {
	   		&process_anom_alert($alert);
	 	} else {
	   		&process_alert($alert);
	   	}
	 }
	&close_out_file($fh);
}

# generate pages
$bgcol= $normbgcol;
$logo_url= $logo_filename;
&construct_sig_indexes();
&output_main_sig_page();
&output_per_sig();
&output_per_source();
&output_per_dest();

$bgcol= $anombgcol;
$logo_url= "../$logo_filename";
&output_anom_sect();


##############################################################################

# process the command line options and leave @ARGV with just the input files
# at the end
sub process_options
{
  my $arg;

  # default values
  $dns_option= undef;
  $log_base= '';
  $homenetaddr= undef;
  $color_opt= 'rotate';
  $cgiavail= 0;
  $cgi_dir= '/cgi-bin';
  $db_file= '';
  $split_threshold= 100;
  $nmap_dir= undef;
  $nmap_url= undef;
  $sisr_config= undef;
  $rules_file= undef;
  $rules_dir= undef;
  $notarget_option= 0;
  $refreshsecs= undef;
  # go through arguments
  while($ARGV[0] =~ /^\-/)
   {
	$arg = shift @ARGV;
	if($arg eq '-dns')
	 {
		  $dns_option = 1;
	 }
	elsif($arg eq '-ldir')
	 {
		  $log_base = shift @ARGV;
		  $log_base.='/' unless $log_base =~ /\/$/;
	 }
	elsif($arg eq '-homenet')
	 {
	 	  my($addr,$bits)= split('/',(shift @ARGV));
	 	  $homenetaddr= &bytenums2bits(split('\.',$addr));
	 	  unless (defined($bits)) {
	 	  	if ($addr =~ /^0\.0\.0\.0$/) {
	 	  		$bits= 0;
	 	  	} elsif ($addr =~ /\.0\.0\.0$/) {
	 	  		$bits= 8;
	 	  	} elsif ($addr =~ /\.0\.0$/) {
	 	  		$bits= 16;
	 	  	} elsif ($addr =~ /\.0$/) {
	 	  		$bits= 24;
	 	  	} else {
	 	  		$bits= 32;
	 	  	}
	 	  }
	 	  my (@bytembits)= ();
	 	  foreach (1..4) {
	 	  	if ($bits <= 0) {
	 	  		push(@bytembits,0);
	 	  	} else {
	 	  		$bits-= 8;
	 	  		if ($bits >= 0) {
	 	  			push(@bytembits,255);
	 	  		} else {
	 	  			push(@bytembits,(0x80,0xC0,0xE0,0xF0,0xF8,0xFC,0xFE)[$bits+7]);
	 	  		}
	 	  	}
	 	  }
	 	  $homenetmask= &bytenums2bits(@bytembits);
		  $homenetaddr &= $homenetmask;
	 }
    elsif($arg =~ s/^-color//)
     {    
		if ($arg =~ /=(.*)/) {
			$color_opt = ($1 eq 'yes')?'rotate':$1;
		} else {
			$color_opt = 'rotate';
		}
	 }
    elsif($arg =~ s/^-split=//)
     {    
		$split_threshold = $arg;
	 }
	elsif($arg eq '-d')
	 {
		  $output_dir = shift @ARGV;
	 }
	elsif($arg eq '-cgidir')
	 {
		  $cgi_dir = shift @ARGV;
		  $cgi_dir =~ s/\/$//;
		  $cgiavail= 1;
	 }
	elsif($arg eq '-db')
	 {
		  $db_file = shift @ARGV;
	 }
	elsif($arg eq '-nmapdir')
	 {
		  $nmap_dir = shift @ARGV;
	 }
	elsif($arg eq '-nmapurl')
	 {
		  $nmap_url = shift @ARGV;
		  $nmap_url.='/' unless $nmap_url =~ /\/$/;
	 }
	elsif($arg eq '-sisr')
	 {
		  $sisr_config = shift @ARGV;
	 }
	elsif($arg eq '-rulesfile')
	 {
		  $rules_file = shift @ARGV;
	 }
	elsif($arg eq '-rulesdir')
	 {
		  $rules_dir = shift @ARGV;
	 }
	elsif($arg eq '-onewindow')
	 {
		  $notarget_option = 1;
	 }
    elsif($arg =~ s/^-refresh=//)
     {    
		$refreshsecs = $arg;
	 }
	else
	 {
	  print "Unknown option $arg\n";
	 }
   }
}

##############################################################################

sub initialize
{
	# Setup to use default file if no args
	if(@ARGV == 0)
	 {
		@ARGV = ($alert_file);
	 }
  	$count = 0;
  
  	$fhnonce='fh00';
  
 	$cwd= getcwd();
  
 	# Setup an output directory named after the first file (unless defined on command line)
  
	my $last = rindex($ARGV[0],$dirsep);
	my $substr = substr($ARGV[0],$last+1);
	$file_name[$count] = $substr;
	$output_dir = "snfout.$file_name[$count]" unless defined($output_dir);
	$anomoutdir= $output_dir.$dirsep.'anomrep';
  
	if(-e $output_dir) {
	 	&clear_dir($output_dir);
	} else {
		mkdir($output_dir,0755);
	}
	&write_logo_file("$output_dir/$logo_filename");
}


sub clear_dir {
	my($dirpath)= shift;
	my $fh= $fhnonce++;
	my ($file,$fullpath);
	die("$dirpath already exists and is not a directory\n")
		unless -d $dirpath;
  	if (opendir($fh,$dirpath)) {
  		while ($file=readdir($fh)) {
 			$fullpath= "$dirpath$dirsep$file";
 			if ($file =~ /^(\d{1,3}|sig)$/ && -d $fullpath) {
  				&clear_dir($fullpath);
				rmdir($fullpath) || warn "could not delete directory $fullpath";
  			} elsif ($file =~ /^((src|dest)\d+\.\d+\.\d+\.\d+.*|(sig\d+)|allbyscore|allbytime|srcips|destips|index)\.$html/) {
				# looks like a file we created, so delete it
				unlink($fullpath) || warn "could not clear $file in $dirpath";
			}
		}
		closedir($fh);
  	} else {
		warn "could not open output directory $dirpath to clear it";
  	}
}

##############################################################################

sub initialize_per_file
{
  my($file) = @_;

  $alert_file = $file;
  my $fh='alertfh00';
  unless (open($fh,"$alert_file")) {
  	warn("Couldn't open input file $alert_file\n");
  	return undef;
  }
  $file_fullpath[$count]= $file;
  $file_fullpath[$count]= "$cwd/$file" unless $file =~ /^\//;
  my $last = rindex($alert_file,$dirsep);
  my $substr = substr($alert_file,$last+1);
  $file_name[$count] = $substr;
  return $fh;
}

##############################################################################

sub close_out_file
{
  close($_[0]);
  $count++;
}

##############################################################################

# return the next alert from the given file handle or undef if we hit the end of the file
sub get_alert
{
	my ($fh)= @_;
	# get and parse the next alert in the file, courtesy of snort_alert_parse.pl
	my ($alerttext,$format)= &next_alert($fh);
	return undef unless defined($alerttext); # must have hit end of file
	my $alert= &parse_alert($alerttext,$format);
	$alert->{'type'}= $format;
	# we used to store source file by 'file' => $file_name[$count], but that doesn't seem to be needed anywhere
	return $alert;
}

##############################################################################

# keep records for a new normal alert
sub process_alert
{
	my($alert) = @_;
#print STDOUT "got $alert->{'sig'}; processing\n";

	# Global stuff
	$earliest_overall = &earliest($alert, $earliest_overall);
	$latest_overall = &latest($alert, $latest_overall);
	$alert_count++;
	
	# Per signature stuff
	$sig_count{$alert->{'sig'}}++;
	$earliest_sig{$alert->{'sig'}}
		= &earliest($alert, $earliest_sig{$alert->{'sig'}});
	$latest_sig{$alert->{'sig'}}
		= &latest($alert, $latest_sig{$alert->{'sig'}});

	# Per source stuff
	$src_count{$alert->{'src'}}++;
	push @{$src_list{$alert->{'src'}}}, $alert;
	# no longer used
	#$earliest_src{$alert->{'src'}}
	#	= &earliest($alert, $earliest_src{$alert->{'src'}});
	#$latest_src{$alert->{'src'}}
	#	= &latest($alert, $latest_src{$alert->{'src'}});

	# Per dest stuff
	$dest_count{$alert->{'dest'}}++;
	push @{$dest_list{$alert->{'dest'}}}, $alert;
	# no longer used
	#$earliest_dest{$alert->{'dest'}}
	#	= &earliest($alert, $earliest_dest{$alert->{'dest'}});
	#$latest_dest{$alert->{'dest'}}
	#	= &latest($alert, $latest_dest{$alert->{'dest'}});
	
	# Per signature-source stuff
	$sig_src_count{$alert->{'sig'}}{$alert->{'src'}}++;
	
	# Per signature-dest stuff
	$sig_dest_count{$alert->{'sig'}}{$alert->{'dest'}}++;
	
	# Per source-signature stuff
	$src_sig_count{$alert->{'src'}}{$alert->{'sig'}}++;
	
	# Per dest-signaturestuff
	$dest_sig_count{$alert->{'dest'}}{$alert->{'sig'}}++;
	
	# Src dest stuff
	$src_dest_count{$alert->{'src'}}{$alert->{'dest'}}++;
	$dest_src_count{$alert->{'dest'}}{$alert->{'src'}}++;
		
	# Sig src dest stuff
	$sig_src_dest_count{$alert->{'sig'}}{$alert->{'src'}}{$alert->{'dest'}}++;
	$sig_dest_src_count{$alert->{'sig'}}{$alert->{'dest'}}{$alert->{'src'}}++;
}


# keep records for a new anomaly alert
sub process_anom_alert
{
	my($alert) = @_;

	# Global anom stuff
	push(@anom_list,$alert);
	
	# Per source stuff
	$anom_src_count{$alert->{'src'}}++;
	push @{$anom_src_list{$alert->{'src'}}}, $alert;
	$earliest_anom_src->{$alert->{'src'}}
		= &earliest($alert, $earliest_anom_src->{$alert->{'src'}});
	$latest_anom_src->{$alert->{'src'}}
		= &latest($alert, $latest_anom_src->{$alert->{'src'}});

	# Per dest stuff
	$anom_dest_count{$alert->{'dest'}}++;
	push @{$anom_dest_list{$alert->{'dest'}}}, $alert;
	$earliest_anom_dest->{$alert->{'dest'}}
		= &earliest($alert, $earliest_anom_dest->{$alert->{'dest'}});
	$latest_anom_dest->{$alert->{'dest'}}
		= &latest($alert, $latest_anom_dest->{$alert->{'dest'}});
		
	# Src dest stuff
	$anom_src_dest_count{$alert->{'src'}}{$alert->{'dest'}}++;	 
	$anom_dest_src_count{$alert->{'dest'}}{$alert->{'src'}}++;	 
}

##############################################################################

# give each signature a number and find the HTML to use
sub construct_sig_indexes
{
  my($sig,$count);
  
  foreach $sig (keys %sig_count)
   {
		$sig_index->{$sig} = $count++;
		if($sig =~ /IDS(\d+)/)
		 {
		 	my $num= $1;
		 	$num =~ s/^0+//;
			# ARACHNIDS signature - can make a nice URL for these.
			$sig_url->{$sig} = "http:\/\/whitehats.com\/IDS\/$num";
			$sig_entry->{$sig} = "<a href=\"$sig_url->{$sig}\"".&target('siginfo').">$sig</a>";
		 }
		else
		 { 
			$sig_entry->{$sig} = $sig;
		 }
    }
}

##############################################################################

sub output_main_sig_page
{
	my($sig);
	my $page_h2 = "All Snort signatures";
	my $page_title = "SnortSnarf: Snort signatures in $file_name[0] et al";
  
  	my $PAGE=&open_page($output_dir,&siglist_page());
	select($PAGE);
	my $base=&siglist_base();
	&print_page_head($page_title,'start page',$page_h2,$base);
  
	print (0+$alert_count);
	print " alerts found among the files:";
	print "<ul>\n<li>";
	print join("\n<li>",@file_name);
	print "\n</ul>\n";
	print "Earliest alert at ".&pretty_time($earliest_overall)."<br>\n";
	print "Latest alert at ".&pretty_time($latest_overall)."</p>\n";
	if (@anom_list) {
		print "<p>The ".+@anom_list." reports from the <A HREF=\"http:\/\/www.silicondefense.com/spice/\">Spade anomaly sensor</A> are in a separte section: <A HREF=\"$base"."anomrep/\">visit it</a></p>\n";
	}


	print "<TABLE BORDER CELLPADDING = 5>\n";
	print "<TR><TD>Signature (click for definition)</TD><TD>\# Alerts</TD>".
				"<TD>\# Sources</TD><TD>\# Destinations</TD><TD>Detail link</TD></TR>\n";

	foreach $sig (sort sort_by_sig_count keys %sig_count)
	{
		print "<TR><TD>$sig_entry->{$sig}</TD><TD>$sig_count{$sig}</TD><TD>".
				scalar(keys %{$sig_src_count{$sig}})."</TD><TD>".
				scalar(keys %{$sig_dest_count{$sig}}).
				"</TD><TD><a href=\"$base".&sig_page($sig_index->{$sig})."\">Summary</a></TD></TR>\n";
	}
	print "</TABLE>\n\n";
	if ($db_file ne '') {
		print "<FORM ACTION=\"$cgi_dir/view_annotations.pl\">View/add annotations of type <SELECT NAME=\"type\"><INPUT TYPE=hidden NAME=\"file\" VALUE=\"$db_file\">";
		foreach ('IP','network','snort message') {
			print "<OPTION VALUE=\"$_\"> $_";
		}
		print "</SELECT> for key : <INPUT NAME=\"key\" SIZE=15><INPUT TYPE=\"submit\" VALUE=\"View\"></FORM>";
	}
  
	&print_page_foot();
	close($PAGE);
}

sub sort_by_sig_count { $sig_count{$a} <=> $sig_count{$b};}

##############################################################################

# produce the anomaly section, if there are any anomaly reports
sub output_anom_sect {
	return unless @anom_list;
	# clear the anomaly output directory if it exists
	if (-e $anomoutdir) {
		&clear_dir($anomoutdir);
	}
	unless (-e $anomoutdir && -d $anomoutdir) {
		mkdir($anomoutdir,0755) || die "could not make directory $anomoutdir for the anomaly report section";
	}
	
	my $page_title = "Snortsnarf: Main Spade report page for $file_name[0] et al";
  
  	my $PAGE= &open_page($anomoutdir,&anomindex_page());
	select($PAGE);
	my $base= &anomindex_base();
	&print_page_head($page_title,'Spade report main page',scalar(@anom_list).' anomaly reports',$base);
  	&output_anom_header($base);
	&print_page_foot();
	close($PAGE);

	&output_all_anom_page("score",\&sort_by_score);
	&output_all_anom_page("time",\&sort_by_time);
	&output_anom_src_page();
	&output_anom_dest_page();
	&output_anom_per_source();
	&output_anom_per_dest();
}

sub sort_by_score 
{ 
 $b->{'anomscore'} <=> $a->{'anomscore'};
}


##############################################################################

# output the page for each signature
sub output_per_sig
{
  my($sig,$sig_file,$src,$dest,$early,$late);
  my $page_title;
  
  foreach $sig (keys %sig_count)
   {  
	# Sort out the file
	$global_sig = $sig;	 # ouch - need to communicate with sort_by_sig_src_count
  	my $PAGE=&open_page($output_dir,&sig_page($sig_index->{$sig}));
	select($PAGE);

	# Print page head stuff
	$page_title = "Summary of alerts in $file_name[0] et al for signature: $sig";
	my $base=&sig_base($sig_index->{$sig});
	&print_page_head($page_title,'signature page',$sig_entry->{$sig},$base);
	#print "<h3>$sig_entry->{$sig}</h3>";
	print "<p>$sig_count{$sig} alerts with this signature among the files:<ul>\n<li>\n";
	print join("\n<li>",@file_name);
	print "\n</ul>\n";

	print "Earliest such alert at ".&pretty_time($earliest_sig{$sig})."<br>\n";
	print "Latest such alert at ".&pretty_time($latest_sig{$sig})."</p>\n";

	# print page head table stuff
	print "<table border cellpadding = 3>\n";
  	print "<tr><td>$sig</td>\n";
  	print "<td><A HREF=#srcsect>",scalar(keys %{$sig_src_count{$sig}})," sources</A></td>\n";
  	print "<td><A HREF=#destsect>",scalar(keys %{$sig_dest_count{$sig}})," destinations</A></td></tr>\n";
	if ($db_file ne '') { #link to annotations
		print "<tr><td colspan=3 align=center><A HREF=\"",&view_ann_url('snort message',$sig),"\">View/add annotations for this signature</A></td></tr>\n";
	}
	if (defined($rules_file) && $sig !~ /^(UDP|TCP|ICMP)\s+scan$/) { # show rule file entries for signature
		my(@rules_html)= &get_rules_html_for_sig($sig,$rules_file);
		if (@rules_html) {
			print "<tr bgcolor=\"#D5E2CE\"><td colspan=3 align=center>Rules with message \"$sig\":</td></tr>\n";
			foreach (@rules_html) {
				print "<tr bgcolor=\"#D5E2CE\"><td colspan=3 align=left>$_</td></tr>\n";
			}
		}
	}
	print "</table>";

	# Print the sources section
	print "<hr><h3><A NAME=srcsect>Sources triggering this attack signature</A></h3>\n";
	print "<TABLE BORDER CELLPADDING = 5>\n";
	print "<tr><td>Source</td><td>\# Alerts (sig)</td>".
				"<td>\# Alerts (total)</td><td>\# Dsts (sig)</td>".
				"<td>\# Dsts (total))</td></tr>\n";
	foreach $src (sort sort_by_sig_src_count keys %{$sig_src_count{$sig}})
		 {
		  print "<tr><td><a href=\"$base".&host_page($src,'src')."\">$src</a></td>".
				"<td>$sig_src_count{$sig}{$src}</td>".
				"<td>$src_count{$src}</td><td>".
				scalar(keys %{$sig_src_dest_count{$sig}{$src}})."</td><td>".
				scalar(keys %{$src_dest_count{$src}})."</td></tr>\n";
		 }
	print "</TABLE>\n";
		
	# Print the destinations section
		print "<hr><h3><A NAME=destsect>Destinations receiving this attack signature</A></h3>\n";
		print "<TABLE BORDER CELLPADDING = 5>\n";
		print "<tr><td>Destinations</td><td>\# Alerts (sig)</td>".
				"<td>\# Alerts (total)</td><td>\# Srcs (sig)</td>".
				"<td>\# Srcs (total))</td></tr>\n";
		foreach $dest (sort sort_by_sig_dest_count
											keys %{$sig_dest_count{$sig}})
		 {
		  print "<tr><td><a href=\"$base".&host_page($dest,'dest')."\">$dest</a></td>".
				"<td>$sig_dest_count{$sig}{$dest}</td>".
				"<td>$dest_count{$dest}</td><td>".
				scalar(keys %{$sig_dest_src_count{$sig}{$dest}})."</td><td>".
				scalar(keys %{$dest_src_count{$dest}})."</td></tr>\n";
		 }
	print "</TABLE>\n"; 
	&print_page_foot();
	close($PAGE);
   }
}

sub sort_by_sig_src_count 
{ 
 $sig_src_count{$global_sig}{$b} <=> 
										$sig_src_count{$global_sig}{$a};
}

sub sort_by_sig_dest_count 
{ 
 $sig_dest_count{$global_sig}{$b} <=> 
										$sig_dest_count{$global_sig}{$a};
}

##############################################################################

# make a page for each source IP
sub output_per_source
{
   my($src,$src_file);
   
   foreach $src (keys %src_count)
	{  
		&output_per_host($output_dir,$src,'src','general',"from $src in $file_name[0] et al",$src_list{$src});
	}
}

##############################################################################

# make a page for each destination IP
sub output_per_dest
{
   my($dest,$dest_file);
   
   foreach $dest (keys %dest_count)
	{  
		&output_per_host($output_dir,$dest,'dest','general',"going to $dest in $file_name[0] et al",$dest_list{$dest});
	}
}

##############################################################################

# make page(s) listing the alerts for an IP address
sub output_per_host {
	my($outdir,$ip,$end,$pagetype,$al_descr,$alertsref)= @_;
	# this is a very general function.  The args are:
	# + the directory to put the file(s) in
	# + the IP address the page is about
	# + the end this page is about ('src' or 'dest')
	# + the type of page to produce ('general' or 'anom')
	# + the description of the alerts
	# + a reference to the a list of alerts to put on the page
	@alerts= sort sort_by_time @{$alertsref};
	my $num_alerts= @alerts;
	my $ip_file = &host_page($ip,$end);
	my $host_base= &host_base($ip,$end);
	my $PAGE= &open_page($outdir,$ip_file);
	my $prevsel= select($PAGE);
	
	$al_descr= ($pagetype eq 'general'?'':'anomaly ')."alerts $al_descr";
	my $page_type= &alert_type_text($pagetype)."alert page";
	my $end_ip= ($end eq 'src'?'Source':'Destination').": <EM>$ip</EM>";
	if ($split_threshold == 0 || $num_alerts <=  $split_threshold) {
		# start the page
		&print_page_head("All $num_alerts $al_descr",$page_type,$end_ip,$host_base);
		&output_per_host_header($ip,$end,$pagetype,$host_base,$alerts[0],$alerts[$#alerts],$num_alerts);
		&output_alert_table(\@alerts,$host_base);
	} else {  # need to split
		# make the page containing the overview the alerts
		&print_page_head("Overview of $num_alerts $al_descr",$page_type,"$end_ip: overview",$host_base);
		&output_per_host_header($ip,$end,$pagetype,$host_base,$alerts[0],$alerts[$#alerts],$num_alerts);

		my $all_file = &host_page($ip,$end,'all');

		print "<hr>This listing contains $num_alerts alerts.  You can:\n";
		print "<UL><LI><A HREF=\"$host_base$all_file\">view the whole listing</A>\n";
		print "<LI>view a range of alerts <A HREF=#rangelist>(see table below)</A></UL>\n";

		print "<hr><A NAME=rangelist>Alert ranges (sorted by time):</A>\n";
		print "<table border cellpadding = 3><TR align=center><B><TD>alert #'s</TD><TD>first time</TD><TD>last time</TD></B></TR>\n";
		my($first,$last);
		foreach ($first=1; $first < $num_alerts; $first+=$split_threshold) { # segment the alerts
			$last= $first+$split_threshold-1;
			$last= $last < $num_alerts ? $last : $num_alerts;
			my(@alertsub)= @alerts[($first-1)..($last-1)];
			my $early= $alertsub[0];
			my $late= $alertsub[$#alertsub];
			my $range_file = &host_page($ip,$end,$first);
			# print this to the "all" page
			print "<tr><td><A HREF=\"$host_base$range_file\">$first to $last</A></td><td>",&pretty_time($early),"</td><td>",&pretty_time($late),"</td></tr>\n";

			# create page for the range
			my $RANGE=&open_page($outdir,$range_file);
			my $prevsel= select($RANGE);
			&print_page_head("$first to $last of $num_alerts $al_descr",$page_type,"$end_ip: #$first-$last",$host_base,$host_base);
			&output_per_host_header($ip,$end,$pagetype,$host_base,$early,$late);
			print "<hr>";
			my $nav= "Go to: ".
				($first == 1?'':"<A HREF=\"$host_base".&host_page($ip,$end,($first-$split_threshold))."\">previous range</A>, ").
				(($first+$split_threshold >= $num_alerts)?'':"<A HREF=\"$host_base".&host_page($ip,$end,($first+$split_threshold))."\">next range</A>, ").
				" <A HREF=\"$host_base$all_file\">all alerts</A>, <A HREF=\"$host_base$ip_file\">overview page</A>";
			print $nav;
			&output_alert_table(\@alertsub,$host_base);
			print $nav;
			&print_page_foot();
			close($RANGE); select($prevsel);
		}		
		print "</table>\n";
		
		# now make the page for all the alerts
		my $ALL= &open_page($outdir,$all_file);
		my $prevsel= select($ALL);
		&print_page_head("All $num_alerts $al_descr",$page_type,$end_ip,$host_base);
		&output_per_host_header($ip,$end,$pagetype,$host_base,$alerts[0],$alerts[$#alerts],$num_alerts);
		my $nav= "Go to: <A HREF=\"$host_base$ip_file\">overview page</A>";
		print $nav;
		&output_alert_table(\@alerts,$host_base);
		print $nav;
		&print_page_foot();
		close($ALL); select($prevsel);
	}
	
	# finish the page we started with
	&print_page_foot();
	close($PAGE); select($prevsel);
}

##############################################################################

# make the header section at the top of a host page
sub output_per_host_header
{
	my ($ip,$end,$pagetype,$base,$early,$late,$num_alerts)=@_;
	# the args are:
	# + the IP address the header is about
	# + the end this page is about ('src' or 'dest')
	# + the type of page to produce ('general' or 'anom')
	# + the earliest alert for the page
	# + the latest alert
	# + the number of alerts this page is about (optional)
	# + path to the base directory of this page type
	if (defined($num_alerts)) {
		print (0+$num_alerts);
		print " such alerts among the files:<ul>\n<li>\n";
	} else {
		print "Looking in files:<ul>\n<li>\n";
	}
	print join("\n<li>",@file_name);
	print "\n</ul>\n";		
	print "<br>Earliest: ".&pretty_time($early)."<br>\n";
	print "Latest: ".&pretty_time($late)."\n<P>";
	
	&output_host_siglist($ip,$end,$base) if $pagetype eq 'general';
	
	my $distinct_ips= ($pagetype eq 'general')?
		(($end eq 'src') ?
			0+(keys %{$src_dest_count{$ip}})." distinct destination" :
			0+(keys %{$dest_src_count{$ip}})." distinct source" ) :
		(($end eq 'src') ?
			0+(keys %{$anom_src_dest_count{$ip}})." distinct destination" :
			0+(keys %{$anom_dest_src_count{$ip}})." distinct source" );
	print "There are $distinct_ips IPs in the alerts of the type on this page.<P>\n";

	&output_anom_pagelist($base) if $pagetype eq 'anom';
	
	# start the table and get the width
	print "<table border cellpadding = 3>\n";
	my $cols= &print_ip_lookup($ip);

	# find the paths to the general and anomaly directories
	my($genpath,$anompath);
	if ($pagetype eq 'general') {
		$genpath=$base;
		$anompath="$base"."anomrep/";
	} else  {
		$genpath="$base../";
		$anompath=$base;
	}
	
	my(@seealso)= (); # make a list of "see also"s that we want
	if (!(($pagetype eq 'general') && ($end eq 'src')) && exists $src_count{$ip}) {
		push(@seealso,"<A HREF=\"$genpath".&host_page($ip,'src')."\">an alert source</A> [".$src_count{$ip}." alerts]");
	}
	if (!(($pagetype eq 'general') && ($end eq 'dest')) && exists $dest_count{$ip}) {
		push(@seealso,"<A HREF=\"$genpath".&host_page($ip,'dest')."\">an alert destination</A> [".$dest_count{$ip}." alerts]");
	}
	if (!(($pagetype eq 'anom') && ($end eq 'src')) && exists $anom_src_count{$ip}) {
		push(@seealso,"<A HREF=\"$anompath".&host_page($ip,'src')."\">an anomaly source</A> [".$anom_src_count{$ip}." alerts]");
	}
	if (!(($pagetype eq 'anom') && ($end eq 'dest')) && exists $anom_dest_count{$ip}) {
		push(@seealso,"<A HREF=\"$anompath".&host_page($ip,'dest')."\">an anomaly destination</A> [".$anom_dest_count{$ip}." alerts]");
	}
	if (@seealso == 1) { # print this with one row
		print "<tr bgcolor=\"#E0CDD0\"><td align=center colspan=$cols>See also $ip as $seealso[0]</td></tr>\n";
	} elsif (@seealso) { # make a common first 2 columns and distinct last columns
		my $colwidth=$cols-2;
		print "<tr bgcolor=\"#E0CDD0\"><td rowspan=".(0+@seealso)." align=center colspan=2>See also $ip as:</td>";
		print "<td colspan=$colwidth align=left>$seealso[0]</td></tr>\n";
		foreach (@seealso[1..$#seealso]) {
			print "<tr bgcolor=\"#E0CDD0\"><td colspan=$colwidth align=left>$_</td></tr>\n";
		}
	}
	
	if ($db_file ne '') {
		# generate links to annotations
		print "<tr><td colspan=$cols align=center><A HREF=\"",&view_ann_url('IP',$ip),"\">View/add annotations for this IP address</A></td></tr>\n";
		$ip =~ /^(\d+\.\d+\.\d+)/;
		my $netkey= "$1.0/24";
		print "<tr><td colspan=$cols align=center><A HREF=\"",&view_ann_url('network',$netkey),"\">View/add annotations for $netkey</A></td></tr>\n";
		$ip =~ /^(\d+\.\d+)/;
		$netkey= "$1.0.0/16";
		print "<tr><td colspan=$cols align=center><A HREF=\"",&view_ann_url('network',$netkey),"\">View/add annotations for $netkey</A></td></tr>\n";
	}
	
	if (defined($nmap_url) && (!defined($nmap_dir) || -e "$nmap_dir/$ip.html")) {
		# generate links to the corresponding nmap2html page
		print "<tr bgcolor=\"#DDDDDD\"><td colspan=$cols align=center><A HREF=\"$nmap_url$ip.html\">View nmap log page for $ip".(defined($nmap_dir)?'':' (if any)')."</A></td></tr>\n";
	}
	
	my $endip= ($end eq 'src') ? "from $ip" : "to $ip";
	my $typefulldescr= &alert_type_text($pagetype)."alerts $endip";
	if ($cgiavail) {
		print "<td colspan=$cols align=center><A HREF=\"$cgi_dir/text4sel.pl?".join('&',"end=$end","ip=$ip",'include='.($pagetype eq 'general'?'g':'a'),'logs='.&url_encode(join(',',@file_fullpath))).'"',">Fresh grab of all $typefulldescr (as text)</A></td></tr>\n";
	}
	
	if (defined($sisr_config)) {
		# make links to SISR
		my $encconfig= &url_encode($sisr_config);
		# we want to "create" links if there are some anom entries and some normal ones for this IP and end 
		my $showcreate2= ($pagetype eq 'anom') && ($end eq 'src') && exists $src_count{$ip};
		$showcreate2 ||= ($pagetype eq 'anom') && ($end eq 'dest') && exists $dest_count{$ip};
		$showcreate2 ||= ($pagetype eq 'general') && ($end eq 'src') && exists $anom_src_count{$ip};
		$showcreate2 ||= ($pagetype eq 'general') && ($end eq 'dest') && exists $anom_dest_count{$ip};
		my $rows= 3+ ($showcreate2 ? 1 : 0);
		my $colwidth=$cols-1;
		print "<tr bgcolor=\"#D5E2CE\"><td rowspan=$rows align=center>Incident handling</td>";
		my $linktext= "Add some the $typefulldescr to a labeled set";
		print "<td colspan=$colwidth align=center><A HREF=\"$cgi_dir/sel_to_add.pl?".join('&',"configfile=$encconfig","end=$end","ip=$ip",'include='.($pagetype eq 'general'?'g':'a'),'logs='.&url_encode(join(',',@file_fullpath))).'"',&target('sisrwin'),">$linktext</A></td></tr>\n";
		if ($showcreate2) {
			$linktext= "Add some of both type alerts $endip to a labeled set";
			print "<tr bgcolor=\"#D5E2CE\"><td colspan=$colwidth align=center><A HREF=\"$cgi_dir/sel_to_add.pl?".join('&',"configfile=$encconfig","end=$end","ip=$ip",'include=ga','logs='.&url_encode(join(',',@file_fullpath))).'"',&target('sisrwin'),">$linktext</A></td></tr>\n";
		}
		print "<tr bgcolor=\"#D5E2CE\"><td colspan=$colwidth align=center><A HREF=\"$cgi_dir/lsetlist.pl?configfile=$encconfig\"",&target('sisrwin'),">List stored sets</td></tr>\n";
		print "<tr bgcolor=\"#D5E2CE\"><td colspan=$colwidth align=center><A HREF=\"$cgi_dir/inclist.pl?configfile=$encconfig\"",&target('sisrwin'),">List stored incidents</td></tr>\n";
	}
	print "</table>\n";
}


sub output_host_siglist {
	my($ip,$end,$base)= @_;
	my($sighash)= $end eq 'src' ? $src_sig_count{$ip} : $dest_sig_count{$ip};
	my @sigs= keys %{$sighash};
	print 0+@sigs," different signatures are present for <EM>$ip</EM> as a ",($end eq 'src' ? 'source' : 'destination'),"\n<UL>";
	foreach (sort {$sighash->{$a} <=> $sighash->{$b}} @sigs) {
		print "<LI>$sighash->{$_} instances of <a href=\"$base".&sig_page($sig_index->{$_})."\"><EM>$_</EM></A></LI>\n";
	}
	print "</UL>";
}

sub alert_type_text {
	return @anom_list?(($_[0] eq 'general')?'standard ':'anomaly '):'';
}


# return a URL to view annotations for and type and key
sub view_ann_url {
	my($type,$key)= @_;
	return "$cgi_dir/view_annotations.pl\?".join('&','file='.&url_encode($db_file),'type='.&url_encode($type),'key='.&url_encode($key));
}


##############################################################################
# make a page listing all anomaly alerts
sub output_all_anom_page {
	my($sortbytext,$sortfn)= @_;
	# args are:
	# + the text for "sorted by..."
	# + a function to use to sort the alerts
	my @alerts= sort $sortfn @anom_list;
	my $num_alerts= @alerts;
	my $PAGE= &open_page($anomoutdir,&anomall_page($sortbytext));
	my $prevsel= select($PAGE);
		
	my $num_anom= scalar(@anom_list);
	my $base=&anomall_base($sortbytext);
	&print_page_head("All $num_anom Anomaly reports, sorted by $sortbytext",'Spade report listing',"All $num_anom reports sorted by $sortbytext",$base);
	&output_anom_header($base);
	&output_alert_table(\@alerts,$base);
	
	&print_page_foot();
	close($PAGE); select($prevsel);
}

# print out the anomaly specific header
sub output_anom_header {
	my $base=shift;
	my $totcount= @anom_list;
	print "$totcount anomaly reports found among the files:<ul>\n<li>\n";
	print join("\n<li>",@file_name);
	print "\n</ul>\n";
	&output_anom_pagelist($base);

}

sub output_anom_pagelist {
	my $base=shift;
	my $srccount= keys %anom_src_list;
	my $destcount= keys %anom_dest_list;
	print "There are 4 top level pages for alerts produced by the <A HREF=\"http:\/\/www.silicondefense.com/spice/\">Spade anomaly sensor</A>:\n<UL>";
	print "<LI><A HREF=\"$base".&anomall_page('score')."\">All alerts sorted by score\n";
	print "<LI><A HREF=\"$base".&anomall_page('time')."\">All alerts sorted by time\n";
	print "<LI><A HREF=\"$base".&anomsrcs_page()."\">A list of the $srccount source IP addresses in the alerts\n";
	print "<LI><A HREF=\"$base".&anomdests_page()."\">A list of the $destcount destination IP addresses in the alerts\n";
	print "</UL>\n<P><A HREF=\"$base../\">See also the main signature page</A>\n";
}

##############################################################################
# display the pages with anomaly source IPS
sub output_anom_src_page
{
	my $PAGE= &open_page($anomoutdir,&anomsrcs_page());
	select($PAGE);

	# Print page head stuff
	my $page_title = "Source IPs of anomaly reports in $file_name[0] et al";
	my $base= &anomsrcs_base();
	&print_page_head($page_title,'Spade report IP list','All source IPs',$base);
	&output_anom_header($base);
	my $ipcount= keys %anom_src_list;
	print "<p>$ipcount distinct source IPs are present in the anomaly reports.</p>\n";

	# Print the sources section
	print "<hr><h3><A NAME=srcsect>Source IPs in anomaly reports</A></h3>\n";
	print "<TABLE BORDER CELLPADDING = 5>\n";
	print "<tr><td>Source</td><td>\# Alerts</td>".
				"<td>\# Anom Dsts</td>".
				"<td>\# Other Alerts </td></tr>\n";
	my $src;
	foreach $src (sort sort_by_anom_src_count keys %anom_src_list) {
		print "<tr><td><a href=\"$base".&host_page($src,'src')."\">$src</a></td>".
			"<td>$anom_src_count{$src}</td><td>".
			scalar(keys %{$anom_src_dest_count{$src}})."</td>".
			"<td>".(0+($src_count{$src}))."</td></tr>\n";
	}
	print "</TABLE>\n";
		
	&print_page_foot();
	close($PAGE);
}

sub sort_by_anom_src_count 
{ 
 $anom_src_count{$b} <=> $anom_src_count{$a};
}

# display the pages with anomaly destination IPS
sub output_anom_dest_page
{
	my $PAGE= &open_page($anomoutdir,&anomdests_page());
	select($PAGE);

	# Print page head stuff
	my $page_title = "Destination IPs of anomaly reports in $file_name[0] et al";
	my $base=&anomdests_base();
	&print_page_head($page_title,'Spade report IP list','All destination IPs',$base);
	&output_anom_header($base);
	my $ipcount= keys %anom_dest_list;
	print "<p>$ipcount distinct destination IPs are present in the anomaly reports.</p>\n";

	# Print the dests section
	print "<hr><h3><A NAME=srcsect>Destination IPs in anomaly reports</A></h3>\n";
	print "<TABLE BORDER CELLPADDING = 5>\n";
	print "<tr><td>Destination</td><td>\# Alerts</td>".
				"<td>\# Anom Srcs</td>".
				"<td>\# Other Alerts </td></tr>\n";
	my $dest;
	foreach $dest (sort sort_by_anom_dest_count keys %anom_dest_list) {
		print "<tr><td><a href=\"$base".&host_page($dest,'dest')."\">$dest</a></td>".
			"<td>$anom_dest_count{$dest}</td><td>".
			scalar(keys %{$anom_dest_src_count{$dest}})."</td>".
			"<td>".(0+($dest_count{$dest}))."</td></tr>\n";
	}
	print "</TABLE>\n";
		
	&print_page_foot();
	close($PAGE);
}

sub sort_by_anom_dest_count 
{ 
 $anom_dest_count{$b} <=> $anom_dest_count{$a};
}

##############################################################################

# make a page for each anomaly source IP
sub output_anom_per_source
{
	my($src);
   
	foreach $src (keys %anom_src_list) {  
		&output_per_host($anomoutdir,$src,'src','anom',"from $src in $file_name[0] et al",$anom_src_list{$src});
	}
}

# make a page for each anomaly destination IP
sub output_anom_per_dest
{
	my($dest);
   
	foreach $dest (keys %anom_dest_list) {  
		&output_per_host($anomoutdir,$dest,'dest','anom',"to $dest in $file_name[0] et al",$anom_dest_list{$dest});
	}
}


##############################################################################

# output a table of alerts
sub output_alert_table {
	my($alertsref,$base)= @_;
	print "<HR>";
	print "<table border cellpadding = 3>\n";
	for (my $i=0; $i <= $#{$alertsref}; $i++) {
		&output_table_entry($alertsref->[$i],$base);
	 }
	print "</table>\n";
}

# output a table entry for a given alert
sub output_table_entry { 
	my($alert,$base)= @_;
	my  $tdopts= $color_opt eq 'rotate'?" bgcolor=".&alert_color($alert):'';
	print "<tr><td$tdopts>".&alert_as_html($alert,$base)."</td></tr>\n";
}

##############################################################################

# find a color to use as the background for an alert
sub alert_color { 
	my($alert)= @_;
	return $color[$::old_color] if (defined($::old_alert) && (($::old_alert->{'sig'} eq $alert->{'sig'}) || defined($alert->{'anomscore'})) && ($::old_alert->{'src'} eq $alert->{'src'}) && ($::old_alert->{'dest'} eq $alert->{'dest'}));
	$::old_color= ++$::old_color % +@color;
	$::old_alert= $alert;
	return $color[$::old_color];
}

##############################################################################

# make an alert into HTML
sub alert_as_html { 
	my($alert,$base)= @_;
	$append= '';
	$text= $alert->{'text'};
    unless (defined($alert->{'anomscore'})) {
	    my $sig= $alert->{'sig'};
	    my $sigre= $sig;
	    $sigre =~ s/([^\w ])/\\$1/g;
	    $sigurl= &sig_page($sig_index->{$sig});
	    $text =~ s/(\[\*\*\]\s*)($sigre)(\s*\[\*\*\])/$1<a href=\"$base$sigurl\">$2<\/A>$3/;
	}
	if (defined($alert->{'sport'})) {
		my $newwindow = &target('lookup'); # Port lookup code contrib by Mike Biesele
		$text =~ s/(\d+\.\d+\.\d+\.\d+):(\d+) ->/"<A HREF=\"$base".&host_page($1,'src')."\">$1<\/A>:<A HREF=\"http:\/\/www.snort.org\/Database\/portsearch.asp?Port=$2\" $newwindow>$2<\/A>->"/e;
		$text =~ s/->(\s*)(\d+\.\d+\.\d+\.\d+):(\d+)/"->$1<A HREF=\"$base".&host_page($2,'dest')."\">$2<\/A>:<A HREF=\"http:\/\/www.snort.org\/Database\/portsearch.asp?Port=$3\" $newwindow>$3<\/A>"/e;
	} else {
		$text =~ s/(\d+\.\d+\.\d+\.\d+)(.*)->/"<A HREF=\"$base".&host_page($1,'src')."\">$1<\/A>$2->"/e;
		$text =~ s/->(\s*)(\d+\.\d+\.\d+\.\d+)/"->$1<A HREF=\"$base".&host_page($2,'dest')."\">$2<\/A>"/e;
	}
	$text =~ s/[\n\r]+/<br>/g;
    if ($log_base ne '' && ($alert->{'type'} =~ /alert$/) && defined($alert->{'proto'})) {
       $append.= " <A HREF=\"".&get_alert_logpage($alert)."\">[Snort log]</A>\n";
    } 
	return "<code>$text</code>$append";
}

##############################################################################

# text to add to use a given locatation as a target link or '' if -onewindow was given
sub target {
	return '' if $notarget_option;
	return " target=$_[0]";
}

##############################################################################

# given a snort message, dig through the given rules file and generate HTML
# for each line than has that signature.  A list is produced and included
# files are followed
sub get_rules_html_for_sig {
	my ($sig,$rule_file)= @_;
	my(@rules_html)=();
	my $fh= $fhnonce++;
	# file names assumed to be absolute or relative to the current directory
	# but if -rulesdir was given, always use that as the dirctory (with the
	# file name appended to make the file location)
	$rule_file =~ s/\s+$//;
	my($file)= $rule_file =~ /([^\/]+)$/; 
	if (defined($rules_dir)) {
		$rule_file= "$rules_dir/$file";
	}
	unless (open($fh,"<$rule_file")) {
		warn "could not open $rule_file to read rules from -- skipping\n";
		return;
	}
	while (<$fh>) {
		chomp;
		next if /^(\#|\s*$)/;
		if (s/^\s*include\s+//) {
			s/\s+$//;
			push(@rules_html,&get_rules_html_for_sig($sig,$_));
		} elsif (/\(.*msg\s*:\s*\"([^\"]*)\"/) {
			my $mess= $1;
			$mess =~ s/\\(.)/$1/g;
			$mess =~ s/^\s+//;
			$mess =~ s/\s+$//;
			if ($mess eq $sig) { # message in rule is same as one looked for (modulo whitespace)
				my $html= "<SMALL>$_</SMALL> (from <EM>$file</EM>)";
				push(@rules_html,$html);
			}
		}
	}
	close $fh;
	return @rules_html;
}

##############################################################################

# return the page in the full snort logs that corresponds to a given alert
sub get_alert_logpage 
{
	my($alert)=@_;
	my $sport=$alert->{'sport'};
	my $dport=$alert->{'dport'};
	my $src=$alert->{'src'};
	my $dest=$alert->{'dest'};
	my $proto=$alert->{'proto'};
	my ($ip,$port1,$port2);

	my $srcishome= &in_homenet($src);
	my $destishome= &in_homenet($dest);

	if ($destishome && !$srcishome) {
		$ip= $src; $port1= $sport; $port2= $dport;
	} elsif ($srcishome && !$destishome) {
		$ip= $dest; $port1= $dport; $port2= $sport;
	} elsif ($sport >= $dport) {
		$ip= $src; $port1= $sport; $port2= $dport;
	} else {
		$ip= $dest; $port1= $dport; $port2= $sport;
	}
   	
   	# win32 version of snort uses different file name; contrib by silverdragon
	if ($proto eq 'ICMP') {
		my $ICMP_type= $alert->{'ICMP_type'};
		print STDOUT "Warning: \"$ICMP_type\" text not found in \%ICMP_text_to_filename table\n" unless defined($ICMP_text_to_filename{$ICMP_type});
		return $log_base.$ip.'/'.$ICMP_text_to_filename{$ICMP_type}.$logfileext;
	} else {
		my $prototext= ($proto eq 'UDP')?'UDP':'TCP';
		return $log_base.$ip."/".$prototext.$logfileprototerm.$port1."-".$port2.$logfileext;
	} 

}

##############################################################################

# print the line of a table to look up an IP address in whois databases;
# returns the number of columns used
sub print_ip_lookup
{
  my($ip) = @_;
 
  my $host = gethostbyaddr(inet_aton($ip), AF_INET) if defined $dns_option;
  
  my $target= &target('lookup');
  print "<tr><td rowspan=2>$ip</td>\n";
  print "<td rowspan=2>($host)</td>\n" if defined $host;
  print "<td>Whois lookup at:</td>\n";
  print "<td><a href=\"http://www.arin.net/cgi-bin/whois.pl".
				"?queryinput=$ip&B1=Submit+Query\"$target>ARIN</a></td>\n";
  print "<td><a href=\"http://www.ripe.net/cgi-bin/whois".
				"?query=$ip&.=Submit+Query\"$target>RIPE</a></td>\n";
  print "<td><a href=\"http://www.apnic.net/apnic-bin/whois.pl".
				"?search=$ip\"$target>APNIC</a></td>\n";
  print "<td><a href=\"http://www.geektools.com/cgi-bin/proxy.cgi".
  				"?query=$ip&targetnic=auto\"$target>Geektools</a></td>\n";
  				# thanks to Dr. Paul Mitchell for this add
  print "</tr>\n<tr>\n";
  print "<td>DNS lookup at:</td>\n";
  my(@ipparts)= split(/\./,$ip);
  print "<td><a href=\"http://www.amnesi.com/hostinfo/ipinfo.jhtml?Search=Lookup+Name&wholeIp=$ip&ip1=".$ipparts[0]."&ip2=".$ipparts[1]."&ip3=".$ipparts[2]."&ip4=".$ipparts[3]."\"$target>Amenesi</a></td>\n";
  # GET method doesn't work: print "<td><a href=\"http://www.infiltration.net/cgi-bin/dnsptr.cgi?ipaddr=$ip\"$target>Infiltration</a></td>\n";
  print "<td><a href=\"http://andrew.triumf.ca/cgi-bin/gethost?$ip\"$target>TRIUMF</a></td>\n";
  print "<td><a href=\"http://riherds.com/cgi-bin/cgiwrap/riherds/rns?ip=$ip\"$target>Riherds</a></td>\n";
  print "<td><a href=\"http://wwwnet.princeton.edu/cgi-bin/dnslookup.pl?advanced_output=on&target=$ip\"$target>Princeton</a></td>\n";
  print "</tr>\n";
  return 6 + (defined $host ? 1: 0); # (number of cols used)

#				 alias jpnic  "/usr/ucb/whois -h whois.nic.ad.jp"
#				 alias aunic  "/usr/ucb/whois -h whois.aunic.net"
#				 alias milnic "/usr/ucb/whois -h whois.nic.mil"
#				 alias govnic "/usr/ucb/whois -h whois.nic.gov"
#				 alias krnic  "/usr/ucb/whois -h whois.krnic.net"
}

##############################################################################

# prints out a standard SnortSnarf HTML header
sub print_page_head
{
  my($page_title,$page_type,$page_h2,$base) = @_;
  
  print "<html>\n<head>\n";
  print "<title>$page_title</title>\n";
  print "<META HTTP-EQUIV=\"refresh\" CONTENT=\"$refreshsecs;\">" if defined($refreshsecs);
  print "</head>\n<body BGCOLOR=\"$bgcol\">\n";
  print "<table><tr>\n";
  print "<td width=130><A HREF=\"http://www.silicondefense.com/\"><IMG BORDER=0 width=123 height=72 SRC=\"$base$logo_url\" ALT=\"[Silicon Defense logo]\"></A></td>\n";
  print "<td><CENTER><h1>SnortSnarf $page_type</h1><h2>$page_h2</h2>$script $version</CENTER></td></tr></table><hr>\n";
  print "\n\n";
}


##############################################################################

# prints out a standard SnortSnarf HTML footer
sub print_page_foot
{
 print "<hr>\n<CENTER>$script brought to you courtesy of <A HREF=\"http://www.silicondefense.com/\">Silicon Defense</A><BR>\n";
 print "Authors: $author<BR>\n";
 print "See also the <a href=\"http://www.snort.org/\">".
						"Snort Page</a> by Marty Roesch<BR>\n";
 print "Page generated at ".localtime(time())."</CENTER></html>";
}

##############################################################################

# sort $a and $b (parsed alerts) by time, returning a <=>/cmp type result
sub sort_by_time
{  
  my(@pieces1) = ($a->{'month'},$a->{'date'},split(':',$a->{'time'}));
  my(@pieces2) = ($b->{'month'},$b->{'date'},split(':',$b->{'time'}));
  
  foreach (0..$#pieces1)
   {
	return -1 if $pieces1[$_] < $pieces2[$_];
	return 1 if $pieces1[$_] > $pieces2[$_];
   }
 return 0;
}

##############################################################################

# find the earlier of two alerts and return it
sub earliest
{
 my($alert1,$alert2) = @_;
 
 return $alert1 unless defined $alert2;
 return $alert2 unless defined $alert1;
 
 my(@pieces1) = ($alert1->{'month'},$alert1->{'date'},split(':',$alert1->{'time'}));
 my(@pieces2) = ($alert2->{'month'},$alert2->{'date'},split(':',$alert2->{'time'}));
 
 foreach (0..$#pieces1)
  {
   return $alert1 if $pieces1[$_] < $pieces2[$_];
   return $alert2 if $pieces1[$_] > $pieces2[$_];
  }
 return $alert1;
}

##############################################################################

# find the later of two alerts and return it
sub latest
{
 my($alert1,$alert2) = @_;
 
 return $alert1 unless defined $alert2;
 return $alert2 unless defined $alert1;
 
 my(@pieces1) = ($alert1->{'month'},$alert1->{'date'},split(':',$alert1->{'time'}));
 my(@pieces2) = ($alert2->{'month'},$alert2->{'date'},split(':',$alert2->{'time'}));
 
 foreach (0..$#pieces1)
  {
   return $alert1 if $pieces1[$_] > $pieces2[$_];
   return $alert2 if $pieces1[$_] < $pieces2[$_];  
  }
 return $alert2;
}

##############################################################################

# return HTML to pretty-print the alert's time
sub pretty_time
{
  my($alert) = @_;
  my(@bits) = split(/\./,$alert->{'time'});
  return "<b>$bits[0]</b>".
						(defined($bits[1])&&($bits[1] ne '')?".$bits[1]":'').
						" <i>on $alert->{'month'}/$alert->{'date'}</i>";
}

##############################################################################

# make a bit mask/array out of a list of 4 byte-size ints, where the first
#  element is the most significant
# really this should adjust for big endian/little endian, but this is fine
#  for our purposes
sub bytenums2bits {
	return ($_[0] << 24) | ($_[1] << 16) | ($_[2] << 8) | $_[3];
}

sub in_homenet {
	return 0 unless defined($homenetaddr);
	my $ipaddr= &bytenums2bits(split('\.',$_[0]));
	return ($ipaddr & $homenetmask) == $homenetaddr;
}

##############################################################################

sub open_page {
	my ($basedir,$file)= @_;
	my $fh= $fhnonce++;
	my $path=$basedir;
	my $filepath= $file;
	$filepath =~ s:[^/]+$::;
	foreach (split(/\//,$filepath)) {
		$path.= "/$_";
		if (-e $path) {
			unless (-d $path) {
				die "$path exists but is not a directory (trying to create $file)";
			}
		} else {
			mkdir($path,0755) || die "could not make directory $path to store $file in";
		}
	}
	open($fh,">$basedir$dirsep$file") || 
								die("Couldn't create page $file in $basedir\n");
	return $fh;
}

sub siglist_page {
	return $index_page;
}
sub siglist_base {
	return '';
}
sub sig_page {
	my($signum)= shift;
	return "sig/sig$signum.$html";
}
sub sig_base {
	return '../';
}
sub host_page {
	my($ip,$dir,$sub)= @_;
	my(@ippcs)= split(/\./,$ip);
	my $ipdir= join('/',@ippcs[0..2]);
	my $suffix= defined($sub) ? "-$sub" : '';
	return "$ipdir/$dir$ip$suffix.$html";
}
sub host_base {
	return '../../../';
}
sub anomindex_page {
	return $index_page;
}
sub anomindex_base {
	return '';
}
sub anomall_page {
	my($sortby)= @_;
	return "allby$sortby.$html";
}
sub anomall_base {
	return '';
}
sub anomsrcs_page {
	return "srcips.$html";
}
sub anomsrcs_base {
	return '';
}
sub anomdests_page {
	return "destips.$html";
}
sub anomdests_base {
	return '';
}



##############################################################################

sub write_logo_file {
	my $file= shift;
	return if -e $file;
	open(LOGO,">$file") || die "could not open logo file $file for write";
	print LOGO unpack("u",'M1TE&.#EA>P!(`/<`,?___^_U^<S__]_K],;U][_O[\_D\\\\_<Y[SDZ;_>\KC>
MY;_9[+;:X;_5YJ_6\+_-V[#1VJ7.SI_)Y:_"U9;+[9_%WZ"^U:2\R8G$ZIF_
MOZ&WQ7^^YI*WMY"SSINNOG^TV&JUY7^MSHJLK)6DMG^EPY*>L82EI7^?O8V6
MJV"CT7Z>GF^9O(F/I8>,HV^0LG62DC^=VH*#G%"5Q7&-C4"5SG]_?R^7VGQZ
ME4",PFB"@GEUD4^#K2"-U&)[>T"`L\'1MBC"#O1&)UEQS<V]D@P"#UT!NFC!S
MIU5J:FI;?`!]SF=6>$!DCTYB8C-FF611=`!UP@!RO@!ONF%-<4M;73!=C`!L
MME]);5Y(;5M#:0!DJA!;E0!AIC]/3UD_90!>H@!9FE4Y82!,?@!3D@!3BE`Q
M6C-`0P!+B`!)A`!$@"XZ.DHF40!#>S,S,R`T1P`\:0`Y<P`W6@`S9D$712`J
M,``O6B`H*``J3@@G0CH+/!HA(1,8&``<,P\/#P`2)`<)"P```/X!`@``````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````"\'Y!`$```(`+`````![`$@```C_``4(\',@@1YD_
M?\IPR3!00(0<$"&**$!0142(#!L*+"#B8HX(&AN*$%*&#T(^"G.("+E1!9<Y
M".>4R<&`I<V;.!M&X)-\'188,7/[DT%C`I%"6.4Y2O"D"X9^E(47D^9,GQ\\\\,
M+Y@X93F##Q\A5X48+5.391F0-AD(B3A#Q<VB?\H*Y.)6X\$_&35F0%@FYUZ$
M-I/^><&R`,R06N=`W:CU#Q^T=A_C5(%P<<@7"?7F\'7AW\\\\"_?7\'^_8,4(6&;
M135JS6-Y8&/)=AU##EF`:LZD?"Q[%M"9)>B<#+9JC,`W9VB!3?_4+3PU<VS9
M-YW?%,PD)V^$NP7\SAD3,<+E-C<W_\])&>\'LZR?/#Y1NDSC"ZCA[A]R.LWC#
MV@CE6G=_\'+73H0V5,0=,5+4F`\'LV!<67@0+)IY=]]2$XFG4-888@2P3VQQM9
M1BD6TH6T$4B5>@UBYQN$T2%H(6D4"B28ABQ-@>*&#G5H&8@A.@7;<PQY1A]+
M?!R(H&`LMG@7C"&]J%$>H>T4TV(XTJ9@@2\'U5D!_/P[\'T(SE%4FADK?-*)V3
M?W@H4)0L-398E28*$.1G,S8$X(RC96?3BM;)>%1#[(UF)IHL"1&G?%C&R=F9
M%QHE1(L..64G9^9IA&"7\'C*9DWI3Y<$F7@(M"B>.AB$*HV!\Z)<3@5-P9YND
M;S;4)1<TUO\74E`PRO<"9%D*4),(&<4)%YK@)<>I3>6M)*F7`W7)1!E(\CGK
M\'YX&V"96GVJJT:)"4#1A2&0RP>`<JCW)W!_P\'4NL4X"N.1!QI6Z:40\'6"B#L
M;!&`:^VV`RU%YAR>%<`$N.%2U2],Y9I[+D*P1CB%3V+MF*]3T;8JJ%,R,9L0
M`]9V.5NY_CJ5QQ00E40N2R]TF(,**HB5A[$L5793>0#>E,&RS);Q`D6\#M21
M1VZ5D4$!\'@6=0<H"!/V1=OVF7#.S7-S\,LW,"L\'RB>J>&S.CA^:T%H5,G&83
MP%B\'75C0)`J$LM@$17LGV#CQ\>@+IJ(M]]QBTR3:4Y?B71C;=/?_[7=.5^)4
MV]0DMXK8HW\G_K<(!8?$1^-5-IL!WXI7KG@9X-FEQPQ3+,WLR<&IW=`<F5=.
M@`(09,#!ZAQD$`$#!%A.(0,."Q3!%\'Z@BU`>S2&D1QF$"_!"O*AAS0`\'G)?!
MNQY^-)^[4W[P5,84+W`0N^PA"6&F=G=1A>X#``!P`$)I&%55645Y79CH+"%@
MPA3+ZR[__\']$SP<;0G"`?;Z/ZWH7\'T<PC%-J$+[PL>`/L,K!@!`R!08P(3<W
MB0#[!D*`",Q`>7J@GP8WZ!@_<"%XB:-,4/A@E2`%QRD7*"``+@"8VN@J!\WI
M0003%A($J*`,S..@#CG(!ST(`0)84P$(_^]CDAQHJR^C^4,:5#BQFN0A+R*H
M@Q]`&($RM`8H>C#*#NGGE2WZ(0],*!L!@*B3JPUG#FF0BT&THSLA#&``-1".
MS_2BD,6H(`_GN6,&MSB_C]6@!![0DPGNR(4RU"&+6G3*[]23%"%<SW9GX18?
MU,<]`>!\'@];B%TM4,`>T4(9E!##!(?FHR#2@*P\!"%\`,B60)PIDC"J8PARZ
MJ$@NZ$<%)JD#O>9PM0C@D5O64M/\YL2@`@BA+7LB@`C2L$<^ZJ$.[1(F\'VI0
M@^8,Q86%R8">GK>Y9,UA#\V;`E0,TTE+SB%N`I$,[>C\'&GE13B\#LAT7FKG#
M+_8`).I,Y"DIPO_)MZ0A#SWHP_.4"!).QF$/ONL!`C9R$`4RJ&CP(=,IT4(Z
MZQ3@"!#@PD#K60<3/%(($=5GF=#2D_9PH0YT>$`<XF`\'@=;O9\'-8J1V<EX<?
M"H0):7BH)=W&F.9492G#HQ`"A)##+:)$/45Y%Q-\:D2!!)4E$4A#\'>S@@@,4
M80DKC0,=^L`&%>QAI6&@`Q]RYX<Z\'"$"/[&."&JG$5R6;2`0P-U&>3@%,K)D
MK6\5@%LWF89O+F$";W`!5K-J!R:HH`]O>,,)$HM0\'<WA!8]\V1]F,#:>,@6\'
M<Z6K)E_&!\K2)@=L-68=N\'"\'Q!X@`"Y(K&H+>]C$+C:Q<=AH\QZK`)S_5+$J
M&6``6H7`)!(1(`.\I><.^T`\'.Z0!AG-LSQQPJUO@\'G4^>9!J:=]`A24\8`E+
M4.T;6(O8-P!6NW=H\'KJ\DH<CB&"A=_T77\"B$0(P0`138(-PATL\'U=(A#UPX
M0D(>Q;@,L5<C##@(3NV@W3<\H,#;-2QB-?"`\':16NWV@GQ[R,`<NG"P#$(BL
M1A`0`1/D0)9YF*\._?!5*B`XO!>DR@S0F1,1\'`2,>:AO@5^K7=:VH0@=`,`$
MBE"$-?C8QVZ(\`;U\$S>I>\'(2.9=B$7LQ3VH]@0_CG(<[C"3N_#R9X(;R50`
M6`8G1_G+7V;M\'M9P`@!`&<P^?D-C2<EF#1(W_\I-Z,`.>@SF.,CDPUIDTEHB
M0I+>/4Y[<$"SH-=`!\/NH0D[.$`3FH"&1COZT6B8:9LG_8<^V`\'2:#!#F1_`
M:$PW>LIZ.$(.CM"[^?&!"ST00AM*6P1/NQH-A?9JHQ_P:DR[`9R4UN&;7UV$
M`W2@UHZVPQP\*(07>&0M/?C7\'`B,AB:<H`FM!G:C8[V\',YSA`=;.MK:WK>WM
MNC37"+&T&]#`[7*3H-SH/@.A[Q!B/Q`Y>G.X`QW6D.TBD$!\.TCWMJEM[7/K
M^]]OH,,=!"I>A#@/>G[H@Z7K^V]SZUL+__:QOGMM@89G&PZ&-H,9?*#QCGO\
MXR`/N<;7X(:2PZ\'D;O_PL<A7\'G(2L\'P%\'&>YS+6P`RTT0>8=Q[@*VB"&GOO\
MYT`/NM"\'3O2B&YT$13\`THUN=!\P\'>B&UH(/2$#UJEL=YD.7NM6WW@`C^-P\'
M*]AZU3O0@9YK7>Q4;T`3Q.!T,1@A`%H`^@1VT\'.P5WT\':_^Y%DAP@/`=``).
M=[H17(!V$EA@Z6*(@Z\'-7L`&4+T##2A@`"P0]Z!K(97B:X#F!P"`MIN=\YG7
M?.01+P8M%/``HN=\W>L.@`8`W?,]C[P%@O[V`<#<!Q```-+S+H8<`R``HA^]
MS]U@Z"\8/_*=-[[R5P#Z`.Q`^=!\'O@^@#P\'H\'S]\TU>^#RQ@_0)FW_@\'L+[_
M#\+\'?>A__PLA`$`(K*\%SAL!^A98/_33WWKH[T#^7R#^#.Z@?.E;_PM:8`\'>
M]W_^IWPR0(#89WT-T\'T):(#B5T#E]W_HIW[6)P._9WU&$(\'&1W\+&\'W*IW_\
M=WW))X&Y]WM;H(`-*($B>\'X2.(`J^`7C!X$JF\'[XMX\'A<X+F9WT<^(+Z9P=>
M\(/(AP,_.(1#F`68%P)$Z`5!F(1,N(1,2(2@)X1/^(,X```54$`5\(0TF(0I
M$#Y9.(4_R(%@N`9,,`-VL`5HN`#A@P-HV(9NZ\'L#X(9;H(8`P(9R*(=T:(=W
MV(9YN(=M6(4XT(5>>(?IIX=H:`18F`5^N`46"``+_["(9&B&63")>3B)EGB)
M0%!`0\'")E6B),G")E+B&F`B*6=")D_B)EPB(67"%X=,!H-B%.$"*O@<`!K")
MI)@%5>B(H(B*61")=F")IGB+!12+P!@^!K``R#@`"T"*=\'B,R*B&S&B,SZB,
MH*B*JUA`*9"*=4B*56``DF>+U:A*SZB&Q-B+96@\'59".>9B.[-B.53",[4B\'
M$O`!](B,[E@%\DB/\'R`!`\'"/^5B/$N".5?@![,B/X9,"[`B(]YB.WJA*0\'"/
MN3@`^O@!`?"0Z8@&Y\B.=$@#"\F.\*B1:]B.`^"/(<F.!D"2==B._=B.`]F.
M!@D`"%D%"MF1+QD`4""0X?^S`.U(D.QX!N<8!4"YD4`YE$09!054E$(YE"E0
ME%&0E$!)`TSIE%\'P`45)`P!`E43YDDMIE5#)E$I90%@YE%;IB$39E4!Y!E,P
M`W0`!6R9`.%#`VP9EW*9B0#@`\'()!6X)`\'!YEW>9EWO)EW\'IEX`9EU;Y`7RI
ME5PYF\'$)`L9XEV.9`(HY!FD)!T]0F7Y9F9B9F8*8`IGY!)?9F9WYF:")F:(Y
MFC```!LPFB_)CS30F38`FJ#7F:<)``DPFI4IF3-`F9;YEK;Y!)Q7FZ$9/C#0
MF4$0G``PG+WIF\+9F3R0F1N`FK;YDL?9F0$`FL\Y`)WYG+0)FJWY!+@)!TD0
MGJ#_!P/A69[ER8\!$`3F&9YY29[F*0\'KF9<@L)[T64#N69X)8)[/N0\'T>9[V
MN9X`0)_\")_Z&3[Y:9XPX)[?69X%-)_KR0,.0(L\T)\-R9_E:0,&$)_A8Z\']
M&9X-:IXV,``%2J`=:I#WZ:$DF@0V$``!,*$%"@`#H)[EZ0`V$)YBD)9N$)XV
MH$(.L`$^N@%N.0`<NIY!H$(#D`!(6I?F6:3AXP`=&IZS:8Q(ZHU#&J$Q^J1)
M0`\'\'":"T"0(P0`$L6J,:JDI(F@"I5)XW.@-A$`0@@`%NZJ8.4*8.@`$P$`1V
M>J=WVJ9ONJ=N6J=VJJ=[N@%XFJ=\RJ=^"@.%.JB#2@$V8C"H7UJF";`!/#"H
M-E"H@7JG33"91+"IG-JIGOJIH!JJHCJJI%JJIGJJI_H%FHJJK-JJKOJJL&JJ
LJIJ;L5JKMGJKN-JIL[H\'<-"KOOJKP!JLPCJLQ%JLQGJLR)JLR7H\'4Q`0`#L`
');
	close LOGO;
}
	

##############################################################################

# only used in debugging, print full details of a parsed alert
sub debug_print_alert {
	foreach $key (keys %{$alert}) {
		print STDOUT "	$key: ",$alert->{$key},"\n";
	}
	print STDOUT "\n";
}

