#!/usr/bin/perl

# Snortsnarf, a utility to convert snort log files to HTML pages
# Authors: Stuart Staniford, Silicon Defense (stuart@SiliconDefense.com)
#          James Hoagland, Silicon Defense (hoagland@SiliconDefense.com)
# copyright (c) 2000 by Silicon Defense (http://www.silicondefense.com/)

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.  

# Snortsnarf description:
#
# Code to parse files of snort alerts and portscan logs, and produce
# HTML output intended to allow an intrusion detection analyst to
# engage in diagnostic inspection and tracking down problems.  
# The model is that one is using a cron job or similar to
# produce a daily/hourly/whatever file of snort alerts.	 This script
# can be run on each such file to produce a convenient HTML breakout
# of all the alerts.
#
# The idea is that the analyst can click around the alerts instead
# of wearily grepping and awking through them.

# Please send complaints, kudos, and especially improvements and bugfixes to
# hoagland@SiliconDefense.com. This is a quick hack with features added and
# may only be worth what you paid for it.  It is still under active
# development and may change at any time.

# this file (snortsnarf.pl) is part of Snortsnarf v062000.1

##############################################################################

# Usage:

# snortsnarf.pl <options> <file1 file2 ...>

# Any filenames with the string 'scan' in the name are assumed to be
# portscan files rather than alert files.  Similarly, filenames containing
# the string 'syslog' are assumed to be syslog files.

# The script will produce a directory snfout.file1 (by default) full of
# a large number of html files.	 These are placed under the current
# directory.  It also produces a file index.html in that directory.	 
# This is a good place to start with a browser.

# Options include:

# -d directory			directory is the path to the directory the HTML pages will
#							be generated in, overriding the default.

# -dns					This will cause the script to lookup the DNS name 
#								of each IP it writes a page for.  On a large alert 
#								file, this will take a very long time and will hammer 
#								your DNS server.

# -ldir URL				URL is the URL of a base directory in which the 
#								log files usually found in /var/log/snort are living,
#								With this option, snortsnarf will generate lots of 
#								links into those files based at that URL.  Eg: with
#								-ldir "http://host.name.here/logs" we get links like
#								http://host.name.here/logs/10.0.0.1/UDP-137-137
#								Note that the logs themselves aren't parsed.

# -homenet network-ip	network-ip is the network IP address for your home network,
#								as told to snort.  The 1-3 zeros that are the last
#								bytes of the address may be omited.	 If this
#								argument is not provided, the home network is
#								assumed to be 0.0.0.0 (no home).  This is currently
#								used only with the -ldir option.

# -color=<option>		option is 'yes' or 'no' or 'rotate'.  The style of coloring
#								is given by the option if any.  'rotate' is the only
#								style currently available.  This changes the background
#								color between an alert/portscan report and the next one
#								iff the next one has a different message (signature)
#								source host, or destination host.  If 'yes' or this not
#								given on the command line, the the default ('rotate' in
#								this version) is used. 

# -split=<threshold>	To speed up display of pages containing lots of alerts,
#							<threshold> is a maximum number of alerts to display on a
#							page unless explicitly requested.  Instead the alerts are
#							broken into several pages.  Default is 100.  Can be set to
#							0 to never split the list alerts being displayed.

# -db path				path is the full path to the annotation database (an XML file),
#							from the point of view of the server hosting the CGI scripts
#							that access it, or the empty string to not use an annotation
#							database.  The default is to not use it.

# -cgidir URL 			URL is the location of the cgi-bin directory that CGI scripts
#							than go along with the program are stored.  This may be
#							relative to the pages that are generated (so that when a
#							link appears on a page with URL as a prefix, it will be
#							valid).  "/cgi-bin" is the default.

# -sisr configfile		Generate links with SnortSnarf Incident Storage and Reporting 
#							(SISR).  The argument is the full path to the SISR
#							configuration file to use.  This file is not parsed by
#							snortsnarf.pl.

# -nmapurl URL			URL is the URL of a base URL directory in which the html
#							output of running nmap2html on nmap output is living. 
#							With this option, snortsnarf will generate links to that
#							output for IP addresses.  If the path to that directory
#							is given with the -nmapdir option, then only those pages
#							that actually exist (at the time snortsnarf is run) are
#							linked to.

# -nmapdir dir			dir is the directory on the file system in which nmap2html
#							output is stored.  When the -nmapurl option is given,
#							this is used to verify that a nmaplog.pl page actually
#							exists before linking to it. 

# -rulesfile file		file is the base rule file (e.g., snort.lib) that snort was
#							run with.  If this option is given, the rulesin that
#							file (and included files) generating a signature  are
#							included in the page for that signatures alerts.  Note
#							that the processing of the rule files are pretty rough.
#							E.g., variables are ignored. 

# -rulesdir dir			dir is a directry to use as the base path for rules files
#							rather than the one given or the paths listed in include
#							directives.  (Useful if the files have been relocated.)


# Snortsnarf is believed to work with Perl 5 on the following OS's
# OpenBSD 2.6
# RedHat Linux 6.1 and 6.2
# Windows NT 4.0 (NTFS only) (tweak the $os parameter below.  Wildcards won't).

# It probably works with most other Unixen/Linuxen, and is certain not to 
# work on the FAT filesystem.

# Believed to work with alert files generated with these snort command lines:
# snort -D -c oursnort.lib -o -d 
# snort -i eth0 -c oursnort.lib -o -e  -d

# This script is pretty memory intensive while it is running.  A previous
# version of the script was known to handle 5MB of alerts on a machine with
# 64MB of memory (no virtual memory).  The script should be studied to figure
# out why it uses more memory than expected.

# HTML looks marginally ok with Netscape 4.7 and Explorer 4.72.	 
# Other browsers not tested.

# Tested on alert files up to about 1MB.  
# HTML takes up 5 times the space of the original alert file.  
# .tar.gz of the dir is about half the size of the alert file.
# YMMV!

##############################################################################

# Credits, etc

# Initial alert parsing code borrowed from Joey McAlerney, Silicon 
# Defense.
# A couple of ideas were stolen from Snort2html by Dan Swan.
# Thanks to SANS GIAC and Donald McLachlan for a paper with ideas 
# on how to look up IP addresses.

# Huge thanks to DARPA for supporting us for part of the time developing
# this code.  Major thanks to Paul Arabelo for being our test customer
# while we learned to do operational intrusion detection instead of being
# only a researcher.

# Shouts to Marty Roesch and Patrick Mullen and the rest of the snort
# community for developing snort and the snort portscan preprocessor.  

# Kudos to Steve Northcutt who has taught us all what an intrusion
# detection analyst should be.

# Version control info: $Id: snortsnarf.pl,v 1.18 2000/06/21 03:45:43 jim Exp $

$script = "<a href=\"http://www.silicondefense.com/snortsnarf/\">".
						"Snortsnarf</a>";
$version = "v062000.1";
$author_email = "hoagland\@SiliconDefense.com";
$author = "<a href=\"mailto:hoagland\@SiliconDefense.com\">Jim Hoagland</a> and <a href=\"mailto:stuart\@SiliconDefense.com\">Stuart Staniford</a>";

use Socket;
#use Sys::Hostname;
use Cwd;
require "snort_alert_parse.pl";
require "web_utils.pl";

##############################################################################

# portability stuff - toggle for Unix/Windows.

$os = 'unix';  # Either 'windows' or 'unix'
if($os eq 'windows')
{
 $dirsep = "\\";		
 #$root = "d:\\";		
 $root = "e:\\";
}
elsif($os eq 'unix')
{
 $dirsep = "\/";		# Unix
 $root = "\/";			# Unix
}

$html = 'html';			# usually html or htm

# Various global variables

$alert_file = $root."var".$dirsep."log".$dirsep."snort.alert";
$sig_page = "index.$html";
$main_page = "index.$html";

%monthnum = ('Jan' => 1,
			 'Feb' => 2,
			 'Mar' => 3,
			 'Apr' => 4,
			 'May' => 5,
			 'Jun' => 6,
			 'Jul' => 7,
			 'Aug' => 8,
			 'Sep' => 9,
			 'Oct' => 10,
			 'Nov' => 11,
			 'Dec' => 12);

%ICMP_text_to_filename= ( # maps the text for ICMP messages found in an alert message to its component in a file name for an snort log of the connection
		'ECHO REPLY' => 'ICMP_ECHO_REPLY',
		'DESTINATION UNREACHABLE: NET UNREACHABLE' => 'ICMP_NET_UNRCH', 
		'DESTINATION UNREACHABLE: HOST UNREACHABLE' => 'ICMP_HST_UNRCH', 
		'DESTINATION UNREACHABLE: PROTOCOL UNREACHABLE' => 'ICMP_PROTO_UNRCH', 
		'DESTINATION UNREACHABLE: PORT UNREACHABLE' => 'ICMP_PORT_UNRCH', 
		'DESTINATION UNREACHABLE: FRAGMENTATION NEEDED' => 'ICMP_UNRCH_FRAG_NEEDED', 
		'DESTINATION UNREACHABLE: SOURCE ROUTE FAILED' => 'ICMP_UNRCH_SOURCE_ROUTE_FAILED', 
		'DESTINATION UNREACHABLE: NET UNKNOWN' => 'ICMP_UNRCH_NETWORK_UNKNOWN', 
		'DESTINATION UNREACHABLE: HOST UNKNOWN' => 'ICMP_UNRCH_HOST_UNKNOWN', 
		'DESTINATION UNREACHABLE: HOST ISOLATED' => 'ICMP_UNRCH_HOST_ISOLATED', 
		'DESTINATION UNREACHABLE: NET ANO' => 'ICMP_UNRCH_NET_ANO',
		'DESTINATION UNREACHABLE: HOST ANO' => 'ICMP_UNRCH_HOST_ANO',
		'DESTINATION UNREACHABLE: NET UNREACHABLE TOS' => 'ICMP_UNRCH_NET_UNR_TOS', 
		'DESTINATION UNREACHABLE: HOST UNREACHABLE TOS' => 'ICMP_UNRCH_HOST_UNR_TOS', 
		'DESTINATION UNREACHABLE: PACKET FILTERED' => 'ICMP_UNRCH_PACKET_FILT', 
		'DESTINATION UNREACHABLE: PREC VIOLATION' => 'ICMP_UNRCH_PREC_VIOL', 
		'DESTINATION UNREACHABLE: PREC CUTOFF' => 'ICMP_UNRCH_PREC_CUTOFF', 
		'DESTINATION UNREACHABLE: UNKNOWN' => 'ICMP_UNKNOWN', 
		'SOURCE QUENCH' => 'ICMP_SRC_QUENCH', 
		'REDIRECT' => 'ICMP_REDIRECT', 
		'ECHO' => 'ICMP_ECHO', 
		'TTL EXCEEDED' => 'ICMP_TTL_EXCEED', 
		'PARAMATER PROBLEM' => 'ICMP_PARAM_PROB', 
		'TIMESTAMP REQUEST'=> 'ICMP_TIMESTAMP', 
		'TIMESTAMP REPLY' => 'ICMP_TIMESTAMP_RPL', 
		'INFO REQUEST' => 'ICMP_INFO_REQ', 
		'INFO REPLY' => 'ICMP_INFO_RPL', 
		'ADDRESS REQUEST'=> 'ICMP_ADDR', 
		'ADDRESS REPLY' => 'ICMP_ADDR_RPL', 
		'UNKNOWN' => 'ICMP_UNKNOWN' 
);

@color= ('#E7DEBD','#E0CDD0','#D5E2CE');

##############################################################################

# Main program

&process_options();
&initialize();

foreach $current_file (@ARGV)
{
  my $file_type;
  if($current_file =~ /scan/)
   { $file_type = 'scan';}
  elsif($current_file =~ /syslog/)
   { $file_type = 'syslog';}
  elsif($current_file =~ /messages/)
   { $file_type = 'syslog';}
  else
   { $file_type = 'alert';}

  my $fh= &initialize_per_file($current_file);

  while($alert = &get_alert($fh,$file_type))
   {
	&process_alert($alert)
   }
 &close_out_file($fh);
}

&construct_sig_indexes();
&output_main_sig_page();
&output_per_sig();
&output_per_source();
&output_per_dest();


##############################################################################

sub process_options
{
  my $arg;

  $dns_option= undef;
  $log_base= '';
  $homenet= '';
  $color_opt= 'rotate';
  $cgi_dir= '/cgi-bin';
  $db_file= '';
  $split_threshold= 100;
  $nmap_dir= undef;
  $nmap_url= undef;
  $sisr_config= undef;
  $rules_file= undef;
  $rules_dir= undef;
  $notarget_option= 0;
  while($ARGV[0] =~ /^\-/)
   {
	$arg = shift @ARGV;
	if($arg eq '-dns')
	 {
		  $dns_option = 1;
	 }
	elsif($arg eq '-ldir')
	 {
		  $log_base = shift @ARGV;
		  $log_base.='/' unless $log_base =~ /\/$/;
	 }
	elsif($arg eq '-homenet')
	 {
		  $homenet = shift @ARGV;
		  $homenet=~ s/(\.0+)+$//;
		  $homenet = '' if $homenet =~ /^0+$/;
	 }
    elsif($arg =~ s/^-color//)
     {    
		if ($arg =~ /=(.*)/) {
			$color_opt = ($1 eq 'yes')?'rotate':$1;
		} else {
			$color_opt = 'rotate';
		}
	 }
    elsif($arg =~ s/^-split=//)
     {    
		$split_threshold = $arg;
	 }
	elsif($arg eq '-d')
	 {
		  $output_dir = shift @ARGV;
	 }
	elsif($arg eq '-cgidir')
	 {
		  $cgi_dir = shift @ARGV;
	 }
	elsif($arg eq '-db')
	 {
		  $db_file = shift @ARGV;
	 }
	elsif($arg eq '-nmapdir')
	 {
		  $nmap_dir = shift @ARGV;
	 }
	elsif($arg eq '-nmapurl')
	 {
		  $nmap_url = shift @ARGV;
		  $nmap_url.='/' unless $nmap_url =~ /\/$/;
	 }
	elsif($arg eq '-sisr')
	 {
		  $sisr_config = shift @ARGV;
	 }
	elsif($arg eq '-rulesfile')
	 {
		  $rules_file = shift @ARGV;
	 }
	elsif($arg eq '-rulesdir')
	 {
		  $rules_dir = shift @ARGV;
	 }
	elsif($arg eq '-onewindow')
	 {
		  $notarget_option = 1;
	 }
	else
	 {
	  print "Unknown option $arg\n";
	 }
   }
}

##############################################################################

sub initialize
{
  # Setup to use default file if no args
  if(@ARGV == 0)
   {
	@ARGV = ($alert_file);
   }
  $count = 0;
  
  $fhnonce='fh00';
  
  $cwd= getcwd();
  
  # Setup an output directory named after the first file (unless defined on command line)
  
  my $last = rindex($ARGV[0],$dirsep);
  my $substr = substr($ARGV[0],$last+1);
  $file_name[$count] = $substr;
  $output_dir = "snfout.$file_name[$count]" unless defined($output_dir);
  
  if(-e $output_dir)
   {
	die("$output_dir already exists and is not a directory\n")
	  unless -d $output_dir;
		if($os eq 'unix')
		 {
		  system("rm -rf $output_dir");
		  mkdir($output_dir,0755);
		 }
		elsif($os eq 'windows')
		 {
		  system("del $output_dir\*.*");
		 }
   }
  else
   {
	mkdir($output_dir,0755);
   }
}

##############################################################################

sub initialize_per_file
{
  my($file) = @_;

  $alert_file = $file;
  $file_fullpath[$count]= $file;
  $file_fullpath[$count]= "$cwd/$file" unless $file =~ /^\//;
  my $fh='alertfh00';
  open($fh,"$alert_file") || die("Couldn't open $file_type input file ".
														"$alert_file\n");
  my $last = rindex($alert_file,$dirsep);
  my $substr = substr($alert_file,$last+1);
  $file_name[$count] = $substr;
  return $fh;
}

##############################################################################

sub close_out_file
{
  close($_[0]);
  $count++;
}

##############################################################################

sub get_alert
{
	my ($fh,$fileformat)= @_;
	my ($alerttext,$format)= &next_alert($fh,$fileformat);
	return undef unless defined($alerttext); # must have hit end of file
	my $alert= &parse_alert($alerttext,$format);
	$alert->{'type'}= $format;
	# we used to store source file by 'file' => $file_name[$count], but that doesn't seem to be needed anywhere
	return $alert;
}

##############################################################################

sub process_alert
{
	my($alert) = @_;

	# Global stuff
	$earliest_overall = &earliest($alert, $earliest_overall);
	$latest_overall = &latest($alert, $latest_overall);
	$alert_count++;
	
	# Per signature stuff
	$sig_count->{$alert->{'sig'}}++;
	$earliest_sig->{$alert->{'sig'}}
		= &earliest($alert, $earliest_sig->{$alert->{'sig'}});
	$latest_sig->{$alert->{'sig'}}
		= &latest($alert, $latest_sig->{$alert->{'sig'}});

	# Per source stuff
	$src_count->{$alert->{'src'}}++;
	push @{$src_list->{$alert->{'src'}}}, $alert;
	$earliest_src->{$alert->{'src'}}
		= &earliest($alert, $earliest_src->{$alert->{'src'}});
	$latest_src->{$alert->{'src'}}
		= &latest($alert, $latest_src->{$alert->{'src'}});

	# Per dest stuff
	$dest_count->{$alert->{'dest'}}++;
	push @{$dest_list->{$alert->{'dest'}}}, $alert;
	$earliest_dest->{$alert->{'dest'}}
		= &earliest($alert, $earliest_dest->{$alert->{'dest'}});
	$latest_dest->{$alert->{'dest'}}
		= &latest($alert, $latest_dest->{$alert->{'dest'}});
	
	# Per signature-source stuff
	$sig_src_count->{$alert->{'sig'}}{$alert->{'src'}}++;
	
	# Per signature-dest stuff
	$sig_dest_count->{$alert->{'sig'}}{$alert->{'dest'}}++;
	
	# Src dest stuff
	$src_dest_count->{$alert->{'src'}}{$alert->{'dest'}}++;	 
	$dest_src_count->{$alert->{'dest'}}{$alert->{'src'}}++;	 
		
	# Sig src dest stuff
	$sig_src_dest_count->{$alert->{'sig'}}{$alert->{'src'}}{$alert->{'dest'}}++;
	$sig_dest_src_count->{$alert->{'sig'}}{$alert->{'dest'}}{$alert->{'src'}}++;
}

##############################################################################

sub construct_sig_indexes
{
  my($sig,$count);
  
  foreach $sig (keys %$sig_count)
   {
		$sig_index->{$sig} = $count++;
		if($sig =~ /^IDS(\d+)/)
		 {
			# ARACHNIDS signature - can make a nice URL for these.
			$sig_url->{$sig} = "http:\/\/whitehats.com\/IDS\/$1";
			$sig_entry->{$sig} = "<a href=\"$sig_url->{$sig}\">$sig</a>";
		 }
		else
		 { 
			$sig_entry->{$sig} = $sig;
		 }
    }
}

##############################################################################

sub output_main_sig_page
{
  my($sig);
  my $page_title = "Snortsnarf: Snort signatures in $file_name[0] et al";
  
  open(PAGE,">$output_dir$dirsep$sig_page") || 
								die("Couldn't open sig page $sig_page\n");
  select(PAGE);
  &print_page_head($page_title);
  
  print "<p>$alert_count alerts processed.</p>\n";
  print "<p>Files included:<ul>\n<li>\n";
  print join("<li>\n",@file_name);
  print "\n</ul>\n";
  print "Earliest alert at ".&pretty_time($earliest_overall)."<br>\n";
  print "Latest alert at ".&pretty_time($latest_overall)."</p>\n";

  print "<TABLE BORDER CELLPADDING = 5>\n";
  print "<TR><TD>Signature (click for definition)</TD><TD>\# Alerts</TD>".
				"<TD>\# Sources</TD><TD>\# Destinations</TD><TD>Detail link</TD></TR>\n";

  foreach $sig (sort sort_by_sig_count keys %$sig_count)
   {

		print "<TR><TD>$sig_entry->{$sig}</TD><TD>$sig_count->{$sig}</TD><TD>".
				scalar(keys %{$sig_src_count->{$sig}})."</TD><TD>".
				scalar(keys %{$sig_dest_count->{$sig}}).
				"</TD><TD><a href=\"sig$sig_index->{$sig}.$html\">Summary</a></TD></TR>\n";
   }
  print "</TABLE>\n\n";
  &print_page_foot();
  close(PAGE);
}

sub sort_by_sig_count { $sig_count->{$a} <=> $sig_count->{$b};}

##############################################################################

sub output_per_sig
{
  my($sig,$sig_file,$src,$dest,$early,$late);
  my $page_title;
  
  foreach $sig (keys %$sig_count)
   {  
	# Sort out the file
	$global_sig = $sig;	 # ouch - need to communicate with sort_by_sig_src_count
	$sig_file = "sig$sig_index->{$sig}.$html";

	open(PAGE,">$output_dir$dirsep$sig_file") || 
										die("Couldn't open sig output file $sig_file\n");
	select(PAGE);

	# Print page head stuff
	$page_title = "Summary of alerts in $file_name[0] et al for signature:";
	&print_page_head($page_title);
	print "<h3>$sig_entry->{$sig}</h3>";
	print "<p>$sig_count->{$sig} alerts on this signature.</p>\n";
	print "<p>Looking in files:<ul>\n<li>\n";
	print join("<li>\n",@file_name);
	print "\n</ul>\n";
	print "Earliest such alert at ".&pretty_time($earliest_sig->{$sig})."<br>\n";
	print "Latest such alert at ".&pretty_time($latest_sig->{$sig})."</p>\n";

	print "<table border cellpadding = 3>\n";
  	print "<tr><td>$sig</td>\n";
  	print "<td><A HREF=#srcsect>",scalar(keys %{$sig_src_count->{$sig}})," sources</A></td>\n";
  	print "<td><A HREF=#destsect>",scalar(keys %{$sig_dest_count->{$sig}})," destinations</A></td></tr>\n";
	if ($db_file ne '') { #link to annotations
		print "<tr><td colspan=3 align=center><A HREF=\"",&view_ann_url('snort message',$sig),"\">View/add annotations for this signature</A></td></tr>\n";
	}
	if (defined($rules_file) && $sig !~ /^(UDP|TCP|ICMP)\s+scan$/) { # show rule file entries for signature
		my(@rules_html)= &get_rules_html_for_sig($sig,$rules_file);
		if (@rules_html) {
			print "<tr bgcolor=\"#D5E2CE\"><td colspan=3 align=center>Rules with message \"$sig\":</td></tr>\n";
			foreach (@rules_html) {
				print "<tr bgcolor=\"#D5E2CE\"><td colspan=3 align=left>$_</td></tr>\n";
			}
		}
	}
	print "</table>";

	# Print the sources section
	print "<hr><h3><A NAME=srcsect>Sources triggering this attack signature</A></h3>\n";
	print "<TABLE BORDER CELLPADDING = 5>\n";
	print "<tr><td>Source</td><td>\# Alerts (sig)</td>".
				"<td>\# Alerts (total)</td><td>\# Dsts (sig)</td>".
				"<td>\# Dsts (total))</td></tr>\n";
	foreach $src (sort sort_by_sig_src_count keys %{$sig_src_count->{$sig}})
		 {
		  $global_src = $src; # ouch ouch
		  print "<tr><td><a href=\"src$src.$html\">$src</a></td>".
				"<td>$sig_src_count->{$sig}{$src}</td>".
				"<td>$src_count->{$src}</td><td>".
				scalar(keys %{$sig_src_dest_count->{$sig}{$src}})."</td><td>".
				scalar(keys %{$src_dest_count->{$src}})."</td></tr>\n";
		 }
	print "</TABLE>\n";
		
	# Print the destinations section
		print "<hr><h3><A NAME=destsect>Destinations receiving this attack signature</A></h3>\n";
		print "<TABLE BORDER CELLPADDING = 5>\n";
		print "<tr><td>Destinations</td><td>\# Alerts (sig)</td>".
				"<td>\# Alerts (total)</td><td>\# Srcs (sig)</td>".
				"<td>\# Srcs (total))</td></tr>\n";
		foreach $dest (sort sort_by_sig_dest_count
											keys %{$sig_dest_count->{$sig}})
		 {
		  $global_dest = $dest; # ouch ouch
		  print "<tr><td><a href=\"dest$dest.$html\">$dest</a></td>".
				"<td>$sig_dest_count->{$sig}{$dest}</td>".
				"<td>$dest_count->{$dest}</td><td>".
				scalar(keys %{$sig_dest_src_count->{$sig}{$dest}})."</td><td>".
				scalar(keys %{$dest_src_count->{$dest}})."</td></tr>\n";
		 }
	print "</TABLE>\n"; 
	&print_page_foot();
	close(PAGE);
   }
}

sub sort_by_sig_src_count 
{ 
 $sig_src_count->{$global_sig}{$b} <=> 
										$sig_src_count->{$global_sig}{$a};
}

sub sort_by_sig_dest_count 
{ 
 $sig_dest_count->{$global_sig}{$b} <=> 
										$sig_dest_count->{$global_sig}{$a};
}

##############################################################################

sub output_per_source
{
   my($src,$src_file);
   
   foreach $src (keys %$src_count)
	{  
		&output_per_host($src,'src',"from $src in $file_name[0] et al",$src_list->{$src});
	}
}

##############################################################################

sub output_per_dest
{
   my($dest,$dest_file,$alert,$early,$late);
   my $page_title;
   
   foreach $dest (keys %$dest_count)
	{  
		&output_per_host($dest,'dest',"going to $dest in $file_name[0] et al",$dest_list->{$dest});
	}
}

##############################################################################

sub output_per_host {
	my($ip,$end,$al_descr,$alertsref)= @_;
	@alerts= sort sort_by_time @{$alertsref};
	my $num_alerts= @alerts;
	my $ip_file = "$end$ip.$html";
	open(PAGE,">$output_dir$dirsep$ip_file") 
		|| die("Couldn't open $ip output file $ip_file\n");
	my $prevsel= select(PAGE);
		
	if ($split_threshold == 0 || $num_alerts <=  $split_threshold) {
		&print_page_head("All $num_alerts alerts $al_descr");
		&output_per_host_header($ip,$end,$alerts[0],$alerts[$#alerts]);
		&output_alert_table(\@alerts);
	} else {  # need to split
		&print_page_head("Overview of $num_alerts alerts $al_descr");
		&output_per_host_header($ip,$end,$alerts[0],$alerts[$#alerts]);

		my $all_file = "$end$ip-all.$html";

		print "<hr>This listing contains $num_alerts alerts.  You can:\n";
		print "<UL><LI><A HREF=\"$all_file\">view the whole listing</A>\n";
		print "<LI>view a range of alerts <A HREF=#rangelist>(see table below)</A></UL>\n";

		print "<hr><A NAME=rangelist>Alert ranges (sorted by time):</A>\n";
		print "<table border cellpadding = 3><TR align=center><B><TD>alert #'s</TD><TD>first time</TD><TD>last time</TD></B></TR>\n";
		my($first,$last);
		foreach ($first=1; $first < $num_alerts; $first+=$split_threshold) {
			$last= $first+$split_threshold-1;
			$last= $last < $num_alerts ? $last : $num_alerts;
			my(@alertsub)= @alerts[($first-1)..($last-1)];
			my $early= $alertsub[0];
			my $late= $alertsub[$#alertsub];
			my $range_file = "$end$ip-$first.$html";
			print "<tr><td><A HREF=\"$range_file\">$first to $last</A></td><td>",&pretty_time($early),"</td><td>",&pretty_time($late),"</td></tr>\n";

			# create page for the range
			open(RANGE,">$output_dir$dirsep$range_file") 
				|| die("Couldn't open $ip output file $range_file\n");
			my $prevsel= select(RANGE);
			&print_page_head("Alerts $first to $last of $num_alerts $al_descr");
			&output_per_host_header($ip,$end,$early,$late);
			print "<hr>";
			my $nav= "Go to: ".
				($first == 1?'':"<A HREF=\"$end$ip-".($first-$split_threshold).".$html\">previous range</A>, ").
				(($first+$split_threshold >= $num_alerts)?'':"<A HREF=\"$end$ip-".($first+$split_threshold).".$html\">next range</A>, ").
				" <A HREF=\"$all_file\">all alerts</A>, <A HREF=\"$ip_file\">overview page</A>";
			print $nav;
			&output_alert_table(\@alertsub);
			print $nav;
			&print_page_foot();
			close(RANGE); select($prevsel);
		}		
		print "</table>\n";
		
		open(ALL,">$output_dir$dirsep$all_file") 
			|| die("Couldn't open $ip output file $all_file\n");
		my $prevsel= select(ALL);
		&print_page_head("All $num_alerts alerts $al_descr");
		&output_per_host_header($ip,$end,$alerts[0],$alerts[$#alerts]);
		my $nav= "Go to: <A HREF=\"$ip_file\">overview page</A>";
		print $nav;
		&output_alert_table(\@alerts);
		print $nav;
		&print_page_foot();
		close(ALL); select($prevsel);
	}
	
	&print_page_foot();
	close(PAGE); select($prevsel);
}

##############################################################################

sub output_per_host_header
{
	my ($ip,$end,$early,$late)=@_;
	print "<p>Looking in files:<ul>\n<li>\n";
	print join("<li>\n",@file_name);
	print "\n</ul>\n";		
	print "Earliest: ".&pretty_time($early)."<br>\n";
	print "Latest: ".&pretty_time($late)."</p>\n";
	print "<table border cellpadding = 3>\n";
	my $cols= &print_ip_lookup($ip);
	if ($end eq 'src' && exists $dest_count->{$ip}) {
		print "<tr><td colspan=$cols align=center><A HREF=\"dest$ip.$html\">See also $ip as a destination</A></td></tr>\n";
	} elsif ($end eq 'dest' && exists $src_count->{$ip}) {
		print "<tr><td colspan=$cols align=center><A HREF=\"src$ip.$html\">See also $ip as a source</A></td></tr>\n";
	}
	if ($db_file ne '') {
		print "<tr><td colspan=$cols align=center><A HREF=\"",&view_ann_url('IP',$ip),"\">View/add annotations for this IP address</A></td></tr>\n";
	}
	if (defined($nmap_url) && (!defined($nmap_dir) || -e "$nmap_dir/$ip.html")) {
		print "<tr><td colspan=$cols align=center><A HREF=\"$nmap_url$ip.html\">View nmap log page for $ip".(defined($nmap_dir)?'':' (if any)')."</A></td></tr>\n";
	}
	if (defined($sisr_config)) {
		my $encconfig= &url_encode($sisr_config);
		print "<tr bgcolor=\"#D5E2CE\"><td rowspan=3 align=center>Incident handling</td>";
		my $colwidth=$cols-1;
		print "<td colspan=$colwidth align=center><A HREF=\"$cgi_dir/sel_to_add.pl?".join('&',"configfile=$encconfig","end=$end","ip=$ip",'logs='.&url_encode(join(',',@file_fullpath))).'"',&target('sisrwin'),">Add some of these alerts to a labeled set</A></td></tr>\n";
		print "<tr bgcolor=\"#D5E2CE\"><td colspan=$colwidth align=center><A HREF=\"$cgi_dir/lsetlist.pl?configfile=$encconfig\"",&target('sisrwin'),">List stored sets</td></tr>\n";
		print "<tr bgcolor=\"#D5E2CE\"><td colspan=$colwidth align=center><A HREF=\"$cgi_dir/inclist.pl?configfile=$encconfig\"",&target('sisrwin'),">List stored incidents</td></tr>\n";
	}
	print "</table>\n";
}

sub view_ann_url {
	my($type,$key)= @_;
	return "$cgi_dir/view_annotations.pl\?".join('&','file='.&url_encode($db_file),'type='.&url_encode($type),'key='.&url_encode($key));
}

##############################################################################
sub output_alert_table {
	my($alertsref)= @_;
	print "<HR>";
	print "<table border cellpadding = 3>\n";
	for (my $i=0; $i <= $#{$alertsref}; $i++) {
		&output_table_entry($alertsref->[$i]);
	 }
	print "</table>\n";
}

sub output_table_entry { 
	my($alert)= @_;
	my  $tdopts= $color_opt eq 'rotate'?" bgcolor=".&alert_color($alert):'';
	print "<tr><td$tdopts>".&alert_as_html($alert)."</td></tr>\n";
}

##############################################################################

sub alert_color { 
	my($alert)= @_;
	return $color[$::old_color] if (defined($::old_alert) && ($::old_alert->{'sig'} eq $alert->{'sig'}) && ($::old_alert->{'src'} eq $alert->{'src'}) && ($::old_alert->{'dest'} eq $alert->{'dest'}));
	$::old_color= ++$::old_color % +@color;
	$::old_alert= $alert;
	return $color[$::old_color];
}

##############################################################################

sub alert_as_html { 
	my($alert)= @_;
	$append= '';
	$text= $alert->{'text'};
    my $sig= $alert->{'sig'};
    my $sigre= $sig;
    $sigre =~ s/([^\w ])/\\$1/g;
    $text =~ s/(\[\*\*\]\s*)($sigre)(\s*\[\*\*\])/$1<a href=\"sig$sig_index->{$sig}.$html\">$2<\/A>$3/;
	$text =~ s/(\d+\.\d+\.\d+\.\d+)(.*)->/<A HREF="src$1.html">$1<\/A>$2->/;
	$text =~ s/->(\s*)(\d+\.\d+\.\d+\.\d+)/->$1<A HREF="dest$2.html">$2<\/A>/;
	$text =~ s/[\n\r]+/<br>/g;
    if ($log_base ne '' && ($alert->{'type'} =~ /alert$/) && defined($alert->{'proto'})) {
       $append.= " <A HREF=\"".&get_alert_logpage($alert)."\">[Snort log]</A>\n";
    } 
	return "<code>$text</code>$append";
}

##############################################################################

sub target {
	return '' if $notarget_option;
	return " target=$_[0]";
}

##############################################################################

sub get_rules_html_for_sig {
	my ($sig,$rule_file)= @_;
	my(@rules_html)=();
	my $fh= $fhnonce++;
	# file names assumed to be absolute or relative to the current directory
	# but if -rulesdir was given, always use that as the dirctory (with the file name appended to make the file location)
	$rule_file =~ s/\s+$//;
	my($file)= $rule_file =~ /([^\/]+)$/; 
	if (defined($rules_dir)) {
		$rule_file= "$rules_dir/$file";
	}
	unless (open($fh,"<$rule_file")) {
		warn "could not open $rule_file to read rules from -- skipping\n";
		return;
	}
	while (<$fh>) {
		chomp;
		next if /^(\#|\s*$)/;
		if (s/^\s*include\s+//) {
			s/\s+$//;
			push(@rules_html,&get_rules_html_for_sig($sig,$_));
		} elsif (/\(.*msg\s*:\s*\"([^\"]*)\"/) {
			my $mess= $1;
			$mess =~ s/\\(.)/$1/g;
			$mess =~ s/^\s+//;
			$mess =~ s/\s+$//;
			if ($mess eq $sig) { # message in rule is same as one looked for (modulo whitespace)
				my $html= "<SMALL>$_</SMALL> (from <EM>$file</EM>)";
				push(@rules_html,$html);
			}
		}
	}
	close $fh;
	return @rules_html;
}

##############################################################################

sub get_alert_logpage 
{
	my($alert)=@_;
	my $sport=$alert->{'sport'};
	my $dport=$alert->{'dport'};
	my $src=$alert->{'src'};
	my $dest=$alert->{'dest'};
	my $proto=$alert->{'proto'};
	my ($ip,$port1,$port2);

	my $srcishome= $src =~ /^$homenet\./;
	my $destishome= $dest =~ /^$homenet\./;
	
	if ($destishome && !$srcishome) {
		$ip= $src; $port1= $sport; $port2= $dport;
	} elsif ($srcishome && !$destishome) {
		$ip= $dest; $port1= $dport; $port2= $sport;
	} elsif ($sport >= $dport) {
		$ip= $src; $port1= $sport; $port2= $dport;
	} else {
		$ip= $dest; $port1= $dport; $port2= $sport;
	}
   
	if ($proto eq 'ICMP') {
	   my $ICMP_type= $alert->{'ICMP_type'};
		print STDOUT "Warning: \"$ICMP_type\" text not found in %ICMP_text_to_filename table\n" unless defined($ICMP_text_to_filename{$ICMP_type});
		return $log_base.$ip.'/'.$ICMP_text_to_filename{$ICMP_type};
	} else {
		my $prototext= ($proto eq 'UDP')?'UDP':'TCP';
		return $log_base.$ip."/".$prototext.":".$port1."-".$port2;
	} 
}

##############################################################################

sub print_ip_lookup
{
  my($ip) = @_;
 
  my $host = gethostbyaddr(inet_aton($ip), AF_INET) if defined $dns_option;
  
  print "<tr><td>$ip</td>\n";
  print "<td>($host)</td>\n" if defined $host;
  print "<td>Lookup at:</td>\n";
  print "<td><a href=\"http://www.arin.net/cgi-bin/whois.pl".
				"?queryinput=$ip&B1=Submit+Query\"",&target('lookup'),">ARIN</a></td>\n";
  print "<td><a href=\"http://www.ripe.net/cgi-bin/whois".
				"?query=$ip&.=Submit+Query\"",&target('lookup'),">RIPE</a></td>\n";
  print "<td><a href=\"http://www.apnic.net/apnic-bin/whois.pl".
				"?search=$ip\"",&target('lookup'),">APNIC</a></td>\n";
  print "</tr>\n";
  return 5 + (defined $host ? 1: 0); # (number of cols used)

#				 alias jpnic  "/usr/ucb/whois -h whois.nic.ad.jp"
#				 alias aunic  "/usr/ucb/whois -h whois.aunic.net"
#				 alias milnic "/usr/ucb/whois -h whois.nic.mil"
#				 alias govnic "/usr/ucb/whois -h whois.nic.gov"
#				 alias krnic  "/usr/ucb/whois -h whois.krnic.net"
}

##############################################################################

sub print_page_head
{
  my($page_title) = @_;
  
  print "<html>\n<head>\n<title>$page_title</title>\n";
  print "</head>\n<body BGCOLOR=\"#E7DEBD\">\n";
  print "<h2>$page_title</h2>\n\n";
}

##############################################################################

sub print_page_foot
{
 print "<hr>\n";
 print "<i>Generated by $script $version ($author)</i><p>\n";
 print "See also the <a href=\"http://www.snort.org/\">".
						"Snort Page</a> by Marty Roesch<p>";
 print "Page generated at ".localtime(time())."<p>\n";
 print "</body></html>\n";
}

##############################################################################

sub sort_by_time
{  
  my(@pieces1) = ($a->{'month'},$a->{'date'},split(':',$a->{'time'}));
  my(@pieces2) = ($b->{'month'},$b->{'date'},split(':',$b->{'time'}));
  
  foreach (0..$#pieces1)
   {
	return -1 if $pieces1[$_] < $pieces2[$_];
	return 1 if $pieces1[$_] > $pieces2[$_];
   }
 return 0;
}

##############################################################################

sub earliest
{
 my($alert1,$alert2) = @_;
 
 return $alert1 unless defined $alert2;
 return $alert2 unless defined $alert1;
 
 my(@pieces1) = ($alert1->{'month'},$alert1->{'date'},split(':',$alert1->{'time'}));
 my(@pieces2) = ($alert2->{'month'},$alert2->{'date'},split(':',$alert2->{'time'}));
 
 foreach (0..$#pieces1)
  {
   return $alert1 if $pieces1[$_] < $pieces2[$_];
   return $alert2 if $pieces1[$_] > $pieces2[$_];
  }
 return $alert1;
}

##############################################################################

sub latest
{
 my($alert1,$alert2) = @_;
 
 return $alert1 unless defined $alert2;
 return $alert2 unless defined $alert1;
 
 my(@pieces1) = ($alert1->{'month'},$alert1->{'date'},split(':',$alert1->{'time'}));
 my(@pieces2) = ($alert2->{'month'},$alert2->{'date'},split(':',$alert2->{'time'}));
 
 foreach (0..$#pieces1)
  {
   return $alert1 if $pieces1[$_] > $pieces2[$_];
   return $alert2 if $pieces1[$_] < $pieces2[$_];  
  }
 return $alert2;
}

##############################################################################

sub pretty_time
{
  my($alert) = @_;
  my(@bits) = split(/\./,$alert->{'time'});
  return "<b>$bits[0]</b>".
						(defined($bits[1])&&($bits[1] ne '')?".$bits[1]":'').
						" <i>on $alert->{'month'}/$alert->{'date'}</i>";
}

##############################################################################

sub debug_print_alert {
	foreach $key (keys %{$alert}) {
		print STDOUT "	$key: ",$alert->{$key},"\n";
	}
	print STDOUT "\n";
}
