#!/usr/bin/perl

# snort_alert_parse.pl, distributed as part of Snortsnarf v062000.1
# Author: James Hoagland, Silicon Defense (hoagland@SiliconDefense.com)
# copyright (c) 2000 by Silicon Defense (http://www.silicondefense.com/)
# Released under GNU General Public License, see the COPYING file included
# with the distribution or http://www.silicondefense.com/snortsnarf/ for
# details.

# snort_alert_parse.pl is a file that contains helpful functions for
#   parsing snort alert files and decomposing alerts.

# Portions of this file are based on code in snortsnarf.pl by Stuart
#   Staniford that was based on code wrttien by Joe McAlerney.

# Please send complaints, kudos, and especially improvements and bugfixes to
# hoagland@SiliconDefense.com.  As described in GNU General Public License, no
# warranty is expressed for this program.

# Used by: snortsnarf.pl, sel_to_add.pl, extr_alerts.pl
# Depends on: nada


# next_alert takes a file handle and the type of the file ('alert', 'syslog', or 'portscan') and returns the next alert found by reading that file handle along with the format of the alert ('fullalert', 'fastalert', 'syslog', or 'portscan') or returns undef if EOF was encountered (i.e., here are no more alerts)
sub next_alert {
	my($fh,$file_type)= @_;
	if ($file_type eq 'alert') {	
		my $alert= '';	
		while(<$fh>) {
			if(/spp_portscan/) {
				<$fh>;						 # line that follows
				next;						 # ok, lets try this again...
			}
			last unless /^\s*$/;	  # who needs a blank line? Not me.
		}
		return undef unless defined($_);
		if (/^\s*\[\*\*\]/) { # full alert format
			$alert= $_;
			while(<$fh>) {
				last if /^\s*$/;
				$alert.= $_;
			}
			return ($alert,'fullalert');
		} else {
			return ($_,'fastalert');
		}
	} elsif ($file_type eq 'syslog') {	
		while(<$fh>) {
			last if /snort/;
		}
		return undef unless defined($_);
		return ($_,'syslog');
	} else {	
		while(<$fh>) {
			last unless /^\s*$/;	  # who needs a blank line? Not me.
		}
		return undef unless defined($_);
		return ($_,'portscan');
	} 
}

# retrieve the alerts from the given files and number (position) ["file:pos","file:pos"]
sub get_alerts_parsed {
	my $fileinfo= pop(@_);
#warn "to get: ",join(',',@_),"\n";
	my %to_get= ();
	foreach (@_) {
		s/:(\d+)$//;
		push(@{$to_get{$_}},$1); # add loc to list for file
	}
	my(@alerts)= ();
	my $file;
	foreach $file (keys %to_get) {
		my ($fileformat,$filepath)= @{$fileinfo->{$file}};
#warn "$file => $format ; $filepath\n";
		my (@poss)= sort {$a <=> $b} @{$to_get{$file}};
		$fh='fh00';
		open($fh,"<$filepath") || die "could not open $filepath for reading alerts from";
		my $count= 0;
		while (1) {
			my ($alert,$format)= &next_alert($fh,$fileformat);
			unless (defined($alert)) {
				warn "could not find $poss[0] alerts in $filepath, only $count";
				last;
			}
			$count++;
			if ($count == $poss[0]) {
#warn "alert $count= >>$alert<<\n";
				shift(@poss);
				push(@alerts,&parse_alert($alert,$format));
				last unless (@poss);
			}
		}
		close $fh;
	}
	return @alerts;
}

# parses give alert text assuming that it is in a given format ('fullalert', 'fastalert', 'syslog', or 'portscan').  A hash reference is returned with 'field name' => 'contents' for various fields of interest.  Not that not all fields are extracted and recorded, just those we have needed.
sub parse_alert {
	my($alerttext,$format)= @_;
	my %monthnum=('Jan' => 1,
			 'Feb' => 2,
			 'Mar' => 3,
			 'Apr' => 4,
			 'May' => 5,
			 'Jun' => 6,
			 'Jul' => 7,
			 'Aug' => 8,
			 'Sep' => 9,
			 'Oct' => 10,
			 'Nov' => 11,
			 'Dec' => 12);
	my %alert= ('text' => $alerttext,'format' => $format);
	$_= $alerttext;
	
	if ($format eq 'fullalert') {
		my(@lines)= split("\n",$alerttext);
		$_= shift(@lines);
		
		# ---- Process the first line -----
		#
		# the first line just holds the attack id
		s/^\[\*\*\]\s*//; s/\s*\[\*\*\]\s*$//;
		$alert{'sig'} = $_;
			# Note: does not handle preprocessor log output

		# ---- Process the second line -----		
		#
		$_= shift(@lines);
				
		$alert{'month'} = substr($_,0,2);
		$alert{'date'} = substr($_,3,2);
		$alert{'time'} = substr($_,6,15);
		my $remainder =	 substr($_,22,(length $_));	 # grab the rest for regex matching

		my $e_option;
		if ($remainder =~ /^\d+\:/)
		{
			# Looks like an ethernet address - assume -e was set in snort command line
			$e_option = 1;
			# We could parse for ethernet stuff here but we don't 
			# feel like it right now.
		}
		else
		{
			# No -e option
			$e_option = 0;
			$remainder =~ s/ \-\> /-/; 
			my ($source,$destination) = split('-',$remainder);
			($alert{'src'},$alert{'sport'}) = split(':',$source);
			($alert{'dest'},$alert{'dport'}) = split(':',$destination);
		}
				 
		# ---- Process the third line -----
		#
		$_= shift(@lines);
				
		if($e_option)
		{
			# Ethernet stuff was on the previous line and now the IP source
			# and destination are here at the beginning of the third line.
			($alert{'src'},$alert{'sport'},$alert{'dest'},$alert{'dport'},$remainder) = /^(\d+\.\d+\.\d+\.\d+)\:(\d+)\s+\-\>\s+(\d+\.\d+\.\d+\.\d+)\:(\d+)\s+(.*)$/;
			unless(defined $alert{'src'})
			{
				#ICMP case
				($alert{'src'},$alert{'dest'},$remainder) = /^(\d+\.\d+\.\d+\.\d+)\s+\-\>\s+(\d+\.\d+\.\d+\.\d+)\s+(.*)$/;			
			}		   
			$_ = $remainder;			  
		}
		my($ttl,$tos,$id,$df); # not stored
		($alert{'proto'},$ttl,$tos,$id,$df) = /^(\w*)\sTTL\:(\d*)\sTOS\:(\w*)\sID\:(\d*)\s?\s?(DF)?$/;

		# ---- Process the fourth line -----
		#
		$_= shift(@lines);				
				
		my($flags,$seq,$ack,$win,$UDPlength,$ICMPid,$seq);
		if ($alert{'proto'} eq "TCP") {
			my($seq,$ack,$win); # not stored
			($alert{'flags'},$seq,$ack,$win) = /^([SFRUAP12*]*)\sSeq\:\s(\w*)\s*Ack\:\s(\w*)\s*Win\:\s(\w*)$/;
		} elsif ($alert{'proto'} eq "UDP") {
			# my($UDPlength); # not stored
			# ($UDPlength) = /^Len\:\s(\d*)$/;
		} elsif ($alert{'proto'} eq "ICMP") {
			my($ICMPid,$seq); # not stored
			if (/^ID/) {
				($ICMPid,$seq,$alert{'ICMP_type'}) = /^ID\:(\d*)\s*Seq\:(\d*)\s*(.*)/;
			} else {
				($alert{'ICMP_type'}) = /^(.*)/;
				# ($ICMPid,$seq) = (undef,undef); 
			}
		}

		# ---- Process the fifth line if there is one -----
		#
		$_= shift(@lines);				
				
		#if(defined($_) && $_ ne '') {
			#my $TCPoptions = "";  # not stored
			#$TCPoptions = substr($line5,16,(length $line5));
		#}
	} elsif ($format eq 'fastalert') {
		s/^\s*(\d+)\/(\d+)\-(\S*)//;
		($alert{'month'},$alert{'date'},$alert{'time'})= ($1,$2,$3);
		s/^\s+\[\*\*\]\s*(.+)\s*\[\*\*\]\s*//;
		$alert{'sig'}= $1;
		    # note: not handled: just a message (i.e. no packet) (e.g., from spp_portscan) -- ignore
		$alert{'sig'} =~ s/\s+$//;
		if (/:/) { 
			($alert{'src'},$alert{'sport'},$alert{'dest'},$alert{'dport'})= /^([\d\.]+):(\d+)\s*->\s*([\d\.]+):(\d+)/;
		} else { # just addresses
			($alert{'src'},$alert{'dest'})= /^([\d\.]+)\s*->\s*([\d\.]+)/;
			$alert{'proto'}= 'ICMP';
		}
	} elsif ($format eq 'syslog') {
		s/(\w+)\s+(\d+)\s+([\d:]+)\s+(\S+)\s+([^\[]+)\[\d+\]:\s*//;
		my($logginghost,$loggingprog); # not stored
		($alert{'month'},$alert{'date'},$alert{'time'},$logginghost,$loggingprog)=($1,$2,$3,$4,$5);
		$alert{'month'}= $monthnum{$alert{'month'}};
		
		my($src,$sport,$dest,$sport,$msg);
		if (s/\s*:\s*([\d\.]+)\:(\d+)\s*->\s*([\d\.]+)\:(\d+)\s*$//) {
			($alert{'src'},$alert{'sport'},$alert{'dest'},$alert{'dport'})= ($1,$2,$3,$4);
		} elsif (s/\s*:\s*([\d\.]+)\s*->\s*([\d\.]+)\s*$//) {
			($alert{'src'},$alert{'dest'})= ($1,$2);
			$alert{'proto'}= 'ICMP';
		} else {
			# must be just a message (not really handled)
		}
		$alert{'sig'}= $_;

	} else { # $format eq 'portscan'
  		($alert{'month'},$alert{'date'},$alert{'time'},$alert{'src'},$alert{'sport'},$alert{'dest'},$alert{'dport'},$alert{'proto'},$alert{'flags'}) = /^(\w+)\s+(\d+)\s+(\d\d\:\d\d:\d\d)\s+(\d+\.\d+\.\d+\.\d+)\:(\d+)\s+\-\>\s+(\d+\.\d+\.\d+\.\d+)\:(\d+)\s+(\w+)\s+(.*)$/;
		$alert{'month'}= $monthnum{$alert{'month'}};
		$alert{'proto'}= 'TCP' unless ($alert{'proto'} eq 'UDP') || ($alert{'proto'} eq 'ICMP');  # was SYN, etc
		
		$alert{'sig'} = "$alert{'proto'} $alert{'flags'} scan";
	}
	
	#unlike in the snortsnarf code, we do not record 'file' and we record format, not file type
	
	return \%alert;
}

1;

# $Id: snort_alert_parse.pl,v 1.3 2000/06/21 00:31:30 jim Exp $
