/*
** Copyright (C) 1998,1999 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spo_log_tcpdump.c,v 1.8 2000/06/03 04:56:04 roesch Exp $ */

/* spo_log_tcpdump 
 * 
 * Purpose:
 *
 * This plugin generates tcpdump formatted binary log files
 *
 * Arguments:
 *   
 * filename of the output log (default: snort.log)
 *
 * Effect:
 *
 * Packet logs are written (quickly) to a tcpdump formatted output
 * file
 *
 * Comments:
 *
 * First logger...
 *
 */

/* your preprocessor header file goes here */
#include "spo_log_tcpdump.h"

/* external globals from rules.c */
extern char *file_name;
extern int file_line;
extern pcap_dumper_t *dumpd;
extern PV pv;

/* If you need to instantiate the preprocessor's data structure, do it here */
LogTcpdumpData *TcpdumpLogInfo;

/*
 * Function: SetupLogTcpdump()
 *
 * Purpose: Registers the output plugin keyword and initialization 
 *          function into the output plugin list.  This is the function that
 *          gets called from InitOutputPlugins() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupLogTcpdump()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterOutputPlugin("log_tcpdump", NT_OUTPUT_LOG, LogTcpdumpInit);

#ifdef DEBUG
    printf("Output plugin: Log-Tcpdump is setup...\n");
#endif
}


/*
 * Function: LogTcpdumpInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void LogTcpdumpInit(u_char *args)
{
#ifdef DEBUG
    printf("Output: Log-Tcpdump Initialized\n");
#endif

    /* tell command line loggers to go away */
    pv.log_plugin_active = 1;

    /* parse the argument list from the rules file */
    ParseTcpdumpArgs((char *)args);

    TcpdumpInitLogFile(TcpdumpLogInfo);

    /* Set the preprocessor function into the function list */
    AddFuncToOutputList(LogTcpdump, NT_OUTPUT_LOG, TcpdumpLogInfo);
    AddFuncToCleanExitList(SpoLogTcpdumpCleanExitFunc, TcpdumpLogInfo);
    AddFuncToRestartList(SpoLogTcpdumpRestartFunc, TcpdumpLogInfo);
}



/*
 * Function: ParseTcpdumpArgs(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.  This function doesn't
 *          have to exist if it makes sense to parse the args in the init 
 *          function.
 *
 * Arguments: args => argument list
 *
 * Returns: void function
 *
 */
LogTcpdumpData *ParseTcpdumpArgs(char *args)
{
    TcpdumpLogInfo = (LogTcpdumpData *)malloc(sizeof(LogTcpdumpData));
    printf("Args: %s<>\n", args);

    if (args != NULL)
    {
        while (isspace((int)*args)) args++;

        TcpdumpLogInfo->filename = (char *) calloc(strlen(args) + 1, sizeof(char));  

        if (TcpdumpLogInfo->filename == NULL)
        {
            FatalError("ParseTcpdumpArgs() filename calloc failed: %s\n", strerror(errno));
        }

        strncpy(TcpdumpLogInfo->filename, args, strlen(args));
    }
    else
    {
        TcpdumpLogInfo->filename = (char *) calloc(strlen("snort.log")+1, sizeof(char));
        strncpy(TcpdumpLogInfo->filename, "snort.log", strlen("snort.log"));
    }
       return TcpdumpLogInfo;
}


/*
 * Function: PreprocFunction(Packet *)
 *
 * Purpose: Perform the preprocessor's intended function.  This can be
 *          simple (statistics collection) or complex (IP defragmentation)
 *          as you like.  Try not to destroy the performance of the whole
 *          system by trying to do too much....
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void LogTcpdump(Packet *p, char *msg, void *arg)
{      
    /* sizeof(struct pcap_pkthdr) = 16 bytes */
    pcap_dump((u_char *)TcpdumpLogInfo->dumpd,p->pkth,p->pkt);
    fflush((FILE *)TcpdumpLogInfo->dumpd);
}


/****************************************************************************
 *
 * Function: TcpdumpInitLogFile()
 *
 * Purpose: Initialize the tcpdump log file header
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void TcpdumpInitLogFile(LogTcpdumpData *data)
{
    time_t curr_time;      /* place to stick the clock data */
    struct tm *loc_time;   /* place to stick the adjusted clock data */
    char timebuf[10];
    char logdir[STD_BUF];

    bzero(logdir, STD_BUF);
    bzero(timebuf, 10);
    curr_time = time(NULL);
    loc_time = localtime(&curr_time);
    strftime(timebuf,91,"%m%d@%H%M",loc_time);

    if ((strlen(TcpdumpLogInfo->filename)+2+strlen(pv.log_dir)+strlen(timebuf)) > STD_BUF)
    {
        FatalError("ERROR: log file logging path and file name are too long, aborting!\n");
    }

    sprintf(logdir,"%s/%s-%s", pv.log_dir, timebuf, TcpdumpLogInfo->filename);

#ifdef DEBUG
    printf("Opening %s\n", logdir);
#endif

    /* overwrites the last frag file */
    if ((TcpdumpLogInfo->dumpd=pcap_dump_open(pd,logdir)) == NULL)
    {
        FatalError("log_tcpdump TcpdumpInitLogFile(): %s", strerror(errno));
    }

    return;
}

void SpoLogTcpdumpCleanExitFunc(int signal, void *arg)
{
#ifdef DEBUG
    printf("SpoLogTcpdumpCleanExitFunc\n");
#endif
    pcap_dump_close(TcpdumpLogInfo->dumpd);
    free(TcpdumpLogInfo->filename);
    free(TcpdumpLogInfo->dumpd);
    free(TcpdumpLogInfo);
}

void SpoLogTcpdumpRestartFunc(int signal, void *arg)
{
#ifdef DEBUG
    printf("SpoLogTcpdumpRestartFunc\n");
#endif
    pcap_dump_close(TcpdumpLogInfo->dumpd);
    free(TcpdumpLogInfo->filename);
    free(TcpdumpLogInfo->dumpd);
    free(TcpdumpLogInfo);
}

