/* $Id: snort.c,v 1.44 2000/03/20 08:40:58 roesch Exp $ */
/*
** Copyright (C) 1998,1999,2000 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/******************************************************************************
 *
 * Program: Snort
 *
 * Purpose: Check out the README file for info on what you can do
 *          with Snort.
 *
 * Author: Martin Roesch (roesch@clark.net)
 *
 * Comments: Ideas and code stolen liberally from Mike Borella's IP Grab 
 *           program. Check out his stuff at http://www.borella.net.  I
 *           also have ripped some util functions from TCPdump, plus Mike's
 *           prog is derived from it as well.  All hail TCPdump....
 *
 * HP-UX 10.x note from Chris Sylvain:
 * if you run snort and receive the error message
 *  "ERROR: OpenPcap() device lan0 open:
 *                    recv_ack: promisc_phys: Invalid argument"
 * it's because there's another program running using the DLPI service.
 * The HP-UX implementation doesn't allow more than one libpcap program
 * at a time to run, unlike Linux.
 *
 ******************************************************************************/

/*  I N C L U D E S  **********************************************************/
#include "snort.h"


extern OutputFuncNode *AlertList;
extern OutputFuncNode *LogList;

/****************************************************************************
 *
 * Function: main(int, char *)
 *
 * Purpose:  Handle program entry and exit, call main prog sections
 *
 * Arguments: See command line args in README file
 *
 * Returns: 0 => normal exit, 1 => exit on error
 *
 ****************************************************************************/
int main(int argc, char *argv[])
{
#if defined(LINUX) || defined (FREEBSD) || defined(OPENBSD) || defined(SOLARIS)
    sigset_t set;
    sigemptyset(&set);
    sigprocmask(SIG_SETMASK, &set, NULL);
#else
    sigsetmask(0);
#endif

    /* make this prog behave nicely when signals come along */
    signal(SIGKILL, CleanExit);
    signal(SIGTERM, CleanExit);
    signal(SIGINT, CleanExit);
    signal(SIGQUIT, CleanExit);
    signal(SIGHUP, CleanExit);
    /* sane permissions */
    umask(077);

    /* set a global ptr to the program name so other functions can tell
       what the program name is */
    progname = argv[0];
    progargs = argv;

    InitNetmasks();
    InitProtoNames();

    /* initialize the packet counter to loop forever */
    pv.pkt_cnt = -1;

    /* set the default alert mode */
    pv.alert_mode = ALERT_FULL;

    /* set the timezone (ripped from tcpdump) */
    thiszone = gmt2local(0);

    pv.quiet_flag = 0;

    /* chew up the command line */
    ParseCmdLine(argc, argv);

    /* if the logging flag hasn't been set yet... */
    if(!pv.log_flag)
    {
        /* ...set the default logging dir */
        strncpy(pv.log_dir,DEFAULT_LOG_DIR,strlen(DEFAULT_LOG_DIR)+1);  
    }

    if (pv.use_rules || pv.log_flag)
    {
        /* perform some sanity checks on the output directory, etc*/
        SanityChecks();
    }

    if (pv.use_rules && pv.rules_order_flag)
    {
        printf("Rule application order changed to Pass->Alert->Log\n");
    }

    /* we're not using rules, sniffing to the screen, or logging packets
       so we don't really need to run... */
    if (!pv.use_rules && !pv.verbose_flag && !pv.log_flag)
    {
        /* give them a nice little reminder that Snort needs to be told what
           to do */
        printf("\n\nUh, you need to tell me to do something....\n\n");
        DisplayBanner();
        ShowUsage(progname);
        exit(0);
    }


    /* if daemon mode requested, fork daemon first, otherwise
     * on linux interface will be reset.
     */
    if (pv.daemon_flag)
    {
#ifdef DEBUG
        printf("Entering daemon mode\n");
#endif
        GoDaemon();
    }

    if (!pv.readmode_flag)
    {
#ifdef DEBUG
        printf("Opening interface: %s\n", pv.interface);
#endif
        /* open up our libpcap packet capture interface */
        OpenPcap(pv.interface);
    }
    else
    {

#ifdef DEBUG
        printf("Opening file: %s\n", pv.readfile);
#endif

        OpenPcap(pv.readfile);
    }

    CreatePidFile(pv.interface);

#ifdef DEBUG
    printf("Setting Packet Processor\n");
#endif

    /* set the packet processor (ethernet, slip or raw)*/
    SetPktProcessor();

    if (pv.use_rules)
    {
        InitPreprocessors();
        InitPlugIns();
        InitOutputPlugins();
#ifdef DEBUG
        DumpPreprocessors();
        DumpPlugIns();
        DumpOutputPlugins();
#endif
        ParseRulesFile(pv.config_file, 0);
    }

    /* Tell 'em who wrote it, and what "it" is */
    if ( !pv.quiet_flag )
        DisplayBanner();

    if (pv.alert_cmd_override)
    {
        /* if a command line alert override has been specified, setup the
           approriate function pointer */

        /* if we're using syslog alerting set the alert function pointer to
           the syslog alerting function */
        if (pv.syslog_flag)
        {
            AddFuncToOutputList(SyslogAlert, NT_OUTPUT_ALERT, NULL);
            openlog("snort", LOG_PID|LOG_NDELAY|LOG_NOWAIT, LOG_AUTH);
        }
        else if (pv.smbmsg_flag)
        {
#ifdef ENABLE_SMB_ALERTS
            AddFuncToOutputList(SmbAlert, NT_OUTPUT_ALERT, NULL);
#else
            FatalError("ERROR: SMB support not compiled into program, exiting...\n");
#endif
        }
        else
        {
            switch (pv.alert_mode)
            {
                case ALERT_FAST:
                    alert = OpenAlertFile(NULL);
                    AddFuncToOutputList(FastAlert, NT_OUTPUT_ALERT, NULL);

                    break;

                case ALERT_FULL:
                    alert = OpenAlertFile(NULL);
                    AddFuncToOutputList(FullAlert, NT_OUTPUT_ALERT, NULL);

                    break;

                case ALERT_NONE:
                    SetOutputList(NoAlert, NT_OUTPUT_ALERT, NULL);

                    break;

                case ALERT_UNSOCK:
                    AddFuncToOutputList(UnixSockAlert, NT_OUTPUT_ALERT, NULL);
                    OpenAlertSock();

                    break;
            }
        }
    }
    
    if (AlertList == NULL && pv.use_rules) {
        alert = OpenAlertFile(NULL);
        AddFuncToOutputList(FullAlert, NT_OUTPUT_ALERT, NULL);
    }

    AlertFunc = CallAlertPlugins;

    if (pv.log_cmd_override)
    {
        if (pv.logbin_flag)
        {
#ifdef DEBUG
            printf("Initializing output file\n");
#endif
            InitBinLogFile();

            AddFuncToOutputList(LogBin, NT_OUTPUT_LOG, NULL);
        }
        else /* if we're not logging in tcpdump format... */
        {
            /* set the logging function pointer */
            if (!pv.nolog_flag)
                AddFuncToOutputList(LogPkt, NT_OUTPUT_LOG, NULL);
            else
                SetOutputList(NoLog, NT_OUTPUT_LOG, NULL);
        }
    }

    if (LogList == NULL)
        AddFuncToOutputList(LogPkt, NT_OUTPUT_LOG, NULL);

    LogFunc = CallLogPlugins;

#ifdef ENABLE_RESPONSE
#ifdef DEBUG
    printf("Opening Raw Socket for Sending\n");
#endif

    if ((nd = libnet_open_raw_sock(IPPROTO_RAW)) < 0)
    {
        FatalError("ERROR: cannot open raw socket for libnet, exiting...\n");
    }
#endif

    if(pv.chroot_dir != NULL) 
    {
       if(chdir(pv.chroot_dir) < 0)
          FatalError("Can not chdir to \"%s\"\n",pv.chroot_dir);

       if(chroot(pv.chroot_dir) < 0)
          FatalError("Can not chroot to %s\n",pv.chroot_dir);

       if(chdir("/") < 0)
          FatalError("Can not chdir to \"/\"\n");
    }

   
    /* set safe userID/groupID if needed */
    SetUidGid();


#ifdef DEBUG
    printf("Entering pcap loop\n");
#endif
    /* Read all packets on the device.  Continue until cnt packets read */
    if (pcap_loop(pd, pv.pkt_cnt, (pcap_handler)ProcessPacket, NULL) < 0)
    {
        if (pv.daemon_flag)
            syslog(LOG_CONS|LOG_DAEMON,"pcap_loop: %s", pcap_geterr(pd));
        else
            PrintFmtError("pcap_loop: %s", pcap_geterr(pd));

        CleanExit(SIGQUIT);
    }

    /* close the capture interface */
    pcap_close(pd);

    CleanExit(SIGQUIT);

    return 0;
}




void ProcessPacket(char *user, struct pcap_pkthdr *pkthdr, u_char *pkt)
{
    Packet p;

    /* call the packet decoder */
    (*grinder)(&p, pkthdr, pkt);

    /* print the packet to the screen */
    if (pv.verbose_flag)
    {
        if (p.iph != NULL)
        {
            PrintIPPkt(stdout, p.iph->ip_proto, &p);
        }
        else if (p.ah != NULL && pv.showarp_flag)
        {
            PrintArpHeader(stdout, &p);

            if (pv.log_flag)
            {
                LogArpPkt(&p);
            }
        }
    }

    /* check or log the packet as necessary */
    if (!pv.use_rules)
    {
        if (pv.log_flag)
        {
            if (pv.logbin_flag)
            {
                LogBin(&p, NULL, NULL);
            }
            else
            {
                if (p.iph != NULL)
                    LogPkt(&p, NULL, NULL);

                if (p.ah != NULL)
                {
                    if (!pv.nolog_flag)
                    {
                        OpenLogFile(ARP, &p);

                        PrintArpHeader(log_ptr, &p);

                        fclose(log_ptr);
                    }
                }
            }
        }
    }
    else
    {
        Preprocess(&p);
    }

    ClearDumpBuf();

}


/****************************************************************************
 *
 * Function: ShowUsage(char *)
 *
 * Purpose:  Display the program options and exit
 *
 * Arguments: progname => name of the program (argv[0])
 *
 * Returns: 0 => success
 *
 ****************************************************************************/
int ShowUsage(char *progname)
{
    printf("USAGE: %s [-options] <filter options>\n", progname);
    puts("Options:");
    puts("        -A         Set alert mode: fast, full, or none "
         " (alert file alerts only)"
         "\n                  \"unsock\" enables UNIX socket logging (experimental).");
    puts("        -a         Display ARP packets");
    puts("        -b         Log packets in tcpdump format (much faster!)");
    puts("        -c <rules> Use Rules File <rules>");
    puts("        -C         Print out payloads with character data only (no hex)");
    puts("        -d         Dump the Application Layer");
    puts("        -D         Run Snort in background (daemon) mode");
    puts("        -e         Display the second layer header info");
    puts("        -F <bpf>   Read BPF filters from file <bpf>");
    puts("        -g <gname> Run snort gid as `gname' user or uid after initialization");
    puts("        -h <hn>    Home network = <hn>");
    puts("        -i <if>    Listen on interface <if>");
    puts("        -l <ld>    Log to directory <ld>");
#ifdef ENABLE_SMB_ALERTS
    puts("        -M <wrkst> Sends SMB message to workstations in file <wrkst>");
    puts("                   (Requires smbclient to be in PATH)");
#endif
    puts("        -n <cnt>   Exit after receiving <cnt> packets");
    puts("        -N         Turn off logging (alerts still work)");
    puts("        -o         Change the rule testing order to Pass|Alert|Log");
    puts("        -O         Obfuscate the logged IP addresses");
    puts("        -p         Disable promiscuous mode sniffing");
    printf("        -P <snap>  set explicit snaplen of packet (default: %d)\n",
           SNAPLEN);
    puts("        -q         Quiet. Don't show banner and status report");
    puts("        -r <tf>    Read and process tcpdump file <tf>");
    puts("        -s         Log alert messages to syslog");
    puts("        -S <n=v>   Set rules file variable n equal to value v");
    puts("        -t <chrt>  Chroots process to <chrt> after initialisaton");
    puts("        -u <uname> Run snort uid as `uname' user (or uid) after initialization");
    puts("        -v         Be verbose");
    puts("        -V         Show version number");
/* Nobody really uses this, do they?
   puts("        -6         Display IPv6 packets");
   puts("        -x         Display IPX packets");
*/
    puts("        -?         Show this information");
    puts("<Filter Options> are standard BPF options, as seen in TCPDump");

    fflush(stdout);

    return 0;
}



/****************************************************************************
 *
 * Function: ParseCmdLine(int, char *)
 *
 * Purpose:  Parse command line args
 *
 * Arguments: argc => count of arguments passed to the routine
 *            argv => 2-D character array, contains list of command line args
 *
 * Returns: 0 => success, 1 => exit on error
 *
 ****************************************************************************/

extern char *optarg;          /* for getopt */
extern int optind;            /* for getopt */

int ParseCmdLine(int argc, char *argv[])
{
    int ch;                      /* storage var for getopt info */
    int read_bpf = 0;
    char bpf_file[STD_BUF];
    char *eq_n;
    char *eq_p;

#ifdef DEBUG
    printf("Parsing command line...\n");
#endif
    /* generally speaking, Snort works best when it's in promiscuous mode */
    pv.promisc_flag = 1;

    /* loop through each command line var and process it */
    while ((ch = getopt(argc, argv, "Og:u:t:CqS:pNA:F:DM:br:xeh:l:dc:n:P:i:vV?aso6")) != -1)
    {
#ifdef DEBUG
        printf("Processing cmd line switch: %c\n", ch);
#endif
        switch (ch)
        {
            case 'A': /* alert mode */
                if (!strncasecmp(optarg,"none", 4))
                {
                    pv.alert_mode = ALERT_NONE;
                }
                else if (!strncasecmp(optarg,"full", 4))
                {
                    pv.alert_mode = ALERT_FULL;
                }
                else if (!strncasecmp(optarg,"fast", 4))
                {
                    pv.alert_mode = ALERT_FAST;
                }
                else if (!strncasecmp(optarg,"unsock", 4))
                {
                    pv.alert_mode = ALERT_UNSOCK;
                }
                else
                {
                    FatalError("ERROR => Unknown command line alert option: %s\n", optarg);
                }

                pv.alert_cmd_override = 1;

                break;

            case 'C':
                pv.char_data_flag = 1;
                break;

            case 'D': /* daemon mode */
#ifdef DEBUG
                printf("Daemon mode flag set\n");
#endif
                pv.daemon_flag = 1;
                pv.quiet_flag  = 1;
                break;

            case 'q': /* no stdout output mode */
                pv.quiet_flag  = 1;
                break;

            case 't':
                if ((pv.chroot_dir = calloc(strlen(optarg)+1, 1)) == NULL)
                {
                    FatalError("[!] ERROR: chroot directory variable calloc failed!\n");
                }

                bcopy(optarg,pv.chroot_dir,strlen(optarg));
#ifdef DEBUG
                printf("chroot directory set to: %s\n", pv.chroot_dir);
#endif
            break;

            case 'u': /* setuid */
                if((pv.user_name = calloc(strlen(optarg)+1, 1)) == NULL) 
                   FatalError("[!] ERROR: UID variable calloc failed!\n");
        
                bcopy(optarg,pv.user_name,strlen(optarg));
        
                if((pv.userid = atoi(pv.user_name)) == 0) 
                {
                   pv.pw = getpwnam(pv.user_name);
                   if(pv.pw == NULL) 
                   {
                      FatalError("[!] ERROR: User \"%s\" unknown\n",pv.user_name);
                   }

                   pv.userid = pv.pw->pw_uid;
                } 
                else 
                {
                   pv.pw = getpwuid(pv.userid);
                   if(pv.pw == NULL) 
                   {
                       FatalError("[!] ERROR: Can not obtain username"
                                  " for uid: %lu\n", (u_long) pv.userid);
                   }
                }

                if(pv.group_name == NULL) 
                {
                   char name[STD_BUF]; 

                   bzero(name, STD_BUF);
                   sprintf(name,"%lu",(u_long)pv.pw->pw_gid);

                   if ((pv.group_name = calloc(strlen(name)+1, 1)) == NULL) 
                   {
                      FatalError("[!] ERROR: Unable to calloc for pv.group_name\n");
                   }

                   pv.groupid = pv.pw->pw_gid;
                }
#if DEBUG
                printf("UserID: %lu GroupID: %lu\n", (unsigned long)pv.userid, 
                        (unsigned long)pv.groupid);
#endif
                break;
            
           case 'g':

                if (pv.group_name != NULL) 
                    free(pv.group_name);
                if ((pv.group_name = calloc(strlen(optarg)+1, 1)) == NULL) 
                    FatalError("[!] ERROR: malloc for group name failed!\n");
        
                bcopy(optarg,pv.group_name,strlen(optarg));

                if ((pv.groupid = atoi(pv.group_name)) == 0) {
                pv.gr = getgrnam(pv.group_name);
                if (pv.gr == NULL) 
                    FatalError("[!] ERROR: Group \"%s\" unknown\n",pv.group_name);
    
                pv.groupid = pv.gr->gr_gid;
                } 
#if DEBUG
                printf("GroupID: %lu\n",(unsigned long)pv.groupid);
#endif
                break;

            case 'N': /* no logging mode */
#ifdef DEBUG
                printf("Logging deactivated\n");
#endif

                pv.nolog_flag = 1;
                pv.log_cmd_override = 1;

                break;

            case 'O': /* obfuscate the logged IP addresses for privacy */
                pv.obfuscation_flag = 1;
                
                break;

            case 'l': /* use log dir <X> */
                strncpy(pv.log_dir, optarg, STD_BUF-1);
#ifdef DEBUG
                printf("Log directory = %s\n", pv.log_dir);
#endif
                pv.log_flag = 1;
                break;

            case 'e': /* show second level header info */
#ifdef DEBUG
                printf("Show 2nd level active\n");
#endif
                pv.show2hdr_flag = 1;

                break;

            case 'b': /* log packets in binary format for post-processing */
#ifdef DEBUG
                printf("Tcpdump logging mode active\n");
#endif
                pv.logbin_flag = 1;
                pv.log_cmd_override = 1;

                break;

            case 'F': /* read BPF filter in from a file */
#ifdef DEBUG
                printf("Tcpdump logging mode active\n");
#endif
                strncpy(bpf_file, optarg, STD_BUF - 1);

                read_bpf = 1;

                break;

            case 'a': /* show ARP packets */
#ifdef DEBUG
                printf("Show ARP active\n");
#endif
                pv.showarp_flag = 1;

                break;

            case 'd': /* dump the application layer data */
                pv.data_flag = 1;
#ifdef DEBUG
                printf("Data Flag active\n");
#endif
                break;

            case 'v': /* be verbose */
                pv.verbose_flag = 1;
#ifdef DEBUG
                printf("Verbose Flag active\n");
#endif
                break;

            case 'n': /* grab x packets and exit */
                pv.pkt_cnt = atoi(optarg);
#ifdef DEBUG
                printf("Exiting after %d packets\n", pv.pkt_cnt);
#endif
                break;

            case 'c': /* use configuration file x ( which currently isn't used) */
                strncpy(pv.config_file, optarg, STD_BUF - 1);
                pv.use_rules = 1;
#ifdef DEBUG
                printf("Config file = %s\n", pv.config_file);
#endif
                break;

            case 'i': /* listen on interface x */
                pv.interface = (char *) malloc(strlen(optarg) + 1);
                bzero((char *)pv.interface, strlen(optarg)+1);
                strncpy(pv.interface, optarg, strlen(optarg));
#ifdef DEBUG
                printf("Interface = %s\n", pv.interface);
#endif
                break;

            case 'o': /* change the rules processing order to passlist first */
                pv.rules_order_flag = 1;
#ifdef DEBUG
                printf("Rule application order changed to Pass->Alert->Log\n");
#endif

                break;

            case 'p': /* disable explicit promiscuous mode */
                pv.promisc_flag = 0;
#ifdef DEBUG
                printf("Promiscuous mode disabled!\n");
#endif

                break;

            case 'P': /* explicitly define snaplength of packets */
                pv.pkt_snaplen = atoi(optarg);
#ifdef DEBUG
                printf("Snaplength of Packets set to: %d\n", pv.pkt_snaplen);
#endif
                break;


            case 'r': /* read packets from a TCPdump file instead of the net */
                strncpy(pv.readfile, optarg, STD_BUF - 1);
                pv.readmode_flag = 1;

                break;

            case 's': /* log alerts to syslog */
                pv.syslog_flag = 1;
#ifdef DEBUG
                printf("Logging alerts to syslog\n");
#endif
                pv.alert_cmd_override = 1;

                break;

            case '6': /* display IPv6 packets (decoder not implemented yet)*/
#ifdef DEBUG
                printf("Show IPv6 active\n");
#endif
                pv.showipv6_flag = 1;

                break;

            case 'x': /* display IPX packets (decoder not implemented yet)*/
#ifdef DEBUG
                printf("Show IPX active\n");
#endif
                pv.showipx_flag = 1;

                break;

            case 'M': /* SMB Message Option */

                pv.smbmsg_flag = 1;
                strncpy(pv.smbmsg_dir, optarg, STD_BUF-1);
                pv.alert_cmd_override = 1;

                break;

            case '?': /* show help and exit */
                DisplayBanner();
                ShowUsage(progname);
                exit(0);

            case 'V': /* prog ver already gets printed out, so we just exit */
                DisplayBanner();
                exit(0);

            case 'h': /* set home network to x, this will help determine what to
                         set logging diectories to */

                GenHomenet(optarg);

                break;

            case 'S': /* set a rules file variable */
                if ((eq_p = strchr(optarg, '=')) != NULL)
                {
                    eq_n = (char *) malloc(eq_p - optarg + 1);
                    bzero(eq_n, eq_p - optarg + 1);
                    strncpy(eq_n, optarg, eq_p - optarg);
                    VarDefine(eq_n, eq_p + 1);
            free(eq_n);
                }
        }
    }

    /* if we're reading in BPF filters from a file */
    if (read_bpf)
    {
        /* suck 'em in */
        pv.pcap_cmd = read_infile(bpf_file);
    }
    else
    {
        /* set the BPF rules string (thanks Mike!) */
        pv.pcap_cmd = copy_argv(&argv[optind]);
    }

#ifdef DEBUG
    if (pv.pcap_cmd != NULL)
    {
        printf("pcap_cmd = %s\n", pv.pcap_cmd);
    }
    else
    {
        printf("pcap_cmd is NULL!\n");
    }
#endif

    return 0;
}



/****************************************************************************
 *
 * Function: GenHomenet(char *)
 *
 * Purpose: Translate the command line character string into its equivalent
 *          32-bit network byte ordered value (with netmask)
 *
 * Arguments: netdata => The address/CIDR block
 *
 * Returns: void function
 *
 ****************************************************************************/
void GenHomenet(char *netdata)
{
    struct in_addr net;           /* place to stick the local network data */
    char **toks;                  /* dbl ptr to store mSplit return data in */
    int num_toks;                 /* number of tokens mSplit returns */
    int nmask;                    /* temporary netmask storage */

    /* break out the CIDR notation from the IP address */
    toks = mSplit(optarg,"/",2,&num_toks,0);

    if (num_toks > 1)
    {
        /* convert the CIDR notation into a real live netmask */
        nmask = atoi(toks[1]);

        if ((nmask > 0) && (nmask < 33))
        {
            pv.netmask = netmasks[nmask];
        }
        else
        {
            FatalError("ERROR: Bad CIDR block [%s:%d], 1 to 32 please!\n",
                       toks[1],nmask);
        }
    }
    else
    {
        FatalError("ERROR: No netmask specified for home network!\n");
    }

    /* since PC's store things the "wrong" way, shuffle the bytes into 
       the right order */
#ifndef WORDS_BIGENDIAN
    pv.netmask = htonl(pv.netmask);
#endif

#ifdef DEBUG
    printf("homenet netmask = %#8lX\n", pv.netmask);
#endif
    /* convert the IP addr into its 32-bit value */
    if ((net.s_addr = inet_addr(toks[0])) ==-1)
    {
        FatalError("ERROR: Homenet (%s) didn't x-late, WTF?\n",
                   toks[0]);
    }
    else
    {
#ifdef DEBUG
        struct in_addr sin;
        printf("Net = %s (%X)\n", inet_ntoa(net), net.s_addr);
#endif
        /* set the final homenet address up */
        pv.homenet = ((u_long)net.s_addr & pv.netmask);
#ifdef DEBUG
        sin.s_addr = pv.homenet;
        printf("Homenet = %s (%X)\n", inet_ntoa(sin), sin.s_addr);
#endif
    }

    free(toks);
}



/****************************************************************************
 *
 * Function: SetPktProcessor()
 *
 * Purpose:  Set which packet processing function we're going to use based on 
 *           what type of datalink layer we're using
 *
 * Arguments: None.
 *
 * Returns: 0 => success
 *
 ****************************************************************************/
int SetPktProcessor()
{
    switch (datalink)
    {
        case DLT_EN10MB: /* Ethernet */
            if (!pv.readmode_flag)
                printf("   => Decoding Ethernet on interface %s\n", pv.interface);
            else
                printf("Entering readback mode....\n");

            grinder = DecodeEthPkt;
            break;

        case 13:
        case DLT_IEEE802: /* Token Ring */
            if (!pv.readmode_flag)
                printf("   => Decoding Token Ring on interface %s\n", pv.interface);
            else
                printf("Entering readback mode...\n");

            grinder = DecodeTRPkt;

            break;

        case DLT_FDDI: /* FDDI */
            if (!pv.readmode_flag)
            {
                printf("   => Decoding FDDI on interface %s\n", pv.interface);
            }
            else
            {
                printf("Entering readback mode...\n");
            }

            grinder = DecodeFDDIPkt;

            break;


        case DLT_SLIP:  /* Serial Line Internet Protocol */
            if (!pv.readmode_flag)
                printf("   => Decoding Slip on interface %s\n", pv.interface);
            else
                printf("Entering readback mode....\n");

            if (pv.show2hdr_flag == 1)
            {
                printf("Second layer header parsing for this datalink "
                       "isn't implemented yet\n");
                pv.show2hdr_flag = 0;
            }

            grinder = DecodeSlipPkt;

            break;

        case DLT_PPP: /* point-to-point protocol */
            if (!pv.readmode_flag)
                printf("   => Decoding PPP on interface %s\n", pv.interface);
            else
                printf("Entering readback mode....\n");

            if (pv.show2hdr_flag == 1)
            {
                /* do we need ppp header showup? it's only 4 bytes anyway ;-) */
                printf("Second layer header parsing for this datalink "
                       "isn't implemented yet\n");
                pv.show2hdr_flag = 0;
            }

            grinder = DecodePppPkt;
            break;

        case DLT_NULL: /* loopback and stuff.. you wouldn't perform intrusion
                        * detection on it, but it's ok for testing.
                        */
            if (!pv.readmode_flag)
                printf("   => Decoding LoopBack on interface %s\n", pv.interface);
            else
                printf("Entering readback mode....\n");

            if (pv.show2hdr_flag == 1)
            {
                printf("Second layer header parsing for this datalink "
                       "isn't implemented yet\n");
                pv.show2hdr_flag = 0;
            }

            grinder = DecodeNullPkt;

            break;

#ifdef DLT_RAW /* Not supported in some arch or older pcap versions */
        case DLT_RAW:
            if (!pv.readmode_flag)
                printf("   => Decoding raw data on interface %s\n", pv.interface);
            else
                printf("Entering readback mode....\n");

            if (pv.show2hdr_flag == 1)
            {
                printf("There's no second layer header available for this datalink\n");
                pv.show2hdr_flag = 0;
            }

            grinder = DecodeRawPkt;

            break;
#endif
            /* you need the I4L modified version of libpcap
            to get this stuff working */
#ifdef DLT_I4L_RAWIP
        case DLT_I4L_RAWIP:  
            printf("   => Decoding I4L-rawip on interface %s\n", pv.interface);
            grinder = DecodeI4LRawIPPkt;
            break;
#endif

#ifdef DLT_I4L_IP
        case DLT_I4L_IP:
            printf("   => Decoding I4L-ip on interface %s\n", pv.interface);
            grinder = DecodeEthPkt;
            break;
#endif

#ifdef DLT_I4L_CISCOHDLC
        case DLT_I4L_CISCOHDLC:
            printf("   => Decoding I4L-cisco-h on interface %s\n", pv.interface);
            grinder = DecodeI4LCiscoIPPkt;
            break;
#endif


        default:  /* oops, don't know how to handle this one */
            PrintFmtError("\n%s cannot handle data link type %d", 
                          progname, datalink);
            CleanExit(SIGQUIT);
    }

    return 0;
}


/****************************************************************************
 *
 * Function: OpenPcap(char *)
 *
 * Purpose:  Open the libpcap interface
 *
 * Arguments: intf => name of the interface to open 
 *
 * Returns: 0 => success, exits on problems
 *
 ****************************************************************************/
int OpenPcap(char *intf)
{
    bpf_u_int32 localnet, netmask;    /* net addr holders */
    struct bpf_program fcode;         /* Finite state machine holder */
    char errorbuf[PCAP_ERRBUF_SIZE];  /* buffer to put error strings in */
    bpf_u_int32 defaultnet = 0xFFFFFF00;

    printf("\nInitializing Network Interface...\n");

    if (pv.interface == NULL)
    {
#ifdef DEBUG
        printf("pv.interface is NULL, looking up interface....   ");
#endif
        /* look up the device and get the handle */
        pv.interface = pcap_lookupdev(errorbuf);
#ifdef DEBUG
        printf("found interface %s\n", pv.interface);
#endif 
        /* uh oh, we couldn't find the interface name */
        if (pv.interface == NULL)
        {
            FatalError("ERROR: OpenPcap() interface lookup: \n\t%s\n", errorbuf);
        }
    }

    /* if we're not reading packets from a file */
    if (!pv.readmode_flag)
    {
        if (pv.pkt_snaplen)  /* if it's set let's try it... */
        {
            if (pv.pkt_snaplen < MIN_SNAPLEN)   /* if it's < MIN set it to MIN */
            {
                snaplen = MIN_SNAPLEN;
            }
            else
            {
                snaplen = pv.pkt_snaplen;
            }
        }
        else
        {
            snaplen = SNAPLEN; /* otherwise let's put the compiled value in */
        }

#ifdef DEBUG
        printf("snaplength info: set=%d/compiled=%d/wanted=%d\n", snaplen,
               SNAPLEN, pv.pkt_snaplen);
#endif

        /* get the device file descriptor */
        pd = pcap_open_live(pv.interface, snaplen,
                            pv.promisc_flag ? PROMISC : 0, READ_TIMEOUT, errorbuf);
    }
    else /* reading packets from a file */
    {
        /* open the file */
        pd = pcap_open_offline(intf, errorbuf);

        /* the file didn't open correctly */
        if (pd == NULL)
        {
            FatalError("ERROR => unable to open file %s for readback: %s\n", 
                       intf, errorbuf);
        }

        /* set the snaplen for the file (so we don't get a lot of extra
           crap in the end of packets */
        snaplen = pcap_snapshot(pd);

        printf("snaplen = %d\n", snaplen);
    }

    /* something is wrong with the opened packet socket */
    if (pd == NULL)
    {
        FatalError("ERROR: OpenPcap() device %s open: \n\t%s\n", 
                   pv.interface, errorbuf);
    }

    /* get local net and netmask */
    if (pcap_lookupnet(pv.interface, &localnet, &netmask, errorbuf) < 0)
    {
        ErrorMessage("WARNING: OpenPcap() device %s network lookup: \n\t%s\n", 
                   pv.interface, errorbuf);
    
        /* set the default netmask to 255.255.255.0 (for stealthed interfaces) */
        netmask = htonl(defaultnet);
    }

    

    /* compile BPF filter spec info fcode FSM */
    if (pcap_compile(pd, &fcode, pv.pcap_cmd, 1, netmask) < 0)
    {
        ErrorMessage("ERROR: OpenPcap() FSM compilation failed: \n\t%s\n", 
                     pcap_geterr(pd));
        FatalError("PCAP command: %s\n", pv.pcap_cmd);
    }

    /* set the pcap filter */
    if (pcap_setfilter(pd, &fcode) < 0)
    {
        FatalError("ERROR: OpenPcap() setfilter: \n\t%s\n", pcap_geterr(pd));
    }

    /* get data link type */
    datalink = pcap_datalink(pd);

    if (datalink < 0)
    {
        FatalError("ERROR: OpenPcap() datalink grab: \n\t%s\n", pcap_geterr(pd));
    }

    return 0;
}

/****************************************************************************
 *
 * Function: CleanExit()
 *
 * Purpose:  Clean up misc file handles and such and exit
 *
 * Arguments: Signal
 *
 * Returns: void function
 *
 ****************************************************************************/

extern PluginSignalFuncNode *PluginCleanExitList;
extern PluginSignalFuncNode *PluginRestartList;

void CleanExit(int sig)
{
    struct pcap_stat ps;
    float drop;
    float recv;
    PluginSignalFuncNode *idx;

    /* make sure everything that needs to go to the screen gets there */
    fflush(stdout);

    if (sig != SIGHUP)
        printf("\nExiting...\n");
    else
        printf("\nRestaring...\n");

    if (pv.logbin_flag)
    {
        pcap_dump_close(dumpd);
    }

    unlink(pv.pid_filename);
    pv.pid_filename[0] = 0;

    if (pv.alert_mode == ALERT_FAST || pv.alert_mode == ALERT_FULL)
    {
        if (alert != NULL)
            fclose(alert);
    }

    /* carry signals down to plugins */
    if(sig != SIGHUP) 
    {
        idx = PluginCleanExitList;
    } 
    else 
    {
        idx = PluginRestartList;
    }

    while(idx != NULL) 
    {
         idx->func(sig, idx->arg);
         idx = idx->next;
    }

    if (pv.readmode_flag)
    {
        puts("\n\n===============================================================================");

        recv = pc.tcp+pc.udp+pc.icmp+pc.arp+pc.ipx+pc.ipv6+pc.other;

        printf("Snort processed %d packets.\n", (int) recv);

        puts("Breakdown by protocol:");
        printf("    TCP: %-10ld (%.3f%%)\n", pc.tcp, CalcPct((float)pc.tcp, recv));
        printf("    UDP: %-10ld (%.3f%%)\n", pc.udp, CalcPct((float)pc.udp, recv));
        printf("   ICMP: %-10ld (%.3f%%)\n", pc.icmp, CalcPct((float)pc.icmp, recv));
        printf("  FRAGS: %-10ld (%.3f%%)\n", pc.frags, CalcPct((float)pc.frags, recv));
        printf("    ARP: %-10ld (%.3f%%)\n", pc.arp, CalcPct((float)pc.arp, recv));
        printf("   IPv6: %-10ld (%.3f%%)\n", pc.ipv6, CalcPct((float)pc.ipv6, recv));
        printf("    IPX: %-10ld (%.3f%%)\n", pc.ipx, CalcPct((float)pc.ipx, recv));
        printf("  OTHER: %-10ld (%.3f%%)\n", pc.other, CalcPct((float)pc.other, recv));
        puts("\n\n===============================================================================");

        exit_or_exec(0, sig);
        exit(0);
    }

    if (pd == NULL)
        exit(1);

    /* collect the packet stats */
    if (pcap_stats(pd, &ps))
    {
        pcap_perror(pd, "pcap_stats");
    }
    else
    {
        recv = ps.ps_recv;
        drop = ps.ps_drop;

        puts("\n\n===============================================================================");
        printf("Snort received %d packets", ps.ps_recv);

        if (ps.ps_recv)
        {
#ifndef LINUX
            printf(" and dropped %d(%.3f%%) packets\n\n", ps.ps_drop, 
                   CalcPct(drop, recv));
#else
            printf(".\nPacket loss statistics are unavailable under Linux.  Sorry!\n\n");
#endif
        }
        else
        {
            puts(".\n");
        }
        puts("Breakdown by protocol:");
        printf("    TCP: %-10ld (%.3f%%)\n", pc.tcp, CalcPct((float)pc.tcp, recv));
        printf("    UDP: %-10ld (%.3f%%)\n", pc.udp, CalcPct((float)pc.udp, recv));
        printf("   ICMP: %-10ld (%.3f%%)\n", pc.icmp, CalcPct((float)pc.icmp, recv));
        printf("  FRAGS: %-10ld (%.3f%%)\n", pc.frags, CalcPct((float)pc.frags, recv));
        printf("    ARP: %-10ld (%.3f%%)\n", pc.arp, CalcPct((float)pc.arp, recv));
        printf("   IPv6: %-10ld (%.3f%%)\n", pc.ipv6, CalcPct((float)pc.ipv6, recv));
        printf("    IPX: %-10ld (%.3f%%)\n", pc.ipx, CalcPct((float)pc.ipx, recv));
        printf("  OTHER: %-10ld (%.3f%%)\n", pc.other, CalcPct((float)pc.other, recv));

        puts("===============================================================================");
    }


    pcap_close(pd);

    exit_or_exec(0, sig);
    exit(0);
}



/*
 *
 * exit_or_exec()
 * Arguments: status, signal received.
 *
 * This function performs exec on SIGHUP signal and exit otherwise
 *
 */
void exit_or_exec(int stat, int sig)
{

    if ( sig != SIGHUP )
        exit(stat);
    else
    {
        syslog(LOG_DAEMON|LOG_NOTICE,"Received SIGHUP. Restarting");

#ifdef PARANOID
        execv(progname,progargs);
#else
        execvp(progname,progargs);
#endif

        syslog(LOG_DAEMON|LOG_NOTICE,"Restarting %s failed",progname);
        exit(1);
    }
}

/****************************************************************************
 *
 * Function: CalcPct(float, float)
 *
 * Purpose:  Calculate the percentage of a value compared to a total
 *
 * Arguments: cnt => the numerator in the equation
 *            total => the denominator in the calculation
 *
 * Returns: pct -> the percentage of cnt to value
 *
 ****************************************************************************/
float CalcPct(float cnt, float total)
{
    float pct;

    if (cnt > 0.0)
        pct = cnt/total;
    else
        return 0.0;

    pct *= 100.0;

    return pct;
}


/****************************************************************************
 *
 * Function: DisplayBanner()
 *
 * Purpose:  Show valuable proggie info
 *
 * Arguments: None.
 *
 * Returns: 0 all the time
 *
 ****************************************************************************/
int DisplayBanner()
{
    printf("\n-*> Snort! <*-\nVersion %s\nBy Martin Roesch (roesch@clark.net, www.snort.org)\n", VERSION);
    return 0;
}



/****************************************************************************
 *  
 * Function: ts_print(register const struct, char *)
 * 
 * Purpose: Generate a time stamp and stuff it in a buffer.  This one has
 *          millisecond precision.  Oh yeah, I ripped this code off from
 *          TCPdump, props to those guys.
 * 
 * Arguments: timeval => clock struct coming out of libpcap
 *            timebuf => buffer to stuff timestamp into
 *      
 * Returns: void function
 * 
 ****************************************************************************/
void ts_print(register const struct timeval *tvp, char *timebuf)
{
    register int s;  
    struct timeval tv;
    struct timezone tz;
    struct tm *lt;   /* place to stick the adjusted clock data */

    /* if null was passed, we use current time */
    if (!tvp)
    {
        /* manual page (for linux) says tz is never used, so.. */
        bzero((char *)&tz,sizeof(tz));
        gettimeofday(&tv,&tz);
        tvp=&tv;
    }
    lt = localtime((time_t *)&tvp->tv_sec);

    s = (tvp->tv_sec + thiszone) % 86400;

    (void)sprintf(timebuf, "%02d/%02d-%02d:%02d:%02d.%06u ", lt->tm_mon+1, 
                  lt->tm_mday, s / 3600, (s % 3600) / 60, s % 60, 
                  (u_int)tvp->tv_usec);
}



/****************************************************************************
 *  
 * Function: gmt2local(time_t)
 * 
 * Purpose: Figures out how to adjust the current clock reading based on the
 *          timezone you're in.  Ripped off from TCPdump.
 * 
 * Arguments: time_t => offset from GMT
 *      
 * Returns: offset seconds from GMT
 * 
 ****************************************************************************/
int gmt2local(time_t t)
{
    register int dt, dir;
    register struct tm *gmt, *loc;
    struct tm sgmt;

    if (t == 0)
        t = time(NULL);

    gmt = &sgmt;
    *gmt = *gmtime(&t);
    loc = localtime(&t);

    dt = (loc->tm_hour - gmt->tm_hour) * 60 * 60 + 
         (loc->tm_min - gmt->tm_min) * 60;

    /* If the year or julian day is different, we span 00:00 GMT
      and must add or subtract a day. Check the year first to
      avoid problems when the julian day wraps. */

    dir = loc->tm_year - gmt->tm_year;

    if (dir == 0)
        dir = loc->tm_yday - gmt->tm_yday;

    dt += dir * 24 * 60 * 60;

    return(dt);
}




/****************************************************************************
 *
 * Function: copy_argv(u_char **)
 *
 * Purpose: Copies a 2D array (like argv) into a flat string.  Stolen from
 *          TCPDump.
 *
 * Arguments: argv => 2D array to flatten
 *
 * Returns: Pointer to the flat string
 *
 ****************************************************************************/
char *copy_argv(char **argv)
{
    char **p;
    u_int len = 0;
    char *buf;
    char *src, *dst;
    void ftlerr(char *, ...);

    p = argv;
    if (*p == 0) return 0;

    while (*p)
        len += strlen(*p++) + 1;

    buf = (char *) malloc (len);

    if (buf == NULL)
    {
        FatalError("malloc() failed: %s\n", strerror(errno));
    }

    p = argv;
    dst = buf;

    while ((src = *p++) != NULL)
    {
        while ((*dst++ = *src++) != '\0');
        dst[-1] = ' ';
    }
    dst[-1] = '\0';

    return buf;
}




/****************************************************************************
 *
 * Function: strip(char *)
 *
 * Purpose: Strips a data buffer of CR/LF/TABs.  Replaces CR/LF's with
 *          NULL and TABs with spaces.
 *
 * Arguments: data => ptr to the data buf to be stripped
 *
 * Returns: size of the newly stripped string
 *
 ****************************************************************************/
int strip(char *data)
{
    int size;
    char *end;
    char *idx;

    idx = data;
    end = data + strlen(data);
    size = end - idx;

    while (idx != end)
    {
        if ((*idx == '\n') ||
            (*idx == '\r'))
        {
            *idx = 0;
            size--;
        }

        if (*idx == '\t')
        {
            *idx = ' ';
        }

        idx++;
    }

    return size;
}




void strip_quotes(char *data)
{
    if(index(data, '"') != NULL)
    {
        *(index(data,'"')) = ' ';
    }

    if(strrchr(data,'"') != NULL)
    {
        *(strrchr(data,'"')) = ' ';
    }

    return;
}




/****************************************************************************
 *
 * Function: InitNetMasks()
 *
 * Purpose: Loads the netmask struct in network order.  Yes, I know I could
 *          just load the array when I define it, but this is what occurred
 *          to me when I wrote this at 3:00 AM.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ****************************************************************************/
void InitNetmasks()
{
    netmasks[0] = 0x0;
    netmasks[1] = 0x80000000;
    netmasks[2] = 0xC0000000;
    netmasks[3] = 0xE0000000;
    netmasks[4] = 0xF0000000;
    netmasks[5] = 0xF8000000;
    netmasks[6] = 0xFC000000;
    netmasks[7] = 0xFE000000;
    netmasks[8] = 0xFF000000;
    netmasks[9] = 0xFF800000;
    netmasks[10] = 0xFFC00000;
    netmasks[11] = 0xFFE00000;
    netmasks[12] = 0xFFF00000;
    netmasks[13] = 0xFFF80000;
    netmasks[14] = 0xFFFC0000;
    netmasks[15] = 0xFFFE0000;
    netmasks[16] = 0xFFFF0000;
    netmasks[17] = 0xFFFF8000;
    netmasks[18] = 0xFFFFC000;
    netmasks[19] = 0xFFFFE000;
    netmasks[20] = 0xFFFFF000;
    netmasks[21] = 0xFFFFF800;
    netmasks[22] = 0xFFFFFC00;
    netmasks[23] = 0xFFFFFE00;
    netmasks[24] = 0xFFFFFF00;
    netmasks[25] = 0xFFFFFF80;
    netmasks[26] = 0xFFFFFFC0;
    netmasks[27] = 0xFFFFFFE0;
    netmasks[28] = 0xFFFFFFF0;
    netmasks[29] = 0xFFFFFFF8;
    netmasks[30] = 0xFFFFFFFC;
    netmasks[31] = 0xFFFFFFFE;
    netmasks[32] = 0xFFFFFFFF;
}




/****************************************************************************
 *
 * Function: GoDaemon()
 *
 * Purpose: Puts the program into daemon mode, nice and quiet like....
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ****************************************************************************/
void GoDaemon(void) 
{
    pid_t fs;

    printf("Initializing daemon mode\n");

    if (getppid() != 1)
    {
        fs=fork();

        if (fs > 0)
            exit(0); /* parent */

        if (fs < 0)
        {
            perror("fork");
            exit(1);
        }

        setsid();
    }

    /* redirect stdin/stdout/stderr to /dev/null */
    close(0);
    close(1);
    close(2);

#ifdef DEBUG
    open("/tmp/snort.debug", O_CREAT|O_RDWR);
#else
    open("/dev/null",O_RDWR);
#endif

    dup(0);
    dup(0);
    umask(077);

    return;
}






/****************************************************************************
 *
 * Function: SanityChecks()
 *
 * Purpose: CyberPsychotic sez: basically we only check if logdir exist and 
 *          writable, since it might screw the whole thing in the middle. Any
 *          other checks could be performed here as well.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ****************************************************************************/
void SanityChecks(void)
{
    struct stat st;
    struct stat pt;

    stat(pv.log_dir,&st);

    if (!S_ISDIR(st.st_mode) || access(pv.log_dir,W_OK) == -1)
    {
        FatalError("\n[!] ERROR:"
                   "Can not get write to logging directory %s.\n"
                   "(directory doesn't "
                   "exist or permissions are set incorrectly)\n\n",
                   pv.log_dir);
    }

#ifndef _PATH_VARRUN
    strncpy(_PATH_VARRUN, "/var/run/", 10);
#endif

    stat(_PATH_VARRUN, &pt);

    if(!S_ISDIR(pt.st_mode) || access(_PATH_VARRUN, W_OK) == -1)
    {
        ErrorMessage("[?] NOTICE: _PATH_VARRUN is unavailable!\n   => Logging Snort PID to log directory (%s)\n", pv.log_dir);

#ifndef _PATH_VARRUN
        sprintf(pv.pid_path,"%s/", pv.log_dir );
#endif
    }
    else
    {
        strncpy(pv.pid_path, _PATH_VARRUN, strlen(_PATH_VARRUN));
    }
}



/****************************************************************************
 *
 * Function: read_infile(char *)
 *
 * Purpose: Reads the BPF filters in from a file.  Ripped from tcpdump.
 *
 * Arguments: fname => the name of the file containing the BPF filters
 *
 * Returns: the processed BPF string
 *
 ****************************************************************************/
char *read_infile(char *fname)
{
    register int fd, cc;
    register char *cp;
    struct stat buf;

    fd = open(fname, O_RDONLY);

    if (fd < 0)
        PrintFmtError("can't open %s: %s", fname, pcap_strerror(errno));

    if (fstat(fd, &buf) < 0)
        PrintFmtError("can't stat %s: %s", fname, pcap_strerror(errno));

    cp = malloc((u_int)buf.st_size + 1);

    cc = read(fd, cp, (int)buf.st_size);

    if (cc < 0)
        PrintFmtError("read %s: %s", fname, pcap_strerror(errno));

    if (cc != buf.st_size)
        PrintFmtError("short read %s (%d != %d)", fname, cc, (int)buf.st_size);

    cp[(int)buf.st_size] = '\0';

    return(cp);
}



/****************************************************************************
 *
 * Function: InitProtoNames()
 *
 * Purpose: Initializes the protocol names
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ****************************************************************************/
void InitProtoNames()
{
    int i;
    int j;
    struct protoent *pt;

    /* I should really think about filling this in more completely.... */
    strncpy(protocol_names[IPPROTO_ICMP], "ICMP", 5);
    strncpy(protocol_names[IPPROTO_TCP], "TCP", 4);
    strncpy(protocol_names[IPPROTO_UDP], "UDP", 4);


    for(i=0;i<256;i++)
    {
        pt = getprotobynumber(i);      
 
        if(pt)
        {
            strncpy(protocol_names[i], pt->p_name, 10);
        }
        else
        {
            sprintf(protocol_names[i],"PROTO%03d",i);
        }
    }

    for(i=0;i<256;i++)
    {
        for(j=0;j<10;j++)
        {
            if(isalpha((int)protocol_names[i][j]))
            {
                protocol_names[i][j] = toupper((int)protocol_names[i][j]);
            }
        }
    }
}

/*
 * error message printing routines. in daemon mode these would go into
 * syslog.
 * 
 * first would allow to print formatted error messages (similar to printf) and
 * the second is similar to perror.
 * 
 */

void PrintFmtError(char *fmt, ...)
{
    char buf[STD_BUF+1];
    va_list ap;

    va_start(ap,fmt);
    vsprintf(buf,fmt,ap);
    if (pv.daemon_flag)
        syslog(LOG_CONS|LOG_DAEMON|LOG_ERR,
               "%s",buf);
    else
        ErrorMessage("%s\n",buf);

}

void PrintError(char *str) 
{
    if (pv.daemon_flag)
        syslog(LOG_CONS|LOG_DAEMON|LOG_ERR,"%s:%m",str);
    else
        perror(str);
}


/* 
 * Function: ErrorMessage(const char *, ...)
 * 
 * Purpose: Print a message to stderr.
 * 
 * Arguments: format => the formatted error string to print out
 *            ... => format commands/fillers
 * 
 * Returns: void function
 */
void ErrorMessage(const char *format, ...)
{
    char buf[STD_BUF];
    va_list ap;

    va_start(ap, format);

    if (pv.daemon_flag)
    {
        vsprintf(buf,format,ap);
        syslog(LOG_CONS|LOG_DAEMON|LOG_ERR, "%s",buf);
    }
    else
    {
        vfprintf(stderr, format, ap);
    }
}



/*
 * Function: FatalError(const char *, ...)
 *
 * Purpose: When a fatal error occurs, this function prints the error message
 *          and cleanly shuts down the program
 *
 * Arguments: format => the formatted error string to print out
 *            ... => format commands/fillers
 *    
 * Returns: void function
 */ 
void FatalError(const char *format, ...)
{
    char buf[STD_BUF];
    va_list ap;  

    va_start(ap, format);

    if (pv.daemon_flag)
    {
        vsprintf(buf,format,ap);
        syslog(LOG_CONS|LOG_DAEMON|LOG_ERR, "%s",buf);
    }
    else
    {
        vfprintf(stderr, format, ap);
    }

    exit(1);
}


/****************************************************************************
 *
 * Function: CreatePidFile(char *)
 *
 * Purpose:  Creates a PID file
 *
 * Arguments: Interface opened.
 *
 * Returns: void function
 *
 ****************************************************************************/
void CreatePidFile(char *intf)
{
    FILE *pid_file;

    sprintf(pv.pid_filename, "%ssnort_%s.pid", pv.pid_path, intf);
    pid_file = fopen(pv.pid_filename, "w");

    if (pid_file)
    {
        fprintf(pid_file, "%d\n", (int) getpid());
        fclose(pid_file);
    }
    else
    {
        ErrorMessage("Failed to create pid file %s", pv.pid_filename);
        pv.pid_filename[0] = 0;
    }
}



/****************************************************************************
 *
 * Function: SetUidGid(char *)
 *
 * Purpose:  Sets safe UserID and GroupID if needed
 *
 * Arguments: none
 *
 * Returns: void function
 *
 ****************************************************************************/


void SetUidGid(void) 
{
    if(pv.group_name != NULL) 
    {
        if(setgid(pv.groupid) < 0) 
            FatalError("[!] ERROR: Can not set gid: %lu\n",(u_long) pv.groupid);
#ifdef DEBUG
        printf("Set gid to %lu\n",pv.groupid);
#endif
    }

   if(pv.user_name != NULL) 
   {
      if(getuid() == 0 && initgroups(pv.user_name, pv.groupid) < 0)
      {
          FatalError("[!] ERROR: Can not initgroups(%s,%lu)", 
                     pv.user_name,(u_long) pv.groupid);
      }
        
      /** just to be on a safe side ... **/
      endgrent();
      endpwent();

      if(setuid(pv.userid) < 0) 
      {
         FatalError("Can not set uid: %lu\n",(u_long) pv.userid);
      }

#ifdef DEBUG
      printf("Set gid to %lu\n",pv.groupid);
#endif
   }

}
