/* $Id: plugbase.c,v 1.22 2000/06/03 04:56:02 roesch Exp $ */
/*
** Copyright (C) 1998,1999,2000 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "plugbase.h"

KeywordXlateList *KeywordList;
PreprocessKeywordList *PreprocessKeywords;
PreprocessFuncNode *PreprocessList;
OutputKeywordList *OutputKeywords;
OutputFuncNode *AlertList;
OutputFuncNode *LogList;
PluginSignalFuncNode *PluginCleanExitList;
PluginSignalFuncNode *PluginRestartList;

extern int file_line;
extern char *file_name;

void InitPlugIns()
{
    if ( !pv.quiet_flag )
    {
        printf("Initializing Plug-ins!\n");
    }

    SetupPatternMatch();
    SetupTCPFlagCheck();
    SetupIcmpTypeCheck();
    SetupIcmpCodeCheck();
    SetupTtlCheck();
    SetupIpIdCheck();
    SetupTcpAckCheck();
    SetupTcpSeqCheck();
    SetupDsizeCheck();
    SetupIpOptionCheck();
    SetupRpcCheck();
    SetupIcmpIdCheck();
    SetupIcmpSeqCheck();
    SetupSession();
}


void InitPreprocessors()
{
    if ( !pv.quiet_flag )
    {
        printf("Initializing Preprocessors!\n");
    }

    SetupHttpDecode();
    SetupMinfrag();
    SetupPortscan(); 
    SetupPortscanIgnoreHosts();
    SetupDefrag();
}



void InitOutputPlugins()
{
    if (!pv.quiet_flag)
    {
        printf("Initializating Output Plugins!\n");
    }

    SetupAlertSyslog();
    SetupLogTcpdump();
    SetupLogDatabase();
    SetupFastAlert();
    SetupFullAlert();
    SetupAlertSmb();
    SetupAlertUnixSock();
}


/****************************************************************************
 *
 * Function: RegisterPlugin(char *, void (*func)())
 *
 * Purpose:  Associates a rule option keyword with an option setup/linking
 *           function.
 *
 * Arguments: keyword => The option keyword to associate with the option 
 *                       handler
 *            *func => function pointer to the handler 
 *
 * Returns: void function
 *
 ***************************************************************************/
void RegisterPlugin(char *keyword, void (*func)(char *, OptTreeNode *, int))
{
    KeywordXlateList *idx;

#ifdef DEBUG
    printf("Registering keyword:func => %s:%p\n", keyword, func);
#endif

    idx = KeywordList;

    if (idx == NULL)
    {
        KeywordList = (KeywordXlateList *) calloc(sizeof(KeywordXlateList), sizeof(char));

        KeywordList->entry.keyword = (char *) calloc(strlen(keyword)+1, sizeof(char));
        strncpy(KeywordList->entry.keyword, keyword, strlen(keyword));
        KeywordList->entry.func = func;
    }
    else
    {
        /* go to the end of the list */
        while (idx->next != NULL)
        {
            if (!strncasecmp(idx->entry.keyword, keyword, strlen(keyword)))
            {
                FatalError( "ERROR %s (%d) => Duplicate detection plugin keyword!\n", file_name, file_line);
            }

            idx = idx->next;
        }

        idx->next = (KeywordXlateList *) calloc(sizeof(KeywordXlateList), sizeof(char));

        idx = idx->next;

        idx->entry.keyword = (char *) calloc(strlen(keyword)+1, sizeof(char));
        strncpy(idx->entry.keyword, keyword, strlen(keyword));
        idx->entry.func = func;
    }
}




/****************************************************************************
 *
 * Function: DumpPlugIns()
 *
 * Purpose:  Prints the keyword->function list
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void DumpPlugIns()
{
    KeywordXlateList *idx;

    if ( pv.quiet_flag )
        return;

    idx = KeywordList;

    printf("-------------------------------------------------\n");
    printf(" Keyword     |      Plugin Registered @\n");  
    printf("-------------------------------------------------\n");
    while (idx != NULL)
    {
        printf("%-13s:      %p\n", idx->entry.keyword, idx->entry.func);
        idx = idx->next;
    }
    printf("-------------------------------------------------\n\n");
}



/****************************************************************************
 *
 * Function: RegisterPreprocessor(char *, void (*func)(u_char *))
 *
 * Purpose:  Associates a preprocessor statement with its function.
 *
 * Arguments: keyword => The option keyword to associate with the 
 *                       preprocessor 
 *            *func => function pointer to the handler 
 *
 * Returns: void function
 *
 ***************************************************************************/
void RegisterPreprocessor(char *keyword, void (*func)(u_char *))
{
    PreprocessKeywordList *idx;

#ifdef DEBUG
    printf("Registering keyword:preproc => %s:%p\n", keyword, func);
#endif

    idx = PreprocessKeywords;

    if (idx == NULL)
    {
        /* alloc the node */
        PreprocessKeywords = (PreprocessKeywordList *) calloc(sizeof(PreprocessKeywordList), sizeof(char));

        /* alloc space for the keyword */
        PreprocessKeywords->entry.keyword = (char *) calloc(strlen(keyword)+1, sizeof(char));

        /* copy the keyword into the struct */
        strncpy(PreprocessKeywords->entry.keyword, keyword, strlen(keyword));

        /* set the function pointer to the keyword handler function */
        PreprocessKeywords->entry.func = (void *) func;
    }
    else
    {
        /* loop to the end of the list */
        while (idx->next != NULL)
        {
            if (!strncasecmp(idx->entry.keyword, keyword, strlen(keyword)))
            {
                FatalError( "ERROR %s (%d) => Duplicate preprocessor keyword!\n", file_name, file_line);
            }

            idx = idx->next;
        }

        idx->next = (PreprocessKeywordList *) calloc(sizeof(PreprocessKeywordList), sizeof(char));

        idx = idx->next;

        /* alloc space for the keyword */
        idx->entry.keyword = (char *) calloc(strlen(keyword)+1, sizeof(char));

        /* copy the keyword into the struct */
        strncpy(idx->entry.keyword, keyword, strlen(keyword));

        /* set the function pointer to the keyword handler function */
        idx->entry.func = (void *) func;
    }
}




/****************************************************************************
 *
 * Function: DumpPreprocessors()
 *
 * Purpose:  Prints the keyword->preprocess list
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void DumpPreprocessors()
{
    PreprocessKeywordList *idx;

    if ( pv.quiet_flag )
        return;
    idx = PreprocessKeywords;

    printf("-------------------------------------------------\n");
    printf(" Keyword     |       Preprocessor @ \n");  
    printf("-------------------------------------------------\n");
    while (idx != NULL)
    {
        printf("%-13s:       %p\n", idx->entry.keyword, idx->entry.func);
        idx = idx->next;
    }
    printf("-------------------------------------------------\n\n");
}



/****************************************************************************
 *
 * Function: RegisterOutputPlugin(char *, void (*func)(Packet *, u_char *))
 *
 * Purpose:  Associates an output statement with its function.
 *
 * Arguments: keyword => The output keyword to associate with the 
 *                       output processor 
 *            type => alert or log types
 *            *func => function pointer to the handler 
 *
 * Returns: void function
 *
 ***************************************************************************/
void RegisterOutputPlugin(char *keyword, int type, void (*func)(u_char *))
{
    OutputKeywordList *idx;

#ifdef DEBUG
    printf("Registering keyword:output => %s:%p\n", keyword, func);
#endif

    idx = OutputKeywords;

    if (idx == NULL)
    {
        /* alloc the node */
        OutputKeywords = (OutputKeywordList *) calloc(sizeof(OutputKeywordList), sizeof(char));

        idx = OutputKeywords;
    }
    else
    {
        /* loop to the end of the list */
        while (idx->next != NULL)
        {
            if (!strncasecmp(idx->entry.keyword, keyword, strlen(keyword)))
            {
                FatalError( "ERROR %s (%d) => Duplicate output keyword!\n", file_name, file_line);
            }

            idx = idx->next;
        }

        idx->next = (OutputKeywordList *) calloc(sizeof(OutputKeywordList), sizeof(char));

        idx = idx->next;
    }

    /* alloc space for the keyword */
    idx->entry.keyword = (char *) calloc(strlen(keyword)+1, sizeof(char));

    /* copy the keyword into the struct */
    strncpy(idx->entry.keyword, keyword, strlen(keyword));

    /* set the plugin type, needed to determine whether an overriding
       command line arg has been specified */
    idx->entry.node_type = type;

    /* set the function pointer to the keyword handler function */
    idx->entry.func = (void *) func;
}




/****************************************************************************
 *
 * Function: DumpOutputPlugins()
 *
 * Purpose:  Prints the keyword->preprocess list
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 ***************************************************************************/
void DumpOutputPlugins()
{
    OutputKeywordList *idx;

    if (pv.quiet_flag)
        return;

    idx = OutputKeywords;

    printf("-------------------------------------------------\n");
    printf(" Keyword     |          Output @ \n");  
    printf("-------------------------------------------------\n");
    while (idx != NULL)
    {
        printf("%-13s:       %p\n", idx->entry.keyword, idx->entry.func);
        idx = idx->next;
    }
    printf("-------------------------------------------------\n\n");
}


int PacketIsIP(Packet *p)
{
    if (p->iph != NULL)
        return 1;

    return 0;
}



int PacketIsTCP(Packet *p)
{
    if (p->iph != NULL && p->tcph != NULL)
        return 1;

    return 0;
}



int PacketIsUDP(Packet *p)
{
    if (p->iph != NULL && p->udph != NULL)
        return 1;

    return 0;
}



int PacketIsICMP(Packet *p)
{
    if (p->iph != NULL && p->icmph != NULL)
        return 1;

    return 0;
}



int DestinationIpIsHomenet(Packet *p)
{
    if ((p->iph->ip_dst.s_addr & pv.netmask) == pv.homenet)
    {
        return 1;
    }

    return 0;
}



int SourceIpIsHomenet(Packet *p)
{
    if ((p->iph->ip_src.s_addr & pv.netmask) == pv.homenet)
    {
        return 1;
    }

    return 0;
}


int IsTcpSessionTraffic(Packet *p)
{
    if (p->tcph == NULL)
        return 0;

    if (p->tcph->th_flags != (TH_PUSH | TH_ACK))
    {
        return 0;
    }

    return 1;
}


int CheckNet(struct in_addr *compare, struct in_addr *compare2)
{
    if (compare->s_addr == compare2->s_addr)
    {
        return 1;
    }

    return 0;
}

/* functions to aid in cleaning up aftre plugins */
void AddFuncToRestartList(void (*func)(int, void*), void *arg)
{
       PluginRestartList = AddFuncToSignalList(func, arg, PluginRestartList);
}

void AddFuncToCleanExitList(void (*func)(int, void*), void *arg)
{
       PluginCleanExitList = AddFuncToSignalList(func, arg, PluginCleanExitList);
}

PluginSignalFuncNode *AddFuncToSignalList(void (*func)(int, void*), void *arg,
        PluginSignalFuncNode *list)
{
    PluginSignalFuncNode *idx;

    idx = list;

    if (idx == NULL)
    {
        idx = (PluginSignalFuncNode *) calloc(sizeof(PluginSignalFuncNode), sizeof(char));

        idx->func = func;
        idx->arg = arg;
        list = idx;
    }
    else
    {
        while (idx->next != NULL)
            idx = idx->next;

        idx->next = (PluginSignalFuncNode *) calloc(sizeof(PluginSignalFuncNode), sizeof(char));

        idx = idx->next;
        idx->func = func;
        idx->arg = arg;
    }
    idx->next = NULL;

    return list;
}
