#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <netdb.h>
#include <errno.h>
#include <time.h>
#include <pcap.h>
#include "config.h"
#include "smurflog.h"
#include "hash.h"

u_int  num	 = 0;
u_long pps	 = 0;
float  kbps	 = 0.0;
u_char smurfmode = 0;
u_char resetsecs = RESET_SECONDS;
#if LOGGING_OPTIONS >= 2
float maxkbps	 = 0.0;
u_long maxpps	 = 0;
#endif

FILE *logfile;
struct hash hashtable[256];
struct bpf_program fcode;
char ebuf[PCAP_ERRBUF_SIZE];
struct in_addr localnet, netmask;
pcap_handler printer;
pcap_t *pd;

u_char o_promisc	= 0;		/* promisc mode flag */
u_char o_syslog		= 1;		/* log to syslog or file */
char  *o_logfile	= NULL;		/* file to log to if not syslog */
char  *o_iface		= NULL;		/* interface */

void usage(char *progname)
{
   fprintf(stderr, "usage: %s [-hp] [-i iface] [-f logfile]\n", progname);
   exit(1);
}

inline void log(char *format, ...)
{
   va_list args;
#if LOGGING_OPTIONS >= 1
   time_t current;
#endif

   va_start(args, format);

   if (o_syslog)
      vsyslog(LOG_NOTICE, format, args);
   else {
      time(&current);
      fprintf(logfile, "%.15s ", ctime(&current) + 4);
      vfprintf(logfile, format, args);
      putc('\n', logfile);
      fflush(logfile);
   }

   va_end(args);
}

inline RETSIGTYPE alarm_handler(int status)
{
   if (!smurfmode) {
      if (kbps >= THRESHOLD_KBPS && pps >= THRESHOLD_PPS) {
         smurfmode = 1;
#if LOGGING_OPTIONS >= 1
         log("Threshold reached, %4.02fkbps %ldpkt/s, Looks like a smurf.", kbps, pps);
         fflush(stdout);
#endif
      }
   } else {
#if LOGGING_OPTIONS == 3
      if (kbps || pps) {
         log("Update %4.02fkbps %ldpkt/s", kbps, pps);
         fflush(stdout);
      }
#endif
      if (kbps && pps) {
         resetsecs = RESET_SECONDS;
#if LOGGING_OPTIONS >= 2
         if (kbps > maxkbps)
            maxkbps = kbps;
         if (pps > maxpps)
            maxpps = pps;
#endif
      } else {
         if (resetsecs == 0) {
            resetsecs = RESET_SECONDS;
            smurfmode = 0;
#if LOGGING_OPTIONS == 1
            log("No longer in smurf mode.");
            fflush(stdout);
#endif
#if LOGGING_OPTIONS >= 2
            log("Leaving Smurf mode. Maximium seen %4.02fkbps %ldpkt/s.", maxkbps, maxpps);
            fflush(stdout);
#endif
            clear_hash();
            num = 0;
         } else
            --resetsecs;
      }
   }

#if LOGGING_OPTIONS >= 2
   {
#else
   if (!smurfmode) {
#endif
      pps = 0;
      kbps = 0.0;
   }
   alarm(1);
}

inline void handlepkt(const u_char *packet)
{
   u_short length;
   u_long  srcip;
   u_char  type;

   type   = *((u_char *)  (packet + 20));

   if (type != 0)			/* If not ICMP Echo Reply, skip it */
      return;

   length = *((u_short *) (packet + 2));
   srcip  = *((u_long *)  (packet + 12));
   srcip &= 0x00FFFFFF;			/* Mask 255.255.255.0 */

#if (LOGGING_OPTIONS == 0) || (LOGGING_OPTIONS >= 2)
#if LOGGING_OPTIONS >= 2
   kbps += ntohs(length) / 1024.0;
   pps  += 1;
#endif
   if (!smurfmode)
      return;
#else
   if (!smurfmode) {
      kbps += ntohs(length) / 1024.0;
      pps  += 1;
      return;
   }
#endif

   if (num > MAX_LOGABLE)
      return;
   if (!unnamed(srcip))
      return;

   num += 1;
   log("#%d - Probable Smurf attack detected from %s/24 (%d bytes)",
	num, inet_ntoa(*((struct in_addr *) &srcip)), ntohs(length));
}

inline void eth_printer(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   handlepkt(p+14);
}

inline void ppp_printer(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   handlepkt(p+4);
}

inline pcap_handler lookup_printer(int type)
{
   if (type == DLT_EN10MB || type == DLT_IEEE802)
      return eth_printer;
   if (type == DLT_PPP);
      return ppp_printer;

   fatal("unknown data link");
}

int main(int argc, char *argv[])
{
   int ch, i;

   printf("Smurflog v2.1 by Richard Steenbergen <humble@lightning.net>\n");

   while ((ch = getopt(argc, argv, "i:f:pu:h")) != EOF)
      switch(ch) {
         case 'i':
            o_iface = optarg;
            break;
         case 'f':
            o_syslog = 0;
            o_logfile = optarg;
            break;
         case 'p':
            o_promisc = 1;
            break;
         case 'h':
         default:
            usage(argv[0]);
      }

   if (o_iface == NULL) {
      o_iface = pcap_lookupdev(ebuf);
      if (o_iface == NULL)
         fatal(ebuf);
   }

   if (o_syslog)
      openlog("smurflog", LOG_PID, LOG_DAEMON);
   else {
      if ((logfile = fopen(o_logfile, "a")) == NULL)
         fatal("Cannot open specified logfile.");
   }

   if ((pd = pcap_open_live(o_iface, 128, o_promisc, 100, ebuf)) == NULL)
      fatal(ebuf);
   if (pcap_lookupnet(o_iface, &localnet.s_addr, &netmask.s_addr, ebuf) < 0)
      fatal(ebuf);
   if (pcap_compile(pd, &fcode, "icmp", 1, (u_long)netmask.s_addr) < 0)
      fatal(pcap_geterr(pd));
   if (pcap_setfilter(pd, &fcode) < 0)
      fatal(pcap_geterr(pd));
   if (o_iface) {
      printf("Now monitoring %s for smurf attacks.\n", o_iface);

      if (o_syslog)
         log("Now monitoring %s for smurf attacks.\n", o_iface);
      else
         fprintf(logfile, "Now monitoring %s for smurf attacks.\n", o_iface);
   }

   i = fork();

   if (i) {
      printf("Launching into background (pid: %d)\n", i); fflush(stdout);
      exit(0);
   }

   close(0); close(1); close(2);

   printer = lookup_printer(pcap_datalink(pd));

   fflush(stdout);
   memset(hashtable, 0, sizeof(hashtable));
   signal(SIGALRM, alarm_handler);
   alarm(1);

   for(;;)
      pcap_loop(pd, 0, printer, NULL);

   pcap_close(pd);

   if (o_syslog)
      closelog();
   else
      fclose(logfile);

   exit(0);
}
