/*	$Id: smtp-vilter.c,v 1.172 2007/01/21 13:09:15 mbalmer Exp $	*/

/*
 * Copyright (c) 2003 - 2007 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/queue.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <dlfcn.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#ifdef ENABLE_LDAP
#include <ldap.h>
#endif
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "pathnames.h"
#include "smtp-vilter.h"

#include "imsg.h"

#ifdef __linux__
#include <sys/param.h>
#include "strlfunc.h"
#include "strtonum.h"
#else
#include <sys/syslimits.h>
#endif

#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256
#define POLL_MAX	1

#define LOG_UNSET	-1

char	*port;
int	 virus_strategy;
int	 spam_strategy;
int	 unwanted_strategy;
int	 error_strategy;

int	 virus_reaction;
int	 spam_reaction;
int	 unwanted_reaction;
int	 clean_reaction;

/* there is only one reaction type: add to/clr from table */
char	*virus_table;
char	*spam_table;
char	*unwanted_table;
char	*clean_table;

int	 markall;
int	 logall;
int	 verbose;
char	*cfgfile;
char	*tmpdir;
int	 tmpfiles_gr;
int	 tmpfiles_setgrp;
char	*backend;
char	*backend_path;
char	*recipient_notification;
char	*notify_only;
int	 log_facility;
char	*logfile;
char	*statfile;
time_t	 stat_begin;
unsigned int stat_interval;
FILE	*logfp;
char	*user;
char	*group;
char	*chrootdir;
rlim_t	 rlimit_nofile;
rlim_t	 rlimit_nproc;
int	 nbe;
char	*spamprefix;
char	*spam_headerf;
char	*spam_headerv;
int	 logvirus, logspam, logunwanted;
int	 keep_tmpfiles;

/* statistics */
u_int32_t	n_conn;
u_int32_t	n_aborts;
u_int32_t	n_msgs;
u_int32_t	n_virus;
u_int32_t	n_spam;
u_int32_t	n_unwanted;
u_int32_t	n_err;

struct imsgbuf	*ibuf_e;

extern char *__progname;

#define DEFLT_LDAPHOST	"localhost"
#define DEFLT_LDAPPORT	389

#ifdef ENABLE_LDAP
LDAP	*ld;
#endif
char	*ldaphost;
int	 ldapport;
char	*ldapurl;
char	*searchbase;
char	*binddn;
char	*bindpasswd;
int	 ldap_use_tls;

volatile sig_atomic_t quit = 0;
volatile sig_atomic_t child_quit = 0;
volatile sig_atomic_t reconfig = 0;
volatile sig_atomic_t dumpstat = 0;
volatile sig_atomic_t alrm_expired = 0;

struct backend *be;
int	 dev = -1;

struct privdata {
	struct connection	*conn;
	struct message		*msg;
	void			*backend_data;
};

int dispatch_imsg(struct imsgbuf *);

extern void smtp_vilter_init(void);
extern int e_main(uid_t, gid_t, int[2]);

static void
sighdlr(int signum)
{
	switch (signum) {
	case SIGCHLD:
		child_quit = 1;
		break;
	case SIGINT:
	case SIGTERM:
		quit = 1;
		break;
	case SIGHUP:
		/* reconfigure */
		reconfig = 1;
		break;
	case SIGALRM:
		alrm_expired = 1;
		/* FALLTHROUGH */
	case SIGUSR1:
		dumpstat = 1;
		break;
	}
}

#if __OpenBSD__
__dead static void
#else
static void
#endif
usage(void)
{
	fprintf(stderr, "usage: %s"
	    " [-Vvmkx"
#ifdef ENABLE_LDAP
	    "Z[Z]"
#endif
	    "]"
	    " [-A path]"
	    " [-a spam-subject-prefix]"
#ifdef ENABLE_LDAP
	    " [-B searchbase]"
#endif
	    " [-b backend]"
	    " [-C configfile]"
	    " [-d dir] "
#ifdef ENABLE_LDAP
	    " [-D binddn]"
#endif
	    " [-e logfile]"
	    " [-f maxfiles]"
	    " [-g group]"
#ifdef ENABLE_LDAP
	    " [-h ldaphost]"
#endif
	    " [-i interval]"
#ifdef ENABLE_LDAP
	    " [-L ldapport]"
#endif
	    " [-n recipient-notification]"
	    " [-o notify-only]" 
 	    " [-p port]"
	    " [-s statfile]"
	    " [-T tmpfile-opt]"
	    " [-t chroot-dir]"
#ifdef ENABLE_LDAP
	    " [-U ldapurl]"
#endif
	    " [-u user]"
#ifdef ENABLE_LDAP
	    " [-w bindpasswd]"
#endif
	    "\n"
		, __progname);
	exit(1);
}

void
decode_backend(char *s)
{
	char	*p;
	int	 n;
	char	*q, *r;

	if ((q = strdup(s)) == NULL)
		err(1, "memory allocation error");

	nbe = 0;
	r = q;
	do {
		p = strsep(&r, ",");
		if (*p != '\0')
			++nbe;
	} while (r != NULL);

	free(q);

	if ((be = (struct backend *)calloc(nbe, sizeof(struct backend))) ==
	    NULL)
		err(1, "error allocating memory for backends");

	n = 0;
	do {
		p = strsep(&s, ",");
		if (*p != '\0') {
			if ((be[n].name = strdup(p)) == NULL)
				err(1, "memory allocation error");
			++n;
		}
	} while (s != NULL);
}

#ifdef ENABLE_LDAP
static int
get_rebind_credentials(LDAP *ld, char **dnp, char **pwp, int *authmethodp,
    int freeit)
{
	if (verbose)
		warnx("get_rebind_credentials() called");
	return LDAP_SUCCESS;
}
#endif

int
main(int argc, char *argv[])
{
	int		 ch;
	char		 libname[MAX_FILE];
	struct passwd	*passwd;
	struct group	*grp;
	uid_t		 uid;
	gid_t		 gid;
	struct rlimit	 rl;
	struct pollfd	 pfd[1];
	int		 n;
	pid_t		 child_pid, pid;
	char		*fnam;	/* file to analyze */
	FILE		*fp;
#ifdef ENABLE_LDAP
	int		 version;
#endif
	int		 xmode;
	const char	*errstr;
	rlim_t		 rlimit_max;
	int		 pipe_m2e[2];
	int		 nfds;
	int		 flags;
	time_t		 now;

	/* Set initial values */
	port = NULL;
	tmpdir = NULL;
	tmpfiles_gr = 0;
	tmpfiles_setgrp = 0;
	user = NULL;
	group = NULL;
	chrootdir = NULL;
	cfgfile = _PATH_CFGFILE;
	backend = NULL;
	backend_path = NULL;
	virus_strategy = -1;
	spam_strategy = -1;
	unwanted_strategy = -1;
	error_strategy = -1;

	fnam = NULL;

	virus_reaction = REACTION_UNSET;
	spam_reaction = REACTION_UNSET;
	unwanted_reaction = REACTION_UNSET;
	clean_reaction = REACTION_UNSET;

	virus_table = spam_table = unwanted_table = clean_table = NULL;

	log_facility = LOG_UNSET;
	logfile = NULL;
	statfile = NULL;
	stat_interval = 0;
	recipient_notification = NULL;
	notify_only = NULL;
	markall = 0;
	logall = 0;
	rlimit_nofile = 0;
	rlimit_nproc = 0;
	spamprefix = NULL;
	spam_headerf = NULL;
	spam_headerv = NULL;
	logvirus = logspam = logunwanted = 0;
	keep_tmpfiles = 0;
#ifdef ENABLE_LDAP
	ldaphost = NULL;
	ldapport = -1;
	ldapurl = NULL;
	searchbase = NULL;
	binddn = NULL;
	bindpasswd = NULL;
	ldap_use_tls = -1;
#endif
	xmode = 0;

	/* Process the commandline */
#ifdef ENABLE_LDAP	
	while ((ch = getopt(argc, argv,
	    "A:a:B:b:C:D:d:e:f:g:h:i:kL:mn:o:p:s:T:t:u:U:Vvw:xZ?")) != -1) {
#else
	while ((ch = getopt(argc, argv,
	    "A:a:b:C:d:e:f:g:i:kmn:o:p:T:t:s:u:VvxZ?")) != -1) {
#endif
		switch (ch) {
		case 'A':
			fnam = optarg;
			break;
		case 'a':
			spamprefix = optarg;
			break;
#ifdef ENABLE_LDAP
		case 'B':
			searchbase = optarg;
			break;
#endif
		case 'b':
			backend = optarg;
			decode_backend(backend);
			break;
		case 'C':
			cfgfile = optarg;
			break;
#ifdef ENABLE_LDAP
		case 'D':
			binddn = optarg;
			break;
#endif
		case 'd':
			tmpdir = optarg;
			break;
		case 'e':
			logfile = optarg;
			break;
		case 'f':
			if (getuid()) {
				getrlimit(RLIMIT_NOFILE, &rl);
				rlimit_max = rl.rlim_max;
			} else
				rlimit_max = RLIM_INFINITY;
			rlimit_nofile = (int)strtonum(optarg, 1LL,
			    (long long)rlimit_max, &errstr);
			if (errstr)
				errx(1, "number of files is %s: %s", errstr,
				    optarg);
			break; 
		case 'g':
			group = optarg;
			break;
#ifdef ENABLE_LDAP
		case 'h':
			ldaphost = optarg;
			break;
#endif
		case 'i':
			stat_interval = (unsigned int)strtonum(optarg, 1LL,
			    100000000LL, &errstr);
			if (errstr)
				errx(1, "statistics dump interval is %s: %s",
				    errstr, optarg);
			break;
		case 'k':
			keep_tmpfiles = 1;
			break;
#ifdef ENABLE_LDAP
		case 'L':
			ldapport = strtonum(optarg, 1, 65535, &errstr);
			if (errstr)
				errx(1, "ldap port number is %s: %s", errstr,
				    optarg);
			break;
#endif
		case 'm':
			markall = 1;
			break;
		case 'n':
			recipient_notification = optarg;
			break;
		case 'o':
			notify_only = optarg;
			break;
		case 'p':
			port = optarg;
			break;
		case 's':
			statfile = optarg;
			break;
		case 'T':
			if (!strcmp(optarg, "g+r+"))
				tmpfiles_gr = 1;
			else if (!strcmp(optarg, "setgrp"))
				tmpfiles_setgrp = 1;
			else
				errx(1, "unknown tmp file option %s", optarg);
			break;
		case 't':
			chrootdir = optarg;
			break;
		case 'u':
			user = optarg;
			break;
#ifdef ENABLE_LDAP
		case 'U':
			ldapurl = optarg;
			break;
#endif
		case 'V':
			printf("%s %s\n", __progname, VERSION);
#ifdef ENABLE_LDAP
			printf("LDAP support enabled\n");
#endif
			exit(0);
		case 'v':
			++verbose;
			break;
#ifdef ENABLE_LDAP
		case 'w':
			bindpasswd = optarg;
			break;
#endif
		case 'x':
			xmode = 1;
			break;
#ifdef ENABLE_LDAP
		case 'Z':
			switch (ldap_use_tls) {
			case -1:
				ldap_use_tls = 1;
				break;
			case 1:
				ldap_use_tls = 2;
				break;
			default:
				usage();
			}
			break;
#endif
		default:
			usage();
		}
	}

	/*
	argc -= optind;
	argv += optind;
	*/

	/* Read config file */
	smtp_vilter_init();

	if (!nbe)
		errx(1, "no backends defined");

	/* Set default values if some variables are not set */
	if (port == NULL)
		port = strdup(_PATH_PORT);
	if (tmpdir == NULL)
		tmpdir = strdup(_PATH_TMPDIR);
	if (backend == NULL)
		backend = strdup("clamd");
	if (backend_path == NULL)
		backend_path = strdup(_PATH_BACKEND);
	if (virus_strategy == -1)
		virus_strategy = STRATEGY_DISCARD;
	if (spam_strategy == -1)
		spam_strategy = STRATEGY_MARK;
	if (unwanted_strategy == -1)
		unwanted_strategy = STRATEGY_MARK;
	if (error_strategy == -1)
		error_strategy = STRATEGY_TEMPFAIL;
	if (log_facility == LOG_UNSET)
		log_facility = LOG_MAIL;

#ifdef ENABLE_LDAP
	if (ldaphost == NULL)
		ldaphost = strdup(DEFLT_LDAPHOST);
	if (ldapport == -1)
		ldapport = DEFLT_LDAPPORT;
	if (ldap_use_tls == -1)
		ldap_use_tls = 0;
#endif

	if (xmode) {
		printf("backends: ");
		for (n = 0; n < nbe; n++)
			printf("%s ", be[n].name);
		printf("\n");
		printf("user: %s\n", user != NULL ? user : "not set");
		printf("group: %s\n", group != NULL ? group : "not set");
		printf("chroot: %s\n", chrootdir != NULL ? chrootdir :
		    "not set");
		printf("recipient-notification: %s\n", recipient_notification
		    != NULL ? recipient_notification : "not set");
		printf("spam-subject-prefix: %s\n", spamprefix != NULL ?
		    spamprefix : "not set");
		printf("port: %s\n", port);
		printf("tmpdir: %s\n", tmpdir);
		printf("logfile: %s\n", logfile != NULL ? logfile : "not set");
		if (rlimit_nofile != 0)
			printf("maxfiles: %lld\n", (long long)rlimit_nofile);
		else
			printf("maxfiles: not set\n"); 
		if (rlimit_nproc != 0)
			printf("maxprocs: %lld\n", (long long)rlimit_nproc);
		else
			printf("maxprocs: not set\n");
#ifdef ENABLE_LDAP
		printf("ldapurl: %s\n", ldapurl != NULL ? ldapurl : "not set");
		printf("ldaphost: %s\n", ldaphost);
		printf("ldapport: %s\n", ldapport);
		printf("use tls: ");
		switch (ldap_use_tls) {
		case 0:
			printf("never\n");
			break;
		case 1:
			printf("try\n");
			break;
		case 2:
			printf("always\n");
			break;
		}
		printf("searchbase: %s\n", searchbase != NULL ?
		    searchbase : "not set");
		printf("binddn: %s\n", binddn != NULL ? binddn : "not set");
		printf("bindpasswd: %s\n", bindpasswd != NULL ?
		    bindpasswd : "not set");
#endif
		printf("virus reaction: ");
		switch (virus_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", virus_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", virus_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		printf("spam reaction: ");
		switch (spam_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", spam_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", spam_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		printf("unwanted-content reaction: ");
		switch (unwanted_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", unwanted_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", unwanted_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		printf("clean reaction: ");
		switch (clean_reaction) {
			case REACTION_ADDTBL:
				printf("add to table %s\n", clean_table);
				break;
			case REACTION_DELTBL:
				printf("clear from table %s\n", clean_table);
				break;
			default:
				printf("not set\n");
				break;
		}
		return 0;
	}

#ifdef ENABLE_LDAP
	if (ldapurl == NULL) {
		if ((ld = ldap_init(ldaphost, ldapport)) == NULL)
			errx(1, "No directory server at %s, %m", ldaphost);
		else if (verbose)
			warnx("using LDAP server %s:%d", ldaphost, ldapport);
	} else {
		if (ldap_initialize(&ld, ldapurl) != LDAP_SUCCESS)
			errx(1, "No directory server at %s, %m", ldapurl);
		else if (verbose)
			warnx("using LDAP server %s", ldapurl);
	}
	version = LDAP_VERSION3;
	if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version) !=
	    LDAP_OPT_SUCCESS)
		errx(1, "Failed to set LDAP version 3 protocol");

	if (ldap_use_tls > 0 && ldap_start_tls_s(ld, NULL, NULL) !=
	    LDAP_SUCCESS) {
		if (ldap_use_tls > 1)
			errx(1, "Failed to start TLS for LDAP");
		else
			warnx("Failed to start TLS for LDAP");
	}

	/*
	 * Setup LDAP rebind, eventually continue with system defaults if
	 * the initial bind fails?
 	 */
	if (ldap_simple_bind_s(ld, binddn, bindpasswd) != LDAP_SUCCESS)
		errx(1, "Failed to bind to directory server as '%s'", binddn);

	ldap_set_rebind_proc(ld, get_rebind_credentials, NULL);

	if (verbose)
		warnx("Directory server at %s server is up, %susing TLS",
		    ldaphost, ldap_use_tls ? "" : "not ");
#endif
	openlog(__progname, verbose ? LOG_CONS | LOG_NDELAY | LOG_PID |
	    LOG_PERROR : LOG_CONS | LOG_NDELAY | LOG_PID, log_facility);

	/* Load the backends */
	for (n = 0; n < nbe; n++) {
		snprintf(libname, sizeof(libname), "%s/vilter-%s.so",
		    backend_path, be[n].name);

		if (verbose)
			warnx("loading backend %s from file %s", be[n].name,
			    libname);
			
		if ((be[n].dlhandle = dlopen(libname, RTLD_LAZY)) == NULL)
			err(1, "error loading backend %s (%s), %s", be[n].name,
			    libname, dlerror());
		
		be[n].be_init = dlsym(be[n].dlhandle, "vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "vilter_name")) ==
		    NULL)
			errx(1, "no function vilter_name in backend %s",
			    be[n].name);

		/* Optional */
		be[n].be_envfrom = dlsym(be[n].dlhandle, "vilter_envfrom");
		if ((be[n].be_type = dlsym(be[n].dlhandle, "vilter_type")) ==
		    NULL)
			errx(1, "no function vilter_type in backend %s",
			    be[n].name);

		be[n].type = be[n].be_type();

		if (be[n].type & BE_SCAN_VIRUS)	{
			if ((be[n].be_scan.be_scan_virus =
			    dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s",
				    be[n].name);
			if (be[n].type & (BE_SCAN_SPAM | BE_SCAN_UNWANTED))
				errx(1, "illegal backend type combination %s",
				    be[n].name);
		} else if (be[n].type & BE_SCAN_SPAM) {
			if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle,
			    "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s",
				    be[n].name);
			if (be[n].type & BE_SCAN_UNWANTED)
				errx(1, "illegal backend type combination %s",
				    be[n].name);
		} else if (be[n].type & BE_SCAN_UNWANTED) {
			if ((be[n].be_scan.be_scan_unwanted =
			    dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s",
				    be[n].name);
		} else
			errx(1, "backend %s has an unknown vilter type",
			    be[n].name);

		be[n].be_exit = dlsym(be[n].dlhandle, "vilter_exit");
		if (be[n].be_init != NULL) {
			if (be[n].be_init(be[n].config_file))
				errx(1, "error initializing backend %s",
				    be[n].name);
		}
	}

	/*
	 * Analyze a single file if requested by the user,
	 * else normal operation
	 */
	if (fnam != NULL) {
		char		 scanfn[PATH_MAX];
		char		 virus[MAXLEN];
		char		 reason[MAXLEN];
		double		 score, threshold;
		int		 scanresult;
		int		 exval;

		exval = 0;
		for (n = 0; n < nbe; n++) {
			scanresult = SCAN_OK;
			if (be[n].type & BE_SCAN_VIRUS) {
				if (verbose)
					warnx("scanning %s for viruses using "
					    "backend %s", fnam, be[n].name);
				strlcpy(scanfn, fnam, sizeof(scanfn));
				bzero(virus, sizeof(virus));
				scanresult = be[n].be_scan.be_scan_virus(NULL,
				    scanfn, sizeof(scanfn), NULL, virus,
				    sizeof(virus));
			} else if (be[n].type & BE_SCAN_SPAM) {
				if (verbose)
					warnx("scanning %s for spam using "
					    "backend %s", fnam, be[n].name);
				strlcpy(scanfn, fnam, sizeof(scanfn));
				scanresult = be[n].be_scan.be_scan_spam(NULL,
				    scanfn, sizeof(scanfn), NULL, &score,
				    &threshold);
			} else if (be[n].type & BE_SCAN_UNWANTED) {
				if (verbose)
					warnx("scanning %s for unwanted content"
					    " using backend %s", fnam,
					    be[n].name);
				strlcpy(scanfn, fnam, sizeof(scanfn));
				bzero(reason, sizeof(reason));
				scanresult =
				    be[n].be_scan.be_scan_unwanted(NULL,
				    scanfn, sizeof(scanfn), NULL, reason,
				    sizeof(reason));
			}
			switch (scanresult) {
			case SCAN_OK:
				printf("file scanned ok\n");
				break;
			case SCAN_ERROR:
				printf("error during scan\n");
				exval = 1;
				break;
			case SCAN_VIRUS:
				printf("virus '%s' detected in file\n", virus);
				exval = 1;
				break;
			case SCAN_SPAM:
				printf("classified as spam, score = %f, "
				    "threshold %f\n", score, threshold);
				exval = 1;
				break;
			case SCAN_UNWANTED:
				printf("unwanted content, reson = %s\n",
				    reason);
				exval = 1;
				break;
			}
		}

#ifdef ENABLE_LDAP
		ldap_unbind(ld);
#endif
		return exval;
	}

	uid = getuid();
	gid = getgid();

	if (user != NULL) {
		if ((passwd = getpwnam(user)) != NULL) {
			uid = passwd->pw_uid;
			gid = passwd->pw_gid;
		} else
			err(1, "no such user '%s'", user);
		
		if (group != NULL) {
			if ((grp = getgrnam(group)) != NULL)
				gid = grp->gr_gid;
			else
				err(1, "no such group '%s'", group);
		}
	}

	if (verbose == 0 &&  daemon(0, 0))
		err(1, "can't run as daemon");

	/* initialize statistics */
	n_conn = n_msgs = n_aborts = 0;
	n_virus = n_spam = n_unwanted = n_err = 0;
	time(&stat_begin);

	while (!quit) {

		/* Setup imsg stuff, this is a one-way com only */
		if (socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, pipe_m2e) == -1)
			errx(1, "imsg setup failed");
		if ((flags = fcntl(pipe_m2e[0], F_GETFL, 0)) == -1)
			errx(1, "fcntl failed");
		flags |= O_NONBLOCK;
		if (fcntl(pipe_m2e[0], F_SETFL, flags) == -1)
			errx(1, "fcntl can't set flags");
		if ((flags = fcntl(pipe_m2e[1], F_GETFL, 0)) == -1)
			errx(1, "fcntl failed");
		flags |= O_NONBLOCK;
		if (fcntl(pipe_m2e[1], F_SETFL, flags) == -1)
			errx(1, "fcntl can't set flags");

		/*
		 * Fork into two processes, one to run privileged, one to do the
		 * work.
		 */
		if (verbose)
			warnx("start unprivileged child process");

		child_quit = 0;
		child_pid = e_main(uid, gid, pipe_m2e);

		/* We are the privileged process */
#ifndef __linux
		setproctitle("parent");
#endif

		signal(SIGCHLD, sighdlr);
		signal(SIGINT, sighdlr);
		signal(SIGTERM, sighdlr);
		signal(SIGHUP, sighdlr);
		if (statfile != NULL) {
			signal(SIGUSR1, sighdlr);
			if (stat_interval > 0) {
				signal(SIGALRM, sighdlr);
				alarm(stat_interval);
			}
		} else
			signal(SIGUSR1, SIG_IGN);

		close(pipe_m2e[1]);

		if ((ibuf_e = malloc(sizeof(struct imsgbuf))) == NULL)
			errx(1, "memory error");

		imsg_init(ibuf_e, pipe_m2e[0]);

		while (!quit && !child_quit) {
			pfd[0].fd = ibuf_e->fd;
			pfd[0].events = POLLIN;
			if ((nfds = poll(pfd, 1, 1000)) == -1) {
				if (errno != EINTR) {
					syslog(LOG_WARNING, "main: poll error");
					/* quit = 1; */
				}
			} else if (nfds > 0 && pfd[0].revents & POLLIN) {
				/* nfds --; */
				if (!child_quit && dispatch_imsg(ibuf_e) == -1)
					quit = 1;
			}
			if (child_quit && verbose)
				warnx("child process terminated");

			if (dumpstat) {
				if (statfile != NULL) {
					if (verbose)
						warnx("dumping statistics");
					time(&now);
					if ((fp = fopen(statfile, "a")) !=
					    NULL) {
						fprintf(fp, "%u\t%u\t%u\t%u\t"
						    "%u\t%u\t%u\t%u\t%u\n",	
						    stat_begin, now, n_conn,
						    n_aborts, n_msgs,
						    n_virus, n_spam,
						    n_unwanted, n_err);
						n_conn = n_aborts = n_msgs = 0;
						n_virus = n_spam = n_unwanted =
						    n_err = 0;
						stat_begin = now;
						fclose(fp);
					} else
						syslog(LOG_ERR, "can not open "
						    "'%s' for writing",
						    statfile);
					if (alrm_expired && stat_interval) {
						alrm_expired = 0;
						alarm(stat_interval);
					}
				} else
					syslog(LOG_WARNING, "received SIGUSR1 "
					    "but no statfile defined");
				dumpstat = 0;
			}
		}

		if (stat_interval && statfile != NULL)
			alarm(0);

		if (quit && !child_quit) {	/* XXX is this necessary? */
			sleep(1);
			if (!child_quit) {
				if (verbose)
					warnx("signalling child process to "
					    "terminate");
				kill(child_pid, SIGTERM);
			}
		}

		if (verbose)
			warnx("waiting for all child processes to terminate");

		do {
			if ((pid = wait(NULL)) == -1 &&
			    errno != EINTR && errno != ECHILD)
				errx(1, "wait");
		} while (pid != child_pid || (pid == -1 && errno == EINTR));

		if (verbose)
			warnx("child processes have terminated");

		msgbuf_clear(&ibuf_e->w);
		free(ibuf_e);
		close(pipe_m2e[0]);

		signal(SIGCHLD, SIG_DFL);
		signal(SIGINT, SIG_DFL);
		signal(SIGTERM, SIG_DFL);
		signal(SIGHUP, SIG_DFL);
		signal(SIGUSR1, SIG_DFL);
		signal(SIGALRM, SIG_DFL);

		/* Wait one second before restarting the child process */
		if (!quit)
			sleep(1);
	}

	closelog();
#ifdef ENABLE_LDAP
	ldap_unbind(ld);
#endif	
	return 0;
}

int
dispatch_imsg(struct imsgbuf *ibuf)
{
	struct imsg	imsg;
	int		n;
	int		rv;
	struct stats_msg	*stat;

	if ((n = imsg_read(ibuf)) == -1)
		return -1;

	if (n == 0) { /* connection closed */
		syslog(LOG_WARNING, "dispatch_imsg in main: pipe closed");
		return -1;
	}

	rv = 0;
	for (;;) {
		if ((n = imsg_get(ibuf, &imsg)) == -1)
			return -1;

		if (n == 0)
			break;

		switch (imsg.hdr.type) {
		case IMSG_RECONFIGURE:
			break;
#ifdef ENABLE_PF 
		case IMSG_PFTABLE_ADD:
			if (imsg.hdr.len != IMSG_HEADER_SIZE +
			    sizeof(struct pftable_msg))
				syslog(LOG_WARNING, "wrong imsg size");
			else if (pftable_addr_add(imsg.data) != 0) {
				rv = 1;
			}
			break;
		case IMSG_PFTABLE_DEL:
			if (imsg.hdr.len != IMSG_HEADER_SIZE +
			    sizeof(struct pftable_msg))
				syslog(LOG_WARNING, "wrong imsg size");
			else if (pftable_addr_del(imsg.data) != 0) {
				rv = 1;
			}
			break;
#endif
		case IMSG_STATS:
			if (verbose)
				warnx("received statistics from child process");
			++n_conn;
			stat = (struct stats_msg *)imsg.data;
			n_msgs += stat->msgs;
			n_aborts += stat->aborts;
			n_virus += stat->virus;
			n_spam += stat->spam;
			n_unwanted += stat->unwanted;
			n_err += stat->err;
			break;
		default:
			break;
		}
		imsg_free(&imsg);
		if (rv != 0)
			return rv;
	}
	return 0;
}
