/*	$Id: vilter-clamd.c,v 1.34 2007/01/16 15:31:14 mbalmer Exp $	*/

/*
 * Copyright (c) 2003 - 2007 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/un.h>
#include <sys/wait.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "pathnames.h"
#include "smtp-vilter.h"
#include "vilter-clamd.h"

#ifdef __linux__
#include "strlfunc.h"
#endif

#define CLAMD_FROM	"From smtp-vilter\r\n"

#define TIMEOUT 	60
#define MAXBUF		1024
#define MAXSTR		64
#define MAXTRIES	256


int	 clamd_port;
char	*clamd_sock;
int	 write_from;
char	*clamd_host;
char	*bind_addr;
int	 clamd_tries;
int	 clamd_timeout;
int	 scantype;
int	 chroot_scanrealpath;

int
vilter_envfrom(char *env_from, FILE *fp)
{
	/*
	 * Write "From <sender>" to the temporary file so that clamd
	 * recognizes the file as e-mail
	 */
	if (scantype == SCANTYPE_FILESYSTEM && fp != NULL && env_from != NULL)
		fprintf(fp, "From %s\r\n", env_from);

	return 0;
}

int
connect_inet()
{
	struct sockaddr_in	 server_sockaddr;
	struct sockaddr_in	 bind_sockaddr;
	struct hostent		*hostent;
	int			 fd;
	int			 try;

	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog(LOG_ERR, "clamd: unable to obtain network");
		return -1;
	}

	if (bind_addr != NULL) {
		bzero((char *)&server_sockaddr, sizeof(server_sockaddr));
		bind_sockaddr.sin_family = AF_INET;

		if (inet_aton(bind_addr, &bind_sockaddr.sin_addr) == 0) {
			hostent = gethostbyname(bind_addr);
			if (!hostent) {
				syslog(LOG_ERR, "clamd: unknown bind address: "
				    "%s", bind_addr);
				close(fd);
				return -1;
			}

			bind_sockaddr.sin_family = hostent->h_addrtype;
			memcpy(&bind_sockaddr.sin_addr, hostent->h_addr,
			    hostent->h_length);
		}

		if (bind(fd, (struct sockaddr *)&bind_sockaddr,
		    sizeof(bind_sockaddr))) {
			syslog(LOG_ERR, "clamd: can't bind to address %s",
			    bind_addr);
			close(fd);
			return -1;
		}
	}
  
	bzero((char *)&server_sockaddr, sizeof(server_sockaddr));
	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(clamd_port);

	if (inet_aton(clamd_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(clamd_host);
		if (!hostent) {
			syslog(LOG_ERR, "clamd: unknown host: %s", clamd_host);
			close(fd);
			return -1;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr,
		    hostent->h_length);
	}

	try = 0;
	while (connect(fd, (struct sockaddr *)&server_sockaddr,
	    sizeof(server_sockaddr)) == -1) {
		++try;

		if (try == clamd_tries) {
			syslog(LOG_ERR, "clamd: unable to connect socket");
			close(fd);
			return -1;
		} else
			sleep(1);
	}
	return fd;
}

static int
connect_local(void)
{
	struct sockaddr_un	 addr;
	int			 fd;

	fd = socket(AF_UNIX, SOCK_STREAM, 0);
	bzero(&addr, sizeof(addr));
	addr.sun_family = AF_UNIX;
	strlcpy(addr.sun_path, clamd_sock, sizeof(addr.sun_path));
	if (connect(fd, (struct sockaddr *)&addr, sizeof(addr))) {
		syslog(LOG_ERR, "clamd: unable to connect to socket %s",
		    clamd_sock);
		return -1;
	}
	return fd;
}

int
vilter_scan(SMFICTX *ctx, char *fn, size_t fnlen, char *chroot, char *virus,
    size_t namelen)
{
	int			 fd, datfd;
	int			 nread;
	FILE			*fp;
	SOCK			*sp;
	unsigned short		 port;
	struct sockaddr_in	 bind_sockaddr, data_sockaddr;
	struct sockaddr_in	 server_sockaddr;
	struct hostent		*hostent;
	int			 try;
	struct timeval		 tv, *t;
	char			 buf[MAXBUF];
	char			*p, *q;
	struct msghdr		 msg;
	struct cmsghdr		*cmsg;
	unsigned char		 fdbuf[CMSG_SPACE(sizeof(int))];

	if (clamd_sock != NULL)
		fd = connect_local();
	else
		fd = connect_inet();
	if (fd == -1)
		return SCAN_ERROR;

	if (clamd_timeout > 0) {
		tv.tv_sec = clamd_timeout;
		tv.tv_usec = 0;
		t = &tv;
	} else
		t = NULL;
	if ((sp = fdsock(fd)) == NULL) {
		syslog(LOG_ERR, "clamd: unable to buffer socket");
		close(fd);
		return SCAN_ERROR;
	}

	switch (scantype) {
	case SCANTYPE_FILESYSTEM:
		if (chroot_scanrealpath && chroot != NULL)
			snprintf(buf, sizeof(buf), "SCAN %s/%s\r\n", chroot,
			    *fn == '/' ? fn + 1 : fn);
		else
			snprintf(buf, sizeof(buf), "SCAN %s\r\n", fn);

		if (to_send(fd, buf, strlen(buf), 0, t) != strlen(buf)) {
			syslog(LOG_ERR, "clamd: send error");
			close(fd);
			return SCAN_ERROR;
		}
		break;
	case SCANTYPE_SOCKET:
		if (to_send(fd, "STREAM\r\n", 8, 0, t) != 8) {
			syslog(LOG_ERR, "clamd: timeout");
			close(fd);
			return SCAN_ERROR;
		}
		if (to_readln(buf, sizeof(buf), sp, t) <= 0) {
			syslog(LOG_ERR, "clamd: no response to STREAM cmd");
			sclose(sp);
			return SCAN_ERROR;
		}
		if (sscanf(buf, "PORT %hu\n", &port) != 1) {
			syslog(LOG_ERR, "clamd: did not receive port number");
			sclose(sp);
			return SCAN_ERROR;
		}
		if ((datfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
			syslog(LOG_ERR, "clamd: unable to obtain network");
			sclose(sp);
			return SCAN_ERROR;
		}
		if (bind_addr != NULL) {
			bzero((char *)&server_sockaddr,
			    sizeof(server_sockaddr));
			bind_sockaddr.sin_family = AF_INET;

			if (inet_aton(bind_addr, &bind_sockaddr.sin_addr) ==
			    0) {
				hostent = gethostbyname(bind_addr);
				if (!hostent) {
					syslog(LOG_ERR, "clamd: unknown bind "
					    "address: %s", bind_addr);
					close(datfd);
					return SCAN_ERROR;
				}

				bind_sockaddr.sin_family = hostent->h_addrtype;
				memcpy(&bind_sockaddr.sin_addr,
				    hostent->h_addr, hostent->h_length);
			}
			if (bind(datfd, (struct sockaddr *)&bind_sockaddr,
			    sizeof(bind_sockaddr))) {
				syslog(LOG_ERR, "clamd: can't bind to address "
				    "%s", bind_addr);
				close(datfd);
				return SCAN_ERROR;
			}
		}
		bzero((char *)&data_sockaddr, sizeof(data_sockaddr));

		data_sockaddr.sin_family = AF_INET;
		data_sockaddr.sin_port = htons(port);

		if (inet_aton(clamd_host, &data_sockaddr.sin_addr) == 0) {
			hostent = gethostbyname(clamd_host);
			if (!hostent) {
				syslog(LOG_ERR, "clamd: unknown host: %s",
				    clamd_host);
				close(datfd);
				close(fd);
				return SCAN_ERROR;
			}
			data_sockaddr.sin_family = hostent->h_addrtype;
			memcpy(&data_sockaddr.sin_addr, hostent->h_addr,
			    hostent->h_length);
		} 
		try = 0;
		while (connect(datfd, (struct sockaddr *)&data_sockaddr,
		    sizeof(data_sockaddr)) == -1) {
			++try;
			if (try == clamd_tries) {
				syslog(LOG_ERR, "clamd: unable to connect "
				    "socket");
				close(datfd);
				sclose(sp);
				return SCAN_ERROR;
			} else
				sleep(1);
		}
		if ((fp = fopen(fn, "r")) == NULL) {
			syslog(LOG_ERR, "clamd: error opening file '%s' for "
			    "reading", fn);
			close(datfd);
			sclose(sp);
			return SCAN_ERROR;
		}
		if (to_send(datfd, CLAMD_FROM, sizeof(CLAMD_FROM), 0, t) !=
		    sizeof(CLAMD_FROM)) {
			syslog(LOG_ERR, "clamd: error sending From");
			close(datfd);
			sclose(sp);
			fclose(fp);
			return SCAN_ERROR;	
		}
		while ((nread = fread(buf, 1, sizeof(buf), fp)) > 0) {
			if (to_send(datfd, buf, nread, 0, t) != nread) {
				syslog(LOG_ERR, "clamd: error sending file");
				close(datfd);
				sclose(sp);
				fclose(fp);
				return SCAN_ERROR;
			}
		}
		fclose(fp);	
		close(datfd);
		break;
	case SCANTYPE_FILDES:
		if ((datfd = open(fn, O_RDONLY)) == -1) {
			syslog(LOG_ERR, "clamd: can't open file to scan");
			sclose(sp);
			return SCAN_ERROR;
		}
		memset(&msg, 0, sizeof(msg));
		msg.msg_control = fdbuf;
		msg.msg_controllen = CMSG_LEN(sizeof(int));

		cmsg = CMSG_FIRSTHDR(&msg);
		cmsg->cmsg_len = CMSG_LEN(sizeof(int));
		cmsg->cmsg_level = SOL_SOCKET;
		cmsg->cmsg_type = SCM_RIGHTS;
		*(int *)CMSG_DATA(cmsg) = datfd;

		strlcpy(buf, "FILDES\n", sizeof(buf));
		if (to_send(fd, buf, strlen(buf), 0, t) != strlen(buf)) {
			syslog(LOG_ERR, "clamd: send error");
			close(datfd);
			close(fd);
			return SCAN_ERROR;
		}
		if (sendmsg(fd, &msg, 0) == -1) {
			syslog(LOG_ERR, "clamd: sendmsg failed, errno %d",
			    errno);
			close(datfd);
			close(fd);
			return SCAN_ERROR;
		}
		close(datfd);
		break;
	default:
		syslog(LOG_ERR, "clamd: internal error, unknown scantype %d",
		    scantype);
		close(fd);
		return SCAN_ERROR;
	}
		
	if (to_readln(buf, sizeof(buf), sp, t) <= 0) {
		syslog(LOG_ERR, "clamd: no response from clamd");
		sclose(sp);
		return SCAN_ERROR;
	}

	sclose(sp);

	if ((p = strrchr(buf, ' ')) == NULL) {
		syslog(LOG_ERR, "clamd: unable to interpret result '%s'", buf);
		return SCAN_ERROR;
	} else {
		++p;
		if (!strncmp(p, "OK", 2))	/* File is OK */
			return SCAN_OK;
		else if (!strncmp(p, "FOUND", 5)) {
			*--p = '\0';
			q = strrchr(buf, ' ') + 1;
			syslog(LOG_ERR, "clamd: found '%s'", q);
			strlcpy(virus, q, namelen);
			return SCAN_VIRUS;
		} else if (!strncmp(p, "ERROR", 5)) {
			*--p = '\0';
			syslog(LOG_ERR, "clamd: error '%s'", buf);
			return SCAN_ERROR;
		} else
			syslog(LOG_ERR, "clamd: unrecognized response: '%s'",
			    p);
 	}
	return SCAN_ERROR;
}

char *
vilter_name(void)
{
	return "Clam AntiVirus Daemon (clamd)";
}

int
vilter_type(void)
{
	return BE_SCAN_VIRUS;
}

void
vilter_exit(void)
{
	if (verbose)
		warnx("clamd: vilter_exit()");

	free(clamd_host);
}
