%{
/*	$Id: parse.y,v 1.46 2006/06/24 10:44:37 mbalmer Exp $	*/

/*
 * Copyright (c) 2005, 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <ctype.h>
#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include "smtp-vilter.h"

#if defined (__linux__)
#include "strtonum.h"
#endif

extern FILE	*vilterin;
extern int	 vilterlineno;
extern char	*viltertext;

extern char	*port;
extern int	 virus_strategy;
extern int	 virus_reaction;
extern long	 virus_duration;
extern int	 spam_strategy;
extern int	 spam_reaction;
extern long	 spam_duration;
extern int	 unwanted_strategy;
extern int	 unwanted_reaction;
extern long	 unwanted_duration;
extern int	 clean_reaction;
extern long	 clean_duration;
extern char	*virus_table;
extern char	*spam_table;
extern char	*unwanted_table;
extern char	*clean_table;
extern int	 error_strategy;
extern int	 markall;
extern int	 logall;
extern int	 verbose;
extern char	*cfgfile;
extern char	*tmpdir;
extern int	 tmpfiles_gr;
extern int	 tmpfiles_setgrp;
extern char	*backend;
extern char	*recipient_notification;
extern char	*notify_only;
extern int	 log_facility;
extern char	*logfile;
extern FILE	*logfp;
extern char	*user;
extern char	*group;
extern char	*chrootdir;
extern rlim_t	 rlimit_nofile;
extern rlim_t	 rlimit_nproc;
extern int	 nbe;
extern char	*spamprefix;
extern char	*spam_headerf;
extern char	*spam_headerv;

extern int	 logvirus, logspam, logunwanted;
extern int	 keep_tmpfiles;

extern struct backend	*be;

extern char	*cfgfile;
extern char	*ldaphost;
extern int	 ldapport;
extern char	*ldapurl;
extern char	*searchbase;
extern char	*rulebase;
extern char	*binddn;
extern char	*bindpasswd;
extern int	 ldap_use_tls;
extern char	*statfile;
extern unsigned int stat_interval;

extern void decode_backend(char *);
extern int pftable_exists(const char *);

int	viltererror(const char *, ...);
int	vilterparse(void);
int	vilterlex(void);

static int 	 viltererrcnt;

%}

%union {
	long	 number;
	char	*string;
}

%token	USER GROUP CHROOT TMPFILES BACKEND CONFIG_FILE
%token	VIRUS_STRATEGY RECIPIENT_NOTIFICATION SPAM_STRATEGY
%token	SPAM_SUBJECT_PREFIX UNWANTED_STRATEGY ERROR_STRATEGY
%token	PORT LOGFILE OPTION NOTIFY_ONLY LOGALL STATFILE
%token	DISCARD MARK NOTIFY_RECIPIENT INTERVAL SPAMHDR
%token	RJECT TEMPFAIL IGNORE
%token	GRPRD SETGRP CFGFILE TMPDIR MAXFILES MAXPROCS MARKALL
%token	LOGVIRUS LOGSPAM LOGUNWANTED LDAP_USE_TLS
%token	LOGFAC LDAPHOST LDAPPORT LDAPURL BINDDN BINDPASSWD SEARCHBASE
%token	USETLS LOCAL0 LOCAL1 LOCAL2 LOCAL3
%token	LOCAL4 LOCAL5 LOCAL6 LOCAL7 COMMENT NEVER TRY ALWAYS
%token	DAEMON MAIL USER STRATEGY
%token	FOR SPAM VIRUS UNWANTED_CONTENT CLEAN
%token	REACT ON ADD TO TABLE DELETE FROM
%token	SECONDS MINUTES HOURS DAYS
%token	<string>	NUMBER
%token	<string>	TEXT
%type	<number>	event
%type	<number>	reaction
%type	<number>	duration
%type	<number>	number
%type	<number>	logfac
%type	<number>	v_strategy
%type	<number>	e_strategy
%type	<number>	su_strategy

%%
statement	: /* empty */
		| statement '\n'
		| statement user '\n'
		| statement group '\n'
		| statement chroot '\n'
		| statement tmpfiles '\n'
		| statement backend '\n'
		| statement config_file '\n'
		| statement virus_strategy '\n'
		| statement rcpt_notification '\n'
		| statement notify_only '\n'
		| statement spam_strategy '\n'
		| statement spam_subject_prefix '\n'
		| statement spam_header '\n'
		| statement unwanted_strategy '\n'
		| statement error_strategy '\n'
		| statement port '\n'
		| statement tmpdir '\n'
		| statement maxfiles '\n'
		| statement maxprocs '\n'
		| statement logfacility '\n'
		| statement logfile '\n'
		| statement statfile '\n'
		| statement interval '\n'
		| statement option '\n'
		| statement ldaphost '\n'
		| statement ldapport '\n'
		| statement ldapurl '\n'
		| statement searchbase '\n'
		| statement binddn '\n'
		| statement bindpasswd '\n'
		| statement usetls '\n'
		| statement react '\n'
		;

user		: USER '=' TEXT				{
			if (user == NULL)
				user = $3;
			else
				free($3);
		}
		;

group		: GROUP '=' TEXT			{
			if (group == NULL)
				group = $3;
			else
				free($3);
		}
		;

chroot		: CHROOT '=' TEXT			{
			if (chrootdir == NULL)
				chrootdir = $3;
			else
				free($3);
		}
		;

tmpfiles	: TMPFILES '=' GRPRD			{
			tmpfiles_gr = 1;
		}
		| TMPFILES '=' SETGRP			{
			tmpfiles_setgrp = 1;
		}
		;

backend		: BACKEND '=' TEXT			{
			if (backend == NULL)
				decode_backend($3);
			free($3);
		}
		;

config_file	: CFGFILE '=' TEXT			{
			char *be_cfgfile;
			int n;
			int be_found;

			be_cfgfile = strchr($3, ':');
			if (be_cfgfile == NULL)
				errx(1, "config-file syntax error");

			*be_cfgfile++ = '\0';
			be_found = 0;
			for (n = 0; n < nbe; n++) {
				if (!strcmp(be[n].name, $3)) {
					if ((be[n].config_file =
					    strdup(be_cfgfile)) == NULL)
						err(1, "memory error, can't "
						    "store config file %s for "
						    "backend %s", be_cfgfile,
						    be[n].name);
					be_found = 1;
					break;
				}
			}
			if (!be_found && verbose) 
				warnx("config-file for unused backend "
				    "%s defined", $3);
		}
		;
notify_only	: NOTIFY_ONLY '=' TEXT			{
			notify_only = $3;
		}
		;

rcpt_notification	: RECIPIENT_NOTIFICATION '=' TEXT	{
			recipient_notification = $3;
		}
		;

port		: PORT '=' TEXT				{
			if (port == NULL)
				port = $3;
			else
				free($3);
		}
		;

tmpdir		: TMPDIR '=' TEXT			{
			if (tmpdir == NULL)
				tmpdir = $3;
			else
				free($3);
		}
		;

maxfiles	: MAXFILES '=' NUMBER			{
			rlim_t rlimit_max;
			struct rlimit rl;
			const char *errstr;

			if (rlimit_nofile == 0) {
				if (getuid()) {
					getrlimit(RLIMIT_NOFILE, &rl);
					rlimit_max = rl.rlim_max;
				} else
					rlimit_max = RLIM_INFINITY;
				rlimit_nofile = (rlim_t)strtonum($3, 1LL,
				    (long long)rlimit_max, &errstr);
				if (errstr)
					errx(1, "maxfiles is %s: %s", errstr,
					    $3);
			}
			free($3);
		}
		;

maxprocs	: MAXPROCS '=' NUMBER			{
			rlim_t rlimit_max;
			struct rlimit rl;
			const char *errstr;

			if (rlimit_nproc == 0) {
				if (getuid()) {
					getrlimit(RLIMIT_NPROC, &rl);
					rlimit_max = rl.rlim_max;
				} else
					rlimit_max = RLIM_INFINITY;
				rlimit_nproc = (rlim_t)strtonum($3, 1LL,
				    (long long)rlimit_max, &errstr);
				if (errstr)
					errx(1, "maxprocs is %s: %s", errstr,
					    $3);
			}
			free($3);
		}
		;

logfile		: LOGFILE '=' TEXT			{
			if (logfile == NULL)
				logfile = $3;
			else
				free($3);
		}
		;

statfile	: STATFILE '=' TEXT			{
			if (statfile == NULL)
				statfile = $3;
			else
				free($3);
		}
		;

interval	: INTERVAL '=' TEXT			{
			const char *errstr;

			if (stat_interval == 0) {
				stat_interval = (unsigned int)strtonum($3, 1LL,
				    100000000LL, &errstr);
				if (errstr)
					errx(1, "interval is %s: %s", errstr,
					    $3);
			}
			free($3);
		}
		;

virus_strategy	: VIRUS_STRATEGY '=' v_strategy		{
			if (virus_strategy == -1)
				virus_strategy = $3;
		}
		;

v_strategy	: DISCARD		{ $$ = STRATEGY_DISCARD; }
		| MARK			{ $$ = STRATEGY_MARK; }
		| NOTIFY_RECIPIENT	{ $$ = STRATEGY_NOTIFY_RECIPIENT; }
		| RJECT			{ $$ = STRATEGY_REJECT; }
		| IGNORE		{ $$ = STRATEGY_IGNORE; }
		;

error_strategy	: ERROR_STRATEGY '=' e_strategy		{
			if (error_strategy == -1)
				error_strategy = $3;
		}
		;

e_strategy	: DISCARD		{ $$ = STRATEGY_DISCARD; }
		| MARK			{ $$ = STRATEGY_MARK; }
		| TEMPFAIL		{ $$ = STRATEGY_TEMPFAIL; }
		| IGNORE		{ $$ = STRATEGY_IGNORE; }
		;

spam_strategy	: SPAM_STRATEGY '=' su_strategy		{
			if (spam_strategy == -1)
				spam_strategy = $3;
		}
		;

unwanted_strategy	: UNWANTED_STRATEGY '=' su_strategy		{
			if (unwanted_strategy == -1)
				unwanted_strategy = $3;
		}
		;
su_strategy	: DISCARD		{ $$ = STRATEGY_DISCARD; }
		| MARK			{ $$ = STRATEGY_MARK; }
		| RJECT			{ $$ = STRATEGY_REJECT; }
		| IGNORE		{ $$ = STRATEGY_IGNORE; }
		;

spam_subject_prefix	: SPAM_SUBJECT_PREFIX '=' TEXT	{
			if (spamprefix == NULL)
				spamprefix = $3;
			else
				free($3);
		}
		;

spam_header	: SPAMHDR '=' TEXT			{
			spam_headerf = $3;
			if ((spam_headerv = strchr(spam_headerf, ':')) != NULL) {
				*spam_headerv++ = '\0';
				for (; *spam_headerv && isspace(*spam_headerv);
				    spam_headerv++)
					;
				if (!strlen(spam_headerf) ||
				    !strlen(spam_headerv))
					spam_headerv = NULL;
			}
			if (spam_headerv == NULL)
				errx(1, "spam-header requires a "
				    "colon-separated header field and value");
		}
		;

option		: OPTION '=' MARKALL			{
			markall = 1;
		}
		| OPTION '=' LOGALL			{
			logall = 1;
		}
		| OPTION '=' LOGVIRUS			{
			logvirus = 1;
		}
		| OPTION '=' LOGSPAM			{
			logspam = 1;
		}
		| OPTION '=' LOGUNWANTED		{
			logunwanted = 1;
		}
		| OPTION '=' LDAP_USE_TLS		{
			ldap_use_tls = 1;
		}
		;

ldaphost	: LDAPHOST '=' TEXT			{
			if (ldaphost == NULL)
				ldaphost = $3;
			else
				free($3);
		}
		;

ldapport	: LDAPPORT '=' NUMBER			{
			if (ldapport == -1) {
				const char *errstr;
			
				ldapport = (int)strtonum($3, 1LL, 65535LL,
				    &errstr);
				if (errstr)
					errx(1, "port number is %s: %s",
					    errstr, $3);
			}
			free($3);
		}
		;

ldapurl		: LDAPURL '=' TEXT			{
			if (ldapurl == NULL)
				ldapurl = $3;
			else
				free($3);
		}
		;

searchbase	: SEARCHBASE '=' TEXT		{
			if (searchbase == NULL)
				searchbase = $3;
			else
				free($3);
		}
		;


binddn		: BINDDN '=' TEXT		{
			if (binddn == NULL)
				binddn = $3;
			else
				free($3);

		}
		;

bindpasswd	: BINDPASSWD '=' TEXT		{
			if (bindpasswd == NULL)
				bindpasswd = $3;
			else
				free($3);
		}
		;

logfacility	:	LOGFAC '=' logfac	{
				log_facility = $3;
		}
		;

logfac		:	LOCAL0		{ $$ = LOG_LOCAL0; }
		|	LOCAL1		{ $$ = LOG_LOCAL1; }
		|	LOCAL2		{ $$ = LOG_LOCAL2; }
		|	LOCAL3		{ $$ = LOG_LOCAL3; }
		|	LOCAL4		{ $$ = LOG_LOCAL4; }
		|	LOCAL5		{ $$ = LOG_LOCAL5; }
		|	LOCAL6		{ $$ = LOG_LOCAL6; }
		|	LOCAL7		{ $$ = LOG_LOCAL7; }
		|	DAEMON		{ $$ = LOG_DAEMON; }
		|	MAIL		{ $$ = LOG_MAIL; }
		|	USER		{ $$ = LOG_USER; }
		;
	
usetls		: USETLS '=' NEVER		{
			if (ldap_use_tls == -1)
				ldap_use_tls = 0;
		}
		| USETLS '=' TRY		{
			if (ldap_use_tls == -1)
				ldap_use_tls = 1;
		}
		| USETLS '=' ALWAYS		{
			if (ldap_use_tls == -1)
				ldap_use_tls = 2;
		}
		;

react		: REACT ON event reaction TEXT duration	{
#ifndef ENABLE_PF
			fprintf(stderr, "reactions are a no-op on this "
			    "platform\n");
#else
			if (pftable_exists($5)) {
				/* XXX create the table? */
				fprintf(stderr, "table %s is not defined in "
				    "pf\n", $5);
				free($5);
			} else {
				switch ($3) {
				case VIRUS:
					virus_reaction = $4;
					virus_table = $5;
					virus_duration = $6;
					break;
				case SPAM:
					spam_reaction = $4;
					spam_table = $5;
					spam_duration = $6;
					break;
				case UNWANTED_CONTENT:
					unwanted_reaction = $4;
					unwanted_table = $5;
					unwanted_duration = $6;
					break;
				case CLEAN:
					clean_reaction = $4;
					clean_table = $5;
					clean_duration = $6;
					break;
				}
			}
#endif
		}
		;

event		: SPAM				{
			$$ = SPAM;
		}
		| VIRUS				{
			$$ = VIRUS;
		}
		| UNWANTED_CONTENT		{
			$$ = UNWANTED_CONTENT;
		}
		| CLEAN				{
			$$ = CLEAN;
		}
		;

reaction	: ADD TO TABLE			{
			$$ = REACTION_ADDTBL;
		}
		| DELETE FROM TABLE		{
			$$ = REACTION_DELTBL;
		}
		;

duration	: /* empty */			{
			$$ = 0;
		}	
		| FOR number SECONDS		{
			$$ = $2;
		}
		| FOR number MINUTES		{
			$$ = $2 * 60;
		}
		| FOR number HOURS		{
			$$ = $2 * 3600;
		}
		| FOR number DAYS		{
			$$ = $2 * 86400;
		}
		;

number		: NUMBER			{
			$$ = atol($1);
			free($1);
		}
		;
%%

void
smtp_vilter_init(void)
{
	vilterlineno = 1;
	viltererrcnt = 0;
	if ((vilterin = fopen(cfgfile, "r")) != NULL) {
		while (!feof(vilterin))
			vilterparse();

		fclose(vilterin);
	}
	if (viltererrcnt)
		errx(1, "configuration file contains errors, terminating");
}

int
viltererror(const char *fmt, ...)
{
	va_list		 ap;
	char		*nfmt;

	++viltererrcnt;
	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s, line %d: %s near '%s'",
	    cfgfile, vilterlineno, fmt, viltertext) == -1)
		errx(1, "asprintf failed");
	fprintf(stderr, "%s\n", nfmt);
	va_end(ap);
	free(nfmt);
	return (0);
}
